/* vi: set sw=4 ts=4: */
/*
 *	Copied from Linux Monitor (LiMon) - Networking.
 *
 *	Copyright 1994 - 2000 Neil Russell.
 *	(See License)
 *	Copyright 2000 Roland Borde
 *	Copyright 2000 Paolo Scaffardi
 *	Copyright 2000-2002 Wolfgang Denk, wd@denx.de
 */

/*
 * General Desription:
 *
 * The user interface supports commands for BOOTP, RARP, and TFTP.
 * Also, we support ARP internally. Depending on available data,
 * these interact as follows:
 *
 * BOOTP:
 *
 *	Prerequisites:	- own ethernet address
 *	We want:	- own IP address
 *			- TFTP server IP address
 *			- name of bootfile
 *	Next step:	ARP
 *
 * RARP:
 *
 *	Prerequisites:	- own ethernet address
 *	We want:	- own IP address
 *			- TFTP server IP address
 *	Next step:	ARP
 *
 * ARP:
 *
 *	Prerequisites:	- own ethernet address
 *			- own IP address
 *			- TFTP server IP address
 *	We want:	- TFTP server ethernet address
 *	Next step:	TFTP
 *
 * DHCP:
 *
 *     Prerequisites:   - own ethernet address
 *     We want:         - IP, Netmask, ServerIP, Gateway IP
 *                      - bootfilename, lease time
 *     Next step:       - TFTP
 *
 * TFTP:
 *
 *	Prerequisites:	- own ethernet address
 *			- own IP address
 *			- TFTP server IP address
 *			- TFTP server ethernet address
 *			- name of bootfile (if unknown, we use a default name
 *			  derived from our own IP address)
 *	We want:	- load the boot file
 *	Next step:	none
 *
 * NFS:
 *
 *	Prerequisites:	- own ethernet address
 *			- own IP address
 *			- name of bootfile (if unknown, we use a default name
 *			  derived from our own IP address)
 *	We want:	- load the boot file
 *	Next step:	none
 */


#include <common.h>
#include <watchdog.h>
#include <command.h>
#include <net.h>
#include "bootp.h"
#include "tftp.h"
#ifdef HTTPSVR_SUPPORT
#include "http.h"
#endif
#include "rarp.h"
//#include "nfs.h"
#include <asm/addrspace.h>
#undef DEBUG
#ifdef CONFIG_STATUS_LED
#include <status_led.h>
#include <miiphy.h>
#endif

#if (CONFIG_COMMANDS & CFG_CMD_NET)

#define ARP_TIMEOUT		5		/* Seconds before trying ARP again */
#ifndef	CONFIG_NET_RETRY_COUNT
# define ARP_TIMEOUT_COUNT	5		/* # of timeouts before giving up  */
#else
# define ARP_TIMEOUT_COUNT  (CONFIG_NET_RETRY_COUNT)
#endif

#if 0
#define ET_DEBUG
#endif

/** BOOTP EXTENTIONS **/

IPaddr_t	NetOurSubnetMask=0;		/* Our subnet mask (0=unknown)	*/
IPaddr_t	NetOurGatewayIP=0;		/* Our gateways IP address	*/
IPaddr_t	NetOurDNSIP=0;			/* Our DNS IP address		*/
#if (CONFIG_BOOTP_MASK & CONFIG_BOOTP_DNS2)
IPaddr_t	NetOurDNS2IP=0;			/* Our 2nd DNS IP address	*/
#endif
char		NetOurNISDomain[32]={0,};	/* Our NIS domain		*/
char		NetOurHostName[32]={0,};	/* Our hostname			*/
char		NetOurRootPath[64]={0,};	/* Our bootpath			*/
ushort		NetBootFileSize=0;		/* Our bootfile size in blocks	*/

/** END OF BOOTP EXTENTIONS **/

ulong		NetBootFileXferSize;	/* The actual transferred size of the bootfile (in bytes) */
uchar		NetOurEther[6];		/* Our ethernet address			*/
uchar		NetServerEther[6] =	/* Boot server enet address		*/
			{ 0, 0, 0, 0, 0, 0 };
IPaddr_t	NetOurIP;		/* Our IP addr (0 = unknown)		*/
IPaddr_t	NetServerIP;		/* Our IP addr (0 = unknown)		*/
volatile uchar *NetRxPkt;		/* Current receive packet		*/
int		NetRxPktLen;		/* Current rx packet length		*/
unsigned	NetIPID;		/* IP packet ID				*/
uchar		NetBcastAddr[6] =	/* Ethernet bcast address		*/
			{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
uchar		NetEtherNullAddr[6] =
			{ 0, 0, 0, 0, 0, 0 };
#if (CONFIG_COMMANDS & CFG_CMD_CDP)
uchar		NetCDPAddr[6] =		/* Ethernet bcast address		*/
			{ 0x01, 0x00, 0x0c, 0xcc, 0xcc, 0xcc };
#endif
int		NetState;		/* Network loop state			*/
#ifdef CONFIG_NET_MULTI
int		NetRestartWrap = 0;	/* Tried all network devices		*/
static int	NetRestarted = 0;	/* Network loop restarted		*/
static int	NetDevExists = 0;	/* At least one device configured	*/
#endif

/* XXX in both little & big endian machines 0xFFFF == ntohs(-1) */
ushort		NetOurVLAN = 0xFFFF;		/* default is without VLAN	*/
ushort		NetOurNativeVLAN = 0xFFFF;	/* ditto			*/

char		BootFile[128];		/* Boot File name			*/

#if (CONFIG_COMMANDS & CFG_CMD_PING)
IPaddr_t	NetPingIP;		/* the ip address to ping 		*/

static void PingStart(void);
#endif

#if (CONFIG_COMMANDS & CFG_CMD_CDP)
static void CDPStart(void);
#endif

#ifdef CONFIG_NETCONSOLE
void NcStart(void);
int nc_input_packet(uchar *pkt, unsigned dest, unsigned src, unsigned len);
#endif

  
volatile uchar	*PktBuf;
	
volatile uchar	Pkt_Buf_Pool[(PKTBUFSRX+2) * PKTSIZE_ALIGN + PKTALIGN];

//FRANK request
//

volatile uchar *NetRxPackets[PKTBUFSRX]; /* Receive packets			*/

static rxhand_f *	packetHandler=NULL;	/* Current RX packet handler		*/
static rxhand_f *	TcpPacketHandler=NULL;
static thand_f *	timeHandler;		/* Current timeout handler		*/
static ulong		timeStart;			/* Time base value			*/
static ulong		timeDelta;			/* Current timeout value		*/
volatile uchar *	NetTxPacket = 0;	/* THE transmit packet			*/

static int net_check_prereq (proto_t protocol);

/**********************************************************************/

IPaddr_t	NetArpWaitPacketIP;
IPaddr_t	NetArpWaitReplyIP;
uchar	       *NetArpWaitPacketMAC;	/* MAC address of waiting packet's destination	*/
uchar          *NetArpWaitTxPacket;	/* THE transmit packet			*/
int		NetArpWaitTxPacketSize;
uchar 		NetArpWaitPacketBuf[PKTSIZE_ALIGN + PKTALIGN];
ulong		NetArpWaitTimerStart;
int		NetArpWaitTry;


//===================================================
/*=======================================*/




extern VALID_BUFFER_STRUCT  rt2880_free_buf_list;
//kaiker
extern BUFFER_ELEM *rt2880_free_buf_entry_dequeue(VALID_BUFFER_STRUCT *hdr);



/*=======================================*/
//===================================================

void ArpRequest(void)
{
	int i;
	volatile uchar *pkt;
	ARP_t *arp;

	etdebug("%s: ARP broadcast %d\n",__func__,NetArpWaitTry);

	pkt = NetTxPacket;
	pkt += NetSetEther(pkt, NetBcastAddr, PROT_ARP);

	arp = (ARP_t *)pkt;
	arp->ar_hrd = htons(ARP_ETHER);
	arp->ar_pro = htons(PROT_IP);
	arp->ar_hln = 6;
	arp->ar_pln = 4;
	arp->ar_op = htons(ARPOP_REQUEST);

	memcpy(&arp->ar_data[0], NetOurEther, 6);		/* source ET addr */
	NetWriteIP((uchar *)&arp->ar_data[6], NetOurIP);/* source IP addr */
	for (i = 10; i < 16; ++i) arp->ar_data[i] = 0;	/* dest ET addr = 0 */

	if ((NetArpWaitPacketIP & NetOurSubnetMask) !=
	    (NetOurIP & NetOurSubnetMask))
	{
		if (NetOurGatewayIP == 0)
		{
			puts ("## Warning: gatewayip needed but not set\n");
		}
		NetArpWaitReplyIP = NetOurGatewayIP;
	}
	else
	{
		NetArpWaitReplyIP = NetArpWaitPacketIP;
	}

	NetWriteIP((uchar *) & arp->ar_data[16], NetArpWaitReplyIP);
	(void)eth_send(NetTxPacket, (pkt - NetTxPacket) + ARP_HDR_SIZE);
}

static void ArpTimeoutCheck(void)
{
	ulong t;

	if (!NetArpWaitPacketIP) return;

	t = get_timer(0);

	/* check for arp timeout */
	if ((t - NetArpWaitTimerStart) > ARP_TIMEOUT * CFG_HZ)
	{
		NetArpWaitTry++;
		if (NetArpWaitTry >= ARP_TIMEOUT_COUNT)
		{
			puts("ARP Retry count exceeded; starting again\n");
			NetArpWaitTry = 0;
			NetStartAgain();
		}
		else
		{
			NetArpWaitTimerStart = t;
			debug("%s: ArpTimeoutCheck\n",__func__);
			ArpRequest();
		}
	}
}

/**********************************************************************/
/*
 *	Main network processing loop.
 */
int NetLoop(proto_t protocol)
{
	DECLARE_GLOBAL_DATA_PTR;
	bd_t *bd = gd->bd;

	//debug("%s: >>>\n",__func__);

#ifdef CONFIG_NET_MULTI
	NetRestarted = 0;
	NetDevExists = 0;
#endif

	/* XXX problem with bss workaround */
	NetArpWaitPacketMAC = NULL;
	NetArpWaitTxPacket = NULL;
	NetArpWaitPacketIP = 0;
	NetArpWaitReplyIP = 0;
	NetArpWaitTxPacket = NULL;

	if (!NetTxPacket)
	{
		int	i;
		BUFFER_ELEM *buf;

		/* Setup packet buffers, aligned correctly. */
		buf = rt2880_free_buf_entry_dequeue(&rt2880_free_buf_list); 
		NetTxPacket = buf->pbuf;
		
		for (i = 0; i < NUM_RX_DESC; i++)
		{
			buf = rt2880_free_buf_entry_dequeue(&rt2880_free_buf_list); 
			if (buf == NULL)
			{
				debug("%s: Packet Buffer is empty !\n",__func__);
				return -1;
			}
			NetRxPackets[i] = buf->pbuf;
			debug("%s: NetRxPackets[%d] = 0x%08x\n",__func__,i,NetRxPackets[i]);
		}
	}

	debug("%s: NetTxPacket = 0x%08x\n",__func__,NetTxPacket);
	NetTxPacket = KSEG1ADDR(NetTxPacket);
	debug("%s: KSEG1ADDR(NetTxPacket) = 0x%08x\n",__func__,NetTxPacket);

	if (!NetArpWaitTxPacket)
	{
		NetArpWaitTxPacket = &NetArpWaitPacketBuf[0] + (PKTALIGN - 1);
		NetArpWaitTxPacket -= (ulong)NetArpWaitTxPacket % PKTALIGN;
		NetArpWaitTxPacketSize = 0;
	}

	eth_halt();
#ifdef CONFIG_NET_MULTI
	eth_set_current();
#endif
	if (eth_init(bd) < 0)
	{
		debug("%s: eth_init is fail !!\n",__func__);
		return -1;
	}	

restart:
#ifdef CONFIG_NET_MULTI
	memcpy(NetOurEther, eth_get_dev()->enetaddr, 6);
#else
	memcpy(NetOurEther, bd->bi_enetaddr, 6);
#endif

	debug("%s: NetOurEther: %02X:%02X:%02X:%02X:%02X:%02X\n",__func__,
		NetOurEther[0], NetOurEther[1], NetOurEther[2],
		NetOurEther[3], NetOurEther[4], NetOurEther[5]);

	NetState = NETLOOP_CONTINUE;

	/*
	 *	Start the ball rolling with the given start function.  From
	 *	here on, this code is a state machine driven by received
	 *	packets and timer events.
	 */

	switch (protocol)
	{
#if (CONFIG_COMMANDS & CFG_CMD_NFS)
	case NFS:
#endif
#if (CONFIG_COMMANDS & CFG_CMD_PING)
	case PING:
#endif
	case NETCONS:
	case TFTP:
#ifdef HTTPSVR_SUPPORT
	case HTTP:
#endif
		NetCopyIP(&NetOurIP, &bd->bi_ip_addr);
		NetOurGatewayIP = getenv_IPaddr ("gatewayip");
		NetOurSubnetMask= getenv_IPaddr ("netmask");
		NetOurVLAN = getenv_VLAN("vlan");
		NetOurNativeVLAN = getenv_VLAN("nvlan");

		switch (protocol)
		{
#if (CONFIG_COMMANDS & CFG_CMD_NFS)
		case NFS:
#endif
		case NETCONS:
		case TFTP:
			NetServerIP = getenv_IPaddr("serverip");
			break;
#if (CONFIG_COMMANDS & CFG_CMD_PING)
		case PING:
			/* nothing */
			break;
#endif
		default:
			break;
		}
		break;

	case BOOTP:
	case RARP:
		/*
		 * initialize our IP addr to 0 in order to accept ANY
		 * IP addr assigned to us by the BOOTP / RARP server
		 */
		NetOurIP = 0;
		NetServerIP = getenv_IPaddr ("serverip");
 		NetOurVLAN = getenv_VLAN("vlan");	/* VLANs must be read */
 		NetOurNativeVLAN = getenv_VLAN("nvlan");
 	case CDP:
 		NetOurVLAN = getenv_VLAN("vlan");	/* VLANs must be read */
 		NetOurNativeVLAN = getenv_VLAN("nvlan");
		break;
	default:
		break;
	}

	switch (net_check_prereq(protocol))
	{
	case 1:
		/* network not configured */
		return (-1);

#ifdef CONFIG_NET_MULTI
	case 2:
		/* network device not configured */
		break;
#endif /* CONFIG_NET_MULTI */

	case 0:
#ifdef CONFIG_NET_MULTI
		NetDevExists = 1;
#endif
		switch (protocol)
		{
		case TFTP:
			/* always use ARP to get server ethernet address */
			TftpStart();
			break;

#ifdef HTTPSVR_SUPPORT
		case HTTP:
			HttpStart();
			break;
#endif

#if (CONFIG_COMMANDS & CFG_CMD_DHCP)
		case DHCP:
			/* Start with a clean slate... */
			BootpTry = 0;
			NetOurIP = 0;
			NetServerIP = getenv_IPaddr ("serverip");
			DhcpRequest();		/* Basically same as BOOTP */
			break;
#endif /* CFG_CMD_DHCP */

#if 0
		case BOOTP:
			BootpTry = 0;
			BootpRequest ();
			break;

		case RARP:
			RarpTry = 0;
			RarpRequest ();
			break;
#endif
#if (CONFIG_COMMANDS & CFG_CMD_PING)
		case PING:
			PingStart();
			break;
#endif
#if (CONFIG_COMMANDS & CFG_CMD_NFS)
		case NFS:
			NfsStart();
			break;
#endif
#if (CONFIG_COMMANDS & CFG_CMD_CDP)
		case CDP:
			CDPStart();
			break;
#endif
#ifdef CONFIG_NETCONSOLE
		case NETCONS:
			NcStart();
			break;
#endif
		default:
			break;
		}

		NetBootFileXferSize = 0;
		break;
	}

#if defined(CONFIG_MII) || (CONFIG_COMMANDS & CFG_CMD_MII)
#error "ONFIG_MII"
#if defined(CFG_FAULT_ECHO_LINK_DOWN) && defined(CONFIG_STATUS_LED) && defined(STATUS_LED_RED)
	/* Echo the inverted link state to the fault LED. */
	if (miiphy_link(CFG_FAULT_MII_ADDR))
	{
		status_led_set(STATUS_LED_RED, STATUS_LED_OFF);
	}
	else
	{
		status_led_set(STATUS_LED_RED, STATUS_LED_ON);
	}
#endif /* CFG_FAULT_ECHO_LINK_DOWN, ... */
#endif /* CONFIG_MII, ... */

	/* Main packet reception loop.  Loop receiving packets until
	 * someone sets `NetQuit'. */
	for (;;)
	{
		WATCHDOG_RESET();
#ifdef CONFIG_SHOW_ACTIVITY
		{
			extern void show_activity(int arg);
			show_activity(1);
		}
#endif

//		extern void toggle_activity(void);
//		toggle_activity();

		/* Check the ethernet for a new packet.  The ethernet
		 * receive routine will process it. */
		eth_rx();

		/* Abort if ctrl-c was pressed. */
		if (ctrlc())
		{
			eth_halt();
			puts ("\nAbort\n");
			return (-1);
		}

		ArpTimeoutCheck();

		/* Check for a timeout, and run the timeout handler
		 * if we have one. */
		if (timeHandler && ((get_timer(0) - timeStart) > timeDelta))
		{
			thand_f *x;

#if defined(CONFIG_MII) || (CONFIG_COMMANDS & CFG_CMD_MII)
#if defined(CFG_FAULT_ECHO_LINK_DOWN) && defined(CONFIG_STATUS_LED) && defined(STATUS_LED_RED)
			/* Echo the inverted link state to the fault LED. */
			if (miiphy_link(CFG_FAULT_MII_ADDR))
			{
				status_led_set(STATUS_LED_RED, STATUS_LED_OFF);
			}
			else
			{
				status_led_set(STATUS_LED_RED, STATUS_LED_ON);
			}
#endif /* CFG_FAULT_ECHO_LINK_DOWN, ... */
#endif /* CONFIG_MII, ... */
			x = timeHandler;
			timeHandler = (thand_f *)0;
			(*x)();
		}

		switch (NetState)
		{
		case NETLOOP_RESTART:
#ifdef CONFIG_NET_MULTI
			NetRestarted = 1;
#endif
			goto restart;

		case NETLOOP_SUCCESS:
			if (NetBootFileXferSize > 0)
			{
				char buf[10];
				printf("Bytes transferred = %ld (%lx hex)\n",
					NetBootFileXferSize, NetBootFileXferSize);
				sprintf(buf, "%lx", NetBootFileXferSize);
				setenv("filesize", buf);
				sprintf(buf, "%lX", (unsigned long)load_addr);
				setenv("fileaddr", buf);
			}
			eth_halt();
			return NetBootFileXferSize;

		case NETLOOP_FAIL:
			return (-1);
		}
	}
}

/**********************************************************************/

static void
startAgainTimeout(void)
{
	NetState = NETLOOP_RESTART;
}

static void
startAgainHandler(uchar * pkt, unsigned dest, unsigned src, unsigned len)
{
	/* Totally ignore the packet */
}

void NetStartAgain (void)
{
#ifdef	CONFIG_NET_MULTI
	DECLARE_GLOBAL_DATA_PTR;
#endif
	char *nretry;
	int noretry = 0, once = 0;

	if ((nretry = getenv ("netretry")) != NULL) {
		noretry = (strcmp (nretry, "no") == 0);
		once = (strcmp (nretry, "once") == 0);
	}
	if (noretry) {
		eth_halt ();
		NetState = NETLOOP_FAIL;
		return;
	}
#ifndef CONFIG_NET_MULTI
	NetSetTimeout (10 * CFG_HZ, startAgainTimeout);
	NetSetHandler (startAgainHandler);
#else	/* !CONFIG_NET_MULTI*/
	eth_halt ();
	eth_try_another (!NetRestarted);
	eth_init (gd->bd);
	if (NetRestartWrap) {
		NetRestartWrap = 0;
		if (NetDevExists && !once) {
			NetSetTimeout (10UL * CFG_HZ, startAgainTimeout);
			NetSetHandler (startAgainHandler);
		} else {
			NetState = NETLOOP_FAIL;
		}
	} else {
		NetState = NETLOOP_RESTART;
	}
#endif	/* CONFIG_NET_MULTI */
}

/**********************************************************************/
/*
 *	Miscelaneous bits.
 */

void
NetSetHandler(rxhand_f * f)
{
	packetHandler = f;
}

void
NetSetTCPHandler(rxhand_f * f)
{
	TcpPacketHandler = f;
}


void
NetSetTimeout(ulong iv, thand_f * f)
{
	if (iv == 0) {
		timeHandler = (thand_f *)0;
	} else {
		timeHandler = f;
		timeStart = get_timer(0);
		timeDelta = iv;
	}
}


void
NetSendPacket(volatile uchar * pkt, int len)
{
	(void) eth_send(pkt, len);
}

#if 0
static uchar *	cache_ether;
static IPaddr_t	cache_dest;
#endif

int NetSendUDPPacket(uchar *ether, IPaddr_t dest, int dport, int sport, int len)
{
	uchar *pkt;

	/* convert to new style broadcast */
	if (dest == 0) dest = 0xFFFFFFFF;

	/* if broadcast, make the ether address a broadcast and don't do ARP */
	if (dest == 0xFFFFFFFF) ether = NetBcastAddr;

	/* if MAC address was not discovered yet, save the packet and do an ARP request */
	if (memcmp(ether, NetEtherNullAddr, 6) == 0)
	{
		etdebug("%s: sending ARP for 0x%08x\n",__func__, dest);

		NetArpWaitPacketIP = dest;
		NetArpWaitPacketMAC = ether;

		pkt = NetArpWaitTxPacket;
		pkt += NetSetEther(pkt, NetArpWaitPacketMAC, PROT_IP);

		NetSetIP (pkt, dest, dport, sport, len);
		memcpy(pkt + IP_HDR_SIZE, (uchar *)NetTxPacket + (pkt - (uchar *)NetArpWaitTxPacket) + IP_HDR_SIZE, len);

		/* size of the waiting packet */
		NetArpWaitTxPacketSize = (pkt - NetArpWaitTxPacket) + IP_HDR_SIZE + len;

		/* and do the ARP request */
		NetArpWaitTry = 1;
		NetArpWaitTimerStart = get_timer(0);
		ArpRequest();
		return 1;	/* waiting */
	}

#if 0
	etdebug("%s: sending UDP to %08lx/%02x:%02x:%02x:%02x:%02x:%02x\n",__func__,
		dest, ether[0], ether[1], ether[2], ether[3], ether[4], ether[5]);
#endif

	pkt = (uchar *)NetTxPacket;
	pkt += NetSetEther (pkt, ether, PROT_IP);
	NetSetIP (pkt, dest, dport, sport, len);
	(void) eth_send(NetTxPacket, (pkt - NetTxPacket) + IP_HDR_SIZE + len);

	return 0;	/* transmitted */
}

#if (CONFIG_COMMANDS & CFG_CMD_PING)
static ushort PingSeqNo;

int PingSend(void)
{
	static uchar mac[6];
	volatile IP_t *ip;
	volatile ushort *s;
	uchar *pkt;

	/* XXX always send arp request */

	memcpy(mac, NetEtherNullAddr, 6);

#ifdef ET_DEBUG
	printf("sending ARP for %08lx\n", NetPingIP);
#endif
	printf("kaiker,PingSend, sending ARP for %08lx\n", NetPingIP);
	NetArpWaitPacketIP = NetPingIP;
	NetArpWaitPacketMAC = mac;

	pkt = NetArpWaitTxPacket;
	pkt += NetSetEther(pkt, mac, PROT_IP);

	ip = (volatile IP_t *)pkt;

	/*
	 *	Construct an IP and ICMP header.  (need to set no fragment bit - XXX)
	 */
	ip->ip_hl_v  = 0x45;		/* IP_HDR_SIZE / 4 (not including UDP) */
	ip->ip_tos   = 0;
	ip->ip_len   = htons(IP_HDR_SIZE_NO_UDP + 8);
	ip->ip_id    = htons(NetIPID++);
	ip->ip_off   = htons(0x4000);	/* No fragmentation */
	ip->ip_ttl   = 255;
	ip->ip_p     = 0x01;		/* ICMP */
	ip->ip_sum   = 0;
	NetCopyIP((void*)&ip->ip_src, &NetOurIP); /* already in network byte order */
	NetCopyIP((void*)&ip->ip_dst, &NetPingIP);	   /* - "" - */
	ip->ip_sum   = ~NetCksum((uchar *)ip, IP_HDR_SIZE_NO_UDP / 2);

	s = &ip->udp_src;		/* XXX ICMP starts here */
	s[0] = htons(0x0800);		/* echo-request, code */
	s[1] = 0;			/* checksum */
	s[2] = 0; 			/* identifier */
	s[3] = htons(PingSeqNo++);	/* sequence number */
	s[1] = ~NetCksum((uchar *)s, 8/2);

	/* size of the waiting packet */
	NetArpWaitTxPacketSize = (pkt - NetArpWaitTxPacket) + IP_HDR_SIZE_NO_UDP + 8;

	/* and do the ARP request */
	NetArpWaitTry = 1;
	NetArpWaitTimerStart = get_timer(0);
	ArpRequest();
	return 1;	/* waiting */
}

static void
PingTimeout (void)
{
	eth_halt();
	NetState = NETLOOP_FAIL;	/* we did not get the reply */
}

static void
PingHandler (uchar * pkt, unsigned dest, unsigned src, unsigned len)
{
	IPaddr_t tmp;
	volatile IP_t *ip = (volatile IP_t *)pkt;

	tmp = NetReadIP((void *)&ip->ip_src);
	if (tmp != NetPingIP)
		return;

	NetState = NETLOOP_SUCCESS;
}

static void PingStart(void)
{
#if defined(CONFIG_NET_MULTI)
	printf ("Using %s device\n", eth_get_name());
#endif	/* CONFIG_NET_MULTI */
	NetSetTimeout (10 * CFG_HZ, PingTimeout);
	NetSetHandler (PingHandler);

	PingSend();
}
#endif	/* CFG_CMD_PING */

#if (CONFIG_COMMANDS & CFG_CMD_CDP)

#define CDP_DEVICE_ID_TLV		0x0001
#define CDP_ADDRESS_TLV			0x0002
#define CDP_PORT_ID_TLV			0x0003
#define CDP_CAPABILITIES_TLV		0x0004
#define CDP_VERSION_TLV			0x0005
#define CDP_PLATFORM_TLV		0x0006
#define CDP_NATIVE_VLAN_TLV		0x000a
#define CDP_APPLIANCE_VLAN_TLV		0x000e
#define CDP_TRIGGER_TLV			0x000f
#define CDP_POWER_CONSUMPTION_TLV	0x0010
#define CDP_SYSNAME_TLV			0x0014
#define CDP_SYSOBJECT_TLV		0x0015
#define CDP_MANAGEMENT_ADDRESS_TLV	0x0016

#define CDP_TIMEOUT			(CFG_HZ/4)	/* one packet every 250ms */

static int CDPSeq;
static int CDPOK;

ushort CDPNativeVLAN;
ushort CDPApplianceVLAN;

static const uchar CDP_SNAP_hdr[8] = { 0xAA, 0xAA, 0x03, 0x00, 0x00, 0x0C, 0x20, 0x00 };

static ushort CDP_compute_csum(const uchar *buff, ushort len)
{
	ushort csum;
	int     odd;
	ulong   result = 0;
	ushort  leftover;

	if (len > 0) {
		odd = 1 & (ulong)buff;
		if (odd) {
			result = *buff << 8;
			len--;
			buff++;
		}
		while (len > 1) {
			result += *((const ushort *)buff)++;
			if (result & 0x80000000)
				result = (result & 0xFFFF) + (result >> 16);
			len -= 2;
		}
		if (len) {
			leftover = (signed short)(*(const signed char *)buff);
			/* * XXX CISCO SUCKS big time! (and blows too) */
			result = (result & 0xffff0000) | ((result + leftover) & 0x0000ffff);
		}
		while (result >> 16)
			result = (result & 0xFFFF) + (result >> 16);

		if (odd)
			result = ((result >> 8) & 0xff) | ((result & 0xff) << 8);
	}

	/* add up 16-bit and 17-bit words for 17+c bits */
	result = (result & 0xffff) + (result >> 16);
	/* add up 16-bit and 2-bit for 16+c bit */
	result = (result & 0xffff) + (result >> 16);
	/* add up carry.. */
	result = (result & 0xffff) + (result >> 16);

	/* negate */
	csum = ~(ushort)result;

	/* run time endian detection */
	if (csum != htons(csum))	/* little endian */
		csum = htons(csum);

	return csum;
}

int CDPSendTrigger(void)
{
	volatile uchar *pkt;
	volatile ushort *s;
	volatile ushort *cp;
	Ethernet_t *et;
	int len;
	ushort chksum;
#if defined(CONFIG_CDP_DEVICE_ID) || defined(CONFIG_CDP_PORT_ID)   || \
    defined(CONFIG_CDP_VERSION)   || defined(CONFIG_CDP_PLATFORM)
	char buf[32];
#endif

	pkt = NetTxPacket;
	et = (Ethernet_t *)pkt;

	/* NOTE: trigger sent not on any VLAN */

	/* form ethernet header */
	memcpy(et->et_dest, NetCDPAddr, 6);
	memcpy(et->et_src, NetOurEther, 6);

	pkt += ETHER_HDR_SIZE;

	/* SNAP header */
	memcpy((uchar *)pkt, CDP_SNAP_hdr, sizeof(CDP_SNAP_hdr));
	pkt += sizeof(CDP_SNAP_hdr);

	/* CDP header */
	*pkt++ = 0x02;				/* CDP version 2 */
	*pkt++ = 180;				/* TTL */
	s = (volatile ushort *)pkt;
	cp = s;
	*s++ = htons(0);			/* checksum (0 for later calculation) */

	/* CDP fields */
#ifdef CONFIG_CDP_DEVICE_ID
	*s++ = htons(CDP_DEVICE_ID_TLV);
	*s++ = htons(CONFIG_CDP_DEVICE_ID);
	memset(buf, 0, sizeof(buf));
	sprintf(buf, CONFIG_CDP_DEVICE_ID_PREFIX "%02X%02X%02X%02X%02X%02X",
		NetOurEther[0] & 0xff, NetOurEther[1] & 0xff,
		NetOurEther[2] & 0xff, NetOurEther[3] & 0xff,
		NetOurEther[4] & 0xff, NetOurEther[5] & 0xff);
	memcpy((uchar *)s, buf, 16);
	s += 16 / 2;
#endif

#ifdef CONFIG_CDP_PORT_ID
	*s++ = htons(CDP_PORT_ID_TLV);
	memset(buf, 0, sizeof(buf));
	sprintf(buf, CONFIG_CDP_PORT_ID, eth_get_dev_index());
	len = strlen(buf);
	if (len & 1)	/* make it even */
		len++;
	*s++ = htons(len + 4);
	memcpy((uchar *)s, buf, len);
	s += len / 2;
#endif

#ifdef CONFIG_CDP_CAPABILITIES
	*s++ = htons(CDP_CAPABILITIES_TLV);
	*s++ = htons(8);
	*(ulong *)s = htonl(CONFIG_CDP_CAPABILITIES);
	s += 2;
#endif

#ifdef CONFIG_CDP_VERSION
	*s++ = htons(CDP_VERSION_TLV);
	memset(buf, 0, sizeof(buf));
	strcpy(buf, CONFIG_CDP_VERSION);
	len = strlen(buf);
	if (len & 1)	/* make it even */
		len++;
	*s++ = htons(len + 4);
	memcpy((uchar *)s, buf, len);
	s += len / 2;
#endif

#ifdef CONFIG_CDP_PLATFORM
	*s++ = htons(CDP_PLATFORM_TLV);
	memset(buf, 0, sizeof(buf));
	strcpy(buf, CONFIG_CDP_PLATFORM);
	len = strlen(buf);
	if (len & 1)	/* make it even */
		len++;
	*s++ = htons(len + 4);
	memcpy((uchar *)s, buf, len);
	s += len / 2;
#endif

#ifdef CONFIG_CDP_TRIGGER
	*s++ = htons(CDP_TRIGGER_TLV);
	*s++ = htons(8);
	*(ulong *)s = htonl(CONFIG_CDP_TRIGGER);
	s += 2;
#endif

#ifdef CONFIG_CDP_POWER_CONSUMPTION
	*s++ = htons(CDP_POWER_CONSUMPTION_TLV);
	*s++ = htons(6);
	*s++ = htons(CONFIG_CDP_POWER_CONSUMPTION);
#endif

	/* length of ethernet packet */
	len = (uchar *)s - ((uchar *)NetTxPacket + ETHER_HDR_SIZE);
	et->et_protlen = htons(len);

	len = ETHER_HDR_SIZE + sizeof(CDP_SNAP_hdr);
	chksum = CDP_compute_csum((uchar *)NetTxPacket + len, (uchar *)s - (NetTxPacket + len));
	if (chksum == 0)
		chksum = 0xFFFF;
	*cp = htons(chksum);

	(void) eth_send(NetTxPacket, (uchar *)s - NetTxPacket);
	return 0;
}

static void
CDPTimeout (void)
{
	CDPSeq++;

	if (CDPSeq < 3) {
		NetSetTimeout (CDP_TIMEOUT, CDPTimeout);
		CDPSendTrigger();
		return;
	}

	/* if not OK try again */
	if (!CDPOK)
		NetStartAgain();
	else
		NetState = NETLOOP_SUCCESS;
}

static void
CDPDummyHandler (uchar * pkt, unsigned dest, unsigned src, unsigned len)
{
	/* nothing */
}

static void
CDPHandler(const uchar * pkt, unsigned len)
{
	const uchar *t;
	const ushort *ss;
	ushort type, tlen;
	uchar applid;
	ushort vlan, nvlan;

	/* minimum size? */
	if (len < sizeof(CDP_SNAP_hdr) + 4)
		goto pkt_short;

	/* check for valid CDP SNAP header */
	if (memcmp(pkt, CDP_SNAP_hdr, sizeof(CDP_SNAP_hdr)) != 0)
		return;

	pkt += sizeof(CDP_SNAP_hdr);
	len -= sizeof(CDP_SNAP_hdr);

	/* Version of CDP protocol must be >= 2 and TTL != 0 */
	if (pkt[0] < 0x02 || pkt[1] == 0)
		return;

	/* if version is greater than 0x02 maybe we'll have a problem; output a warning */
	if (pkt[0] != 0x02)
		printf("** WARNING: CDP packet received with a protocol version %d > 2\n",
				pkt[0] & 0xff);

	if (CDP_compute_csum(pkt, len) != 0)
		return;

	pkt += 4;
	len -= 4;

	vlan = htons(-1);
	nvlan = htons(-1);
	while (len > 0) {
		if (len < 4)
			goto pkt_short;

		ss = (const ushort *)pkt;
		type = ntohs(ss[0]);
		tlen = ntohs(ss[1]);
		if (tlen > len) {
			goto pkt_short;
		}

		pkt += tlen;
		len -= tlen;

		ss += 2;	/* point ss to the data of the TLV */
		tlen -= 4;

		switch (type) {
			case CDP_DEVICE_ID_TLV:
				break;
			case CDP_ADDRESS_TLV:
				break;
			case CDP_PORT_ID_TLV:
				break;
			case CDP_CAPABILITIES_TLV:
				break;
			case CDP_VERSION_TLV:
				break;
			case CDP_PLATFORM_TLV:
				break;
			case CDP_NATIVE_VLAN_TLV:
				nvlan = *ss;
				break;
			case CDP_APPLIANCE_VLAN_TLV:
				t = (const uchar *)ss;
				while (tlen > 0) {
					if (tlen < 3)
						goto pkt_short;

					applid = t[0];
					ss = (const ushort *)(t + 1);

#ifdef CONFIG_CDP_APPLIANCE_VLAN_TYPE
					if (applid == CONFIG_CDP_APPLIANCE_VLAN_TYPE)
						vlan = *ss;
#else
					vlan = ntohs(*ss);	/* XXX will this work; dunno */
#endif
					t += 3; tlen -= 3;
				}
				break;
			case CDP_TRIGGER_TLV:
				break;
			case CDP_POWER_CONSUMPTION_TLV:
				break;
			case CDP_SYSNAME_TLV:
				break;
			case CDP_SYSOBJECT_TLV:
				break;
			case CDP_MANAGEMENT_ADDRESS_TLV:
				break;
		}
	}

	CDPApplianceVLAN = vlan;
	CDPNativeVLAN = nvlan;

	CDPOK = 1;
	return;

 pkt_short:
	printf("** CDP packet is too short\n");
	return;
}

static void CDPStart(void)
{
#if defined(CONFIG_NET_MULTI)
	printf ("Using %s device\n", eth_get_name());
#endif
	CDPSeq = 0;
	CDPOK = 0;

	CDPNativeVLAN = htons(-1);
	CDPApplianceVLAN = htons(-1);

	NetSetTimeout (CDP_TIMEOUT, CDPTimeout);
	NetSetHandler (CDPDummyHandler);

	CDPSendTrigger();
}
#endif	/* CFG_CMD_CDP */

#ifdef HTTPSVR_SUPPORT

static void send_tcp_packet(void);

/* Connection info */
static tcpconn_t tcpconn;

void NetInitTcp(ushort port)
{
	memset(&tcpconn, 0, sizeof(tcpconn));
	tcpconn.state = TCP_LISTENING;
	tcpconn.myport = port;
	debug("%s: port = %d\n",__func__, ntohs(tcpconn.myport));
}

/* saddr    : source IP in network byte order
 * daddr    : dest IP in network byte order.
 * protocol : protocol number
 * tcphdr   : pointer to TCP header.
 * tcplen   : total length of TCP payload, including header & data. */
ushort NetTcpCksum(ulong saddr, ulong daddr, uchar protocol, uchar * tcphdr, ushort tcplen)
{
	tcpphdr_t phdr;
	ulong cksum;

	/* fill up the pseudo header */
	phdr.saddr = saddr;
	phdr.daddr = daddr;
	phdr.reserved = 0;
	phdr.protocol = protocol;
	phdr.length = htons(tcplen);
	cksum = NetCksum((uchar *)&phdr, sizeof(phdr)/2);
	cksum += NetCksum((uchar *)tcphdr, tcplen/2);
	if (tcplen & 1) cksum += (ushort)tcphdr[tcplen-1];
	cksum = (cksum & 0xffff) + (cksum >> 16);
	return (cksum & 0xffff);
}

int NetTcpSend(uchar * buf, int size)
{
	if (tcpconn.bufsize + size > sizeof(tcpconn.buffer))
	{
		debug("%s: not buffer to send (bufsize=%d)!\n",__func__, tcpconn.bufsize);
		return -1;
	}

	etdebug("%s: size = %d\n",__func__,size);
	if (buf && size > 0) memcpy(&tcpconn.buffer[tcpconn.bufsize], buf, size);
	tcpconn.bufsize += size;
	send_tcp_packet();
	return 0;
}

static void send_syn_ack(void)
{
	volatile uchar * pkt;
	volatile iphdr_t * iphdr;
	volatile tcphdr_t * tcphdr;
	ulong seq;
	ushort cksum;

	/* 8 bytes of TCP option for MSS (1460 bytes) */
	static uchar options[] = { 0x02,0x04,0x05,0xb4,0x01,0x01,0x04,0x02 };

	pkt = NetTxPacket;
	pkt += NetSetEther(pkt, tcpconn.yourmac, PROT_IP);
	iphdr = (iphdr_t *)pkt;
	tcphdr = (tcphdr_t *)(pkt + sizeof(iphdr_t));
	pkt += (sizeof(iphdr_t) + sizeof(tcphdr_t));

	/* fill up TCP header */
	tcphdr->source	= tcpconn.myport;
	tcphdr->dest	= tcpconn.yourport;
	seq = htonl(tcpconn.myseq);
	NetCopyLong((void *)&tcphdr->seq, &seq);
	seq = htonl(tcpconn.yourseq);
	tcpconn.hasack = tcpconn.yourseq;
	NetCopyLong((void *)&tcphdr->ack_seq, &seq);
	tcphdr->doff	= (sizeof(tcphdr_t)+sizeof(options)) * 4;
	tcphdr->flags	= TCPFLAG_SYN | TCPFLAG_ACK;
	tcphdr->window	= htons(2048);
	tcphdr->check	= 0;
	/* fill up TCP option */
	memcpy((void *)pkt, options, sizeof(options));
	pkt += sizeof(options);
	/* fill up IP header */
	iphdr->ihl		= 0x45;
	iphdr->tos		= 0;
	iphdr->tot_len	= htons(sizeof(iphdr_t) + sizeof(tcphdr_t) + sizeof(options));
	iphdr->id		= htons(NetIPID++);
	iphdr->frag_off	= htons(0x4000);	// No fragmentation
	iphdr->ttl		= 64;
	iphdr->protocol	= 0x06;
	iphdr->check	= 0;
	NetCopyIP((uchar *)&iphdr->saddr, &NetOurIP);
	NetCopyIP((uchar *)&iphdr->daddr, &tcpconn.youraddr);
	cksum = ~NetCksum((uchar *)iphdr, sizeof(iphdr_t)/2);
	iphdr->check = cksum;
	debug("%s: iphdr->check = 0x%04x\n",__func__,cksum);
	cksum = ~NetTcpCksum(NetOurIP, tcpconn.youraddr, 0x06, (uchar *)tcphdr, sizeof(tcphdr_t)+sizeof(options));
	tcphdr->check = cksum;
	debug("%s: tcphdr->check = 0x%04x\n",__func__,cksum);

	eth_send(NetTxPacket, pkt - NetTxPacket);
	tcpconn.state = TCP_SYN_SENT;
}

static void send_reset(void)
{
	volatile uchar * pkt;
	volatile iphdr_t * iphdr;
	volatile tcphdr_t * tcphdr;
	ulong seq;
	ushort cksum;

	pkt = NetTxPacket;
	pkt += NetSetEther(pkt, tcpconn.yourmac, PROT_IP);
	iphdr = (iphdr_t *)pkt;
	tcphdr = (tcphdr_t *)(pkt + sizeof(iphdr_t));
	pkt += (sizeof(iphdr_t) + sizeof(tcphdr_t));

	/* fill the TCP header */
	tcphdr->source	= tcpconn.myport;
	tcphdr->dest	= tcpconn.yourport;
	seq = htonl(tcpconn.myseq);
	NetCopyLong((void *)&tcphdr->seq, &seq);
	seq = htonl(tcpconn.yourseq);
	tcpconn.hasack = tcpconn.yourseq;
	NetCopyLong((void *)&tcphdr->ack_seq, &seq);
	tcphdr->doff	= sizeof(tcphdr) * 4;
	tcphdr->flags	= TCPFLAG_RST | TCPFLAG_ACK;
	tcphdr->window	= htons(2048);
	tcphdr->check	= 0;
	/* fill the IP header */
	iphdr->ihl		= 0x45;
	iphdr->tos		= 0;
	iphdr->tot_len	= htons(sizeof(iphdr_t) + sizeof(tcphdr_t));
	iphdr->id		= htons(NetIPID++);
	iphdr->frag_off	= htons(0x4000);
	iphdr->ttl		= 64;
	iphdr->protocol	= 0x06;
	iphdr->check	= 0;
	NetCopyIP((uchar *)&iphdr->saddr, &NetOurIP);
	NetCopyIP((uchar *)&iphdr->daddr, &tcpconn.youraddr);
	/* Calculate ckecksum */
	cksum = ~NetCksum((uchar *)iphdr, sizeof(iphdr_t)/2);
	iphdr->check = cksum;
	debug("%s: iphdr->check = 0x%04x\n",__func__,cksum);
	cksum = ~NetTcpCksum(NetOurIP, tcpconn.youraddr, 0x6, (uchar *)tcphdr, sizeof(tcphdr_t));
	tcphdr->check = cksum;
	debug("%s: tcphdr->check = 0x%04x\n",__func__,cksum);

	eth_send(NetTxPacket, pkt - NetTxPacket);
	NetInitTcp(tcpconn.myport);
}

static void send_tcp_packet(void)
{
	volatile uchar * pkt;
	volatile iphdr_t * iphdr;
	volatile tcphdr_t * tcphdr;
	int dlen = 0;
	ulong seq;
	ushort cksum;

	pkt = NetTxPacket;
	pkt += NetSetEther(pkt, tcpconn.yourmac, PROT_IP);
	iphdr = (iphdr_t *)pkt;
	tcphdr = (tcphdr_t *)(pkt + sizeof(iphdr_t));
	pkt += (sizeof(iphdr_t) + sizeof(tcphdr_t));

	/* fill the TCP header */
	tcphdr->source	= tcpconn.myport;
	tcphdr->dest	= tcpconn.yourport;
	seq = htonl(tcpconn.myseq);
	NetCopyLong((void *)&tcphdr->seq, &seq);
	seq = htonl(tcpconn.yourseq);
	tcpconn.hasack = tcpconn.yourseq;
	NetCopyLong((void *)&tcphdr->ack_seq, &seq);
	tcphdr->doff	= sizeof(tcphdr_t) * 4;
	tcphdr->flags	= TCPFLAG_ACK | ((tcpconn.bufsize>0) ? TCPFLAG_PSH : 0);
	tcphdr->window	= htons(1460*3);
	tcphdr->check	= 0;
	/* Ack is sending. */
	tcpconn.ack		= 0;
	/* copy data */
	dlen = (tcpconn.bufsize > 1460) ? 1460 : tcpconn.bufsize;
	memcpy((uchar *)pkt, tcpconn.buffer, dlen);
	pkt += dlen;
	/* fill the IP header */
	iphdr->ihl		= 0x45;
	iphdr->tos		= 0;
	iphdr->tot_len	= htons(sizeof(iphdr_t) + sizeof(tcphdr_t) + dlen);
	iphdr->id		= htons(NetIPID++);
	iphdr->frag_off	= htons(0x4000);
	iphdr->ttl		= 64;
	iphdr->protocol	= 0x06;
	iphdr->check	= 0;
	/* Calculate checksum */
	NetCopyIP((uchar *)&iphdr->saddr, &tcpconn.myaddr);
	NetCopyIP((uchar *)&iphdr->daddr, &tcpconn.youraddr);
	cksum = ~NetCksum((uchar *)iphdr, sizeof(iphdr_t)/2);
	iphdr->check = cksum;
	//debug("%s: iphdr->check = 0x%04x\n",__func__,cksum);
	cksum = ~NetTcpCksum(tcpconn.myaddr, tcpconn.youraddr, 0x06, (uchar *)tcphdr, sizeof(tcphdr_t) + dlen);
	tcphdr->check = cksum;
	//debug("%s: tcphdr->check = 0x%04x\n",__func__,cksum);
	etdebug("%s: dlen = %d, total size = %d\n",__func__,dlen, pkt-NetTxPacket);

	eth_send(NetTxPacket, pkt - NetTxPacket);
}

static void TCPTimeout(void)
{
	if (tcpconn.ack)
	{
		etdebug("%s: send ack!\n",__func__);
		send_tcp_packet();
	}
}

extern unsigned long mips_cpu_feq;

static void HandleTCP(uchar * mac, uchar * pkt, int len)
{
	iphdr_t * iphdr;
	tcphdr_t * tcphdr;
	int iplen, tcplen, dlen;
	ushort cksum;
	ulong tmp32, i, l;
	ulong ticks, delta;

	if (len < sizeof(iphdr_t)+sizeof(tcphdr_t))
	{
		debug("%s: len < %d, packet size too small!\n",__func__,sizeof(iphdr_t)+sizeof(tcphdr_t));
		return;
	}

	iphdr	= (iphdr_t *)pkt;
	iplen	= (iphdr->ihl & 0xf) * 4;
	tcphdr	= (tcphdr_t *)((ulong)pkt+iplen);
	tcplen	= (tcphdr->doff & 0xf0) >> 2;
	dlen	= ntohs(iphdr->tot_len) - iplen - tcplen;

	etdebug("%s: len=%d, iplen=%d, tcplen=%d, dlen=%d, source=%d, dest=%d\n",__func__,
		len, iplen, tcplen, dlen, ntohs(tcphdr->source), ntohs(tcphdr->dest));

	cksum = NetTcpCksum(NetReadIP(&iphdr->saddr), NetReadIP(&iphdr->daddr),
						iphdr->protocol, (uchar *)tcphdr, tcplen + dlen);
	if (cksum != 0xffff)
	{
		debug("%s: checksum error (cksum = 0x%04x).\n",__func__,cksum);
		return;
	}

	ticks = (ulong)(get_ticks() & 0xffffffff);
	tmp32 = (ulong)(ticks - tcpconn.lasttick);
	delta = tmp32/((mips_cpu_feq/2)/1000000);
	etdebug("%s: delta = %d\n",__func__,delta);

	if (tcphdr->flags == TCPFLAG_SYN && delta > 3000000)
	{
		/* If the last packet is 10 sec ago, clear the session. */
		debug("%s: Clear old session !!!\n",__func__);
		NetInitTcp(tcpconn.myport);
	}

	switch (tcpconn.state)
	{
	case TCP_LISTENING:
		if (tcphdr->flags != TCPFLAG_SYN) { debug("%s: DEBUG 1!\n",__func__); break; }
		if (tcphdr->dest != tcpconn.myport) { debug("%s: DEBUG2\n",__func__); break; }
		tcpconn.myaddr	= NetOurIP;
		tcpconn.myport	= tcphdr->dest;
		tcpconn.myseq	= get_timer(0);
		tcpconn.youraddr= NetReadIP(&iphdr->saddr);
		tcpconn.yourport= tcphdr->source;
		tcpconn.yourseq = ntohl(NetReadLong(&tcphdr->seq));
		memcpy(&tcpconn.yourmac, mac, 6);
		etdebug("%s: Got SYN from 0x%08x (%02x:%02x:%02x:%02x:%02x:%02x)\n",__func__,tcpconn.youraddr,
			mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

		tcpconn.yourseq++;
		tcpconn.lasttick = (ulong)(get_ticks() & 0xffffffff);
		send_syn_ack();
		break;

	case TCP_SYN_SENT:
		if (tcphdr->flags != TCPFLAG_ACK) break;
		tmp32 = ntohl(NetReadLong(&tcphdr->ack_seq));
		if (tmp32 < tcpconn.myseq+1)
		{
			debug("%s: ackseq : 0x%08x, expected : 0x%08x\n",__func__,tmp32, tcpconn.myseq+1);
			send_reset();
		}
		else
		{
			/* They got my SYN. */
			tcpconn.myseq++;
			tcpconn.state = TCP_ESTABLISHED;
			/* Indicat a new connection. */
			if (TcpPacketHandler) (*TcpPacketHandler)(NULL, ntohs(tcpconn.myport), ntohs(tcpconn.yourport), 0);
			etdebug("%s: Got ACK from 0x%08x\n",__func__,tcpconn.youraddr);
		}
		break;

	case TCP_ESTABLISHED:
		tmp32 = NetReadIP(&iphdr->saddr);
		if (tcpconn.youraddr != NetReadIP(&iphdr->saddr) ||
			tcpconn.yourport != tcphdr->source || tcpconn.myport != tcphdr->dest) break;

		etdebug("%s: Got packet from 0x%08x\n",__func__,tmp32);
		if (tcphdr->flags & TCPFLAG_FIN)
		{
			tcpconn.yourseq++;
			debug("%s: Got FIN !!!!!\n", __func__);
		}
		else if (!(tcphdr->flags & TCPFLAG_SYN))
		{
			tcpconn.yourseq += dlen;
			tcpconn.lasttick = (ulong)(get_ticks() & 0xffffffff);
			tcpconn.ack++;	/* need to send ACK. */
			etdebug("%s: dlen=%d, need to ack to 0x%08x\n",__func__,dlen, tcpconn.yourseq);
		}

		/* Handle ACK, if exist */
		if (tcphdr->flags & TCPFLAG_ACK)
		{
			tmp32 = ntohl(NetReadLong((void *)&tcphdr->ack_seq));
			etdebug("%s: Got ack = 0x%08x, \n", __func__,tmp32);
			if (tmp32 > tcpconn.myseq)
			{
				l = tmp32 - tcpconn.myseq;
				debug("%s: 0x%08x = 0x%08x - 0x%08x\n",__func__,l,tmp32,tcpconn.myseq);
				tcpconn.myseq = tmp32;
				for (i=0; l<tcpconn.bufsize; i++,l++) tcpconn.buffer[i] = tcpconn.buffer[l];
				tcpconn.bufsize = i;
				debug("%s: bufsize = 0x%08x\n",__func__,tcpconn.bufsize);
			}
		}

		if (tcphdr->flags & TCPFLAG_FIN)
		{
			send_tcp_packet();
			tcpconn.state = TCP_FIN_SENT;
		}
		/* Handle the data */
		else if (TcpPacketHandler && dlen > 0)
		{
			(*TcpPacketHandler)(pkt+iplen+tcplen, ntohs(tcpconn.myport), ntohs(tcpconn.yourport), dlen);
			if (tcpconn.yourseq - tcpconn.hasack > 1460*2+100) send_tcp_packet();
		}

		/* Schedule a timer to send ack. */
		//NetSetTimeout(CFG_HZ/4, TCPTimeout);
		NetSetTimeout(CFG_HZ/6, TCPTimeout);
		break;

	case TCP_FIN_SENT:
		/* This should be an ACK. */
		NetInitTcp(tcpconn.myport);
		break;
	}
}
#endif


void NetReceive(volatile uchar * inpkt, int len)
{
	Ethernet_t * et;
	IP_t * ip;
	ARP_t * arp;
	IPaddr_t tmp;
	int	x;
	uchar *pkt;
#if (CONFIG_COMMANDS & CFG_CMD_CDP)
	int iscdp;
#endif
	ushort cti = 0, vlanid = VLAN_NONE, myvlanid, mynvlanid;

	//etdebug("%s: >>>\n",__func__);

	NetRxPkt = inpkt;
	NetRxPktLen = len;
	et = (Ethernet_t *)inpkt;

	/* too small packet? */
	if (len < ETHER_HDR_SIZE)
	{
		etdebug("%s: en[%d] < ETHER_HDR_SIZE\n",__func__,len);
		return;
	}

#if (CONFIG_COMMANDS & CFG_CMD_CDP)
	/* keep track if packet is CDP */
	iscdp = memcmp(et->et_dest, NetCDPAddr, 6) == 0;
#endif

	myvlanid = ntohs(NetOurVLAN);
	if (myvlanid == (ushort)-1) myvlanid = VLAN_NONE;
	mynvlanid = ntohs(NetOurNativeVLAN);
	if (mynvlanid == (ushort)-1) mynvlanid = VLAN_NONE;

	x = ntohs(et->et_protlen);

	if (x < 1514)
	{
		/* Got a 802 packet.  Check the other protocol field. */
		x = ntohs(et->et_prot);
		ip = (IP_t *)(inpkt + E802_HDR_SIZE);
		len -= E802_HDR_SIZE;
	}
	else if (x != PROT_VLAN)
	{	/* normal packet */
		ip = (IP_t *)(inpkt + ETHER_HDR_SIZE);
		len -= ETHER_HDR_SIZE;
	}
	else
	{	/* VLAN packet */
		VLAN_Ethernet_t *vet = (VLAN_Ethernet_t *)et;
		etdebug("%s: VLAN packet received\n",__func__);

		/* too small packet? */
		if (len < VLAN_ETHER_HDR_SIZE) return;

		/* if no VLAN active */
		if ((ntohs(NetOurVLAN) & VLAN_IDMASK) == VLAN_NONE
#if (CONFIG_COMMANDS & CFG_CMD_CDP)
				&& iscdp == 0
#endif
			) return;

		cti = ntohs(vet->vet_tag);
		vlanid = cti & VLAN_IDMASK;
		x = ntohs(vet->vet_type);

		ip = (IP_t *)(inpkt + VLAN_ETHER_HDR_SIZE);
		len -= VLAN_ETHER_HDR_SIZE;
	}

	//etdebug("%s: protocol 0x%x\n",__func__,x);

#if (CONFIG_COMMANDS & CFG_CMD_CDP)
	if (iscdp)
	{
		CDPHandler((uchar *)ip, len);
		return;
	}
#endif

	if ((myvlanid & VLAN_IDMASK) != VLAN_NONE)
	{
		if (vlanid == VLAN_NONE) vlanid = (mynvlanid & VLAN_IDMASK);
		/* not matched? */
		if (vlanid != (myvlanid & VLAN_IDMASK)) return;
	}

	switch (x)
	{
	case PROT_ARP:
		/*
		 * We have to deal with two types of ARP packets:
		 * - REQUEST packets will be answered by sending  our
		 *   IP address - if we know it.
		 * - REPLY packates are expected only after we asked
		 *   for the TFTP server's or the gateway's ethernet
		 *   address; so if we receive such a packet, we set
		 *   the server ethernet address
		 */
		etdebug("%s: Got ARP\n",__func__);

		arp = (ARP_t *)ip;
		if (len < ARP_HDR_SIZE)				{ etdebug("%s: bad length %d < %d\n",__func__,len,ARP_HDR_SIZE); break; }
		if (ntohs(arp->ar_hrd)!=ARP_ETHER)	{ etdebug("%s: ntohs(arp->ar_hrd) != ARP_ETHER\n",__func__); break; }
		if (ntohs(arp->ar_pro)!=PROT_IP)	{ etdebug("%s: ntohs(arp->ar_pro) != PROT_IP\n",__func__); break; }
		if (arp->ar_hln != 6)				{ etdebug("%s: arp->ar_hln != 6\n",__func__); break; }
		if (arp->ar_pln != 4)				{ etdebug("%s: arp->ar_pln != 4\n",__func__); break; }
		if (NetOurIP == 0)					{ etdebug("%s: NetOurIP\n",__func__); break; }
		if (NetReadIP(&arp->ar_data[16]) != NetOurIP) { break; }

		switch (ntohs(arp->ar_op))
		{
		case ARPOP_REQUEST:		/* reply with our IP address	*/
			etdebug("%s: Got ARP REQUEST, return our IP\n",__func__);
			pkt = (uchar *)et;
			pkt += NetSetEther(pkt, et->et_src, PROT_ARP);
			arp->ar_op = htons(ARPOP_REPLY);
			memcpy   (&arp->ar_data[10], &arp->ar_data[0], 6);
			NetCopyIP(&arp->ar_data[16], &arp->ar_data[6]);
			memcpy   (&arp->ar_data[ 0], NetOurEther, 6);
			NetCopyIP(&arp->ar_data[ 6], &NetOurIP);
			(void)eth_send((uchar *)et, (pkt - (uchar *)et) + ARP_HDR_SIZE);
			break;

		case ARPOP_REPLY:		/* arp reply */
			/* are we waiting for a reply */
			if (!NetArpWaitPacketIP || !NetArpWaitPacketMAC) break;
			etdebug("%s: Got ARP REPLY, set server/gtwy eth addr (%02x:%02x:%02x:%02x:%02x:%02x)\n",__func__,
				arp->ar_data[0], arp->ar_data[1], arp->ar_data[2], arp->ar_data[3], arp->ar_data[4], arp->ar_data[5]);
			tmp = NetReadIP(&arp->ar_data[6]);

			/* matched waiting packet's address */
			if (tmp == NetArpWaitReplyIP)
			{
				//etdebug("%s: Got it\n",__func__);
				/* save address for later use */
				memcpy(NetArpWaitPacketMAC, &arp->ar_data[0], 6);

#ifdef CONFIG_NETCONSOLE
				if (packetHadler) (*packetHandler)(0,0,0,0);
#endif
				/* modify header, and transmit it */
				memcpy(((Ethernet_t *)NetArpWaitTxPacket)->et_dest, NetArpWaitPacketMAC, 6);
#if 1
				/* Invalid payload was send out with buffer 'NetArpWaitTxPacket',
				 * Move to 'NetTxPacket' can solve this problem.
				 * by David Hsieh <david_hsieh@alphanetworks.com> */
				memcpy((void *)NetTxPacket, NetArpWaitTxPacket, NetArpWaitTxPacketSize);
				(void)eth_send(NetTxPacket, NetArpWaitTxPacketSize);
#else
				(void) eth_send(NetArpWaitTxPacket, NetArpWaitTxPacketSize);
#endif

				/* no arp request pending now */
				NetArpWaitPacketIP = 0;
				NetArpWaitTxPacketSize = 0;
				NetArpWaitPacketMAC = NULL;
			}
			break;

		default:
			etdebug("%s: Unexpected ARP opcode 0x%x\n", ntohs(arp->ar_op));
			break;
		}
		break;

	case PROT_RARP:
		etdebug("%s: Got RARP\n",__func__);

		arp = (ARP_t *)ip;
		if (len < ARP_HDR_SIZE)	{ debug("%s: bad length %d < %d\n",__func__, len, ARP_HDR_SIZE); break; }
		if ((ntohs(arp->ar_op) != RARPOP_REPLY) ||
			(ntohs(arp->ar_hrd) != ARP_ETHER)   ||
			(ntohs(arp->ar_pro) != PROT_IP)     ||
			(arp->ar_hln != 6) || (arp->ar_pln != 4))
		{
			debug("%s: invalid RARP header\n",__func__);
			break;
		}
		
		NetCopyIP(&NetOurIP, &arp->ar_data[16]);
		if (NetServerIP == 0) NetCopyIP(&NetServerIP, &arp->ar_data[ 6]);
		memcpy (NetServerEther, &arp->ar_data[ 0], 6);
		if (packetHandler) (*packetHandler)(0,0,0,0);
		break;

	case PROT_IP:
		//etdebug("%s: Got IP\n",__func__);
		if (len < IP_HDR_SIZE)			{ debug("%s: len bad %d < %d\n",__func__,len, IP_HDR_SIZE); break; } 
		if (len < ntohs(ip->ip_len))	{ debug("%s: len bad %d < %d\n",__func__, len, ntohs(ip->ip_len)); break; }

		len = ntohs(ip->ip_len);

		//etdebug("%s: Got IP, len=%d, v=%02x\n",__func__, len, ip->ip_hl_v & 0xff);
		if ((ip->ip_hl_v & 0xf0) != 0x40) break;
		if (ip->ip_off & htons(0x1fff)) break;	/* Can't deal w/ fragments */
		if (!NetCksumOk((uchar *)ip, IP_HDR_SIZE_NO_UDP/2))	{ debug("%s: checksum bad\n",__func__); break; }

		tmp = NetReadIP(&ip->ip_dst);
		if (NetOurIP && tmp != NetOurIP && tmp != 0xFFFFFFFF) break;

		/*
		 * watch for ICMP host redirects
		 *
		 * There is no real handler code (yet). We just watch
		 * for ICMP host redirect messages. In case anybody
		 * sees these messages: please contact me
		 * (wd@denx.de), or - even better - send me the
		 * necessary fixes :-)
		 *
		 * Note: in all cases where I have seen this so far
		 * it was a problem with the router configuration,
		 * for instance when a router was configured in the
		 * BOOTP reply, but the TFTP server was on the same
		 * subnet. So this is probably a warning that your
		 * configuration might be wrong. But I'm not really
		 * sure if there aren't any other situations.
		 */
		if (ip->ip_p == IPPROTO_ICMP)
		{
			ICMP_t *icmph = (ICMP_t *)&(ip->udp_src);
			switch (icmph->type)
			{
			case ICMP_REDIRECT:
				if (icmph->code != ICMP_REDIR_HOST) return;
				printf("%s: ICMP Host Redirect to "); print_IPaddr(icmph->un.gateway); printf("\n");
				break;
#if (CONFIG_COMMANDS & CFG_CMD_PING)
			case ICMP_ECHO_REPLY:
				/* IP header OK.  Pass the packet to the current handler. */
				/* XXX point to ip packet */
				if (packetHadler) (*packetHandler)((uchar *)ip, 0, 0, 0);
				break;
#endif
			default:
				break;
			}
		}
		else if (ip->ip_p == IPPROTO_UDP)
		{
#ifdef CONFIG_NETCONSOLE
			nc_input_packet((uchar *)ip +IP_HDR_SIZE,
				ntohs(ip->udp_dst), ntohs(ip->udp_src), ntohs(ip->udp_len) - 8);
#endif
			/* IP header OK.  Pass the packet to the current handler. */
			if (packetHandler)
				(*packetHandler)((uchar *)ip +IP_HDR_SIZE,
					ntohs(ip->udp_dst), ntohs(ip->udp_src), ntohs(ip->udp_len) - 8);
		}
#ifdef HTTPSVR_SUPPORT
		else if (ip->ip_p == IPPROTO_TCP) HandleTCP(et->et_src, (uchar *)ip, len);
#endif
		break;
	}
}


/**********************************************************************/

static int net_check_prereq (proto_t protocol)
{
	switch (protocol) {
		/* Fall through */
#if (CONFIG_COMMANDS & CFG_CMD_PING)
	case PING:
		if (NetPingIP == 0) {
			puts ("*** ERROR: ping address not given\n");
			return (1);
		}
		goto common;
#endif
#if (CONFIG_COMMANDS & CFG_CMD_NFS)
	case NFS:
#endif
	case NETCONS:
	case TFTP:
		if (NetServerIP == 0) {
			puts ("*** ERROR: `serverip' not set\n");
			return (1);
		}
#if (CONFIG_COMMANDS & CFG_CMD_PING)
	      common:
#endif

		if (NetOurIP == 0) {
			puts ("*** ERROR: `ipaddr' not set\n");
			return (1);
		}
		/* Fall through */

	case DHCP:
	case RARP:
	case BOOTP:
	case CDP:
		if (memcmp (NetOurEther, "\0\0\0\0\0\0", 6) == 0) {
#ifdef CONFIG_NET_MULTI
			extern int eth_get_dev_index (void);
			int num = eth_get_dev_index ();

			switch (num) {
			case -1:
				puts ("*** ERROR: No ethernet found.\n");
				return (1);
			case 0:
				puts ("*** ERROR: `ethaddr' not set\n");
				break;
			default:
				printf ("*** ERROR: `eth%daddr' not set\n",
					num);
				break;
			}

			NetStartAgain ();
			return (2);
#else
			puts ("*** ERROR: `ethaddr' not set\n");
			return (1);
#endif
		}
		/* Fall through */
	default:
		return (0);
	}
	return (0);		/* OK */
}
/**********************************************************************/

int
NetCksumOk(uchar * ptr, int len)
{
	return !((NetCksum(ptr, len) + 1) & 0xfffe);
}


unsigned
NetCksum(uchar * ptr, int len)
{
	ulong	xsum;

	xsum = 0;
	while (len-- > 0)
		xsum += *((ushort *)ptr)++;
	xsum = (xsum & 0xffff) + (xsum >> 16);
	xsum = (xsum & 0xffff) + (xsum >> 16);
	return (xsum & 0xffff);
}

int
NetEthHdrSize(void)
{
	ushort myvlanid;

	myvlanid = ntohs(NetOurVLAN);
	if (myvlanid == (ushort)-1)
		myvlanid = VLAN_NONE;

	return ((myvlanid & VLAN_IDMASK) == VLAN_NONE) ? ETHER_HDR_SIZE : VLAN_ETHER_HDR_SIZE;
}

int
NetSetEther(volatile uchar * xet, uchar * addr, uint prot)
{
	Ethernet_t *et = (Ethernet_t *)xet;
	ushort myvlanid;

	myvlanid = ntohs(NetOurVLAN);
	if (myvlanid == (ushort)-1)
		myvlanid = VLAN_NONE;

	memcpy (et->et_dest, addr, 6);
	memcpy (et->et_src, NetOurEther, 6);
	if ((myvlanid & VLAN_IDMASK) == VLAN_NONE) {
	et->et_protlen = htons(prot);
		return ETHER_HDR_SIZE;
	} else {
		VLAN_Ethernet_t *vet = (VLAN_Ethernet_t *)xet;

		vet->vet_vlan_type = htons(PROT_VLAN);
		vet->vet_tag = htons((0 << 5) | (myvlanid & VLAN_IDMASK));
		vet->vet_type = htons(prot);
		return VLAN_ETHER_HDR_SIZE;
	}
}

void
NetSetIP(volatile uchar * xip, IPaddr_t dest, int dport, int sport, int len)
{
	volatile IP_t *ip = (IP_t *)xip;

	/*
	 *	If the data is an odd number of bytes, zero the
	 *	byte after the last byte so that the checksum
	 *	will work.
	 */
	if (len & 1)
		xip[IP_HDR_SIZE + len] = 0;

	/*
	 *	Construct an IP and UDP header.
	 *	(need to set no fragment bit - XXX)
	 */
	ip->ip_hl_v  = 0x45;		/* IP_HDR_SIZE / 4 (not including UDP) */
	ip->ip_tos   = 0;
	ip->ip_len   = htons(IP_HDR_SIZE + len);
	ip->ip_id    = htons(NetIPID++);
	ip->ip_off   = htons(0x4000);	/* No fragmentation */
	ip->ip_ttl   = 255;
	ip->ip_p     = 17;		/* UDP */
	ip->ip_sum   = 0;
	NetCopyIP((void*)&ip->ip_src, &NetOurIP); /* already in network byte order */
	NetCopyIP((void*)&ip->ip_dst, &dest);	   /* - "" - */
	ip->udp_src  = htons(sport);
	ip->udp_dst  = htons(dport);
	ip->udp_len  = htons(8 + len);
	ip->udp_xsum = 0;
	ip->ip_sum   = ~NetCksum((uchar *)ip, IP_HDR_SIZE_NO_UDP / 2);
}

void copy_filename (uchar *dst, uchar *src, int size)
{
	if (*src && (*src == '"')) {
		++src;
		--size;
	}

	while ((--size > 0) && *src && (*src != '"')) {
		*dst++ = *src++;
	}
	*dst = '\0';
}

#endif /* CFG_CMD_NET */

void ip_to_string (IPaddr_t x, char *s)
{
	x = ntohl (x);
	sprintf (s, "%d.%d.%d.%d",
		 (int) ((x >> 24) & 0xff),
		 (int) ((x >> 16) & 0xff),
		 (int) ((x >> 8) & 0xff), (int) ((x >> 0) & 0xff)
	);
}

IPaddr_t string_to_ip(char *s)
{
	IPaddr_t addr;
	char *e;
	int i;

	if (s == NULL)
		return(0);

	for (addr=0, i=0; i<4; ++i) {
		ulong val = s ? simple_strtoul(s, &e, 10) : 0;
		addr <<= 8;
		addr |= (val & 0xFF);
		if (s) {
			s = (*e) ? e+1 : e;
		}
	}

	return (htonl(addr));
}

void VLAN_to_string(ushort x, char *s)
{
	x = ntohs(x);

	if (x == (ushort)-1)
		x = VLAN_NONE;

	if (x == VLAN_NONE)
		strcpy(s, "none");
	else
		sprintf(s, "%d", x & VLAN_IDMASK);
}

ushort string_to_VLAN(char *s)
{
	ushort id;

	if (s == NULL)
		return htons(VLAN_NONE);

	if (*s < '0' || *s > '9')
		id = VLAN_NONE;
	else
		id = (ushort)simple_strtoul(s, NULL, 10);

	return htons(id);
}

void print_IPaddr (IPaddr_t x)
{
	char tmp[16];

	ip_to_string (x, tmp);

	puts (tmp);
}

IPaddr_t getenv_IPaddr (char *var)
{
	return (string_to_ip(getenv(var)));
}

ushort getenv_VLAN(char *var)
{
	return (string_to_VLAN(getenv(var)));
}
