/* vi: set sw=4 ts=4: */
/*
 *	Copyright (C) 2008 Alpha Networks, Inc.
 *	by David Hsieh <david_hsieh@alphanetworks.com>
 */

#include <common.h>
#include <command.h>
#include <net.h>
#include "tftp.h"

#ifdef HTTPSVR_SUPPORT

/************************************************************************/

char * eatwhite(char * ptr)
{
	while (*ptr==' ') ptr++;
	return ptr;
}

/************************************************************************/

enum
{
	HTTPSTATE_WAITING = 0,	/* Waiting for METHOD */
	HTTPSTATE_READ_HEADER,	/* Reading one line */
	HTTPSTATE_READ_BODY,	/* Reading body now. */
};

#define MAX_HLINE		1024
#define HLINE_CLEAN()	do { hlen=0; hline[0]='\0'; } while (0)
#define HLINE_ADDC(c)	do { hline[hlen++]=(c); hline[hlen]='\0'; } while (0)

static int state;
static char hline[MAX_HLINE+1];		/* header line buffer */
static int hlen;					/* length of header line (in buffer) */
static int method;					/* 1:get, 2:post */
static char boundary[MAX_HLINE+1];	/* Boundary */
static int blen;					/* strlen(boundary) */
static ulong clen;					/* Content-Length */
static ulong csize;					/* Received Content size.  */
static ulong upsize;				/* uploaded image size. */

static void init_vars()
{
	state = HTTPSTATE_WAITING;
	memset(hline, 0, sizeof(hline));
	memset(boundary, 0, sizeof(boundary));
	method = 0;
	hlen = blen = 0;
	clen = csize = upsize = 0;
}

static char page_header[] =
	"HTTP/1.1 200 OK\r\n"
	"Date: Fri, 11 Jul 2008 12:00:00 GMT\r\n"
	"Content-Type: text/html; charset=utf-8\r\n"
	"Content-Length: %d\r\n"
	"Server: Emergency Room\r\n"
	"Conection: Close\r\n"
	"\r\n";
static char page_body[] =
	"<html>"
	"<head><title>Emergency Room</title></head>"
	"<body><center>"
	"<h1>UPDATING FIRMWARE</h1>"
	"<p>This page can be used to update the firmware of this device.</p>"
	"<form name=upload method=post enctype=multipart/form-data>"
	"<p><input type=file name=firmware size=30></p>"
	"<p><input type=submit name=post value='Upload firmware NOW'></p>"
	"</form>"
	"</center></body>"
	"</html>";

static char page_success[] =
	"<html>"
	"<head><title>SUCCESS</title>"
	"<script>\n"
	"var count = 90;\n"
	"function init() { countdown(); }\n"
	"function countdown()\n"
	"{\n"
	"	count--;\n"
	"	document.getElementById('WaitInfo').value = count;\n"
	"	if (count >= 1) setTimeout('countdown()',1000);\n"
	"	else top.location.href=\"\";\n"
	"}\n"
	"</script></head>"
	"<body onload='init()'><center>"
	"<h1>IMAGE UPLOADED SUCCESSFULLY</h1>"
	"<p>%d bytes uploaded.</p>"
	"<p>The WEB server is shuting down and the firmware updating will start immediately."
	" Please <font color=red><strong>DO NOT POWER OFF</strong></font> the device."
	" And please wait for <input type='Text' readonly id='WaitInfo' value='90' size='3'"
	" style='border-width:0; color:#FF3030; text-align:center'> seconds...</p>"
	"</center></body>"
	"</html>";

static char page_fail[] =
	"<html>"
	"<head><title>FAILED</title></head>"
	"<body><center>"
	"<h1>FAILED TO UPLOAD IMAGE</h1>"
	"<p>Unable to upload the image, "
	"please make sure the uploaded file is the correct image.</p>"
	"</center></body>"
	"</html>";

static void read_content_type(char * line)
{
	char * ptr = eatwhite(line);

	if (strncasecmp(ptr, "multipart/form-data;", 20)!=0) return;
	ptr = eatwhite(&ptr[20]);
	if (strncasecmp(ptr, "boundary=", 9)!=0) return;
	strcpy(boundary, &ptr[9]);
	blen = strlen(boundary);
	//debug("%s: found boundary = [%s]\n",__func__, boundary);
}

static void read_content_length(char * line)
{
	clen = simple_strtoul(eatwhite(line), NULL, 10);
	//debug("%s: clen = %d\n",__func__,clen);
}

static int mpfd_decode(uchar c)
{
	int ret = 0;
	static int state;
	static ulong cache;		/* store the lastest 4 characters of the received content */
	static uchar * ptr;

#define MPFD_WAIT			0
#define MPFD_CHECK_BOUNDARY	1	/* read and save character in hline, and check if we reach boundary. */
#define MPFD_FOUND_BOUNDARY	2	/* We reach boundary, skip "\r\n". */
#define MPFD_BOUNDARY_HEAD	3	/* Read boundary head. */
#define MPFD_READ_BODY		4

	/* initialize the decoder */
	if (csize == 0)
	{
		cache = 0x0d0a;
		hlen = 0;
		ptr = NULL;
		state = MPFD_WAIT;
		debug("%s: start, boundary = [%s], %d bytes, clen = %d\n", __func__,boundary,blen,clen);
	}

	/* Save this character in buffer & cache */
	cache = (ulong)((cache << 8) | (c & 0xff));
	csize++;
	//debug("%s: c = 0x%02x, csize = %d, cache = 0x%08x\n",__func__, c, csize, cache);
	if (ptr) *ptr++ = c;

	/* If we reach "\r\n--", check for boundary next. */
	if (cache == 0x0d0a2d2d && state != MPFD_BOUNDARY_HEAD)
	{
		//debug("%s: Reach boundary !!\n",__func__);
		state = MPFD_CHECK_BOUNDARY;
		HLINE_CLEAN();
	}
	else
	{
		/* cache the characters, if we match "\r\n--", check boundary. */
		switch (state)
		{
		case MPFD_CHECK_BOUNDARY:
			/* read and save in hline, and check for boundary. */
			HLINE_ADDC(c);
			if (hlen == blen)
			{
				if (memcmp(hline, boundary, hlen)==0)
				{
					/* Seek back */
					if (ptr)
					{
						/* We have got the first octet-stream. */
						ptr -= (blen + 4);
						/* TODO: we should check the image here!! */
						upsize = (ulong)(ptr - load_addr);
						ptr = NULL;
						printf("%s: UPLOADED, upsize = %d\n",__func__, upsize);
						ret = upsize;
					}
					else
					{
						debug("%s: FOUND [%s]\n",__func__,hline);
						/* Reach boundary */
						state = MPFD_FOUND_BOUNDARY;
					}
				}
				else
				{
					state = MPFD_WAIT;
				}
				HLINE_CLEAN();
			}
			break;

		case MPFD_FOUND_BOUNDARY:
			//debug("%s: MPFD_FOUND_BOUNDARY: c = 0x%02x\n",__func__,c);
			if (c == '\r') break;
			if (c == '\n')
			{
				state = MPFD_BOUNDARY_HEAD;
				ptr = NULL;
				debug("%s: BOUNDARY end with [%s]\n",__func__,hline);
				if (strcmp(hline, "--")==0)
				{
					debug("%s: This should be the last boundary! csize=%d, clen=%d\n",__func__, csize, clen);
					ret = 1;
				}
				HLINE_CLEAN();
			}
			else if (hlen < MAX_HLINE)
			{
				HLINE_ADDC(c);
			}
			break;

		case MPFD_BOUNDARY_HEAD:
			if (c == '\r') break;
			if (c == '\n')
			{
				debug("%s: PARTHEAD [%s] %d\n",__func__,hline, strlen(hline));
				if (strncasecmp(hline, "content-type: application/octet-stream", 38)==0)
				{
					/* Use ptr to mark that we have application/octet-stream. */
					ptr = (uchar *)load_addr;
				}
				else if (strlen(hline)==0)
				{
					debug("%s: entering MPFD_WAIT!\n",__func__);
					state = MPFD_WAIT;
					if (ptr)
					{
						/* ptr should be advanced to store this empty line, CR & LF.
						 * Reset it now for the real payload. */
						debug("%s: load_addr = 0x%08x\n",__func__,load_addr);
						ptr = (uchar *)load_addr;
					}
				}
				HLINE_CLEAN();
			}
			else if (hlen < MAX_HLINE)
			{
				HLINE_ADDC(c);
			}
			break;

		case MPFD_WAIT:
			break;
		}
	}

	return ret;
}

static void HttpHandler(uchar * pkt, unsigned dest, unsigned src, unsigned len)
{
	unsigned i;

	etdebug("%s: dest:%d, src:%d, len:%d (0x%08x)\n",__func__,dest,src,len,len);
	if (pkt == NULL || len == 0)
	{
		/* New session coming, reset the global variables. */
		debug("%s: prepare for new session!\n",__func__);
		init_vars();
		return;
	}

	for (i=0; i<len; i++)
	{
		switch (state)
		{
		case HTTPSTATE_WAITING:
			if (pkt[i] == '\r') break;
			if (pkt[i] == '\n')
			{
				debug("%s: HEADER[%s]\n",__func__,hline);

				if		(strncasecmp(hline, "GET", 3) ==0) method = 1;
				else if	(strncasecmp(hline, "POST", 4)==0) method = 2;
				hlen = 0;
				state = HTTPSTATE_READ_HEADER;
			}
			else if (hlen < MAX_HLINE)
			{
				HLINE_ADDC(pkt[i]);
			}
			else
			{
				debug("%s: First line too long !!\n",__func__);
			}
			break;

		case HTTPSTATE_READ_HEADER:
			if (pkt[i] == '\r') break;
			if (pkt[i] == '\n')
			{
				if (hlen==0)
				{
					state = HTTPSTATE_READ_BODY;
					debug("%s: End of header!\n",__func__);
				}
				else
				{
					//debug("%s: header[%s]\n",__func__,hline);

					/* parse this line. */
					if (strncasecmp(hline, "content-type:", 13)==0)
						read_content_type(&hline[13]);
					else if	(strncasecmp(hline, "content-length:", 15)==0)
						read_content_length(&hline[15]);

					/* Done on this line, reset the header length. */
					hlen = 0;
				}
			}
			else if (hlen < MAX_HLINE)
			{
				HLINE_ADDC(pkt[i]);
			}
			else
			{
				debug("%s: line to long, ignore [%c]\n",__func__,pkt[i]);
			}
			break;

		case HTTPSTATE_READ_BODY:
			if (method == 2 && blen > 0)
			{
				/* method is POST and content-type is multipart/form-data */
				if (csize < clen)
				{
					if (mpfd_decode(pkt[i]) > 0)
					{
						/* TODO: Check the image */
						debug("%s: DONE!!! upsize = %d\n",__func__, upsize);
						if (verify_IMGAlpha(load_addr, upsize))
						{
							sprintf(boundary, page_success, upsize);
							sprintf(hline, page_header, strlen(boundary));
							strcat(hline, boundary);
						}
						else
						{
							sprintf(hline, page_header, strlen(page_fail));
							strcat(hline, page_fail);
						}
						NetTcpSend(hline, strlen(hline));

						/* reset the HTTP state */
						init_vars();

						NetState = NETLOOP_SUCCESS;
					}
				}
			}
			break;
		}
	}
	if (method == 1 && state == HTTPSTATE_READ_BODY)
	{
		debug("%s: got method GET!\n",__func__);
		/* method is GET. Always return our only one page.*/
		sprintf(hline, page_header, strlen(page_body));
		strcat(hline, page_body);
		NetTcpSend(hline, strlen(hline));
		init_vars();
	}

	return;
}

void HttpStart(void)
{
#if defined(CONFIG_NET_MULTI)
	printf("Using %s device\n", eth_get_name());
#endif
	puts("Start HTTP server on port 80, "); print_IPaddr(NetOurIP);

	/* Check if we need to send across this subnet */
	if (NetOurGatewayIP && NetOurSubnetMask)
	{
		IPaddr_t OurNet		= NetOurIP    & NetOurSubnetMask;
		IPaddr_t ServerNet	= NetServerIP & NetOurSubnetMask;
		if (OurNet != ServerNet)
		{
			puts("; sending through gateway ");
			print_IPaddr(NetOurGatewayIP);
		}
	}
	putc('\n');
	puts("Running ...\n");

	NetInitTcp(htons(80));
	NetSetTCPHandler(HttpHandler);
}

#endif
