/*******************************************************************************
#	 	uvcview: Sdl video Usb Video Class grabber           .         #
#This package work with the Logitech UVC based webcams with the mjpeg feature. #
#All the decoding is in user space with the embedded jpeg decoder              #
#.                                                                             #
# 		Copyright (C) 2005 2006 Laurent Pinchart &&  Michel Xhaard     #
#     Copyright (C) 2007 Tom Stöveken (minor modifications for uvc_streamer)   #
#                                                                              #
# This program is free software; you can redistribute it and/or modify         #
# it under the terms of the GNU General Public License as published by         #
# the Free Software Foundation; either version 2 of the License, or            #
# (at your option) any later version.                                          #
#                                                                              #
# This program is distributed in the hope that it will be useful,              #
# but WITHOUT ANY WARRANTY; without even the implied warranty of               #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                #
# GNU General Public License for more details.                                 #
#                                                                              #
# You should have received a copy of the GNU General Public License            #
# along with this program; if not, write to the Free Software                  #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    #
#                                                                              #
*******************************************************************************/

#include <stdlib.h>

#include "uvcvideo.h"
#include "v4l2uvc.h"
#include "utils.h"
#include "nvram.h"

#define	DEBUG_VIDEO_PARAMETER	0x01
#define	DEBUG_IMAGE_PARAMETER	0x02
#define	DEBUG_UVC_PARAMETER	0x04
#define	DEBUG_MOTION		0x08
#define	DEBUG_ALL		-1
#define	DEBUG_VIDEO		(0)

#define SET_PARAMETER_FROM_UVC		0x00

#include "ip297x.c"

#define swab32(x) \
	((__u32)( \
		(((__u32)(x) & (__u32)0x000000ffUL) << 24) | \
		(((__u32)(x) & (__u32)0x0000ff00UL) <<  8) | \
		(((__u32)(x) & (__u32)0x00ff0000UL) >>  8) | \
		(((__u32)(x) & (__u32)0xff000000UL) >> 24) ))

static int debug = 0;

static int init_v4l2(struct vdIn *Video);

__s32 V4L2Query(struct vdIn *Video, __u32 V4L2id, struct v4l2_queryctrl	*ctrl)
{
int			ret_val;

	ctrl->id = V4L2id;
	if ( (ret_val=ioctl(Video->fd, VIDIOC_QUERYCTRL, ctrl)) < 0 ) {
		fprintf(stderr, "V4L2Read error : %i\n", ret_val);
		ctrl->default_value = -1;
	}
	return ret_val;
}

__s32 V4L2Read(struct vdIn *Video, __u32 V4L2id)
{
struct v4l2_control	ctrl;
int			ret_val;

	ctrl.id = V4L2id;
	if ( (ret_val=ioctl(Video->fd, VIDIOC_G_CTRL, &ctrl)) < 0 ) {
		fprintf(stderr, "V4L2Read error : %i\n", ret_val);
		ctrl.value = -1;
	}
	return ctrl.value;
}

int V4L2Write(struct vdIn *Video, __u32 V4L2id, __s32 Value)
{
struct v4l2_control	ctrl;
int			ret_val;

	ctrl.id = V4L2id;
	ctrl.value = Value;
	if ( (ret_val=ioctl(Video->fd, VIDIOC_S_CTRL, &ctrl)) < 0 ) {
		fprintf(stderr, "V4L2Write error : %i\n", ret_val);
		return -1;
	}
	return 0;
}

void UVCGetDefaultValues(struct vdIn *Video)
{
	// get min/max/default value
#if (SET_PARAMETER_FROM_UVC != 0)
	V4L2Query(Video, V4L2_CID_BRIGHTNESS, &Video->QueryBrightness);
	V4L2Query(Video, V4L2_CID_CONTRAST, &Video->QueryContrast);
	V4L2Query(Video, V4L2_CID_SATURATION, &Video->QuerySaturation);
#endif
	V4L2Query(Video, V4L2_CID_GAMMA, &Video->QueryQuality);
#if  ((DEBUG_VIDEO & DEBUG_UVC_PARAMETER) != 0 )
#if (SET_PARAMETER_FROM_UVC != 0)
	printf("Brightness : max=%3d, min=%3d, default=%3d, step=%3d, flag=%x\n",
		Video->QueryBrightness.maximum, Video->QueryBrightness.minimum, Video->QueryBrightness.default_value,
		Video->QueryBrightness.step, Video->QueryBrightness.flags);
	printf("Contrast   : max=%3d, min=%3d, default=%3d, step=%3d, flag=%x\n",
		Video->QueryContrast.maximum, Video->QueryContrast.minimum, Video->QueryContrast.default_value,
		Video->QueryContrast.step, Video->QueryContrast.flags);
	printf("Saturation : max=%3d, min=%3d, default=%3d, step=%3d, flag=%x\n",
		Video->QuerySaturation.maximum, Video->QuerySaturation.minimum, Video->QuerySaturation.default_value,
		Video->QuerySaturation.step, Video->QuerySaturation.flags);
#endif
	printf("Quality    : max=%3d, min=%3d, default=%3d, step=%3d, flag=%x\n",
		Video->QueryQuality.maximum, Video->QueryQuality.minimum, Video->QueryQuality.default_value,
		Video->QueryQuality.step, Video->QueryQuality.flags);
#endif
}

void SetVideoParameter(struct vdIn *Video,int SysInfoFD)
{
char	*pImageCfg;
int	ret_val, max_frame_size;
	//
	pImageCfg = nvram_bufget(RT2860_NVRAM, "VideoResolution");
	Video->CfgResolution = atoi(pImageCfg);
	pImageCfg = nvram_bufget(RT2860_NVRAM, "CompressionRate");
	Video->CfgCompressionRate = atoi(pImageCfg);
	pImageCfg = nvram_bufget(RT2860_NVRAM, "FrameRate");
	Video->CfgFrameRate = atoi(pImageCfg);
	pImageCfg = nvram_bufget(RT2860_NVRAM, "LightFrequency");
	Video->CfgLightFrequency = atoi(pImageCfg);
	nvram_close(RT2860_NVRAM);
	//
	switch ( Video->CfgResolution ) {
	case 0:
		Video->width = 160;
		Video->height = 120;
		Video->framesizeIn = 0x08000;
		break;
	case 1:
		Video->width = 320;
		Video->height = 240;
		Video->framesizeIn = 0x10000;
		break;
	case 2:
		Video->width = 640;
		Video->height = 480;
		Video->framesizeIn = 0x18000;
		break;
	}
	//
	Video->fps = 30;
	if ( (ret_val = ioctl(SysInfoFD, (SYSIF_READ |
			(FRAME_BUFFER_SIZE << SUBCMD_IDX_SHIFT) |
			(sizeof(long) << BUF_LEN_SHIFT)), &max_frame_size)) < 0 ) {
		fprintf(stderr, "Error read FRAME_BUFFER_SIZE : %i\n", ret_val);
	}
	if ( max_frame_size )	Video->framesizeIn = max_frame_size;
#if  ((DEBUG_VIDEO & DEBUG_VIDEO_PARAMETER) != 0 )
	fprintf(stderr, "Using V4L2 device: %s\n", Video->videodevice);
	fprintf(stderr, "Resolution.......: %i x %i\n", Video->width, Video->height);
	fprintf(stderr, "frames per second: %i\n", Video->fps);
	fprintf(stderr, "Max Frame Size   : %i\n", Video->framesizeIn);
	fprintf(stderr, "Resolution       : %i\n", Video->CfgResolution);
	fprintf(stderr, "CompressionRate  : %i\n", Video->CfgCompressionRate);
	fprintf(stderr, "FrameRate        : %i\n", Video->CfgFrameRate);
	fprintf(stderr, "LightFrequency   : %i\n", Video->CfgLightFrequency);
#endif
}

void SetImageParameter(struct vdIn *Video)
{
char	*pImageCfg;
__s32	v4l2_value;
int	reg_value;
__u8	sensor_data;

	pImageCfg = nvram_bufget(RT2860_NVRAM, "BrightnessControl");
	Video->CfgBrightness = atoi(pImageCfg);
	pImageCfg = nvram_bufget(RT2860_NVRAM, "ContrastControl");
	Video->CfgContrast = atoi(pImageCfg);
	pImageCfg = nvram_bufget(RT2860_NVRAM, "SharpnessControl");
	Video->CfgSharpness = atoi(pImageCfg);
	pImageCfg = nvram_bufget(RT2860_NVRAM, "SaturationControl");
	Video->CfgSaturation = atoi(pImageCfg);
	pImageCfg = nvram_bufget(RT2860_NVRAM, "Mirror");
	Video->CfgMirror = atoi(pImageCfg);
	pImageCfg = nvram_bufget(RT2860_NVRAM, "AntiFlickerEnable");
	reg_value = Video->CfgAntiFlickerEnable;
	Video->CfgAntiFlickerEnable = atoi(pImageCfg);
	pImageCfg = nvram_bufget(RT2860_NVRAM, "ViewMode");
	Video->CfgViewMode = atoi(pImageCfg);
	nvram_close(RT2860_NVRAM);
	if ( reg_value == Video->CfgAntiFlickerEnable ) {
		Video->ChangeFlag &= ~IMAGE_CHANGE_FLAG_ANTI_FLICKER;
	} else {
		Video->ChangeFlag |= IMAGE_CHANGE_FLAG_ANTI_FLICKER;
	}
	// set video parameters
#if (SET_PARAMETER_FROM_UVC == 0)
	VendorSetImageParameters(Video);
#else
	reg_value = Video->CfgBrightness*2;
	if ( reg_value > Video->QueryBrightness.maximum )	reg_value = Video->QueryBrightness.maximum;
	if ( reg_value < Video->QueryBrightness.minimum )	reg_value = Video->QueryBrightness.minimum;
	V4L2Write(Video, V4L2_CID_BRIGHTNESS, reg_value);
	reg_value = Video->CfgContrast*2;
	if ( reg_value > Video->QueryContrast.maximum )	reg_value = Video->QueryContrast.maximum;
	if ( reg_value < Video->QueryContrast.minimum )	reg_value = Video->QueryContrast.minimum;
	V4L2Write(Video, V4L2_CID_CONTRAST, reg_value);
	reg_value = Video->CfgSaturation*2;
	if ( reg_value > Video->QuerySaturation.maximum )	reg_value = Video->QuerySaturation.maximum;
	if ( reg_value < Video->QuerySaturation.minimum )	reg_value = Video->QuerySaturation.minimum;
	V4L2Write(Video, V4L2_CID_SATURATION, reg_value);
#endif
	//
#if  ((DEBUG_VIDEO & DEBUG_IMAGE_PARAMETER) != 0 )
	fprintf(stderr, "Brightness       : %i\n", Video->CfgBrightness);
	fprintf(stderr, "Contrast         : %i\n", Video->CfgContrast);
	fprintf(stderr, "Saturation       : %i\n", Video->CfgSaturation);
	fprintf(stderr, "Mirror           : %i\n", Video->CfgMirror);
	fprintf(stderr, "AntiFlicker      : %i\n", Video->CfgAntiFlickerEnable);
	fprintf(stderr, "ViewMode         : %i\n", Video->CfgViewMode);
	v4l2_value = V4L2Read(Video, V4L2_CID_GAMMA);
	fprintf(stderr, "Current Quality : %i\n", v4l2_value);
	//
#if (SET_PARAMETER_FROM_UVC != 0)
	v4l2_value = V4L2Read(Video, V4L2_CID_BRIGHTNESS);
	fprintf(stderr, "Current Brightness : %i\n", v4l2_value);
	v4l2_value = V4L2Read(Video, V4L2_CID_CONTRAST);
	fprintf(stderr, "Current Contrast : %i\n", v4l2_value);
	v4l2_value = V4L2Read(Video, V4L2_CID_SATURATION);
	fprintf(stderr, "Current Saturation : %i\n", v4l2_value);
#endif
#endif
}

void SetMotionDetection(struct vdIn *Video)
{
char	*pMotionCfg;
struct motion_control	motion;
int	idx, mask;

	memset(&motion, 0, sizeof(struct motion_control));
//	IP297xMotionGet(Video, (__u8*) &motion);
	pMotionCfg = nvram_bufget(RT2860_NVRAM, "MotionDetectionEnable");
	Video->CfgMotionEnable = atoi(pMotionCfg);
	motion.MotionFunSet = (__u8) Video->CfgMotionEnable;
	motion.FrameInterval = IP297X_MOTION_FRAME_INTERVAL;
	pMotionCfg = nvram_bufget(RT2860_NVRAM, "MotionDetectionBlockSet");
	motion.MotionBlockSet = 0;
	for ( idx = 0, mask = 1; idx < 25; idx++, mask = (mask << 1) ) {
		if ( pMotionCfg[idx] == '1' ) {
			motion.MotionBlockSet |= mask;
		}
	}
	motion.MotionBlockSet = swab32(motion.MotionBlockSet);
	motion.MotionBThd = 1;
	Video->CfgMotionBlockSet = (int) motion.MotionBlockSet;
	pMotionCfg = nvram_bufget(RT2860_NVRAM, "MotionDetectionSensitivity");
	Video->CfgMotionSensitivity = atoi(pMotionCfg);
	motion.MotionYThd = (__u8) ((IP297X_MOTION_SENSITIVITY_MAX - Video->CfgMotionSensitivity) / 2);
	if ( motion.MotionYThd <= 0 )	motion.MotionYThd = 1;
	nvram_close(RT2860_NVRAM);
	IP297xMotionSet(Video, (__u8*) &motion);
#if  ((DEBUG_VIDEO & DEBUG_MOTION) != 0 )
	fprintf(stderr, "MotionFunSet   = %.2d\n", motion.MotionFunSet);
	fprintf(stderr, "FrameInterval  = %.2d\n", motion.FrameInterval);
	fprintf(stderr, "MotionYThd     = %.2d\n", motion.MotionYThd);
	fprintf(stderr, "MotionBThd     = %.2d\n", motion.MotionBThd);
	fprintf(stderr, "MotionBlockSet = 0x%.8x\n", motion.MotionBlockSet);
	fprintf(stderr, "FrameCount     = %.2d\n", motion.FrameCount);
	fprintf(stderr, "MotionBlockNo  = %.2d\n", motion.MotionBlockNo);
#endif
}

int SetGrayImage(struct vdIn *Video, int Flag)
{
	return IP2970SetGrayImage(Video, Flag);
}

int InitVideoIn(struct vdIn *Video,int SysInfoFD, char *Device, int Format, int Grabmethod)
{
struct v4l2_control ctrl;

	if ((Video == NULL) || (Device == NULL) || (SysInfoFD==0))	return -1;
	if (Grabmethod < 0 || Grabmethod > 1)	Grabmethod = 1;             // mmap by default;
	Video->videodevice = NULL;
	Video->videodevice = (char *) calloc(1, 16 * sizeof(char));
	snprintf(Video->videodevice, 12, "%s", Device);
	Video->framecount = 0;
	Video->signalquit = 1;
	Video->formatIn = Format;
	Video->grabmethod = Grabmethod;
	SetVideoParameter(Video, SysInfoFD);
	if (init_v4l2(Video) < 0) {
		printf(" Init v4L2 failed !! exit fatal \n");
		goto error;;
	}
	UVCGetDefaultValues(Video);
	/*
	* alloc a temp buffer to reconstruct the pict
	*/
	Video->framebuffer =(unsigned char *) calloc(1, (size_t) Video->framesizeIn);
	if (!Video->framebuffer)	goto error;

	// add vendor specific extension controls
	AddExtensionUnitControls(Video);
	// set parameter from configuration
#if (SET_PARAMETER_FROM_UVC == 0)
	VendorSetVideoParameters(Video);
#endif
	Video->ChangeFlag |= IMAGE_CHANGE_FLAG_ANTI_FLICKER;	// reset AEC
	// set image quality 0 - 30(best)
	switch ( Video->CfgCompressionRate ) {
	case IMAGE_COMPRESSION_RATE_VERY_HIGH:
		V4L2Write(Video, V4L2_CID_GAMMA, 0);
		break;
	case IMAGE_COMPRESSION_RATE_HIGH:
		V4L2Write(Video, V4L2_CID_GAMMA, 10);
		break;
	case IMAGE_COMPRESSION_RATE_MEDIUM:
		V4L2Write(Video, V4L2_CID_GAMMA, 20);
		break;
	case IMAGE_COMPRESSION_RATE_LOW:
		V4L2Write(Video, V4L2_CID_GAMMA, 25);
		break;
	case IMAGE_COMPRESSION_RATE_VERY_LOW:
		V4L2Write(Video, V4L2_CID_GAMMA, 30);
		break;
	}
	SetImageParameter(Video);
	SetMotionDetection(Video);
	return 0;
error:
	free(Video->videodevice);
	close(Video->fd);
	return -1;
}


static int init_v4l2(struct vdIn *Video)
{
int i;
int ret = 0;

	if ((Video->fd = open(Video->videodevice, O_RDWR|O_NONBLOCK)) == -1) {
		perror("ERROR opening V4L interface \n");
		exit(1);
	}
	memset(&Video->cap, 0, sizeof(struct v4l2_capability));
	ret = ioctl(Video->fd, VIDIOC_QUERYCAP, &Video->cap);
	if (ret < 0) {
		printf("Error opening device %s: unable to query device.\n", Video->videodevice);
		goto fatal;
	}

	if ((Video->cap.capabilities & V4L2_CAP_VIDEO_CAPTURE) == 0) {
		printf("Error opening device %s: video capture not supported.\n", Video->videodevice);
		goto fatal;
	}
	if (Video->grabmethod) {
		if (!(Video->cap.capabilities & V4L2_CAP_STREAMING)) {
			printf("%s does not support streaming i/o\n", Video->videodevice);
			goto fatal;
		}
	} else {
		if (!(Video->cap.capabilities & V4L2_CAP_READWRITE)) {
			printf("%s does not support read i/o\n", Video->videodevice);
			goto fatal;
		}
	}
	/* set format in */
	memset(&Video->fmt, 0, sizeof(struct v4l2_format));
	Video->fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	Video->fmt.fmt.pix.width = Video->width;
	Video->fmt.fmt.pix.height = Video->height;
	Video->fmt.fmt.pix.pixelformat = Video->formatIn;
	Video->fmt.fmt.pix.field = V4L2_FIELD_ANY;
	ret = ioctl(Video->fd, VIDIOC_S_FMT, &Video->fmt);
	if (ret < 0) {
		printf("Unable to set format: %d.\n", errno);
		goto fatal;
	}
	if ((Video->fmt.fmt.pix.width != Video->width) || (Video->fmt.fmt.pix.height != Video->height)) {
		printf(" format asked unavailable get width %d height %d \n",
			Video->fmt.fmt.pix.width, Video->fmt.fmt.pix.height);
		Video->width = Video->fmt.fmt.pix.width;
		Video->height = Video->fmt.fmt.pix.height;
		/* look the format is not part of the deal ??? */
		// Video->formatIn = Video->fmt.fmt.pix.pixelformat;
	}

	/* set framerate */
	struct v4l2_streamparm *setfps;
	setfps = (struct v4l2_streamparm *) calloc(1, sizeof(struct v4l2_streamparm));
	memset(setfps, 0, sizeof(struct v4l2_streamparm));
	setfps->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	setfps->parm.capture.timeperframe.numerator = 1;
	setfps->parm.capture.timeperframe.denominator = Video->fps;
	ret = ioctl(Video->fd, VIDIOC_S_PARM, setfps);

	/* request buffers */
	memset(&Video->rb, 0, sizeof(struct v4l2_requestbuffers));
	Video->rb.count = NB_BUFFER;
	Video->rb.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	Video->rb.memory = V4L2_MEMORY_MMAP;

	ret = ioctl(Video->fd, VIDIOC_REQBUFS, &Video->rb);
	if (ret < 0) {
		printf("Unable to allocate buffers: %d.\n", errno);
		goto fatal;
	}
	/* map the buffers */
	for (i = 0; i < NB_BUFFER; i++) {
		memset(&Video->buf, 0, sizeof(struct v4l2_buffer));
		Video->buf.index = i;
		Video->buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		Video->buf.memory = V4L2_MEMORY_MMAP;
		ret = ioctl(Video->fd, VIDIOC_QUERYBUF, &Video->buf);
		if (ret < 0) {
			printf("Unable to query buffer (%d).\n", errno);
			goto fatal;
		}
		if (debug)
			printf("length: %u offset: %u\n", Video->buf.length, Video->buf.m.offset);
		Video->mem[i] = mmap(0 /* start anywhere */ ,
				Video->buf.length, PROT_READ, MAP_SHARED, Video->fd,
				Video->buf.m.offset);
		if (Video->mem[i] == MAP_FAILED) {
			printf("Unable to map buffer (%d)\n", errno);
			goto fatal;
		}
		if (debug)	printf("Buffer mapped at address %p.\n", Video->mem[i]);
	}
	/* Queue the buffers. */
	for (i = 0; i < NB_BUFFER; ++i) {
		memset(&Video->buf, 0, sizeof(struct v4l2_buffer));
		Video->buf.index = i;
		Video->buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		Video->buf.memory = V4L2_MEMORY_MMAP;
		ret = ioctl(Video->fd, VIDIOC_QBUF, &Video->buf);
		if (ret < 0) {
			printf("Unable to queue buffer (%d).\n", errno);
			goto fatal;;
		}
	}
	return 0;
fatal:
	return -1;

}


static int VideoEnable(struct vdIn *Video)
{
int type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
int ret;

	ret = ioctl(Video->fd, VIDIOC_STREAMON, &type);
	if (ret < 0) {
		printf("Unable to %s capture: %d.\n", "start", errno);
		return ret;
	}
	Video->isstreaming = 1;
	return 0;
}

static int VideoDisable(struct vdIn *Video)
{
int type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
int ret;

	ret = ioctl(Video->fd, VIDIOC_STREAMOFF, &type);
	if (ret < 0) {
		printf("Unable to %s capture: %d.\n", "stop", errno);
		return ret;
	}
	Video->isstreaming = 0;
	return 0;
}


int uvcGrab(struct vdIn *Video)
{
#define HEADERFRAME1 520
int	ret;

	if (!Video->isstreaming) {
		if (VideoEnable(Video))	goto err;
	}

	memset(&Video->buf, 0, sizeof(struct v4l2_buffer));
	Video->buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	Video->buf.memory = V4L2_MEMORY_MMAP;

	ret = ioctl(Video->fd, VIDIOC_DQBUF, &Video->buf);
	if (ret < 0) {
//		printf("Unable to dequeue buffer (%i).\n", errno);
		goto err;
	}

	switch (Video->formatIn) {
	case V4L2_PIX_FMT_MJPEG:
	case V4L2_PIX_FMT_YUYV:
		if (Video->buf.bytesused <= HEADERFRAME1) {    /* Prevent crash on empty image */
			printf("v4luvc:frame no data.\n");
			break;
		}
		if ( Video->buf.bytesused <= (Video->framesizeIn-1) ) {	// reserve last byte for motion flag
			memcpy(Video->framebuffer, (__u8 *) Video->mem[Video->buf.index], Video->buf.bytesused);
    		} else	fprintf(stderr, "<data overflow:%i> ...\n", Video->buf.bytesused);
		break;

	default:
		goto err;
		break;
	}

	ret = ioctl(Video->fd, VIDIOC_QBUF, &Video->buf);
	if (ret < 0) {
		printf("Unable to requeue buffer (%d).\n", errno);
		goto err;
	}

	return 0;

err:
	Video->signalquit = 0;
	return -1;
}

int close_v4l2(struct vdIn *Video)
{
	if (Video->isstreaming)	VideoDisable(Video);
	free(Video->framebuffer);
	Video->framebuffer = NULL;
	free(Video->videodevice);
	Video->videodevice = NULL;
	return 0;
}

