/*******************************************************************************
# "uvc_stream" is a command line application to stream JPEG files over an      #
# IP-based network from the webcam to a viewer like Firefox, Cambozola,        #
# Videolanclient or even to a Windows Mobile device running the TCPMP-Player.  #
#                                                                              #
# It was written for embedded devices with very limited ressources in terms of #
# RAM and CPU. The decision for Linux-UVC was done, because supported cameras  #
# directly produce JPEG-data, allowing fast and perfomant M-JPEG streams even  #
# from slow embedded devices like those supported by OpenWRT.                  #
#                                                                              #
# I would suggest not to open this server to the internet. Use it as input to  #
# the programm "motion" [http://motion.sf.net] running at your DMZ instead.    #
# Motion has many users and i expect it to be checked more often for security  #
# issues. Keep in mind, that motions advanced capabilties like                 #
# motion-detection, writing of avifiles etc require much more ressources.      #
#                                                                              #
# In contrast to the better known SPCA5XX-LE, UVC-cameras in average produce   #
# better image quality (See Michel XHaards comparison table and rating at      #
# his site) [http://mxhaard.free.fr/embedded.html].                            #
#                                                                              #
# This programm was written in 2007 by Tom Stöveken, basing on luvcview.       #
# The luvcview sources were modified using the tool "indent" and afterwards    #
# SDL dependencies were removed to reduce dependencies to other packages.      #
#                                                                              #
# This package work with the Logitech UVC based webcams with the mjpeg feature.#
#                                                                              #
#     Copyright (C) 2005 2006 Laurent Pinchart &&  Michel Xhaard               #
#     Copyright (C) 2007      Tom Stöveken                                     #
#                                                                              #
# This program is free software; you can redistribute it and/or modify         #
# it under the terms of the GNU General Public License as published by         #
# the Free Software Foundation; version 2 of the License.                      #
#                                                                              #
# This program is distributed in the hope that it will be useful,              #
# but WITHOUT ANY WARRANTY; without even the implied warranty of               #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                #
# GNU General Public License for more details.                                 #
#                                                                              #
# You should have received a copy of the GNU General Public License            #
# along with this program; if not, write to the Free Software                  #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    #
#                                                                              #
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <linux/videodev.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <signal.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <getopt.h>
#include <pthread.h>

#include "v4l2uvc.h"
#include "utils.h"

#define SOURCE_VERSION		"1.0"
#define DEFAULT_UVC_DEVICE	"/dev/video0"
#define UVC_STREAM_PID_FILE	"/var/run/uvc_stream.pid"
#define TEST_802_3		0	// add <uvc_stream -test802 &> to /user/rt2880_app/scripts/internet.sh

#define	UVC_STREAM_DEBUG_FRAME_RATE	0x01
#define	UVC_STREAM_DEBUG_FRAME_SIZE	0x02
#define	UVC_STREAM_DEBUG_MOTION		0x04
#define	UVC_STREAM_DEBUG_ALL		-1
#define	UVC_STREAM_DEBUG		(0)

#define IP297X_MOTION_FLAG_OFFSET	4
#define	MOTION_ACTIVE_TIMEOUT		3000
#define MOTION_FLAG_ACTIVE		0x01
#define MOTION_FLAG_DETECT		0x02
#define MOTION_FLAG_ENABLE		0x80

#define FRAME_DATA_FLAG_MOTION_DETECT	0x01

/* globals */
struct vdIn	*VideoIn;
int		VideoStop=0;
int		fd_SysInfo;
static char	*SavedPidFile=NULL;
int		FrameRate=0;
int		MotionFlag=0;
void 	SetImageParameter(struct vdIn *Video);

void FrameGrab()
{
int	ret, motion_flag;
__u16   *p_data_head, *p_data_tail;
__u8	*p_data, p_jpeg_check, p_jpeg_error;
struct	timeval	time_value;
__u64	time_current, time_start, time_motion_start;
__u32	event_flag, clear_event_flag;
__u32	bytesused;

	gettimeofday(&time_value, NULL);
	time_start = (time_value.tv_sec*1000) + (time_value.tv_usec/1000);
	while( !VideoStop ) {
		usleep(1000*10);
		/* grab a frame */
		if( (ret = uvcGrab(VideoIn)) < 0 )	continue;
		if ( VideoIn->buf.bytesused > (VideoIn->framesizeIn-1) )	continue;

		gettimeofday(&time_value, NULL);
		time_current = (time_value.tv_sec*1000) + (time_value.tv_usec/1000);
		/* copy frame to global buffer */
		bytesused = VideoIn->buf.bytesused;
		p_data = (__u8 *)&VideoIn->framebuffer[0];
		p_data_head = (__u16 *)&VideoIn->framebuffer[0];
		p_data_tail = (__u16 *)&VideoIn->framebuffer[bytesused-2];
		p_jpeg_check = VideoIn->framebuffer[519];
		p_jpeg_error = VideoIn->framebuffer[bytesused-3];
		/* uvc_video.c:uvc_video_decode_isoc() */
		/* set motion flag to 0xff, 0xd8, 0xff, 0xc4, (0x01) ==> (0x81) */
		if ( VideoIn->framebuffer[IP297X_MOTION_FLAG_OFFSET] & 0x80 ) {
			VideoIn->framebuffer[IP297X_MOTION_FLAG_OFFSET] &= 0x7f;
			MotionFlag |= MOTION_FLAG_DETECT;
			time_motion_start = time_current;
			VideoIn->framebuffer[bytesused] |= FRAME_DATA_FLAG_MOTION_DETECT;	// set motion flag in frame data
		} else	VideoIn->framebuffer[bytesused] &= ~FRAME_DATA_FLAG_MOTION_DETECT;
		if ( (*p_data_head == 0xd8ff) && (*p_data_tail == 0xd9ff) && 
			(p_jpeg_check==0x3f) && (p_jpeg_error != 0x14) ) {
			if ( (ret = ioctl(fd_SysInfo, (SYSIF_WRITE | 
				(FRAME_READY_LIST_Q << SUBCMD_IDX_SHIFT) |
				((bytesused+1) << BUF_LEN_SHIFT)),
				VideoIn->framebuffer)) < 0 ) {
				fprintf(stderr, "Error queue image\n");
			}
			FrameRate++;
#if ((UVC_STREAM_DEBUG & UVC_STREAM_DEBUG_FRAME_SIZE) != 0)
			fprintf(stderr, "<G:%6d>", bytesused);
		} else {
			fprintf(stderr, "Error image : (%4x), (%4x),(%2x)(%2x)", *p_data_head, *p_data_tail,
				p_jpeg_check, p_jpeg_error);
#endif			
		}
#if ((UVC_STREAM_DEBUG & UVC_STREAM_DEBUG_FRAME_RATE) != 0)
		if ( (time_current - time_start) >= 1000 ) {
			time_start = time_current;
			fprintf(stderr, "<FrameRate:%4d (%llums)>\n", FrameRate, time_current);
			FrameRate = 0;
		}
#endif
		if ( VideoIn->CfgMotionEnable ) {
			if ( (MotionFlag & MOTION_FLAG_DETECT) && ((MotionFlag & MOTION_FLAG_ACTIVE) == 0)) {
				// set motion active event
				MotionFlag |= MOTION_FLAG_ACTIVE;
				event_flag = 1;
				ioctl(fd_SysInfo, (SYSIF_WRITE | (MOTION_DETECTION_STATUS << SUBCMD_IDX_SHIFT) | 
					(4 << BUF_LEN_SHIFT)), 	&event_flag);
				event_flag = EVENT_MOTION_DETECTED;
				ioctl(fd_SysInfo, (SYSIF_WRITE | (SET_EVENT_FLAG << SUBCMD_IDX_SHIFT) | 
					(4 << BUF_LEN_SHIFT)), 	&event_flag);
#if ((UVC_STREAM_DEBUG & UVC_STREAM_DEBUG_MOTION) != 0)
				fprintf(stderr, "<Motion-Start:(%llums)>\n", time_current);
#endif
			}
			if ( (MotionFlag & MOTION_FLAG_ACTIVE) && 
				(time_current - time_motion_start >= MOTION_ACTIVE_TIMEOUT) ) {
				// motion timeout, set motion end event
				MotionFlag &= ~MOTION_FLAG_ACTIVE;
				event_flag = 0;
				ioctl(fd_SysInfo, (SYSIF_WRITE | (MOTION_DETECTION_STATUS << SUBCMD_IDX_SHIFT) | 
					(4 << BUF_LEN_SHIFT)), 	&event_flag);
				event_flag = EVENT_MOTION_ENDED;
				ioctl(fd_SysInfo, (SYSIF_WRITE | (SET_EVENT_FLAG << SUBCMD_IDX_SHIFT) | 
					(4 << BUF_LEN_SHIFT)), 	&event_flag);
#if ((UVC_STREAM_DEBUG & UVC_STREAM_DEBUG_MOTION) != 0)
				fprintf(stderr, "<Motion-End  :(%llums)>\n", time_current);
#endif
			}
		}
		MotionFlag &= ~MOTION_FLAG_DETECT;
		ioctl(fd_SysInfo, (SYSIF_READ | (READ_EVENT_FLAG << SUBCMD_IDX_SHIFT) | (4 << BUF_LEN_SHIFT)), 
				&event_flag);
		if ( event_flag & EVENT_IMAGE_CONFIG ) {
			clear_event_flag = EVENT_IMAGE_CONFIG;
			ioctl(fd_SysInfo, (SYSIF_WRITE | (CLEAR_EVENT_FLAG << SUBCMD_IDX_SHIFT) | (4 << BUF_LEN_SHIFT)), 
				&clear_event_flag);
			SetImageParameter(VideoIn);
		}
#if (TEST_AEC_AGC_DISABLE != 0)	/* close julie schedule.c ftp config event check */
		if ( event_flag & EVENT_FTP_CONFIG ) {
			clear_event_flag = EVENT_FTP_CONFIG;
			ioctl(fd_SysInfo, (SYSIF_WRITE | (CLEAR_EVENT_FLAG << SUBCMD_IDX_SHIFT) | (4 << BUF_LEN_SHIFT)), 
				&clear_event_flag);
			SetImageParameter(VideoIn);
		}
#endif		
		if ( event_flag & EVENT_MOTION_CONFIG ) {
			clear_event_flag = EVENT_MOTION_CONFIG;
			ioctl(fd_SysInfo, (SYSIF_WRITE | (CLEAR_EVENT_FLAG << SUBCMD_IDX_SHIFT) | (4 << BUF_LEN_SHIFT)), 
				&clear_event_flag);
			SetMotionDetection(VideoIn);
			if ( (VideoIn->CfgMotionEnable == 0) && (MotionFlag & MOTION_FLAG_ACTIVE) ) {
				MotionFlag &= ~MOTION_FLAG_ACTIVE;
				event_flag = 0;
				ioctl(fd_SysInfo, (SYSIF_WRITE | (MOTION_DETECTION_STATUS << SUBCMD_IDX_SHIFT) | 
					(4 << BUF_LEN_SHIFT)), 	&event_flag);
				event_flag = EVENT_MOTION_ENDED;
				ioctl(fd_SysInfo, (SYSIF_WRITE | (SET_EVENT_FLAG << SUBCMD_IDX_SHIFT) | 
					(4 << BUF_LEN_SHIFT)), 	&event_flag);
#if ((UVC_STREAM_DEBUG & UVC_STREAM_DEBUG_MOTION) != 0)
				fprintf(stderr, "<Motion-End by config>\n");
#endif
			}
		}
		if ( event_flag & EVENT_IR_LED_CHANGE ) {
			short	ir_mode;
			clear_event_flag = EVENT_IR_LED_CHANGE;
			ioctl(fd_SysInfo, (SYSIF_WRITE | (CLEAR_EVENT_FLAG << SUBCMD_IDX_SHIFT) | (4 << BUF_LEN_SHIFT)), 
				&clear_event_flag);
			ioctl(fd_SysInfo, (SYSIF_READ | (IR_LED_ON_OFF << SUBCMD_IDX_SHIFT) | (4 << BUF_LEN_SHIFT)), 
				&ir_mode);
			SetGrayImage(VideoIn, (int)ir_mode);
		}
	}
	return;
}

void help(char *progname)
{
	fprintf(stderr, "------------------------------------------------------------------\n");
	fprintf(stderr, "Usage: %s\n" \
			" [-h | --help ]........: display this help\n" \
			" [-d | --device ]......: video device to open (your camera)\n" \
			" [-v | --version ].....: display version information\n" \
			" [-b | --background]...: fork to the background, daemon mode\n" \
			" [-k ].................: kill process\n" \
			" [-f <filename>].......: set/get iP2970/OV7740 register\n" \
			" [-dump ]..............: display iP2970/OV7740 register data\n" \
			" [-sysinfo ]...........: set image parameter changed\n" \
			" [-test802 ]...........: 802.3 test\n" \
			, progname);
	fprintf(stderr, "------------------------------------------------------------------\n");
}

void signal_handler(int sig)
{
	/* signal "stop" to threads */
	VideoStop = 1;

	/* cleanup most important structures */
	fprintf(stderr, "shutdown...wait for 1secs \n");
	usleep(500*1000);
	close_v4l2(VideoIn);
	usleep(500*1000);
	close(fd_SysInfo);
	free(VideoIn);

	exit(0);
	return;
}

void daemon_mode(void)
{
int	fr=0;

	fr = fork();
	if( fr < 0 ) {
		fprintf(stderr, "fork() failed\n");
		exit(1);
	}
	if ( fr > 0 ) {
		exit(0);
	}

	if( setsid() < 0 ) {
		fprintf(stderr, "setsid() failed\n");
		exit(1);
	}

	fr = fork();
	if( fr < 0 ) {
		fprintf(stderr, "fork() failed\n");
		exit(1);
	}
	if ( fr > 0 ) {
#if (UVC_STREAM_DEBUG != 0)
		fprintf(stderr, "forked to background (%d)\n", fr);
#endif		
		exit(0);
	}

	umask(0);

	chdir("/");
	close(0);
	close(1);
	close(2);

	open("/dev/null", O_RDWR);
	dup(0);
	dup(0);
}

static void FileDeleteAtExit(void)
{
	if ( SavedPidFile )	unlink(SavedPidFile);
} /* FileDeleteAtExit() */

int PidFileGet(const char *PidFile)
{
int	pid_fd, pid_num;
char	buffer[10];

	if (!PidFile)	return -1;

	pid_fd = open(PidFile, O_RDONLY, 0644);
	if (pid_fd < 0) {
		printf("Unable to open pidfile %s: %m\n", PidFile);
		return -1;
	}
	pid_num = read(pid_fd, buffer, 10);
	if ( pid_num <= 0 ) {
		printf("read %s error : %d\n", PidFile, pid_num);
		close(pid_fd);
		return -1;
	}
	pid_num = atoi(buffer);
	return pid_num;
} /* PidFileAcquire() */

int PidFileAcquire(const char *PidFile)
{
int	pid_fd;

	if (!PidFile)	return -1;

	pid_fd = open(PidFile, O_CREAT | O_WRONLY, 0644);
	if (pid_fd < 0) {
		printf("Unable to open pidfile %s\n", PidFile);
	} else {
		lockf(pid_fd, F_LOCK, 0);
		if ( !SavedPidFile )	atexit(FileDeleteAtExit);
		SavedPidFile = (char *) PidFile;
	}
	return pid_fd;
} /* PidFileAcquire() */

void PidFileWriteRelease(int pid_fd)
{
FILE	*out;

	if (pid_fd < 0) return;

	if ((out = fdopen(pid_fd, "w")) != NULL) {
		fprintf(out, "%d\n", getpid());
		fclose(out);
	}
	lockf(pid_fd, F_UNLCK, 0);
	close(pid_fd);
} /* PidFileWriteRelease() */

int SensorCommand(char *FileName)
{
#define	OPTION_BUFFER_SIZE	80
FILE	*fp;
__u8	buf[OPTION_BUFFER_SIZE], *cmd, *cmd_argv1, *cmd_argv2, *cmd_argv3;
int	arg1, arg2, arg3;

	if (NULL == (fp = fopen(FileName, "ro"))) {
		perror("fopen");
		return -1;
	}

	while (NULL != fgets(buf, OPTION_BUFFER_SIZE, fp)) {
		cmd_argv1 = cmd_argv2 = cmd_argv3 = NULL;
		arg1 = arg2 = arg3 = 0;
		cmd = buf;
		while ( (*cmd == ' ') || (*cmd == '\t') )	cmd++;
		if ( (*cmd == '#') || (*cmd == '\n') || (*cmd == '\0') )	continue;
		cmd_argv1 = cmd + 2;
		while ( (*cmd_argv1 == ' ') || (*cmd_argv1 == '\t') )	cmd_argv1++;
		if ( (*cmd_argv1 != '#') && (*cmd_argv1 != '\n') && (*cmd_argv1 != '\0') ) {
			arg1 = strtol(cmd_argv1, NULL, 16);
			cmd_argv2 = cmd_argv1 + 2;
			while ( (*cmd_argv2 == ' ') || (*cmd_argv2 == '\t') )	cmd_argv2++;
			if ( (*cmd_argv2 != '#') && (*cmd_argv2 != '\n') && (*cmd_argv2 != '\0') ) {
				arg2 = strtol(cmd_argv2, NULL, 16);
				cmd_argv3 = cmd_argv2 + 2;
				while ( (*cmd_argv3 == ' ') || (*cmd_argv3 == '\t') )	cmd_argv3++;
				if ( (*cmd_argv3 != '#') && (*cmd_argv3 != '\n') && (*cmd_argv3 != '\0') ) {
					arg3 = strtol(cmd_argv3, NULL, 16);
				}
			}
		}
		if ( cmd[1] == 's' ) {	// OV7740
			if ( cmd[0] == 'r' ) {
				fprintf(stderr, "Read  I2C    Reg=0x%.2X, ", arg2);
				fprintf(stderr, "Data = 0x%.2X \n", 
					IP297xI2CGetByte(VideoIn, (__u8) arg1, (__u8) arg2));
			} else if ( cmd[0] == 'w' ) {
				fprintf(stderr, "Write I2C    Reg=0x%.2X, Data = 0x%.2X", arg2, arg3);
				IP297xI2CSetByte(VideoIn, (__u8) arg1, (__u8) arg2, (__u8) arg3);
				fprintf(stderr, "\n");
			}
		} else if ( cmd[1] == 'i' ){	// iP2970
			if ( cmd[0] == 'r' ) {
				fprintf(stderr, "Read  iP2970 Reg=0x%.2X, ", arg1);
				fprintf(stderr, "Data = 0x%.2X \n", 
					IP297xRegisterGet(VideoIn, (__u8) arg1));
			} else if ( cmd[0] == 'w' ) {
				fprintf(stderr, "Write iP2970 Reg=0x%.2X, Data = 0x%.2X", arg1, arg2);
				IP297xRegisterSet(VideoIn, (__u8) arg1, (__u8) arg2);
				fprintf(stderr, "\n");
			}
		}
	}

	fclose(fp);
	return 0;
}

/* Main */
int main(int argc, char *argv[])
{
char	*device = DEFAULT_UVC_DEVICE;
char	*sensor_command_file = NULL;
int	format = V4L2_PIX_FMT_MJPEG;	// always use MJPEG frame format
int	daemon;
int	fd, saved_pid;
int	command_flag=0;
__u32	event_flag;

	daemon = 0;
	while(1) {
		int option_index = 0, c=0;
		static struct option long_options[] = \
					{
					{"h", no_argument, 0, 0},
					{"help", no_argument, 0, 0},
					{"d", required_argument, 0, 0},
					{"device", required_argument, 0, 0},
					{"v", no_argument, 0, 0},
					{"version", no_argument, 0, 0},
					{"b", no_argument, 0, 0},
					{"background", no_argument, 0, 0},
					{"k", no_argument, 0, 0},
					{"f", required_argument, 0, 0},
					{"dump", no_argument, 0, 0},
					{"sysinfo", no_argument, 0, 0},
					{"test802", no_argument, 0, 0},
					{0, 0, 0, 0}
					};

		c = getopt_long_only(argc, argv, "", long_options, &option_index);

		/* no more options to parse */
		if (c == -1) break;

		/* unrecognized option */
		if(c=='?'){ help(argv[0]); return 0; }

		switch (option_index) {
			/* h, help */
		case 0:
		case 1:
			help(argv[0]);
			return 0;
			break;
		/* d, device */
		case 2:
		case 3:
			device = strdup(optarg);
			break;
		/* v, version */
		case 4:
		case 5:
			printf("UVC Streamer Version: %s\n" \
				"Compilation Date....: %s\n" \
				"Compilation Time....: %s\n", SOURCE_VERSION, __DATE__, __TIME__);
			return 0;
			break;
		/* b, background */
		case 6:
		case 7:
			daemon=1;
			break;
		case 8:
			saved_pid = PidFileGet(UVC_STREAM_PID_FILE);
			if ( saved_pid > 0 )	kill(saved_pid, SIGINT);
			return 0;
		case 9:
			sensor_command_file = strdup(optarg);
			command_flag |= 0x80;
			break;
		case 10:
			command_flag |= 0x40;
			break;
		case 11:
			fd_SysInfo = open(SYSIF_DEV, O_RDONLY);
			if ( fd_SysInfo < 0 ) {
				fprintf(stderr, "SysInfo: %s open fail, code %d\n", SYSIF_DEV, fd_SysInfo);
				exit(1);
			}
			event_flag = EVENT_IMAGE_CONFIG;
			ioctl(fd_SysInfo, (SYSIF_WRITE | (SET_EVENT_FLAG << SUBCMD_IDX_SHIFT) | 
				(4 << BUF_LEN_SHIFT)), 	&event_flag);
			close(fd_SysInfo);
			return 0;
		case 12:
#if (TEST_802_3 != 0)
			Test802();
#else
			fprintf(stderr,"must compile 802.3 test code!!\n");
#endif
			return 0;
		default:
			help(argv[0]);
			return 0;
		} // switch (option_index)
	} // while(1)

	/* ignore SIGPIPE (send if transmitting to closed sockets) */
	signal(SIGPIPE, SIG_IGN);
	if (signal(SIGINT, signal_handler) == SIG_ERR) {
		fprintf(stderr, "could not register signal handler\n");
		exit(1);
	}

	fd_SysInfo = open(SYSIF_DEV, O_RDONLY);
	if ( fd_SysInfo < 0 ) {
		fprintf(stderr, "SysInfo: %s open fail, code %d\n", SYSIF_DEV, fd_SysInfo);
		exit(1);
	}
	/* allocate webcam datastructure */
	VideoIn = (struct vdIn *) calloc(1, sizeof(struct vdIn));

	/* open video device and prepare data structure */
	if (InitVideoIn(VideoIn, fd_SysInfo, device, format, 1) < 0) {
		fprintf(stderr, "init_VideoIn failed\n");
		exit(1);
	}

	if ( (command_flag & 0x80) != 0 ) {
		SensorCommand(sensor_command_file);
	}
	if ( (command_flag & 0x40) != 0 ) {
		DumpAllRegister(VideoIn);
	}
	/* fork to the background */
	if ( daemon ) {
		daemon_mode();
	}
	// save pid
	fd = PidFileAcquire(UVC_STREAM_PID_FILE);
	PidFileWriteRelease(fd);
	//
	fprintf(stderr, "<Start Grab Frame!>\n");
	FrameGrab();
	return 0;
}

#if (TEST_802_3 != 0)
#include <sys/socket.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#include <linux/if_arp.h>
#include "nvram.h"

#define MAX_ETHERNET_FRAME_LENGTH 		1514
#define MIN_ETHERNET_FRAME_LENGTH 		64
#define NVRAM_COMMAND_SIZE			64

#define PACKET_DEFAULT_PATTERN			0xff
#define PACKET_MODE_PATTERN			1
#define PACKET_MODE_RANDOM_ONE			2
#define PACKET_MODE_RANDOM_ALL			3

int Test802()
{
struct sockaddr_ll	socket_address;
unsigned char		*buffer, *p_pattern;
int			sock, idx, packet_length, packet_mode, time_vaule;
__u8			temp_buff[NVRAM_COMMAND_SIZE], pattern;
struct USER_SIG_FLAG 	signal_flag;


	fprintf(stderr, "test 802.3........\n");
	system("rm /sbin/reboot");
	buffer = (unsigned char *)malloc(MAX_ETHERNET_FRAME_LENGTH);
	if ( (sock=socket(AF_PACKET,SOCK_RAW,0)) == -1 ) {
		perror("raw socket"); 
		exit(0);
	}
	fd_SysInfo = open(SYSIF_DEV, O_RDONLY);
	if ( fd_SysInfo < 0 ) {
		fprintf(stderr, "SysInfo: %s open fail, code %d\n", SYSIF_DEV, fd_SysInfo);
		exit(1);
	}

	memset(&socket_address, 0, sizeof(socket_address));
	socket_address.sll_family   = AF_PACKET;
	socket_address.sll_protocol = htons(ETH_P_IP);
	socket_address.sll_ifindex  = 3;

	// read RESET button
	signal_flag.process = 1;	//nvram_daemon    
	while (1) {
		if (ioctl(fd_SysInfo, (SYSIF_READ | (USER2_SIG_FLAG<<SUBCMD_IDX_SHIFT) |
					(sizeof(struct USER_SIG_FLAG)<<BUF_LEN_SHIFT)), &signal_flag) < 0) {
			fprintf(stderr, "....xxxxxxxxxxxxxxxxxxxx\n");
			perror("ioctl");
			usleep(1000*3000);
			continue;
		}
		if( signal_flag.sig_flag & USER2_SIG_RESTART ) {
			fprintf(stderr, "....start\n");
			break;
		}
		usleep(1000*500);
	} // while (1)

	// set pattern
	p_pattern = nvram_bufget(RT2860_NVRAM, "CameraName");
	memset(temp_buff, 0, NVRAM_COMMAND_SIZE);
	memcpy(temp_buff, p_pattern, NVRAM_COMMAND_SIZE);
	nvram_close(RT2860_NVRAM);
	if ( temp_buff[0] == '\0' ) {
		packet_length = MAX_ETHERNET_FRAME_LENGTH;
		packet_mode = PACKET_MODE_PATTERN;
		pattern = PACKET_DEFAULT_PATTERN;
	} else {
		// get pattern
		packet_mode = 0;
		p_pattern = strstr(temp_buff, "p=");
		if ( p_pattern == NULL )	p_pattern = strstr(temp_buff, "P=");
		if ( p_pattern == NULL ) {
			pattern = PACKET_DEFAULT_PATTERN;
			packet_mode = PACKET_MODE_PATTERN;
		} else {
			p_pattern += 2;
			if ( (*p_pattern == 'r') || (*p_pattern == 'R') ) {
				if ( *(p_pattern+1) == '1' ) {
					packet_mode = PACKET_MODE_RANDOM_ONE;
				} else {
					packet_mode = PACKET_MODE_RANDOM_ALL;
				}
			} else {
				packet_mode = PACKET_MODE_PATTERN;
				pattern = (__u8) strtol(p_pattern, NULL, 16);
			}
		}
		// get length
		p_pattern = strstr(temp_buff, "l=");
		if ( p_pattern == NULL )	p_pattern = strstr(temp_buff, "L=");
		if ( p_pattern == NULL ) {
			packet_length = MAX_ETHERNET_FRAME_LENGTH;
		} else {
			p_pattern += 2;
			packet_length = strtol(p_pattern, NULL, 10);
		}
		if ( packet_length < 60 )	packet_length = MIN_ETHERNET_FRAME_LENGTH;
		if ( packet_length > MAX_ETHERNET_FRAME_LENGTH )	packet_length = MAX_ETHERNET_FRAME_LENGTH;
	} // if ( temp_buff[0] == '\0' )
	
	// set packet data
	if ( packet_mode == PACKET_MODE_PATTERN ) {
		memset(buffer, pattern, packet_length);
	} else if ( packet_mode == PACKET_MODE_RANDOM_ONE ) { // just create one random packet
		srand(time(NULL));
		for ( idx = 0; idx < packet_length; idx++ ) {
			buffer[idx] = (__u8)rand();
		}
	}
	buffer[0] |= 0x01;	// set multicast, RT3050 internal switch will discard unicast packet
	
	while (1) {
		if( (sendto(sock, buffer, packet_length, 0,
					(struct sockaddr*)&socket_address,
					sizeof(socket_address))) ==- 1 ) {
			perror("send");
			exit(0);
		}
		if ( packet_mode == PACKET_MODE_RANDOM_ALL ) { // create random packet 
			srand(time(NULL)+rand());
			for ( idx = 0; idx < packet_length; idx++ ) {
				buffer[idx] = (__u8)rand();
			}
			buffer[0] |= 0x01;	// set multicast, RT3050 internal switch will discard unicast packet
		}
	} // while (1)
  	return 0;
}
#endif	// TEST_802_3
