#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <linux/videodev.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <signal.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <getopt.h>
#include <pthread.h>
#include <fcntl.h>

#include "uvcvideo.h"

#define DEFAULT_UVC_DEVICE	"/dev/video0"

#define	COMMAND_FLAG_SENSOR_INIT	0x01
#define	COMMAND_FLAG_SENSOR_ACCESS	0x02
#define	COMMAND_FLAG_DUMP_SENSOR	0x04

#define	NB_BUFFER	4
struct vdIn {
	int				fd;
	char				*videodevice;
	int				SensorID;
	struct v4l2_capability		cap;
	struct v4l2_format		fmt;
	struct v4l2_buffer		buf;
	struct v4l2_requestbuffers	rb;
	void				*mem[NB_BUFFER];
	unsigned char 			*framebuffer;
	int				isstreaming;
	int				grabmethod;
	int				width;
	int				height;
	int				fps;
	int				formatIn;
	int				framesizeIn;
	int				signalquit;
	int				framecount;
};

int InitVideoIn(struct vdIn *Video, int SysInfoFD, char *Device, int Format, int Grabmethod);
int init_v4l2(struct vdIn *Video);
int close_v4l2(struct vdIn *vd);
/* globals */
struct vdIn	*VideoIn;

///////////////////////////////////////////////////////////
/*iP297x Extension_Unit Control Selectors */
#define EU_RIGISTER_CONTROL		0x01
#define EU_EEPROM_CONTROL		0x02
#define EU_RAM_CONTROL			0x03
#define EU_I2C_CONTROL			0x04

#define EXTENSION_RIGISTER_READ		0x01
#define EXTENSION_RIGISTER_WRITE	0x00
#define EXTENSION_I2C_READ		0x80
#define EXTENSION_I2C_WRITE		0x00

#define EXTU_SIF_8BIT			0x01
#define EXTU_SIF_16BIT			0x02
#define IP297X_EXTENSION_UNIT_ID	4
#define UVC_GUID_UVC_EXTENSION_UNIT_IP2970 \
	{0x3a, 0xab, 0x91, 0x99, 0xef, 0xb2, 0xc9, 0x48, \
	 0x8f, 0xe9, 0x8f, 0xe3, 0x63, 0x47, 0x71, 0xd0}

#define I2C_OV_SENSOR_ID		0x21
struct motion_control {	// default = 0x01, 0x10, 0x10, 0x0A, 0xFFFFFFFF, 0x01, 0x00
	__u8	MotionFunSet;	// b0:enable
	__u8	FrameInterval;
	__u8	MotionYThd;
	__u8	MotionBThd;
	__u32	MotionBlockSet;
	__u8	FrameCount;
	__u8	MotionBlockNo;
}  __attribute__ ((packed));

struct ip297x_rw_register {
	__u8	Function;
	__u8	Address;
	__u8	Data;
} __attribute__ ((packed));

struct ip297x_i2c {
	__u8	Status;
	__u8	SensorID;;
	__u8	Address;
	__u8	DataL;
	__u8	DataH;
}  __attribute__ ((packed));
static struct uvc_xu_control_info ip297x_uvc_ctrls[] = {
	{
		.entity		= UVC_GUID_UVC_EXTENSION_UNIT_IP2970,
		.selector	= EU_RIGISTER_CONTROL,
		.index		= 1,
		.size		= sizeof(struct ip297x_rw_register),
		.flags		= UVC_CONTROL_SET_CUR | UVC_CONTROL_GET_CUR,
	},
	{
		.entity		= UVC_GUID_UVC_EXTENSION_UNIT_IP2970,
		.selector	= EU_RAM_CONTROL,
		.index		= 2,
		.size		= sizeof(struct motion_control),
		.flags		= UVC_CONTROL_SET_CUR | UVC_CONTROL_GET_CUR,
	},
	{
		.entity		= UVC_GUID_UVC_EXTENSION_UNIT_IP2970,
		.selector	= EU_I2C_CONTROL,
		.index		= 3,
		.size		= sizeof(struct ip297x_i2c),
		.flags		= UVC_CONTROL_SET_CUR | UVC_CONTROL_GET_CUR,
	},
#if 0
	{
		.entity		= UVC_GUID_UVC_EXTENSION_UNIT_IP2970,
		.selector	= EU_EEPROM_CONTROL,
		.index		= 4,
		.size		= 10,
		.flags		= UVC_CONTROL_SET_CUR | UVC_CONTROL_GET_CUR,
	},
#endif
	{
		.entity		= 0,
		.selector	= 0,
		.index		= 0,
		.size		= 0,
		.flags		= 0,
	},
};
///////////////////////////////////////////////////////////
__u8 I2CGetByte(struct vdIn *Video, __u8 Address)
{
struct ip297x_i2c		i2c_data;
struct uvc_xu_control		xctrl;
int				ret_val;

	i2c_data.Status = EXTENSION_I2C_READ + EXTU_SIF_8BIT;
	i2c_data.SensorID = I2C_OV_SENSOR_ID;
	i2c_data.Address = Address;
	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_I2C_CONTROL;
	xctrl.size = sizeof(struct ip297x_i2c);
	xctrl.data = (__u8 *) &i2c_data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_SET, &xctrl);
	if ( ret_val < 0 )	fprintf(stderr, "IP297xI2CGet error 1: %i\n", ret_val);
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_GET, &xctrl);
	if ( (ret_val < 0) || (i2c_data.Status == 0) ) {
		fprintf(stderr, "IP297xI2CGetByte error : %i\n", ret_val);
	}
	return i2c_data.DataL;
}

int I2CSetByte(struct vdIn *Video, __u8 Address, __u8 Data)
{
struct ip297x_i2c		i2c_data;
struct uvc_xu_control		xctrl;
int				ret_val;

	i2c_data.Status = EXTENSION_I2C_WRITE + EXTU_SIF_8BIT;
	i2c_data.SensorID = I2C_OV_SENSOR_ID;
	i2c_data.Address = Address;
	i2c_data.DataL = Data;
	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_I2C_CONTROL;
	xctrl.size = sizeof(struct ip297x_i2c);
	xctrl.data = (__u8 *) &i2c_data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_SET, &xctrl);
	if ( (ret_val < 0) || (i2c_data.Status == 0) ) {
		fprintf(stderr, "IP297xI2CSetByte error : %i\n", ret_val);
	}
	return 0;
}

int AddExtensionUnitControls(struct vdIn *Video)
{
struct uvc_xu_control_info	*xinfo;
int	ret_val, idx, sensor_id;

	xinfo = ip297x_uvc_ctrls;
	while( xinfo->entity[0] != 0 ) {
		ret_val=ioctl(Video->fd, UVCIOC_CTRL_ADD, xinfo);
		if ( (ret_val < 0) && (ret_val != -1) ) {
			printf(" Adding control failed (%d)!!\n", ret_val);
		}
		xinfo++;
	}
	return 0;
}


int DumpAllRegister(struct vdIn *Video, int Mode)
{
int	idx, ret;
__u8	ret_data;

	printf("\nDump OV7740 :\n");
	for ( idx=0; idx <=0xff; idx++ ) {
		ret_data = I2CGetByte(Video, (__u8) idx);
		if ( Mode == 0 ) {
			printf("42 %.2x %.2x\n", idx, ret_data);
		} else {
			printf("(%.2x, %.2x)", idx, ret_data);
			if ( ((idx+1) % 8) == 0 )	printf("\n");
		}
	}
	printf("\n");
}

int InitVideoIn(struct vdIn *Video,int SysInfoFD, char *Device, int Format, int Grabmethod)
{
struct v4l2_control ctrl;
int	ret_val=0;

	if ((Video == NULL) || (Device == NULL) )	return -1;
	Video->videodevice = NULL;
	Video->videodevice = (char *) calloc(1, 16 * sizeof(char));
	snprintf(Video->videodevice, 12, "%s", Device);
	Video->framecount = 0;
	Video->signalquit = 1;
	Video->formatIn = Format;
	Video->grabmethod = Grabmethod;
	if (init_v4l2(Video) < 0) {
		printf(" Init v4L2 failed !! exit fatal \n");
		goto error;
	}
	// add vendor specific extension controls
//	AddExtensionUnitControls(Video);
	return 0;
error:
	free(Video->videodevice);
	close(Video->fd);
	return -1;
}


int init_v4l2(struct vdIn *Video)
{
int i;
int ret = 0;

	if ((Video->fd = open(Video->videodevice, O_RDWR|O_NONBLOCK)) == -1) {
		fprintf(stderr, "ERROR opening V4L interface \n");
		exit(1);
	}
	memset(&Video->cap, 0, sizeof(struct v4l2_capability));
	ret = ioctl(Video->fd, VIDIOC_QUERYCAP, &Video->cap);
	if (ret < 0) {
		fprintf(stderr, "Error opening device %s: unable to query device.\n", Video->videodevice);
		goto fatal;
	}

	if ((Video->cap.capabilities & V4L2_CAP_VIDEO_CAPTURE) == 0) {
		fprintf(stderr, "Error opening device %s: video capture not supported.\n", Video->videodevice);
		goto fatal;
	}
	if (Video->grabmethod) {
		if (!(Video->cap.capabilities & V4L2_CAP_STREAMING)) {
			fprintf(stderr, "%s does not support streaming i/o\n", Video->videodevice);
			goto fatal;
		}
	} else {
		if (!(Video->cap.capabilities & V4L2_CAP_READWRITE)) {
			fprintf(stderr, "%s does not support read i/o\n", Video->videodevice);
			goto fatal;
		}
	}
	//
	return 0;
fatal:
	return -1;

}



int close_v4l2(struct vdIn *Video)
{
	free(Video->videodevice);
	Video->videodevice = NULL;
	close(Video->fd);
	return 0;
}

///////////////////////////////////////////////////////////
void help(char *progname)
{
	fprintf(stderr, "------------------------------------------------------------------\n");
	fprintf(stderr, "Usage: %s\n", progname);
	fprintf(stderr, " [-h]............: display this help\n");
	fprintf(stderr, " [-f <filename>].: set/get OV7740 register\n");
	fprintf(stderr, " [-dump].........: display OV7740 register data\n");
	fprintf(stderr, " [-m mode].......: display OV7740 register data, 0=for ov FAE, 1=list\n");
	fprintf(stderr, " [-a <register>].: OV7740 register\n");
	fprintf(stderr, " [-d <data>].....: set OV7740 data\n");
	fprintf(stderr, "------------------------------------------------------------------\n");
}


int SensorCommand(char *FileName)
{
#define	OPTION_BUFFER_SIZE	80
FILE	*fp;
__u8	buf[OPTION_BUFFER_SIZE], *cmd, *cmd_argv1, *cmd_argv2, *cmd_argv3;
int	arg1, arg2, arg3;

	if (NULL == (fp = fopen(FileName, "ro"))) {
		fprintf(stderr, "fopen fail (%s)\n", FileName);
		return -1;
	}

	while (NULL != fgets(buf, OPTION_BUFFER_SIZE, fp)) {
		cmd_argv1 = cmd_argv2 = cmd_argv3 = NULL;
		arg1 = arg2 = arg3 = 0;
		cmd = buf;
		while ( (*cmd == ' ') || (*cmd == '\t') )	cmd++;
		if ( (*cmd == '#') || (*cmd == '\n') || (*cmd == '\0') )	continue;
		cmd_argv1 = cmd + 2;
		while ( (*cmd_argv1 == ' ') || (*cmd_argv1 == '\t') )	cmd_argv1++;
		if ( (*cmd_argv1 != '#') && (*cmd_argv1 != '\n') && (*cmd_argv1 != '\0') ) {
			arg1 = strtol(cmd_argv1, NULL, 16);
			cmd_argv2 = cmd_argv1 + 2;
			while ( (*cmd_argv2 == ' ') || (*cmd_argv2 == '\t') )	cmd_argv2++;
			if ( (*cmd_argv2 != '#') && (*cmd_argv2 != '\n') && (*cmd_argv2 != '\0') ) {
				arg2 = strtol(cmd_argv2, NULL, 16);
				cmd_argv3 = cmd_argv2 + 2;
				while ( (*cmd_argv3 == ' ') || (*cmd_argv3 == '\t') )	cmd_argv3++;
				if ( (*cmd_argv3 != '#') && (*cmd_argv3 != '\n') && (*cmd_argv3 != '\0') ) {
					arg3 = strtol(cmd_argv3, NULL, 16);
				}
			}
		}
		if ( cmd[1] == '2' ) {	// OV7740
			if ( cmd[0] == '5' ) {
				fprintf(stderr, "Read  OV7740    Reg=0x%.2X, ", arg1);
				fprintf(stderr, "Data = 0x%.2X \n", I2CGetByte(VideoIn, (__u8) arg1));
			} else if ( cmd[0] == '4' ) {
				fprintf(stderr, "Write OV7740    Reg=0x%.2X, Data = 0x%.2X", arg1, arg2);
				I2CSetByte(VideoIn, (__u8) arg1, (__u8) arg2);
				fprintf(stderr, "\n");
			}
		}
	}

	fclose(fp);
	return 0;
}

/* Main */
int main(int argc, char *argv[])
{
char	*device = DEFAULT_UVC_DEVICE;
char	*sensor_command_file = NULL;
int	format = V4L2_PIX_FMT_MJPEG;	// always use MJPEG frame format
int	daemon;
int	fd, saved_pid;
int	command_flag=0, dump_mode=0;
__u32	event_flag;
int	addr, reg_data;

	addr = reg_data = -1;
	while(1) {
		int option_index = 0, c=0;
		static struct option long_options[] = \
					{
					{"h", no_argument, 0, 0},
					{"dump", no_argument, 0, 0},
					{"f", required_argument, 0, 0},
					{"a", required_argument, 0, 0},
					{"d", required_argument, 0, 0},
					{"m", required_argument, 0, 0},
					{0, 0, 0, 0}
					};

		c = getopt_long_only(argc, argv, "", long_options, &option_index);

		/* no more options to parse */
		if (c == -1) break;

		/* unrecognized option */
		if(c=='?'){ help(argv[0]); return 0; }

		switch (option_index) {
		case 0:	/* h, help */
			help(argv[0]);
			return 0;
			break;
		case 1:	/* dump */
			command_flag |= COMMAND_FLAG_DUMP_SENSOR;
			break;
		case 2:	/* f, input sensor command filename */
			sensor_command_file = strdup(optarg);
			command_flag |= COMMAND_FLAG_SENSOR_INIT;
			break;
		case 3:	/* -a, address */
			addr = strtol(optarg, NULL, 16);
			command_flag |= COMMAND_FLAG_SENSOR_ACCESS;
			break;
		case 4:	/* -d, data */
			reg_data = strtol(optarg, NULL, 16);
			break;
		case 5:	/* -m, dump mode */
			dump_mode = atoi(optarg);
			break;
		default:
			help(argv[0]);
			return 0;
		} // switch (option_index)
	} // while(1)

	VideoIn = (struct vdIn *) calloc(1, sizeof(struct vdIn));

	/* open video device and prepare data structure */
	if (InitVideoIn(VideoIn, 0, device, format, 1) < 0) {
		fprintf(stderr, "init_VideoIn failed\n");
		exit(1);
	}
	if ( (command_flag & COMMAND_FLAG_SENSOR_INIT) != 0 ) {
		SensorCommand(sensor_command_file);
	}
	if ( (command_flag & COMMAND_FLAG_SENSOR_ACCESS) != 0 ) {
		if ( reg_data == -1 ) {
			fprintf(stderr, "Read  OV7740    Reg=0x%.2X, ", addr);
			fprintf(stderr, "Data = 0x%.2X \n", I2CGetByte(VideoIn, (__u8) addr));
		} else {
			fprintf(stderr, "Write OV7740    Reg=0x%.2X, Data = 0x%.2X\n", addr, reg_data);
			I2CSetByte(VideoIn, (__u8) addr, (__u8) reg_data);
		}
	}
	if ( (command_flag & COMMAND_FLAG_DUMP_SENSOR) != 0 ) {
		DumpAllRegister(VideoIn, dump_mode);
	}
	close_v4l2(VideoIn);
	return 0;
}

