#ifndef	IP297X_INCLUDE
#define	IP297X_INCLUDE	1

#define	LENS_MODEL_DCS930	1
#define	LENS_MODEL_TVIP551W	2
#define	LENS_MODEL_TVIP551WI	3
#define	LENS_MODEL_TVIP651W	LENS_MODEL_TVIP551W
#define	LENS_MODEL_TVIP651WI	LENS_MODEL_TVIP551WI
char	LensModel[4][16]={"Unknown", "DCS930", "TVIP551W/651W", "TVIP551WI/651WI"};

/*==============================================================================*/
#define	LENS_MODEL		(LENS_MODEL_DCS930)

#ifdef MODEL_TVIP551W
#undef	LENS_MODEL
#define	LENS_MODEL		(LENS_MODEL_TVIP551W)
#endif	// #ifdef MODEL_TVIP551W

#ifdef MODEL_TVIP551WI
#undef	LENS_MODEL
#define	LENS_MODEL		(LENS_MODEL_TVIP551WI)
#endif	// #ifdef MODEL_TVIP551WI

#ifdef MODEL_TVIP651W
#undef	LENS_MODEL
#define	LENS_MODEL		(LENS_MODEL_TVIP651W)
#endif	// #ifdef MODEL_TVIP651W

#ifdef MODEL_TVIP651WI
#undef	LENS_MODEL
#define	LENS_MODEL		(LENS_MODEL_TVIP651WI)
#endif	// #ifdef MODEL_TVIP651WI

/*==============================================================================*/
#define	IP297X_DEBUG_MOTION	0x01
#define	IP297X_DEBUG_EXTENSION	0x02
#define	IP297X_DEBUG_ALL	-1
#define	IP297X_DEBUG		(0)

/*iP297x Extension_Unit Control Selectors */
#define EU_RIGISTER_CONTROL		0x01
#define EU_EEPROM_CONTROL		0x02
#define EU_RAM_CONTROL			0x03
#define EU_I2C_CONTROL			0x04

#define EXTENSION_RIGISTER_READ		0x01
#define EXTENSION_RIGISTER_WRITE	0x00
#define EXTENSION_I2C_READ		0x80
#define EXTENSION_I2C_WRITE		0x00

#define EXTU_SIF_8BIT			0x01
#define EXTU_SIF_16BIT			0x02

#define IP297X_EXTENSION_UNIT_ID	4
#define UVC_GUID_UVC_EXTENSION_UNIT_IP2970 \
	{0x3a, 0xab, 0x91, 0x99, 0xef, 0xb2, 0xc9, 0x48, \
	 0x8f, 0xe9, 0x8f, 0xe3, 0x63, 0x47, 0x71, 0xd0}

#define I2C_OV_SENSOR_ID		0x21

struct	IP297X_REGISTER	{
	__u8	Address;
	__u8	Data;
};

#define BRIGHTNESS_MIN		0x00
#define BRIGHTNESS_MAX		0xb0
#define BRIGHTNESS_DEFAULT	0x0c

#define CONTRAST_MIN		0x10
#define CONTRAST_MAX		0x48
#define CONTRAST_DEFAULT	0x1f

#define SATURATION_MIN		0x01
#define SATURATION_MAX		0xff
#define SATURATION_DEFAULT	0x46

#define END_REG                 0xFF
struct	IP297X_REGISTER	IP297x_Init[] = {
{0xa2, 0x11}, {0xa1, 0x03}, {0xa4, 0x33}, {0xac, 0x12}, {0xB6, 0x50}, {0xB7, 0x30}, {0xB8, 0x84}, {0xD0, 0x7D},
{0xD1, 0x02}, {0xD2, 0x02}, {0xD3, 0x30}, {0xD4, 0x80}, {0xD5, 0x00}, {0xD6, 0x14}, {0xD7, 0x01}, {0xD8, 0x9A},
{0xD9, 0x00}, {0xDA, 0x5A}, {0xDB, 0x01}, {0xDC, 0x00}, {0xDD, 0x08}, {0xDE, 0x20}, {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_Init[] = {
#if ( LENS_MODEL == LENS_MODEL_DCS930 )
{0x12, 0x80}, {0x13, 0x00}, {0x11, 0x01}, {0x12, 0x00}, {0xd5, 0x10}, {0x0c, 0x02}, {0x0d, 0x34}, {0x17, 0x25},
{0x18, 0xa0}, {0x19, 0x03}, {0x1a, 0xf0}, {0x1b, 0x89}, {0x1e, 0x13}, {0x22, 0x03}, {0x28, 0x02}, {0x29, 0x17},
{0x2b, 0xf8}, {0x2c, 0x01}, {0x31, 0xa0}, {0x32, 0xf0}, {0x33, 0xc4}, {0x35, 0x05}, {0x36, 0x3f}, {0x04, 0x60},
{0x27, 0x80}, {0x3d, 0x0f}, {0x3e, 0x82}, {0x3f, 0x40}, {0x40, 0x7f}, {0x41, 0x6a}, {0x42, 0x29}, {0x44, 0xe5},
{0x45, 0x41}, {0x47, 0x42}, {0x48, 0x00}, {0x49, 0x61}, {0x4a, 0xa1}, {0x4b, 0x46}, {0x4c, 0x18}, {0x4d, 0x50},
{0x4e, 0x13}, {0x64, 0x00}, {0x67, 0x88}, {0x68, 0x1a}, {0x14, 0x38}, {0x24, 0x3c}, {0x25, 0x30}, {0x26, 0x72},
{0x50, 0x97}, {0x51, 0x7e}, {0x52, 0x00}, {0x53, 0x00}, {0x20, 0x00}, {0x21, 0x23}, {0x38, 0x14}, {0xe9, 0x00},
{0x56, 0x55}, {0x57, 0xff}, {0x58, 0xff}, {0x59, 0xff}, {0x5f, 0x04}, {0x13, 0xff}, {0x81, 0x3f}, {0x82, 0x32},
{0x83, 0x03}, {0x38, 0x11}, {0x84, 0x70}, {0x85, 0x00}, {0x86, 0x03}, {0x87, 0x01}, {0x88, 0x05}, {0x89, 0x30},
{0x80, 0x7F}, {0x8C, 0x01}, {0x8a, 0x40}, {0x8b, 0xf0}, {0x8d, 0xf0}, {0x8f, 0x85}, {0x8e, 0xc2}, {0x92, 0x01},
{0x90, 0x40}, {0x91, 0xf0}, {0x93, 0xb0}, {0x95, 0x85}, {0x94, 0xc2}, {0x98, 0x01}, {0x96, 0x40}, {0x97, 0xf0},
{0x99, 0xb4}, {0x9b, 0x85}, {0x9a, 0xc2}, {0x9c, 0x08}, {0x9d, 0x12}, {0x9e, 0x23}, {0x9f, 0x45}, {0xa0, 0x55},
{0xa1, 0x64}, {0xa2, 0x72}, {0xa3, 0x7f}, {0xa4, 0x8b}, {0xa5, 0x95}, {0xa6, 0xa7}, {0xa7, 0xb5}, {0xa8, 0xcb},
{0xa9, 0xdd}, {0xaa, 0xec}, {0xab, 0x1a}, {0xce, 0x69}, {0xcf, 0x5B}, {0xd0, 0x0D}, {0xd1, 0x15}, {0xd2, 0x90},
{0xd3, 0xA6}, {0xd4, 0x1e}, {0x5a, 0x24}, {0x5b, 0x1f}, {0x5c, 0x88}, {0x5d, 0x60}, {0xac, 0x6e}, {0xbe, 0xff},
{0xbf, 0x00}, {0x70, 0x00}, {0x71, 0x34}, {0x74, 0x28}, {0x75, 0x98}, {0x76, 0x00}, {0x77, 0x08}, {0x78, 0x01},
{0x79, 0xc2}, {0x7d, 0x02}, {0x7a, 0x9c}, {0x7b, 0x40}, {0xEC, 0x00}, {0x7c, 0x0c}, {0x24, 0x50}, {0x25, 0x40},
{0x26, 0x82}, {0xcc, 0x42}, {0x85, 0x02}, {0x86, 0x03}, {0x14, 0x28}, {0x15, 0x98}, {0xda, 0x06}, {0xdd, 0x44},
{0xde, 0x44}, {0x5d, 0x00}, {0x67, 0x90}, {END_REG, 0x00}};
#endif
/**********************************/
#if ( LENS_MODEL == LENS_MODEL_TVIP551W )
// DVP(Digital Video Port)
{0x12, 0x80}, {0x13, 0x00}, {0x11, 0x01}, {0x12, 0x00}, {0xd5, 0x10}, {0x0c, 0x02}, {0x0d, 0x34},
{0x17, 0x25}, {0x18, 0xa0}, {0x19, 0x03}, {0x1a, 0xf0}, {0x1b, 0x89}, {0x1e, 0x13}, {0x22, 0x03},
{0x28, 0x02}, {0x29, 0x17}, {0x2b, 0xf8}, {0x2c, 0x01}, {0x31, 0xa0}, {0x32, 0xf0}, {0x33, 0xc4},
// Analog setting/BLC(Black Level Calibration)
{0x3a, 0xb4}, {0x36, 0x3f}, {0x04, 0x60}, {0x27, 0x80}, {0x3d, 0x0f}, {0x3e, 0x82}, {0x3f, 0x40},
{0x40, 0x7f}, {0x41, 0x6a}, {0x42, 0x29}, {0x44, 0xe5}, {0x45, 0x41}, {0x47, 0x42}, {0x48, 0x00},
{0x49, 0x61}, {0x4a, 0xa1}, {0x4b, 0x46}, {0x4c, 0x18}, {0x4d, 0x50}, {0x4e, 0x13}, {0x64, 0x00},
{0x67, 0x88}, {0x68, 0x1a},
// AEC/AGC 50Hz/60Hz
{0x14, 0x38}, {0x24, 0x3c}, {0x25, 0x30}, {0x26, 0x72}, {0x50, 0x97}, {0x51, 0x7e}, {0x52, 0x00},
{0x53, 0x00}, {0x20, 0x00}, {0x21, 0x23}, {0x38, 0x14}, {0xe9, 0x00}, {0x56, 0x55}, {0x57, 0xff},
{0x58, 0xff}, {0x59, 0xff}, {0x5f, 0x04}, {0x13, 0xff},
// ISP controls, sharpness, denoise
{0x80, 0x7d}, {0x81, 0x3f}, {0x82, 0x32}, {0x83, 0x03}, {0x38, 0x11}, {0x84, 0x70}, {0x85, 0x01},
{0x86, 0x04}, {0x87, 0x01}, {0x88, 0x05}, {0x89, 0x30},
// Gamma
{0x9c, 0x08}, {0x9d, 0x12}, {0x9e, 0x23}, {0x9f, 0x45}, {0xa0, 0x55}, {0xa1, 0x64}, {0xa2, 0x72},
{0xa3, 0x7f}, {0xa4, 0x8b}, {0xa5, 0x95}, {0xa6, 0xa7}, {0xa7, 0xb5}, {0xa8, 0xcb}, {0xa9, 0xdd},
{0xaa, 0xec}, {0xab, 0x1a},
// CTX (color matrix)
{0xce, 0x78}, {0xcf, 0x6c}, {0xd0, 0x0a}, {0xd1, 0x0c}, {0xd2, 0x84}, {0xd3, 0x92}, {0xd4, 0x1e},
// auto color stauration adjust
{0x5a, 0x24}, {0x5b, 0x1f}, {0x5c, 0x88}, {0x5d, 0x60},
// AWB/WBC (reg 0xAC-0xCA)
{0xac, 0x6e}, {0xbe, 0xff}, {0xbf, 0x00},
//50/60Hz auto detection is XCLK dependent , the following is based on XCLK = 24MHz
{0x70, 0x00}, {0x71, 0x34}, {0x74, 0x28}, {0x75, 0x98}, {0x76, 0x00}, {0x77, 0x08}, {0x78, 0x01},
{0x79, 0xc2}, {0x7d, 0x02}, {0x7a, 0x9c}, {0x7b, 0x40}, {0xec, 0x02}, {0x7c, 0x0c},
// Lens Correction
{0x80, 0x7f}, {0x8a, 0x40}, {0x8b, 0xf0}, {0x8c, 0x01}, {0x8d, 0x48}, {0x8e, 0xc2}, {0x8f, 0x85},
{0x90, 0x40}, {0x91, 0xf0}, {0x92, 0x01}, {0x93, 0x3d}, {0x94, 0xc2}, {0x95, 0x85}, {0x96, 0x40},
{0x97, 0xf0}, {0x98, 0x01}, {0x99, 0x30}, {0x9a, 0xc2}, {0x9b, 0x85},
// edge
{0xcc, 0x42}, {0xcd, 0x04},
//
{0x14, 0x38},	// 0x08 => 0x38, AGC gain ceiling = 16x
{0x15, 0x94},	// 0x00 => 0x94, night mod inserting frame up to 1 frame, trigger point = 4x gain
{0x4a, 0xa9},	// 0xa1 => 0xa9, bypass internal regulator
{0xda, 0x06},	// 0x04 => 0x06, enable Contrast & Saturation
{END_REG, 0x00}};
#endif	// ...( LENS_MODEL == LENS_MODEL_TVIP551W )
/**********************************/
#if ( LENS_MODEL == LENS_MODEL_TVIP551WI )
// DVP(Digital Video Port)
{0x12, 0x80}, {0x13, 0x00}, {0x11, 0x01}, {0x12, 0x00}, {0xd5, 0x10}, {0x0c, 0x02}, {0x0d, 0x34},
{0x17, 0x25}, {0x18, 0xa0}, {0x19, 0x03}, {0x1a, 0xf0}, {0x1b, 0x89}, {0x1e, 0x13}, {0x22, 0x03},
{0x28, 0x02}, {0x29, 0x17}, {0x2b, 0xf8}, {0x2c, 0x01}, {0x31, 0xa0}, {0x32, 0xf0}, {0x33, 0xc4},
// Analog setting/BLC(Black Level Calibration)
{0x3a, 0xb4}, {0x36, 0x3f}, {0x04, 0x60}, {0x27, 0x80}, {0x3d, 0x0f}, {0x3e, 0x82}, {0x3f, 0x40},
{0x40, 0x7f}, {0x41, 0x6a}, {0x42, 0x29}, {0x44, 0xe5}, {0x45, 0x41}, {0x47, 0x42}, {0x48, 0x00},
{0x49, 0x61}, {0x4a, 0xa1}, {0x4b, 0x46}, {0x4c, 0x18}, {0x4d, 0x50}, {0x4e, 0x13}, {0x64, 0x00},
{0x67, 0x88}, {0x68, 0x1a},
// AEC/AGC 50Hz/60Hz
{0x14, 0x38}, {0x24, 0x3c}, {0x25, 0x30}, {0x26, 0x72}, {0x50, 0x97}, {0x51, 0x7e}, {0x52, 0x00},
{0x53, 0x00}, {0x20, 0x00}, {0x21, 0x23}, {0x38, 0x14}, {0xe9, 0x00}, {0x56, 0x55}, {0x57, 0xff},
{0x58, 0xff}, {0x59, 0xff}, {0x5f, 0x04}, {0x13, 0xff},
// ISP controls, sharpness, denoise
{0x80, 0x7d}, {0x81, 0x3f}, {0x82, 0x32}, {0x83, 0x03}, {0x38, 0x11}, {0x84, 0x70}, {0x85, 0x01},
{0x86, 0x04}, {0x87, 0x01}, {0x88, 0x05}, {0x89, 0x30},
// Gamma
{0x9c, 0x08}, {0x9d, 0x12}, {0x9e, 0x23}, {0x9f, 0x45}, {0xa0, 0x55}, {0xa1, 0x64}, {0xa2, 0x72},
{0xa3, 0x7f}, {0xa4, 0x8b}, {0xa5, 0x95}, {0xa6, 0xa7}, {0xa7, 0xb5}, {0xa8, 0xcb}, {0xa9, 0xdd},
{0xaa, 0xec}, {0xab, 0x1a},
// CTX (color matrix)
{0xce, 0x76}, {0xcf, 0x6c}, {0xd0, 0x0a}, {0xd1, 0x0c}, {0xd2, 0x84}, {0xd3, 0x91}, {0xd4, 0x1e},
// auto color stauration adjust
{0x5a, 0x24}, {0x5b, 0x1f}, {0x5c, 0x88}, {0x5d, 0x60},
// AWB/WBC (reg 0xAC-0xCA)
{0xac, 0x6e}, {0xbe, 0xff}, {0xbf, 0x00},
//50/60Hz auto detection is XCLK dependent , the following is based on XCLK = 24MHz
{0x70, 0x00}, {0x71, 0x34}, {0x74, 0x28}, {0x75, 0x98}, {0x76, 0x00}, {0x77, 0x08}, {0x78, 0x01},
{0x79, 0xc2}, {0x7d, 0x02}, {0x7a, 0x9c}, {0x7b, 0x40}, {0xec, 0x02}, {0x7c, 0x0c},
// Lens Correction
{0x80, 0x7f}, {0x8a, 0x40}, {0x8b, 0xf0}, {0x8c, 0x01}, {0x8d, 0x40}, {0x8e, 0xc2}, {0x8f, 0x85},
{0x90, 0x40}, {0x91, 0xf0}, {0x92, 0x01}, {0x93, 0x3d}, {0x94, 0xc2}, {0x95, 0x85}, {0x96, 0x40},
{0x97, 0xf0}, {0x98, 0x01}, {0x99, 0x30}, {0x9a, 0xc2}, {0x9b, 0x85},
// edge
{0xcc, 0x42}, {0xcd, 0x04},
//
{0x14, 0x38},	// 0x08 => 0x38, AGC gain ceiling = 16x
{0x15, 0x94},	// 0x00 => 0x94, night mod inserting frame up to 1 frame, trigger point = 4x gain
{0x4a, 0xa9},	// 0xa1 => 0xa9, bypass internal regulator
{0xda, 0x06},	// 0x04 => 0x06, enable Contrast & Saturation
{END_REG, 0x00}};
#endif	// ...( LENS_MODEL == LENS_MODEL_TVIP551WI )
struct	IP297X_REGISTER	IP297x_Init_QQVGA[] = {
{0xac, 0x34}, {0xa2, 0x10}, {0xa1, 0x03}, {0xb6, 0x31}, {0xb7, 0x30}, {0xb8, 0x84},  {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_Init_QQVGA[] = {
{0x31, 0x50}, {0x32, 0x78}, {0x82, 0x3f}, {0x14, 0x28}, {0x26, 0x72}, {0x50, 0x97}, {0x51, 0x7e}, {0x52, 0x00},
{0xe2, 0x20}, {0xcc, 0x41}, {END_REG, 0x00}};
struct	IP297X_REGISTER	IP297x_Init_QVGA[] = {
{0xac, 0x30}, {0xa2, 0x10}, {0xa1, 0x03}, {0xb6, 0x53}, {0xb7, 0x30}, {0xb8, 0xa3}, {0xD0, 0x7D}, {0xD1, 0x02},
{0xD2, 0x02}, {0xD3, 0x30}, {0xD4, 0x80}, {0xD5, 0x00}, {0xD6, 0x14}, {0xD7, 0x01}, {0xD8, 0x9A}, {0xD9, 0x00},
{0xDA, 0x5A}, {0xDB, 0x01}, {0xDC, 0x00}, {0xDD, 0x08}, {0xDE, 0x20}, {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_Init_QVGA[] = {
{0x31, 0x50}, {0x32, 0x78}, {0x82, 0x3f}, {0x14, 0x28}, {0x26, 0x72}, {0x50, 0x97}, {0x51, 0x7e}, {0x52, 0x00},
{0xe2, 0x20}, {0xcc, 0x46}, {END_REG, 0x00}};
struct	IP297X_REGISTER	IP297x_Init_VGA_10[] = {
{0xa2, 0x12}, {0xa1, 0x03}, {0xa4, 0x33}, {0xac, 0x12}, {0xB6, 0x30}, {0xB7, 0x30}, {0xB8, 0x84}, {0xD0, 0x7D},
{0xD1, 0x02}, {0xD2, 0x02}, {0xD3, 0x30}, {0xD4, 0x80}, {0xD5, 0x00}, {0xD6, 0x14}, {0xD7, 0x01}, {0xD8, 0x9A},
{0xD9, 0x00}, {0xDA, 0x5A}, {0xDB, 0x01}, {0xDC, 0x00}, {0xDD, 0x08}, {0xDE, 0x20}, {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_Init_VGA_10[] = {
{0x31, 0xA0}, {0x32, 0xF0}, {0x82, 0x32}, {0x14, 0x28}, {0x26, 0x72}, {0x50, 0x32}, {0x51, 0x2A}, {0x52, 0x00},
{0xe2, 0x20}, {0xcc, 0x43}, {END_REG, 0x00}};
struct	IP297X_REGISTER	IP297x_Init_VGA_15[] = {
{0xa2, 0x11}, {0xa1, 0x03}, {0xa4, 0x33}, {0xac, 0x12}, {0xB6, 0x30}, {0xB7, 0x30}, {0xB8, 0x84}, {0xD0, 0x7D},
{0xD1, 0x02}, {0xD2, 0x02}, {0xD3, 0x30}, {0xD4, 0x80}, {0xD5, 0x00}, {0xD6, 0x14}, {0xD7, 0x01}, {0xD8, 0x9A},
{0xD9, 0x00}, {0xDA, 0x5A}, {0xDB, 0x01}, {0xDC, 0x00}, {0xDD, 0x08}, {0xDE, 0x20}, {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_Init_VGA_15[] = {
{0x31, 0xA0}, {0x32, 0xF0}, {0x82, 0x32}, {0x14, 0x28}, {0x26, 0x72}, {0x50, 0x4c}, {0x51, 0x3F}, {0x52, 0x00},
{0xe2, 0x20}, {0xcc, 0x43}, {END_REG, 0x00}};
struct	IP297X_REGISTER	IP297x_Init_VGA_20[] = {
{0xa2, 0x02}, {0xa1, 0x03}, {0xa4, 0x33}, {0xac, 0x12}, {0xB6, 0x30}, {0xB7, 0x30}, {0xB8, 0x84}, {0xD0, 0x7D},
{0xD1, 0x02}, {0xD2, 0x02}, {0xD3, 0x30}, {0xD4, 0x80}, {0xD5, 0x00}, {0xD6, 0x14}, {0xD7, 0x01}, {0xD8, 0x9A},
{0xD9, 0x00}, {0xDA, 0x5A}, {0xDB, 0x01}, {0xDC, 0x00}, {0xDD, 0x08}, {0xDE, 0x20}, {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_Init_VGA_20[] = {
{0x31, 0xA0}, {0x32, 0xF0}, {0x82, 0x32}, {0x14, 0x28}, {0x26, 0x72}, {0x50, 0x65}, {0x51, 0x54}, {0x52, 0x00},
{0xe2, 0x20}, {0xcc, 0x43}, {END_REG, 0x00}};
struct	IP297X_REGISTER	IP297x_Init_VGA_20_BadQuality[] = {
{0xa2, 0x02}, {0xa1, 0x03}, {0xa4, 0x33}, {0xac, 0x12}, {0xB6, 0x30}, {0xB7, 0x30}, {0xB8, 0x84}, {0xD0, 0x7D},
{0xD1, 0x02}, {0xD2, 0x02}, {0xD3, 0x30}, {0xD4, 0x40}, {0xD5, 0x00}, {0xD6, 0x88}, {0xD7, 0x00}, {0xD8, 0x60},
{0xD9, 0x00}, {0xDA, 0xA0}, {0xDB, 0x00}, {0xDC, 0x00}, {0xDD, 0x08}, {0xDE, 0x20}, {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_Init_VGA_20_BadQuality[] = {
{0x31, 0xA0}, {0x32, 0xF0}, {0x82, 0x32}, {0x14, 0x28}, {0x26, 0x72}, {0x50, 0x65}, {0x51, 0x54}, {0x52, 0x00},
{0xe2, 0x20}, {0xcc, 0x43}, {END_REG, 0x00}};
struct	IP297X_REGISTER	IP297x_Init_VGA_30[] = {
{0xa2, 0x10}, {0xa1, 0x03}, {0xa4, 0x33}, {0xac, 0x12}, {0xB6, 0x30}, {0xB7, 0x30}, {0xB8, 0x84}, {0xD0, 0x7D},
{0xD1, 0x02}, {0xD2, 0x02}, {0xD3, 0x30}, {0xD4, 0x40}, {0xD5, 0x00}, {0xD6, 0x88}, {0xD7, 0x00}, {0xD8, 0x60},
{0xD9, 0x00}, {0xDA, 0xA0}, {0xDB, 0x00}, {0xDC, 0x00}, {0xDD, 0x08}, {0xDE, 0x20}, {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_Init_VGA_30[] = {
{0x31, 0xA0}, {0x32, 0xF0}, {0x82, 0x32}, {0x14, 0x28}, {0x26, 0x72}, {0x50, 0x97}, {0x51, 0x7E}, {0x52, 0x00},
{0xe2, 0x20}, {0xcc, 0x43}, {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_AE_WEIGHTING_QVGA[] = {
{0x38, 0x15}, {0xE9, 0x50}, {0x38, 0x16}, {0xE9, 0x78}, {0x38, 0x12}, {0xE9, 0x00}, {0x38, 0x13}, {0xE9, 0x00},
{0x38, 0x14}, {0xE9, 0x20}, {0x53, 0x02}, {0x56, 0xff}, {0x57, 0xff}, {0x58, 0xff}, {0x59, 0xff}, {END_REG, 0x00}};
struct	IP297X_REGISTER	OV7740_AE_WEIGHTING_VGA[] = {
{0x38, 0x15}, {0xE9, 0x50}, {0x38, 0x16}, {0xE9, 0x78}, {0x38, 0x12}, {0xE9, 0x00}, {0x38, 0x13}, {0xE9, 0x00},
{0x38, 0x14}, {0xE9, 0x20}, {0x53, 0x02}, {0x56, 0x55}, {0x57, 0x55}, {0x58, 0x55}, {0x59, 0x55}, {END_REG, 0x00}};

#define IP297X_MOTION_FLAG_OFFSET	4
#define	IP297X_MOTION_FRAME_INTERVAL	6
#define	IP297X_MOTION_SENSITIVITY_MIN	1
#define	IP297X_MOTION_SENSITIVITY_MAX	101

struct motion_control {	// default = 0x01, 0x10, 0x10, 0x0A, 0xFFFFFFFF, 0x01, 0x00
	__u8	MotionFunSet;	// b0:enable
	__u8	FrameInterval;
	__u8	MotionYThd;
	__u8	MotionBThd;
	__u32	MotionBlockSet;
	__u8	FrameCount;
	__u8	MotionBlockNo;
}  __attribute__ ((packed));

struct ip297x_rw_register {
	__u8	Function;
	__u8	Address;
	__u8	Data;
} __attribute__ ((packed));

struct ip297x_i2c {
	__u8	Status;
	__u8	SensorID;;
	__u8	Address;
	__u8	DataL;
	__u8	DataH;
}  __attribute__ ((packed));

static struct uvc_xu_control_info ip297x_uvc_ctrls[] = {
	{
		.entity		= UVC_GUID_UVC_EXTENSION_UNIT_IP2970,
		.selector	= EU_RIGISTER_CONTROL,
		.index		= 1,
		.size		= sizeof(struct ip297x_rw_register),
		.flags		= UVC_CONTROL_SET_CUR | UVC_CONTROL_GET_CUR,
	},
	{
		.entity		= UVC_GUID_UVC_EXTENSION_UNIT_IP2970,
		.selector	= EU_RAM_CONTROL,
		.index		= 2,
		.size		= sizeof(struct motion_control),
		.flags		= UVC_CONTROL_SET_CUR | UVC_CONTROL_GET_CUR,
	},
	{
		.entity		= UVC_GUID_UVC_EXTENSION_UNIT_IP2970,
		.selector	= EU_I2C_CONTROL,
		.index		= 3,
		.size		= sizeof(struct ip297x_i2c),
		.flags		= UVC_CONTROL_SET_CUR | UVC_CONTROL_GET_CUR,
	},
#if 0
	{
		.entity		= UVC_GUID_UVC_EXTENSION_UNIT_IP2970,
		.selector	= EU_EEPROM_CONTROL,
		.index		= 4,
		.size		= 10,
		.flags		= UVC_CONTROL_SET_CUR | UVC_CONTROL_GET_CUR,
	},
#endif
	{
		.entity		= 0,
		.selector	= 0,
		.index		= 0,
		.size		= 0,
		.flags		= 0,
	},
};

__u8 IP297xRegisterGet(struct vdIn *Video, __u8 Address)
{
struct ip297x_rw_register	reg_data;
struct uvc_xu_control		xctrl;
int				ret_val;

	reg_data.Function = EXTENSION_RIGISTER_READ;
	reg_data.Address = Address;
	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_RIGISTER_CONTROL;
	xctrl.size = sizeof(struct ip297x_rw_register);
	xctrl.data = (__u8 *) &reg_data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_SET, &xctrl);
	if ( ret_val < 0 )	fprintf(stderr, "IP297xRegisterRead error 1: %i\n", ret_val);
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_GET, &xctrl);
	if ( ret_val < 0 )	fprintf(stderr, "IP297xRegisterRead error 2: %i\n", ret_val);
	return reg_data.Data;
}

int IP297xRegisterSet(struct vdIn *Video, __u8 Address, __u8 Data)
{
struct ip297x_rw_register	reg_data;
struct uvc_xu_control		xctrl;
int				ret_val;

	reg_data.Function = EXTENSION_RIGISTER_WRITE;
	reg_data.Address = Address;
	reg_data.Data = Data;
	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_RIGISTER_CONTROL;
	xctrl.size = sizeof(struct ip297x_rw_register);
	xctrl.data = (__u8 *) &reg_data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_SET, &xctrl);
	if ( ret_val < 0 )	fprintf(stderr, "IP297xRegisterWrite error : %i\n", ret_val);
	return 0;
}

int IP297xSetRegisters(struct vdIn *Video, struct IP297X_REGISTER *CamRegister)
{
int	idx=0;

	while( CamRegister[idx].Address != END_REG ) {
//		printf("iP2970 Address= %2x, Data = %2x\n",CamRegister[idx].Address, CamRegister[idx].Data);
		IP297xRegisterSet(Video, CamRegister[idx].Address, CamRegister[idx].Data);
		idx++;
	}
}

__u8 IP297xI2CGetByte(struct vdIn *Video, __u8 SensorID, __u8 Address)
{
struct ip297x_i2c		i2c_data;
struct uvc_xu_control		xctrl;
int				ret_val;

	i2c_data.Status = EXTENSION_I2C_READ + EXTU_SIF_8BIT;
	i2c_data.SensorID = SensorID;
	i2c_data.Address = Address;
	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_I2C_CONTROL;
	xctrl.size = sizeof(struct ip297x_i2c);
	xctrl.data = (__u8 *) &i2c_data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_SET, &xctrl);
	if ( ret_val < 0 )	fprintf(stderr, "IP297xI2CGet error 1: %i\n", ret_val);
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_GET, &xctrl);
	if ( (ret_val < 0) || (i2c_data.Status == 0) ) {
		fprintf(stderr, "IP297xI2CGetByte error : %i\n", ret_val);
	}
	return i2c_data.DataL;
}

int IP297xI2CSetByte(struct vdIn *Video, __u8 SensorID, __u8 Address, __u8 Data)
{
struct ip297x_i2c		i2c_data;
struct uvc_xu_control		xctrl;
int				ret_val;

	i2c_data.Status = EXTENSION_I2C_WRITE + EXTU_SIF_8BIT;
	i2c_data.SensorID = SensorID;
	i2c_data.Address = Address;
	i2c_data.DataL = Data;
	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_I2C_CONTROL;
	xctrl.size = sizeof(struct ip297x_i2c);
	xctrl.data = (__u8 *) &i2c_data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_SET, &xctrl);
	if ( (ret_val < 0) || (i2c_data.Status == 0) ) {
		fprintf(stderr, "IP297xI2CSetByte error : %i\n", ret_val);
	}
	return 0;
}

int IP297xI2CSetByteRegisters(struct vdIn *Video, __u8 SensorID, struct IP297X_REGISTER *CamRegister)
{
int	idx=0;

	while( CamRegister[idx].Address != END_REG ) {
//		printf("I2C Address= %2x, Data = %2x\n",CamRegister[idx].Address, CamRegister[idx].Data);
		IP297xI2CSetByte(Video, SensorID, CamRegister[idx].Address, CamRegister[idx].Data);
		idx++;
	}
}

__u16 IP297xI2CGetWord(struct vdIn *Video, __u8 SensorID, __u8 Address)
{
struct ip297x_i2c		i2c_data;
struct uvc_xu_control		xctrl;
int				ret_val;
__u16				word_data;

	i2c_data.Status = EXTENSION_I2C_READ + EXTU_SIF_16BIT;
	i2c_data.SensorID = SensorID;
	i2c_data.Address = Address;
	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_I2C_CONTROL;
	xctrl.size = sizeof(struct ip297x_i2c);
	xctrl.data = (__u8 *) &i2c_data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_SET, &xctrl);
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_GET, &xctrl);
	if ( (ret_val < 0) || (i2c_data.Status == 0) ) {
		fprintf(stderr, "IP297xI2CGetWord error : %i\n", ret_val);
	}
	word_data = ((i2c_data.DataL) << 8) + i2c_data.DataH;
	return word_data;
}

int IP297xI2CSetWord(struct vdIn *Video, __u8 SensorID, __u8 Address, __u16 Data)
{
struct ip297x_i2c		i2c_data;
struct uvc_xu_control		xctrl;
int				ret_val;

	i2c_data.Status = EXTENSION_I2C_WRITE + EXTU_SIF_16BIT;
	i2c_data.SensorID = SensorID;
	i2c_data.Address = Address;
	i2c_data.DataL = (__u8)(Data & 0x00ff);
	i2c_data.DataH = (__u8)(Data >> 8);
	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_I2C_CONTROL;
	xctrl.size = sizeof(struct ip297x_i2c);
	xctrl.data = (__u8 *) &i2c_data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_SET, &xctrl);
	if ( (ret_val < 0) || (i2c_data.Status == 0) ) {
		fprintf(stderr, "IP297xI2CSetWord error : %i\n", ret_val);
	}
	return 0;
}

int IP297xMotionGet(struct vdIn *Video, __u8 *Data)
{
struct uvc_xu_control	xctrl;
int			ret_val;

	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_RAM_CONTROL;
	xctrl.size = sizeof(struct motion_control);
	xctrl.data = Data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_GET, &xctrl);
	if ( ret_val < 0 )	fprintf(stderr, "IP297xMotionRead error : %i\n", ret_val);
}

int IP297xMotionSet(struct vdIn *Video, __u8 *Data)
{
struct uvc_xu_control	xctrl;
int			ret_val;

	xctrl.unit = IP297X_EXTENSION_UNIT_ID;
	xctrl.selector = EU_RAM_CONTROL;
	xctrl.size = sizeof(struct motion_control);
	xctrl.data = Data;
	ret_val = ioctl(Video->fd, UVCIOC_CTRL_SET, &xctrl);
	if ( ret_val < 0 )	fprintf(stderr, "IP297xMotionRead error : %i\n", ret_val);
}

int IP2970SetGrayImage(struct vdIn *Video, int Flag)
{
__u8	sensor_data;

	/* gray enable/disable , Flag=0(day mode,color), =1(night mode, gray image) */
	sensor_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0xDA);
	if ( Flag )	sensor_data |= 0x20;
	else		sensor_data &= ~0x20;
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0xDA, sensor_data);
	return 0;
}

int AddExtensionUnitControls(struct vdIn *Video)
{
struct uvc_xu_control_info	*xinfo;
struct ip297x_rw_register	*p_reg;
int	ret_val, idx, sensor_id;
__u8	tmp_buf[10];

	xinfo = ip297x_uvc_ctrls;
	while( xinfo->entity[0] != 0 ) {
		ret_val=ioctl(Video->fd, UVCIOC_CTRL_ADD, xinfo);
		xinfo++;
	}
	ret_val = (int) IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0x0a);
	sensor_id = (ret_val<<8);
	ret_val = (int) IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0x0b);
	sensor_id += ret_val;
	Video->SensorID = sensor_id;
#if ((IP297X_DEBUG & IP297X_DEBUG_EXTENSION) != 0)
	fprintf(stderr, "Sensor ID : %x\n", Video->SensorID);
	memset(tmp_buf, 0, 10);
	fprintf(stderr, "Motion : ", ret_val);
	IP297xMotionGet(Video, tmp_buf);
	for( idx=0; idx < 10; idx++ ) {
		fprintf(stderr, " %2x", tmp_buf[idx]);
	}
	fprintf(stderr, "\n");
#endif
	return 0;
}

int DumpAllRegister(struct vdIn *Video)
{
int	idx, ret;
__u8	ret_data;

	fprintf(stderr,"\nDump iP2970 :");
	for ( idx=0; idx <=0xff; idx++ ) {
		if ( (idx % 8) == 0 )	fprintf(stderr,"\n");
		ret = IP297xRegisterGet(Video, (__u8) idx);
		fprintf(stderr, "(%2x, %2x)", idx, ret);
	}
	fprintf(stderr,"\nDump OV7740 :");
	for ( idx=0; idx <=0xff; idx++ ) {
		if ( (idx % 8) == 0 )	fprintf(stderr,"\n");
		ret_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, (__u8) idx);
		fprintf(stderr, "(%2x, %2x)", idx, ret_data);
	}
	fprintf(stderr,"\n");
}

int VendorSetVideoParameters(struct vdIn *Video)
{
__u8	sensor_data;

#if ( LENS_MODEL != LENS_MODEL_DCS930 )
	fprintf(stderr,"uvc_stream: Lens model (%s)\n", LensModel[LENS_MODEL]);
	IP297xSetRegisters(Video, IP297x_Init);
	IP297xI2CSetByteRegisters(Video, I2C_OV_SENSOR_ID, OV7740_Init);
#endif
	switch ( Video->CfgResolution ) {
	case IMAGE_RESOLUTION_QCIF:	// QQVGA
		IP297xSetRegisters(Video, IP297x_Init_QQVGA);
		IP297xI2CSetByteRegisters(Video, I2C_OV_SENSOR_ID, OV7740_Init_QQVGA);
		IP297xI2CSetByteRegisters(Video, I2C_OV_SENSOR_ID, OV7740_AE_WEIGHTING_QVGA);
		break;
	case IMAGE_RESOLUTION_CIF:	// QVGA
		IP297xSetRegisters(Video, IP297x_Init_QVGA);
		IP297xI2CSetByteRegisters(Video, I2C_OV_SENSOR_ID, OV7740_Init_QVGA);
		IP297xI2CSetByteRegisters(Video, I2C_OV_SENSOR_ID, OV7740_AE_WEIGHTING_QVGA);
		break;
	case IMAGE_RESOLUTION_VGA:	// VGA
		switch ( Video->CfgCompressionRate ) {
		case IMAGE_COMPRESSION_RATE_VERY_HIGH:
			IP297xSetRegisters(Video, IP297x_Init_VGA_20_BadQuality);
			IP297xI2CSetByteRegisters(Video, I2C_OV_SENSOR_ID, OV7740_Init_VGA_20_BadQuality);
			break;
		case IMAGE_COMPRESSION_RATE_HIGH:
		case IMAGE_COMPRESSION_RATE_MEDIUM:
			IP297xSetRegisters(Video, IP297x_Init_VGA_20);
			IP297xI2CSetByteRegisters(Video, I2C_OV_SENSOR_ID, OV7740_Init_VGA_20);
			break;
		case IMAGE_COMPRESSION_RATE_LOW:
		case IMAGE_COMPRESSION_RATE_VERY_LOW:
			IP297xSetRegisters(Video, IP297x_Init_VGA_15);
			IP297xI2CSetByteRegisters(Video, I2C_OV_SENSOR_ID, OV7740_Init_VGA_15);
			break;
		}
		IP297xI2CSetByteRegisters(Video, I2C_OV_SENSOR_ID, OV7740_AE_WEIGHTING_VGA);
		break;
	}
#if ( LENS_MODEL == LENS_MODEL_DCS930 )
	/* bypass internal regulator */
	sensor_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0x4A);
	sensor_data |= 0x08;
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x4A, sensor_data);
	/* LANC correction */
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x8F, 0x86);
	/* enable contrast/brightness/saturation control */
	sensor_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0xDA);
	sensor_data |= 0x06;
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0xDA, sensor_data);
#endif
	// reset iP2970 video/jpeg-fifo function
	IP297xRegisterSet(Video, 0xff, 0x86);
	usleep(1000*100);
	IP297xRegisterSet(Video, 0xff, 0x8f);
}

int VendorSetImageParameters(struct vdIn *Video)
{
__u8	sensor_data, reg_data;
int	target_value, brightness_default;
#if (TEST_AEC_AGC_DISABLE != 0)
__u8	*p_pattern;
#endif

	/* Brightness , Reg 0xE3: -0x40..+0x40 */
	sensor_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0xE4);
	switch ( Video->CfgResolution ) {
	case IMAGE_RESOLUTION_QCIF:	// QQVGA
		brightness_default = -24;
		break;
	case IMAGE_RESOLUTION_CIF:	// QVGA
		brightness_default = -20;
		break;
	case IMAGE_RESOLUTION_VGA:	// VGA
		brightness_default = -12;
	}
#if ( (LENS_MODEL == LENS_MODEL_TVIP551W) || (LENS_MODEL == LENS_MODEL_TVIP551WI) )
	brightness_default += 20;
#endif
	target_value = (Video->CfgBrightness - 64) * 2 + brightness_default;
	if ( target_value < 0 ) {
		sensor_data |= 0x08;
		target_value = ~target_value;
	} else {
		sensor_data &= ~0x08;
	}
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0xE4, sensor_data);	// set brightness sign bit
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0xE3, (__u8)target_value);
	/* Contrast, Reg 0xE2:	0x00-0xff */
	if ( Video->CfgContrast >= 64 ) {
		target_value = Video->CfgContrast - 64;
		target_value = target_value * (CONTRAST_MAX-CONTRAST_DEFAULT) / 64;
		target_value = target_value + CONTRAST_DEFAULT;
	} else {
		target_value = 64 - Video->CfgContrast;
		target_value = target_value * (CONTRAST_DEFAULT-CONTRAST_MIN) / 64;
		target_value = CONTRAST_DEFAULT - target_value;
	}
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0xE2, (__u8)target_value);
	/* Saturation, Sensor Reg 0xDD, 0xDE :	0x00-0xff */
	if ( Video->CfgSaturation >= 64 ) {
		target_value = Video->CfgSaturation - 64;
		target_value = target_value * (SATURATION_MAX-SATURATION_DEFAULT) / 64;
		target_value = target_value + SATURATION_DEFAULT;
	} else {
		target_value = 64 - Video->CfgSaturation;
		target_value = target_value * (SATURATION_DEFAULT-SATURATION_MIN) / 64;
		target_value = SATURATION_DEFAULT - target_value;
	}
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0xDD, (__u8)target_value);
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0xDE, (__u8)target_value);
	/* Sharpness , Sensor Reg 0xCC : 0x00-0x1f, if > 0x1f then use default value */
	if ( (Video->CfgSharpness >= 0) && (Video->CfgSharpness <= 31) ) {
		sensor_data = 0x40 + (__u8)Video->CfgSharpness;
		IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0xCC, sensor_data);
	}
	/* Anti-Flicker */
	sensor_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0x13);
	if ( Video->CfgAntiFlickerEnable ) {	// Anti-Flicker
		sensor_data &= ~0x10;
	} else {
		sensor_data |= 0x10;
	}
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x13, sensor_data);
	// H/V Mirror
	sensor_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0x0c);
	reg_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0x16);
	reg_data &= 0xfc;	// clear Horizontal start point AHSTART LSB=0
	if ( Video->CfgMirror & IMAGE_FLAG_MIRROR_H ) {
		sensor_data |= 0x40;
		reg_data |= 1;	// Horizontal start point AHSTART LSB=1
	} else {
		sensor_data &= ~0x40;
//		reg_data |= 0;	// Horizontal start point AHSTART LSB=0
	}
	if ( Video->CfgMirror & IMAGE_FLAG_MIRROR_V ) {
		sensor_data |= 0x80;
	} else {
		sensor_data &= ~0x80;
	}
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x0c, sensor_data);
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x16, reg_data);
	// Light Frequency
	sensor_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0xec);
	sensor_data &= ~0xc0;	// select manual mode, 60Hz
	if ( Video->CfgLightFrequency == 0 )	sensor_data |= 0x40;	// select 50Hz
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0xec, sensor_data);
	//
	if ( Video->ChangeFlag & IMAGE_CHANGE_FLAG_ANTI_FLICKER ) {
		// reset AEC
		sensor_data = IP297xI2CGetByte(Video, I2C_OV_SENSOR_ID, 0x13);
		sensor_data &= ~0x01;
		IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x13, sensor_data);
		IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x0F, 0x03);	// MSB
		IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x10, 0x03);	// LSB
		usleep(1000*30);
		sensor_data |= 0x01;
		IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x13, sensor_data);
		Video->ChangeFlag &= ~IMAGE_CHANGE_FLAG_ANTI_FLICKER;
	}
#if (TEST_AEC_AGC_DISABLE != 0)
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x13, 0x00);
	// get Eexpsure
	p_pattern = nvram_bufget(RT2860_NVRAM, "FTPHostAddress");
	target_value = strtol(p_pattern, NULL, 16);
	sensor_data = (__u8)((target_value & 0xff00)>>8);
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x0f, sensor_data);
	fprintf(stderr,"<OV Register=0x0f, Data=0x%.2x>\n", sensor_data);
	sensor_data = (__u8)(target_value & 0xff);
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x10, sensor_data);
	fprintf(stderr,"<OV Register=0x10, Data=0x%.2x (Exposure))>\n", sensor_data);
	// get Gain
	p_pattern = nvram_bufget(RT2860_NVRAM, "FTPUserName");
	target_value = strtol(p_pattern, NULL, 16);
	sensor_data = (__u8)(target_value & 0xff);
	nvram_close(RT2860_NVRAM);
	IP297xI2CSetByte(Video, I2C_OV_SENSOR_ID, 0x00, sensor_data);
	fprintf(stderr,"<OV Register=0x00, Data=0x%.2x (Gain)>\n", sensor_data);
#endif
}

#endif	/* IP297X_INCLUDE */
