///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2000 Intel Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// * Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright notice,
// this list of conditions and the following disclaimer in the documentation
// and/or other materials provided with the distribution.
// * Neither name of Intel Corporation nor the names of its contributors
// may be used to endorse or promote products derived from this software
// without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
// OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
///////////////////////////////////////////////////////////////////////////
//
// $Revision: 1.1.1.5 $
// $Date: 2001/06/15 00:21:55 $
//     

//Utility functions for url parsing
//and connecting to an http server as a client
#include <string.h>
#include <ctype.h>
#include <netinet/in.h>
#include "upnp_types.h"
#include "http_pas.h"
#include "upnp_wra.h"



//*************************************************************************
//* Name: copy_token
//*
//* Description:  Tokens are generally pointers into other strings
//*               this copies the offset and size from a token (in)
//*               relative to one string (in_base) into a token (out)
//*               relative to another string (out_base)
//*
//* In:           char * in_base, token * in, char * out_base 
//*
//* Out:          The fields of out are set relative to out_base.
//*
//* Return Codes: None 
//* Error Codes:  None               
//*************************************************************************
void copy_token(const token *in, const char * in_base, token * out, char * out_base)
{
  out->size=in->size;
  out->buff = out_base + (int)(in->buff-in_base);
}

//*************************************************************************
//* Name: free_URL_list
//*
//* Description:  Frees the memory associated with a URL_list.
//*               Frees the dynamically allocated members of of list.
//*               Does NOT free the pointer to the list itself 
//*               ( i.e. does NOT free(list))
//* In:           URL_list *list
//*
//* Out:          list->URLs is freed 
//*               list->parsedURLs is freed
//*
//* Return Codes: None
//* Error Codes:  None       
//*************************************************************************  
void free_URL_list(URL_list * list)
{
  if (list->URLs)
    ALFreeMemory(list->URLs);
  if (list->parsedURLs)
    ALFreeMemory(list->parsedURLs);
  list->size=0;
}


//*************************************************************************
//* Name: is_reserved
//*
//* Description:  Returns a 1 if a char is a RESERVED char
//*               as defined in http://www.ietf.org/rfc/rfc2396.txt
//*               (RFC explaining URIs)
//*
//* In:           char in
//*
//* Out:          1 if reserved, 0 if not
//*
//* Return Codes: None
//* Error Codes:  None       
//*************************************************************************
int is_reserved(char in)
{
  if (strchr(RESERVED,in))
    return 1;
  else 
    return 0;
}

//*************************************************************************
//* Name: is_mark
//*
//* Description:  Returns a 1 if a char is a MARK char
//*               as defined in http://www.ietf.org/rfc/rfc2396.txt
//*               (RFC explaining URIs)
//*
//* In:           char in
//*
//* Out:          1 if a Mark, O if not
//*
//* Return Codes: None
//* Error Codes:  None       
//*************************************************************************
int is_mark(char in )
{
  if (strchr(MARK,in))
    return 1;
  else
    return 0;
}

//*************************************************************************
//* Name: is_unreserved
//*
//* Description:  Returns a 1 if a char is an unreserved char
//*               as defined in http://www.ietf.org/rfc/rfc2396.txt
//*               (RFC explaining URIs)
//*
//* In:           char in
//*
//* Out:          1 if unreserved, O if not
//*
//* Return Codes: None
//* Error Codes:  None       
//*************************************************************************
int is_unreserved(char in)
{
  if (isalnum(in) || (is_mark(in)))
    return 1;
 else 
    return 0;
}

//*************************************************************************
//* Name: is_escaped
//*
//* Description:  Returns a 1 if a char[3] sequence is escaped
//*               as defined in http://www.ietf.org/rfc/rfc2396.txt
//*               (RFC explaining URIs)
//*               size of array is NOT checked (MUST be checked by caller)
//*
//* In:           char *in
//*
//* Out:          1 if escaped, 0 if not
//*
//* Return Codes: None
//* Error Codes:  None       
//*************************************************************************
int is_escaped(  char * in)
{
  
  if ( ( in[0]=='%') && (isxdigit(in[1])) && isxdigit(in[2]))
   {
     
     return 1;
   }
  else
    return 0;
}

//*************************************************************************
//* Name: parse_header_value
//*
//* Description:  Parses the input for a header value.
//*               the header value is equal to all characters up to /r/n 
//*               (end of line)
//*               with beginning and trailing whitespace, removed  
//*               (TAB and ' ')
//*               The memory is NOT copied.  The returned token simply 
//*               points into the original char array. 
//*               Assumes that the whitespace has been stripped from
//*               the beginning of value
//* In:            char *in , int max_size (max_size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          the size of the header value that was parsed, out.buff 
//*               points into the original string. There is NO new memory 
//*               created.
//*
//* Return Codes: None
//* Error Codes:  None       
//*************************************************************************
int parse_header_value(  char *in, token *out, int max_size)
{
  int counter =0;
   char * finger=in;
  counter = parse_http_line(in,max_size);
  if (!counter)
    { 
      out->size=0;
      out->buff=NULL;
      return 0;
    }

  counter-=2;

  finger+=(counter);

  //strip whitespace from end 
  while ( (counter>=0)
	  && ( (*finger==' ')
	       || (*finger==TAB)))
   {
     counter--;
     finger--;
   }

  out->buff=in;
  out->size=counter;
  return counter;

}

//*************************************************************************
//* Name: parse_http_line
//*
//* Description:  Returns the number of characters between 
//*               the first character of in, and the first \r\n
//*               sequence
//* In:            char *in , int max_size (max_size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          the size of the line (including \r\n)
//* Return Codes: None
//* Error Codes:  0 if no \r\n is found within the max_size       
//*************************************************************************
int parse_http_line(  char * in, int max_size)
{
  int counter =0;
  while ( (counter+1<max_size)
	  && ( (in[counter]!=CR)
	       || (in[counter+1]!=LF)))
    counter++;
  if (counter>max_size-2 )
    return 0;
  return counter+2;
}

//*************************************************************************
//* Name: upnp_parse_status_line
//*
//* Description:  parses the status line of an http response
//* In:            char *in , http_status *out, int max_size 
//*              ( to avoid running into bad memory)
//*
//* Out:          returns size of line (including /r/n)
//* Return Codes: None
//* Error Codes:  0 if any piece is missing (NOTE: NO LWS is expected at 
//*                                          beginning of in) 
//*                                         values are not to be trusted
//*                                         if return code is zero      
//*************************************************************************
int upnp_parse_status_line( char * in, http_status * out, int max_size)
{
  int temp=0;
  int max_len=max_size;
   char * finger=in;
  int size=0;

  out->http_version.buff=NULL;
  out->http_version.size=0;
  out->status_code.buff=NULL;
  out->status_code.size=0;
  out->reason_phrase.buff=NULL;
  out->reason_phrase.size=0;

  if (! (temp=parse_not_LWS(in,&out->http_version,max_size)))
      {
	return 0;
      }
  finger+=temp;
  max_len-=temp;
  size+=temp;

  temp=parse_LWS(finger,max_len);

  max_len-=temp;
  finger+=temp;
  size+=temp;

  if (! (temp=parse_not_LWS(finger,&out->status_code,max_len)))
    { 
      return 0;
    }
  finger+=temp;
  max_len-=temp;
  size+=temp;

  temp=parse_LWS(finger,max_len);

  max_len-=temp;
  finger+=temp;
  size+=temp;

  if (! (temp=parse_http_line(finger,max_len)))
    { 
      return 0;
    }
  out->reason_phrase.buff=finger;

  //remove /r/n
  out->reason_phrase.size=temp-2;

  size+=temp;
  return size;
      
}

//*************************************************************************
//* Name: upnp_parse_request_line
//*
//* Description:  parses the request line of an http response
//* In:            char *in , http_status *out, int max_size 
//*              ( to avoid running into bad memory)
//*
//* Out:          returns size of line (including /r/n)
//* Return Codes: None
//* Error Codes:  0 if any piece is missing or bad (NOTE: 
//*                 NO LWS is expected at 
//*                                          beginning of in) 
//*                                          values are not to be trusted
//*                                          if return code is zero      
//*************************************************************************
int upnp_parse_request_line( char * in, http_request * out, int max_size)
{
  int temp=0;
  int max_len=max_size;
  char * finger=in;
  int size=0;
  token tempToken;

  out->http_version.buff=NULL;
  out->http_version.size=0;
  out->method.buff=NULL;
  out->method.size=0;
 

  if (! (temp=parse_token(in,&out->method,max_size)))
    {
      
      return 0;
    }
  finger+=temp;
  max_len-=temp;
  size+=temp;

  temp=parse_LWS(finger,max_len);

  max_len-=temp;
  finger+=temp;
  size+=temp;

    //parse request URI (path)
  if (!(temp=parse_uri(finger,max_len,&out->request_uri)))
    {
      
      return 0;
    }
  temp=parse_not_LWS(finger,&tempToken,max_len);

  max_len-=temp;
  finger+=temp;
  size+=temp;

  temp=parse_LWS(finger,max_len);

  max_len-=temp;
  finger+=temp;
  size+=temp;

  if (! (temp=parse_http_line(finger,max_len)))
    {
      
      return 0;
    }
  //remove /r/n
  out->http_version.size=temp-2;
  out->http_version.buff=finger;
  size+=temp;
  
  return size;
      
}

//*************************************************************************
//* Name: parse_not_LWS
//*
//* Description:  Returns the number of characters between 
//*               the first character of in, and the first LWS (for this 
//*               function LWS: Tab 
//*               or space )
//*               sets the value of the token.buff=in, token.size= (num of 
//*               characters of nonLWS)
//*
//*               Note: does not handle multiple line white space  
//* In:           char *in , int max_size (max_size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          out->size (length of non LWS at beginning of in)
//*
//* Return Codes: None
//* Error Codes:  None       
//*************************************************************************
int parse_not_LWS(  char *in, token *out, int max_size)
{
  int counter=0;

  while ( (counter<max_size)
	  && (in[counter]!=' ')
	  && (in[counter]!=TAB) )
    counter++;

  out->buff=in;
  out->size=counter;
  return counter;
  
}

//*************************************************************************
//* Name: parse_LWS
//*
//* Description:  Returns the number of characters between 
//*               the first character of in, and the first non LWS (LWS is 
//*               TAB or SPACE)
//*               for this function
//* 
//* In:            char *in , int max_size (max_size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          length of LWS at beginning of in
//* Return Codes: None
//* Error Codes:  None       
//*************************************************************************
int parse_LWS( char * in, int max_size)
{
  int counter=0;
  while ((counter<max_size)
	 && ( (in[counter]==' ')
	      || (in[counter]==TAB) ))
    {
      counter++;
    }
  return counter;
}

//*************************************************************************
//* Name: parse_token
//*
//* Description:  parses a token as defined in HTTP/1.1 
//* In:            char *in , int max_size (max_size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          out->size (length of token)
//*               Note: out->buff simply points to in. (no new memory)
//* Return Codes: length of token
//* Error Codes:  0 if no char is matched
//*************************************************************************
int parse_token( char * in, token * out, int max_size)
{
  int counter=0;
  int prefix_sp=0;

  prefix_sp = parse_LWS(in, max_size);
  in += prefix_sp;
  max_size -= prefix_sp;

  while ( ( counter<max_size)
	  && (in[counter]>31) 
	  && (strchr(SEPARATORS,in[counter])==NULL) 
	  && (in[counter]!=127))
    {
     
	  counter++;
    }
    out->buff=in;
    out->size=counter;      
    //return counter;
    return (counter+prefix_sp);
}

//*************************************************************************
//* Name: parse_uric
//*
//* Description:  parses a string of uric characters starting at in[0]
//*               as defined in http://www.ietf.org/rfc/rfc2396.txt
//*               (RFC explaining URIs)
//* In:            char *in , int max_size (max_size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          out->size (length of uric string)
//*               Note: out->buff simply points to in.  (no new memory)
//* Return Codes: length of uric string
//* Error Codes:  0 if no char is matched.      
//*************************************************************************
int parse_uric(  char *in, int max, token *out)
{
  int i=0;

  while ( (i<max) && ( ( is_unreserved(in[i])) || (is_reserved(in[i])) ||
		       ( (i+2<max) && (is_escaped(&in[i])) ) ) )
    {
      i++;
    }

  out->size=i;
  out->buff=in;
  return i;
}

//*************************************************************************
//* Name: parse_port
//*
//* Description:  parses a port (i.e. '4000') and converts it into a network
//*               ordered unsigned short int.
//* In:            char *port , int max (max_size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          unsigned short int *out. (memory allocated by caller)
//*               
//* Return Codes: Number of characters contained in the port value
//* Error Codes:  0 if no char is matched.      
//*************************************************************************
int parse_port(int max,   char * port, unsigned short * out)
{

   char * finger=port;
   char * max_ptr=finger+max;
  unsigned short temp=0;

  while ( (finger<max_ptr) && ( isdigit(*finger) ) )
    {
      temp=temp*10;
      temp+=(*finger)-'0';
      finger++;
    }

  *out= htons(temp);
  return (int)(finger-port);
}

//*************************************************************************
//* Name: parse_hostport
//*
//* Description:  parses a string representing a host and port
//*               (e.g. "127.127.0.1:80" or "localhost")
//*               and fills out a hostport_type struct with internet
//*               address and a token representing the full host
//*               and port.  uses gethostbyname.
//*
//* In:           char *in , int max  (max size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          out->text.size (size of text parsed as host and port)
//*               Note: out->text.buff points to in. No new memory is created.
//*
//* Return Codes: length of host port text, out->text.size
//* Error Codes:  UPNP_E_INVALID_URL (if either host or port can not be 
//*               resolved) 
//*               UPNP_E_OUTOF_MEMORY (if the function ran out
//*               of memory)
//*************************************************************************
int parse_hostport( char *in , int max, hostport_type *out)
{
  
  int i=0; 
  int begin_port;
  int hostport_size=0;
  int host_size=0;
  //struct  hostent * h;
  char * temp_host_name;
  

  out->IPv4address.sin_port=htons(80); //default port is 80
  bzero( &out->IPv4address.sin_zero,8);
  
  while ( (i<max) && (in[i]!=':') && (in[i]!='/') 
	  && ( (isalnum(in[i])) || (in[i]=='.') || (in[i]=='-') ))
      i++;
  
  host_size=i;
  
  if ( (i<max) && (in[i]==':'))
    {  
      begin_port=i+1;
    //convert port
    if (! ( hostport_size=parse_port(max-begin_port,
				     &in[begin_port],
				     &out->IPv4address.sin_port)))
      return -1;
      //return UPNP_E_INVALID_URL;
    hostport_size+=begin_port;
    }
  else
    hostport_size=host_size;
  
  //convert to temporary null terminated string
  temp_host_name=(char * )ALAllocateMemory(host_size+1);
  
  if (temp_host_name==NULL)
    return -1;
    //return UPNP_E_OUTOF_MEMORY;

  memcpy(temp_host_name,in,host_size);
  temp_host_name[host_size]='\0';
#if 0
      
  if ( ( h=gethostbyname(temp_host_name)) && (h->h_addrtype==AF_INET)
       && (h->h_length==4) )
    {
      
      out->IPv4address.sin_addr= (* (struct in_addr *) h->h_addr);
      out->IPv4address.sin_family=AF_INET;
    }
  else
    {
      out->IPv4address.sin_addr.s_addr=0;
      out->IPv4address.sin_family=AF_INET;
      ALFreeMemory(temp_host_name);
      return -1;
      //return UPNP_E_INVALID_URL;
    }
#endif
  
  ALFreeMemory(temp_host_name);
  
  out->text.size=hostport_size;
  out->text.buff=in;
  return hostport_size;
  
}

//*************************************************************************
//* Name: parse_scheme
//*
//* Description:  parses a uri scheme starting at in[0]
//*               as defined in http://www.ietf.org/rfc/rfc2396.txt
//*               (RFC explaining URIs)
//*               (e.g. "http:" -> scheme= "http")
//*               Note string MUST include ':' within the max charcters
//* In:           char *in , int max (max size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          out->size (length of scheme)
//*               Note: out->buff simply points to in.  (no new memory)
//* Return Codes: length of scheme
//* Error Codes:  0 if scheme is not found   
//*************************************************************************
int parse_scheme(  char * in, int max, token * out)
{
  int i=0;
  out->size=0;
  out->buff=NULL;
 
 
  if ( (max==0) || ( ! isalpha(in[0])))
    return FALSE;

  i++;
  while ( (i<max) && (in[i]!=':'))
    {
 
      if (  !( isalnum(in[i]) || ( in[i]=='+') || (in[i]=='-')
	       || (in[i]=='.')))
	return FALSE;
      
      i++;
    }
  if (i<max) 
    {
      out->size=i;
      out->buff=&in[0];
      return i;
    }

  return FALSE;

}


//*************************************************************************
//* Name: remove_dots
//*
//* Description:  removes ".", and ".." from a path. If a ".." can not
//*               be resolved (i.e. the .. would go past the root of
//*               the path) an error is returned. The input IS
//*               modified in place.)
//* In:           char *in , int max (max size to be considered 
//*               to avoid 
//*               running into bad memory)
//*
//* Out:          char * in (with ".", and ".." rmeoved from the path)
//* Return Codes: UPNP_E_SUCCESS, UPNP_E_OUTOF_MEMORY, UPNP_E_INVALID_URL
//* Examples: 
//*          char path[30]="/../hello";
//*          remove_dots(path, strlen(path)) -> UPNP_E_INVALID_URL
//*          char path[30]="/./hello";
//*          remove_dots(path, strlen(path)) -> UPNP_E_SUCCESS, 
//*           in = "/hello"
//*          char path[30]="/./hello/foo/../goodbye" -> 
//*            UPNP_E_SUCCESS, in = "/hello/goodbye"
//*************************************************************************
int remove_dots(char *in, int size)
{
  char * copyTo=in;
  char * copyFrom=in;
  char * max=in+size;
  char **Segments=NULL;
  int lastSegment=-1;
  

  Segments=ALAllocateMemory(sizeof(char*) * size);

  if (Segments==NULL)
    return -1;
    //return UPNP_E_OUTOF_MEMORY;

  Segments[0]=NULL;
  
  while  ( (copyFrom<max) && (*copyFrom!='?')
	   && (*copyFrom!='#'))
    {
     
      if  ( ((*copyFrom)=='.')
	    && ( (copyFrom==in) || (*(copyFrom-1)=='/') ))
	{
	  if  ( (copyFrom+1==max)
		|| ( *(copyFrom+1)=='/'))
	    {
	      
	      copyFrom+=2;
	      continue;
	    }
	  else
	    if ( ( *(copyFrom+1)=='.')
		 && ( (copyFrom+2==max)
		      || ( *(copyFrom+2)=='/')))
	      {
		copyFrom+=3;
		
		if (lastSegment>0)
		  {
		    copyTo=Segments[--lastSegment];
		  }
		else
		  {
		    ALFreeMemory(Segments);
		    
		    return -1;
		    //return UPNP_E_INVALID_URL;
		  }
		continue;
	      }
	}
      
      if ( (*copyFrom)=='/')
	{
	  
	  lastSegment++;
	  Segments[lastSegment]=copyTo+1;
	}
      (*copyTo)=(*copyFrom);
      copyTo++;
      copyFrom++;
    }
  if (copyFrom<max)
    {
      while (copyFrom<max)
      {
	(*copyTo)=(*copyFrom);
	copyTo++;
	copyFrom++;
      }
    }
  (*copyTo)=0;
  ALFreeMemory(Segments);
  
  return -1;
  //return UPNP_E_SUCCESS;
}


//*************************************************************************
//* Name: parse_uri
//*
//* Description:  parses a uri 
//*               as defined in http://www.ietf.org/rfc/rfc2396.txt
//*               (RFC explaining URIs)
//*               handles absolute, relative, and opaque uris
//*               Parses into the following pieces:
//*                  scheme, hostport, pathquery, fragment
//*                  (path and query are treated as one token)
//*               Caller should check for the pieces they require.
//* In:           char *in , int max (max size to be considered to avoid 
//*               running into bad memory)
//*
//* Out:          uri_type * out (memory allocated by caller)
//*               Each of the pieces that are present are filled in.
//*               Plus the type of uri (ABSOULTE, RELATIVE), and type of 
//*               path (ABS_PATH, REL_PATH, OPAQUE_PART)
//* Return Codes: HTTP_SUCCESS
//* Error Codes:  returns the error code returned from
//*               parse_hostport if an address is expected but not resolved   
//*************************************************************************
int parse_uri(  char * in, int max, uri_type * out)
{
  int begin_path=0;
  int begin_hostport=0;
  int begin_fragment=0;
  

  //if ( (begin_hostport=parse_scheme(in, max,&out->scheme) ))
  begin_hostport=parse_scheme(in, max,&out->scheme);
  if ( begin_hostport )
  {
    out->type=ABSOLUTE;
    out->path_type=OPAQUE_PART;
    begin_hostport++;
  }
  else
    {    
      out->type=RELATIVE;
      out->path_type=REL_PATH;
    }

  if ( ( (begin_hostport+1) <max) && (in[begin_hostport]=='/')
       && (in[begin_hostport+1]=='/'))
    {
      begin_hostport+=2;
      
      if ( ( begin_path=parse_hostport(&in[begin_hostport],
				       max-begin_hostport,
				       &out->hostport)) >= 0 )
	{
	  begin_path+=begin_hostport;
	}
      else
	return begin_path;			  
	
    }
  else
    {
       out->hostport.IPv4address.sin_port=0;
       out->hostport.IPv4address.sin_addr.s_addr=0;
       out->hostport.text.size=0;
       out->hostport.text.buff=0;
       begin_path=begin_hostport;
    }

  begin_fragment=parse_uric(&in[begin_path],max-begin_path,&out->pathquery);
  
  if ( (out->pathquery.size) && (out->pathquery.buff[0]=='/'))
    {
      out->path_type=ABS_PATH;
    }

  begin_fragment+=begin_path;
 
  if ( (begin_fragment<max) && ( in[begin_fragment]=='#'))
    {
      begin_fragment++;
      parse_uric(&in[begin_fragment],max-begin_fragment,&out->fragment);
    }
  else{
    out->fragment.buff=NULL;
    out->fragment.size=0;
  }
  
  return HTTP_SUCCESS;
}


//*************************************************************************
//* Name: concat_socket_buffers
//*
//* Description:  concats a list of socket buffers (used to read data of 
//*               unknown size) into one buffer (out)
//*               size of out is assumed to be large enough to hold the data
//*               the caller is responsible for maintaining this
//*
//* In:           socket_buffer * head (head of the list of buffers)
//*
//* Out:          the concatenated socket_buffers are copied into out
//*
//* Return Codes: None
//* Error Codes:  None   
//*************************************************************************
void concat_socket_buffers(socket_buffer * head, char * out)
{
  socket_buffer *finger=head;
  finger=head;
  while (finger)
    {
      memcpy(out,finger->buff,finger->size);
      out+=finger->size;
      finger=finger->next;
     }
  
}

//*************************************************************************
//* Name: free_socket_buffers
//*
//* Description:  frees the space allocated in a list of socket_buffers
//*              
//* In:           socket_buffer * head (head of list of buffers to free)
//*
//* Out:          None
//*           
//* Return Codes: None
//* Error Codes:  None  
//*************************************************************************
void free_socket_buffers(socket_buffer * head)
{
  socket_buffer * temp=NULL;
  socket_buffer * finger=NULL;
  if (head)
    {
      finger=head->next;
      while (finger)
	{
	  temp=finger->next;
	  ALFreeMemory(finger);
	  finger=temp;
	}
    }
}


void free_http_headers(http_header * list)
{
  http_header * next;
  while (list)
    {
      next=list->next;
      ALFreeMemory(list);
      list=next;
    }
}

void free_http_message(http_message * message)
{
    free_http_headers(message->header_list);
}


int parse_headers(char *in, http_message *out, int max_len)
{
  int counter;
  char * finger=in;
  char * max_finger=in+max_len;
  http_header *current=NULL;
  http_header *head =NULL;

  //parse headers

  while ( (finger<max_finger-1) &&
	  ( ( (*finger)!=CR) || ((*(finger+1))!=LF)))
    {
      
      if (current)
	{
	  current->next=(http_header *) ALAllocateMemory(sizeof(http_header));
	  current=current->next;
	}
      else
	{
	  head=(http_header *) ALAllocateMemory(sizeof(http_header));
	  current=head;
	}
      if (!current)
	{
	  free_http_headers(head);
	  return -1;
	  //return UPNP_E_OUTOF_MEMORY;
	}
      current->next=NULL;
      
      //parse header
      if (!(counter=parse_token(finger, &current->header,max_len)))
	{ free_http_headers(head);
	  return -1;
	  //return UPNP_E_BAD_REQUEST;
	
	}
      
      finger+=counter;
      if ( (*finger!=':'))
	{
	  free_http_headers(head);
	  return -1;
	  //return UPNP_E_BAD_REQUEST;
	}
      counter++;
      finger++;
      max_len-=counter;

      //get rid of whitespace
      counter=parse_LWS(finger,max_len);
      finger+=counter;
      max_len-=counter;
      
      //parse value, allows empty values
      if (!(counter=parse_header_value(finger,&current->value,max_len)))
	{
	  current->value.size=0;
	  current->value.buff=NULL;
	}
      
      finger+=counter;
      max_len-=counter;
      
      counter=parse_http_line(finger,max_len);
      finger+=counter;
      max_len-=counter;
    }
  
  out->header_list=head;

  if (finger<max_finger-1)
    finger+=2;
    
  out->content.buff=finger;
  if (max_len>0)
    out->content.size=max_len-2;
  else
    out->content.size=0;

  return HTTP_SUCCESS;
}


int parse_http_response( char * in, http_message *out, int max_len)
{
  int temp;
  int max=max_len;
  char *finger=in;
  
  if ( !(temp=upnp_parse_status_line(in,&out->status,max_len)))
    {
      return -1;
      //return UPNP_E_BAD_RESPONSE;
    }
  max-=temp;
  finger+=temp;
  
  if ( parse_headers(finger,out,max)!=HTTP_SUCCESS)
    {
      return -1;
      //return UPNP_E_BAD_REQUEST;
    }
  return HTTP_SUCCESS;

}

int parse_http_request(  char * in, http_message * out, int max_len)
{
  int temp;
  int max=max_len;
  char *finger=in;
  
  if ( !(temp=upnp_parse_request_line(in,&out->request,max_len)))
    {
      return -1;
      //return UPNP_E_BAD_REQUEST;
    }
  max-=temp;
  finger+=temp;
  
  if ( parse_headers(finger,out,max)!=HTTP_SUCCESS)
    {
      return -1;
      //return UPNP_E_BAD_REQUEST;
    }
  return HTTP_SUCCESS;
  
}
