/*
 * A.L. memory management utility(AL_MEM.C)
 */
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "upnp_types.h"
#include "semaphore_api.h"
#include "dbgtrace.h"
#include "al_mem.h"

extern int almem_sem_id;

#define AL_MEM_C 
#define AL_MEM_MONITOR_DURATION  0*18

static char far AL_ALLOC_MEM[]="__ALAllocateMemory";
static char far AL_FREE_MEM[]="__ALFreeMemory";
static char far AL_NOT_ALLOC[]="Not Alloc By AL";
static char far AL_EMPTY[]="EMPTY";
static char far AL_INVALID[]="INVALID";
static char far AL_NTH_ALLOC[]="nth allocation=";
static char far AL_GROW_BUF2[]="ALgrowBuffer2";

void ALDumpMemoryLeak(int level);

ULONG
ALUlongToString(ULONG  ulValue, char*  string)
{
    ULONG                           counter = 0;
    ULONG                           result  = ulValue;
    ULONG                           length  = 0;

    while ( result )
    {
        result = result / 10;
        counter++;
    }

    length = counter;
    result = ulValue;

    if ( length != 0 )
    {
        for ( counter = 0; counter < length; counter++ )
        {
            string[(int)(length - 1 - counter)] = (char)(result % 10) + '0';
            result                       = result / 10;
        }

        string[(int)length] = 0;
    }
    else
    {
        string[0]  = '0';
        length      = 1;
    }

    return  length;
}


/* DOUBLY LINKED LIST */
typedef struct
_LIST_ENTRY
{
   struct _LIST_ENTRY               *Flink;
   struct _LIST_ENTRY               *Blink;
}
LIST_ENTRY, *PLIST_ENTRY;

#define  ALInitializeListHead(ListHead)             ((ListHead)->Flink = (ListHead)->Blink = (ListHead))


#define  ALInsertTailList(ListHead,Entry)                       \
            {                                                   \
                PLIST_ENTRY _EX_Blink;                          \
                PLIST_ENTRY _EX_ListHead;                       \
                _EX_ListHead        = (ListHead);               \
                _EX_Blink           = _EX_ListHead->Blink;      \
                (Entry)->Flink      = _EX_ListHead;             \
                (Entry)->Blink      = _EX_Blink;                \
                _EX_Blink->Flink    = (Entry);                  \
                _EX_ListHead->Blink = (Entry);                  \
            }

#define  ALIsListEmpty(ListHead)                    ((ListHead)->Flink == (ListHead))
#define  ALGetFirstEntryList(ListHead)              (ListHead)->Flink
#define  ALGetNextEntryList(Entry)                  (Entry)->Flink
#define  ALIsListEntryValid(pListEntry)             ( ((pListEntry)->Flink != NULL) && ((pListEntry)->Blink != NULL) )

#define  ALRemoveEntryList(Entry)                               \
            {                                                   \
                PLIST_ENTRY _EX_Blink;                          \
                PLIST_ENTRY _EX_Flink;                          \
                _EX_Flink           = (Entry)->Flink;           \
                _EX_Blink           = (Entry)->Blink;           \
                _EX_Blink->Flink    = _EX_Flink;                \
                _EX_Flink->Blink    = _EX_Blink;                \
            }

#define  ALSizeOfString                             strlen
#define  ALStringCopy                               strcpy
#define  ALStringCat                                strcat


typedef	unsigned char	UCHAR ;
typedef  UCHAR                      *PUCHAR;
typedef  char                       *PCHAR;
typedef  void                       *PVOID;

#define  ALZeroMemory(pMemory, ulMemorySize)        memset(pMemory, 0, ulMemorySize)
ULONG           gMemAllocTotalSize      = 0;
ULONG           gNumberMemAllocated     = 0;
ULONG           gCVMemAllocTotalSize      = 0;
ULONG           gCVNumberMemAllocated     = 0;
ULONG           gIndex= 0;  /* Accumulated No. of Allocation*/


#ifdef AL_MEM_S 
typedef struct _AL_MEM_PREHEADER
{
    /* true size of memory allocation we were requested */
    ULONG               MemTrueSize;

    /*
    UCHAR               MemData[0];
     */
}
AL_MEM_PREHEADER, *PAL_MEM_PREHEADER;


#define ACCESS_MEM_FROM_PREHEADER(pHeader)          (PVOID)( ((PUCHAR)pHeader) + sizeof(AL_MEM_PREHEADER))
#define ACCESS_PREHEADER_FROM_MEM(pMem)             (PAL_MEM_PREHEADER)( ((PUCHAR)pMem) - sizeof(AL_MEM_PREHEADER))



void *
ALAllocateMemory(ULONG  ulMemorySize)
{
    PAL_MEM_PREHEADER           pHeader;

    //pHeader =  (PAL_MEM_PREHEADER)POSEMemoryAllocate((size_t)ulMemorySize + sizeof(AL_MEM_PREHEADER));
    pHeader =  (PAL_MEM_PREHEADER)malloc((size_t)ulMemorySize + sizeof(AL_MEM_PREHEADER));

    if (pHeader)
    {
        ALZeroMemory(pHeader, (size_t)ulMemorySize + sizeof(AL_MEM_PREHEADER));

        pHeader->MemTrueSize = ulMemorySize;

        gMemAllocTotalSize += ulMemorySize;
        gNumberMemAllocated += 1;

        return ACCESS_MEM_FROM_PREHEADER(pHeader);
    }
    else
    {
        return NULL;
    }
}

void
ALFreeMemory(void *pMem)
{
    PAL_MEM_PREHEADER           pHeader;

    pHeader             = ACCESS_PREHEADER_FROM_MEM(pMem);

    gMemAllocTotalSize -= pHeader->MemTrueSize;
    gNumberMemAllocated -= 1;

    //POSEMemoryFree(pHeader);
    free(pHeader);

    return;
}
void *
ALReAllocateMemory(void **pOrigMem, ULONG  ulMemorySize)
{
    return NULL;
}
#endif

#ifdef AL_MEM_C 
typedef struct _AL_MEM_PREHEADER
{
    LIST_ENTRY          Linkage;

    /* true size of memory allocation we were requested */
    ULONG               MemTrueSize;

    char                Description[32];   /* GNU compiler put full-path file name for __FILE__ */

    char                index[8];
    /*
     *  eliminate usage of zero-sized array
     *
    UCHAR               MemData[0];
     */
}
AL_MEM_PREHEADER, *PAL_MEM_PREHEADER;

#define  CONTAINING_RECORD(address, type, field)    \
            ((type*)((PCHAR)(address) - (ULONG)(&((type*)0)->field)))


#define ACCESS_MEM_FROM_PREHEADER(pHeader)          (PVOID)( ((PUCHAR)pHeader) + sizeof(AL_MEM_PREHEADER))
#define ACCESS_PREHEADER_FROM_MEM(pMem)             (PAL_MEM_PREHEADER)( ((PUCHAR)pMem) - sizeof(AL_MEM_PREHEADER))
#define ACCESS_PREHEADER_FROM_LINKAGE(pListEntry)   (PAL_MEM_PREHEADER)CONTAINING_RECORD(pListEntry, AL_MEM_PREHEADER, Linkage)

LIST_ENTRY      gMemAllocMonitorList;


static int IsALMemMonitorExpired(void);
static nuint32 ALMemMonitorExpireTime =0;
static char far UPNP_AL_AM[]="AM";
static char far UPNP_AL_FM[]="FM";


void
ALInitializeMemAllocMonitor(void)
{
    ALInitializeListHead(&gMemAllocMonitorList);
}

void*   __ALAllocateMemory(ULONG ulMemorySize, PCSTR pFileName, ULONG LineNumber)
{
    PAL_MEM_PREHEADER           pHeader;
    PVOID                       pMem    = NULL;
    PVOID                       ret     = NULL;
    //PRIVILEGED_PROTECT_HANDLE protectID;

    //protectID = POSEPrivilegedProtect();
    if (!semaphore_p(almem_sem_id)) exit(EXIT_FAILURE);
    //pHeader =  (PAL_MEM_PREHEADER)POSEMemoryAllocate((size_t)ulMemorySize + sizeof(AL_MEM_PREHEADER));
    pHeader =  (PAL_MEM_PREHEADER)malloc((size_t)ulMemorySize + sizeof(AL_MEM_PREHEADER));

    DEBUG_trace(DBG_LEVEL_VERBOSE, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
                      "%s=%p\r\n", UPNP_AL_AM, pHeader);
    if (pHeader)
    {
        ALZeroMemory(pHeader, (size_t)ulMemorySize + sizeof(AL_MEM_PREHEADER));

        pHeader->MemTrueSize = ulMemorySize;

        pMem = ACCESS_MEM_FROM_PREHEADER(pHeader);

        /* put it into monitoring list */
        ALInsertTailList(&gMemAllocMonitorList, &pHeader->Linkage);

        gMemAllocTotalSize += ulMemorySize;

        /*
         * added by shx. 05/30/01. 
         * increase gNumberMemAllocated by 1.
         */
        gNumberMemAllocated += 1;
        gIndex+= 1;

        ALStringCopy(pHeader->Description, pFileName);
        ALStringCat(pHeader->Description, " -- ");
        ALUlongToString(LineNumber, &pHeader->Description[ALSizeOfString(pHeader->Description)]);
        ALUlongToString(gIndex, &pHeader->index[0]);
        //ALUlongToString(gIndex, &pHeader->index[ALSizeOfString(pHeader->index)]);
        DEBUG_trace(DBG_LEVEL_TRACE_INFO, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
                      "%s, %lu, %s, %s\r\n",
                      AL_ALLOC_MEM, 
                      pHeader->MemTrueSize, 
                      pHeader->Description, 
                      pHeader->index);
        ret=pMem;
        //return pMem;
    }
    else
    {
        DEBUG_trace(DBG_LEVEL_CRITICAL, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
                      "%s, %s",
                      AL_ALLOC_MEM, 
                      FAIL);

        ALDumpMemoryLeak(DBG_LEVEL_CRITICAL);
        ret=NULL;
        //return NULL;
    }
    //POSEPrivilegedUnProtect(protectID);
    if (!semaphore_v(almem_sem_id)) exit(EXIT_FAILURE);
    return ret;
}


void    __ALFreeMemory(void *pMem, PCSTR pFileName, ULONG LineNumber)
{
    PAL_MEM_PREHEADER           pHeader;
    PLIST_ENTRY                 pListEntryTobeFreed;
    BOOLEAN                     bFound                  = FALSE;
    PLIST_ENTRY                 pListEntry;
    ULONG                       ulMemorySize;
    //PRIVILEGED_PROTECT_HANDLE protectID;

    //protectID = POSEPrivilegedProtect();
    if (!semaphore_p(almem_sem_id)) exit(EXIT_FAILURE);
    /*
     *  no need to validate the memory, since we don't really touch the memory 
     *  before we do the check against the list
     */
    pHeader             = ACCESS_PREHEADER_FROM_MEM(pMem);
    pListEntryTobeFreed = &pHeader->Linkage;

    /*
     *  check if this one is still in memory pool though it may be too late
     */

    if (!ALIsListEmpty(&gMemAllocMonitorList))
    {
        pListEntry = ALGetFirstEntryList(&gMemAllocMonitorList);

        while (pListEntry != NULL && pListEntry != &gMemAllocMonitorList)
        {
            if (pListEntryTobeFreed == pListEntry)
            {
                bFound = TRUE;
                DEBUG_trace(DBG_LEVEL_TRACE_INFO, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
                      "%s, %lu, %s, %s\r\n",
                      AL_FREE_MEM, 
                      pHeader->MemTrueSize, 
                      pHeader->Description, 
                      pHeader->index);
                break;
            }

            pListEntry = ALGetNextEntryList(pListEntry);
        }

        if (!bFound)
        {
            DEBUG_trace(DBG_LEVEL_CRITICAL, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
                  "\r\n%s\r\n", AL_NOT_ALLOC);
            goto ErrorExit;
        }
    }
    else
    {
        DEBUG_trace(DBG_LEVEL_CRITICAL, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
                  "\r\n%s\r\n", AL_EMPTY);
        goto ErrorExit;
    }

    if (!ALIsListEntryValid(&pHeader->Linkage))
    {
        DEBUG_trace(DBG_LEVEL_CRITICAL, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
                  "\r\n%s\r\n", AL_INVALID);
        goto ErrorExit;
    }

    ulMemorySize = pHeader->MemTrueSize;

    /* remove it from monitoring list */
    ALRemoveEntryList(&pHeader->Linkage);

    /* reset everything in the memory block */
    ALZeroMemory(pHeader, (size_t)ulMemorySize + sizeof(AL_MEM_PREHEADER));

    DEBUG_trace(DBG_LEVEL_VERBOSE, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
                      "%s=%p\r\n", UPNP_AL_FM, pHeader);
    //POSEMemoryFree(pHeader);
    free(pHeader);

    gMemAllocTotalSize -= ulMemorySize;
    /*
     * added by shx. 05/30/01. 
     * decrease gNumberMemAllocated by 1.
     */
    gNumberMemAllocated -= 1;

ErrorExit:
    //POSEPrivilegedUnProtect(protectID);
    if (!semaphore_v(almem_sem_id)) exit(EXIT_FAILURE);
    return;
}

void*   __ALReAllocateMemory(void **pOrigMem, ULONG ulMemorySize, PCSTR pFileName, ULONG LineNumber)
{
    void *pNewMem;
    //PRIVILEGED_PROTECT_HANDLE protectID;

    //protectID = POSEPrivilegedProtect();
    if (!semaphore_p(almem_sem_id)) exit(EXIT_FAILURE);
    pNewMem=__ALAllocateMemory(ulMemorySize,__FILE__, __LINE__);
    if (pNewMem != NULL){
        memcpy(pNewMem, *pOrigMem, (size_t)ulMemorySize); 
        __ALFreeMemory(*pOrigMem, __FILE__, __LINE__);
        *pOrigMem=NULL;
    }
    //POSEPrivilegedUnProtect(protectID);
    if (!semaphore_v(almem_sem_id)) exit(EXIT_FAILURE);
    return pNewMem;
}

static int IsALMemMonitorExpired(void)
{
    nuint32 now;
    time_t t;

    //now = POSEClockGetTick();
    now = time(&t);
    if (ALMemMonitorExpireTime <= now ) {
        ALMemMonitorExpireTime=now + AL_MEM_MONITOR_DURATION; 
        return 1;
    }
    else
        return 0;
}


void
ALDumpMemoryLeak(int level)
{
    PLIST_ENTRY                 pListEntry  = NULL;
    PAL_MEM_PREHEADER           pHeader     = NULL;
    int ret;
    //PRIVILEGED_PROTECT_HANDLE protectID;

    //protectID = POSEPrivilegedProtect();
    if (!semaphore_p(almem_sem_id)) exit(EXIT_FAILURE);
    ret=IsALMemMonitorExpired();
    if (ret == 0) {
        goto Exit;
        //return;
    }

    if (!ALIsListEmpty(&gMemAllocMonitorList))
    {
        pListEntry = ALGetFirstEntryList(&gMemAllocMonitorList);

        while (pListEntry != NULL && pListEntry != &gMemAllocMonitorList)
        {
            pHeader    = ACCESS_PREHEADER_FROM_LINKAGE(pListEntry);

            DEBUG_trace(level, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
                  "%s=%s, size=%lu, %s\r\n",
                  AL_NTH_ALLOC, 
                  pHeader->index,
                  pHeader->MemTrueSize, 
                  pHeader->Description);
            pListEntry = ALGetNextEntryList(pListEntry);
        }
    }
Exit:
    //POSEPrivilegedUnProtect(protectID);
    if (!semaphore_v(almem_sem_id)) exit(EXIT_FAILURE);
    return;
}

int ALgrowBuffer2(void **pOrigMem, int block, int block_size) 
{
    int returnStatus=PASS;
    void *pNewMem;

    DEBUG_trace(DBG_LEVEL_VERBOSE, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
              "%s S\r\n", AL_GROW_BUF2);
    pNewMem=__ALReAllocateMemory(pOrigMem, block * block_size, __FILE__, __LINE__);
    if (pNewMem != NULL) {
        *pOrigMem=pNewMem;
    } else {
        returnStatus=FAIL;
    }
    DEBUG_trace(DBG_LEVEL_VERBOSE, PROJ_DBG_MASK_MEM, MEM_DBG_MASK_AL_MEM_C, 
              "%s E\r\n", AL_GROW_BUF2);
    return returnStatus;
}
#endif

#ifdef CV_MEM
void *
CVAllocateMemory(ULONG  ulMemorySize)
{
    PAL_MEM_PREHEADER           pHeader;

    pHeader =  (PAL_MEM_PREHEADER)POSEMemoryAllocate((size_t)ulMemorySize + sizeof(AL_MEM_PREHEADER));

    if (pHeader)
    {
        ALZeroMemory(pHeader, (size_t)ulMemorySize + sizeof(AL_MEM_PREHEADER));

        pHeader->MemTrueSize = ulMemorySize;

        gCVMemAllocTotalSize += ulMemorySize;
        gCVNumberMemAllocated +=1;

        return ACCESS_MEM_FROM_PREHEADER(pHeader);
    }
    else
    {
        return NULL;
    }
}

void
CVFreeMemory(void *pMem)
{
    PAL_MEM_PREHEADER           pHeader;

    pHeader             = ACCESS_PREHEADER_FROM_MEM(pMem);

    gCVMemAllocTotalSize -= pHeader->MemTrueSize;
    gCVNumberMemAllocated -=1;

    POSEMemoryFree(pHeader);

    return;
}
#endif
