/*
 * parser.c : an XML 1.0 non-verifying parser
 *
 * See Copyright for the status of this software.
 *
 * Daniel.Veillard@w3.org
 */

#if 0
#include "nplat.h"
#include "nest.h"
#include "poseapi.h"
#include "system.h"
#include "nppkt.h"
#include "nptapi.h"
#include "syslib.h"
#include "string.h"
#include "timer.h"
#include "err.h"
#include "netcfg.h"
#include "socket.h"
#include "in.h"
#include "syscfg.h"
#endif

//#include "upnp.h"
//#include "upnp_utl.h"
//#include "upnp_wra.h"

#include <stdlib.h>
#include "xml_dump.h"
#include "xml_tree.h"
#include "xml_pars.h"
#include "xml_enti.h"
#include "xml_vali.h"
#include "xml_util.h"
#include "xml_mem.h"
#include "xml_dbg.h"




unsigned int xmlParserMaxDepth = 10;
/*
 * Forward definition for recusive behaviour.
 */
xmlNodePtr xmlParseElement(xmlParserCtxtPtr ctxt);

/*
 * Macros for accessing the content. Those should be used only by the parser,
 * and not exported.
 *
 * Dirty macros, i.e. one need to make assumption on the context to use them
 *
 *   CUR_PTR return the current pointer to the CHAR to be parsed.
 *   CUR     returns the current CHAR value, i.e. a 8 bit value if compiled
 *           in ISO-Latin or UTF-8, and the current 16 bit value if compiled
 *           in UNICODE mode. This should be used internally by the parser
 *           only to compare to ASCII values otherwise it would break when
 *           running with UTF-8 encoding.
 *   NXT(n)  returns the n'th next CHAR. Same as CUR is should be used only
 *           to compare on ASCII based substring.
 *   SKIP(n) Skip n CHAR, and must also be used only to skip ASCII defined
 *           strings within the parser.
 *
 * Clean macros, not dependent of an ASCII context.
 *
 *   CURRENT Returns the current char value, with the full decoding of
 *           UTF-8 if we are using this mode. It returns an int.
 *   NEXT    Skip to the next character, this does the proper decoding
 *           in UTF-8 mode. It also pop-up unfinished entities on the fly.
 *           It returns the pointer to the current CHAR.
 */

#define CUR (*ctxt->input->cur)
#define SKIP(val) ctxt->input->cur += (val)
#define NXT(val) ctxt->input->cur[(val)]
#define CUR_PTR ctxt->input->cur

#define SKIP_BLANKS 							\
    while (IS_BLANK(*(ctxt->input->cur))) NEXT

#ifndef USE_UTF_8
#define CURRENT (*ctxt->input->cur)
#define NEXT ((*ctxt->input->cur) ?					\
                (((*(ctxt->input->cur) == '\n') ?			\
		    (ctxt->input->line++, ctxt->input->col = 1) :	\
		    (ctxt->input->col++)), ctxt->input->cur++) :	\
		(xmlPopInput(ctxt), ctxt->input->cur))
#else
#endif



/*
 * A few macros needed to help building the parser.
 */

#ifdef UNICODE
#else
/************************************************************************
 *									*
 * 8bits / ISO-Latin version of the macros.				*
 *									*
 ************************************************************************/
/*
 * [2] Char ::= #x9 | #xA | #xD | [#x20-#xD7FF] | [#xE000-#xFFFD]
 *                  | [#x10000-#x10FFFF]
 * any Unicode character, excluding the surrogate blocks, FFFE, and FFFF.
 */
#if 0 /*OXYGEN*/
#define IS_CHAR(c)							\
    ((((c) == 0x09) || ((c) == 0x0a) || ((c) == 0x0d) ||		\
      (((c) >= 0x20) && ((c) != 0xFFFE) && ((c) != 0xFFFF))) &&		\
      (((c) <= 0xD7FF) || ((c) >= 0xE000)) && ((c) <= 0x10FFFF))
#else
#define IS_CHAR(c)							\
    ( ((c) == 0x09) || ((c) == 0x0a) || ((c) == 0x0d) ||		\
      ((c) >= 0x20) )
#endif


/*
 * [85] BaseChar ::= ... long list see REC ...
 */
#if 0 /*OXYGEN*/
#define IS_BASECHAR(c)							\
    ((((c) >= 0x41) && ((c) <= 0x5a)) ||				\
     (((c) >= 0x61) && ((c) <= 0x7a)) ||				\
     (((c) >= 0xaa) && ((c) <= 0x5b)) ||				\
     (((c) >= 0xc0) && ((c) <= 0xd6)) ||				\
     (((c) >= 0xd8) && ((c) <= 0xf6)) ||				\
     (((c) >= 0xf8) && ((c) <= 0xff)) ||				\
      ((c) == 0xba))
#else
#define IS_BASECHAR(c)							\
    ((((c) >= 0x41) && ((c) <= 0x5a)) ||				\
     (((c) >= 0x61) && ((c) <= 0x7a)))
#endif

/*
 * [88] Digit ::= ... long list see REC ...
 */
#define IS_DIGIT(c) (((c) >= 0x30) && ((c) <= 0x39))

/*
 * [84] Letter ::= BaseChar | Ideographic 
 */
#define IS_LETTER(c) IS_BASECHAR(c)


/*
 * [87] CombiningChar ::= ... long list see REC ...
 */
#define IS_COMBINING(c) 0

/*
 * [89] Extender ::= #x00B7 | #x02D0 | #x02D1 | #x0387 | #x0640 |
 *                   #x0E46 | #x0EC6 | #x3005 | [#x3031-#x3035] |
 *                   [#x309D-#x309E] | [#x30FC-#x30FE]
 */
#define IS_EXTENDER(c) ((c) == 0xb7)

#endif /* !UNICODE */

/*
 * Blank chars.
 *
 * [3] S ::= (#x20 | #x9 | #xD | #xA)+
 */
#define IS_BLANK(c) (((c) == 0x20) || ((c) == 0x09) || ((c) == 0xa) ||	\
                     ((c) == 0x0D))

/*
 * [13] PubidChar ::= #x20 | #xD | #xA | [a-zA-Z0-9] | [-'()+,./:=?;!*#@$_%]
 */
#define IS_PUBIDCHAR(c)							\
    (((c) == 0x20) || ((c) == 0x0D) || ((c) == 0x0A) ||			\
     (((c) >= 'a') && ((c) <= 'z')) ||					\
     (((c) >= 'A') && ((c) <= 'Z')) ||					\
     (((c) >= '0') && ((c) <= '9')) ||					\
     ((c) == '-') || ((c) == '\'') || ((c) == '(') || ((c) == ')') ||	\
     ((c) == '+') || ((c) == ',') || ((c) == '.') || ((c) == '/') ||	\
     ((c) == ':') || ((c) == '=') || ((c) == '?') || ((c) == ';') ||	\
     ((c) == '!') || ((c) == '*') || ((c) == '#') || ((c) == '@') ||	\
     ((c) == '$') || ((c) == '_') || ((c) == '%'))

#define SKIP_EOL(p) 							\
    if (*(p) == 0x13) { p++ ; if (*(p) == 0x10) p++; }			\
    if (*(p) == 0x10) { p++ ; if (*(p) == 0x13) p++; }

#define MOVETO_ENDTAG(p)						\
    while (IS_CHAR(*p) && (*(p) != '>')) (p)++

#define MOVETO_STARTTAG(p)						\
    while (IS_CHAR(*p) && (*(p) != '<')) (p)++


/**
 * inputPush:
 * @ctxt:  an XML parser context
 * @value:  the parser input
 *
 * Pushes a new parser input on top of the input stack
 *
 * Returns 0 in case of error, the index in the stack otherwise
 */
extern int
inputPush(xmlParserCtxtPtr ctxt, xmlParserInputPtr value)
{
    if (ctxt->inputNr >= ctxt->inputMax) {
#if 0
        ctxt->inputMax *= 2;
        ctxt->inputTab =
            (xmlParserInputPtr *) xmlRealloc(ctxt->inputTab,
                                             ctxt->inputMax *
                                             sizeof(ctxt->inputTab[0]));
        if (ctxt->inputTab == NULL) {
            xmlErrMemory(ctxt, NULL);
            return (0);
        }
#else
	return (0);
#endif
    }
    ctxt->inputTab[ctxt->inputNr] = value;
    ctxt->input = value;
    return (ctxt->inputNr++);
}
/**
 * inputPop:
 * @ctxt: an XML parser context
 *
 * Pops the top parser input from the input stack
 *
 * Returns the input just removed
 */
extern xmlParserInputPtr
inputPop(xmlParserCtxtPtr ctxt)
{
    xmlParserInputPtr ret;

    if (ctxt->inputNr <= 0)
        return (0);
    ctxt->inputNr--;
    if (ctxt->inputNr > 0)
        ctxt->input = ctxt->inputTab[ctxt->inputNr - 1];
    else
        ctxt->input = NULL;
    ret = ctxt->inputTab[ctxt->inputNr];
    ctxt->inputTab[ctxt->inputNr] = 0;
    return (ret);
}


/**
 * xmlPopInput:
 * @ctxt:  an XML parser context
 *
 * xmlPopInput: the current input pointed by ctxt->input came to an end
 *          pop it and return the next char.
 *
 * TODO A deallocation of the popped Input structure is needed
 *
 * Returns the current CHAR in the parser context
 */
CHAR
xmlPopInput(xmlParserCtxtPtr ctxt) 
{
    if (ctxt->inputNr == 1) return(0); /* End of main Input */
    inputPop(ctxt);
    return(CUR);
}

/**
 * xmlPushInput:
 * @ctxt:  an XML parser context
 * @input:  an XML parser input fragment (entity, XML fragment ...).
 *
 * xmlPushInput: switch to a new input stream which is stacked on top
 *               of the previous one(s).
 */
void
xmlPushInput(xmlParserCtxtPtr ctxt, xmlParserInputPtr input) 
{
    xmlDEBUG_trace(XML_DBG_LEVEL_TRACE_INFO, XML_DBG_MASK_XML_PARS,
                   "%s\r\n", "xmlPushInput");
    if (input == NULL) return;
    inputPush(ctxt, input);
}


/**
 * nodePush:
 * @ctxt:  an XML parser context
 * @value:  the element node
 *
 * Pushes a new element node on top of the node stack
 *
 * Returns 0 in case of error, the index in the stack otherwise
 */
extern int
nodePush(xmlParserCtxtPtr ctxt, xmlNodePtr value)
{
    if (ctxt->nodeNr >= ctxt->nodeMax) {
#if 0 /*OXYGEN*/
        ctxt->nodeMax *= 2;
        ctxt->nodeTab =
            (xmlNodePtr *) xmlRealloc(ctxt->nodeTab,
                                      ctxt->nodeMax *
                                      sizeof(ctxt->nodeTab[0]));
        if (ctxt->nodeTab == NULL) {
            xmlErrMemory(ctxt, NULL);
            return (0);
        }
#endif
    }
    if (((unsigned int) ctxt->nodeNr) > xmlParserMaxDepth) {
#if 0 /*OXYGEN*/
	xmlFatalErrMsgInt(ctxt, XML_ERR_INTERNAL_ERROR,
		 "Excessive depth in document: change xmlParserMaxDepth = %d\r\n",
			  xmlParserMaxDepth);
	ctxt->instate = XML_PARSER_EOF;
#endif
	return(0);
    }
    ctxt->nodeTab[ctxt->nodeNr] = value;
    ctxt->node = value;
    return (ctxt->nodeNr++);
}
/**
 * nodePop:
 * @ctxt: an XML parser context
 *
 * Pops the top element node from the node stack
 *
 * Returns the node just removed
 */
xmlNodePtr
nodePop(xmlParserCtxtPtr ctxt)
{
    xmlNodePtr ret;

    if (ctxt->nodeNr <= 0)
        return (0);
    ctxt->nodeNr--;
    if (ctxt->nodeNr > 0)
        ctxt->node = ctxt->nodeTab[ctxt->nodeNr - 1];
    else
        ctxt->node = NULL;
    ret = ctxt->nodeTab[ctxt->nodeNr];
    ctxt->nodeTab[ctxt->nodeNr] = 0;
    return (ret);
}

/**
 * xmlParseEncName:
 * @ctxt:  an XML parser context
 *
 * parse the XML encoding name
 *
 * [81] EncName ::= [A-Za-z] ([A-Za-z0-9._] | '-')*
 *
 * Returns the encoding name value or NULL
 */
CHAR *
xmlParseEncName(xmlParserCtxtPtr ctxt) 
{
    const CHAR *q = CUR_PTR;
    CHAR *ret = NULL;

    if (((CUR >= 'a') && (CUR <= 'z')) ||
        ((CUR >= 'A') && (CUR <= 'Z'))) {
	NEXT;
	while (IS_CHAR(CUR) &&
	       (((CUR >= 'a') && (CUR <= 'z')) ||
		((CUR >= 'A') && (CUR <= 'Z')) ||
		((CUR >= '0') && (CUR <= '9')) ||
		(CUR == '-'))) NEXT;
	ret = xmlStrndup(q, CUR_PTR - q);
    } else {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "Invalid XML encoding name\r\n");
	ctxt->wellFormed = 0;
    }
    return(ret);
}

/**
 * xmlParseEncodingDecl:
 * @ctxt:  an XML parser context
 * 
 * parse the XML encoding declaration
 *
 * [80] EncodingDecl ::= S 'encoding' Eq ('"' EncName '"' |  "'" EncName "'")
 *
 * TODO: this should setup the conversion filters.
 *
 * Returns the encoding value or NULL
 */

CHAR *
xmlParseEncodingDecl(xmlParserCtxtPtr ctxt) 
{
    CHAR *encoding = NULL;
    const CHAR *q;

    SKIP_BLANKS;
    if ((CUR == 'e') && (NXT(1) == 'n') &&
        (NXT(2) == 'c') && (NXT(3) == 'o') &&
	(NXT(4) == 'd') && (NXT(5) == 'i') &&
	(NXT(6) == 'n') && (NXT(7) == 'g')) {
	SKIP(8);
	SKIP_BLANKS;
	if (CUR != '=') {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, "xmlParseEncodingDecl : expected '='\r\n");
	    ctxt->wellFormed = 0;
	    return(NULL);
        }
	NEXT;
	SKIP_BLANKS;
	if (CUR == '"') {
	    NEXT;
	    q = CUR_PTR;
	    encoding = xmlParseEncName(ctxt);
	    if (CUR != '"') {
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt, "String not closed\n%.50s\n", q);
		ctxt->wellFormed = 0;
	    } else
	        NEXT;
	} else if (CUR == '\''){
	    NEXT;
	    q = CUR_PTR;
	    encoding = xmlParseEncName(ctxt);
	    if (CUR != '\'') {
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt, "String not closed\n%.50s\n", q);
		ctxt->wellFormed = 0;
	    } else
	        NEXT;
	} else {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt,
		     "xmlParseEncodingDecl : expected ' or \"\n");
	    ctxt->wellFormed = 0;
	}
    }
    return(encoding);
}

/**
 * xmlParseSDDecl:
 * @ctxt:  an XML parser context
 *
 * parse the XML standalone declaration
 *
 * [32] SDDecl ::= S 'standalone' Eq
 *                 (("'" ('yes' | 'no') "'") | ('"' ('yes' | 'no')'"')) 
 *
 * Returns 1 if standalone, 0 otherwise
 */

int
xmlParseSDDecl(xmlParserCtxtPtr ctxt) 
{
    int standalone = -1;

    SKIP_BLANKS;
    if ((CUR == 's') && (NXT(1) == 't') &&
        (NXT(2) == 'a') && (NXT(3) == 'n') &&
	(NXT(4) == 'd') && (NXT(5) == 'a') &&
	(NXT(6) == 'l') && (NXT(7) == 'o') &&
	(NXT(8) == 'n') && (NXT(9) == 'e')) {
	SKIP(10);
	if (CUR != '=') {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt,
		    "XML standalone declaration : expected '='\r\n");
	    ctxt->wellFormed = 0;
	    return(standalone);
        }
	NEXT;
	SKIP_BLANKS;
        if (CUR == '\''){
	    NEXT;
	    if ((CUR == 'n') && (NXT(1) == 'o')) {
	        standalone = 0;
                SKIP(2);
	    } else if ((CUR == 'y') && (NXT(1) == 'e') &&
	               (NXT(2) == 's')) {
	        standalone = 1;
		SKIP(3);
            } else {
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt, "standalone accepts only 'yes' or 'no'\r\n");
		ctxt->wellFormed = 0;
	    }
	    if (CUR != '\'') {
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt, "String not closed\r\n");
		ctxt->wellFormed = 0;
	    } else
	        NEXT;
	} else if (CUR == '"'){
	    NEXT;
	    if ((CUR == 'n') && (NXT(1) == 'o')) {
	        standalone = 0;
		SKIP(2);
	    } else if ((CUR == 'y') && (NXT(1) == 'e') &&
	               (NXT(2) == 's')) {
	        standalone = 1;
                SKIP(3);
            } else {
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt,
		        "standalone accepts only 'yes' or 'no'\r\n");
		ctxt->wellFormed = 0;
	    }
	    if (CUR != '"') {
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt, "String not closed\r\n");
		ctxt->wellFormed = 0;
	    } else
	        NEXT;
	} else {
            if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, "Standalone value not found\r\n");
	    ctxt->wellFormed = 0;
        }
    }
    return(standalone);
}

/**
 * xmlParseVersionNum:
 * @ctxt:  an XML parser context
 *
 * parse the XML version value.
 *
 * [26] VersionNum ::= ([a-zA-Z0-9_.:] | '-')+
 *
 * Returns the string giving the XML version number, or NULL
 */
CHAR *
xmlParseVersionNum(xmlParserCtxtPtr ctxt) {
    const CHAR *q = CUR_PTR;
    CHAR *ret;

    while (IS_CHAR(CUR) &&
           (((CUR >= 'a') && (CUR <= 'z')) ||
            ((CUR >= 'A') && (CUR <= 'Z')) ||
            ((CUR >= '0') && (CUR <= '9')) ||
            (CUR == '_') || (CUR == '.') ||
	    (CUR == ':') || (CUR == '-'))) NEXT;
    ret = xmlStrndup(q, CUR_PTR - q);
    return(ret);
}
/**
 * xmlParseVersionInfo:
 * @ctxt:  an XML parser context
 * 
 * parse the XML version.
 *
 * [24] VersionInfo ::= S 'version' Eq (' VersionNum ' | " VersionNum ")
 * 
 * [25] Eq ::= S? '=' S?
 *
 * Returns the version string, e.g. "1.0"
 */

CHAR *
xmlParseVersionInfo(xmlParserCtxtPtr ctxt) 
{
    CHAR *version = NULL;
    const CHAR *q;

    if ((CUR == 'v') && (NXT(1) == 'e') &&
        (NXT(2) == 'r') && (NXT(3) == 's') &&
	(NXT(4) == 'i') && (NXT(5) == 'o') &&
	(NXT(6) == 'n')) {
	SKIP(7);
	SKIP_BLANKS;
	if (CUR != '=') {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, "xmlParseVersionInfo : expected '='\r\n");
	    ctxt->wellFormed = 0;
	    return(NULL);
        }
	NEXT;
	SKIP_BLANKS;
	if (CUR == '"') {
	    NEXT;
	    q = CUR_PTR;
	    version = xmlParseVersionNum(ctxt);
	    if (CUR != '"') {
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt, "String not closed\r\n%.50s\r\n", q);
		ctxt->wellFormed = 0;
	    } else
	        NEXT;
	} else if (CUR == '\''){
	    NEXT;
	    q = CUR_PTR;
	    version = xmlParseVersionNum(ctxt);
	    if (CUR != '\'') {
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt, "String not closed\r\n%.50s\r\n", q);
		ctxt->wellFormed = 0;
	    } else
	        NEXT;
	} else {
#if 1 /*OXYGEN*/
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt,
		      "xmlParseVersionInfo : expected ' or \"\r\n");
#endif
		ctxt->wellFormed = 0;
	}
    }
    return(version);
}


/**
 * xmlParseXMLDecl:
 * @ctxt:  an XML parser context
 * 
 * parse an XML declaration header
 *
 * [23] XMLDecl ::= '<?xml' VersionInfo EncodingDecl? SDDecl? S? '?>'
 */

void
xmlParseXMLDecl(xmlParserCtxtPtr ctxt)
{
    CHAR *version;

    /*
     * We know that '<?xml' is here.
     */
    SKIP(5);

    if (!IS_BLANK(CUR)) {
#if 1 /*OXYGEN*/
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "Blank needed after '<?xml'\r\n");
#endif
	ctxt->wellFormed = 0;
    }
    SKIP_BLANKS;

    /*
     * We should have the VersionInfo here.
     */
    version = xmlParseVersionInfo(ctxt);
    if (version == NULL)
	version = xmlCharStrdup(XML_DEFAULT_VERSION);
    ctxt->doc = xmlNewDoc(version);
    xmlMemFree(version);
#if 1 /*OXYGEN*/
    /*
     * We may have the encoding declaration
     */
    if (!IS_BLANK(CUR)) {
        if ((CUR == '?') && (NXT(1) == '>')) {
	    SKIP(2);
	    return;
	}
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "Blank needed here\r\n");
	ctxt->wellFormed = 0;
    }
    ctxt->doc->encoding = xmlParseEncodingDecl(ctxt);

    /*
     * We may have the standalone status.
     */
    if ((ctxt->doc->encoding != NULL) && (!IS_BLANK(CUR))) {
        if ((CUR == '?') && (NXT(1) == '>')) {
	    SKIP(2);
	    return;
	}
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "Blank needed here\r\n");
	ctxt->wellFormed = 0;
    }
    SKIP_BLANKS;
    ctxt->doc->standalone = xmlParseSDDecl(ctxt);
#endif

    SKIP_BLANKS;
    if ((CUR == '?') && (NXT(1) == '>')) {
        SKIP(2);
    } else if (CUR == '>') {
        /* Deprecated old WD ... */
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "XML declaration must end-up with '?>'\r\n");
	ctxt->wellFormed = 0;
	NEXT;
    } else {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "parsing XML declaration: '?>' expected\r\n");
	ctxt->wellFormed = 0;
	MOVETO_ENDTAG(CUR_PTR);
	NEXT;
    }
}

/**
 * xmlParseComment:
 * @ctxt:  an XML parser context
 * @create: should we create a node, or just skip the content
 *
 * Skip an XML (SGML) comment <!-- .... -->
 *  This may or may not create a node (depending on the context)
 *  The spec says that "For compatibility, the string "--" (double-hyphen)
 *  must not occur within comments. "
 *
 * [15] Comment ::= '<!--' ((Char - '-') | ('-' (Char - '-')))* '-->'
 *
 * TODO: this should call a SAX function which will handle (or not) the
 *       creation of the comment !
 *
 * Returns the comment node, or NULL
 */
xmlNodePtr
xmlParseComment(xmlParserCtxtPtr ctxt, int create) 
{
    xmlNodePtr ret = NULL;
    const CHAR *q, *start;
    const CHAR *r;
    CHAR *val;

    /*
     * Check that there is a comment right here.
     */
    if ((CUR != '<') || (NXT(1) != '!') ||
        (NXT(2) != '-') || (NXT(3) != '-')) return(NULL);

    SKIP(4);
    start = q = CUR_PTR;
    NEXT;
    r = CUR_PTR;
    NEXT;
    while (IS_CHAR(CUR) &&
           ((CUR == ':') || (CUR != '>') ||
	    (*r != '-') || (*q != '-'))) {
	if ((*r == '-') && (*q == '-')) {
#if 1 /*OXYGEN*/
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt,
	       "Comment must not contain '--' (double-hyphen)`\r\n");
#endif
	    ctxt->wellFormed = 0;
	}
        NEXT;r++;q++;
    }
    if (!IS_CHAR(CUR)) {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "Comment not terminated \r\n<!--%.50s\r\n", start);
	ctxt->wellFormed = 0;
    } else {
        NEXT;
	if (create) {
	    val = xmlStrndup(start, q - start);
	    ret = xmlNewDocComment(ctxt->doc, val);
	    xmlMemFree(val);
	}
    }
    return(ret);
}

/**
 * xmlParseQuotedString:
 * @ctxt:  an XML parser context
 *
 * [OLD] Parse and return a string between quotes or doublequotes
 *
 * Returns the string parser or NULL.
 */
CHAR *
xmlParseQuotedString(xmlParserCtxtPtr ctxt) {
    CHAR *ret = NULL;
    const CHAR *q;

    if (CUR == '"') {
        NEXT;
	q = CUR_PTR;
	while (IS_CHAR(CUR) && (CUR != '"')) NEXT;
	if (CUR != '"') {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, "String not closed \"%.50s\"\r\n", q);
	    ctxt->wellFormed = 0;
        } else {
            ret = xmlStrndup(q, CUR_PTR - q);
	    NEXT;
	}
    } else if (CUR == '\''){
        NEXT;
	q = CUR_PTR;
	while (IS_CHAR(CUR) && (CUR != '\'')) NEXT;
	if (CUR != '\'') {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, "String not closed \"%.50s\"\r\n", q);
	    ctxt->wellFormed = 0;
        } else {
            ret = xmlStrndup(q, CUR_PTR - q);
	    NEXT;
	}
    }
    return(ret);
}

/**
 * xmlParseNamespace:
 * @ctxt:  an XML parser context
 *
 * [OLD] xmlParseNamespace: parse specific PI '<?namespace ...' constructs.
 *
 * This is what the older xml-name Working Draft specified, a bunch of
 * other stuff may still rely on it, so support is still here as
 * if ot was declared on the root of the Tree:-(
 */

void
xmlParseNamespace(xmlParserCtxtPtr ctxt) 
{
    CHAR *href = NULL;
    CHAR *prefix = NULL;
    int garbage = 0;

    /*
     * We just skipped "namespace" or "xml:namespace"
     */
    SKIP_BLANKS;

    while (IS_CHAR(CUR) && (CUR != '>')) {
	/*
	 * We can have "ns" or "prefix" attributes
	 * Old encoding as 'href' or 'AS' attributes is still supported
	 */
	if ((CUR == 'n') && (NXT(1) == 's')) {
	    garbage = 0;
	    SKIP(2);
	    SKIP_BLANKS;

	    if (CUR != '=') continue;
	    NEXT;
	    SKIP_BLANKS;

	    href = xmlParseQuotedString(ctxt);
	    SKIP_BLANKS;
	} else if ((CUR == 'h') && (NXT(1) == 'r') &&
	    (NXT(2) == 'e') && (NXT(3) == 'f')) {
	    garbage = 0;
	    SKIP(4);
	    SKIP_BLANKS;

	    if (CUR != '=') continue;
	    NEXT;
	    SKIP_BLANKS;

	    href = xmlParseQuotedString(ctxt);
	    SKIP_BLANKS;
	} else if ((CUR == 'p') && (NXT(1) == 'r') &&
	           (NXT(2) == 'e') && (NXT(3) == 'f') &&
	           (NXT(4) == 'i') && (NXT(5) == 'x')) {
	    garbage = 0;
	    SKIP(6);
	    SKIP_BLANKS;

	    if (CUR != '=') continue;
	    NEXT;
	    SKIP_BLANKS;

	    prefix = xmlParseQuotedString(ctxt);
	    SKIP_BLANKS;
	} else if ((CUR == 'A') && (NXT(1) == 'S')) {
	    garbage = 0;
	    SKIP(2);
	    SKIP_BLANKS;

	    if (CUR != '=') continue;
	    NEXT;
	    SKIP_BLANKS;

	    prefix = xmlParseQuotedString(ctxt);
	    SKIP_BLANKS;
	} else if ((CUR == '?') && (NXT(1) == '>')) {
	    garbage = 0;
	    CUR_PTR ++;
	} else {
            /*
	     * Found garbage when parsing the namespace
	     */
	    if (!garbage)
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt, "xmlParseNamespace found garbage\r\n");
	    ctxt->wellFormed = 0;
            NEXT;
        }
    }

    MOVETO_ENDTAG(CUR_PTR);
    NEXT;

    /*
     * Register the DTD.
     */
    if (href != NULL)
        xmlNewGlobalNs(ctxt->doc, href, prefix);

    if (prefix != NULL) xmlMemFree(prefix);
    if (href != NULL) xmlMemFree(href);
}


/**
 * xmlParseName:
 * @ctxt:  an XML parser context
 *
 * parse an XML name.
 *
 * [4] NameChar ::= Letter | Digit | '.' | '-' | '_' | ':' |
 *                  CombiningChar | Extender
 *
 * [5] Name ::= (Letter | '_' | ':') (NameChar)*
 *
 * [6] Names ::= Name (S Name)*
 *
 * Returns the Name parsed or NULL
 */

CHAR *
xmlParseName(xmlParserCtxtPtr ctxt) {
    const CHAR *q;
    CHAR *ret = NULL;

    if (!IS_LETTER(CUR) && (CUR != '_') &&
        (CUR != ':')) return(NULL);
    q = NEXT;

    while ((IS_LETTER(CUR)) || (IS_DIGIT(CUR)) ||
           (CUR == '.') || (CUR == '-') ||
	   (CUR == '_') || (CUR == ':') || 
	   (IS_COMBINING(CUR)) ||
	   (IS_EXTENDER(CUR)))
	NEXT;
    
    ret = xmlStrndup(q, CUR_PTR - q);

    return(ret);
}

/**
 * xmlParsePITarget:
 * @ctxt:  an XML parser context
 * 
 * parse the name of a PI
 *
 * [17] PITarget ::= Name - (('X' | 'x') ('M' | 'm') ('L' | 'l'))
 *
 * Returns the PITarget name or NULL
 */

CHAR *
xmlParsePITarget(xmlParserCtxtPtr ctxt) 
{
    CHAR *name;

    name = xmlParseName(ctxt);
    if ((name != NULL) && (name[3] == 0) &&
        ((name[0] == 'x') || (name[0] == 'X')) &&
        ((name[1] == 'm') || (name[1] == 'M')) &&
        ((name[2] == 'l') || (name[2] == 'L'))) {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "xmlParsePItarget: invalid name prefix 'xml'\r\n");
	return(NULL);
    }
    return(name);
}


/**
 * xmlParsePI:
 * @ctxt:  an XML parser context
 * 
 * parse an XML Processing Instruction.
 *
 * [16] PI ::= '<?' PITarget (S (Char* - (Char* '?>' Char*)))? '?>'
 *
 * The processing is transfered to SAX once parsed.
 */

void
xmlParsePI(xmlParserCtxtPtr ctxt) 
{
    CHAR *target;

    if ((CUR == '<') && (NXT(1) == '?')) {
	/*
	 * this is a Processing Instruction.
	 */
	SKIP(2);

	/*
	 * Parse the target name and check for special support like
	 * namespace.
	 *
	 * TODO : PI handling should be dynamically redefinable using an
	 *        API. Only namespace should be in the code IMHO ...
	 */
        target = xmlParsePITarget(ctxt);
	if (target != NULL) {
	    /*
	     * Support for the old Processing Instruction related to namespace.
	     */
	    if ((target[0] == 'n') && (target[1] == 'a') &&
		(target[2] == 'm') && (target[3] == 'e') &&
		(target[4] == 's') && (target[5] == 'p') &&
		(target[6] == 'a') && (target[7] == 'c') &&
		(target[8] == 'e')) {
		xmlParseNamespace(ctxt);
	    } else if ((target[0] == 'x') && (target[1] == 'm') &&
		       (target[2] == 'l') && (target[3] == ':') &&
		       (target[4] == 'n') && (target[5] == 'a') &&
		       (target[6] == 'm') && (target[7] == 'e') &&
		       (target[8] == 's') && (target[9] == 'p') &&
		       (target[10] == 'a') && (target[11] == 'c') &&
		       (target[12] == 'e')) {
		xmlParseNamespace(ctxt);
	    } else {
	        const CHAR *q = CUR_PTR;

		while (IS_CHAR(CUR) &&
		       ((CUR != '?') || (NXT(1) != '>')))
		    NEXT;
		if (!IS_CHAR(CUR)) {
#if 1 /*OXYGEN*/
		    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		        ctxt->sax->error(ctxt,
			  "xmlParsePI: PI %s never end ...\r\n", target);
#endif
		    ctxt->wellFormed = 0;
		} else {
		    CHAR *tmpdata;

		    tmpdata = xmlStrndup(CUR_PTR, CUR_PTR - q);
		    SKIP(2);

		    /*
		     * SAX: PI detected.
		     */
		    if (ctxt->sax) 
			ctxt->sax->processingInstruction(ctxt, target, tmpdata);
		    /*
		     * Unknown PI, ignore it !
		     */
#if 0 /*OXYGEN*/
		    else 
			xmlParserWarning(ctxt,
		           "xmlParsePI : skipping unknown PI %s\r\n",
				         target);
#endif
	            xmlMemFree(tmpdata);
                }
	    }
	    xmlMemFree(target);
	} else {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, "xmlParsePI : no target name\r\n");
	    ctxt->wellFormed = 0;

	    /********* Should we try to complete parsing the PI ???
	    while (IS_CHAR(CUR) &&
		   (CUR != '?') && (CUR != '>'))
		NEXT;
	    if (!IS_CHAR(CUR)) {
		fprintf(stderr, "xmlParsePI: PI %s never end ...\r\n",
			target);
	    }
	     ********************************************************/
	}
    }
}


/**
 * xmlParseMisc:
 * @ctxt:  an XML parser context
 * 
 * parse an XML Misc* optionnal field.
 *
 * [27] Misc ::= Comment | PI |  S
 */

void
xmlParseMisc(xmlParserCtxtPtr ctxt) 
{
    while (((CUR == '<') && (NXT(1) == '?')) ||
           ((CUR == '<') && (NXT(1) == '!') &&
	    (NXT(2) == '-') && (NXT(3) == '-')) ||
           IS_BLANK(CUR)) {
        if ((CUR == '<') && (NXT(1) == '?')) {
	    xmlParsePI(ctxt);
	} else if (IS_BLANK(CUR)) {
	    NEXT;
	} else
	    xmlParseComment(ctxt, 0);
    }
}

/**
 * xmlNamespaceParseNCName:
 * @ctxt:  an XML parser context
 *
 * parse an XML namespace name.
 *
 * [NS 3] NCName ::= (Letter | '_') (NCNameChar)*
 *
 * [NS 4] NCNameChar ::= Letter | Digit | '.' | '-' | '_' |
 *                       CombiningChar | Extender
 *
 * Returns the namespace name or NULL
 */

CHAR *
xmlNamespaceParseNCName(xmlParserCtxtPtr ctxt)
{
    const CHAR *q;
    CHAR *ret = NULL;

    if (!IS_LETTER(CUR) && (CUR != '_')) return(NULL);
    q = NEXT;

    while ((IS_LETTER(CUR)) || (IS_DIGIT(CUR)) ||
           (CUR == '.') || (CUR == '-') ||
	   (CUR == '_') ||
	   (IS_COMBINING(CUR)) ||
	   (IS_EXTENDER(CUR)))
	NEXT;

    ret = xmlStrndup(q, CUR_PTR - q);

    return(ret);
}


/**
 * xmlNamespaceParseQName:
 * @ctxt:  an XML parser context
 * @prefix:  a CHAR ** 
 *
 * parse an XML qualified name
 *
 * [NS 5] QName ::= (Prefix ':')? LocalPart
 *
 * [NS 6] Prefix ::= NCName
 *
 * [NS 7] LocalPart ::= NCName
 *
 * Returns the function returns the local part, and prefix is updated
 *   to get the Prefix if any.
 */

CHAR *
xmlNamespaceParseQName(xmlParserCtxtPtr ctxt, CHAR **prefix)
{
    CHAR *ret = NULL;

    *prefix = NULL;
    ret = xmlNamespaceParseNCName(ctxt);
    if (CUR == ':') {
        *prefix = ret;
	NEXT;
	ret = xmlNamespaceParseNCName(ctxt);
    }

    return(ret);
}

/**
 * xmlParseEntityRef:
 * @ctxt:  an XML parser context
 *
 * parse ENTITY references declarations
 *
 * [68] EntityRef ::= '&' Name ';'
 *
 * Returns the entity ref string or NULL if directly as input stream.
 */
CHAR *
xmlParseEntityRef(xmlParserCtxtPtr ctxt) 
{
    CHAR *ret = NULL;
    const CHAR *q;
    CHAR *name;
    xmlEntityPtr ent;
    xmlParserInputPtr input = NULL;

    xmlDEBUG_trace(XML_DBG_LEVEL_TRACE_INFO, XML_DBG_MASK_XML_PARS,
                  "%s\r\n", "xmlParseEntityRef");
    q = CUR_PTR;
    if (CUR == '&') {
        NEXT;
        name = xmlParseName(ctxt);
	if (name == NULL) {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, "xmlParseEntityRef: no name\r\n");
	    ctxt->wellFormed = 0;
	} else {
	    if (CUR == ';') {
	        NEXT;
		/*
		 * Well Formedness Constraint if:
		 *   - standalone
		 * or
		 *   - no external subset and no external parameter entities
		 *     referenced
		 * then
		 *   the entity referenced must have been declared
		 *
		 * TODO: to be double checked !!!
		 */
		ent = xmlGetDocEntity(ctxt->doc, name);
		if ((ctxt->doc->standalone) ||
		    ((ctxt->doc->intSubset == NULL) &&
		     (ctxt->doc->extSubset == NULL))) {
		    if (ent == NULL) {
			if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
			    ctxt->sax->error(ctxt, 
			         "Entity '%s' not defined\r\n", name);
			ctxt->wellFormed = 0;
		    }
		}

		/*
		 * Well Formedness Constraint :
		 *   The referenced entity must be a parsed entity.
		 */
		if (ent != NULL) {
		    switch (ent->type) {
			case XML_INTERNAL_PARAMETER_ENTITY:
			case XML_EXTERNAL_PARAMETER_ENTITY:
			if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
			    ctxt->sax->error(ctxt, 
		     "Attempt to reference the parameter entity '%s'\r\n", name);
			ctxt->wellFormed = 0;
			break;
                        
			case XML_EXTERNAL_GENERAL_UNPARSED_ENTITY:
			if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
			    ctxt->sax->error(ctxt, 
		     "Attempt to reference unparsed entity '%s'\r\n", name);
			ctxt->wellFormed = 0;
			break;
		    }
		}

		/*
		 * Well Formedness Constraint :
		 *   The referenced entity must not lead to recursion !
		 */
		 
		/*
		 * We parsed the entity reference correctly, call SAX
		 * interface for the proper behaviour:
		 *   - get a new input stream
		 *   - or keep the reference inline
		 */
		if (ctxt->sax)
		    input = ctxt->sax->resolveEntity(ctxt, NULL, name);
		if (input != NULL) {
		    xmlPushInput(ctxt, input);
                } else {
#if 1 /*OXYGEN*/
		    ret = xmlStrndup(q, CUR_PTR - q);
#else
		    ret = xmlStrndup(ent->content, strlen(ent->content));
#endif
		}
	    } else {
		unsigned char cst[2] = { '&', 0 };

		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt,
		                     "xmlParseEntityRef: expecting ';'\r\n");
		ctxt->wellFormed = 0;
		ret = xmlStrndup(cst, 1);
		ret = xmlStrcat(ret, name);
	    }
	    xmlMemFree(name);
	}
    }
    return(ret);
}

/**
 * xmlParseReference:
 * @ctxt:  an XML parser context
 * 
 * parse Reference declarations
 *
 * [67] Reference ::= EntityRef | CharRef
 *
 * Returns the entity string or NULL if handled directly by pushing
 *      the entity value as the input.
 */
CHAR *
xmlParseReference(xmlParserCtxtPtr ctxt) 
{
    xmlDEBUG_trace(XML_DBG_LEVEL_TRACE_INFO, XML_DBG_MASK_XML_PARS,
                  "%s\r\n", "xmlParseReference");
#if 0 /*OXYGEN*/
    if ((CUR == '&') && (NXT(1) == '#')) {
        CHAR *val = xmlParseCharRef(ctxt);
	xmlParserInputPtr in;

	if (val != NULL) {
	    in = xmlNewStringInputStream(ctxt, val);
	    xmlPushInput(ctxt, in);
	}
	return(NULL);
    } else if (CUR == '&') {
        return(xmlParseEntityRef(ctxt));
    }
#else
    if (CUR == '&') {
        return(xmlParseEntityRef(ctxt));
    }
#endif
    return(NULL);
}



/**
 * xmlParseAttValue:
 * @ctxt:  an XML parser context
 *
 * parse a value for an attribute
 *
 * [10] AttValue ::= '"' ([^<&"] | Reference)* '"' |
 *                   "'" ([^<&'] | Reference)* "'"
 *
 * Returns the AttValue parsed or NULL.
 */

CHAR *
xmlParseAttValue(xmlParserCtxtPtr ctxt)
{
#if 1 /*OXYGEN*/
    CHAR *ret = NULL;
#else
    CHAR *ret = NULL, *cur;
#endif
    const CHAR *q;

    if (CUR == '"') {
        NEXT;

	q = CUR_PTR;
	while ((IS_CHAR(CUR)) && (CUR != '"')) {
	    if (CUR == '<') {
#if 1 /*OXYGEN*/
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt,
		       "Unescaped '<' not allowed in attributes values\r\n");
#endif
		ctxt->wellFormed = 0;
	    }
	    if (CUR == '&') {
#if 0 /*OXYGEN*/
	        ret = xmlStrncat(ret, q, CUR_PTR - q);
	        cur = xmlParseReference(ctxt);
		if (cur != NULL) {
		    /*
		     * Special case for '&amp;', we don't want to
		     * resolve it here since it will break later
		     * when searching entities in the string.
		     */
		    if ((cur[0] == '&') && (cur[1] == 0)) {
		        CHAR buf[6] = { '&', 'a', 'm', 'p', ';', 0 };
		        ret = xmlStrncat(ret, buf, 5);
		    } else
		        ret = xmlStrcat(ret, cur);
		    xmlMemFree(cur);
		}
		q = CUR_PTR;
#endif
	    } else 
	        NEXT;
	    /*
	     * Pop out finished entity references.
	     */
	    while ((CUR == 0) && (ctxt->inputNr > 1)) {
		if (CUR_PTR != q)
		    ret = xmlStrncat(ret, q, CUR_PTR - q);
	        xmlPopInput(ctxt);
		q = CUR_PTR;
	    }
	}
	if (!IS_CHAR(CUR)) {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, "Unfinished AttValue\r\n");
	    ctxt->wellFormed = 0;
	} else {
	    ret = xmlStrncat(ret, q, CUR_PTR - q);
	    NEXT;
        }
    } else if (CUR == '\'') {
        NEXT;
	q = CUR_PTR;
	while ((IS_CHAR(CUR)) && (CUR != '\'')) {
	    if (CUR == '<') {
#if 1 /*OXYGEN*/
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt,
		       "Unescaped '<' not allowed in attributes values\r\n");
#endif
		ctxt->wellFormed = 0;
	    }
	    if (CUR == '&') {
#if 0
	        ret = xmlStrncat(ret, q, CUR_PTR - q);
	        cur = xmlParseReference(ctxt);
		if (cur != NULL) {
		    /*
		     * Special case for '&amp;', we don't want to
		     * resolve it here since it will break later
		     * when searching entities in the string.
		     */
		    if ((cur[0] == '&') && (cur[1] == 0)) {
		        CHAR buf[6] = { '&', 'a', 'm', 'p', ';', 0 };
		        ret = xmlStrncat(ret, buf, 5);
		    } else
		        ret = xmlStrcat(ret, cur);
		    xmlMemFree(cur);
		}
		q = CUR_PTR;
#endif
	    } else 
	        NEXT;
	    /*
	     * Pop out finished entity references.
	     */
	    while ((CUR == 0) && (ctxt->inputNr > 1)) {
		if (CUR_PTR != q)
		    ret = xmlStrncat(ret, q, CUR_PTR - q);
	        xmlPopInput(ctxt);
		q = CUR_PTR;
	    }
	}
	if (!IS_CHAR(CUR)) {
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, "Unfinished AttValue\r\n");
	    ctxt->wellFormed = 0;
	} else {
	    ret = xmlStrncat(ret, q, CUR_PTR - q);
	    NEXT;
        }
    } else {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "AttValue: \" or ' expected\r\n");
	ctxt->wellFormed = 0;
    }
    
    return(ret);
}


/**
 * xmlParseAttribute:
 * @ctxt:  an XML parser context
 * @node:  the node carrying the attribute
 *
 * parse an attribute
 *
 * [41] Attribute ::= Name Eq AttValue
 *
 * [25] Eq ::= S? '=' S?
 *
 * With namespace:
 *
 * [NS 11] Attribute ::= QName Eq AttValue
 *
 * Also the case QName == xmlns:??? is handled independently as a namespace
 * definition.
 *
 * Returns the attribute just parsed of NULL in case of error.
 */

xmlAttrPtr
xmlParseAttribute(xmlParserCtxtPtr ctxt, xmlNodePtr node)
{
    CHAR *name, *val;
    CHAR *ns;
    CHAR *value = NULL;
    xmlAttrPtr ret;

    name = xmlNamespaceParseQName(ctxt, &ns);
    if (name == NULL) {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "error parsing attribute name\r\n");
	ctxt->wellFormed = 0;
        return(NULL);
    }

    /*
     * read the value
     */
    SKIP_BLANKS;
    if (CUR == '=') {
        NEXT;
	SKIP_BLANKS;
	value = xmlParseAttValue(ctxt);
    } else {
#if 1 /*OXYGEN*/
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt,
	       "Specification mandate value for attribute %s\r\n", name);
#endif
	ctxt->wellFormed = 0;
    }

    /*
     * Check whether it's a namespace definition
     */
    if ((ns == NULL) &&
        (name[0] == 'x') && (name[1] == 'm') && (name[2] == 'l') &&
        (name[3] == 'n') && (name[4] == 's') && (name[5] == 0)) {
	/* a default namespace definition */
	xmlNewNs(node, value, NULL);
	if (name != NULL) 
	    xmlMemFree(name);
	if (value != NULL)
	    xmlMemFree(value);
	return(NULL);
    }
    if ((ns != NULL) && (ns[0] == 'x') && (ns[1] == 'm') && (ns[2] == 'l') &&
        (ns[3] == 'n') && (ns[4] == 's') && (ns[5] == 0)) {
	/* a standard namespace definition */
	xmlNewNs(node, value, name);
	xmlMemFree(ns);
	if (name != NULL) 
	    xmlMemFree(name);
	if (value != NULL)
	    xmlMemFree(value);
	return(NULL);
    }

    /*
     * Well formedness requires at most one declaration of an attribute
     */
    if ((val = xmlGetProp(ctxt->node, name)) != NULL) {
        xmlMemFree(val);
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "Attribute %s redefined\r\n", name);
	ctxt->wellFormed = 0;
	ret = NULL;
    } else {
	ret = xmlNewProp(ctxt->node, name, NULL);
	if (ret != NULL)
	    ret->val = xmlStringGetNodeList(ctxt->doc, value);
    }

    if (ns != NULL)
      xmlMemFree(ns);
    if (value != NULL)
	xmlMemFree(value);
    xmlMemFree(name);
    return(ret);
}


/**
 * xmlParseStartTag:
 * @ctxt:  an XML parser context
 * 
 * parse a start of tag either for rule element or
 * EmptyElement. In both case we don't parse the tag closing chars.
 *
 * [40] STag ::= '<' Name (S Attribute)* S? '>'
 *
 * [44] EmptyElemTag ::= '<' Name (S Attribute)* S? '/>'
 *
 * With namespace:
 *
 * [NS 8] STag ::= '<' QName (S Attribute)* S? '>'
 *
 * [NS 10] EmptyElement ::= '<' QName (S Attribute)* S? '/>'
 *
 * Returns the XML new node or NULL.
 */

xmlNodePtr
xmlParseStartTag(xmlParserCtxtPtr ctxt)
{
    CHAR *namespace, *name;
    xmlNsPtr ns = NULL;
    xmlNodePtr ret = NULL;
    xmlNodePtr parent = ctxt->node;

    if (CUR != '<') return(NULL);
    NEXT;

    name = xmlNamespaceParseQName(ctxt, &namespace);
    if (name == NULL) {
#if 1 /*OXYGEN*/
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, 
	     "xmlParseStartTag: invalid element name\r\n");
#endif
	ctxt->wellFormed = 0;
        return(NULL);
    }

    /*
     * Note : the namespace resolution is deferred until the end of the
     *        attributes parsing, since local namespace can be defined as
     *        an attribute at this level.
     */
    ret = xmlNewDocNode(ctxt->doc, ns, name, NULL);
    if (ret == NULL) {
	if (namespace != NULL)
	    xmlMemFree(namespace);
	xmlMemFree(name);
        return(NULL);
    }

    /*
     * We are parsing a new node.
     */
    nodePush(ctxt, ret);

    /*
     * Now parse the attributes, it ends up with the ending
     *
     * (S Attribute)* S?
     */
    SKIP_BLANKS;
    while ((IS_CHAR(CUR)) &&
           (CUR != '>') && 
	   ((CUR != '/') || (NXT(1) != '>'))) {
	const CHAR *q = CUR_PTR;

	xmlParseAttribute(ctxt, ret);
	SKIP_BLANKS;

        if (q == CUR_PTR) {
#if 1 /*OXYGEN*/
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt, 
	         "xmlParseStartTag: problem parsing attributes\r\n");
#endif
	    ctxt->wellFormed = 0;
	    break;
	}
    }

    /*
     * Search the namespace
     */
    ns = xmlSearchNs(ctxt->doc, ret, namespace);
    if (ns == NULL) /* ret still doesn't have a parent yet ! */
	ns = xmlSearchNs(ctxt->doc, parent, namespace);
    xmlSetNs(ret, ns);
    if (namespace != NULL)
	xmlMemFree(namespace);

    /*
     * SAX: Start of Element !
     */
    if (ctxt->sax != NULL)
        ctxt->sax->startElement(ctxt, name);
    xmlMemFree(name);

    /*
     * Link the child element
     */
    if (ctxt->nodeNr < 2) return(ret);
    parent = ctxt->nodeTab[ctxt->nodeNr - 2];
    if (parent != NULL)
	xmlAddChild(parent, ctxt->node);

    return(ret);
}

/**
 * xmlParseEndTag:
 * @ctxt:  an XML parser context
 * @nsPtr:  the current node namespace definition
 * @tagPtr:  CHAR** receive the tag value
 *
 * parse an end of tag
 *
 * [42] ETag ::= '</' Name S? '>'
 *
 * With namespace
 *
 * [9] ETag ::= '</' QName S? '>'
 *    
 * tagPtr receive the tag name just read
 */

void
xmlParseEndTag(xmlParserCtxtPtr ctxt, xmlNsPtr *nsPtr, CHAR **tagPtr)
{
    CHAR *namespace, *name;
    xmlNsPtr ns = NULL;

    *nsPtr = NULL;
    *tagPtr = NULL;

    if ((CUR != '<') || (NXT(1) != '/')) {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "xmlParseEndTag: '</' not found\r\n");
	ctxt->wellFormed = 0;

	return;
    }
    SKIP(2);

    name = xmlNamespaceParseQName(ctxt, &namespace);

    /*
     * Search the namespace
     */
    ns = xmlSearchNs(ctxt->doc, ctxt->node, namespace);
    if (namespace != NULL)
	xmlMemFree(namespace);

    *nsPtr = ns;
    *tagPtr = name;

    /*
     * We should definitely be at the ending "S? '>'" part
     */
    SKIP_BLANKS;
    if ((!IS_CHAR(CUR)) || (CUR != '>')) {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "End tag : expected '>'\r\n");
	ctxt->wellFormed = 0;
    } else
	NEXT;

    return;
}

/**
 * areBlanks:
 * @ctxt:  an XML parser context
 * @str:  a CHAR *
 * @len:  the size of @str
 *
 * Is this a sequence of blank chars that one can ignore ?
 *
 * TODO: to be corrected accodingly to DTD information if available
 *
 * Returns 1 if ignorable 0 otherwise.
 */
#if 0
static int areBlanks(xmlParserCtxtPtr ctxt, const CHAR *str, int len) 
{
    int i;
    xmlNodePtr lastChild;

    for (i = 0;i < len;i++)
        if (!(IS_BLANK(str[i]))) return(0);

    if (CUR != '<') return(0);
    lastChild = xmlGetLastChild(ctxt->node);
    if (lastChild == NULL) {
        if (ctxt->node->content != NULL) return(0);
    } else if (xmlNodeIsText(lastChild))
        return(0);
    return(1);
}
#endif


/**
 * xmlParseCharData:
 * @ctxt:  an XML parser context
 * @cdata:  int indicating whether we are within a CDATA section
 *
 * parse a CharData section.
 * if we are within a CDATA section ']]>' marks an end of section.
 *
 * [14] CharData ::= [^<&]* - ([^<&]* ']]>' [^<&]*)
 */

void
xmlParseCharData(xmlParserCtxtPtr ctxt, int cdata) 
{
    const CHAR *q;

    q = CUR_PTR;
    while ((IS_CHAR(CUR)) && (CUR != '<') &&
           (CUR != '&')) {
	if ((CUR == ']') && (NXT(1) == ']') &&
	    (NXT(2) == '>')) {
	    if (cdata) break;
	    else {
#if 1 /*OXYGEN*/
		if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
		    ctxt->sax->error(ctxt,
		       "Sequence ']]>' not allowed in content\r\n");
#endif
		ctxt->wellFormed = 0;
	    }
	}
        NEXT;
    }
    if (q == CUR_PTR) return;

    /*
     * Ok the segment [q CUR_PTR] is to be consumed as chars.
     */
    if (ctxt->sax != NULL) {
#if 0 /*Do not care the content*/
	if (areBlanks(ctxt, q, (int)(CUR_PTR - q)))
	    ctxt->sax->ignorableWhitespace(ctxt, q, 0, (int)(CUR_PTR - q));
	else
#endif
	    ctxt->sax->characters(ctxt, q, 0, (int)(CUR_PTR - q));
    }
}


/**
 * xmlParseContent:
 * @ctxt:  an XML parser context
 *
 * Parse a content:
 *
 * [43] content ::= (element | CharData | Reference | CDSect | PI | Comment)*
 */

void
xmlParseContent(xmlParserCtxtPtr ctxt) 
{
#if 1 /*OXYGEN*/
    xmlNodePtr ret = NULL;
#endif

    while ((CUR != '<') || (NXT(1) != '/')) {
	const CHAR *test = CUR_PTR;
#if 0 /*OXYGEN*/
        ret = NULL;
#endif
	/*
	 * First case :  a sub-element.
	 */
	if (CUR == '<') {
#if 1 /*OXYGEN*/
	    xmlParseElement(ctxt);
#else
	    ret = xmlParseElement(ctxt);
#endif
	}
#if 1 /*WCN*/
	/*
	 * Fifth case : a reference. If if has not been resolved,
	 *    parsing returns it's Name, create the node 
	 */
	else if (CUR == '&') {
	    CHAR *val = xmlParseReference(ctxt);
	    if (val != NULL) {
	        if (val[0] != '&') {
		    /*
		     * inline predefined entity.
		     */
                    if (ctxt->sax != NULL)
			ctxt->sax->characters(ctxt, val, 0, xmlStrlen(val));
		} else {
		    /*
		     * user defined entity, create a node.
		     */
		    ret = xmlNewReference(ctxt->doc, val);
		    xmlAddChild(ctxt->node, ret);
		}
		xmlMemFree(val);
	    }
	}
#endif

	/*
	 * Last case, text. Note that References are handled directly.
	 */
	else {
	    xmlParseCharData(ctxt, 0);
	}
	/*
	 * Pop-up of finished entities.
	 */
	while ((CUR == 0) && (ctxt->inputNr > 1)) xmlPopInput(ctxt);

	if (test == CUR_PTR) {
#if 1 /*OXYGEN*/
	    if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	        ctxt->sax->error(ctxt,
		     "detected an error in element content\r\n");
#endif
	    ctxt->wellFormed = 0;
            break;
	}
    }
}


/**
 * xmlParseElement:
 * @ctxt:  an XML parser context
 *
 * parse an XML element, this is highly recursive
 *
 * [39] element ::= EmptyElemTag | STag content ETag
 *
 * [41] Attribute ::= Name Eq AttValue
 *
 * Returns the XML new node or NULL
 */


xmlNodePtr
xmlParseElement(xmlParserCtxtPtr ctxt)
{
    xmlNodePtr ret;
#if 1 /*OXYGEN*/
    const CHAR *openTag = CUR_PTR;
#endif
    xmlParserNodeInfo node_info;
    CHAR *endTag;
    xmlNsPtr endNs;

    /* Capture start position */
    node_info.begin_pos = CUR_PTR - ctxt->input->base;
    node_info.begin_line = ctxt->input->line;

    ret = xmlParseStartTag(ctxt);
    if (ret == NULL) {
        return(NULL);
    }

    /*
     * Check for an Empty Element.
     */
    if ((CUR == '/') && (NXT(1) == '>')) {
        SKIP(2);
	if (ctxt->sax != NULL)
	    ctxt->sax->endElement(ctxt, ret->name);

	/*
	 * end of parsing of this node.
	 */
	nodePop(ctxt);

	return(ret);
    }

    if (CUR == '>') NEXT;
    else {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "Couldn't find end of Start Tag\r\n%.30s\r\n",
	                     openTag);
	ctxt->wellFormed = 0;

	/*
	 * end of parsing of this node.
	 */
	nodePop(ctxt);
       
#if 1 /*OXYGEN*/
        return ret;
#else
	return(NULL);
#endif
    }

    /*
     * Parse the content of the element:
     */
    xmlParseContent(ctxt);


    if (!IS_CHAR(CUR)) {
#if 1 /*OXYGEN*/
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt,
	         "Premature end of data in tag %.30s\r\n", openTag);
#endif

        xmlDEBUG_trace(XML_DBG_LEVEL_TRACE_INFO, XML_DBG_MASK_XML_PARS,
                  "CUR = %x\r\n", CUR);
	ctxt->wellFormed = 0;

	/*
	 * end of parsing of this node.
	 */
	nodePop(ctxt);

#if 1 /*OXYGEN*/
        return ret;
#else
	return(NULL);
#endif
    }

    /*
     * parse the end of tag: '</' should be here.
     */
    xmlParseEndTag(ctxt, &endNs, &endTag);

    /*
     * Check that the Name in the ETag is the same as in the STag.
     */
    if (endNs != ret->ns) {
#if 1 /*OXYGEN*/
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, 
	    "Start and End tags don't use the same namespace\r\n%.30s\r\n%.30s\r\n",
	               openTag, endTag);
#endif
	ctxt->wellFormed = 0;
    }


    if (endTag == NULL ) {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "The End tag has no name\r\n%.30s\r\n", openTag);
	ctxt->wellFormed = 0;
    } else if (xmlStrcmp(ret->name, endTag)) {
#if 1 /*OXYGEN*/
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, 
	    "Start and End tags don't use the same name\r\n%.30s\r\n%.30s\r\n",
	               openTag, endTag);
#endif
	ctxt->wellFormed = 0;
    }
    /*
     * SAX: End of Tag
     */
    else if (ctxt->sax != NULL)
        ctxt->sax->endElement(ctxt, endTag);

    if (endTag != NULL)
	xmlMemFree(endTag);
#if 0 /*OXYGEN*/
    /* Capture end position and add node */
    if ( ret != NULL && ctxt->record_info ) {
      node_info.end_pos = CUR_PTR - ctxt->input->base;
      node_info.end_line = ctxt->input->line;
      node_info.node = ret;
      xmlParserAddNodeInfo(ctxt, &node_info);
    }
#endif

    /*
     * end of parsing of this node.
     */
    nodePop(ctxt);

    return(ret);
}


/**
 * xmlParseDocument :
 * @ctxt:  an XML parser context
 * 
 * parse an XML document (and build a tree if using the standard SAX
 * interface).
 *
 * [1] document ::= prolog element Misc*
 *
 * [22] prolog ::= XMLDecl? Misc* (doctypedecl Misc*)?
 *
 * Returns 0, -1 in case of error. the parser context is augmented
 *                as a result of the parsing.
 */

int
xmlParseDocument(xmlParserCtxtPtr ctxt) 
{
    xmlDefaultSAXHandlerInit();

    /*
     * SAX: beginning of the document processing.
     */
    if (ctxt->sax) 
        ctxt->sax->setDocumentLocator(ctxt, &xmlDefaultSAXLocator);
    if (ctxt->sax)
        ctxt->sax->startDocument(ctxt);

    /*
     * We should check for encoding here and plug-in some
     * conversion code TODO !!!!
     */

    /*
     * Wipe out everything which is before the first '<'
     */
    if (IS_BLANK(CUR)) {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL)) {
#if 1 /*OXYGEN*/
	    ctxt->sax->error(ctxt,
	    "Extra spaces at the beginning of the document are not allowed\r\n");
#endif
        }
	ctxt->wellFormed = 0;
	SKIP_BLANKS;
    }

    while ('<' != CUR) {
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL)) {
	    ctxt->sax->error(ctxt,
	    "Extra bytes at the beginning of the document are not allowed\r\n");
        }
	ctxt->wellFormed = 0;
        //if (NULL == *(NEXT)) goto Exit;
        if (0 == *(NEXT)) goto Exit;
    }

    if (CUR == 0) {
#if 1 /*OXYGEN*/
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "Document is empty\r\n");
#endif
	ctxt->wellFormed = 0;
    }

    /*
     * Check for the XMLDecl in the Prolog.
     */
    if ((CUR == '<') && (NXT(1) == '?') &&
        (NXT(2) == 'x') && (NXT(3) == 'm') &&
	(NXT(4) == 'l')) {
	xmlParseXMLDecl(ctxt);
	/* SKIP_EOL(cur); */
	SKIP_BLANKS;
    } else {
	CHAR *version;

	version = xmlCharStrdup(XML_DEFAULT_VERSION);
	ctxt->doc = xmlNewDoc(version);
	xmlMemFree(version);
    }

#if 1 /*OXYGEN*/
    if (ctxt->doc == NULL) {
	ctxt->wellFormed = 0;
        goto Exit;
    }
#endif

#if 0 /*OXYGEN*/
    /*
     * The Misc part of the Prolog
     */
    xmlParseMisc(ctxt);
    /*
     * Then possibly doc type declaration(s) and more Misc
     * (doctypedecl Misc*)?
     */
    if ((CUR == '<') && (NXT(1) == '!') &&
	(NXT(2) == 'D') && (NXT(3) == 'O') &&
	(NXT(4) == 'C') && (NXT(5) == 'T') &&
	(NXT(6) == 'Y') && (NXT(7) == 'P') &&
	(NXT(8) == 'E')) {
	xmlParseDocTypeDecl(ctxt);
	xmlParseMisc(ctxt);
    }
#endif

    /*
     * Time to start parsing the tree itself
     */
    ctxt->doc->root = xmlParseElement(ctxt);

#if 0 /*OXYGEN*/
    /*
     * The Misc part at the end
     */
    xmlParseMisc(ctxt);
#endif

    if (CUR != 0) {
#if 1 /*OXYGEN*/
	if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL)) {
	    ctxt->sax->error(ctxt,
	        "Extra content at the end of the document\r\n");
        }
#endif
	ctxt->wellFormed = 0;
    }

#if 1 /*OXYGEN*/
Exit:
#endif
    /*
     * SAX: end of the document processing.
     */
    if (ctxt->sax) 
        ctxt->sax->endDocument(ctxt);
    if (! ctxt->wellFormed) return(-1);
    return(0);
}







/**
 * xmlInitNodeInfoSeq :
 * @seq:  a node info sequence pointer
 *
 * -- Initialize (set to initial state) node info sequence
 */
void
xmlInitNodeInfoSeq(xmlParserNodeInfoSeqPtr seq)
{
  seq->length = 0;
  seq->maximum = 0;
  seq->buffer = NULL;
}


/**
 * xmlInitParserCtxt:
 * @ctxt:  an XML parser context
 *
 * Initialize a parser context
 */

void
xmlInitParserCtxt(xmlParserCtxtPtr ctxt)
{
  /* Allocate the Input stack */
  ctxt->inputTab = (xmlParserInputPtr *) xmlMemAlloc(5 * sizeof(xmlParserInputPtr));
  ctxt->inputNr = 0;
  ctxt->inputMax = 5;
  ctxt->input = NULL;

  /* Allocate the Node stack */
  ctxt->nodeTab = (xmlNodePtr *) xmlMemAlloc(10 * sizeof(xmlNodePtr));
  ctxt->nodeNr = 0;
  ctxt->nodeMax = 10;
  ctxt->node = NULL;

  ctxt->sax = &xmlDefaultSAXHandler;
  ctxt->doc = NULL;
  ctxt->wellFormed = 1;
  ctxt->record_info = 0;
  xmlInitNodeInfoSeq(&ctxt->node_seq);
}





/**
 * xmlSAXParseMemory :
 * @sax:  the SAX handler block
 * @buffer:  an pointer to a char array
 * @size:  the siwe of the array
 * @recovery:  work in recovery mode, i.e. tries to read no Well Formed
 *             documents
 *
 * parse an XML in-memory block and use the given SAX function block
 * to handle the parsing callback. If sax is NULL, fallback to the default
 * DOM tree building routines.
 * 
 * TODO : plug some encoding conversion routines here. !!!
 *
 * Returns the resulting document tree
 */
xmlDocPtr
xmlSAXParseMemory(xmlSAXHandlerPtr sax, char *buffer, int size,
                            int recovery)
{
    xmlDocPtr ret;
    xmlParserCtxtPtr ctxt;
    xmlParserInputPtr input;

    //buffer[size - 1] = '\0';

    ctxt = (xmlParserCtxtPtr) xmlMemAlloc(sizeof(xmlParserCtxt));
    if (ctxt == NULL) {
        //perror("xmlMemAlloc");
		return(NULL);
    }
    xmlInitParserCtxt(ctxt);
    if (sax != NULL) ctxt->sax = sax;
    input = (xmlParserInputPtr) xmlMemAlloc(sizeof(xmlParserInput));
    if (input == NULL) {
        //perror("xmlMemAlloc");
        xmlMemFree(ctxt->nodeTab);
		xmlMemFree(ctxt->inputTab);
		xmlMemFree(ctxt);
		return(NULL);
    }

    input->filename = NULL;
    input->line = 1;
    input->col = 1;

    /*
     * TODO : plug some encoding conversion routines here. !!!
     */
    input->base = (unsigned char *)buffer;
    input->cur = (unsigned char *)buffer;

    inputPush(ctxt, input);

    xmlParseDocument(ctxt);

    if ((ctxt->wellFormed) || recovery) ret = ctxt->doc;
    else {
       ret = NULL;
       xmlFreeDoc(ctxt->doc);
       ctxt->doc = NULL;
    }
    xmlMemFree(ctxt->nodeTab);
    xmlMemFree(ctxt->inputTab);
    if (input->filename != NULL)
    xmlMemFree((char *)input->filename);
    xmlMemFree(input);

    xmlMemFree(ctxt);
    
    return(ret);
}









/**
 * xmlParseMemory :
 * @buffer:  an pointer to a char array
 * @size:  the size of the array
 *
 * parse an XML in-memory block and build a tree.
 * 
 * Returns the resulting document tree
 */

xmlDocPtr xmlParseMemory(char *buffer, int size) 
{
   return(xmlSAXParseMemory(NULL, buffer, size, 1));
}

/**
 * xmlParserFindNodeInfoIndex:
 * @seq:  a node info sequence pointer
 * @node:  an XML node pointer
 *
 * 
 * xmlParserFindNodeInfoIndex : Find the index that the info record for
 *   the given node is or should be at in a sorted sequence
 *
 * Returns a long indicating the position of the record
 */
unsigned long xmlParserFindNodeInfoIndex(const xmlParserNodeInfoSeq* seq,
                                         const xmlNode* node)
{
  unsigned long upper, lower, middle;
  int found = 0;

  /* Do a binary search for the key */
  lower = 1;
  upper = seq->length;
  middle = 0;
  while ( lower <= upper && !found) {
    middle = lower + (upper - lower) / 2;
    if ( node == seq->buffer[(int)(middle - 1)].node )
      found = 1;
    else if ( node < seq->buffer[(int)(middle - 1)].node )
      upper = middle - 1;
    else
      lower = middle + 1;
  }

  /* Return position */
  if ( middle == 0 || seq->buffer[(int)(middle - 1)].node < node )
    return middle;
  else 
    return middle - 1;
}

#if 0 /*OXYGEN*/
/**
 * xmlParserAddNodeInfo:
 * @ctxt:  an XML parser context
 * @info:  a node info sequence pointer
 *
 * Insert node info record into the sorted sequence
 */
void
xmlParserAddNodeInfo(xmlParserCtxtPtr ctxt, 
                     const xmlParserNodeInfo* info)
{
  unsigned long pos;
#if 0 /*OXYGEN*/
  static unsigned int block_size = 5;
#else
  static unsigned int block_size = 12;
#endif

  /* Find pos and check to see if node is already in the sequence */
  pos = xmlParserFindNodeInfoIndex(&ctxt->node_seq, info->node);
  if ( pos < ctxt->node_seq.length
       && ctxt->node_seq.buffer[pos].node == info->node ) {
    ctxt->node_seq.buffer[pos] = *info;
  }

  /* Otherwise, we need to add new node to buffer */
  else {
    /* Expand buffer by 5 if needed */
    if ( ctxt->node_seq.length + 1 > ctxt->node_seq.maximum ) {
      xmlParserNodeInfo* tmp_buffer;
      unsigned int byte_size = (sizeof(*ctxt->node_seq.buffer)
                                *(ctxt->node_seq.maximum + block_size));

      if ( ctxt->node_seq.buffer == NULL )
        tmp_buffer = (xmlParserNodeInfo*)xmlMemAlloc(byte_size);
      else 
#if 0 /*OXYGEN*/
        tmp_buffer = (xmlParserNodeInfo*)realloc(ctxt->node_seq.buffer, byte_size);
#else
	  //exit(1);
          return;
#endif

      if ( tmp_buffer == NULL ) {
        if ((ctxt->sax != NULL) && (ctxt->sax->error != NULL))
	    ctxt->sax->error(ctxt, "");//Out of memory\r\n");
        return;
      }
      ctxt->node_seq.buffer = tmp_buffer;
      ctxt->node_seq.maximum += block_size;
    }

    /* If position is not at end, move elements out of the way */
    if ( pos != ctxt->node_seq.length ) {
      unsigned long i;

      for ( i = ctxt->node_seq.length; i > pos; i-- )
        ctxt->node_seq.buffer[i] = ctxt->node_seq.buffer[i - 1];
    }
  
    /* Copy element and increase length */
    ctxt->node_seq.buffer[pos] = *info;
    ctxt->node_seq.length++;
  }   
}
#endif
