/* After the #includes, the function prototypes and the global variable, we come to the
 main function. There the semaphore is created with a call to semget, which returns the
 semaphore ID. If the program is the first to be called (i.e. it's called with a parameter
 and argc > 1), a call is made to set_semvalue to initialize the semaphore and op_char is
 set to X. */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include <sys/sem.h>

#include "semun.h"
#include "semaphore_api.h"

int xmldbg_sem_id;
int xmltree_sem_id;
int xmlmem_sem_id;
int almem_sem_id;
int ucpopen_sem_id;
int ucpdbg_sem_id;


void semaphore_init(void)
{
	static int sem_init=0;
	if (0 == sem_init) {
		xmldbg_sem_id = semget((key_t)1234, 1, 0666 | IPC_CREAT);
		xmltree_sem_id = semget((key_t)1235, 1, 0666 | IPC_CREAT);
		xmlmem_sem_id = semget((key_t)1236, 1, 0666 | IPC_CREAT);
		almem_sem_id = semget((key_t)1237, 1, 0666 | IPC_CREAT);
		ucpopen_sem_id = semget((key_t)1238, 1, 0666 | IPC_CREAT);
		ucpdbg_sem_id = semget((key_t)1239, 1, 0666 | IPC_CREAT);
		if (!set_semvalue(xmldbg_sem_id)) {
			fprintf(stderr, "Failed to initialize semaphore, xmldbg_sem_id\n");
			exit(EXIT_FAILURE);
		}
		if (!set_semvalue(xmltree_sem_id)) {
			fprintf(stderr, "Failed to initialize semaphore, xmltree_sem_id\n");
			exit(EXIT_FAILURE);
		}
		if (!set_semvalue(xmlmem_sem_id)) {
			fprintf(stderr, "Failed to initialize semaphore, xmlmem_sem_id\n");
			exit(EXIT_FAILURE);
		}
		if (!set_semvalue(almem_sem_id)) {
			fprintf(stderr, "Failed to initialize semaphore, almem_sem_id\n");
			exit(EXIT_FAILURE);
		}
		if (!set_semvalue(ucpopen_sem_id)) {
			fprintf(stderr, "Failed to initialize semaphore, ucpopen_sem_id\n");
			exit(EXIT_FAILURE);
		}
		if (!set_semvalue(ucpdbg_sem_id)) {
			fprintf(stderr, "Failed to initialize semaphore, ucpdbg_sem_id\n");
			exit(EXIT_FAILURE);
		}
	}
}

void semaphore_exit(void)
{
	del_semvalue(xmldbg_sem_id);
	del_semvalue(xmltree_sem_id);
	del_semvalue(xmlmem_sem_id);
	del_semvalue(almem_sem_id);
	del_semvalue(ucpopen_sem_id);
	del_semvalue(ucpdbg_sem_id);
}

/* The function set_semvalue initializes the semaphore using the SETVAL command in a
 semctl call. We need to do this before we can use the semaphore. */

int set_semvalue(int sem_id)
{
    union semun sem_union;

    sem_union.val = 1;
    if (semctl(sem_id, 0, SETVAL, sem_union) == -1) return(0);
    return(1);
}

/* The del_semvalue function has almost the same form, except the call to semctl uses
 the command IPC_RMID to remove the semaphore's ID. */

void del_semvalue(int sem_id)
{
    union semun sem_union;
    
    if (semctl(sem_id, 0, IPC_RMID, sem_union) == -1)
        fprintf(stderr, "Failed to delete semaphore\n");
}

/* semaphore_p changes the semaphore by -1 (waiting). */

int semaphore_p(int sem_id)
{
    struct sembuf sem_b;
    
    sem_b.sem_num = 0;
    sem_b.sem_op = -1; /* P() */
    sem_b.sem_flg = SEM_UNDO;
    if (semop(sem_id, &sem_b, 1) == -1) {
        fprintf(stderr, "semaphore_p failed\n");
        return(0);
    }
    return(1);
}

/* semaphore_v is similar except for setting the sem_op part of the sembuf structure to 1,
 so that the semaphore becomes available. */

int semaphore_v(int sem_id)
{
    struct sembuf sem_b;
    
    sem_b.sem_num = 0;
    sem_b.sem_op = 1; /* V() */
    sem_b.sem_flg = SEM_UNDO;
    if (semop(sem_id, &sem_b, 1) == -1) {
        fprintf(stderr, "semaphore_v failed\n");
        return(0);
    }
    return(1);
}

