#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <linux/videodev.h>
#include <linux/soundcard.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <getopt.h>
#include <pthread.h>
#include <fcntl.h>
#include <linux/types.h>
#include <ralink_gpio.h>
#include "nvram.h"
#include "i2c_drv.h"

#if ( defined(MODEL_TVIP651W) || defined(MODEL_TVIP651WI) )
#define	SWING_DEBUG			0

#define	SWING_PID_FILE			"/var/run/swing.pid"
#define	SWING_DEVICE			"/dev/i2cM0"

#define	SWING_STATE_IDLE		0
#define	SWING_STATE_PRESET		1
#define	SWING_STATE_HORIZONTAL		2
#define	SWING_STATE_VERTICAL		3

#define	SWING_DIRECTION_ASCENDING	0x01
#define	SWING_DIRECTION_DESCENDING	0x02
#define	SWING_DIRECTION_REVERSE		0x80

#define	SWING_DEFAULT_STEP		5
#define	SWING_DEFAULT_STAY_TIME		7
#define	SWING_DEFAULT_LOOP_COUNT	0	// infinite loop
#define	SWING_PRESET_POSITION_NUMBER	24

typedef	struct	_SWING_PRESET_POS_ {
	int	PanPosition;
	int	TiltPosition;
} SWING_PRESET_POS, *pSWING_PRESET_POS;

int	fd_SysInfo, fd_Swing;
char	*SavedPidFile;
int	SwingState;
int	MaxPanPosition, MaxTiltPosition;
__u64	LastPanTiltToggleTime;
SWING_PRESET_POS	PresetPosition[SWING_PRESET_POSITION_NUMBER];
struct ACTIVEUSER	ActiveUser[MAX_USER_NUMBER];

int CheckActiveUser()
{
int	idx;

	ioctl(fd_SysInfo, (SYSIF_READ|(ACTIVE_USER_TABLE<<SUBCMD_IDX_SHIFT)|
			((sizeof(struct ACTIVEUSER)*MAX_USER_NUMBER)<<BUF_LEN_SHIFT)),
		 	&ActiveUser[0]);
	for ( idx = 0; idx < MAX_USER_NUMBER; idx++ ) {
		if ( ActiveUser[idx].Pid != 0 )	return 1;
	}
	return 0;
}

void SwingWaitMotorStop(void)
{
PAN_TILT_COMMAND pan_tilt_cmd;

	memset(&pan_tilt_cmd, 0, sizeof(PAN_TILT_COMMAND));
	do {
		usleep(100*1000);
		pan_tilt_cmd.Function = PT_FUNC_GET_MOVING_STATUS;
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
	} while (  (pan_tilt_cmd.MovingStatus & 0x03) != 0 );
#if ( SWING_DEBUG != 0 )
	fprintf(stderr, "(MotorStop)\n");
#endif
}

int SwingProcess(void)
{
PAN_TILT_COMMAND swing_status, pan_tilt_cmd;
int	active_swing_function=PT_FUNC_SWING_STOP, loop_count, stay_time;
int	idx, preset_idx, direction;
int	saved_pan_pos, saved_tilt_pos, pan_pos, tilt_pos, steps;
struct	timeval	time_value;
__u64	time_current, time_swing;

	pan_tilt_cmd.Function = PT_FUNC_GET_MAX_POSITION;
	ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
	MaxPanPosition = pan_tilt_cmd.PanPosition;
	MaxTiltPosition = pan_tilt_cmd.TiltPosition;
	while ( 1 ) {
		usleep(200*1000);
		swing_status.Function = PT_FUNC_SWING_STATUS;
		if ( ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &swing_status) < 0 ) {
			SwingState = SWING_STATE_IDLE;
			active_swing_function = PT_FUNC_SWING_STOP;
			continue;
		}
		//
		if ( swing_status.Function != active_swing_function ) {
			active_swing_function = swing_status.Function;
			loop_count = swing_status.LoopCount;
			stay_time = swing_status.StayTime;
			if ( swing_status.StayTime == 0 )	stay_time = SWING_DEFAULT_STAY_TIME * 1000;
			else	stay_time = swing_status.StayTime * 1000;
			time_swing = 0;
			if ( active_swing_function != PT_FUNC_SWING_STOP ) {
				pan_tilt_cmd.Function = PT_FUNC_GET_POTISION;
				ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
				saved_pan_pos = pan_tilt_cmd.PanPosition;
				saved_tilt_pos = pan_tilt_cmd.TiltPosition;
			}
			switch ( active_swing_function ) {
			case PT_FUNC_SWING_STOP:
#if 0
				pan_tilt_cmd.Function = PT_FUNC_GO_POSITION;
				pan_tilt_cmd.PanPosition = saved_pan_pos;
				pan_tilt_cmd.TiltPosition = saved_tilt_pos;
				ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
#endif
				SwingState = SWING_STATE_IDLE;
				break;
			case PT_FUNC_SWING_PRESET:
				preset_idx = -1;
				for ( idx = 0 ; idx < SWING_PRESET_POSITION_NUMBER; idx++ ) {
				char	temp_buf[64], *p_pos;
					// format: Name/PanPosition/TiltPosition , MaxNameLength=12
					PresetPosition[idx].PanPosition = -1;
					PresetPosition[idx].TiltPosition = -1;
					sprintf(temp_buf, "SetPosition%d", idx+1);
					p_pos = nvram_bufget(RT2860_NVRAM, temp_buf);
					memset(temp_buf, 0, 64);
					memcpy(temp_buf, p_pos, 63);
					if ( temp_buf[0] == '\0' )	continue;
					p_pos = strstr(temp_buf, "/");
					if ( p_pos != NULL ) {
						p_pos++;
						PresetPosition[idx].PanPosition = atoi(p_pos);
					}
					p_pos = strstr(p_pos, "/");
					if ( p_pos != NULL ) {
						p_pos++;
						PresetPosition[idx].TiltPosition = atoi(p_pos);
					}
					if ( (preset_idx == -1) && (PresetPosition[idx].PanPosition != -1) &&
								(PresetPosition[idx].TiltPosition != -1) ) {
						preset_idx = idx;
					}
#if ( SWING_DEBUG != 0 )
					fprintf(stderr, "swing: preset(%d) pan=(%d), tilt=(%d)\n", idx+1,
						PresetPosition[idx].PanPosition, PresetPosition[idx].TiltPosition);
#endif
				}
				nvram_close(RT2860_NVRAM);
				if ( preset_idx != -1)	SwingState = SWING_STATE_PRESET;
				break;
			case PT_FUNC_SWING_HORIZONTAL:
				steps = swing_status.Steps;
				if ( steps == 0 )	steps = SWING_DEFAULT_STEP;
				SwingState = SWING_STATE_HORIZONTAL;
				pan_pos = saved_pan_pos + steps;
				direction = SWING_DIRECTION_ASCENDING;
				break;
			case PT_FUNC_SWING_VERTICAL:
				steps = swing_status.Steps;
				if ( steps == 0 )	steps = SWING_DEFAULT_STEP;
				SwingState = SWING_STATE_VERTICAL;
				tilt_pos = saved_tilt_pos + steps;
				direction = SWING_DIRECTION_ASCENDING;
				break;
			} // ....switch ( active_swing_function )
		} // ....if ( swing_status.Function != active_swing_function )
		if ( SwingState == SWING_STATE_IDLE )	continue;
		//
		if ( CheckActiveUser() == 0 ) {
			SwingState = SWING_STATE_IDLE;
			pan_tilt_cmd.Function = PT_FUNC_SWING_STOP;
			ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
			continue;
		} // ....if ( CheckActiveUser() == 0 )
		gettimeofday(&time_value, NULL);
		time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
		if ( time_swing > time_current )	continue;
		//
		switch ( SwingState ) {
		case SWING_STATE_PRESET:
			for ( idx = preset_idx; idx < SWING_PRESET_POSITION_NUMBER; idx++ ) {
				if ( (PresetPosition[idx].PanPosition != -1) &&
					(PresetPosition[idx].TiltPosition != -1) )	break;
			}
			if ( idx < SWING_PRESET_POSITION_NUMBER ) {
			int	tilt_movement, max_movement;
				pan_tilt_cmd.Function = PT_FUNC_GO_POSITION;
				pan_tilt_cmd.PanPosition = PresetPosition[idx].PanPosition;
				pan_tilt_cmd.TiltPosition = PresetPosition[idx].TiltPosition;
				ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
				max_movement = pan_tilt_cmd.PanPosition - PresetPosition[preset_idx].PanPosition;
				tilt_movement = pan_tilt_cmd.TiltPosition - PresetPosition[preset_idx].TiltPosition;
				if ( tilt_movement > max_movement )	max_movement = tilt_movement;
				preset_idx = idx + 1;
				SwingWaitMotorStop();
				gettimeofday(&time_value, NULL);
				time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
				time_swing = time_current + (__u64)stay_time;
			} else {
#if ( SWING_DEBUG != 0 )
				fprintf(stderr, "(%d)===================================\n", loop_count);
#endif
				loop_count--;
				if ( loop_count == 0 ) { // stop preset swing
					SwingState = SWING_STATE_IDLE;
					pan_tilt_cmd.Function = PT_FUNC_SWING_STOP;
					ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
				} else {
					preset_idx = 0;	// start next swing when process
					time_swing = 0;
					if ( loop_count < 0 )	loop_count = 0;
				}
			}
			break;
		case SWING_STATE_HORIZONTAL:
			if ( (direction & SWING_DIRECTION_ASCENDING) != 0 ) {
#if ( PAN_MOTOR_ASCENDING_FOR_LEFT == 1 )
				pan_tilt_cmd.Function = PT_FUNC_MOVE_LEFT;
#else
				pan_tilt_cmd.Function = PT_FUNC_MOVE_RIGHT;
#endif
				pan_tilt_cmd.Steps = steps;
				ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
				if ( (direction & SWING_DIRECTION_REVERSE) != 0 ) {
					if ( (pan_tilt_cmd.PanPosition + steps) >= saved_pan_pos ) {
#if ( SWING_DEBUG != 0 )
						fprintf(stderr, "(%d)===================================\n", loop_count);
#endif
						loop_count--;
						if ( loop_count == 0 ) { // stop preset swing
							SwingState = SWING_STATE_IDLE;
							pan_tilt_cmd.Function = PT_FUNC_SWING_STOP;
							ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
						} else {
							if ( loop_count < 0 )	loop_count = 0;
							direction &= ~SWING_DIRECTION_REVERSE;
						}
					}
				}
				if ( pan_tilt_cmd.PanPosition >= MaxPanPosition ) {
					direction = SWING_DIRECTION_DESCENDING;
				}
			} else if ( direction == SWING_DIRECTION_DESCENDING ) {
#if ( PAN_MOTOR_ASCENDING_FOR_LEFT == 1 )
				pan_tilt_cmd.Function = PT_FUNC_MOVE_RIGHT;
#else
				pan_tilt_cmd.Function = PT_FUNC_MOVE_LEFT;
#endif
				pan_tilt_cmd.Steps = steps;
				ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
				if ( pan_tilt_cmd.PanPosition <= 0 ) {
					direction = SWING_DIRECTION_ASCENDING + SWING_DIRECTION_REVERSE;
				}
			}
			SwingWaitMotorStop();
			gettimeofday(&time_value, NULL);
			time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
			time_swing = time_current + (__u64)stay_time;
			break;
		case SWING_STATE_VERTICAL:
			if ( (direction & SWING_DIRECTION_ASCENDING) != 0 ) {
#if ( TILT_MOTOR_ASCENDING_FOR_UP == 1 )
				pan_tilt_cmd.Function = PT_FUNC_MOVE_UP;
#else
				pan_tilt_cmd.Function = PT_FUNC_MOVE_DOWN;
#endif
				pan_tilt_cmd.Steps = steps;
				ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
				if ( (direction & SWING_DIRECTION_REVERSE) != 0 ) {
					if ( (pan_tilt_cmd.TiltPosition + steps) >= saved_tilt_pos ) {
#if ( SWING_DEBUG != 0 )
						fprintf(stderr, "(%d)===================================\n", loop_count);
#endif
						loop_count--;
						if ( loop_count == 0 ) { // stop preset swing
							SwingState = SWING_STATE_IDLE;
							pan_tilt_cmd.Function = PT_FUNC_SWING_STOP;
							ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
						} else {
							if ( loop_count < 0 )	loop_count = 0;
							direction &= ~SWING_DIRECTION_REVERSE;
						}
					}
				}
				if ( pan_tilt_cmd.TiltPosition >= MaxTiltPosition ) {
					direction = SWING_DIRECTION_DESCENDING;
				}
			} else if ( direction == SWING_DIRECTION_DESCENDING ) {
#if ( TILT_MOTOR_ASCENDING_FOR_UP == 1 )
				pan_tilt_cmd.Function = PT_FUNC_MOVE_DOWN;
#else
				pan_tilt_cmd.Function = PT_FUNC_MOVE_UP;
#endif
				pan_tilt_cmd.Steps = steps;
				ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_cmd);
				if ( pan_tilt_cmd.TiltPosition <= 0 ) {
					direction = SWING_DIRECTION_ASCENDING + SWING_DIRECTION_REVERSE;
				}
			}
			SwingWaitMotorStop();
			gettimeofday(&time_value, NULL);
			time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
			time_swing = time_current + (__u64)stay_time;
			break;
		} // ....switch ( SwingState )
	}
	return 0;
}

void TestPanTiltCommand(int Function, int Step, int LoopCount, int StayTime, int PanPosition, int TiltPosition)
{
PAN_TILT_COMMAND pan_tilt_command;
__u64	time_current, time_start;
struct	timeval	time_value;

	pan_tilt_command.Function = Function;
	switch ( Function ) {
	case PT_FUNC_GET_VERSION:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: pan-tilt version (%d)\n", pan_tilt_command.Version);
		break;
	case PT_FUNC_GO_HOME:
	case PT_FUNC_CALIBRATION:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: cmd (%s) send\n", (Function==PT_FUNC_GO_HOME)?"Home":"Calibration");
		break;
	case PT_FUNC_GET_HOME:
	case PT_FUNC_GET_POTISION:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: (%s) Pan=%d, Tilt=%d\n", (Function==PT_FUNC_GET_HOME)?"Home":"Position",
				pan_tilt_command.PanPosition, pan_tilt_command.TiltPosition);
		break;
	case PT_FUNC_MOVE_LEFT:
	case PT_FUNC_MOVE_RIGHT:
		pan_tilt_command.Steps = Step;
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: Move (%s) step (%d)\n", (Function==PT_FUNC_MOVE_LEFT)?"Left":"Right", Step);
		break;
	case PT_FUNC_MOVE_UP:
	case PT_FUNC_MOVE_DOWN:
		pan_tilt_command.Steps = Step;
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: Move (%s) step (%d)\n", (Function==PT_FUNC_MOVE_UP)?"Up":"Down", Step);
		break;
	case PT_FUNC_GO_POSITION:
		pan_tilt_command.PanPosition = PanPosition;
		pan_tilt_command.TiltPosition = TiltPosition;
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: GoTo Pan=%d, Tilt=%d\n", PanPosition, TiltPosition);
		break;
	case PT_FUNC_PAN_SPEED:
	case PT_FUNC_TILT_SPEED:
		pan_tilt_command.Speed = Step;
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: Set(%s) Speed=(%d)\n", (Function==PT_FUNC_PAN_SPEED)?"Pan":"Tilt", Step);
		break;
	case PT_FUNC_LIMIT_SWITCH_TEST:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		gettimeofday(&time_value, NULL);
		time_start = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
		pan_tilt_command.Function = PT_FUNC_LIMIT_SWITCH_STATUS;
		while ( ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command) < 0 );
		gettimeofday(&time_value, NULL);
		time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
		fprintf(stderr, "swing: Limit Switch (%x) Pan=(%s) Tilt=(%s) (time=%d ms)\n",
			pan_tilt_command.SwitchStatus, (pan_tilt_command.SwitchStatus&0x02)?"Fail":"OK",
			(pan_tilt_command.SwitchStatus&0x01)?"Fail":"OK", (__u32)(time_current-time_start));
		break;
	case PT_FUNC_LOST_STEP_TEST_PAN:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		gettimeofday(&time_value, NULL);
		time_start = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
		pan_tilt_command.Function = PT_FUNC_LOST_STEP_STATUS_PAN;
		while ( ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command) < 0 );
		SwingWaitMotorStop();
		gettimeofday(&time_value, NULL);
		time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
		fprintf(stderr, "swing: Pan Lost Step Test (lost=%d)(time=%d ms)\n",
					pan_tilt_command.PanPosition, (__u32)(time_current-time_start));
		break;
	case PT_FUNC_LOST_STEP_TEST_TILT:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		gettimeofday(&time_value, NULL);
		time_start = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
		pan_tilt_command.Function = PT_FUNC_LOST_STEP_STATUS_TILT;
		while ( ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command) < 0 );
		SwingWaitMotorStop();
		gettimeofday(&time_value, NULL);
		time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
		fprintf(stderr, "swing: Tilt Lost Step Test (lost=%d)(time=%d ms)\n",
					pan_tilt_command.TiltPosition, (__u32)(time_current-time_start));
		break;
	case PT_FUNC_GET_MAX_POSITION:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: Max position pan=(%d), tilt=(%d)\n",
				pan_tilt_command.PanPosition, pan_tilt_command.TiltPosition);
		break;
	case PT_FUNC_GET_MIN_POSITION:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: Min position pan=(%d), tilt=(%d)\n",
				pan_tilt_command.PanPosition, pan_tilt_command.TiltPosition);
		break;
	case PT_FUNC_SWING_STATUS:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: Status = (%d)\n", pan_tilt_command.Function);
		break;
	case PT_FUNC_SWING_STOP:
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: swing (Stop)!!\n");
		break;
	case PT_FUNC_SWING_PRESET:
		if ( LoopCount == -1 )	LoopCount = SWING_DEFAULT_LOOP_COUNT;	// infinite loop
		pan_tilt_command.LoopCount = LoopCount;
		if ( StayTime == -1 )	StayTime = SWING_DEFAULT_STAY_TIME;
		pan_tilt_command.StayTime = StayTime;
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: swing (Preset) LoopCount=(%d) StayTime=(%d)\n", LoopCount, StayTime);
		break;
	case PT_FUNC_SWING_HORIZONTAL:
	case PT_FUNC_SWING_VERTICAL:
		if ( Step == -1 )	Step = SWING_DEFAULT_STEP;
		pan_tilt_command.Steps = Step;
		if ( LoopCount == -1 )	LoopCount = SWING_DEFAULT_LOOP_COUNT;	// infinite loop
		pan_tilt_command.LoopCount = LoopCount;
		if ( StayTime == -1 )	StayTime = SWING_DEFAULT_STAY_TIME;
		pan_tilt_command.StayTime = StayTime;
		ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
		fprintf(stderr, "swing: swing (%s) Step=(%d) LoopCount=(%d) StayTime=(%d)\n",
				(Function==PT_FUNC_SWING_HORIZONTAL)?"Horizontal":"Vertical",
				Step, LoopCount, StayTime);
		break;
	default:
		fprintf(stderr, "swing: invalid pan-tilt command (%d)\n", Function);
		break;
	}
}

void SignalHandler(int sig)
{
struct USER_SIG_FLAG	user_sig_flag;

	switch ( sig ) {
	case SIGTERM:
		fprintf(stderr, "swing: signal SIGTERM(%d)\n", sig);
		if ( fd_SysInfo )	close(fd_SysInfo);
		if ( fd_Swing )		close(fd_Swing);
		exit(0);
		break;
	case SIGUSR1:
		fprintf(stderr, "swing: signal SIGUSR1(%d)\n", sig);
		user_sig_flag.process = 6;	//ptz

		if (ioctl(fd_SysInfo, (SYSIF_READ | (USER1_SIG_FLAG<<SUBCMD_IDX_SHIFT) |
				(sizeof(struct USER_SIG_FLAG)<<BUF_LEN_SHIFT)), &user_sig_flag) < 0) {
			perror("ioctl");
			return;
		}
		if( (user_sig_flag.sig_flag & PTZ_USER1_SIG_PRIVACY) != 0 ) {
			/* Do Privacy Function */
			PAN_TILT_COMMAND pan_tilt_command;
			__u64	time_current;
			struct	timeval	time_value;
			char	*pPrivacyEnable;
			int	privacy_enable;

			pPrivacyEnable = nvram_bufget(RT2860_NVRAM, "PrivacyButton");
			privacy_enable = atoi(pPrivacyEnable);
			nvram_close(RT2860_NVRAM);
			if ( privacy_enable == 0 ) {
				fprintf(stderr, "swing: PrivacyButton Disable\n");
				return;
			}
			gettimeofday(&time_value, NULL);
			time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
			if ( (time_current - LastPanTiltToggleTime) > (__u64)(10 * 1000) ) {
				LastPanTiltToggleTime = time_current;
				pan_tilt_command.Function = PT_FUNC_GET_ACTIVE_MODE;
				ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
				pan_tilt_command.Function = PT_FUNC_SET_PRIVACY;
				if ( pan_tilt_command.ActiveMode == PT_ACTIVE_MODE_PRIVACY ) {
					fprintf(stderr, "swing: switch to Run Mode\n");
					pan_tilt_command.Privacy = 0;
				} else {
					fprintf(stderr, "swing: switch to Privacy Mode\n");
					pan_tilt_command.Privacy = 1;
				}
				ioctl(fd_Swing, RT2880_I2C_PAN_TILT, &pan_tilt_command);
			} else {
				fprintf(stderr, "swing: skip change Privacy/Run mode.\n");
			} // ....if ( (time_current - LastPanTiltToggleTime) > (10 * 1000) )
		}
		break;
	default:
		fprintf(stderr, "swing: unknown signal (%d)\n", sig);
		break;
	}
} /* SignalHandler() */

void DaemonMode(void)
{
int	fr=0;

	fr = fork();
	if( fr < 0 ) {
		fprintf(stderr, "fork() failed\n");
		exit(1);
	}
	if ( fr > 0 ) {
		fprintf(stderr, "swing: forked to background (%d)\n", fr);
		exit(0);
	}

	if( setsid() < 0 ) {
		fprintf(stderr, "setsid() failed\n");
		exit(1);
	}
	umask(0);
	chdir("/");
	close(STDIN_FILENO);
	close(STDOUT_FILENO);
#if ( SWING_DEBUG == 0 )
	close(STDERR_FILENO);
#endif
} /* DaemonMode() */

static void FileDeleteAtExit(void)
{
	if ( SavedPidFile )	unlink(SavedPidFile);
} /* FileDeleteAtExit() */

int PidFileGet(const char *PidFile)
{
int	pid_fd, pid_num;
char	buffer[10];

	if (!PidFile)	return -1;

	pid_fd = open(PidFile, O_RDONLY, 0644);
	if (pid_fd < 0)	return -1;
	pid_num = read(pid_fd, buffer, 10);
	if ( pid_num <= 0 ) {
		printf("read %s error : %d\n", PidFile, pid_num);
		close(pid_fd);
		return -1;
	}
	pid_num = atoi(buffer);
	return pid_num;
} /* PidFileAcquire() */

int PidFileAcquire(const char *PidFile)
{
int	pid_fd;

	if (!PidFile)	return -1;

	pid_fd = open(PidFile, O_CREAT | O_WRONLY, 0644);
	if (pid_fd < 0) {
		printf("Unable to open pidfile %s\n", PidFile);
	} else {
		lockf(pid_fd, F_LOCK, 0);
		if ( !SavedPidFile )	atexit(FileDeleteAtExit);
		SavedPidFile = (char *) PidFile;
	}
	return pid_fd;
} /* PidFileAcquire() */

void PidFileWriteRelease(int pid_fd)
{
FILE	*out;

	if (pid_fd < 0) return;

	if ((out = fdopen(pid_fd, "w")) != NULL) {
		fprintf(out, "%d\n", getpid());
		fclose(out);
	}
	lockf(pid_fd, F_UNLCK, 0);
	close(pid_fd);
} /* PidFileWriteRelease() */

void Help(char *ProgramName)
{
	fprintf(stderr, "------------------------------------------------------------------\n");
	fprintf(stderr, "Usage: %s\n", ProgramName);
	fprintf(stderr, " [-h]..........: display this help\n");
#if ( SWING_DEBUG != 0 )
	fprintf(stderr, " [-f func].....: function\n");
	fprintf(stderr, " [-s step].....: step/speed\n");
	fprintf(stderr, " [-c count]....: loop count\n");
	fprintf(stderr, " [-d sec]......: swing stay time\n");
	fprintf(stderr, " [-p postion]..: pan position\n");
	fprintf(stderr, " [-t position].: tilt position\n");
#endif
	fprintf(stderr, "------------------------------------------------------------------\n");
} /* Help() */

int main(int argc, char *argv[])
{
int	saved_pid, pid_fd, idx;
int	function_id, step, loop_count, stay_time, pan_position, tilt_position;
ralink_gpio_reg_info	gpio_info;

	function_id = step = loop_count = stay_time = pan_position = tilt_position = -1;
	LastPanTiltToggleTime = 0;
	while(1) {
		int option_index = 0, c=0;
		static struct option long_options[] = \
			{
			{"h", no_argument, 0, 0},		/* help */
			{"f", required_argument, 0, 0},		/* function */
			{"s", required_argument, 0, 0},		/* step/speed */
			{"c", required_argument, 0, 0},		/* loop count */
			{"d", required_argument, 0, 0},		/* swing stay time */
			{"p", required_argument, 0, 0},		/* pan position */
			{"t", required_argument, 0, 0},		/* tilt position */
			{0, 0, 0, 0}
			};

		c = getopt_long_only(argc, argv, "", long_options, &option_index);

		/* no more options to parse */
		if (c == -1) break;

		/* unrecognized option */
		if(c=='?'){ Help(argv[0]); return 0; }

		switch (option_index) {
		case 0:/* "-h" : help */
			Help(argv[0]);
			return 0;
		case 1:/* "-f" : function */
			function_id = atoi(optarg);
			break;
		case 2:/* "-s" : step */
			step = atoi(optarg);
			break;
		case 3:/* "-c" : loop count */
			loop_count = atoi(optarg);
			break;
		case 4:/* "-d" : swing stay time */
			stay_time = atoi(optarg);
			break;
		case 5:/* "-p" : pan position */
			pan_position = atoi(optarg);
			break;
		case 6:/* "-t" : tilt position */
			tilt_position = atoi(optarg);
			break;
		default:
			Help(argv[0]);
			return 0;
		} // switch (option_index)
	} // while(1)
	fd_Swing = open(SWING_DEVICE, O_RDONLY);
	if ( fd_Swing < 0 ) {
		fprintf(stderr, "Swing device: %s open fail, code %d\n", SWING_DEVICE, fd_Swing);
		exit(1);
	}
	if ( function_id != -1 ) {	// test pan-tilt command
		TestPanTiltCommand(function_id, step, loop_count, stay_time, pan_position, tilt_position);
		close(fd_Swing);
		exit(0);
	}
	saved_pid = PidFileGet(SWING_PID_FILE);
	if ( saved_pid > 0 ) {
		printf("process running pid=%d.....\n", saved_pid);
		return 0;
	}
	/* ignore SIGPIPE (send if transmitting to closed sockets) */
	signal(SIGPIPE, SIG_IGN);
	if (signal(SIGTERM, SignalHandler) == SIG_ERR) {
		printf("could not register signal handler\n");
		close(fd_Swing);
		exit(1);
	}
	fd_SysInfo = open(SYSIF_DEV, O_RDONLY);
	if ( fd_SysInfo < 0 ) {
		fprintf(stderr, "SysInfo: %s open fail, code %d\n", SYSIF_DEV, fd_SysInfo);
		close(fd_Swing);
		exit(1);
	}
	/* fork to the background */
	DaemonMode();
	// active Privacy button
	gpio_info.pid = getpid();
	gpio_info.irq = PRIVACY_BUTTON;
	//set gpio direction to input
	if ( ioctl(fd_SysInfo, RALINK_GPIO_SET_DIR_IN, (1<<gpio_info.irq)) < 0 )	perror("gpio ioctl..dir");
	//enable gpio interrupt
	if ( ioctl(fd_SysInfo, RALINK_GPIO_ENABLE_INTP) < 0 )	perror("gpio ioctl..intp");
	//register my information
	if ( ioctl(fd_SysInfo, RALINK_GPIO_REG_IRQ, &gpio_info) < 0 )	perror("gpio ioctl...irq");
	//issue a handler to handle SIGUSR1
	signal(SIGUSR1, SignalHandler);
	// save pid
	pid_fd = PidFileAcquire(SWING_PID_FILE);
	PidFileWriteRelease(pid_fd);
	//
	SwingState = SWING_STATE_IDLE;
	SwingProcess();
	return 0;
}
#else
int main(int argc, char *argv[])
{
	fprintf(stderr, "swing: Pan-Tilt function not support\n");
	return 0;
}
#endif	// ...( defined(MODEL_TVIP651W) || defined(MODEL_TVIP651WI) )