#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <getopt.h>
#include <fcntl.h>
#include <linux/types.h>
#include "i2c_drv.h"

#define	I2C_DEBUG			1

#define	I2C_DEVICE			"/dev/i2cM0"

#define	I2C_OPTION_DEVICE_ADDRESS	0x01
#define	I2C_OPTION_CLOCK_DIVISOR	0x02
#define	I2C_OPTION_CONFIGURATION	0x04
#define	I2C_OPTION_SUB_ADDRESS		0x08
#define	I2C_OPTION_DATA			0x10
#define	I2C_OPTION_DUMP			0x40
#define	I2C_OPTION_TEST			0x80
#define	I2C_OPTION_CONFIG_MASK		(I2C_OPTION_DEVICE_ADDRESS+I2C_OPTION_CLOCK_DIVISOR+I2C_OPTION_CONFIGURATION)

#define	I2C_TEST_MODE_RW		0
#define	I2C_TEST_MODE_READ		1
#define	I2C_TEST_MODE_WRITE		2

int	fd_i2c;

void	I2CTest(int Mode, int Count, int SubAddress)
{
I2C_READ_WRITE	i2c_access;
int	idx, sub_addr_idx, total_test, total_error, i2c_data;
__u32	i2c_data_save, saved_data_buf[PT_ADDRESS_MAX];
__u64	time_current, time_start;
struct	timeval	time_value;

	i2c_access.Address = SubAddress;
	ioctl(fd_i2c, RT2880_I2C_READ, &i2c_access);
	i2c_data_save = i2c_access.Data;
	//
	total_test = total_error = 0;
	gettimeofday(&time_value, NULL);
	time_start = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
	switch ( Mode ) {
	case I2C_TEST_MODE_RW:
		for( idx = 0 ; idx < Count; idx ++ ) {
			total_test++;
			i2c_access.Address = SubAddress;
			i2c_data = idx & 0xff;
			i2c_access.Data = (__u32) i2c_data;
			ioctl(fd_i2c, RT2880_I2C_WRITE, &i2c_access);
			ioctl(fd_i2c, RT2880_I2C_READ, &i2c_access);
			if ( i2c_data != (int) i2c_access.Data ) {
				total_error++;
				fprintf(stderr, "<RW-OUT=%x, IN=%x>", i2c_data, i2c_access.Data);
			}
			gettimeofday(&time_value, NULL);
			time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
			if ( (int)(time_current - time_start) > 1000 ) {
				time_start = time_current;
				fprintf(stderr, " I2C test(RW)...total=%.8d, error=%.8d\n", total_test, total_error);
			}
		}
		fprintf(stderr, " I2C test....Mode=(RW), SubAddress=%x, total=%d, error=%d\n",
						SubAddress, total_test, total_error);
		break;
	case I2C_TEST_MODE_READ:
		for ( sub_addr_idx = 0; sub_addr_idx < PT_ADDRESS_MAX; sub_addr_idx++ ) {
			i2c_access.Address = sub_addr_idx;
			ioctl(fd_i2c, RT2880_I2C_READ, &i2c_access);
			saved_data_buf[sub_addr_idx] = i2c_access.Data;
		}
		for( idx = 0 ; idx < Count; idx ++ ) {
			total_test++;
			for ( sub_addr_idx = 0; sub_addr_idx < PT_ADDRESS_MAX; sub_addr_idx++ ) {
				i2c_access.Address = sub_addr_idx;
				ioctl(fd_i2c, RT2880_I2C_READ, &i2c_access);
				if ( saved_data_buf[sub_addr_idx] != i2c_access.Data ) {
					total_error++;
					fprintf(stderr, "<R-Addr=%x, Saved=%x, IN=%x>",
						sub_addr_idx, saved_data_buf[sub_addr_idx], i2c_access.Data);
				}
			}
			gettimeofday(&time_value, NULL);
			time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
			if ( (int)(time_current - time_start) > 1000 ) {
				time_start = time_current;
				fprintf(stderr, " I2C test(R)....total=%.8d, error=%.8d\n", total_test, total_error);
			}
		}
		fprintf(stderr, " I2C test....Mode=(R), SubAddressMax=%x, total=%d, error=%d\n",
						PT_ADDRESS_MAX, total_test, total_error);
		break;
	case I2C_TEST_MODE_WRITE:
		for ( sub_addr_idx = 0; sub_addr_idx < PT_ADDRESS_MAX; sub_addr_idx++ ) {
			i2c_access.Address = sub_addr_idx;
			ioctl(fd_i2c, RT2880_I2C_READ, &i2c_access);
			saved_data_buf[sub_addr_idx] = i2c_access.Data;
		}
		for( idx = 0 ; idx < Count; idx ++ ) {
			total_test++;
			for ( sub_addr_idx = 0; sub_addr_idx < PT_ADDRESS_MAX; sub_addr_idx++ ) {
				i2c_access.Address = sub_addr_idx;
				i2c_access.Data = saved_data_buf[sub_addr_idx];
				ioctl(fd_i2c, RT2880_I2C_WRITE, &i2c_access);
			}
			sub_addr_idx = (Count % PT_ADDRESS_MAX);
			i2c_access.Address = sub_addr_idx;
			ioctl(fd_i2c, RT2880_I2C_READ, &i2c_access);
			if ( saved_data_buf[sub_addr_idx] != i2c_access.Data ) {
				total_error++;
				fprintf(stderr, "<W-Addr=%x, Saved=%x, IN=%x>",
					sub_addr_idx, saved_data_buf[sub_addr_idx], i2c_access.Data);
			}
			gettimeofday(&time_value, NULL);
			time_current = ((__u64)time_value.tv_sec*1000) + ((__u64)time_value.tv_usec/1000);
			if ( (int)(time_current - time_start) > 1000 ) {
				time_start = time_current;
				fprintf(stderr, " I2C test(W)....total=%.8d, error=%.8d\n", total_test, total_error);
			}
		}
		fprintf(stderr, " I2C test....Mode=(W), SubAddressMax=%x, total=%d, error=%d\n",
						PT_ADDRESS_MAX, total_test, total_error);
		break;
	default:
		fprintf(stderr, " I2C test....invalid mode %d\n", Mode);
		break;
	} //....switch ( Mode )
	//
	i2c_access.Address = SubAddress;
	i2c_access.Data = i2c_data_save;
	ioctl(fd_i2c, RT2880_I2C_WRITE, &i2c_access);
}

void Help(char *ProgramName)
{
	fprintf(stderr, "------------------------------------------------------------------\n");
	fprintf(stderr, "Usage: %s\n", ProgramName);
	fprintf(stderr, " [-h]..........: display this help\n");
	fprintf(stderr, " [-dump].......: dump registers data\n");
	fprintf(stderr, " [-t times]....: i2c test\n");
	fprintf(stderr, " [-m mode].....: i2c test mode\n");
	fprintf(stderr, " [-r reg]......: i2c test register\n");
	fprintf(stderr, " [-a hex]......: i2c register address\n");
	fprintf(stderr, " [-d hex]......: i2c register data\n");
	fprintf(stderr, " [-dev hex]....: i2c device address\n");
	fprintf(stderr, " [-clk decimal]: i2c clock-divisor, freq=(400M/3)/((2+clkdiv)+5)\n");
	fprintf(stderr, " [-cfg hex]....: i2c config data,bit5-7:AddrLen,bit2-4:DevAddrLen\n");
	fprintf(stderr, " ..............: bit1:AddrDisable, bit0:DevAddrDisable\n");
	fprintf(stderr, "------------------------------------------------------------------\n");
} /* Help() */

int main(int argc, char *argv[])
{
int		i2c_address, i2c_data, i2c_dev, i2c_clk, i2c_cfg;
int		test_count, test_mode, test_sub_address;
I2C_CONFIG 	i2c_config_data;
I2C_READ_WRITE	i2c_access;
__u32		i2c_options;

	//
	i2c_options = 0;
	test_mode = I2C_TEST_MODE_RW;
	test_sub_address = 0;
	while(1) {
		int option_index = 0, c=0;
		static struct option long_options[] = \
			{
			{"h", no_argument, 0, 0},		/* help */
			{"a", required_argument, 0, 0},		/* i2c register address */
			{"d", required_argument, 0, 0},		/* i2c register data */
			{"dev", required_argument, 0, 0},	/* i2c device address */
			{"clk", required_argument, 0, 0},	/* i2c clock-divisor */
			{"cfg", required_argument, 0, 0},	/* i2c config data */
			{"t", required_argument, 0, 0},		/* i2c test */
			{"m", required_argument, 0, 0},		/* i2c test delay */
			{"r", required_argument, 0, 0},		/* i2c test sub address */
			{"dump", no_argument, 0, 0},		/* i2c dump all register */
			{0, 0, 0, 0}
			};

		c = getopt_long_only(argc, argv, "", long_options, &option_index);

		/* no more options to parse */
		if (c == -1) break;

		/* unrecognized option */
		if(c=='?'){ Help(argv[0]); return 0; }

		switch (option_index) {
		case 0:/* "-h" : help */
			Help(argv[0]);
			return 0;
		case 1:/* "-a" : i2c register address */
			i2c_options |= I2C_OPTION_SUB_ADDRESS;
			i2c_address = (int) strtol(optarg, NULL, 16);
			break;
		case 2:/* "-d" : i2c register data */
			i2c_options |= I2C_OPTION_DATA;
			i2c_data = (int) strtol(optarg, NULL, 16);
			break;
		case 3:/* "-dev" : i2c device address */
			i2c_options |= I2C_OPTION_DEVICE_ADDRESS;
			i2c_dev = (int) strtol(optarg, NULL, 16);
			break;
		case 4:/* "-clk" : i2c clock-divisor */
			i2c_options |= I2C_OPTION_CLOCK_DIVISOR;
			i2c_clk = atoi(optarg);
			break;
		case 5:/* "-cfg" : i2c config data */
			i2c_options |= I2C_OPTION_CONFIGURATION;
			i2c_cfg = (int) strtol(optarg, NULL, 16);
			break;
		case 6:/* "-t" : i2c test */
			i2c_options |= I2C_OPTION_TEST;
			test_count = atoi(optarg);
			break;
		case 7:/* "-m" : i2c test delay */
			test_mode = atoi(optarg);
			break;
		case 8:/* "-r" : i2c test sub address */
			test_sub_address = (int) strtol(optarg, NULL, 16);
			break;
		case 9:/* "-dump" : dump all register data */
			i2c_options |= I2C_OPTION_DUMP;
			break;
		default:
			Help(argv[0]);
			return 0;
		} // switch (option_index)
	} // while(1)
	fd_i2c = open(I2C_DEVICE, O_RDONLY);
	if ( fd_i2c < 0 ) {
		fprintf(stderr, "i2c device: %s open fail, code %d\n", I2C_DEVICE, fd_i2c);
		exit(1);
	}
	if ( (i2c_options & I2C_OPTION_CONFIG_MASK) != 0 ) {
		ioctl(fd_i2c, RT2880_I2C_CONFIG_READ, &i2c_config_data);
		if ( (i2c_options & I2C_OPTION_DEVICE_ADDRESS) != 0 ) {
			i2c_config_data.DeviceAddress = i2c_dev;
		}
		if ( (i2c_options & I2C_OPTION_CLOCK_DIVISOR) != 0 ) {
			i2c_config_data.ClockDivisor = i2c_clk;
		}
		if ( (i2c_options & I2C_OPTION_CONFIGURATION) != 0 ) {
			i2c_config_data.ConfigRegister = i2c_cfg;
		}
		ioctl(fd_i2c, RT2880_I2C_CONFIG_WRITE, &i2c_config_data);
	}
	if ( (i2c_options & I2C_OPTION_SUB_ADDRESS) != 0 ) {
		i2c_access.Address = (__u32) i2c_address;
		if ( (i2c_options & I2C_OPTION_DATA) != 0 ) {
			fprintf(stderr, "i2c : write");
			i2c_access.Data = (__u32) i2c_data;
			ioctl(fd_i2c, RT2880_I2C_WRITE, &i2c_access);
		} else {
			fprintf(stderr, "i2c : read");
			ioctl(fd_i2c, RT2880_I2C_READ, &i2c_access);
			i2c_data = (int) i2c_access.Data;
		}
		fprintf(stderr, " Address=(0x%.2X), Data=(0x%.2X)\n", i2c_address, i2c_data);
	}
	if ( (i2c_options & I2C_OPTION_DUMP) != 0 ) {
		int	idx;
		fprintf(stderr, "I2C dump....");
		for ( idx = 0; idx <= 0x38; idx++ ) {
			i2c_access.Address = (__u32) idx;
			ioctl(fd_i2c, RT2880_I2C_READ, &i2c_access);
			if ( (idx % 8) == 0 )	fprintf(stderr, "\n");
			fprintf(stderr, "(%.2x,%.2X)", idx, i2c_access.Data);
		}
		fprintf(stderr, "\n");
	}
	if ( (i2c_options & I2C_OPTION_TEST) != 0 )	I2CTest(test_mode, test_count, test_sub_address);
	return 0;
}