/* vi: set sw=4 ts=4 si: */
/*
 * ftpget
 *
 * Mini implementation of FTP to retrieve a remote file.
 *
 * Copyright (C) 2002 Jeff Angielski, The PTR Group <jeff@theptrgroup.com>
 * Copyright (C) 2002 Glenn McGrath
 *
 * Based on wget.c by Chip Rosenthal Covad Communications
 * <chip@laserlink.net>
 *
 * Licensed under GPLv2 or later, see file LICENSE in this tarball for details.
 */

//#include <signal.h>
#include <pthread.h>
#include <errno.h>
#include "libbb.h"
#include "../../linux-2.6.21.x/drivers/char/ralink_gpio.h"
#include "../../linux-2.6.21.x/drivers/char/frame.h"
#include "../../lib/libnvram/nvram.h"
#include "../../lib/include/net/if.h"
//#include <sys/select.h>

#define FTP_THREAD_ENABLE 0
#define CTRL_EXTRA_TIME	2

#if 0
int dir_level = 0;
long shift_time;
long frame_buffer_size;
long alloclen;
char *imgbuf;
int fd_data;
int got_signal = 0;
int signal_no = 0;
int insufficient_disk = 0;
int port_pasv_not_support = 0;
int stor_cmd_fail = 0;
int web_config_error = 0;
int folder_fail = 0;
char port_cmd_ip_part_s[64];
char port_cmd_s[64];
int error_control = 0;
int stream_error_control = 0;
int error_data = 0;
int max_seq_no_n_len = 0;

static char day_dir_path_old[20]="";
static char hour_dir_path_old[20]="0000";
static char half_hour_dir_path_old[20]="0030";
#endif

int (*ftp_imagesend)(void);
static int ftpcmd_w_timeout(const char *s1, const char *s2, int timeout, int to_usec);
static int get_cmd_response_wo_select(void);

/*error_code*/
#define NO_TEST_CONDUCTED				0 /*used by goahead*/
#define TESTING							1 /*used by goahead*/
#define TEST_SUCCEEDED					2
#define INVALID_FTP_SERVER_ADDRESS		3
#define IMAGE_NOT_AVAILABLE				4
#define CAN_NOT_CONNECT_TO_FTP_SERVER	5
#define INVALID_USERNAME_PASSWORD		6
#define TCP_IP_SOCKET_ERROR				7	
#define CAN_NOT_UPLOAD_IMAGE_FILE		8	
#define FAIL_TO_CHANGE_DIRECTORY		9 /*fail to change directory*/
#define NOT_SUPPORT_PASV_MODE			10 /*not support PASV mode*/
#define NOT_SUPPORT_PORT_MODE			11 /*not support PORT mode*/
#define INSUFFICIENT_DISK_SPACE 		12 /*insufficient disk space*/
#define CAN_NOT_CREATE_FOLDER			13 /*can not create folder*/
#define FAILURE							14  



#define YEAR_MONTH_DAY_LEN		8
#define HOUR_LEN				2
#define MINUTE_LEN				2 
#define SEC_LEN					2 


#define DIR_ROOT				0
#define DIR_DAY					1
#define DIR_HOUR_OR_HALF_HOUR	2
#define DIR_SEC					3




#define CONTROL_STREAM	1
#define DATA_STREAM		0

struct globals {
	char user[128];
	char password[128];
	struct len_and_sockaddr *lsa;
	FILE *control_stream;
	int control_stream_fd;
	int verbose_flag;
	int do_continue;
	char dst_filename[256];
	char time_part[20];
	char basefile_name[256];
	char basefile_name_mode[8];
	char freq_mode[4];
	char schedule_mode[4];
	struct USER_SIG_FLAG motion_flag;
	int freq_n;
	char frames_per_second[8];
	char seconds_per_frame[8];
	int max_seq_no_n;		/*_n for numeric value*/
	short error_code;
	int ftp_test;
	char server_path[256];
	char ftp_address[256];
	char ftp_port_s[8];
	int suffix;
	int folder_mode;
	int dir_level;
	long shift_time;
	long frame_buffer_size;
	long alloclen;
	char *imgbuf;
	unsigned long frame_seqnum;
	int fd_data;
	int got_signal;
	int signal_no;
	int insufficient_disk;
	int port_pasv_not_support;
	int stor_cmd_fail;
	int web_config_error;
	int folder_fail;
	char port_cmd_ip_part_s[64];
	char port_cmd_s[64];
	int error_control;
	int stream_error_control;
	int error_data;
	int max_seq_no_n_len;
	char day_dir_path_old[20];
	char hour_dir_path_old[20];
	char half_hour_dir_path_old[20];
	char buf[1]; /* actually [BUFSZ] */
};
#define G (*(struct globals*)&bb_common_bufsiz1)
enum { BUFSZ = COMMON_BUFSIZE - offsetof(struct globals, buf) };
struct BUG_G_too_big {
	char BUG_G_too_big[sizeof(G) <= COMMON_BUFSIZE ? 1 : -1];
};
#define user           (G.user          )
#define password       (G.password      )
#define lsa            (G.lsa           )
#define control_stream (G.control_stream)
#define control_stream_fd (G.control_stream_fd)
#define verbose_flag   (G.verbose_flag  )
#define do_continue    (G.do_continue   )
#define dst_filename   (G.dst_filename  )
#define time_part	   (G.time_part		)
#define basefile_name  (G.basefile_name )
#define basefile_name_mode	(G.basefile_name_mode)
#define freq_mode	   (G.freq_mode		)
#define schedule_mode  (G.schedule_mode)
#define motion_flag	   (G.motion_flag)
#define freq_n		   (G.freq_n		)
#define frames_per_second  (G.frames_per_second)
#define seconds_per_frame  (G.seconds_per_frame)
#define max_seq_no_n   (G.max_seq_no_n)
#define error_code	   (G.error_code)
#define ftp_test	   (G.ftp_test)
#define server_path	   (G.server_path)
#define ftp_address	   (G.ftp_address)
#define ftp_port_s	   (G.ftp_port_s)
#define buf            (G.buf           )
#define suffix (G.suffix)
#define folder_mode	   (G.folder_mode)

#define dir_level						(G.dir_level)
#define shift_time						(G.shift_time)
#define frame_buffer_size				(G.frame_buffer_size)
#define alloclen                        (G.alloclen)
#define imgbuf                         	(G.imgbuf)
#define frame_seqnum					(G.frame_seqnum)
#define fd_data                         (G.fd_data)
#define got_signal                      (G.got_signal)
#define signal_no						(G.signal_no)
#define insufficient_disk				(G.insufficient_disk)
#define port_pasv_not_support           (G.port_pasv_not_support)
#define stor_cmd_fail                   (G.stor_cmd_fail)
#define web_config_error				(G.web_config_error)
#define folder_fail						(G.folder_fail)
#define port_cmd_ip_part_s              (G.port_cmd_ip_part_s)
#define port_cmd_s						(G.port_cmd_s)
#define error_control					(G.error_control)
#define stream_error_control			(G.stream_error_control)
#define error_data						(G.error_data)
#define max_seq_no_n_len				(G.max_seq_no_n_len)
#define day_dir_path_old				(G.day_dir_path_old)
#define hour_dir_path_old				(G.hour_dir_path_old)
#define half_hour_dir_path_old			(G.half_hour_dir_path_old)
#define INIT_G() do { } while (0)


static void ftp_die(const char *msg) NORETURN;
static void ftp_die(const char *msg)
{
	char *cp = buf; /* buf holds peer's response */

	/* Guard against garbage from remote server */
	while (*cp >= ' ' && *cp < '\x7f')
		cp++;
	*cp = '\0';
	bb_error_msg_and_die("unexpected server response%s%s: %s",
			(msg ? " to " : ""), (msg ? msg : ""), buf);

}

static int read_sysinfo(int len, char *out, int subfunc)
{
	int fd;

	fd = open(SYSIF_DEV, O_RDONLY);
	if (fd < 0) {
		perror(SYSIF_DEV);
		return -1;
	}
	if (ioctl(fd, (SYSIF_READ | 
					subfunc << SUBCMD_IDX_SHIFT | 
					(len << BUF_LEN_SHIFT)), out) < 0) {
		perror("ioctl");
		close(fd);
		return -1;
	}

	close(fd);
	return 0;
}

static int write_sysinfo_short(short in, int subfunc)
{
	int fd;

	if (true == ftp_test) {
		fd = open(SYSIF_DEV, O_RDONLY);
		if (fd < 0) {
			perror(SYSIF_DEV);
			return -1;
		}
		if (ioctl(fd, (SYSIF_WRITE | 
						subfunc << SUBCMD_IDX_SHIFT | 
						((sizeof in) << BUF_LEN_SHIFT)), &in) < 0) {
			perror("ioctl");
			close(fd);
			return -1;
		}

		close(fd);
	}
	return 0;
}

static int write_sysinfo(int len, char *in, int subfunc)
{
	int fd;

	fd = open(SYSIF_DEV, O_RDONLY);
	if (fd < 0) {
		perror(SYSIF_DEV);
		return -1;
	}
	if (ioctl(fd, (SYSIF_WRITE | 
					subfunc << SUBCMD_IDX_SHIFT | 
					(len << BUF_LEN_SHIFT)), in) < 0) {
		perror("ioctl");
		close(fd);
		return -1;
	}

	close(fd);
	return 0;
}

static int alloc_imagebuf(void)
{
	int ret;

	frame_buffer_size = 0;
	read_sysinfo(sizeof(frame_buffer_size), (char *)&frame_buffer_size, FRAME_BUFFER_SIZE);
	//fprintf(stderr, "frame_buffer_size = %d\n", frame_buffer_size);

	alloclen = sizeof(struct FRM) + frame_buffer_size;
	imgbuf = malloc(alloclen);
	if (imgbuf != NULL) {
		ret = 0;
	} else {
		/*malloc failed*/
		ret = -1;
	}
	return ret;
}

static void free_imagebuf(void)
{
	if (NULL != imgbuf)
		free(imgbuf);
}

static void oneshotreleaseimage(void) 
{
	struct FRAME *sp = (struct FRAME *)imgbuf;

	//fprintf(stderr, "sp->Frame_Header.user_type = %d\n", sp->Frame_Header.user_type);
	//fprintf(stderr, "FRAME_RELEASE_IMAGE\n");
	if (2 == atoi(schedule_mode) && true != ftp_test) {
		sp->Frame_Header.frame_flag = RELEASE_MOTION_FRAME;
	} else {
		sp->Frame_Header.frame_flag = 0 ;
	}

	//write_sysinfo(frame_buffer_size, imgbuf, FRAME_RELEASE_IMAGE);
	write_sysinfo(alloclen, imgbuf, FRAME_RELEASE_IMAGE);

	//fprintf(stderr, "sp->Frame_Header.user_type = %d\n", sp->Frame_Header.user_type);
	//printf("release my_buffer_addr = %lu\n", sp->Frame_Header.my_buffer_addr);
}

static int oneshotgetimage(void) 
{
	int ret ;
	if (imgbuf != NULL) {
		struct FRAME *sp = (struct FRAME *)imgbuf;
		sp->Frame_Header.ticks_interval = 1;
		sp->Frame_Header.sequence_num = frame_seqnum;
		sp->Frame_Header.frame_len = frame_buffer_size;

		if (2 == atoi(schedule_mode) && true != ftp_test) {
			sp->Frame_Header.frame_flag = GET_MOTION_FRAME;
		} else {
			sp->Frame_Header.frame_flag = 0 ;
		}
		//write_sysinfo(frame_buffer_size, imgbuf, FRAME_GET_IMAGE);
		write_sysinfo(alloclen, imgbuf, FRAME_GET_IMAGE);

		if (sp->Frame_Header.frame_len == 0) {
			//fprintf(stderr, "sp->Frame_Header.frame_len = %d\n", sp->Frame_Header.frame_len);
			//fprintf(stderr, "sp->Frame_Header.sequence_num = %d\n", sp->Frame_Header.sequence_num);
			//fprintf(stderr, "frame_len = 0, sp->Frame_Header.user_type = %d\n", sp->Frame_Header.user_type);

			error_code = IMAGE_NOT_AVAILABLE;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
			/* for test*/
			//memset(imgbuf + sizeof (struct FRM), 0x41,  frame_buffer_size);
			//ret = 0;
			ret = -1;
		} else {
			//fprintf(stderr, "sp->Frame_Header.sequence_num = %d\n", sp->Frame_Header.sequence_num);
			//printf("get my_buffer_addr = %lu\n", sp->Frame_Header.my_buffer_addr);
			//fprintf(stderr, "sp->Frame_Header.user_type = %d\n", sp->Frame_Header.user_type);
			frame_seqnum = sp->Frame_Header.sequence_num;
			ret = 0;
		}
	} else {
		ret = -1;
	}
	return ret;
}

static int get_control_sesseion_response(void)
{
	int ret ;

	//ret = ftpcmd(NULL, NULL);
	ret = get_cmd_response_wo_select();
	if (450 == ret || 451 == ret || 452 == ret || 550 == ret || 552 == ret) {
		insufficient_disk = 1;
		error_code = INSUFFICIENT_DISK_SPACE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("get_control_sesseion_response, insufficient disk\n");
	} else if (421 == ret) {
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (426 == ret) {
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}
	return ret;
}

static int is_there_control_session_response(int sec, int usec)
{
	int ret ;

	//ret = ftpcmd(NULL, NULL);
	ret = ftpcmd_w_timeout(NULL, NULL, sec, usec);
	if (450 == ret || 451 == ret || 452 == ret || 550 == ret || 552 == ret) {
		insufficient_disk = 1;
		error_code = INSUFFICIENT_DISK_SPACE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("is_there_control_session_response, insufficient disk\n");
	} else if (421 == ret) {
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (426 == ret) {
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (9999 == ret) {
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);

	}
	return ret;
}

static int is_there_socket_error(int fd, int *error, int control_connection) 
{
	int ret = 0;
	int len = sizeof(int);

	if (getsockopt (fd, SOL_SOCKET, SO_ERROR, error, &len) < 0) {
		perror ("is_there_socket_error getsockopt");
		if (verbose_flag)
			bb_simple_perror_msg("is_there_socket_error() 1\n");

		//if (verbose_flag) {
		if (1 == control_connection) {
			bb_perror_msg("getsockopt < 0, is_there_socket_error(), control_connection, *error = %d\n", *error);
		} else {
			bb_perror_msg("getsockopt < 0, is_there_socket_error(), data_connectioin, *error = %d\n", *error);
		}
		//}

		ret = 1;
		if (1 == control_connection) {
			error_control = 1;
		} 
		goto EXIT;
	} else {
		if (verbose_flag) {
			if (1 == control_connection) {
				bb_perror_msg("is_there_socket_error(), control_connection, *error = %d\n", *error);
			} else {
				bb_perror_msg("is_there_socket_error(), data_connectioin, *error = %d\n", *error);
			}
		}
		/* (ECONNREFUSED == *error || EPIPE == *error || ENOTRECOVERABLE == *error)*/
		if (0 != *error) {
			ret = 1;
			if (1 == control_connection) {
				error_control = 1;
			}
			goto EXIT;
		}
	}
EXIT:
	return ret;
}

static int flush_cmd(const char *s1, const char *s2)
{
	fd_set wset;
	struct timeval tval;
	int sret, timeout = 5;
	int ret = 0;
	sigset_t sigmask;
	sigset_t sigmask_old;
	sigset_t sigmask_pend;

	FD_ZERO (&wset);
	FD_SET (control_stream_fd, &wset);
	tval.tv_sec = timeout;
	tval.tv_usec = 0;
	/* We "select()" until connect() returns its result or timeout */
	if (ferror(control_stream)) {
		if (verbose_flag)
			bb_perror_msg("flush_cmd  ferror\n");
		stream_error_control = 1;
		ret = -1;
		goto EXIT;
	}
	if (feof(control_stream)) {
		if (verbose_flag)
			bb_perror_msg("flush_cmd  feof\n");
		stream_error_control = 1;
		ret = -1;
		goto EXIT;
	}

	if (verbose_flag)
		bb_simple_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 1\n");

#if 1
	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGUSR1);
	if (-1 == sigprocmask(SIG_BLOCK, &sigmask, &sigmask_old)) {
		ret = 9999;
		goto EXIT;
	} else {
		sret = select(control_stream_fd+1, NULL, &wset, NULL, timeout ? &tval : NULL);
		if (verbose_flag) {
			sigpending(&sigmask_pend);
			if (sigismember(&sigmask_pend, SIGUSR1))
				printf("SIGUSR1 pending\n");
		}

		if (-1 == sigprocmask(SIG_SETMASK, &sigmask_old, NULL)) {
			ret = 9999;
			goto EXIT;
		}
	}
#else
	sigemptyset(&sigmask);
	if (-1 == sigprocmask(SIG_SETMASK, &sigmask, &sigmask_old)) {
		ret = 9999;
		goto EXIT;
	} else if ( -1 == sigaddset(&sigmask, SIGUSR1)) {
		ret = 9999;
		goto EXIT;
	} else if (-1 == sigprocmask(SIG_BLOCK, &sigmask, NULL)) {
		ret = 9999;
		goto EXIT;
	} else {
		//sret = pselect(control_stream_fd+1, NULL, &wset, NULL, (timeout)? &tval : NULL, &sigmask);
		sret = select(control_stream_fd+1, NULL, &wset, NULL, timeout ? &tval : NULL);
		if (verbose_flag) {
			sigpending(&sigmask_pend);
			if (sigismember(&sigmask_pend, SIGUSR1))
				printf("SIGUSR1 pending\n");
		}

		if (-1 == sigprocmask(SIG_SETMASK, &sigmask_old, NULL)) {
			ret = 9999;
			goto EXIT;
		}
	}
#endif

	if (verbose_flag)
		bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 2 sret = %d\n", sret);
	switch (sret) {
	case 0: /*timeout*/
		if (verbose_flag)
			bb_simple_perror_msg("flush_cmd timeout \n");
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			if (verbose_flag)
				bb_simple_perror_msg("flush_cmd timeout have error\n");
			ret = -1;
			goto EXIT;
		}
		break;
	case -1: /*socket error*/
		if (verbose_flag)
			bb_simple_perror_msg("flush_cmd socket error\n");
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			if (verbose_flag)
				bb_simple_perror_msg("flush_cmd socket error have error\n");
			ret = -1;
			goto EXIT;
		}
		break;
	default:
		if (verbose_flag)
			bb_simple_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 3\n");
		if (FD_ISSET(control_stream_fd, &wset)) {
			if (verbose_flag)
				bb_simple_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 4\n");

			if (s1) {
				fprintf(control_stream, (s2 ? "%s %s\r\n" : "%s %s\r\n"+3), s1, s2);

				if (verbose_flag)
					bb_simple_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 5\n");
				if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
					if (verbose_flag)
						bb_simple_perror_msg("flush_cmd default have error\n");
					ret = -1;
					goto EXIT;
				}
#if 0
				//int error_control = 0;
				int len = sizeof(error_control);
				if (getsockopt (control_stream_fd, SOL_SOCKET, SO_ERROR, &error_control, &len) < 0) {
					perror ("getsockopt");
					if (verbose_flag)
						bb_simple_perror_msg("flush_cmd() %%%%%%%%%%%%%%%%%%%%%%%%%%%%% 5.1\n");
					ret = -1;
					goto EXIT;
				} else {
					if (verbose_flag)
						bb_perror_msg("error_control = %d\n", error_control);
					if (ECONNREFUSED == error_control || EPIPE == error_control) {
						ret = -1;
						goto EXIT;
					}
				}
#endif
				fflush(control_stream);

				if (verbose_flag)
					bb_simple_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 6\n");
			}
		} else {
			ret = -1;
			goto EXIT;
		}
	}
	if (verbose_flag)
		bb_simple_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 6.1\n");
	return ret;
EXIT:

	if (verbose_flag)
		bb_simple_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 6.2\n");
	if (9999 == ret) {
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
		if (verbose_flag)
			bb_simple_perror_msg("get_cmd_response EXIT socket error have error\n");
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
	}

#if 0	
	error_code = TCP_IP_SOCKET_ERROR;
	write_sysinfo_short(error_code, FTP_TEST_RESULT);
#endif
	return ret;
}

static int get_cmd_response_wo_select(void)
{
	int ret = 0;
	int n = 0;

	memset(buf, 0, sizeof buf);
	do {
		/*(1 == is_there_socket_error(control_stream_fd, &error_control, 1))*/
		if (ferror(control_stream)) {
			bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%ferror 100, errno = %d\n", errno);
			stream_error_control = 1;
			ret = -1;
			goto EXIT;
		} 
		if (feof(control_stream)) {
			if (verbose_flag)
				bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%flush_cmd  feof, errno = %d\n", errno);
			stream_error_control = 1;
		}

		{
			//int c;
			/*strcpy(buf, "EOF"); thunder*/
			memset(buf, 0, sizeof buf);
			if (fgets(buf, BUFSZ - 2, control_stream) == NULL) {
				if (verbose_flag)
					bb_simple_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 100\n");
				if (error_code != INVALID_USERNAME_PASSWORD) {
					error_code = TCP_IP_SOCKET_ERROR;
				}
				write_sysinfo_short(error_code, FTP_TEST_RESULT);
				if (verbose_flag)
					bb_simple_perror_msg("fgets() == NULL\n");
				ret = -1;
				goto EXIT;
				//ftp_die(NULL);
			} else {
				if (verbose_flag)
					bb_perror_msg("ftpcmd, buf = %s\n", buf);
				ret = 0;
			}
		}

		if (verbose_flag)
			bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 100, 14 size = %d\n, buf = %s\n", strlen(buf), buf);
	} while (!isdigit(buf[0]) || buf[3] != ' ');/*for server greeting message more than one line*/

	buf[3] = '\0';

	if (verbose_flag)
		bb_perror_msg("get_cmd_response_wo_select, buf = %s\n", buf);
	//n = xatou(buf);
	n = atoi(buf);
	buf[3] = ' ';

	if (verbose_flag) {
		bb_error_msg("ftpcmd n = %d\n", n);
	}
	return n;
EXIT:
	return ret;
}

static int get_cmd_response(int timeout, int to_usec)
{
	fd_set	rset;
	struct timeval tval;
	int sret;
	int ret = 0;
	int have_response = false;
	sigset_t sigmask;
	sigset_t sigmask_old;
	sigset_t sigmask_pend;


	/*strcpy(buf, "EOF"); thunder*/
	memset(buf, 0, sizeof buf);
	FD_ZERO (&rset);
	FD_SET (control_stream_fd, &rset);
	tval.tv_sec = timeout;
	tval.tv_usec = to_usec;

	/* We "select()" until connect() returns its result or timeout */
	if (ferror(control_stream)) {
		if (verbose_flag)
			bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% ferror 7, errno = %d\n", errno);
			printf("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% ferror 7, errno = %d\n", errno);
		stream_error_control = 1;
		ret = -1;
		goto EXIT;
	}
	if (feof(control_stream)) {
		if (verbose_flag)
			bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% feof 8, errno = %d\n", errno);
			printf("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% feof 8, errno = %d\n", errno);
		stream_error_control = 1;
		goto GET;
	}

	sigemptyset(&sigmask);
	if (-1 == sigprocmask(SIG_SETMASK, &sigmask, &sigmask_old)) {
		ret = 9999;
		goto EXIT;
	} else if ( -1 == sigaddset(&sigmask, SIGUSR1)) {
		ret = 9999;
		goto EXIT;
	} else if (-1 == sigprocmask(SIG_BLOCK, &sigmask, NULL)) {
		ret = 9999;
		goto EXIT;
	} else {
		//sret = select(fileno(control_stream)+1, &rset, NULL, NULL, (timeout || to_usec)? &tval : NULL);
		//sret = pselect(control_stream_fd+1, &rset, NULL, NULL, (timeout || to_usec)? &tval : NULL, &sigmask);
		sret = select(control_stream_fd+1, &rset, NULL, NULL, &tval);
		if (verbose_flag) {
			sigpending(&sigmask_pend);
			if (sigismember(&sigmask_pend, SIGUSR1))
				printf("SIGUSR1 pending\n");
		}

		if (-1 == sigprocmask(SIG_SETMASK, &sigmask_old, NULL)) {
			ret = 9999;
			goto EXIT;
		}
	}

	if (verbose_flag)
		bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 10 sret = %d\n", sret);
	switch (sret) {
	case 0: /*timeout*/
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			if (verbose_flag)
				bb_simple_perror_msg("get_cmd_response timeout have error\n");
			ret = -1;
			goto EXIT;
		}
		break;
	case -1: /*socket error*/
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			if (verbose_flag)
				bb_simple_perror_msg("get_cmd_response socket error have error\n");
			ret = -1;
			goto EXIT;
		}
		break;
	default:
		if (verbose_flag)
			bb_simple_perror_msg("get_cmd_response case default\n");
		if (FD_ISSET(control_stream_fd, &rset)) {
			have_response = true;
		} else {
			ret = -1;
			goto EXIT;
		}
	}

	if (have_response) {
GET:
		do {
			/*(1 == is_there_socket_error(control_stream_fd, &error_control, 1))*/
			if (ferror(control_stream)) {
				bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% ferror 9, errno = %d\n", errno);
				stream_error_control = 1;
				ret = -1;
				goto EXIT;
			} else {

				//int c;
				/*strcpy(buf, "EOF"); thunder*/
				memset(buf, 0, sizeof buf);
				if (fgets(buf, BUFSZ - 2, control_stream) == NULL) {
					if (verbose_flag)
						bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 13, errno = %d\n", errno);
					if (error_code != INVALID_USERNAME_PASSWORD) {
						error_code = TCP_IP_SOCKET_ERROR;
					}
					write_sysinfo_short(error_code, FTP_TEST_RESULT);
					if (verbose_flag)
						bb_simple_perror_msg("fgets() == NULL\n");
					ret = -1;
					goto EXIT;
					//ftp_die(NULL);
				} else {
					if (verbose_flag)
						bb_perror_msg("ftpcmd, buf = %s\n", buf);
					ret = 0;
				}
			}

			if (verbose_flag)
				bb_perror_msg("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 14 size = %d\n, buf = %s\n", strlen(buf), buf);
		} while (!isdigit(buf[0]) || buf[3] != ' ');/*for server greeting message more than one line*/
		return ret;

	}

EXIT:
	if (9999 == ret) {
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
		if (verbose_flag)
			bb_simple_perror_msg("get_cmd_response EXIT socket error have error\n");
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
	}

#if 0	
	else if (0 != timeout && 0 != to_usec) {
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}
#endif
	return ret ;
}

static int ftpcmd_w_timeout(const char *s1, const char *s2, int timeout, int to_usec)
{
	unsigned n;
	if (verbose_flag) {
		bb_error_msg("w_timeout, cmd %s %s timeout = %d, to_usec = %d", s1, s2, timeout, to_usec);
	}
	if(NULL == control_stream)
		goto EXIT;
	if (-1 == control_stream_fd)
		goto EXIT;

	if ( -1 == flush_cmd(s1, s2)) {
		memset(buf, 0, 4);
		goto EXIT;
	}

	if (verbose_flag)
		bb_simple_perror_msg("ftpcmd_w_timeout after fflush\n");

	if ( -1 == get_cmd_response(timeout, to_usec)) goto EXIT;

EXIT:
	buf[3] = '\0';

	if (verbose_flag)
		bb_perror_msg("ftpcmd_w_timeout, buf = %s\n", buf);
	//n = xatou(buf);
	n = atoi(buf);
	buf[3] = ' ';

	if (verbose_flag) {
		bb_error_msg("ftpcmd_w_timeout n = %d\n", n);
	}
	return n;
}

static int ftpcmd(const char *s1, const char *s2)
{
	unsigned n;
	int timeout = 10;

	if (verbose_flag) {
		bb_error_msg("cmd %s %s", s1, s2);
	}
	if(NULL == control_stream)
		goto EXIT;
	if (-1 == control_stream_fd)
		goto EXIT;

	if ( -1 == flush_cmd(s1, s2)) {
		memset(buf, 0, 4);
		goto EXIT;
	}

	if (verbose_flag)
		bb_simple_perror_msg("ftpcmd after fflush\n");

	if ( -1 == get_cmd_response(timeout, 0)) goto EXIT;

EXIT:
	buf[3] = '\0';
	if (verbose_flag)
		bb_perror_msg("ftpcmd, buf = %s\n", buf);
	//n = xatou(buf);
	n = atoi(buf);
	buf[3] = ' ';
	if (verbose_flag) {
		bb_error_msg("ftpcmd n = %d\n", n);
	}
	return n;
}

static int FAST_FUNC ftpconnect(int s, const struct sockaddr *s_addr, socklen_t addrlen)
{
	int ret = 0;
	int error = 0;
	int	flags;
	fd_set	rset, wset;
	int 	n;
	struct timeval tval;
	int timeout = 10;
	sigset_t sigmask;
	sigset_t sigmask_old;
	sigset_t sigmask_pend;

	if (verbose_flag)
		bb_simple_perror_msg("ftpconnect IN\n");

	flags = fcntl (s, F_GETFL, 0);
	fcntl (s, F_SETFL, flags | O_NONBLOCK);		// set the socket as nonblocking IO

	if (verbose_flag)
		bb_simple_perror_msg("ftpconnect 1\n");
	if (connect(s, s_addr, addrlen) < 0) {
		if (verbose_flag)
			printf("ftpconnect, errno = %d\n", errno);
		if (errno == EINPROGRESS) {

			if (verbose_flag)
				bb_simple_perror_msg("ftpconnect 3\n");
			FD_ZERO (&rset);
			FD_ZERO (&wset);
			FD_SET (s, &rset);
			FD_SET (s, &wset);
			tval.tv_sec = timeout;
			tval.tv_usec = 0;

			/* We "select()" until connect() returns its result or timeout
			*/
			sigemptyset(&sigmask);
			if (-1 == sigprocmask(SIG_SETMASK, &sigmask, &sigmask_old)) {
				ret = 9999;
				goto EXIT;
			} else if ( -1 == sigaddset(&sigmask, SIGUSR1)) {
				ret = 9999;
				goto EXIT;
			} else if (-1 == sigprocmask(SIG_BLOCK, &sigmask, NULL)) {
				ret = 9999;
				goto EXIT;
			} else {
				//n = pselect(s+1, &rset, NULL, NULL, (timeout)? &tval : NULL, &sigmask);
				n = select(s+1, &rset, &wset, NULL, timeout ? &tval : NULL);
				if (verbose_flag) {
					sigpending(&sigmask_pend);
					if (sigismember(&sigmask_pend, SIGUSR1))
						printf("SIGUSR1 pending\n");
				}

				if (-1 == sigprocmask(SIG_SETMASK, &sigmask_old, NULL)) {
					ret = 9999;
					goto EXIT;
				}
			}

			if (0 == n) { 
				close (s);
				if (verbose_flag)
					bb_simple_perror_msg(
							"connect_timeout(), Connect Timeout\n");
				error = ETIMEDOUT;
				ret = -1;
				goto EXIT;
			} else if (-1 == n) {
				close (s);

				if (verbose_flag)
					bb_simple_perror_msg(
							"ftpconnect recv RST\n");
				error = EL3RST;
				ret = -1;
				goto EXIT;
			}

			if (FD_ISSET(s, &rset) || FD_ISSET(s, &wset)) {
				int len = sizeof(error);
				if (getsockopt (s, SOL_SOCKET, SO_ERROR, &error, &len) < 0) {
					if (verbose_flag)
						bb_simple_perror_msg("connect_timeout(), getsockopt fail\n");
					perror ("getsockopt");
					error_control = 1;
					ret = -1;
					goto EXIT;
				} else {
					if (verbose_flag)
						bb_perror_msg("error = %d\n", error);
					/*ECONNREFUSED
					 *EHOSTUNREACH
					 */
					if (0 != error) {
						error_control = 1;
						ret = -1;
						goto EXIT;
					}
				}
			} else {
				if (verbose_flag)
					bb_simple_perror_msg(
							"FD_ISSET FALSE \n");
				ret = -1;
				goto EXIT;
			}
		} else {
			error_control = 1;
			ret = -1;
			close(s);
		}

		if (verbose_flag)
			bb_simple_perror_msg("ftpconnect 4\n");
	}
	fcntl (s, F_SETFL, flags);
	if (verbose_flag)
		bb_simple_perror_msg("ftpconnect OUT 4\n");

EXIT:
	if (9999 == ret) {
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else 	if (-1 == ret) {
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}

	return ret;
}

static int FAST_FUNC ftpconnect_stream(const len_and_sockaddr *ftp_lsa, int control)
{
	int ret = 0;
	int fd;
	if (verbose_flag) {
		switch(control) {
		case CONTROL_STREAM:
			bb_simple_perror_msg("frpconnect_stream CONTROL_STREAM\n");
			break;
		case DATA_STREAM:
			bb_simple_perror_msg("frpconnect_stream DATA_STREAM\n");
			break;
		}
	}
	//fd = xsocket(ftp_lsa->u.sa.sa_family, SOCK_STREAM, 0);
	fd = socket(ftp_lsa->u.sa.sa_family, SOCK_STREAM, 0);
	if (verbose_flag) 
		bb_simple_perror_msg("frpconnect_stream xxxx\n");

	if ( -1 != fd ) {
		if (CONTROL_STREAM == control) {
			control_stream_fd = fd;
			if (verbose_flag) {
				bb_perror_msg("frpconnect_stream xxxx, control_stream_fd = %d\n", control_stream_fd);
			}
		}
		ret = ftpconnect(fd, &ftp_lsa->u.sa, ftp_lsa->len);
	} else {
		control_stream_fd = -1;
		control_stream = NULL;
		ret = -2;
	}

	if (verbose_flag)
		bb_perror_msg("frpconnect_stream ret = %d\n", ret);

	if (-1 == ret) {
		close(fd);
		return -1;
	} else if (-2 == ret) {
		return -1;
	} else {
		return fd;
	}
}

static int ftp_syn(void)
{
	int ret, fd;
	/* Connect to the command socket */
	//control_stream = fdopen(xconnect_stream(lsa), "r+");
	if (verbose_flag)
		bb_simple_perror_msg("fdopen 1\n");
	fd = ftpconnect_stream(lsa, CONTROL_STREAM);
	if (-1 == fd) {
		ret = -1;
		goto EXIT;
	}
	control_stream = fdopen(fd, "r+");
	if (verbose_flag)
		bb_simple_perror_msg("fdopen 2\n");
	if (control_stream == NULL) {
		/* fdopen failed - extremely unlikely */
		if (verbose_flag)
			bb_simple_perror_msg("fdopen fail\n");
		close(fd);
		ret = -1;
		goto EXIT;
		//bb_perror_nomsg_and_die();
	}

AGAIN:
	ret = ftpcmd(NULL, NULL);
	switch (ret) {
	case 220:
		break;
	case 120:
		goto AGAIN;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		break;
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("ftp_login, NULL fail\n");
		//ftp_die(NULL);
		break;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	default:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}

EXIT:
	return ret;
}

static int ftp_noop(void)
{
	int ret;
	ret = ftpcmd("NOOP", NULL);

	switch (ret) {
	case 200:
		break;
	case 530:
		error_code = INVALID_USERNAME_PASSWORD;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("ftp_TYPE_I, TYPE I fail\n");
		break;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		if (verbose_flag)
			bb_simple_perror_msg("ftp_TYPE_I, TYPE I socket fail\n");
		break;
		//ftp_die(NULL);
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 500:
	case 501:
	case 504:
	default:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}
	if (verbose_flag)
		bb_perror_msg("ftp_noop ret = %d\n", ret);

	return ret;
}

static int ftp_TYPE_I(void)
{
	int ret;
	ret = ftpcmd("TYPE I", NULL);

	switch (ret) {
	case 200:
		break;
	case 530:
		error_code = INVALID_USERNAME_PASSWORD;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("ftp_TYPE_I, TYPE I fail\n");
		break;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		if (verbose_flag)
			bb_simple_perror_msg("ftp_TYPE_I, TYPE I socket fail\n");
		break;
		//ftp_die(NULL);
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 500:
	case 501:
	case 504:
	default:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}
	if (verbose_flag)
		bb_perror_msg("TYPE I ret = %d\n", ret);

	return ret;
}

static int ftp_mkd(const char *path)
{
	int ret = 0;
	ret = ftpcmd("MKD", path);
	switch (ret) {
	case 257:
		break;
	case 550:
		error_code = CAN_NOT_CREATE_FOLDER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_perror_msg("MKD fail 550 ret = %d \n", ret);
		//ftp_die("PASS");
		break;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		if (verbose_flag)
			bb_perror_msg("MKD fail ret = %d \n", ret);
		//ftp_die("PASS");
		ret = -1;
		break;
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 500:
	case 501:
	case 503:
	default:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}
	return ret;
}

static int ftp_pass(void)
{
	int ret = 0;
	ret = ftpcmd("PASS", password);
	switch (ret) {
	case 230:
		break;
	case 202:
		break;
	case 332:
	case 530:
		error_code = INVALID_USERNAME_PASSWORD;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_perror_msg("PASS fail 530 ret = %d \n", ret);
		//ftp_die("PASS");
		break;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		if (verbose_flag)
			bb_perror_msg("PASS fail ret = %d \n", ret);
		//ftp_die("PASS");
		ret = -1;
		break;
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 500:
	case 501:
	case 503:
	default:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}
	return ret;
}

static int ftp_login(void)
{
	int ret = 0;
	int ret_pass = 0;

	error_code = INVALID_USERNAME_PASSWORD;
	switch ((ret = ftpcmd("USER", user))) {
	case 230:
		break;
	case 331:
		ret_pass = ftp_pass();
		if (530 == ret_pass) {
			/*not login*/
			ret = ret_pass;
		} else if (230 == ret_pass || 202 == ret_pass){
			/*login success*/
			ret = ret_pass;
		} else if (9999 == ret_pass) {
			ret = ret_pass;
		}
		break;
	case 530:
		error_code = INVALID_USERNAME_PASSWORD;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		break;
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 500:
	case 501:
	default:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}

	if (verbose_flag)
		bb_perror_msg("ftp_login ret = %d\n", ret);
	return ret;
}

#if 0
static int ftp_login(void)
{
	int ret = 0;
	int ret_pass = 0;
	/*  Log in the server */
	/*ftpcmd fgets will return NULL if server deny the username password(server will send EOF) pair*/
	error_code = INVALID_USERNAME_PASSWORD;
	switch ((ret = ftpcmd("USER", user))) {
	case 230:
		break;
	case 331:
		ret_pass == ftp_pass();
		if (530 == ret_pass) {
			/*not login*/
			ret = -1;
			/*success*/
		} else if (0 == ret_pass){
			ret = -2;
		} else {
			ret = ret_pass;
		}	
		break;
	case 0:
	default:
		if (verbose_flag)
			bb_perror_msg("USER fail ret = %d\n", ret);
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -2;
		close(fd_data);
	}
	return ret;
}
#endif

static int ftp_pasv(void)
{
	int ret;
	char *buf_ptr;
	unsigned port_num;

	/*
TODO: PASV command will not work for IPv6. RFC2428 describes
IPv6-capable "extended PASV" - EPSV.

"EPSV [protocol]" asks server to bind to and listen on a data port
in specified protocol. Protocol is 1 for IPv4, 2 for IPv6.
If not specified, defaults to "same as used for control connection".
If server understood you, it should answer "229 <some text>(|||port|)"
where "|" are literal pipe chars and "port" is ASCII decimal port#.

There is also an IPv6-capable replacement for PORT (EPRT),
but we don't need that.

NB: PASV may still work for some servers even over IPv6.
For example, vsftp happily answers
"227 Entering Passive Mode (0,0,0,0,n,n)" and proceeds as usual.

TODO2: need to stop ignoring IP address in PASV response.
*/
	ret = ftpcmd("PASV", NULL);
	if (verbose_flag)
		bb_perror_msg("PASV ret = %d\n", ret);

	switch (ret) {
	case 227:
		break;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		//close(fd_data);
		ret = -1;
		goto EXIT;
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		goto EXIT;
	case 530:
		error_code = INVALID_USERNAME_PASSWORD;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		goto EXIT;
	case 502:
		error_code = NOT_SUPPORT_PASV_MODE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("server not support PASV mode\n");
		port_pasv_not_support = 1;
		ret = -1;
		goto EXIT;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		break;
	case 500:
	case 501:
	default:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		goto EXIT;
		//ftp_die("PORT");
	}

#if 0
	if ( 227 != ret && 0 != ret) {
		if (verbose_flag)
			bb_perror_msg("===================PASV 1 ret = %d\n", ret);
		error_code = NOT_SUPPORT_PASV_MODE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("PASV fail\n");
		port_pasv_not_support = 1;
		ret = -1;
		goto EXIT;
		//ftp_die("PASV");
	}
#endif

	if (verbose_flag)
		bb_perror_msg("===================PASV 2 ret = %d\n", ret);
	/* Response is "NNN garbageN1,N2,N3,N4,P1,P2[)garbage]
	 * Server's IP is N1.N2.N3.N4 (we ignore it)
	 * Server's port for data connection is P1*256+P2 */
	buf_ptr = strrchr(buf, ')');

	if (NULL == buf_ptr) {
		goto EXIT2;
	}

	if (verbose_flag)
		bb_perror_msg("===================PASV 3 buf_ptr = %p\n", buf_ptr); 

	if (buf_ptr) *buf_ptr = '\0';

	if (verbose_flag)
		bb_perror_msg("===================PASV 3.1 buf_ptr = %p\n", buf_ptr); 

	buf_ptr = strrchr(buf, ',');
	if (buf_ptr) *buf_ptr = '\0';

	if (verbose_flag)
		bb_perror_msg("===================PASV 4 buf = %s\n", "");
	port_num = xatoul_range(buf_ptr + 1, 0, 255);

	if (verbose_flag)
		bb_perror_msg("===================PASV 5 buf = %s\n", "");

	buf_ptr = strrchr(buf, ',');
	if (buf_ptr) *buf_ptr = '\0';

	if (verbose_flag)
		bb_perror_msg("===================PASV 6 buf = %s\n", "");

	port_num += xatoul_range(buf_ptr + 1, 0, 255) * 256;

	if (verbose_flag)
		bb_perror_msg("===================PASV 7 buf = %s\n", "");

	if (verbose_flag)
		bb_simple_perror_msg("xconnect_ftpdata 1111\n");
	set_nport(lsa, htons(port_num));

	if (verbose_flag)
		bb_perror_msg("===================PASV 8 buf = %s\n", "");

	if (verbose_flag)
		bb_simple_perror_msg("xconnect_ftpdata 2222\n");
	//return xconnect_stream(lsa);
EXIT2:
	if (-1 == ret) {
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
	}
EXIT:
	return ret;
}

#if 0
static int xconnect_ftpdata(void)
{
	int ret;
	char *buf_ptr;
	unsigned port_num;

	/*
TODO: PASV command will not work for IPv6. RFC2428 describes
IPv6-capable "extended PASV" - EPSV.

"EPSV [protocol]" asks server to bind to and listen on a data port
in specified protocol. Protocol is 1 for IPv4, 2 for IPv6.
If not specified, defaults to "same as used for control connection".
If server understood you, it should answer "229 <some text>(|||port|)"
where "|" are literal pipe chars and "port" is ASCII decimal port#.

There is also an IPv6-capable replacement for PORT (EPRT),
but we don't need that.

NB: PASV may still work for some servers even over IPv6.
For example, vsftp happily answers
"227 Entering Passive Mode (0,0,0,0,n,n)" and proceeds as usual.

TODO2: need to stop ignoring IP address in PASV response.
*/
	ret = ftpcmd("PASV", NULL);
	if ( 227 != ret && 0 != ret) {
		if (verbose_flag)
			bb_perror_msg("===================PASV 1 ret = %d\n", ret);
		error_code = NOT_SUPPORT_PASV_MODE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("PASV fail\n");
		port_pasv_not_support = 1;
		ret = -1;
		goto EXIT;
		//ftp_die("PASV");
	}

	if (verbose_flag)
		bb_perror_msg("===================PASV 2 ret = %d\n", ret);
	/* Response is "NNN garbageN1,N2,N3,N4,P1,P2[)garbage]
	 * Server's IP is N1.N2.N3.N4 (we ignore it)
	 * Server's port for data connection is P1*256+P2 */
	buf_ptr = strrchr(buf, ')');

	if (NULL == buf_ptr) {
		goto EXIT2;
	}

	if (verbose_flag)
		bb_perror_msg("===================PASV 3 buf_ptr = %p\n", buf_ptr); 

	if (buf_ptr) *buf_ptr = '\0';

	if (verbose_flag)
		bb_perror_msg("===================PASV 3.1 buf_ptr = %p\n", buf_ptr); 

	buf_ptr = strrchr(buf, ',');
	if (buf_ptr) *buf_ptr = '\0';

	if (verbose_flag)
		bb_perror_msg("===================PASV 4 buf = %s\n", "");
	port_num = xatoul_range(buf_ptr + 1, 0, 255);

	if (verbose_flag)
		bb_perror_msg("===================PASV 5 buf = %s\n", "");

	buf_ptr = strrchr(buf, ',');
	if (buf_ptr) *buf_ptr = '\0';

	if (verbose_flag)
		bb_perror_msg("===================PASV 6 buf = %s\n", "");

	port_num += xatoul_range(buf_ptr + 1, 0, 255) * 256;

	if (verbose_flag)
		bb_perror_msg("===================PASV 7 buf = %s\n", "");

	if (verbose_flag)
		bb_simple_perror_msg("xconnect_ftpdata 1111\n");
	set_nport(lsa, htons(port_num));

	if (verbose_flag)
		bb_perror_msg("===================PASV 8 buf = %s\n", "");

	if (verbose_flag)
		bb_simple_perror_msg("xconnect_ftpdata 2222\n");
	//return xconnect_stream(lsa);
	ret = ftpconnect_stream(lsa, DATA_STREAM);

	if (verbose_flag)
		bb_simple_perror_msg("xconnect_ftpdata 3333\n");
	//return xconnect_stream(lsa);
EXIT2:
	if (-1 == ret) {
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	}
EXIT:
	return ret;
}
#endif

static int wait_select(int dst_fd, int flags)
{
	fd_set	wset;
	struct timeval tval;
	int sret, timeout = 5;
	int ret = 0;
	sigset_t sigmask;
	sigset_t sigmask_old;
	sigset_t sigmask_pend;

	fcntl (dst_fd, F_SETFL, flags);
	while(1)
	{
		if (verbose_flag)
			bb_simple_perror_msg("===================wait_select 0\n");

		FD_ZERO (&wset);
		FD_SET (dst_fd, &wset);
		tval.tv_sec = timeout;
		tval.tv_usec = 0;
		/* We "select()" until connect() returns its result or timeout */
		sigemptyset(&sigmask);
		if (-1 == sigprocmask(SIG_SETMASK, &sigmask, &sigmask_old)) {
			ret = 9999;
			goto EXIT;
		} else if ( -1 == sigaddset(&sigmask, SIGUSR1)) {
			ret = 9999;
			goto EXIT;
		} else if (-1 == sigprocmask(SIG_BLOCK, &sigmask, NULL)) {
			ret = 9999;
			goto EXIT;
		} else {
			//sret = pselect(dst_fd+1, NULL, &wset, NULL, (timeout)? &tval : NULL, &sigmask);
			sret = select(dst_fd+1, NULL, &wset, NULL, timeout ? &tval : NULL);

			if (verbose_flag) {
				sigpending(&sigmask_pend);
				if (sigismember(&sigmask_pend, SIGUSR1))
					printf("SIGUSR1 pending\n");
			}

			if (-1 == sigprocmask(SIG_SETMASK, &sigmask_old, NULL)) {
				ret = 9999;
				goto EXIT;
			}
		}

		if (verbose_flag)
			bb_simple_perror_msg("===================wait_select 1\n");
		switch (sret) {
		case 0: /*timeout*/
			if (verbose_flag)
				bb_simple_perror_msg("===================wait_select 2\n");
			if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
				error_code = TCP_IP_SOCKET_ERROR;
				write_sysinfo_short(error_code, FTP_TEST_RESULT);
			}
			ret = -1;
			goto EXIT;
		case -1: /*socket error*/
			if (verbose_flag)
				bb_simple_perror_msg("===================wait_select 3\n");
			ret = -2;
			goto EXIT;
		default:
			if (verbose_flag)
				bb_simple_perror_msg("===================wait_select 4\n");
			if (FD_ISSET(dst_fd, &wset)) {
				if (verbose_flag)
					bb_simple_perror_msg("===================wait_select 5\n");
				ret = 0;
				goto EXIT;
			} else {
				if (verbose_flag)
					bb_simple_perror_msg("===================wait_select 6\n");
				ret = -3;
			}
		}
	}
EXIT:
	flags = fcntl (dst_fd, F_GETFL, 0);
	fcntl (dst_fd, F_SETFL, flags | O_NONBLOCK);		// set the socket as nonblocking IO

	return ret;
}

static int image_socket_send(int dst_fd, char *from, int buf_len, int *wr)
{
	int out = 0;
	int left_len = buf_len;
	int ret = 0;
	int wret= 0;
	int	flags;

	flags = fcntl (dst_fd, F_GETFL, 0);
	fcntl (dst_fd, F_SETFL, flags | O_NONBLOCK);		// set the socket as nonblocking IO

	if (verbose_flag)
		bb_perror_msg("^^^^^^^^^^^1, dst_fd = %d\n", dst_fd);

	while(1) {
		if (verbose_flag)
			bb_simple_perror_msg("===================send 0.0\n");

		ret = is_there_control_session_response(0, 2);
		//printf("1 !!!!!!!!!!!!!!!!!!!!!ret = %d\n", ret);
		if (450 == ret || 451 == ret || 452 == ret || 550 == ret || 552 == ret || 421 == ret || 426 == ret || 9999 == ret) {
			goto EXIT;
		}

		if (1 == is_there_socket_error(dst_fd, &error_data, 0)) {
			//ret = is_there_control_session_response(5, 0);
			//ret = get_cmd_response_wo_select();

			ret = get_control_sesseion_response();
			//printf("2 !!!!!!!!!!!!!!!!!!!!!ret = %d\n", ret);
			goto EXIT;
		}
		out = send(dst_fd, from, left_len, 0);
		if (verbose_flag)
			bb_perror_msg("===================send 0 out = %d\n", out);
		if ( out == left_len ) {
			if (verbose_flag)
				bb_simple_perror_msg("===================send 1\n");
			*wr += out;
			ret = 0;
			//ret = buf_len;
			goto EXIT;
		} else if ( out >= 0 ) {
			from = from+out;
			left_len = left_len-out;
			*wr += out;
			if (verbose_flag)
				bb_simple_perror_msg("===================send 2\n");
		} else if ( out == -1) {
			if (verbose_flag)
				bb_simple_perror_msg("===================send 3\n");
			wret = wait_select(dst_fd, flags);
			switch (wret) {
			case -1: /*timeout*/
				printf("image_socket_send wait_select, timeout, wret = %d\n", wret);
				ret = -1;
				goto EXIT;
			case -2: /*socket error*/
				ret = is_there_control_session_response(5, 0);
				printf("image_socket_send wait_select, socket error, wret = %d, ret = %d\n", wret, ret);
				if (450 == ret || 451 == ret || 452 == ret || 550 == ret || 552 == ret || 421 == ret || 426 == ret || 9999 == ret) {
					goto EXIT;
				}
				ret = -2;
				goto EXIT;
			case 0: /*data in*/
				break;
			default:
				printf("image_socket_send wait_select, error, wret = %d\n", wret);
				ret = -3;
				goto EXIT;
				//break;
			}
		}
	}
EXIT:
	fcntl (dst_fd, F_SETFL, flags);
	return ret;
}

static int deliver_image(void *image, int dst_fd)
{
	void *from;
	//ssize_t rd = frame_buffer_size;
	ssize_t rd = ((struct FRM *)image)->frame_len ;
	ssize_t ret = 0;
	ssize_t wr = 0;

	//rd = ((struct FRM)(from - sizeof (struct FRM)))->frame_len ;
	from = image + sizeof (struct FRM);

	if (verbose_flag)
		bb_simple_perror_msg("deliver_image, IN\n");
	/* dst_fd == -1 is a fake, else... */
	if (dst_fd >= 0) {
		if (verbose_flag)
			bb_perror_msg("deliver_image, before full_write, rd = %d\n", rd);
		//wr = full_write(dst_fd, from, rd);
		ret = image_socket_send(dst_fd, from, rd, &wr);
		if (verbose_flag)
			bb_simple_perror_msg("deliver_image, after full_write\n");
		if (450 == ret || 451 == ret || 452 == ret || 550 == ret || 552 == ret) {
			insufficient_disk = 1;
			error_code = INSUFFICIENT_DISK_SPACE;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
			if (verbose_flag)
				bb_simple_perror_msg("image_socket_send, insufficient disk\n");
		} else if (421 == ret) {
			error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		} else if (426 == ret) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		} else if (ret < 0 || wr < rd) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
				printf("deliver_image, socket error dst_fd = %d, ret = %d, rd = %d, wr = %d, dst_filename %s\n", dst_fd, ret, rd, wr, dst_filename);
			//bb_perror_msg(bb_msg_write_error);
			if (verbose_flag)
				bb_perror_msg("deliver_image, socket error dst_fd = %d, ret = %d, rd = %d, wr = %d, dst_filename %s\n", dst_fd, ret, rd, wr, dst_filename);
			ret = -1;
		} 
	} else {
		ret = -1;
	}

	if (verbose_flag)
		bb_perror_msg("deliver_image, OUT ret = %d\n", ret);

	return ret;
}

static int ftp_cwd(const char *dir_path)
{
	int ret;

	ret = ftpcmd("CWD", dir_path);
	switch (ret) {
	case 250:
		break;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		if (verbose_flag)
			bb_perror_msg("CWD fail ret = %d\n", ret);
		break;
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 530:
		error_code = INVALID_USERNAME_PASSWORD;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		break;
	case 500:
	case 501:
	case 502:
	case 550:
	default:
		error_code = FAIL_TO_CHANGE_DIRECTORY;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("CWD fail\n");
		//ftp_die("CWD");
	}
	return ret;
}

#if 0
static int get_ifs_ips(void)
{
	struct ifreq   buffer[32];
	struct ifconf  intfc;
	struct ifreq  *pIntfc;
	int            i, fd, num_intfc;

	intfc.ifc_len = sizeof(buffer);
	intfc.ifc_buf = (char*) buffer;

	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
	{
		perror("socket() failed");
		return 1;
	}

	if (ioctl(fd, SIOCGIFCONF, &intfc) < 0)
	{
		perror("ioctl SIOCGIFCONF failed");
		return 1;
	}

	pIntfc    = intfc.ifc_req;
	num_intfc = intfc.ifc_len / sizeof(struct ifreq);

	for (i = 0; i < num_intfc; i++)
	{
		struct ifreq *item = &(pIntfc[i]);
		if (verbose_flag)
			bb_perror_msg("Interface # %d -> %s: IP %s, %x\n",
					i, item->ifr_name,
					inet_ntoa(((struct sockaddr_in *)&item->ifr_addr)->sin_addr), 
					(((struct sockaddr_in *)&item->ifr_addr)->sin_addr).s_addr);
	}

	return 0;
}
#endif

static int set_port_cmd_ip_part(void)
{
	unsigned long  cur_ip;

	read_sysinfo(sizeof(cur_ip), (char *)&cur_ip, CURRENT_IP);
	sprintf(port_cmd_ip_part_s, "%lu,%lu,%lu,%lu", 
			(cur_ip & 0x000000ff),
			(cur_ip & 0x0000ff00) >> 8,
			(cur_ip & 0x00ff0000) >> 16,
			(cur_ip & 0xff000000) >> 24
		   );
	//fprintf(stderr, "port_cmd_ip_part_s = %s\n", port_cmd_ip_part_s);
}

static void set_port_cmd_port_part(int cur_port)
{
	int first_part, second_part;
	//char port_cmd[64];

	first_part = cur_port / 256;
	second_part = cur_port % 256;
	sprintf(port_cmd_s, "%s,%d,%d", port_cmd_ip_part_s, first_part, second_part);

	//fprintf(stderr, "port_cmd_s = %s\n", port_cmd_s);
}

static int set_port_no(void)
{
	/*port number can not be between 1 - 1023, becasue some ftp server 
	 *will response "Port number too low.*/
	static const int port_min = 30000;
	static const int port_max = 60000;
	static int ftp_data_server_port = port_min;

	if (ftp_data_server_port >= port_max) {
		ftp_data_server_port = port_min;
	} else {
		ftp_data_server_port++;
	}

	return ftp_data_server_port;
}

static int FAST_FUNC ftp_socket_type(len_and_sockaddr **lsap, int family, int sock_type)
{
	int fd;
	int len;
	int i = 0;
	len_and_sockaddr *lsatmp;

	do {
		fd = socket(family, sock_type, 0);
		usleep(1000);
		if (verbose_flag)
			bb_perror_msg("ftp_socket_type fd = %d\n", fd);
	} while (fd < 0 && (++i) < 100);


	len = sizeof(struct sockaddr_in);

	i = 0;
	do {
		lsatmp = malloc(offsetof(len_and_sockaddr, u.sa) + len);
		usleep(1000);
		if (verbose_flag)
			bb_perror_msg("ftp_socket_type lsatmp = %p\n", lsatmp);
	} while (NULL == lsatmp && (++i) < 100);
	lsatmp->len = len;
	lsatmp->u.sa.sa_family = family;
	*lsap = lsatmp;
	return fd;
}

static void set_nport_ipv4(len_and_sockaddr *lsa_tmp, unsigned port)
{
	if (lsa_tmp->u.sa.sa_family == AF_INET) {
		lsa_tmp->u.sin.sin_port = port;
		lsa_tmp->u.sin.sin_addr.s_addr = htonl(INADDR_ANY); 
		return;
	}
	/* What? UNIX socket? IPX?? :) */
}

static int FAST_FUNC local_create_and_bind_stream_but_not_die(const char *bindaddr, int port)
{
	int ret = 0;
	int fd;
	len_and_sockaddr *tmp_lsa;

	bindaddr = NULL;

	if (verbose_flag)
		bb_simple_perror_msg("local_create_and_bind_stream_but_not_die IN");

	//fd = xsocket_type(&tmp_lsa, USE_FEATURE_IPV6(AF_UNSPEC,) SOCK_STREAM);
	fd = ftp_socket_type(&tmp_lsa, AF_INET, SOCK_STREAM);

	if (verbose_flag)
		bb_perror_msg("local_create_and_bind_stream_but_not_die fd = %d\n", fd);

	if (fd > 0) {
		//set_nport(tmp_lsa, htons(port));
		set_nport_ipv4(tmp_lsa, htons(port));

		if (verbose_flag)
			bb_simple_perror_msg("local_create_and_bind_stream_but_not_die 0");
		setsockopt_reuseaddr(fd);

		if (verbose_flag)
			bb_simple_perror_msg("local_create_and_bind_stream_but_not_die 1");
		//xbind(fd, &tmp_lsa->u.sa, tmp_lsa->len);
		if (bind(fd, &tmp_lsa->u.sa, tmp_lsa->len)) {
			ret = -1;
			close(fd);

			if (verbose_flag)
				bb_simple_perror_msg("local_create_and_bind_stream_but_not_die 2");
		}

		if (verbose_flag)
			bb_simple_perror_msg("local_create_and_bind_stream_but_not_die 3");

		free(tmp_lsa);
	} else {
		ret = -1;
	}

	if (-1 == ret) {
		return ret;
	} else {
		return fd;
	}
}

/*
 * Create a listen server socket on the designated port.
 */
static int openServer(void)
{
	char bind_addr_or_port[8] ;
	//char port_cmd_s[32];
	//unsigned n;
	int n;
	int ftp_data_srv_port;

SELECT_PORT:
	ftp_data_srv_port = set_port_no();
	set_port_cmd_port_part(ftp_data_srv_port);
	//get_ifs_ips();

	sprintf(bind_addr_or_port, "%d", ftp_data_srv_port);

	/*if success, n = 4 always, but I do not know why.*/
	//n = create_and_bind_stream_or_die(NULL, ftp_data_srv_port);
	/*if success, n = 4 always still, but if fail, we do not let process die, but choose a new port number.*/
	n = local_create_and_bind_stream_but_not_die(NULL, ftp_data_srv_port);

	if (verbose_flag)
		bb_perror_msg("openServer n = %d\n", n);

	if (-1 == n) {
		goto SELECT_PORT;
	}
#if 0
	n = bb_strtou(bind_addr_or_port, NULL, 10);

	//fprintf(stderr, "bind_addr_or_port = %s\n", bind_addr_or_port);
	//fprintf(stderr, "n = %s\n", n);
	if (!errno && n && n <= 0xffff)
		n = create_and_bind_stream_or_die(NULL, n);
	else
		goto EXIT;
#endif
	//xlisten(n, 1);
	listen(n, 1);
	//EXIT:
	return n;
}

static int mini_ftpdatad(int server_socket)
{
	/* NB: it's best to not use xfuncs in this loop before fork().
	 * Otherwise server may die on transient errors (temporary
	 * out-of-memory condition, etc), which is Bad(tm).
	 * Try to do any dangerous calls after fork.
	 */
	int n = 0;
	int ret = -1;
	len_and_sockaddr fromAddr;

	int timeout = 10;
	fd_set	rset;
	struct timeval tval;
	int sret;
	sigset_t sigmask;
	sigset_t sigmask_old;
	sigset_t sigmask_pend;

	FD_ZERO (&rset);
	FD_SET (server_socket, &rset);
	tval.tv_sec = timeout;
	tval.tv_usec = 0;

	/* We "select()" until connect() returns its result or timeout */
	sigemptyset(&sigmask);
	if (-1 == sigprocmask(SIG_SETMASK, &sigmask, &sigmask_old)) {
		ret = 9999;
		goto EXIT;
	} else if ( -1 == sigaddset(&sigmask, SIGUSR1)) {
		ret = 9999;
		goto EXIT;
	} else if (-1 == sigprocmask(SIG_BLOCK, &sigmask, NULL)) {
		ret = 9999;
		goto EXIT;
	} else {
		//sret = pselect(server_socket+1, &rset, NULL, NULL, (timeout)? &tval : NULL, &sigmask);
		sret = select(server_socket+1, &rset, NULL, NULL, (timeout)? &tval : NULL);
		if (verbose_flag) {
			sigpending(&sigmask_pend);
			if (sigismember(&sigmask_pend, SIGUSR1))
				printf("SIGUSR1 pending\n");
		}

		if (-1 == sigprocmask(SIG_SETMASK, &sigmask_old, NULL)) {
			ret = 9999;
			goto EXIT;
		}
	}

	if (verbose_flag)
		bb_perror_msg("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ mini_ftpdatad sret = %d\n", sret);
	switch (sret) {
	case 0: /*timeout*/
		if (verbose_flag)
			bb_simple_perror_msg("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ mini_ftpdatad 0\n");
		ret = -1;
		goto EXIT;
	case -1: /*socket error*/
		if (verbose_flag)
			bb_simple_perror_msg("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ mini_ftpdatad 1\n");

		ret = -1;
		goto EXIT;
	default:
		if (verbose_flag)
			bb_simple_perror_msg("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ mini_ftpdatad 2\n");
		if (!FD_ISSET(server_socket, &rset)) {
			ret = -1;
			goto EXIT;
		}
	}

	if ( 1 == is_there_socket_error(server_socket, &error_data, 0)) {
		ret = is_there_control_session_response(5, 0);
		printf("mini_ftpdatad ret = %d\n", ret);
		goto EXIT;
	}

	/* Wait for connections... */
	fromAddr.len = LSA_SIZEOF_SA;
	n = accept(server_socket, &fromAddr.u.sa, &fromAddr.len);
	//printf ("n = %d\n", n);

	if (n < 0) {
		return ret;
	}
	/* set the KEEPALIVE option to cull dead connections */
	setsockopt(n, SOL_SOCKET, SO_KEEPALIVE, &const_int_1, sizeof(const_int_1));

	//xmove_fd(n, 0);
	//xdup2(0, 1);

	//ret = deliver_image(imgbuf + sizeof(struct FRM), n);
	ret = deliver_image(imgbuf, n);
	//printf("ret = %d\n", ret);
	/* parent, or fork failed */
	/* close data connection */
EXIT:
	close(n);

	close(server_socket);
	return ret;
}

#if 0
static void *ftp_data_connection_server(void *arg)
{
	int server_socket = server_socket; /* for gcc */

	while (0 != oneshotgetimage()) {
		usleep(1000);
	}
	//fprintf(stderr, "after get image\n");

	server_socket = openServer();

	mini_ftpdatad(server_socket);

	//fprintf(stderr, "after deliver image\n");
	oneshotreleaseimage();

	pthread_exit(NULL);
	return NULL;
}
#endif


static int ftp_null(void)
{
	int ret;

	ret = ftpcmd(NULL, NULL);
	if (450 == ret || 451 == ret || 452 == ret || 550 == ret || 552 == ret) {
		insufficient_disk = 1;
		error_code = INSUFFICIENT_DISK_SPACE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (226 == ret) {
		error_code = TEST_SUCCEEDED;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (421 == ret) {
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (426 == ret) {
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (530 == ret) {
		error_code = INVALID_USERNAME_PASSWORD;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		web_config_error = 1;
	} else if (502 == ret) {
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
	} else if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
		if (verbose_flag)
			bb_simple_perror_msg("ftp_null, NULL fail\n");
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
	}
#if 0
	error_code = TCP_IP_SOCKET_ERROR;
	write_sysinfo_short(error_code, FTP_TEST_RESULT);
	if (verbose_flag)
		bb_simple_perror_msg("ftp_null, NULL fail\n");
	ret = -1;
#endif
	return ret;
}

#if 0
static int ftp_null_266(void)
{
	int ret;

	if ((ret = ftpcmd(NULL, NULL)) != 226) {
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("ftp_null_266, NULL fail\n");
		ret = -1;
	} else {
		error_code = TEST_SUCCEEDED;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = 0;
	}
	return ret;
}
#endif

static int ftp_stor(void)
{
	int ret;

	ret = ftpcmd("STOR", dst_filename);
	switch (ret) {
	case 125:
	case 150:
		break;
	case 425:
		/*Server can't build data connection: Cannot assign requested address.*/
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		if (verbose_flag)
			bb_perror_msg("STOR fail 425 ret = %d\n", ret);
		close(fd_data);
		ret = -1;
		break;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		if (verbose_flag)
			bb_perror_msg("STOR fail 0 ret = %d\n", ret);
		close(fd_data);
		ret = -1;
		break;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		break;
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		break;
	case 426:
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		break;
	case 450:
	case 451:
	case 452:
	case 552:
		insufficient_disk = 1;
		error_code = INSUFFICIENT_DISK_SPACE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		break;
	case 532:
	case 550:
	case 553:
		error_code = CAN_NOT_UPLOAD_IMAGE_FILE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("STOR fail\n");
		ret = -1;
		stor_cmd_fail = 1;
		//ftp_die("STOR");
		break;
	case 530:
		error_code = NOT_SUPPORT_PORT_MODE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		break;
	case 500:
	case 501:
	default:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
	}
	return ret;
}

static int ftp_port(void)
{
	int ret;

	ret = ftpcmd("PORT", port_cmd_s);
	//fprintf(stderr, "PORT ret = %d\n", ret);
	switch (ret) {
	case 200:
		break;
	case 0:
		if (1 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
			error_code = TCP_IP_SOCKET_ERROR;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}
		if (verbose_flag)
			bb_perror_msg("PORT fail ret = %d\n", ret);
		//close(fd_data);
		ret = -1;
		break;
	case 9999:
		error_code = FAILURE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		break;
	case 421:
		error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		break;
	case 530:
		error_code = NOT_SUPPORT_PORT_MODE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ret = -1;
		break;
	case 500:
	case 501:
		error_code = NOT_SUPPORT_PORT_MODE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_simple_perror_msg("PORT fail\n");
		port_pasv_not_support = 1;
		ret = -1;
		break;
		//ftp_die("PORT");
	default:
		ret = -1;
	}

	return ret;
}
#if 1
static int ftp_send_port(void)
{
	int ret;
	int server_socket = server_socket; /* for gcc */

	while (0 != oneshotgetimage()) {
		usleep(10000);
	}
	server_socket = openServer();
	if (verbose_flag)
		bb_perror_msg("server_socket = %d\n", server_socket);

	if (-1 == ftp_port()) {
		close(server_socket);
		ret = -1;
		goto EXIT;
	}

	if ( -1 == ftp_stor()) {
		close(server_socket);
		ret = -1;
		goto EXIT;
	}

	ret = mini_ftpdatad(server_socket);

	if (verbose_flag)
		bb_perror_msg("ftp_send_port, mini_ftpdatad ret = %d\n", ret);

	if (450 == ret || 451 == ret || 452 == ret || 550 == ret || 552 == ret || 421 == ret || 426 == ret || 9999 == ret) {
		goto EXIT;
	}
	/* does server confirm that transfer is finished? */

	ret = ftp_null();
	if (-1 == ret) {
		if (verbose_flag)
			bb_simple_perror_msg("after deliver_image port mode, mini_ftpdatad fail\n");
		ret = -1;
	} 
EXIT:
	oneshotreleaseimage();
	return ret;
}
#else
static int ftp_send_port(void)
{
	pthread_t a_thread;
	void *thread_result;
	int ret;
	int result;

#if 1
	result = pthread_create(&a_thread, NULL, ftp_data_connection_server, NULL);
	if (result != 0) {
		perror("Thread creation failed");
		exit(EXIT_FAILURE);
	}
#endif
	/* must let data connection server built first*/
	usleep(1000);

#if 1
	ret = ftpcmd("PORT", port_cmd_s);
	//fprintf(stderr, "PORT ret = %d\n", ret);
	switch (ret) {
	case 200:
		break;
	default:
		error_code = NOT_SUPPORT_PORT_MODE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ftp_die("PORT");
	}
#endif


#if 1
	ret = ftpcmd("STOR", dst_filename);
	switch (ret) {
	case 125:
	case 150:
		break;
	default:
		error_code = CAN_NOT_UPLOAD_IMAGE_FILE;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ftp_die("STOR");
	}
#endif

#if 1
	//printf("Waiting for thread to finish...\n");
	result = pthread_join(a_thread, &thread_result);
	if (result != 0) {
		perror("Thread join failed");
		exit(EXIT_FAILURE);
	}
#endif

#if 1
	/* does server confirm that transfer is finished? */
	if ((ret = ftpcmd(NULL, NULL)) != 226) {
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		ftp_die(NULL);
	}
#endif

	return ret;
}
#endif

static int ftp_send_pasv(void)
{
	int ret;

	while (0 != oneshotgetimage()) {
		usleep(10000);
	}

	if (-1 == ftp_pasv()) {
		ret = -1;	
		goto EXIT;
	}

	/* connect to the data socket */
	//fd_data = xconnect_ftpdata();
	fd_data = ftpconnect_stream(lsa, DATA_STREAM);

	if (verbose_flag)
		bb_perror_msg("ftpconnect_stream, fd_data = %d\n", fd_data);
	//return xconnect_stream(lsa);

	if (-1 != fd_data) {
		if (-1 == ftp_stor()) {
			close(fd_data);
			ret = -1;
			goto EXIT;
		}

		//ret = deliver_image(imgbuf + sizeof(struct FRM), fd_data);
		ret = deliver_image(imgbuf, fd_data);
		/* close data connection */
		close(fd_data);

		if (450 == ret || 451 == ret || 452 == ret || 550 == ret || 552 == ret || 421 == ret || 426 == ret || 9999 == ret) {
			goto EXIT;
		}
		ret = ftp_null();
		if (-1 == ret) {
			if (verbose_flag)
				bb_simple_perror_msg("after deliver_image(pasv mode) fail\n");
			ret = -1;
		}
	} else {
		error_code = TCP_IP_SOCKET_ERROR;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		if (verbose_flag)
			bb_perror_msg("ftp_send_pasv fail, fd_data = %d\n", fd_data);
		ret = -1;
	}
EXIT:
	oneshotreleaseimage();
	return ret;
}

#if ENABLE_FEATURE_FTPGETPUT_LONG_OPTIONS
static const char ftpputimage_longopts[] ALIGN1 =
"continue\0" Required_argument "c"
"verbose\0"  No_argument       "v"
"username\0" Required_argument "u"
"password\0" Required_argument "p"
"port\0"     Required_argument "P"
;
#endif

static void serve_segv_request(int signo)
{
	printf("signo = %d\n", signo);
	//got_signal = 1;
	signal_no = signo;
	//exit(0);
}

static void serve_kill_request(int signo)
{
	//printf("signo = %d\n", signo);
	got_signal = 1;
	signal_no = signo;
}

static void	convert_tm_to_test_str(struct tm *filename_time_tm, char *filename_time_s)
{
	sprintf(filename_time_s, "%04d", filename_time_tm->tm_year + 1900); 
	sprintf(filename_time_s+4, "%02d", filename_time_tm->tm_mon + 1); 
	sprintf(filename_time_s+6, "%02d", filename_time_tm->tm_mday); 
	sprintf(filename_time_s+8, "_"); 
	sprintf(filename_time_s+9, "%02d", filename_time_tm->tm_hour); 
	sprintf(filename_time_s+11, "%02d", filename_time_tm->tm_min); 
	sprintf(filename_time_s+13, "%02d", filename_time_tm->tm_sec); 
	//fprintf(stderr, "time = %s\n", filename_time_s);
}

static void	convert_tm_to_str(struct tm *filename_time_tm, char *filename_time_s)
{
	sprintf(filename_time_s, "%04d", filename_time_tm->tm_year + 1900); 
	sprintf(filename_time_s+4, "%02d", filename_time_tm->tm_mon + 1); 
	sprintf(filename_time_s+6, "%02d", filename_time_tm->tm_mday); 
	sprintf(filename_time_s+8, "%02d", filename_time_tm->tm_hour); 
	sprintf(filename_time_s+10, "%02d", filename_time_tm->tm_min); 
	sprintf(filename_time_s+12, "%02d", filename_time_tm->tm_sec); 
	//fprintf(stderr, "time = %s\n", filename_time_s);
}

static void get_test_time_string(char *filename_time_s)
{
	time_t current_time;
	struct tm filename_time_tm;

	time(&current_time);
	gmtime_r(&current_time, &filename_time_tm);
	convert_tm_to_test_str(&filename_time_tm, filename_time_s);
}

static void get_time_string(char *filename_time_s)
{
	time_t current_time;
	struct tm filename_time_tm;

	time(&current_time);
	gmtime_r(&current_time, &filename_time_tm);
	convert_tm_to_str(&filename_time_tm, filename_time_s);
	//printf("filename_time_s = %s\n", filename_time_s);
}

static void set_test_file_name(void)
{
	char filename_time_s[20];

	memset(filename_time_s, 0, sizeof filename_time_s);
	get_test_time_string(filename_time_s);
	sprintf(dst_filename, "%s%s.jpg", "test_", filename_time_s);
	//fprintf(stderr, "dst_filename = %s\n", dst_filename);
}

static void set_file_name(struct timeval *tv)
{
	static int file_seq = 0;
	static char filename_time_prev_s[20];

	switch (basefile_name_mode[0]) {
	case '0': /*overwrite*/
		if (-1 != freq_n) {
			if (suffix >= freq_n) {
				suffix = 1;
			} else {
				suffix++;
			}
		}
		gettimeofday(tv, NULL);
		shift_time = tv->tv_usec * 1000;
		sprintf(dst_filename, "%s.jpg", basefile_name);
		//fprintf(stderr, "dst_filename = %s\n", dst_filename);
		break;
	case '1': /*Date/Time Suffix*/
		switch (atoi(freq_mode)) {
		case 0:
			if (-1 != freq_n) {
				if (suffix >= freq_n) {
					suffix = 1;
				} else {
					suffix++;
				}
				if (1 == suffix) {
					memset(time_part, 0, sizeof time_part);
					gettimeofday(tv, NULL);
					get_time_string(time_part);
					//fprintf(stderr, "tv_sec = %ld, tv_usec = %ld\n", tv->tv_sec, tv->tv_usec);
					shift_time = tv->tv_usec * 1000;
				} else {
					shift_time = 0;
				}
				sprintf(dst_filename, "%s%s%02d.jpg", basefile_name, time_part, suffix);
				//fprintf(stderr, "1 dst_filename = %s\n", dst_filename);
			} else {
				/* auto */
				memset(time_part, 0, sizeof time_part);
				get_time_string(time_part);
				if (0 == strcmp(filename_time_prev_s, time_part)) {
					suffix++;
				} else {
					strcpy(filename_time_prev_s, time_part);
					suffix = 1;
				}
				sprintf(dst_filename, "%s%s%02d.jpg", basefile_name, time_part, suffix);
				//fprintf(stderr, "2 dst_filename = %s\n", dst_filename);
			}
			break;
		case 1: 
			{
				memset(time_part, 0, sizeof time_part);
				gettimeofday(tv, NULL);
				get_time_string(time_part);
				//fprintf(stderr, "tv_sec = %ld, tv_usec = %ld\n", tv->tv_sec, tv->tv_usec);
				shift_time = tv->tv_usec * 1000;
				suffix = 1;
				sprintf(dst_filename, "%s%s%02d.jpg", basefile_name, time_part, 1);
				//fprintf(stderr, "3 dst_filename = %s\n", dst_filename);
			}
			break;
		}
		break;
	case '2': /*Sequence Number*/
#if 0
		switch (atoi(freq_mode)) {
		case 0:
			if (-1 != freq_n) {
				gettimeofday(tv, NULL);
				shift_time = tv->tv_usec * 1000;
			} else {
				/* auto */
			}
			break;
		case 1:
			gettimeofday(tv, NULL);
			shift_time = tv->tv_usec * 1000;
			break;
		}
#endif
		if (-1 != freq_n) {
			if (suffix >= freq_n) {
				suffix = 1;
			} else {
				suffix++;
			}
		}
		gettimeofday(tv, NULL);
		shift_time = tv->tv_usec * 1000;
		if (file_seq >= max_seq_no_n) {
			file_seq = 1;
		} else {
			file_seq++;
		}
		sprintf(dst_filename, "%s%0*d.jpg", basefile_name, max_seq_no_n_len, file_seq);
		//fprintf(stderr, "dst_filename = %s\n", dst_filename);
		break;
	}
	return;
}


/* We hijack this constant to mean something else */
/* It doesn't hurt because we will remove this bit anyway */
#define DIE_ON_ERROR AI_CANONNAME


/* host: "1.2.3.4[:port]", "www.google.com[:port]"
 * port: if neither of above specifies port # */
static len_and_sockaddr* ftpstr2sockaddr(
		const char *host, int port,
		USE_FEATURE_IPV6(sa_family_t af,)
		int ai_flags)
{
	int rc;
	len_and_sockaddr *r = NULL;
	struct addrinfo *result = NULL;
	struct addrinfo *used_res;
	const char *org_host = host; /* only for error msg */
	const char *cp;
	struct addrinfo hint;

	/* Ugly parsing of host:addr */
	if (ENABLE_FEATURE_IPV6 && host[0] == '[') {
		/* Even uglier parsing of [xx]:nn */
		host++;
		cp = strchr(host, ']');
		if (!cp || cp[1] != ':') { /* Malformed: must have [xx]:nn */
			bb_error_msg("bad address '%s'", org_host);
			if (ai_flags & DIE_ON_ERROR)
				xfunc_die();
			return NULL;
		}
	} else {
		cp = strrchr(host, ':');
		if (ENABLE_FEATURE_IPV6 && cp && strchr(host, ':') != cp) {
			/* There is more than one ':' (e.g. "::1") */
			cp = NULL; /* it's not a port spec */
		}
	}
	if (cp) { /* points to ":" or "]:" */
		int sz = cp - host + 1;
		host = safe_strncpy(alloca(sz), host, sz);
		if (ENABLE_FEATURE_IPV6 && *cp != ':')
			cp++; /* skip ']' */
		cp++; /* skip ':' */
		port = bb_strtou(cp, NULL, 10);
		if (errno || (unsigned)port > 0xffff) {
			bb_error_msg("bad port spec '%s'", org_host);
			if (ai_flags & DIE_ON_ERROR)
				xfunc_die();
			return NULL;
		}
	}

	memset(&hint, 0 , sizeof(hint));
#if !ENABLE_FEATURE_IPV6
	hint.ai_family = AF_INET; /* do not try to find IPv6 */
#else
	hint.ai_family = af;
#endif
	/* Needed. Or else we will get each address thrice (or more)
	 * for each possible socket type (tcp,udp,raw...): */
	hint.ai_socktype = SOCK_STREAM;
	hint.ai_flags = ai_flags & ~DIE_ON_ERROR;
	rc = getaddrinfo(host, NULL, &hint, &result);
	if (rc || !result) {
		bb_error_msg("bad address '%s'", org_host);
		if (ai_flags & DIE_ON_ERROR)
			xfunc_die();
		goto ret;
	}
	used_res = result;
#if ENABLE_FEATURE_PREFER_IPV4_ADDRESS
	while (1) {
		if (used_res->ai_family == AF_INET)
			break;
		used_res = used_res->ai_next;
		if (!used_res) {
			used_res = result;
			break;
		}
	}
#endif
	//r = xmalloc(offsetof(len_and_sockaddr, u.sa) + used_res->ai_addrlen);
	r = malloc(offsetof(len_and_sockaddr, u.sa) + used_res->ai_addrlen);
	r->len = used_res->ai_addrlen;
	memcpy(&r->u.sa, used_res->ai_addr, used_res->ai_addrlen);
	set_nport(r, htons(port));
ret:
	freeaddrinfo(result);
	return r;
}
#if !ENABLE_FEATURE_IPV6
#define str2sockaddr(host, port, af, ai_flags) str2sockaddr(host, port, ai_flags)
#endif

static int is_noop_time_due(struct timeval *tv1, struct timeval *tv2)
{
	unsigned long tv1long, tv2long, tvelapsed;
	int ret = 0;

	tv1long = tv1->tv_sec * 1000000 + tv1->tv_usec;
	tv2long = tv2->tv_sec * 1000000 + tv2->tv_usec;
	tvelapsed = tv2long - tv1long;
	if (tvelapsed >= 60000000) {
		ret = 1;
	} else {
		ret = 0;
	}
	return ret;
}

static void calculate_sleep_time(struct timeval *tv1, struct timeval *tv2, struct timespec *ftp_timeToWait)
{
	unsigned long offset_sec, offset_nsec;
	unsigned long tv1long, tv2long, tvelapsed;
	long avg_ftp_one_img_time, avg_ftp_one_img_time_suffix;

	tv1long = tv1->tv_sec * 1000000 + tv1->tv_usec;
	tv2long = tv2->tv_sec * 1000000 + tv2->tv_usec;
	tvelapsed = tv2long - tv1long;
	if (tvelapsed >= 1000000) {
		tvelapsed %= 1000000;
		tvelapsed *= 1000;
	} else {
		tvelapsed *= 1000;
	}
	offset_nsec = tv2->tv_usec * 1000;
	offset_sec = tv2->tv_sec - tv1->tv_sec;

	memset(ftp_timeToWait, 0, sizeof ftp_timeToWait);

	switch (atoi(freq_mode)) {
	case 0:
		if (-1 != freq_n) {
			if (0 != offset_sec) {
				switch (basefile_name_mode[0]) {
				case '0': /*overwrite*/
					avg_ftp_one_img_time = (1000000000 * 1) / freq_n;
					avg_ftp_one_img_time_suffix = avg_ftp_one_img_time * suffix;
					ftp_timeToWait->tv_sec = 0 ;

					if((offset_nsec) >= avg_ftp_one_img_time_suffix) {
						ftp_timeToWait->tv_nsec = 0;
					} else {
						ftp_timeToWait->tv_nsec = (avg_ftp_one_img_time_suffix) - (offset_nsec);
						ftp_timeToWait->tv_nsec -= CTRL_EXTRA_TIME * 1000;
						if (ftp_timeToWait->tv_nsec < 0) ftp_timeToWait->tv_nsec = 0;
					}
					break;
				case '1': /*Date/Time Suffix*/
					ftp_timeToWait->tv_sec = 0;
					ftp_timeToWait->tv_nsec = 0;
					break;
				case '2': /*Sequence Number*/
					avg_ftp_one_img_time = (1000000000 * 1) / freq_n;
					avg_ftp_one_img_time_suffix = avg_ftp_one_img_time * suffix;
					ftp_timeToWait->tv_sec = 0 ;

					if((offset_nsec) >= avg_ftp_one_img_time_suffix) {
						ftp_timeToWait->tv_nsec = 0;
					} else {
						ftp_timeToWait->tv_nsec = (avg_ftp_one_img_time_suffix) - (offset_nsec);
						ftp_timeToWait->tv_nsec -= CTRL_EXTRA_TIME * 1000;
						if (ftp_timeToWait->tv_nsec < 0) ftp_timeToWait->tv_nsec = 0;
					}
					break;
				default:
					printf("3 nsec Error = %ld\n", ftp_timeToWait->tv_nsec);
				}
			} else {
				avg_ftp_one_img_time = (1000000000 * 1) / freq_n;
				avg_ftp_one_img_time_suffix = avg_ftp_one_img_time * suffix;
				ftp_timeToWait->tv_sec = 0;
				//printf("offset_nsec = %ld\n", offset_nsec);
				//printf("avg_ftp_one_img_time = %ld\n", avg_ftp_one_img_time);
				//printf("avg_ftp_one_img_time_suffix = %ld\n", avg_ftp_one_img_time_suffix);
				if((offset_nsec) >= avg_ftp_one_img_time_suffix) {
					ftp_timeToWait->tv_nsec = 0;
				} else {
					ftp_timeToWait->tv_nsec = (avg_ftp_one_img_time_suffix) - (offset_nsec);
					ftp_timeToWait->tv_nsec -= CTRL_EXTRA_TIME * 1000;
					if (ftp_timeToWait->tv_nsec < 0) ftp_timeToWait->tv_nsec = 0;

				}
			}
		} else {
			ftp_timeToWait->tv_sec = 0;
			ftp_timeToWait->tv_nsec = 0;
		}
		break;
	case 1:
		/*long var 's max value is pow(2,32) = 4234967296,
		 *so we can not use 
		 * unsigned long diff;
		 * diff = atoi(seconds_per_frame) * 1000000000 - tvelapsed - shift_time;
		 * ftp_timeToWait->tv_sec = diff / 1000000000;
		 * ftp_timeToWait->tv_nsec = diff  % 1000000000;
		 */
		if (0 != offset_sec) {
			ftp_timeToWait->tv_sec = atoi(seconds_per_frame) - (offset_sec + 1);
			ftp_timeToWait->tv_nsec = 1000000000 - (offset_nsec) + 400000000;
			if (ftp_timeToWait->tv_nsec >= 1000000000) ftp_timeToWait->tv_nsec = 999999999;
		} else {
			ftp_timeToWait->tv_sec = atoi(seconds_per_frame) - (offset_sec + 1);
			ftp_timeToWait->tv_nsec = 1000000000 - (tvelapsed + shift_time) + 400000000;
			if (ftp_timeToWait->tv_nsec >= 1000000000) ftp_timeToWait->tv_nsec = 999999999;
		}
		break;
	}

	return;
}

static void register_signal_handler(void)
{
	struct sigaction act, act1;
	act.sa_handler = serve_kill_request;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	sigaction(SIGTERM, &act, 0);

	act1.sa_handler = serve_segv_request;
	sigemptyset(&act1.sa_mask);
	act1.sa_flags = 0;
	sigaction(SIGSEGV, &act1, 0);
	return;
}

static void read_ftp_config(void)
{

	char max_seq_no_s[16];	/*_s for string*/
	char passive_mode[2];
	char folder_s[8];

	/* Set default values */
	memset(user, 0, sizeof user);
	memset(password, 0, sizeof password );
	memset(schedule_mode, 0, sizeof schedule_mode);
	strcpy(user, "TEST");
	strcpy(user, nvram_bufget(RT2860_NVRAM, "FTPUserName"));

	if (!strcmp(user, "")) {
		error_code = INVALID_USERNAME_PASSWORD;
		write_sysinfo_short(error_code, FTP_TEST_RESULT);
		nvram_close(RT2860_NVRAM);
		ftp_die("USER");
	}
	strcpy(password, nvram_bufget(RT2860_NVRAM, "FTPPassword"));
	strcpy(freq_mode, nvram_bufget(RT2860_NVRAM, "FTPScheduleVideoFrequencyMode"));
	strcpy(schedule_mode, nvram_bufget(RT2860_NVRAM, "FTPScheduleMode"));

	strcpy(server_path, nvram_bufget(RT2860_NVRAM, "FTPDirectoryPath"));
	strcpy(passive_mode, nvram_bufget(RT2860_NVRAM, "FTPPassiveMode"));

	switch (atoi(passive_mode)) {
	case 0:
		ftp_imagesend = ftp_send_port;
		break;
	case 1:
		ftp_imagesend = ftp_send_pasv;
		break;
	default:
		//fprintf(stderr, "passive mode = %s\n", passive_mode);
		break;
	}

	strcpy(ftp_address, nvram_bufget(RT2860_NVRAM, "FTPHostAddress"));
	strcpy(ftp_port_s, nvram_bufget(RT2860_NVRAM, "FTPPortNumber"));
	//ftp_port_n = atoi(ftp_port_s);


	strcpy(basefile_name, nvram_bufget(RT2860_NVRAM, "FTPScheduleBaseFileName"));
	strcpy(basefile_name_mode, nvram_bufget(RT2860_NVRAM, "FTPScheduleFileMode"));
	strcpy(max_seq_no_s, nvram_bufget(RT2860_NVRAM, "FTPScheduleMaxFileSequenceNumber"));
	max_seq_no_n_len = strlen(max_seq_no_s);
	max_seq_no_n = atoi(max_seq_no_s);
	strcpy(frames_per_second, nvram_bufget(RT2860_NVRAM, "FTPScheduleFramePerSecond")); 
	strcpy(seconds_per_frame, nvram_bufget(RT2860_NVRAM, "FTPScheduleSecondPerFrame"));

	switch (atoi(freq_mode)) {
	case 0:
		freq_n = atoi(frames_per_second);
		break;
	case 1:
		freq_n = 1;
		break;
	}
	set_port_cmd_ip_part();

	memset(folder_s, 0, sizeof folder_s);
	strcpy(folder_s, nvram_bufget(RT2860_NVRAM, "FTPCreateFolderInterval")); 
	folder_mode = atoi(folder_s);

	nvram_close(RT2860_NVRAM);
	return;
}

#if 0
static int create_new_folder_per_sec(char *path)
{
	int ret = 0, ret_cwd = 0, ret_mkd = 0;
	char dir_path_new[20];
	static char dir_path_old[20]="abcd";
	//int different = 0;

	//printf("444444444444444444444444444444444444444444444444444444444444444444444444\n");
	//printf("basefile_name_mode[0]= %c\n", basefile_name_mode[0]);
	switch (basefile_name_mode[0]) {
	case '0': /*overwrite*/
#if 0
		ret_mkd = ftp_mkd("");
		if (257 == ret_mkd) {
		}
#endif
		break;

	case '1': /*Date/Time Suffix*/
		if (1 == suffix) {
			memset(dir_path_new, 0, sizeof dir_path_new);
			//strncpy(dir_path_new, filename_time_prev_s, 10);
			strncpy(dir_path_new, path + YEAR_MONTH_DAY_LEN + HOUR_LEN + MINUTE_LEN, SEC_LEN);
			//printf("filename_time_prev_s = %s\n", filename_time_prev_s);
			//printf("path = %s\n", path);
			//printf("dir_path_new = %s\n", dir_path_new);
			//printf("dir_path_old = %s\n", dir_path_old);
			//different = strncmp(dir_path_old, dir_path_new, 14);
			//printf("different = %d\n", different);
			//if (0 != different) {
			if (DIR_SEC == dir_level) {
				ret_cwd = ftp_cwd("..");
				dir_level = DIR_DAY;
			} 
			//printf("2 dir_path_new = %s\n", dir_path_new);
			ret_mkd = ftp_mkd(dir_path_new);
			if (257 == ret_mkd) {
				ret = 0;
				ret_cwd = ftp_cwd(dir_path_new);
				dir_level = DIR_SEC;
			} else {
				ret = -1;
			}
			strcpy(dir_path_old, dir_path_new);
			//}
		}
		break;
	case '2': /*Sequence Number*/
		ret =ftp_mkd("");
		break;
	default:
		break;

	}
	//}

	//printf("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
return ret;
}
#endif

static int create_new_folder_per_half_hour(char *path)
{
	int ret = 0, ret_cwd = 0, ret_mkd = 0;
	char dir_path_new[20];
	int different = 0, min_diff = 0;

	//printf("333333333333333333333333333333333333333333333333333333333333333333333333333\n");
	//printf("basefile_name_mode[0]= %c\n", basefile_name_mode[0]);
	switch (basefile_name_mode[0]) {
	case '0': /*overwrite*/
		break;
	case '1': /*Date/Time Suffix*/
		if (1 == suffix) {
			memset(dir_path_new, 0, sizeof dir_path_new);
			//strncpy(dir_path_new, filename_time_prev_s, 10);
			strncpy(dir_path_new, path + YEAR_MONTH_DAY_LEN, HOUR_LEN + MINUTE_LEN);
			min_diff = strncmp(dir_path_new + HOUR_LEN, "30", MINUTE_LEN);
			//printf("min_diff = %d\n", min_diff);
			if (min_diff < 0) 
				sprintf(dir_path_new + HOUR_LEN, "%s", "00");
			else if(min_diff > 0)
				sprintf(dir_path_new + HOUR_LEN, "%s", "30");
			else if (0 == min_diff)
				sprintf(dir_path_new + HOUR_LEN, "%s", "30");
			//printf("filename_time_prev_s = %s\n", filename_time_prev_s);
			//printf("half hour path = %s\n", path);
			//printf("dir_path_new = %s\n", dir_path_new);
			//printf("half_hour_dir_path_old = %s\n", half_hour_dir_path_old);
			different = strncmp(half_hour_dir_path_old, dir_path_new, HOUR_LEN + MINUTE_LEN);
			//printf("different = %d\n", different);
			if (0 != different) {
				if (DIR_HOUR_OR_HALF_HOUR == dir_level) {
					ret_cwd = ftp_cwd("..");
					if (250 != ret_cwd) {
						ret = -1;
					} else {
						dir_level = DIR_DAY;
					}
				} 
				//printf("2 dir_path_new = %s\n", dir_path_new);

				if (-1 == ret) {
					goto EXIT;
				} else {
					ret_cwd = ftp_cwd(dir_path_new);
					if (250 != ret_cwd) {
						ret_mkd = ftp_mkd(dir_path_new);
						if (257 == ret_mkd) {
							ret_cwd = ftp_cwd(dir_path_new);
						}
					}
					if (250 == ret_cwd) {
						dir_level = DIR_HOUR_OR_HALF_HOUR;
						strcpy(half_hour_dir_path_old, dir_path_new);
					} else {
						ret = -1;
					}
				}
			} else {
				switch (dir_level) {
				case DIR_HOUR_OR_HALF_HOUR:
					break;
				case DIR_DAY:
					ret_cwd = ftp_cwd(half_hour_dir_path_old);
					if (250 == ret_cwd)
						dir_level = DIR_HOUR_OR_HALF_HOUR;
					else
						ret = -1;
					break;
				default:
					ret = -1;
				}
			}
		}
		break;
	case '2': /*Sequence Number*/
		break;
	default:
		break;
	}

	//printf("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
EXIT:
	return ret;
}

static int create_new_folder_per_hour(char *path)
{
	int ret = 0, ret_cwd = 0, ret_mkd = 0;
	char dir_path_new[20];
	int hour_diff = 0;

	//printf("2222222222222222222222222222222222222222222222222222222222222222222222222\n");
	//printf("basefile_name_mode[0]= %c\n", basefile_name_mode[0]);
	switch (basefile_name_mode[0]) {
	case '0': /*overwrite*/
		break;
	case '1': /*Date/Time Suffix*/
		if (1 == suffix) {
			memset(dir_path_new, 0, sizeof dir_path_new);
			//strncpy(dir_path_new, filename_time_prev_s, 10);
			strncpy(dir_path_new, path + YEAR_MONTH_DAY_LEN, HOUR_LEN);
			sprintf(dir_path_new + HOUR_LEN, "%s", "00");
			hour_diff = strncmp(hour_dir_path_old, dir_path_new, HOUR_LEN);
			//printf("hour_diff = %d\n", hour_diff);
			//printf("hour path = %s\n", path);
			//printf("dir_path_new = %s\n", dir_path_new);
			//printf("hour_dir_path_old = %s\n", hour_dir_path_old);
			if (0 != hour_diff) {
				if (DIR_HOUR_OR_HALF_HOUR == dir_level) {
					ret_cwd = ftp_cwd("..");
					if (250 != ret_cwd) {
						ret = -1;
					} else {
						dir_level = DIR_DAY;
					}
				} 
				//printf("2 dir_path_new = %s\n", dir_path_new);
				if (-1 == ret) {
					goto EXIT;
				} else {
					ret_cwd = ftp_cwd(dir_path_new);
					if (250 != ret_cwd) {
						ret_mkd = ftp_mkd(dir_path_new);
						if (257 == ret_mkd) {
							ret_cwd = ftp_cwd(dir_path_new);
						}
					}
					if (250 == ret_cwd) {
						dir_level = DIR_HOUR_OR_HALF_HOUR;
						strcpy(hour_dir_path_old, dir_path_new);
					} else {
						ret = -1;
					}
				}
			} else {
				switch (dir_level) {
				case DIR_HOUR_OR_HALF_HOUR:
					break;
				case DIR_DAY:
					ret_cwd = ftp_cwd(hour_dir_path_old);
					if (250 == ret_cwd)
						dir_level = DIR_HOUR_OR_HALF_HOUR;
					else
						ret = -1;
					break;
				default:
					ret = -1;
				}
			}
		}
		break;
	case '2': /*Sequence Number*/
		break;
	default:
		break;
	}

	//printf("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
EXIT:
	return ret;
}

static int create_new_folder_per_day(char *path)
{
	int ret = 0, ret_cwd = 0, ret_mkd = 0;
	char dir_path_new[20];
	int different = 0;

	//printf("11111111111111111111111111111111111111111111111111111111111111111111111\n");
	//printf("basefile_name_mode[0]= %c\n", basefile_name_mode[0]);
	switch (basefile_name_mode[0]) {
	case '0': /*overwrite*/
		break;
	case '1': /*Date/Time Suffix*/
		if (1 == suffix) {
			memset(dir_path_new, 0, sizeof dir_path_new);
			//strncpy(dir_path_new, filename_time_prev_s, 10);
			strncpy(dir_path_new, path, YEAR_MONTH_DAY_LEN);
			//printf("filename_time_prev_s = %s\n", filename_time_prev_s);
			//printf("path = %s\n", path);
			//printf("dir_path_new = %s\n", dir_path_new);
			//printf("day_dir_path_old = %s\n", day_dir_path_old);
			different = strncmp(day_dir_path_old, dir_path_new, YEAR_MONTH_DAY_LEN);
			//printf("different = %d\n", different);
			if (0 != different) {
				switch (dir_level) {
				case DIR_HOUR_OR_HALF_HOUR:
					ret_cwd = ftp_cwd("../..");
					if (250 == ret_cwd) {
						dir_level = DIR_ROOT;
					} else {
						ret = -1;
					}
					break;
				case DIR_DAY:
					ret_cwd = ftp_cwd("..");
					if (250 == ret_cwd) {
						dir_level = DIR_ROOT;
					} else {
						ret = -1;
					}
					break;
				case DIR_ROOT:
					break;
				default:
					ret = -1;
				}
				if (-1 == ret) {
					goto EXIT;
				} else {
					//printf("2 dir_path_new = %s\n", dir_path_new);
					ret_cwd = ftp_cwd(dir_path_new);
					if (250 != ret_cwd) {
						ret_mkd = ftp_mkd(dir_path_new);
						if (257 == ret_mkd) {
							ret_cwd = ftp_cwd(dir_path_new);
						}
					}
					if (250 == ret_cwd) {
						dir_level = DIR_DAY;
						/*only for set time by console e.g. Current Time is 06211440, then we do #date -u 06221440*/
						strcpy(hour_dir_path_old, "");
						strcpy(half_hour_dir_path_old, "");
						/*update path*/
						strcpy(day_dir_path_old, dir_path_new);
					} else {
						ret = -1;
						goto EXIT;
					}
				}
			} else {
				switch (dir_level) {
				case DIR_HOUR_OR_HALF_HOUR:
				case DIR_DAY:
					break;
				case DIR_ROOT:
					ret_cwd = ftp_cwd(day_dir_path_old);
					if (250 == ret_cwd)
						dir_level = DIR_DAY;
					else
						ret = -1;
					break;
				default:
					ret = -1;
				}
			}
		}
		break;
	case '2': /*Sequence Number*/
		break;
	default:
		break;

	}
	//printf("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
EXIT:
	return ret;
}

static int do_folder_jobs(void)
{
	int ret = 0;
	//printf("folder_mode = %d\n", folder_mode);
	switch (folder_mode) {
	case 0:/*do not create folder*/
		break;
	case 30:/*per halh-hour*/
		ret = create_new_folder_per_day(time_part);
		if (0 == ret)
			ret = create_new_folder_per_half_hour(time_part);
		break;
	case 60:/*per hour*/
		ret = create_new_folder_per_day(time_part);
		if (0 == ret)
			ret = create_new_folder_per_hour(time_part);
		break;
	case 1440:/*per day*/
		ret = create_new_folder_per_day(time_part);
		break;
	default:
#if 0
		/*per second*/
		ret = create_new_folder_per_day(time_part);
		if (0 == ret)
			ret = create_new_folder_per_sec(time_part);
#endif
		break;
	}
	return ret;
}

static void get_motion_flag(int signo)
{
	if (SIGUSR1 == signo) {
		read_sysinfo(sizeof(motion_flag), (char *)&motion_flag, USER1_SIG_FLAG);
		//printf("motion_flag.process = %d\n", motion_flag.process);
		//printf("motion_flag.sig_flag = %lu\n", motion_flag.sig_flag);
	}
}

static int back_to_home_folder(void)
{
	int ret = 0;

	switch (folder_mode) {
	case 0:/*do not create folder*/
		ret = 250;
		break;
	case 30:/*per halh-hour*/
		ret = ftp_cwd("../../");
		break;
	case 60:/*per hour*/
		ret = ftp_cwd("../../");
		break;
	case 1440:/*per day*/
		ret = ftp_cwd("../");
		break;
	default:
#if 0
		/*per second*/
		ret = create_new_folder_per_day(time_part);
		if (0 == ret)
			ret = create_new_folder_per_sec(time_part);
#endif
		break;
	}
	return ret;
}

static void image_put_test(void)
{
	struct timeval tv;
	int rcode = 0;

	set_file_name(&tv);
	if (0 != do_folder_jobs()) {
		folder_fail = 1;
	} else if (250 == back_to_home_folder()) {
		set_test_file_name();
		rcode = ftp_imagesend();
		if (226 == rcode) errno = 0;
	}
}

static void image_put_by_frequency(void)
{
	int ctrl_extra = 0;
	int rcode = 0;
	struct timespec ftp_timeToWait;

#if 0
	if (-1 != freq_n) {
		struct timeval tv;
		struct timespec req;
		/* wait a few nano seconds for start transfer image from for example 2010-0209-15(hr)-33(min)-47(sec)-10^9(nsec)*/
		gettimeofday(&tv, NULL);
		req.tv_sec = 0;
		req.tv_nsec = (10^9) - tv.tv_usec * 1000;
		nanosleep(&req, NULL);
	}
#endif

	do {
		struct timeval tv1, tv2, tv_noop1, tv_noop2;
		int null_ret = 0;
#if 0
		if (2 == atoi(schedule_mode) && true != ftp_test) {
			/*Motion Detection*/
			int fret = 0;
			int stream_closed = 0;
			while (FTP_USER1_SIG_MOTION_ENDED == motion_flag.sig_flag) {
				if (0 == got_signal) {
					usleep(50000);
					if (0 == stream_closed) {
						null_ret = is_there_control_session_response(0, CTRL_EXTRA_TIME);
						if (0 != null_ret) {
							if (verbose_flag)
								printf("motion, null_ret = %d\n", null_ret);
							fret = fclose(control_stream);

							if (verbose_flag)
								printf("motion, fret = %d\n", fret);
							stream_closed = 1;
						}
					}
				} else {
					goto EXIT;
				}
			}
			if (1 == stream_closed) {
				error_control = 1;
				goto EXIT;
			}
		}

#else
		if (2 == atoi(schedule_mode) && true != ftp_test) {
			/*Motion Detection*/
			gettimeofday(&tv_noop1, NULL);
			while (FTP_USER1_SIG_MOTION_ENDED == motion_flag.sig_flag) {
				if (0 == got_signal) {
					int fret = 0;
					usleep(50000);
					/*receive 421 timeout response from server*/
					null_ret = is_there_control_session_response(0, 0);
					if (0 != null_ret) {
						if (verbose_flag)
							printf("motion, null_ret = %d\n", null_ret);
						fret = fclose(control_stream);

						if (verbose_flag)
							printf("motion, fret = %d\n", fret);
						error_control = 1;
						goto EXIT;
					} else {
						/*keep connection alive*/
						/*(1 min)*/
						gettimeofday(&tv_noop2, NULL);
						if (1 == is_noop_time_due(&tv_noop1, &tv_noop2)) {
							gettimeofday(&tv_noop1, NULL);
							if(200 != ftp_noop()) {
								fret = fclose(control_stream);
								error_control = 1;
								goto EXIT;
							}
						}
					}
				} else {
					goto EXIT;
				}
			}
		}
#endif

		//printf("motion_flag.sig_flag = %lu\n", motion_flag.sig_flag);
		error_control = 0;
		stream_error_control = 0;
		set_file_name(&tv1);
		//printf("tv1.tv_sec = %ld\n", tv1.tv_sec);
		//printf("tv1.tv_usec = %ld\n", tv1.tv_usec);

		if (0 != do_folder_jobs()) {
			folder_fail = 1;
			break;
		}
		errno = 0;
		rcode = ftp_imagesend();
		if (226 == rcode) errno = 0;
		gettimeofday(&tv2, NULL);
		//printf("tv2.tv_sec = %ld\n", tv2.tv_sec);
		//printf("tv2.tv_usec = %ld\n", tv2.tv_usec);

		calculate_sleep_time(&tv1, &tv2, &ftp_timeToWait);
		//printf("tv_sec = %ld\n", ftp_timeToWait.tv_sec);
		//printf("tv_nsec = %ld\n", ftp_timeToWait.tv_nsec);
		nanosleep(&ftp_timeToWait, NULL);
		//system("date");
		if (verbose_flag) {
			bb_perror_msg("ftp_imagesned rcode = %d\n", rcode);
		}
		//get_control_sesseion_response();
		ctrl_extra = is_there_control_session_response(0, CTRL_EXTRA_TIME);
	} while (0 == got_signal && 0 == port_pasv_not_support && 0 == stor_cmd_fail && 0 == insufficient_disk && 226 == rcode && 0 == ctrl_extra);

EXIT:
	if (verbose_flag) {
		printf("got_signal = %d\n", got_signal);
		printf("port_pasv_not_support= %d\n", port_pasv_not_support);
		printf("stor_cmd_fail= %d\n", stor_cmd_fail);
		printf("insufficient_disk = %d\n", insufficient_disk);
		printf("folder_fail = %d\n", folder_fail);
		printf("rcode = %d\n", rcode);
		printf("ctrl_extra = %d\n", ctrl_extra);
	}

	if (verbose_flag) {
		bb_perror_msg("error_control = %d\n", error_control);
		bb_perror_msg("stream_error_control = %d\n", stream_error_control);
	}
}

static void	do_motion_mode_setting(void)
{
	struct sigaction act;
	pid_t mypid = 0;
	struct USER_PID ftp_client_pid;

	if (2 == atoi(schedule_mode) && true != ftp_test) {
		/*Motion Detection*/
		motion_flag.process = 4;
		motion_flag.sig_flag = FTP_USER1_SIG_MOTION_ENDED;
		act.sa_handler = get_motion_flag;
		sigemptyset(&act.sa_mask);
		act.sa_flags = 0;
		sigaction(SIGUSR1, &act, 0);
		//signal(SIGUSR1, get_motion_flag);

		mypid = getpid();
		ftp_client_pid.process = 4;
		ftp_client_pid.pid = mypid;
		if (verbose_flag)
			printf("mypid = %lu\n", (unsigned long)mypid);
		write_sysinfo(sizeof ftp_client_pid, (char *)&ftp_client_pid, USER_SIG_PID);
		strcpy(freq_mode, "0"); /*Frames/Second*/
		freq_n = -1; /*auto*/
	}
}

static void *ftp_loop(void *none)
{
	int login;
	int server_dir_resp;
	/* ftp stauts:
	 * 434: Ftp server not available
	 * 425: Can't build data connection: Connection refused.
	 * 530: Login incorrect
	 * 550: No such file or directory
	 */
	none = NULL;
	do {
		error_code = NO_TEST_CONDUCTED;
		//write_sysinfo_short(error_code, FTP_TEST_RESULT);
		/*init*/
		got_signal = 0;
		signal_no = 0;
		insufficient_disk = 0;
		port_pasv_not_support = 0;
		stor_cmd_fail = 0;
		web_config_error = 0;
		folder_fail = 0;
		error_data = 0;
		suffix = 0;
		errno = 0;
		error_control = 0;
		stream_error_control = 0;
		control_stream_fd = 0;
		control_stream = NULL;
		dir_level = DIR_ROOT;
		strcpy(day_dir_path_old, "");
		strcpy(hour_dir_path_old, "");
		strcpy(half_hour_dir_path_old, "");
		frame_seqnum = 0xffffffff;
		/* We want to do exactly _one_ DNS lookup, since some
		 * sites (i.e. ftp.us.debian.org) use round-robin DNS
		 * and we want to connect to only one IP... */
		lsa = ftpstr2sockaddr(ftp_address, bb_lookup_port(ftp_port_s, "tcp", 21), AF_UNSPEC, 0);
		if (NULL == lsa) {
			printf("xhost2sockaddr fail\n");
			error_code = INVALID_FTP_SERVER_ADDRESS;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
			web_config_error = 1;
			goto EXIT;
		}
		if (verbose_flag) {
			bb_perror_msg("Connecting to %s (%s)\n", ftp_address, xmalloc_sockaddr2dotted(&lsa->u.sa));
		}

		if ( 220 == ftp_syn()) {
		} else {
			error_code = CAN_NOT_CONNECT_TO_FTP_SERVER;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
			sleep(5);
			goto EXIT;
		}

		login = ftp_login();
		if (230 == login || 202 == login) {
			/*success*/
		} else if (530 == login) {
			/*not loggin*/
			if (verbose_flag)
				bb_simple_perror_msg("ftp_login web config fail\n");
			web_config_error = 1;
			goto EXIT;
		} else {
			/*socket error*/
			if (verbose_flag)
				bb_simple_perror_msg("ftp_login socket fail\n");
			goto EXIT;
		}

		if (200 == ftp_TYPE_I()) {
			/*success*/
		} else {
			goto EXIT;
		}

		server_dir_resp = ftp_cwd(server_path);
		if (250 == server_dir_resp) {
			/*success*/
			dir_level = DIR_ROOT;
		} else if (550 == server_dir_resp || 500 == server_dir_resp || 501 == server_dir_resp || 502 == server_dir_resp || 530 == server_dir_resp ) {
			if (verbose_flag)
				bb_simple_perror_msg("ftp_cwd fail\n");
			web_config_error = 1;
			goto EXIT;
		} else {
			/*socket error*/
			if (verbose_flag)
				bb_simple_perror_msg("cwd socket fail\n");
			goto EXIT;
		}

		if (true != ftp_test) {
			error_code = TEST_SUCCEEDED;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
			image_put_by_frequency();
		} else {
			image_put_test();
		}
EXIT:
		if (NULL != lsa)
			free(lsa);
		if (0 == error_control && 0 == stream_error_control) {
			if (0 == is_there_socket_error(control_stream_fd, &error_control, 1)) {
				int fret = 0;
				if (verbose_flag)
					printf("QUIT\n");
				ftpcmd("QUIT", NULL);
				usleep(1000);	
				fret = fclose(control_stream);
				printf("1 fclose ret = %d, errno = %d, dst_filename = %s\n",	fret, errno, dst_filename);
			} else {
				if (NULL != control_stream) {
					int fret = 0;

					if (verbose_flag) {
						int fer = 0, feo = 0;
						fer = ferror(control_stream);
						printf("fer = %d\n", fer);
						feo = feof(control_stream);
						printf("feo = %d\n", feo);
					}
					fret = fclose(control_stream);
					printf("2 fclose ret = %d, errno = %d, dst_filename = %s\n",	fret, errno, dst_filename);
					printf("ftp_loop, control_stream_fd = %d\n", control_stream_fd);

					fret = close(control_stream_fd);
					printf("2 close ret = %d, errno = %d\n",	fret, errno);
					printf("ftp_loop, after close, control_stream_fd = %d\n", control_stream_fd);
				}
			}
		}
		/*server could send RST after receive QUIT or serverself shutdown */
		if (verbose_flag) {
			printf("control_stream = %p\n", control_stream);
			printf("control_stream_fd = %d\n", control_stream_fd);
			printf("error_control = %d\n", error_control);
			printf("stream_error_control = %d\n", stream_error_control);
		}
		if ( 1 == insufficient_disk) {
			error_code = INSUFFICIENT_DISK_SPACE;
			write_sysinfo_short(error_code, FTP_TEST_RESULT);
		}

	} while (0 == got_signal && true != ftp_test);
	//while (0 == got_signal && 0 == web_config_error && 0 == port_pasv_not_support && 0 == stor_cmd_fail && 0 == insufficient_disk && true != ftp_test && 0 == folder_fail);

	if (verbose_flag) {
		printf("got_signal = %d\n", got_signal);
		printf("web_config_error= %d\n", web_config_error);
		printf("port_pasv_not_support= %d\n", port_pasv_not_support);
		printf("stor_cmd_fail= %d\n", stor_cmd_fail);
		printf("insufficient_disk = %d\n", insufficient_disk);
		printf("folder_fail = %d\n", folder_fail);
	}

#if FTP_THREAD_ENABLE
	pthread_exit(NULL);
#else
	return NULL;
#endif
}

int ftpputimage_main(int argc, char **argv) MAIN_EXTERNALLY_VISIBLE;
int ftpputimage_main(int argc UNUSED_PARAM, char **argv)
{
	unsigned opt;
#if FTP_THREAD_ENABLE
	pthread_t a_thread;
	int res;
	void *thread_result;
#endif

	errno = 0;
	register_signal_handler();

	INIT_G();
	read_ftp_config();


	/*
	 * Decipher the command line
	 */
#if ENABLE_FEATURE_FTPGETPUT_LONG_OPTIONS
	applet_long_options = ftpputimage_longopts;
#endif
	//opt_complementary = "=3:vv:cc"; /* must have 3 params; -v and -c count */
	//opt = getopt32(argv, "cvu:p:P:", &user, &password, &port,
	//				&verbose_flag, &do_continue);
	opt_complementary = "vv"; /* -v count */
	opt = getopt32(argv, "tv", &verbose_flag);
	argv += optind;

	ftp_test = false;
	if (opt & 0x0001) {
		ftp_test = true;
	}
	if (verbose_flag)
		printf("opt = %u, ftp_test = %d\n", opt, ftp_test);

	while (0 != alloc_imagebuf()) {
		printf("alloc_imagebuf fail\n");
		sleep(1);
	}

	do_motion_mode_setting();
#if FTP_THREAD_ENABLE
	do {
		res = pthread_create(&a_thread, NULL, ftp_loop, NULL);
		if (res != 0) {
			perror("Thread creation failed");
			continue;
			//exit(EXIT_FAILURE);
		}
		res = pthread_join(a_thread, &thread_result);
		if (res != 0) {
			perror("Thread join failed");
			continue;
			//exit(EXIT_FAILURE);
		}
	} while (0 == got_signal && 0 == web_config_error && 0 == port_pasv_not_support && 0 == stor_cmd_fail && 0 == insufficient_disk && true != ftp_test && 0 == folder_fail);
#else
	ftp_loop(NULL);
#endif

	free_imagebuf();
	xfunc_die();
	return 0;
}
