#include <linux/init.h>
#include <linux/version.h>
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/fcntl.h>
#include <asm/system.h>
#include <linux/delay.h>
#include "system.h"
#if ( defined(MODEL_TVIP651W) || defined(MODEL_TVIP651WI) )
#include "i2c_drv.h"

#define	I2C_DEBUG		0

#ifdef  CONFIG_DEVFS_FS
#include <linux/devfs_fs_kernel.h>
static devfs_handle_t devfs_handle;
#endif

int i2cdrv_major =  218;
int PanTiltFirmwareVersion=0;
I2C_CONFIG	I2CConfigData;
/*----------------------------------------------------------------------*/
void I2CMasterInit(void)
{
int	busy_loop;
u32	reg_data;
	/* reset i2c block */
	reg_data = RT2880_REG(RT2880_RSTCTRL_REG);
	reg_data |= RSTCTRL_I2C_RESET;
	RT2880_REG(RT2880_RSTCTRL_REG) = reg_data;
	udelay(1);
	reg_data &= ~RSTCTRL_I2C_RESET;
	RT2880_REG(RT2880_RSTCTRL_REG) = reg_data;
	for( busy_loop = 0; busy_loop < 50000; busy_loop++ );
	//
	RT2880_REG(RT2880_I2C_CONFIG_REG) = I2CConfigData.ConfigRegister;
	RT2880_REG(RT2880_I2C_CLKDIV_REG) = I2CConfigData.ClockDivisor;
	RT2880_REG(RT2880_I2C_DEVADDR_REG) = I2CConfigData.DeviceAddress;
	RT2880_REG(RT2880_I2C_ADDR_REG) = 0;
	RT2880_REG(RT2880_I2C_BYTECNT_REG) = 0;	// set byte-count = 1 byte
}

int I2CWrite(u32 Address, u32 Data)
{
int	busy_loop;

#if ( I2C_DEBUG != 0 )
//	printk("I2CWrite : address=(%x) Data=(%x)\n", Address, Data);
#endif
	RT2880_REG(RT2880_I2C_CONFIG_REG) = I2CConfigData.ConfigRegister;
	RT2880_REG(RT2880_I2C_ADDR_REG) = Address;
	RT2880_REG(RT2880_I2C_DATAOUT_REG) = Data;
	RT2880_REG(RT2880_I2C_STARTXFR_REG) = WRITE_CMD;
	busy_loop = I2C_BUSY_LOOP;
	do {
		if ( ! IS_BUSY ) {
			udelay(300);
			return 0;
		}
	} while ( --busy_loop );
	printk("I2CWrite timeout: address=(%x) data=(%x) status=(%x)\n",
				Address, Data, RT2880_REG(RT2880_I2C_STATUS_REG));
	return -1;
}

int I2CRead(u32 Address, u32 *Data)
{
int	busy_loop;

#if ( I2C_DEBUG != 0 )
//	printk("I2CRead : address=(%x) \n", Address);
#endif
	// output DevAddress(W) & SubAddress
	RT2880_REG(RT2880_I2C_CONFIG_REG) = I2CConfigData.ConfigRegister;
	RT2880_REG(RT2880_I2C_ADDR_REG) = Address;
	RT2880_REG(RT2880_I2C_STARTXFR_REG) = WRITE_CMD_NODATA;
	busy_loop = I2C_BUSY_LOOP;
	do {
		if ( ! IS_BUSY )	break;
	} while ( --busy_loop );
	if ( busy_loop <= 0 )	printk("I2CRead : busy status=(%x)\n", RT2880_REG(RT2880_I2C_STATUS_REG));
	// output DevAddress(R) & Data
	RT2880_REG(RT2880_I2C_CONFIG_REG) = I2CConfigData.ConfigRegister | I2C_CFG_ADDRDIS;
	RT2880_REG(RT2880_I2C_STARTXFR_REG) = READ_CMD;
	busy_loop = I2C_BUSY_LOOP;
	do {
		if ( (! IS_BUSY) && (IS_DATARDY) ) {
			*Data = RT2880_REG(RT2880_I2C_DATAIN_REG);
			return 0;
		}
	} while ( --busy_loop );
	printk("I2CRead timeout: address=(%x) Loop=(%d) status=(%x)\n",
				 Address, I2C_BUSY_LOOP, RT2880_REG(RT2880_I2C_STATUS_REG));
	return -1;
}

/*----------------------------------------------------------------------*/
/* Pan Tilt Command							*/
/*----------------------------------------------------------------------*/
#define	HOME_PAN_DEGREE		170
#define	HOME_TILT_DEGREE	30
#define	PAN_STEP_MULTIPLICATOR	(64*1)
#define	TILT_STEP_MULTIPLICATOR	(64*1)

int	PanDegreeMax, PanDegreeMin, TiltDegreeMax, TiltDegreeMin;
int	PanStepMax, PanStepMin, PanStepCurrent;
int	TiltStepMax, TiltStepMin, TiltStepCurrent;
int	PanStepPerDegree, PanStepPerDegreeBase=1, TiltStepPerDegree, TiltStepPerDegreeBase=1;
int	PanStepMultiplicator=1, TiltStepMultiplicator=1;
int	PanStepDefaultOffset=0, TiltStepDefaultOffset=0;
int	PanTiltActiveMode=PT_ACTIVE_MODE_RUN;
PAN_TILT_COMMAND	SwingStatus;

#if ( I2C_DEBUG != 0 )
void DumpStepInfo(void)
{
u32	low_byte, high_byte;
int	step;
	
	if ( PanTiltFirmwareVersion != -1 ) {
		printk("StepInfo: ");
		I2CRead(PT_ADDRESS_PAN_CURRENT_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_PAN_CURRENT_STEP_L, &low_byte);
		step = (int)((high_byte<<8) + low_byte);
		printk("Pan=(%d,0x%X,", step, step);
		I2CRead(PT_ADDRESS_PAN_MOVE_TARGET_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_PAN_MOVE_TARGET_STEP_L, &low_byte);
		step = (int)((high_byte<<8) + low_byte);
		printk("T=0x%X), ", step);
		//
		I2CRead(PT_ADDRESS_TILT_CURRENT_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_TILT_CURRENT_STEP_L, &low_byte);
		step = (int)((high_byte<<8) + low_byte);
		printk("Tilt=(%d,0x%X,", step, step);
		I2CRead(PT_ADDRESS_TILT_MOVE_TARGET_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_TILT_MOVE_TARGET_STEP_L, &low_byte);
		step = (int)((high_byte<<8) + low_byte);
		printk("T=0x%X)\n", step);
	}
}
#endif

int PanDegreeToMotorStep(int Degree)
{
int	step;

#if ( PAN_MOTOR_ASCENDING_FOR_LEFT == 0 )
	Degree = PanDegreeMax - Degree;
#endif
	step = Degree * PanStepPerDegree / PanStepPerDegreeBase;
	if ( PanStepMultiplicator > 1 ) {
		step -= (step % PanStepMultiplicator);	/* even pulse number */
	}
	step += PanStepMin;
	return step;
}

int TiltDegreeToMotorStep(int Degree)
{
int	step;

#if ( TILT_MOTOR_ASCENDING_FOR_UP == 0 )
	Degree = TiltDegreeMax - Degree;
#endif
	step = Degree * TiltStepPerDegree / TiltStepPerDegreeBase;
	if ( TiltStepMultiplicator > 1 ) {
		step -= (step % TiltStepMultiplicator);	/* even pulse number */
	}
	step += TiltStepMin;
	return step;
}

int PanMotorStepToDegree(int Step)
{
int	degree;

	Step -= PanStepMin;
	Step *= PanStepPerDegreeBase;
	degree = Step / PanStepPerDegree;
	if ( (Step % PanStepPerDegree) != 0 )	degree++;
#if ( PAN_MOTOR_ASCENDING_FOR_LEFT == 0 )
	degree = PanDegreeMax - degree;
#endif
	return degree;
}

int TiltMotorStepToDegree(int Step)
{
int	degree;

	Step -= TiltStepMin;
	Step *= TiltStepPerDegreeBase;
	degree = Step / TiltStepPerDegree;
	if ( (Step % TiltStepPerDegree) != 0 )	degree++;
#if ( TILT_MOTOR_ASCENDING_FOR_UP == 0 )
	degree = TiltDegreeMax - degree;
#endif
	return degree;
}

int PanGotoStep(int Step, int MotorStarting)
{
u32	byte_data;

	byte_data = (u32)(Step & 0xff);
	I2CWrite(PT_ADDRESS_PAN_MOVE_TARGET_STEP_L, byte_data);
	byte_data = (u32)((Step>>8) & 0xff);
	I2CWrite(PT_ADDRESS_PAN_MOVE_TARGET_STEP_H, byte_data);
	PanStepCurrent = Step;
	if ( MotorStarting )	I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_PAN_TILT_GOTO_POSITION);
#if ( I2C_DEBUG != 0 )
	printk("PanGotoStep: pan step=(0x%X)\n", PanStepCurrent);
#endif
	return 0;
}

int TiltGotoStep(int Step, int MotorStarting)
{
u32	byte_data;

	byte_data = (u32)(Step & 0xff);
	I2CWrite(PT_ADDRESS_TILT_MOVE_TARGET_STEP_L, byte_data);
	byte_data = (u32)((Step>>8) & 0xff);
	I2CWrite(PT_ADDRESS_TILT_MOVE_TARGET_STEP_H, byte_data);
	TiltStepCurrent = Step;
	if ( MotorStarting )	I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_PAN_TILT_GOTO_POSITION);
#if ( I2C_DEBUG != 0 )
	printk("TiltGotoStep: tilt step=(0x%X)\n", TiltStepCurrent);
#endif
	return 0;
}

int PanTiltGetPosition(pPAN_TILT_COMMAND pPanTiltCommand)
{
#if 0
u32	low_byte, high_byte;
int	step;

	if ( PanTiltFirmwareVersion != -1 ) {
		I2CRead(PT_ADDRESS_PAN_CURRENT_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_PAN_CURRENT_STEP_L, &low_byte);
		step = (int)((high_byte<<8) + low_byte);
		pPanTiltCommand->PanPosition = PanMotorStepToDegree(step);
		PanStepCurrent = step;
		//
		I2CRead(PT_ADDRESS_TILT_CURRENT_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_TILT_CURRENT_STEP_L, &low_byte);
		step = (int)((high_byte<<8) + low_byte);
		pPanTiltCommand->TiltPosition = TiltMotorStepToDegree(step);
		TiltStepCurrent = step;
	}
#if ( I2C_DEBUG != 0 )
	if ( PanTiltFirmwareVersion == -1 ) {
		pPanTiltCommand->PanPosition = PanMotorStepToDegree(PanStepCurrent);
		pPanTiltCommand->TiltPosition = TiltMotorStepToDegree(TiltStepCurrent);
	}
	printk("PanTiltGetPosition: pan step=(0x%X), tilt step=(0x%X)\n", PanStepCurrent, TiltStepCurrent);
#endif
#else
	pPanTiltCommand->PanPosition = PanMotorStepToDegree(PanStepCurrent);
	pPanTiltCommand->TiltPosition = TiltMotorStepToDegree(TiltStepCurrent);
#endif
#if ( I2C_DEBUG != 0 )
	DumpStepInfo();
#endif
	return 0;
}

int PanTiltGotoPosition(pPAN_TILT_COMMAND pPanTiltCommand)
{
int	step;

	if ( pPanTiltCommand->PanPosition > PanDegreeMax )	pPanTiltCommand->PanPosition = PanDegreeMax;
	if ( pPanTiltCommand->PanPosition < PanDegreeMin )	pPanTiltCommand->PanPosition = PanDegreeMin;
	if ( pPanTiltCommand->TiltPosition > TiltDegreeMax )	pPanTiltCommand->TiltPosition = TiltDegreeMax;
	if ( pPanTiltCommand->TiltPosition < TiltDegreeMin )	pPanTiltCommand->TiltPosition = TiltDegreeMin;
	step = PanDegreeToMotorStep(pPanTiltCommand->PanPosition);
	PanGotoStep(step, 0);
	step = TiltDegreeToMotorStep(pPanTiltCommand->TiltPosition);
	TiltGotoStep(step, 0);
	I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_PAN_TILT_GOTO_POSITION);
#if ( I2C_DEBUG != 0 )
	printk("PanTiltGotoPosition: pan degree=(%d), tilt degree=(%d)\n", pPanTiltCommand->PanPosition, pPanTiltCommand->TiltPosition);
#endif
	return 0;
}

int PanTiltGetLimitSwitchStatus(pPAN_TILT_COMMAND pPanTiltCommand)
{
u32	byte_data;

	I2CRead(PT_ADDRESS_MOTOR_MOVING_STATUS, &byte_data);
	if ( byte_data != 0 ) {
		pPanTiltCommand->SwitchStatus = -EBUSY;
		return -EBUSY;
	}
	I2CRead(PT_ADDRESS_LIMIT_SWITCH_STATUS, &byte_data);
	pPanTiltCommand->SwitchStatus = (int) byte_data;
	return 0;
}

int PanLostStepStatus(pPAN_TILT_COMMAND pPanTiltCommand)
{
u32	low_byte, high_byte;
int	step;

	// wait lost step test complete
	I2CRead(PT_ADDRESS_PAN_LOST_STEP_H, &high_byte);
	if ( high_byte == 0xff ) {
		pPanTiltCommand->SwitchStatus = -EBUSY;
		return -EBUSY;
	}
#if 0
	I2CRead(PT_ADDRESS_MOTOR_MOVING_STATUS, &low_byte);
	if ( low_byte == 0xff ) {
		pPanTiltCommand->SwitchStatus = -EBUSY;
		return -EBUSY;
	}
#endif
	//
	I2CRead(PT_ADDRESS_PAN_LOST_STEP_L, &low_byte);
	step = (int)((high_byte<<8) + low_byte);
	pPanTiltCommand->PanPosition = step;
	return 0;
}

int TiltLostStepStatus(pPAN_TILT_COMMAND pPanTiltCommand)
{
u32	low_byte, high_byte;
int	step;

	// wait lost step test complete
	I2CRead(PT_ADDRESS_TILT_LOST_STEP_H, &high_byte);
	if ( high_byte == 0xff ) {
		pPanTiltCommand->SwitchStatus = -EBUSY;
		return -EBUSY;
	}
#if 0
	I2CRead(PT_ADDRESS_MOTOR_MOVING_STATUS, &low_byte);
	if ( low_byte == 0xff ) {
		pPanTiltCommand->SwitchStatus = -EBUSY;
		return -EBUSY;
	}
#endif
	//
	I2CRead(PT_ADDRESS_TILT_LOST_STEP_L, &low_byte);
	step = (int)((high_byte<<8) + low_byte);
	pPanTiltCommand->TiltPosition = step;
	return 0;
}

int PanTiltGetMovingStatus(pPAN_TILT_COMMAND pPanTiltCommand)
{
u32	byte_data;

	I2CRead(PT_ADDRESS_MOTOR_MOVING_STATUS, &byte_data);
	pPanTiltCommand->MovingStatus = (int) byte_data;
	return 0;
}

int PanTiltGetHome(pPAN_TILT_COMMAND pPanTiltCommand)
{
u32	low_byte, high_byte;
int	step;

	if ( PanTiltFirmwareVersion != -1 ) {
		I2CRead(PT_ADDRESS_PAN_HOME_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_PAN_HOME_STEP_L, &low_byte);
		step = (int)((high_byte<<8) + low_byte);
		pPanTiltCommand->PanPosition = PanMotorStepToDegree(step);
		//
		I2CRead(PT_ADDRESS_TILT_HOME_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_TILT_HOME_STEP_L, &low_byte);
		step = (int)((high_byte<<8) + low_byte);
		pPanTiltCommand->TiltPosition = TiltMotorStepToDegree(step);
	}
#if ( I2C_DEBUG != 0 )
	if ( PanTiltFirmwareVersion == -1 ) {
		pPanTiltCommand->PanPosition = PanMotorStepToDegree(29696);
		pPanTiltCommand->TiltPosition = TiltMotorStepToDegree(8976);
	}
	printk("PanTiltGetHome: pan degree=(%d), tilt degree=(%d)\n", pPanTiltCommand->PanPosition, pPanTiltCommand->TiltPosition);
#endif
	return 0;
}

int PanAscending(pPAN_TILT_COMMAND pPanTiltCommand)
{
int	last_pan=PanStepCurrent, steps;

	if ( PanStepCurrent >= PanStepMax )	return 1;
	if ( PanStepMultiplicator > 1 ) {
		steps = PanStepDefaultOffset + PanStepMultiplicator * pPanTiltCommand->Steps;	
	} else {
		steps = PanStepDefaultOffset + PanStepPerDegree * pPanTiltCommand->Steps;
	}
	PanStepCurrent += steps;
	if ( PanStepCurrent > PanStepMax )	PanStepCurrent = PanStepMax;
	if ( last_pan != PanStepCurrent )	PanGotoStep(PanStepCurrent, 1);
#if ( I2C_DEBUG != 0 )
	printk("PanAscending: step=(0x%X), new pan=(0x%X)\n", pPanTiltCommand->Steps, PanStepCurrent);
#endif
	return 0;
}

int PanDescending(pPAN_TILT_COMMAND pPanTiltCommand)
{
int	last_pan=PanStepCurrent, steps;

	if ( PanStepCurrent <= PanStepMin )	return 1;
	if ( PanStepMultiplicator > 1 ) {
		steps = PanStepDefaultOffset + PanStepMultiplicator * pPanTiltCommand->Steps;	
	} else {
		steps = PanStepDefaultOffset + PanStepPerDegree * pPanTiltCommand->Steps;
	}
	PanStepCurrent -= steps;
	if ( PanStepCurrent < PanStepMin )	PanStepCurrent = PanStepMin;
	if ( last_pan != PanStepCurrent )	PanGotoStep(PanStepCurrent, 1);
#if ( I2C_DEBUG != 0 )
	printk("PanDescending: step=(0x%X), new pan=(0x%X)\n", pPanTiltCommand->Steps, PanStepCurrent);
#endif
	return 0;
}

int TiltAscending(pPAN_TILT_COMMAND pPanTiltCommand)
{
int	last_tilt=TiltStepCurrent, steps;

	if ( TiltStepCurrent >= TiltStepMax )	return 1;
	if ( TiltStepMultiplicator > 1 ) {
		steps = TiltStepDefaultOffset + TiltStepMultiplicator * pPanTiltCommand->Steps;
	} else {
		steps = TiltStepDefaultOffset + TiltStepPerDegree * pPanTiltCommand->Steps;
	}
	TiltStepCurrent += steps;
	if ( TiltStepCurrent > TiltStepMax )	TiltStepCurrent = TiltStepMax;
	if ( last_tilt != TiltStepCurrent )	TiltGotoStep(TiltStepCurrent, 1);
#if ( I2C_DEBUG != 0 )
	printk("TiltAscending: step=(0x%X), new tilt=(0x%X)\n", pPanTiltCommand->Steps, TiltStepCurrent);
#endif
	return 0;
}

int TiltDescending(pPAN_TILT_COMMAND pPanTiltCommand)
{
int	last_tilt=TiltStepCurrent, steps;

	if ( TiltStepCurrent <= TiltStepMin )	return 1;
	if ( TiltStepMultiplicator > 1 ) {
		steps = TiltStepDefaultOffset + TiltStepMultiplicator * pPanTiltCommand->Steps;
	} else {
		steps = TiltStepDefaultOffset + TiltStepPerDegree * pPanTiltCommand->Steps;
	}
	TiltStepCurrent -= steps;
	if ( TiltStepCurrent < TiltStepMin )	TiltStepCurrent = TiltStepMin;
	if ( last_tilt != TiltStepCurrent )	TiltGotoStep(TiltStepCurrent, 1);
#if ( I2C_DEBUG != 0 )
	printk("TiltDescending: step=(0x%X), new tilt=(0x%X)\n", pPanTiltCommand->Steps, TiltStepCurrent);
#endif
	return 0;
}

int PanTiltCommand(pPAN_TILT_COMMAND pPanTiltCommand)
{
int	ret_value=0;
u32	low_byte, high_byte;

#if ( I2C_DEBUG == 0 )
	if ( PanTiltFirmwareVersion == -1 ) {
		if ( pPanTiltCommand->Function == PT_FUNC_GET_VERSION ) {
			pPanTiltCommand->Version = PanTiltFirmwareVersion;
		} else	memset(pPanTiltCommand, 0, sizeof(PAN_TILT_COMMAND));
		return -EPERM;
	}
#endif
	if ( PanTiltActiveMode == PT_ACTIVE_MODE_PRIVACY ) {
		if ( (pPanTiltCommand->Function != PT_FUNC_SET_PRIVACY) &&
					(pPanTiltCommand->Function != PT_FUNC_GET_ACTIVE_MODE) &&
					(pPanTiltCommand->Function != PT_FUNC_GET_VERSION) ) {
			memset(pPanTiltCommand, 0, sizeof(PAN_TILT_COMMAND));
			return -EACCES;
		}
	} // ....if ( PanTiltActiveMode == PT_ACTIVE_MODE_PRIVACY )
	switch ( pPanTiltCommand->Function ) {
	case PT_FUNC_GET_VERSION:
		pPanTiltCommand->Version = PanTiltFirmwareVersion;
		break;
	case PT_FUNC_GO_HOME:
		PanStepCurrent = PanDegreeToMotorStep(HOME_PAN_DEGREE);
		low_byte = (u32)PanStepCurrent & 0xff;
		high_byte = ((u32)PanStepCurrent & 0xff00)>>8;
		I2CWrite(PT_ADDRESS_PAN_MOVE_TARGET_STEP_L, low_byte);
		I2CWrite(PT_ADDRESS_PAN_MOVE_TARGET_STEP_H, high_byte);
		TiltStepCurrent = TiltDegreeToMotorStep(HOME_TILT_DEGREE);
		low_byte = (u32)TiltStepCurrent & 0xff;
		high_byte = ((u32)TiltStepCurrent & 0xff00)>>8;
		I2CWrite(PT_ADDRESS_TILT_MOVE_TARGET_STEP_L, low_byte);
		I2CWrite(PT_ADDRESS_TILT_MOVE_TARGET_STEP_H, high_byte);
		ret_value = I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_HOME);
		break;
	case PT_FUNC_GET_HOME:
//		ret_value = PanTiltGetHome(pPanTiltCommand);
		pPanTiltCommand->PanPosition = HOME_PAN_DEGREE;
		pPanTiltCommand->TiltPosition = HOME_TILT_DEGREE;
		break;
	case PT_FUNC_CALIBRATION:
		PanStepCurrent = PanDegreeToMotorStep(HOME_PAN_DEGREE);
		low_byte = (u32)PanStepCurrent & 0xff;
		high_byte = ((u32)PanStepCurrent & 0xff00)>>8;
		I2CWrite(PT_ADDRESS_PAN_MOVE_TARGET_STEP_L, low_byte);
		I2CWrite(PT_ADDRESS_PAN_MOVE_TARGET_STEP_H, high_byte);
		TiltStepCurrent = TiltDegreeToMotorStep(HOME_TILT_DEGREE);
		low_byte = (u32)TiltStepCurrent & 0xff;
		high_byte = ((u32)TiltStepCurrent & 0xff00)>>8;
		I2CWrite(PT_ADDRESS_TILT_MOVE_TARGET_STEP_L, low_byte);
		I2CWrite(PT_ADDRESS_TILT_MOVE_TARGET_STEP_H, high_byte);
		ret_value = I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_CALIBRATION);
		break;
	case PT_FUNC_GET_POTISION:
		ret_value = PanTiltGetPosition(pPanTiltCommand);
		break;
	case PT_FUNC_MOVE_LEFT:
		ret_value = PanAscending(pPanTiltCommand);
		pPanTiltCommand->PanPosition = PanMotorStepToDegree(PanStepCurrent);
		pPanTiltCommand->TiltPosition = TiltMotorStepToDegree(TiltStepCurrent);
		break;
	case PT_FUNC_MOVE_RIGHT:
		ret_value = PanDescending(pPanTiltCommand);
		pPanTiltCommand->PanPosition = PanMotorStepToDegree(PanStepCurrent);
		pPanTiltCommand->TiltPosition = TiltMotorStepToDegree(TiltStepCurrent);
		break;
	case PT_FUNC_MOVE_UP:
		ret_value = TiltAscending(pPanTiltCommand);
		pPanTiltCommand->PanPosition = PanMotorStepToDegree(PanStepCurrent);
		pPanTiltCommand->TiltPosition = TiltMotorStepToDegree(TiltStepCurrent);
		break;
	case PT_FUNC_MOVE_DOWN:
		ret_value = TiltDescending(pPanTiltCommand);
		pPanTiltCommand->PanPosition = PanMotorStepToDegree(PanStepCurrent);
		pPanTiltCommand->TiltPosition = TiltMotorStepToDegree(TiltStepCurrent);
		break;
	case PT_FUNC_GO_POSITION:
		ret_value = PanTiltGotoPosition(pPanTiltCommand);
		break;
	case PT_FUNC_PAN_SPEED:
		ret_value = I2CWrite(PT_ADDRESS_PAN_MOTOR_SPEED, (u32)pPanTiltCommand->Speed);
		break;
	case PT_FUNC_TILT_SPEED:
		ret_value = I2CWrite(PT_ADDRESS_TILT_MOTOR_SPEED, (u32)pPanTiltCommand->Speed);
		break;
	case PT_FUNC_GET_MAX_POSITION:
		pPanTiltCommand->PanPosition = PanDegreeMax;
		pPanTiltCommand->TiltPosition = TiltDegreeMax;
		break;
	case PT_FUNC_GET_MIN_POSITION:
		pPanTiltCommand->PanPosition = PanDegreeMin;
		pPanTiltCommand->TiltPosition = TiltDegreeMin;
		break;
	case PT_FUNC_LIMIT_SWITCH_TEST:
		ret_value = I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_CALIBRATION);
		break;
	case PT_FUNC_LIMIT_SWITCH_STATUS:
		ret_value = PanTiltGetLimitSwitchStatus(pPanTiltCommand);
		break;
	case PT_FUNC_LOST_STEP_TEST_PAN:
		ret_value = I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_PAN_LOST_STEP_TEST);
		ret_value = I2CWrite(PT_ADDRESS_PAN_LOST_STEP_H, (u32) 0xff);
		ret_value = I2CWrite(PT_ADDRESS_PAN_LOST_STEP_L, (u32) 0xff);
		break;
	case PT_FUNC_LOST_STEP_STATUS_PAN:
		ret_value = PanLostStepStatus(pPanTiltCommand);
		break;
	case PT_FUNC_LOST_STEP_TEST_TILT:
		ret_value = I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_TILT_LOST_STEP_TEST);
		ret_value = I2CWrite(PT_ADDRESS_TILT_LOST_STEP_H, (u32) 0xff);
		ret_value = I2CWrite(PT_ADDRESS_TILT_LOST_STEP_L, (u32) 0xff);
		break;
	case PT_FUNC_LOST_STEP_STATUS_TILT:
		ret_value = TiltLostStepStatus(pPanTiltCommand);
		break;
	case PT_FUNC_GET_MOVING_STATUS:
		ret_value = PanTiltGetMovingStatus(pPanTiltCommand);
		break;
	case PT_FUNC_SET_PRIVACY: // 19
		if ( pPanTiltCommand->Privacy == 0 ) {
			ret_value = I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_PAN_TILT_GOTO_POSITION);
			PanTiltActiveMode = PT_ACTIVE_MODE_RUN;
		} else {
			SwingStatus.Function = PT_FUNC_SWING_STOP;
			ret_value = I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_PRIVACY);
			PanTiltActiveMode = PT_ACTIVE_MODE_PRIVACY;
		}
		break;
	case PT_FUNC_GET_ACTIVE_MODE: // 20
		pPanTiltCommand->ActiveMode = PanTiltActiveMode;
		break;
	// swing status data
	case PT_FUNC_SWING_STATUS:
		memcpy(pPanTiltCommand, &SwingStatus, sizeof(PAN_TILT_COMMAND));
		break;
	case PT_FUNC_SWING_STOP:
#if ( I2C_DEBUG != 0 )
		if ( SwingStatus.Function == PT_FUNC_SWING_STOP ) {
			ret_value = I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_CALIBRATION);
			PanStepCurrent = PanDegreeToMotorStep(HOME_PAN_DEGREE);
			TiltStepCurrent = TiltDegreeToMotorStep(HOME_TILT_DEGREE);
		}
#endif
		memset(&SwingStatus, sizeof(PAN_TILT_COMMAND), 0);
		SwingStatus.Function = PT_FUNC_SWING_STOP;
		break;
	case PT_FUNC_SWING_PRESET:
	case PT_FUNC_SWING_HORIZONTAL:
	case PT_FUNC_SWING_VERTICAL:
		if ( SwingStatus.Function == PT_FUNC_SWING_STOP ) {
			memcpy(&SwingStatus, pPanTiltCommand, sizeof(PAN_TILT_COMMAND));
#if ( I2C_DEBUG != 0 )
			printk("i2cdrv: set swing (%d)\n", pPanTiltCommand->Function);
#endif
		} else	ret_value = -EBUSY;
		break;
	default:
		printk("i2cdrv: invalid pan-tilt command (%d)\n", pPanTiltCommand->Function);
		ret_value = -EINVAL;
		break;
	}
	return ret_value;
}

int PanTiltInit(void)
{
u32	low_byte, high_byte;
int	home_pan_pos, home_tilt_pos;
int	power_on_mode=0;

	I2CRead(PT_ADDRESS_FIRMWARE_VERSION, &high_byte);
	PanTiltFirmwareVersion = (int) high_byte;
	if ( (high_byte == 0) || (high_byte >= 0x80) )	PanTiltFirmwareVersion = -1;
	//
	if ( PanTiltFirmwareVersion != -1 ) {
		PanTiltActiveMode = PT_ACTIVE_MODE_RUN;
		// get pan-tilt motor information, and set parameters
		I2CRead(PT_ADDRESS_PAN_STEP_PER_DEGREE, &high_byte);
		PanStepPerDegree = (int)high_byte;
		if ( PanStepPerDegree == 0 )	PanStepPerDegree = 1;
		PanStepPerDegreeBase = 1;
		I2CRead(PT_ADDRESS_TILE_STEP_PER_DEGREE, &high_byte);
		TiltStepPerDegree = (int)high_byte;
		if ( TiltStepPerDegree == 0 )	TiltStepPerDegree = 1;
		TiltStepPerDegreeBase = 1;
		//
		PanStepMultiplicator = PAN_STEP_MULTIPLICATOR;
		TiltStepMultiplicator = TILT_STEP_MULTIPLICATOR;
		if ( PanStepMultiplicator > 1 ) {
			if ( (PanStepPerDegree % PanStepMultiplicator) == 0 ) {	
				PanStepMultiplicator = PanStepMultiplicator * (PanStepPerDegree/PanStepMultiplicator);
			} else {
				PanStepMultiplicator = PanStepMultiplicator * ((PanStepPerDegree/PanStepMultiplicator)+1);
			}
		} else	PanStepMultiplicator = 1;
		if ( TiltStepMultiplicator > 1 ) {
			if ( (TiltStepPerDegree % TiltStepMultiplicator) == 0 ) {
				TiltStepMultiplicator = TiltStepMultiplicator * (TiltStepPerDegree/TiltStepMultiplicator);
			} else {
				TiltStepMultiplicator = TiltStepMultiplicator * ((TiltStepPerDegree/TiltStepMultiplicator)+1);
			}
		} else	TiltStepMultiplicator = 1;
		//
		I2CRead(PT_ADDRESS_TILT_MAX_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_TILT_MAX_STEP_L, &low_byte);
		TiltStepMax = (int)((high_byte<<8) + low_byte);
		I2CRead(PT_ADDRESS_TILT_MAX_SITE_BUFFER_STEPS_H, &high_byte);
		I2CRead(PT_ADDRESS_TILT_MAX_SITE_BUFFER_STEPS_L, &low_byte);
		TiltStepMax -= (int)((high_byte<<8) + low_byte);
		I2CRead(PT_ADDRESS_TILT_MIN_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_TILT_MIN_STEP_L, &low_byte);
		TiltStepMin = (int)((high_byte<<8) + low_byte);
		I2CRead(PT_ADDRESS_TILT_MIN_SITE_BUFFER_STEPS_H, &high_byte);
		I2CRead(PT_ADDRESS_TILT_MIN_SITE_BUFFER_STEPS_L, &low_byte);
		TiltStepMin += (int)((high_byte<<8) + low_byte);
		//
		I2CRead(PT_ADDRESS_PAN_MAX_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_PAN_MAX_STEP_L, &low_byte);
		PanStepMax = (int)((high_byte<<8) + low_byte);
		I2CRead(PT_ADDRESS_PAN_MAX_SITE_BUFFER_STEPS_H, &high_byte);
		I2CRead(PT_ADDRESS_PAN_MAX_SITE_BUFFER_STEPS_L, &low_byte);
		PanStepMax -= (int)((high_byte<<8) + low_byte);
		I2CRead(PT_ADDRESS_PAN_MIN_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_PAN_MIN_STEP_L, &low_byte);
		PanStepMin = (int)((high_byte<<8) + low_byte);
		I2CRead(PT_ADDRESS_PAN_MIN_SITE_BUFFER_STEPS_H, &high_byte);
		I2CRead(PT_ADDRESS_PAN_MIN_SITE_BUFFER_STEPS_L, &low_byte);
		PanStepMin += (int)((high_byte<<8) + low_byte);
		//
		TiltDegreeMax = (TiltStepMax - TiltStepMin) * TiltStepPerDegreeBase / TiltStepPerDegree;
		TiltDegreeMin = 0;
		PanDegreeMax = (PanStepMax - PanStepMin) * PanStepPerDegreeBase / PanStepPerDegree;
		PanDegreeMin = 0;
		// check HOME position
		I2CRead(PT_ADDRESS_TILT_HOME_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_TILT_HOME_STEP_L, &low_byte);
		home_tilt_pos = (int)((high_byte<<8) + low_byte);
		home_tilt_pos = TiltMotorStepToDegree(home_tilt_pos);
		I2CRead(PT_ADDRESS_PAN_HOME_STEP_H, &high_byte);
		I2CRead(PT_ADDRESS_PAN_HOME_STEP_L, &low_byte);
		home_pan_pos = (int)((high_byte<<8) + low_byte);
		home_pan_pos = PanMotorStepToDegree(home_pan_pos);
		if ( (home_tilt_pos != HOME_TILT_DEGREE) || (home_pan_pos != HOME_PAN_DEGREE) )	power_on_mode = 1;
		// set HOME and motor init position
		PanStepCurrent = PanDegreeToMotorStep(HOME_PAN_DEGREE);
		low_byte = (u32)PanStepCurrent & 0xff;
		high_byte = ((u32)PanStepCurrent & 0xff00)>>8;
		I2CWrite(PT_ADDRESS_PAN_HOME_STEP_L, low_byte);
		I2CWrite(PT_ADDRESS_PAN_HOME_STEP_H, high_byte);
		I2CWrite(PT_ADDRESS_PAN_MOVE_TARGET_STEP_L, low_byte);
		I2CWrite(PT_ADDRESS_PAN_MOVE_TARGET_STEP_H, high_byte);
		TiltStepCurrent = TiltDegreeToMotorStep(HOME_TILT_DEGREE);
		low_byte = (u32)TiltStepCurrent & 0xff;
		high_byte = ((u32)TiltStepCurrent & 0xff00)>>8;
		I2CWrite(PT_ADDRESS_TILT_HOME_STEP_L, low_byte);
		I2CWrite(PT_ADDRESS_TILT_HOME_STEP_H, high_byte);
		I2CWrite(PT_ADDRESS_TILT_MOVE_TARGET_STEP_L, low_byte);
		I2CWrite(PT_ADDRESS_TILT_MOVE_TARGET_STEP_H, high_byte);
		if ( power_on_mode ) {
			// go Home position
			I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_HOME);
		} else {
			I2CWrite(PT_ADDRESS_MOTOR_STATUS, PT_MOTOR_STATUS_CALIBRATION);
		}
		PanStepDefaultOffset = 0;	//PanStepPerDegree;
		TiltStepDefaultOffset = 0;	//TiltStepPerDegree;
#if ( I2C_DEBUG != 0 )
	} else {
		TiltStepMax = 17952;
		TiltStepMin = 400;
		PanStepMax = 59800;
		PanStepMin = 400;
		PanStepPerDegree = 172;
		PanStepPerDegreeBase = 1;
		TiltStepPerDegree = 91;
		TiltStepPerDegreeBase = 1;
		PanStepMultiplicator =1;
		TiltStepMultiplicator=1;
		TiltStepCurrent = TiltDegreeToMotorStep(HOME_TILT_DEGREE);
		PanStepCurrent = PanDegreeToMotorStep(HOME_PAN_DEGREE);
		//
		TiltDegreeMax = TiltMotorStepToDegree(TiltStepMax);
		TiltDegreeMin = TiltMotorStepToDegree(TiltStepMin);
		PanDegreeMax = PanMotorStepToDegree(PanStepMax);
		PanDegreeMin = PanMotorStepToDegree(PanStepMin);
#endif
	}
	// init swing status data
	memset(&SwingStatus, sizeof(PAN_TILT_COMMAND), 0);
	SwingStatus.Function = PT_FUNC_SWING_STOP;

	printk("pan-tilt: version = (%d)\n", PanTiltFirmwareVersion);
#if ( I2C_DEBUG != 0 )
	printk("pan-tilt: (Pan ) Degree Max=%d, Min=%d; Step Max=0x%X, Min=0x%X\n",
			PanDegreeMax, PanDegreeMin, PanStepMax, PanStepMin);
	printk("pan-tilt: (Tilt) Degree Max=%d, Min=%d; Step Max=0x%X, Min=0x%X\n",
			TiltDegreeMax, TiltDegreeMin, TiltStepMax, TiltStepMin);
	printk("pan-tilt: (Multiplicator) Pan=0x%X, Tilt=0x%X\n", PanStepMultiplicator, TiltStepMultiplicator);
	printk("pan-tilt: PanStepCurrent=0x%X, TiltStepCurrent=0x%X\n",
			PanStepCurrent, TiltStepCurrent);
	printk("pan-tilt: PanStepPerDegree=%d (base=%d), TiltStepPerDegree=%d(base=%d)\n",
			PanStepPerDegree, PanStepPerDegreeBase, TiltStepPerDegree, TiltStepPerDegreeBase);
#endif
	return 0;
}
/*----------------------------------------------------------------------*/

int i2cdrv_ioctl (struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
PAN_TILT_COMMAND	pan_tilt_cmd;
int			ret_value=0;
I2C_READ_WRITE		i2c_access;

	switch ( cmd ) {
	case RT2880_I2C_CONFIG_READ:
		copy_to_user((int __user *)arg, &I2CConfigData, sizeof(I2C_CONFIG));
		break;
	case RT2880_I2C_CONFIG_WRITE:
		copy_from_user(&I2CConfigData, (int __user *)arg, sizeof(I2C_CONFIG));
		I2CMasterInit();
		ret_value = 0;
#if ( I2C_DEBUG != 0 )
		printk("i2cdrv: set CFG=0x%X, CLKDIV=%d, DevAddr=0x%X\n", I2CConfigData.ConfigRegister,
			 	I2CConfigData.ClockDivisor, I2CConfigData.DeviceAddress);
#endif
		break;
	case RT2880_I2C_PAN_TILT:
		copy_from_user(&pan_tilt_cmd, (int __user *)arg, sizeof(PAN_TILT_COMMAND));
		ret_value = PanTiltCommand(&pan_tilt_cmd);
		copy_to_user((int __user *)arg, &pan_tilt_cmd, sizeof(PAN_TILT_COMMAND));
		break;
	case RT2880_I2C_READ:
		copy_from_user(&i2c_access, (int __user *)arg, sizeof(I2C_READ_WRITE));
		ret_value = I2CRead(i2c_access.Address, &i2c_access.Data);
		copy_to_user((int __user *)arg, &i2c_access, sizeof(I2C_READ_WRITE));
		break;
	case RT2880_I2C_WRITE:
		copy_from_user(&i2c_access, (int __user *)arg, sizeof(I2C_READ_WRITE));
		ret_value = I2CWrite(i2c_access.Address, i2c_access.Data);
		copy_to_user((int __user *)arg, &i2c_access, sizeof(I2C_READ_WRITE));
		break;
	default :
		printk("i2c_drv: command format error\n");
	}

	return ret_value;
}

struct file_operations i2cdrv_fops = {
	ioctl:	i2cdrv_ioctl,
};

static int i2cdrv_init(void)
{
u32	reg_data;

	// GPIO-I2C mode
	reg_data = RT2880_REG(RT2880_GPIOMODE_REG);
	reg_data  &= (~(1 << 0));
	RT2880_REG(RT2880_GPIOMODE_REG) = reg_data;

#ifdef  CONFIG_DEVFS_FS
	{
		if(devfs_register_chrdev(i2cdrv_major, I2C_DEV_NAME , &i2cdrv_fops)) {
			printk(KERN_WARNING " i2cdrv: can't create device node\n");
			return -EIO;
		}
		devfs_handle = devfs_register(NULL, I2C_DEV_NAME, DEVFS_FL_DEFAULT, i2cdrv_major, 0, \
				S_IFCHR | S_IRUGO | S_IWUGO, &i2cdrv_fops, NULL);
	}
#else
	{
	int result=0;
		result = register_chrdev(i2cdrv_major, I2C_DEV_NAME, &i2cdrv_fops);
		if (result < 0) {
			printk(KERN_WARNING "i2c_drv: can't get major %d\n",i2cdrv_major);
			return result;
		}
		if (i2cdrv_major == 0) {
			i2cdrv_major = result; /* dynamic */
		}
	}
#endif
	I2CConfigData.ConfigRegister = I2C_CFG_DEFAULT;
	I2CConfigData.ClockDivisor = CLKDIV_VALUE;
	I2CConfigData.DeviceAddress = PAN_TILT_DEVICE_ADDRESS;
	I2CMasterInit();
	printk("i2cdrv_major = %d, CFG=0x%X, CLKDIV=%d, DevAddr=0x%X\n", i2cdrv_major,
		I2CConfigData.ConfigRegister, I2CConfigData.ClockDivisor, I2CConfigData.DeviceAddress);
	PanTiltInit();
	return 0;
}



static void i2cdrv_exit(void)
{
	printk("i2c_drv exit\n");
#ifdef  CONFIG_DEVFS_FS
	devfs_unregister_chrdev(i2cdrv_major, I2C_DEV_NAME);
	devfs_unregister(devfs_handle);
#else
	unregister_chrdev(i2cdrv_major, I2C_DEV_NAME);
#endif
}

module_init(i2cdrv_init);
module_exit(i2cdrv_exit);

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,12)
MODULE_PARM (i2cdrv_major, "i");
#else
module_param (i2cdrv_major, int, 0);
#endif

MODULE_DESCRIPTION("I2C for Pan-Tilt Driver");
#endif	//...( defined(MODEL_TVIP651W) || defined(MODEL_TVIP651WI) )

