/* File: FRAME.C */

#include <linux/spinlock.h> 
#include <linux/jiffies.h>

#include "frame.h"

#define IMAGE_TEST	0

#if IMAGE_TEST
#include "image.h"
#endif

char VideoBuffer[1024L * 1024L];   //reserve 1Mbytes for frame buffer

struct FRM *pFRM_EMPTY_LIST_Head;
struct FRM *pFRM_READY_LIST_Head;
struct FRM *pFRM_LOCK_LIST_Head;
struct FRM *pFRM_READY_LIST_Tail;
struct FRM *pFRM_LOCK_LIST_Tail;

short lock_internet_user = 0;
unsigned long seq_number = 0;
short Ready_List_Cnt = 0;
short Max_Ready_List_Cnt = 30;
short CameraDisable = 0;
short On_Frame_Init = 0;
short LockFrameNum = 0;

extern short OnFlashWrite;
extern unsigned long FrameBufferSize;
extern unsigned long FrameVideoSize;
extern short Resolution;
extern short Compression;
extern short FrameRate;
extern short TotalVideoUser;

extern void Send_User2_Sig_Restart(void);
short Clear_No_Image_Cnt(void);

spinlock_t mr_lock1 = SPIN_LOCK_UNLOCKED;

short Frame_Empty_List_Queue(struct FRM *pFRM)
{
    unsigned long flags;

    if(CameraDisable)
    {
        if(On_Frame_Init)
            ; /* Pass if doing frame init */
        else
	    return -1;
    }

    spin_lock_irqsave(&mr_lock1, flags);

    pFRM->user_type = 0;
    pFRM->lock_count = 0;
    pFRM->max_lock_count = 0;
    pFRM->frame_flag = 0;
    pFRM->sequence_num = 0xffffffff;
    pFRM->frame_len = 0;

    pFRM->link_down = pFRM_EMPTY_LIST_Head;
    pFRM_EMPTY_LIST_Head = pFRM;

    spin_unlock_irqrestore(&mr_lock1, flags);

    return 0;
}


struct FRM *Frame_Empty_List_DeQueue(void)
{
    unsigned long flags;
    struct FRM *pFRM;

    if(CameraDisable)
	return 0;
    if(OnFlashWrite)
	return 0;

    spin_lock_irqsave(&mr_lock1, flags);

    pFRM = pFRM_EMPTY_LIST_Head;
    if(pFRM_EMPTY_LIST_Head)
	pFRM_EMPTY_LIST_Head = pFRM->link_down;

    spin_unlock_irqrestore(&mr_lock1, flags);

    return pFRM;
}


short Frame_Ready_List_Queue(struct FRM *pFRM)
{
    unsigned long flags;
    struct FRM *pTmpFRM;

    if(CameraDisable)
	return -1;

    Clear_No_Image_Cnt();

    spin_lock_irqsave(&mr_lock1, flags);

    seq_number++;
    if(seq_number == 0xfffffff0L)		//Reserved 0xfffffff0 - 0xffffffff for special use
	seq_number = 0;
    pFRM->sequence_num = seq_number;

    FrameVideoSize = pFRM->frame_len;

    pTmpFRM = pFRM_READY_LIST_Head;

    pFRM->link_down = pFRM_READY_LIST_Head;
    pFRM_READY_LIST_Head = pFRM;

    pFRM->link_up = 0;
    if(pTmpFRM)
	pTmpFRM->link_up = pFRM;
    else
	pFRM_READY_LIST_Tail = pFRM;

    Ready_List_Cnt++;

#if DEBUG_MOTION_DETECTION
if(pFRM->frame_flag & IS_MOTION_FRAME)
{
   printk("<Queue Motion Frame, Sequence # = %d>\n", pFRM->sequence_num);
}
#endif

    spin_unlock_irqrestore(&mr_lock1, flags);

    return 0;
}


struct FRM *Frame_Ready_List_DeQueue(void)
{
    unsigned long flags;
    struct FRM *pFRM;

    if(CameraDisable)
	return 0;
    if(OnFlashWrite)
	return 0;

    while(1)
    {
        spin_lock_irqsave(&mr_lock1, flags);

	pFRM = pFRM_READY_LIST_Tail;
	if(pFRM_READY_LIST_Tail)
	    pFRM_READY_LIST_Tail = pFRM->link_up;
	if(pFRM_READY_LIST_Tail)
	    pFRM_READY_LIST_Tail->link_down = 0;
	else
	    pFRM_READY_LIST_Head = 0;

        spin_unlock_irqrestore(&mr_lock1, flags);

	if(Ready_List_Cnt)
	    Ready_List_Cnt--;

	if((pFRM) && (Ready_List_Cnt > Max_Ready_List_Cnt))  /* 2004-06-01 */
	{
	    Frame_Empty_List_Queue(pFRM);
	    continue;
	}

	break;
    }

    return pFRM;
}


struct FRM *Get_Motion_Frame(unsigned long sequence_num)
{
    unsigned long flags;
    struct FRM *pFRM;

    if(CameraDisable)
	return 0;
    if(OnFlashWrite)
	return 0;

    spin_lock_irqsave(&mr_lock1, flags);

    pFRM = pFRM_READY_LIST_Tail;
    if(pFRM == 0)
    {
        spin_unlock_irqrestore(&mr_lock1, flags);
        return 0;
    }

    while(1)
    {
        if((pFRM->sequence_num > sequence_num) && (pFRM->frame_flag & IS_MOTION_FRAME))
        {
            spin_unlock_irqrestore(&mr_lock1, flags);
#if DEBUG_MOTION_DETECTION
            printk("[Got Motion Frame, Sequence # = %d OOOOOOOOOOOOOOOOOOOOO]\n", pFRM->sequence_num);
#endif
            return pFRM;
        }

        if(pFRM->link_up == 0)
        {
            if(pFRM->sequence_num > sequence_num)
            {
                spin_unlock_irqrestore(&mr_lock1, flags);
#if DEBUG_MOTION_DETECTION
                printk("[Got Non-Motion Frame, Sequence # = %d oooooooooooooooooooo]\n", pFRM->sequence_num);
#endif
                return pFRM;
            }
            else
            {
                spin_unlock_irqrestore(&mr_lock1, flags);
                return 0;
            }
        }

        pFRM = pFRM->link_up;
    }
}


short Frame_Lock_List_Queue(struct FRM *pFRM)
{
    unsigned long flags;
    struct FRM *pTmpFRM;

    if(CameraDisable)
	return -1;

    spin_lock_irqsave(&mr_lock1, flags);

    pFRM->lock_count = 1;
    pFRM->max_lock_count = 1;
    pFRM->free_time = jiffies + 24;  //1/10 second

    LockFrameNum++;

    pTmpFRM = pFRM_LOCK_LIST_Head;

    pFRM->link_down = pFRM_LOCK_LIST_Head;
    pFRM_LOCK_LIST_Head = pFRM;

    pFRM->link_up = 0;
    if(pTmpFRM)
	pTmpFRM->link_up = pFRM;
    else
	pFRM_LOCK_LIST_Tail = pFRM;

    spin_unlock_irqrestore(&mr_lock1, flags);

    return 0;
}


/* Return Code: 0 - OK			   */
/*		1 - Still on Lock	   */
/*		2 - Not on Lock List	   */
/*		3 - Error Frame Address    */
/*             -1 - Camera is Disable      */
short Frame_Lock_List_DeQueue(struct FRM *pFRM)
{
    unsigned long flags;
    struct FRM *pPrevFRM;
    struct FRM *pNextFRM;

    if(CameraDisable)
	return -1;

    spin_lock_irqsave(&mr_lock1, flags);

    if(pFRM_LOCK_LIST_Head == 0)
    {
        spin_unlock_irqrestore(&mr_lock1, flags);

	return 2;
    }

    if(pFRM->lock_count == 0)
    {
        spin_unlock_irqrestore(&mr_lock1, flags);

	return 3;
    }

    pFRM->lock_count--;
    if(pFRM->lock_count)
    {
        spin_unlock_irqrestore(&mr_lock1, flags);

	return 1;		/* Still on Lock */
    }

    if((pFRM->free_time > jiffies) && (pFRM->max_lock_count < TotalVideoUser))
    {
        spin_unlock_irqrestore(&mr_lock1, flags);

	return 1;		/* Still on Lock */
    }

//printk("max_lock_count = %d, jiffies = %d\n", pFRM->max_lock_count, jiffies);
//printk("[%d]\n", jiffies);

    /* Need DeQueue */
    pPrevFRM = pFRM->link_up;
    pNextFRM = pFRM->link_down;

    if(pPrevFRM)
	pPrevFRM->link_down = pFRM->link_down;
    else
	pFRM_LOCK_LIST_Head = pFRM->link_down;

    if(pNextFRM)
	pNextFRM->link_up = pFRM->link_up;
    else
	pFRM_LOCK_LIST_Tail = pFRM->link_up;

    LockFrameNum--;

    spin_unlock_irqrestore(&mr_lock1, flags);

    return 0;
}


struct FRM *Frame_Lock_List_Newest(unsigned long sequence_num)
{
    unsigned long flags;
    struct FRM *pFRM;

    if(CameraDisable)
	return 0;

    spin_lock_irqsave(&mr_lock1, flags);

    pFRM = pFRM_LOCK_LIST_Tail;
    if(pFRM)
    {
	if(pFRM->sequence_num == sequence_num)
	{
	    pFRM = 0;
	}
	else if(pFRM->sequence_num > sequence_num)
	{
	    if((pFRM->sequence_num - sequence_num) > 0x00000100L)
		pFRM = 0;
	}
	else if(pFRM->sequence_num < sequence_num)
	{
	    pFRM = 0;
	}

	if(pFRM)
        {
	    pFRM->lock_count++;
	    pFRM->max_lock_count++;

            spin_unlock_irqrestore(&mr_lock1, flags);

            return pFRM;
        }
    }

    pFRM = pFRM_LOCK_LIST_Head;
    if(pFRM)
    {
	if(pFRM->sequence_num == sequence_num)
	{
	    pFRM = 0;
	}
	else if(pFRM->sequence_num > sequence_num)
	{
	    if((pFRM->sequence_num - sequence_num) > 0x00000100L)
		pFRM = 0;
	}
	else if(pFRM->sequence_num < sequence_num)
	{
	    pFRM = 0;
	}

	if(pFRM)
        {
	    pFRM->lock_count++;
	    pFRM->max_lock_count++;
        }
    }

    spin_unlock_irqrestore(&mr_lock1, flags);

    return pFRM;
}


struct FRM *Frame_Lock_List_InternetNewest(unsigned long sequence_num)
{
    unsigned long flags;
    struct FRM *pFRM;
    short no_internet_frame = 1;

    if(CameraDisable)
	return 0;

//  spin_lock_irqsave(&mr_lock1, flags);  //need not cli protect

    pFRM = pFRM_LOCK_LIST_Head;
    while(pFRM)
    {
	if(pFRM->user_type != USER_TYPE_INTERNET)
	{
	    pFRM = pFRM->link_down;
	    continue;
	}

	no_internet_frame = 0;

	if(pFRM->sequence_num == sequence_num)
	    pFRM = 0;
	else if(pFRM->sequence_num > sequence_num)
	{
	    if((pFRM->sequence_num - sequence_num) > 0x00000100L)
		pFRM = 0;
	}
	else
	{
	    pFRM = 0;
	}

	if(pFRM)
        {
	    pFRM->lock_count++;
	    pFRM->max_lock_count++;
        }

	break;
    }

    if(no_internet_frame)
	pFRM = (struct FRM *)1; 	      //No internet frame on lock list

//  spin_unlock_irqrestore(&mr_lock1, flags);

    return pFRM;
}

void Release_Timeout_Lock_Frame(void)
{
    unsigned long flags;
    struct FRM *pPrevFRM;
    struct FRM *pNextFRM;
    struct FRM *pFRM, *pTmpFRM;

    if(CameraDisable)
	return;

    pFRM = pFRM_LOCK_LIST_Tail;
    while(pFRM)
    {
        if(pFRM->free_time > jiffies)
            break;

        if((pFRM->lock_count == 0) && (jiffies >= pFRM->free_time))
        {
            spin_lock_irqsave(&mr_lock1, flags);

            /* Need DeQueue */
            pPrevFRM = pFRM->link_up;
            pNextFRM = pFRM->link_down;

            if(pPrevFRM)
        	pPrevFRM->link_down = pFRM->link_down;
            else
        	pFRM_LOCK_LIST_Head = pFRM->link_down;

            if(pNextFRM)
        	pNextFRM->link_up = pFRM->link_up;
            else
        	pFRM_LOCK_LIST_Tail = pFRM->link_up;

            LockFrameNum--;

            spin_unlock_irqrestore(&mr_lock1, flags);

            Frame_Empty_List_Queue(pFRM);

            pFRM = pPrevFRM;
        }
#if 0   //Inactive User will be removed with 30 seconds -- do at ralink_gpio.c ActiveUser_KeepAlive()
        else if((pFRM->lock_count != 0) && ((jiffies - pFRM->free_time) >= 24*10*6))  //24*10*6 -> 6 seconds -- some pc disconnected
        {
            spin_lock_irqsave(&mr_lock1, flags);

            /* Need DeQueue */
            pPrevFRM = pFRM->link_up;
            pNextFRM = pFRM->link_down;

            if(pPrevFRM)
        	pPrevFRM->link_down = pFRM->link_down;
            else
        	pFRM_LOCK_LIST_Head = pFRM->link_down;

            if(pNextFRM)
        	pNextFRM->link_up = pFRM->link_up;
            else
        	pFRM_LOCK_LIST_Tail = pFRM->link_up;

            LockFrameNum--;

            spin_unlock_irqrestore(&mr_lock1, flags);

            Frame_Empty_List_Queue(pFRM);

            pFRM = pPrevFRM;
        }
#endif
        else 
        {
            pFRM = pFRM->link_up;
        }
    }

    return;
}

void Release_All_Lock_Frame(void)
{
    unsigned long flags;
    struct FRM *pPrevFRM;
    struct FRM *pNextFRM;
    struct FRM *pFRM;
    short total_lock_frame = 0;

    if(CameraDisable)
	return;

    while(pFRM = pFRM_LOCK_LIST_Tail)
    {
        spin_lock_irqsave(&mr_lock1, flags);

        /* Need DeQueue */
        pPrevFRM = pFRM->link_up;
        pNextFRM = pFRM->link_down;

        if(pPrevFRM)
       	    pPrevFRM->link_down = pFRM->link_down;
        else
            pFRM_LOCK_LIST_Head = pFRM->link_down;

        if(pNextFRM)
       	    pNextFRM->link_up = pFRM->link_up;
        else
            pFRM_LOCK_LIST_Tail = pFRM->link_up;

        spin_unlock_irqrestore(&mr_lock1, flags);

        printk("max lock cnt = %d, lock cnt = %d, timeout = %d\n", pFRM->max_lock_count, pFRM->lock_count, pFRM->free_time);

        Frame_Empty_List_Queue(pFRM);

        total_lock_frame++;
    }

    LockFrameNum = 0;

    printk("total lock frame = %d\n", total_lock_frame);

    return;
}


/*****************************************************************************/
short No_Image_Cnt = 0;
unsigned long No_Image_Time = 0xffffffffL;

short Clear_No_Image_Cnt(void)
{
    No_Image_Cnt = 0;
    No_Image_Time = 0xffffffffL;

//  printk("c");

    return 0;
}

short Get_Image_OK(void)
{
    No_Image_Cnt = 0;
    No_Image_Time = 0xffffffffL;

//  printk("o");

    return 0;
}

short Get_Image_Fail(void)
{
    unsigned long Current_Time;
    unsigned long flags;

//  printk("x");

    if(No_Image_Cnt == 0)
    {
	No_Image_Cnt++;
	No_Image_Time = jiffies/100;  //jiffies is 1/100 second
    }
    else
    {
	No_Image_Cnt++;
	Current_Time = jiffies/100;
	if(Current_Time < No_Image_Time)
	    No_Image_Time = Current_Time;

	if((Current_Time - No_Image_Time) >= (60 * 3))   /* 3 minutes */
        {
            printk("Reboot System: No Image\n");

            Send_User2_Sig_Restart();   
            return 0;
         }
    }

    return 0;
}

/*****************************************************************************/

unsigned long Frame_List_Init(void)
{
    short frame_size_index[3][5];
    unsigned long frame_size[6];
    unsigned long FRMAddr;
    struct FRM *pFRM;
    short framecnt;
    short i;

    CameraDisable = 1;
    On_Frame_Init = 1;
    FrameBufferSize = ((unsigned long)1024L * (unsigned long)72L) - sizeof(struct FRM); /* Default Max. */ 
   
    frame_size[0]  = ((unsigned long)1024L * (unsigned long)72L);      /* 1  */
    frame_size[1]  = ((unsigned long)1024L * (unsigned long)64L);      /* 2  */
    frame_size[2]  = ((unsigned long)1024L * (unsigned long)32L);      /* 3  */
    
    frame_size_index[2][0] = 0; 	/* VGA - Compress1 (72K) */
    frame_size_index[2][1] = 0; 	/* VGA - Compress2 (72K) */
    frame_size_index[2][2] = 0; 	/* VGA - Compress3 (72K) */
    frame_size_index[2][3] = 1; 	/* VGA - Compress4 (64K) */
    frame_size_index[2][4] = 1; 	/* VGA - Compress5 (64K) */

    frame_size_index[1][0] = 1; 	/* CIF - Compress1 (64K) */
    frame_size_index[1][1] = 1; 	/* CIF - Compress2 (64K) */
    frame_size_index[1][2] = 1; 	/* CIF - Compress3 (64K) */
    frame_size_index[1][3] = 1; 	/* CIF - Compress4 (64K) */
    frame_size_index[1][4] = 1; 	/* CIF - Compress5 (64k) */

    frame_size_index[0][0] = 1; 	/* QCIF - Compress1 (64K) */
    frame_size_index[0][1] = 1; 	/* QCIF - Compress2 (64K) */
    frame_size_index[0][2] = 1; 	/* QCIF - Compress3 (64K) */
    frame_size_index[0][3] = 2; 	/* QCIF - Compress4 (32K) */
    frame_size_index[0][4] = 2; 	/* QCIF - Compress5 (32K) */

#if IMAGE_TEST
    Resolution = 1;
    Compression = 0;
#endif
    FrameBufferSize = frame_size[frame_size_index[Resolution][Compression]];

    pFRM_EMPTY_LIST_Head = 0;
    pFRM_READY_LIST_Head = 0;
    pFRM_LOCK_LIST_Head = 0;
    pFRM_READY_LIST_Tail = 0;
    pFRM_LOCK_LIST_Tail = 0;

    Ready_List_Cnt = 0;
    if(FrameRate == 0)
	FrameRate = 30;
    if(FrameRate < 5)
	FrameRate = 5;
    Max_Ready_List_Cnt = FrameRate / 2;

    if(Resolution == 2) //VGA mode
    {
	if(Max_Ready_List_Cnt >= 6)
	{
	    Max_Ready_List_Cnt = 6;	//To solve Latcy problem
//#if    WIRELESS_INCLUDED
//	    Max_Ready_List_Cnt = 3;	//To solve Latcy problem
//#endif
	}
    }

//  printk("[MaxCnt = %x]", Max_Ready_List_Cnt);

    framecnt = (short)(1024 / (FrameBufferSize / 1024));

    FRMAddr = (unsigned long)VideoBuffer;
    pFRM = (struct FRM *)FRMAddr;

    for(i = 0; i < framecnt; i++)
    {
        pFRM->my_buffer_addr = FRMAddr;
        Frame_Empty_List_Queue(pFRM);
	FRMAddr += FrameBufferSize;
        pFRM = (struct FRM *)FRMAddr;
    }

    FrameBufferSize -= sizeof(struct FRM);

    CameraDisable = 0;
    On_Frame_Init = 0;
    LockFrameNum = 0;

    Clear_No_Image_Cnt();	//andy

    return FrameBufferSize;
}


unsigned long Frame_Get_Size(void)
{
    return FrameBufferSize;
}

/*************************************************************************************/
struct FRM *Frame_Get_Image(unsigned long prev_sequence_num, unsigned short ticks_interval, unsigned short flag)
{
//  struct IMG *pIMG;
//  char *pAddr;
    struct FRM *pFRM;
    long trycnt;

    if(CameraDisable)
	return 0;

    if(jiffies <= 250*10)	//Wait AWB OK //10 seconds
    {
	Frame_Clear();
	return 0;
    }

    Release_Timeout_Lock_Frame();

    if(flag & GET_MOTION_FRAME)		//Motion Frame request
    {
#if DEBUG_MOTION_DETECTION
printk("prev_sequence_num = %x\n", prev_sequence_num);
#endif

        if(prev_sequence_num >= 0xfffffff0L)
        {
            prev_sequence_num = 0;
        }

	pFRM = Get_Motion_Frame(prev_sequence_num);

	if(pFRM)
	{
	    pFRM->user_type = USER_TYPE_LOCAL;
	    Get_Image_OK();
	    return pFRM;
	}
        else
        {
#if DEBUG_MOTION_DETECTION
            printk("[No Motion Frame XXXXXXXXXXXXXXXXXXXXX]\n");
#endif
            return (struct FRM *)0;
        }
    }

    if(flag)
    {
#if DEBUG_MOTION_DETECTION
printk("Get Frame - frame_flag need to clear to 0 -- ????????????????????\n");
#endif
    }

//  if(ticks_interval >= 18)	/* It is a Internet Users */
    if(0)			/* Not support this case -- andy 2010-04-09 */
    {
	if(lock_internet_user)
	{
	    /* Get Newest Internet Lock Frame from Lock List */
	    pFRM = Frame_Lock_List_InternetNewest(prev_sequence_num);

	    if(pFRM)
	    {
		if(pFRM == (struct FRM *)1)   /* No Internet Frame on Lock List */
		{
		    lock_internet_user = 0;

		    Get_Image_Fail();
		    return (struct FRM *)0;
		}

		pFRM->user_type = USER_TYPE_INTERNET;

//		pAddr = (char *)pFRM;
//		pAddr += (sizeof(struct FRM) - sizeof(struct IMG));
//		pIMG = (struct IMG *)pAddr;

		Get_Image_OK();
		return pFRM;
	    }
            else
            {
	        lock_internet_user = 0;
            }
	}
	else
	{
	    /* Get one Frame from Ready List */
	    pFRM = Frame_Ready_List_DeQueue();

	    if(pFRM)
	    {
		pFRM->user_type = USER_TYPE_INTERNET;
		Frame_Lock_List_Queue(pFRM);
//		pAddr = (char *)pFRM;
//		pAddr += (sizeof(struct FRM) - sizeof(struct IMG));
//		pIMG = (struct IMG *)pAddr;

		lock_internet_user = 1;

		Get_Image_OK();
		return pFRM;
	    }
#if 0	    //andy, 2009-11-26
	    else
	    {
		lock_internet_user = 1;
	    }
#endif

	    /* Get Newest Frame from Lock List */
	    pFRM = Frame_Lock_List_Newest(prev_sequence_num);

	    if(pFRM)
	    {
		pFRM->user_type = USER_TYPE_INTERNET;
		lock_internet_user = 1;

//		pAddr = (char *)pFRM;
//		pAddr += (sizeof(struct FRM) - sizeof(struct IMG));
//		pIMG = (struct IMG *)pAddr;

		Get_Image_OK();
		return pFRM;
	    }
	}
    }
    else			/* It is a Local Users */
    {
#if 1
	/* Get Newest Frame from Lock List */
	pFRM = Frame_Lock_List_Newest(prev_sequence_num);

	if(pFRM)
	{
	    pFRM->user_type = USER_TYPE_LOCAL;

//	    pAddr = (char *)pFRM;
//	    pAddr += (sizeof(struct FRM) - sizeof(struct IMG));
//	    pIMG = (struct IMG *)pAddr;

//	    lock_internet_user = 0;

	    Get_Image_OK();
	    return pFRM;
	}
#endif
	/* Get one Frame from Ready List */
        if(LockFrameNum < 2)			//wait some PC that got low frame number
	    pFRM = Frame_Ready_List_DeQueue();

	if(pFRM)
	{
	    pFRM->user_type = USER_TYPE_LOCAL;
	    Frame_Lock_List_Queue(pFRM);
//	    pAddr = (char *)pFRM;
//	    pAddr += (sizeof(struct FRM) - sizeof(struct IMG));
//	    pIMG = (struct IMG *)pAddr;

//	    lock_internet_user = 0;

	    Get_Image_OK();
	    return pFRM;
	}
#if 0
	else
	{
	    lock_internet_user = 1;
	}
#endif

	if(prev_sequence_num == 0xffffffff)   /* This is for smoe Single Frame Capture on some Application */
	{
	    trycnt = 204800;
	    while(trycnt--)
	    {
		if(pFRM = Frame_Ready_List_DeQueue())
		    break;

        	schedule();
	    }

	    if(pFRM)
	    {

//              printk("[********************]");

		pFRM->user_type = USER_TYPE_LOCAL;
		Frame_Lock_List_Queue(pFRM);
//		pAddr = (char *)pFRM;
//		pAddr += (sizeof(struct FRM) - sizeof(struct IMG));
//		pIMG = (struct IMG *)pAddr;

//		lock_internet_user = 0;

		Get_Image_OK();
		return pFRM;
	    }

//          printk("[????????????????????]");

	}

#if 0
	/* Get Newest Frame from Lock List */
	pFRM = Frame_Lock_List_Newest(prev_sequence_num);

	if(pFRM)
	{
//	    pAddr = (char *)pFRM;
//	    pAddr += (sizeof(struct FRM) - sizeof(struct IMG));
//	    pIMG = (struct IMG *)pAddr;

	    return pFRM;
	}
#endif
    }

    Get_Image_Fail();
    return (struct FRM *)0;
}


/* Return Code: 0 - OK					  */
/*		1 - Error Input 			  */
/*		2 - pIMG not on Lock List		  */
/*		3 - Error Frame Address 		  */
/*             -1 - Camera Disable                        */
short Frame_Release_Image(struct FRM *pFRM)
{
//  struct FRM *pFRM;
//  char *pAddr;
    short de_queue_sts;

    if(CameraDisable)
	return -1;

//  pAddr = (char *)pIMG;
//  pAddr -= (sizeof(struct FRM) - sizeof(struct IMG));
//  pFRM = (struct FRM *)pAddr;

    if(pFRM->frame_flag & RELEASE_MOTION_FRAME)
    {
#if DEBUG_MOTION_DETECTION
        printk("(Free Motion Frame, Sequence # = %x FFFFFFFFFFFFFFFFFFFFF)\n", pFRM->sequence_num); 
#endif
        return 0;
    }

    if(pFRM->frame_flag)
    {
#if DEBUG_MOTION_DETECTION
printk("FREE Frame - frame_flag need to clear to 0 -- ????????????????????\n");
#endif
    }

    if(pFRM->my_buffer_addr != (unsigned long)pFRM)
    {
        printk("Frame_release_image error 1\n");
	return 1;		/* error input */
    }

    if((pFRM->user_type != USER_TYPE_LOCAL) && (pFRM->user_type != USER_TYPE_INTERNET))
    {
        printk("Frame_release_image error 2\n");
	return 1;		/* error input */
    }

    if((pFRM->frame_len == 0) || (pFRM->frame_len > FrameBufferSize))
    {
        printk("Frame_release_image error 3\n");
	return 1;		/* error input */
    }

    de_queue_sts = Frame_Lock_List_DeQueue(pFRM);

    if(de_queue_sts == 0)	/* DeQueue OK */
    {
	Frame_Empty_List_Queue(pFRM);
	return 0;               
    }
    else if(de_queue_sts != 1)	/* 1 - Still on Lock */
    {
        printk("Frame_release_image error 4\n");
	return de_queue_sts;	/* 2 - Frame not on Lock List, 3 - Error Frame Address */
    }

//printk("[on lock]\n");

    return 0;                   /* Still on Lock */
}


short Frame_Clear(void)
{
    unsigned long flags;
    struct FRM *pFRM;

    if(CameraDisable)
	return -1;

//  spin_lock_irqsave(&mr_lock1, flags);

    while(pFRM = Frame_Ready_List_DeQueue())
    {
	Frame_Empty_List_Queue(pFRM);
    }

//  spin_unlock_irqrestore(&mr_lock1, flags);

    return 0;
}

/*****************************************************************************/
#if IMAGE_TEST
unsigned long sequence_number = 0;
short QueueImage()
{
    short i;
    struct FRM *pFRM;
    struct IMG *pDstIMG;
    struct IMG *pSrcIMG;
    unsigned long FrameLen;
    char *pAddr;

    if(CameraDisable)
	return 0;

    pSrcIMG = (struct IMG *)&TestImage;
    FrameLen = pSrcIMG->frame_len;
    FrameLen &= 0x0000ffff;

    if(FrameLen > Frame_Get_Size())
	return 0;

    for(i = 0; i < 8; i++)
    {
	pFRM = Frame_Empty_List_DeQueue();
	if(pFRM)
	{
	    pAddr = (char *)pFRM;
	    pAddr += (sizeof(struct FRM) - sizeof(struct IMG));
	    pDstIMG = (struct IMG *)pAddr;
	    memcpy(pDstIMG, pSrcIMG, (unsigned short)(FrameLen+8));
	    pDstIMG->sequence_num = sequence_number;
	    sequence_number++;

	    Frame_Ready_List_Queue(pFRM);
	}
	else
	{
	    pFRM = Frame_Ready_List_DeQueue();
	    if(pFRM)
	    {
		pAddr = (char *)pFRM;
		pAddr += (sizeof(struct FRM) - sizeof(struct IMG));
		pDstIMG = (struct IMG *)pAddr;
		memcpy(pDstIMG, pSrcIMG, (unsigned short)(FrameLen+8));
		pDstIMG->sequence_num = sequence_number;
		sequence_number++;

		Frame_Ready_List_Queue(pFRM);
	    }
	}
    }

    return 0;
}
#endif

