/*
 * Flash mapping for rtl8196 board
 *
 * Copyright (C) 2008 Realtek Corporation
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/root_dev.h>
#include <linux/mtd/partitions.h>
#include <linux/config.h>
#include <linux/delay.h>
#include <linux/autoconf.h>

#ifdef CONFIG_MTD_CONCAT
#include <linux/mtd/concat.h>
#endif

#include "sn_flash_map.h"
#if DYNAMIC_UDPATE_PARTITION_TBL
#include "../chips/rtl819x/spi_flash.h"
/* Firmware image header, copy from bootloader hdr definition(boot/init/rtk.h) */
typedef struct _rtl_header_ {
	unsigned char signature[4];
	unsigned long startAddr;
	unsigned long burnAddr;
	unsigned long len;	
} RTL_IMG_HEADER_T;
#endif

#define WINDOW_ADDR 0xbfe00000
#ifdef CONFIG_SPANSION_16M_FLASH	
#define WINDOW_SIZE 0x1000000
#define FLASH_BANK_SIZE 0x400000
#else 
#define WINDOW_SIZE 0x400000
#endif 
#define BUSWIDTH 2

#ifdef CONFIG_RTL_TWO_SPI_FLASH_ENABLE
#define MAX_SPI_CS 2		/* Number of CS we are going to test */
#else
#define MAX_SPI_CS 1
#endif



static struct mtd_info *rtl8196_mtd;

__u8 rtl8196_map_read8(struct map_info *map, unsigned long ofs)
{
	//printk("enter %s %d\n",__FILE__,__LINE__);
	return __raw_readb(map->map_priv_1 + ofs);
}

__u16 rtl8196_map_read16(struct map_info *map, unsigned long ofs)
{
	//printk("enter %s %d\n",__FILE__,__LINE__);
	return __raw_readw(map->map_priv_1 + ofs);
}

__u32 rtl8196_map_read32(struct map_info *map, unsigned long ofs)
{
	//printk("enter %s %d\n",__FILE__,__LINE__);
	return __raw_readl(map->map_priv_1 + ofs);
}

void rtl8196_map_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	//printk("enter to %x from  %x len %d\n",to, map->map_priv_1+from , len);
	//11/15/05' hrchen, change the size to fit file systems block size if use different fs
	//4096 for cramfs, 1024 for squashfs
	if (from>0x10000)
	    memcpy(to, map->map_priv_1 + from, (len<=1024)?len:1024);//len);
	else
	    memcpy(to, map->map_priv_1 + from, (len<=4096)?len:4096);//len);
	//printk("enter %s %d\n", __FILE__,__LINE__);

}

void rtl8196_map_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	__raw_writeb(d, map->map_priv_1 + adr);
	mb();
}

void rtl8196_map_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	__raw_writew(d, map->map_priv_1 + adr);
	mb();
}

void rtl8196_map_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	__raw_writel(d, map->map_priv_1 + adr);
	mb();
}

void rtl8196_map_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	//printk("enter %s %d\n",__FILE__,__LINE__);
	memcpy_toio(map->map_priv_1 + to, from, len);
}

static struct map_info spi_map[MAX_SPI_CS] = {
        {
			name: 		"flash_bank_1",
#ifdef CONFIG_RTL_TWO_SPI_FLASH_ENABLE
			size: 		CONFIG_RTL_SPI_FLASH1_SIZE,
#else
			size:			WINDOW_SIZE,
#endif
			phys: 		0xbd000000,
			virt: 			0xbd000000,
			bankwidth: 	BUSWIDTH

        },
#ifdef CONFIG_RTL_TWO_SPI_FLASH_ENABLE
        {
			name: 		"flash_bank_2",
			size: 		CONFIG_RTL_SPI_FLASH2_SIZE,
			phys: 		0xbe000000,
			virt: 			0xbe000000,
			bankwidth: 	BUSWIDTH
        }
#endif
};

static struct mtd_info *my_sub_mtd[MAX_SPI_CS] = {
	NULL,
#ifdef CONFIG_RTL_TWO_SPI_FLASH_ENABLE
	NULL
#endif
};

static struct mtd_info *mymtd;


#if DYNAMIC_UDPATE_PARTITION_TBL
static void sn_update_partition_tbl(struct spi_chip_info *chip_info, int erasesize)
{
	int i, knl_idx, cb_app_idx;
	char flash_hdr[64];
	RTL_IMG_HEADER_T *pt_rtl_hdr;

	/*Find kernel/cb-app index first*/
	for (i=0; i<ARRAY_SIZE(physmap_partitions_t); i++)
	{
#if COMBINED_FW_IMAGE_TYPE == 2
		if (!strncasecmp(physmap_partitions_t[i].name, "app", sizeof("app")))
#else
		if (!strncasecmp(physmap_partitions_t[i].name, "kernel", sizeof("kernel")))
#endif
		{
			knl_idx = i;
		}

#if COMBINED_FW_IMAGE_TYPE == 2
		if (!strncasecmp(physmap_partitions_t[i].name, "cb-app", sizeof("cb-app")))
#else
		if (!strncasecmp(physmap_partitions_t[i].name, "app", sizeof("app")))
#endif
		{
			cb_app_idx = i;
		}
	}
	
	chip_info->read((unsigned int)(physmap_partitions_t[knl_idx].offset), (unsigned int)flash_hdr, 64, chip_info->chip_select);
	pt_rtl_hdr = (RTL_IMG_HEADER_T *)flash_hdr;	


	// kernel real size
	// 00000000  27 05 19 56 6c cf 84 2b  4e 4d da 8f 00 13 5f fb  |'..Vl..+NM...._.|
	// 00000010  80 00 00 00 80 2c e0 00  1d b9 ba 90 05 05 02 03  |.....,..........|
	// 00000020  4c 69 6e 75 78 20 4b 65  72 6e 65 6c 20 49 6d 61  |Linux Kernel Ima|
	// 00000030  67 65 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |ge..............|
	// 00000040  5d 00 00 00 02 35 65 33  00 00 00 00 00 00 00 6f  |]....5e3.......o|
	// 0x40 is header size.
	pt_rtl_hdr->len += 0x40;

	//printk("sn_update_partition_tbl, knl_size=%x\n", pt_rtl_hdr->len);
	pt_rtl_hdr->len += (erasesize-(pt_rtl_hdr->len%erasesize));
	//printk("padding=%x\n", pt_rtl_hdr->len);

	/*New offset of app = kernel offset + real kernel size*/
	/*New size of app = logical kernel size - real kernel size*/
	physmap_partitions_t[cb_app_idx].offset = physmap_partitions_t[knl_idx].offset + pt_rtl_hdr->len;
	physmap_partitions_t[cb_app_idx].size = physmap_partitions_t[knl_idx].size - pt_rtl_hdr->len;
}
#endif

/////////////////////////////////////////////////////////////////////////////

#ifdef CONFIG_RTL_FLASH_MAPPING_ENABLE
#ifdef USER_DEFINITION_FLASH_MAP
#else
#ifndef CONFIG_RTL_FLASH_DUAL_IMAGE_ENABLE
static struct mtd_partition rtl8196_parts1[] = {
        {
                name: "boot+cfg+linux",
                size:           (CONFIG_RTL_ROOT_IMAGE_OFFSET-0),
                offset:         0x00000000,
        },
        {
                name:           "root fs",                
#ifdef CONFIG_RTL_TWO_SPI_FLASH_ENABLE
#ifdef CONFIG_MTD_CONCAT
                size:        (CONFIG_RTL_SPI_FLASH1_SIZE+CONFIG_RTL_SPI_FLASH2_SIZE-CONFIG_RTL_ROOT_IMAGE_OFFSET),
#else
		  size:        (CONFIG_RTL_SPI_FLASH1_SIZE-CONFIG_RTL_ROOT_IMAGE_OFFSET),
#endif
#else
		  size:	  (WINDOW_SIZE - CONFIG_RTL_ROOT_IMAGE_OFFSET),
#endif
                offset:         (CONFIG_RTL_ROOT_IMAGE_OFFSET),
        }
};
#else
static struct mtd_partition rtl8196_parts1[] = {
        {
                name: "boot+cfg+linux(bank1)",
                size:           (CONFIG_RTL_ROOT_IMAGE_OFFSET-0),
                offset:         0x00000000,
        },
        {
                name:           "root fs(bank1)",                
                size:        (CONFIG_RTL_FLASH_SIZE-CONFIG_RTL_ROOT_IMAGE_OFFSET),
                offset:         (CONFIG_RTL_ROOT_IMAGE_OFFSET),
        },
        {
                name: "inux(bank2)",
                size:           (CONFIG_RTL_ROOT_IMAGE_OFFSET-0),
                offset:         CONFIG_RTL_FLASH_DUAL_IMAGE_OFFSET,
        },
        {
                name:           "root fs(bank2)",                
                size:        (CONFIG_RTL_FLASH_SIZE-CONFIG_RTL_ROOT_IMAGE_OFFSET),
                offset:         CONFIG_RTL_FLASH_DUAL_IMAGE_OFFSET+(CONFIG_RTL_ROOT_IMAGE_OFFSET),
        }

};

#endif
#endif /*USER_DEFINITION_FLASH_MAP*/
#else
static struct mtd_partition rtl8196_parts1[] = {
        {
                name: "boot+cfg+linux",
                size:  0x00130000,
                offset:0x00000000,
        },
        {
                name:           "root fs",                
		   		size:        	0x002D0000,
                offset:         0x00130000,
        }
};
#endif
#ifdef CONFIG_RTL_TWO_SPI_FLASH_ENABLE
static struct mtd_partition rtl8196_parts2[] = {
        {
                name: 		"data",
                size:  		CONFIG_RTL_SPI_FLASH2_SIZE,
                offset:		0x00000000,
        }
};
#endif

#if LINUX_VERSION_CODE < 0x20212 && defined(MODULE)
#define init_rtl8196_map init_module
#define cleanup_rtl8196_map cleanup_module
#endif

#define mod_init_t  static int __init
#define mod_exit_t  static void __exit

mod_init_t init_rtl8196_map(void)
{
        int i,chips;
   	chips = 0;
	for (i=0;i<MAX_SPI_CS;i++) {
		simple_map_init(&spi_map[i]);
		my_sub_mtd[i] = do_map_probe(spi_map[i].name, &spi_map[i]);

		if (my_sub_mtd[i]) {
			my_sub_mtd[i]->owner = THIS_MODULE;
			chips++;
			//printk("%s, %d, i=%d, chips=%d\n", __FUNCTION__, __LINE__, i, chips);
		}
	}
	
	#ifdef CONFIG_MTD_CONCAT
	if (chips == 1) 
		mymtd = my_sub_mtd[0];
	else 		
		{
			//printk("%s, %d\n, size=0x%x\n", __FUNCTION__, __LINE__, my_sub_mtd[0]->size);
			mymtd = mtd_concat_create(&my_sub_mtd[0], chips,"flash_concat");
			//printk("%s, %d, size=0x%x\n", __FUNCTION__, __LINE__, (mymtd->size));
		}
	
	if (!mymtd) {
		printk("Cannot create flash concat device\n");
		return -ENXIO;
	}
	#endif
	
	#ifdef CONFIG_MTD_CONCAT
	add_mtd_partitions(mymtd, rtl8196_parts1, ARRAY_SIZE(rtl8196_parts1));
	#ifdef DEBUG_MAP
    	printk(KERN_NOTICE "name=%s, size=0x%x\n", mymtd->name, mymtd->size);
	#endif
	#else
	if (my_sub_mtd[0]) {
      #ifdef USER_DEFINITION_FLASH_MAP
        #if DYNAMIC_UDPATE_PARTITION_TBL
        sn_update_partition_tbl((struct spi_chip_info *)(spi_map[0].fldrv_priv), my_sub_mtd[0]->erasesize);
        #endif
		add_mtd_partitions(my_sub_mtd[0], physmap_partitions_t, ARRAY_SIZE(physmap_partitions_t));        
      #else
	    add_mtd_partitions(my_sub_mtd[0], rtl8196_parts1, ARRAY_SIZE(rtl8196_parts1));
      #endif /*USER_DEFINITION_FLASH_MAP*/
	#ifdef DEBUG_MAP
    	printk(KERN_NOTICE "name=%s, size=0x%x\n", my_sub_mtd[0]->name,
					 my_sub_mtd[0]->size);
	#endif
	}
#ifdef CONFIG_RTL_TWO_SPI_FLASH_ENABLE	
	if (my_sub_mtd[1]) {
    #ifdef USER_DEFINITION_FLASH_MAP
		add_mtd_partitions(my_sub_mtd[1], physmap_partitions2_t, ARRAY_SIZE(physmap_partitions2_t));
	#else
		add_mtd_partitions(my_sub_mtd[1], rtl8196_parts2, 
					ARRAY_SIZE(rtl8196_parts2));
	#endif
	#ifdef DEBUG_MAP
    	printk(KERN_NOTICE "name=%s, size=0x%x\n", my_sub_mtd[1]->name,
					 my_sub_mtd[1]->size);
	#endif
	}
#endif
	#endif

	ROOT_DEV = MKDEV(MTD_BLOCK_MAJOR, 0);
	return 0;
}

mod_exit_t cleanup_rtl8196_map(void)
{
	int i;

	if (mymtd) {
		del_mtd_partitions(mymtd);
		map_destroy(mymtd);
	}

	for(i=0; i<MAX_SPI_CS; i++)
	{
		if (my_sub_mtd[i])
		{
			del_mtd_partitions(my_sub_mtd[i]);
			map_destroy(my_sub_mtd[i]);
		}
	}
}

MODULE_LICENSE("GPL");
module_init(init_rtl8196_map);
module_exit(cleanup_rtl8196_map);

