/* Kernel module to match the port-ranges, trigger related port-ranges,
 * and alters the destination to a local IP address.
 *
 * Copyright (C) 2003, CyberTAN Corporation
 * All Rights Reserved.
 *
 * Description:
 *   This is kernel module for port-triggering.
 *
 *   The module follows the Netfilter framework, called extended packet 
 *   matching modules.
 * 
 *   Luke 2011/6/9: support multiport
 */

#include <linux/types.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/timer.h>
#include <linux/module.h>
#include <linux/netfilter.h>
#include <linux/netdevice.h>
#include <linux/if.h>
#include <linux/inetdevice.h>
#include <net/protocol.h>
#include <net/checksum.h>

#include <linux/netfilter_ipv4.h>
#include <linux/netfilter/x_tables.h>
#include <net/netfilter/nf_conntrack.h>
#include <net/netfilter/nf_conntrack_core.h>
#include <net/netfilter/nf_conntrack_tuple.h>
#include <net/netfilter/nf_nat_rule.h>
#include <linux/netfilter_ipv4/ip_autofw.h>
#include <linux/netfilter_ipv4/lockhelp.h>
#ifdef CONFIG_NF_NAT_NEEDED
#include <net/netfilter/nf_nat_rule.h>
#else
#include <linux/netfilter_ipv4/ip_nat_rule.h>
#endif
#include <linux/netfilter/xt_TRIGGER.h>

/* This rwlock protects the main hash table, protocol/helper/expected
 *    registrations, conntrack timers*/

#define ASSERT_READ_LOCK(x) MUST_BE_READ_LOCKED(&nf_conntrack_lock)
#define ASSERT_WRITE_LOCK(x) MUST_BE_WRITE_LOCKED(&nf_conntrack_lock)

#include <linux/netfilter_ipv4/listhelp.h>

#if 0
#define DEBUGP printk
#else
#define DEBUGP(format, args...)
#endif

struct xt_trigger {
	struct list_head list;		/* Trigger list */
	struct timer_list timeout;	/* Timer for list destroying */
	u_int32_t srcip;		/* Outgoing source address */
	u_int32_t dstip;		/* Outgoing destination address */
	u_int16_t mproto;		/* Trigger protocol */
	u_int16_t rproto;		/* Related protocol */
	struct xt_trigger_ports mports;	/* Trigger and related ports */
	struct xt_trigger_ports rports;	/* Trigger and related ports */
	u_int8_t reply;			/* Confirm a reply connection */
};

LIST_HEAD(trigger_list);
//DECLARE_LOCK(ip_trigger_lock);

static void trigger_refresh(struct xt_trigger *trig, unsigned long extra_jiffies)
{
    DEBUGP("%s: \n", __FUNCTION__);
    NF_CT_ASSERT(trig);
    WRITE_LOCK(&nf_conntrack_lock);

    /* Need del_timer for race avoidance (may already be dying). */
    if (del_timer(&trig->timeout)) {
	trig->timeout.expires = jiffies + extra_jiffies;
	add_timer(&trig->timeout);
    }

    WRITE_UNLOCK(&nf_conntrack_lock);
}

static void __del_trigger(struct xt_trigger *trig)
{
    DEBUGP("%s: \n", __FUNCTION__);
    NF_CT_ASSERT(trig);
    MUST_BE_WRITE_LOCKED(&nf_conntrack_lock);

     /* delete from 'trigger_list' */
    list_del(&trig->list);
    kfree(trig);
}

static void trigger_timeout(unsigned long ul_trig)
{
    struct xt_trigger *trig= (void *) ul_trig;

    DEBUGP("trigger list %p timed out\n", trig);
    WRITE_LOCK(&nf_conntrack_lock);
    __del_trigger(trig);
    WRITE_UNLOCK(&nf_conntrack_lock);
}

static unsigned int
add_new_trigger(struct xt_trigger *trig)
{
    struct xt_trigger *new;

    DEBUGP("!!!!!!!!!!!! %s !!!!!!!!!!!\n", __FUNCTION__);
    WRITE_LOCK(&nf_conntrack_lock);
    new = (struct xt_trigger *)
	kmalloc(sizeof(struct xt_trigger), GFP_ATOMIC);

    if (!new) {
	WRITE_UNLOCK(&nf_conntrack_lock);
	DEBUGP("%s: OOM allocating trigger list\n", __FUNCTION__);
	return -ENOMEM;
    }

    memset(new, 0, sizeof(*trig));
    INIT_LIST_HEAD(&new->list);
    memcpy(new, trig, sizeof(*trig));

    /* add to global table of trigger */
    list_prepend(&trigger_list, &new->list);
    /* add and start timer if required */
    init_timer(&new->timeout);
    new->timeout.data = (unsigned long)new;
    new->timeout.function = trigger_timeout;
    new->timeout.expires = jiffies + (TRIGGER_TIMEOUT * HZ);
    add_timer(&new->timeout);
	    
    WRITE_UNLOCK(&nf_conntrack_lock);

    return 0;
}

static inline bool
ports_match_v1(const struct xt_trigger_ports *minfo, u_int16_t dst)
{
	unsigned int i;
	u_int16_t s, e;

	for (i = 0; i < minfo->count; i++) {
		s = minfo->ports[i];

		if (minfo->pflags[i]) {
			/* range port matching */
			e = minfo->ports[++i];

			if (dst >= s && dst <= e)
				return true;
		} else {
			/* exact port matching */
			if (dst == s)
				return true;
		}
	}

	return false;
}

static inline bool trigger_out_matched(const struct xt_trigger *i,
	const u_int16_t proto, const u_int16_t dport)
{	
	if(i->mproto == proto){
		return ports_match_v1(&i->mports, dport);
	}
	
	return false;
}

static unsigned int
trigger_out(struct sk_buff **pskb,
		unsigned int hooknum,
		const struct net_device *in,
		const struct net_device *out,
		const void *targinfo)
{
    const struct xt_trigger_info *info = targinfo;
    struct xt_trigger trig, *found;
    const struct iphdr *iph = ip_hdr(*pskb);
    struct tcphdr *tcph = (void *)iph + iph->ihl*4;	/* Might be TCP, UDP */

    DEBUGP("############# %s ############\n", __FUNCTION__);
	
	if(!ports_match_v1(&info->mports, ntohs(tcph->dest))){
		return XT_CONTINUE;
	}	
	
    /* Check if the trigger range has already existed in 'trigger_list'. */
    found = LIST_FIND(&trigger_list, trigger_out_matched,
	    struct xt_trigger *, iph->protocol, ntohs(tcph->dest));

	DEBUGP("tcph->dest=%d\n", ntohs(tcph->dest));
	DEBUGP("found=%d\n", found);
		
    if (found) {
		/* Yeah, it exists. We need to update(delay) the destroying timer. */
		trigger_refresh(found, TRIGGER_TIMEOUT * HZ);
		/* In order to allow multiple hosts use the same port range, we update
		   the 'saddr' after previous trigger has a reply connection. */
		if (found->reply)
			found->srcip = iph->saddr;
    }
    else {
		/* Create new trigger */
		DEBUGP("Create new trigger --> port:%d\n", ntohs(tcph->dest));
		memset(&trig, 0, sizeof(trig));
		trig.srcip = iph->saddr;
		trig.mproto = iph->protocol;
		trig.rproto = info->proto;
		memcpy(&trig.mports, &info->mports, sizeof(struct xt_trigger_ports));
		memcpy(&trig.rports, &info->rports, sizeof(struct xt_trigger_ports));
		add_new_trigger(&trig);	/* Add the new 'trig' to list 'trigger_list'. */
    }

    return XT_CONTINUE;	/* We don't block any packet. */
}

static inline int trigger_in_matched(const struct xt_trigger *i,
	const u_int16_t proto, const u_int16_t dport)
{
    u_int16_t rproto;
	rproto = i->rproto;

    if (!rproto)
		rproto = proto;
	
	if(rproto == proto){
		return ports_match_v1(&i->rports, dport);
	}
	
	return false;
}

static unsigned int
trigger_in(struct sk_buff **pskb,
		unsigned int hooknum,
		const struct net_device *in,
		const struct net_device *out,
		const void *targinfo)
{
    struct xt_trigger *found;
    const struct iphdr *iph = ip_hdr(*pskb);
    struct tcphdr *tcph = (void *)iph + iph->ihl*4;	/* Might be TCP, UDP */
    /* Check if the trigger-ed range has already existed in 'trigger_list'. */
    found = LIST_FIND(&trigger_list, trigger_in_matched,
	    struct xt_trigger *, iph->protocol, ntohs(tcph->dest));
		
	DEBUGP("%s found=%d\n", __FUNCTION__, found);
	DEBUGP("%s tcph->dest=%d\n", __FUNCTION__, ntohs(tcph->dest));
    if (found) {
	DEBUGP("############# %s ############\n", __FUNCTION__);
	/* Yeah, it exists. We need to update(delay) the destroying timer. */
	trigger_refresh(found, TRIGGER_TIMEOUT * HZ);
	return NF_ACCEPT;	/* Accept it, or the imcoming packet could be 
				   dropped in the FORWARD chain */
    }
 
    return XT_CONTINUE;	/* Our job is the interception. */
}

static unsigned int
trigger_dnat(struct sk_buff **pskb,
		unsigned int hooknum,
		const struct net_device *in,
		const struct net_device *out,
		const void *targinfo)
{
    struct xt_trigger *found;
    const struct iphdr *iph = ip_hdr(*pskb);
    struct tcphdr *tcph = (void *)iph + iph->ihl*4;	/* Might be TCP, UDP */
    struct nf_conn *ct;
    enum ip_conntrack_info ctinfo;
    struct nf_nat_multi_range_compat newrange;

    NF_CT_ASSERT(hooknum == NF_INET_PRE_ROUTING);
    /* Check if the trigger-ed range has already existed in 'trigger_list'. */
    found = LIST_FIND(&trigger_list, trigger_in_matched,
	    struct xt_trigger *, iph->protocol, ntohs(tcph->dest));

    if (!found || !found->srcip)
		return XT_CONTINUE;	/* We don't block any packet. */

    DEBUGP("############# %s ############\n", __FUNCTION__);
    found->reply = 1;	/* Confirm there has been a reply connection. */
    ct = nf_ct_get(*pskb, &ctinfo);
    NF_CT_ASSERT(ct && (ctinfo == IP_CT_NEW));

    DEBUGP("%s: got ", __FUNCTION__);
    nf_ct_dump_tuple(&ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple);

    /* Alter the destination of imcoming packet. */
    newrange = ((struct nf_nat_multi_range_compat)
	    { 1, { { IP_NAT_RANGE_MAP_IPS,
	             found->srcip, found->srcip,
	             { 0 }, { 0 }
	           } } });

    /* Hand modified range to generic setup. */
    return nf_nat_setup_info(ct, &newrange.range[0], IP_NAT_MANIP_DST);
}

static unsigned int
trigger_target(struct sk_buff *skb, const struct xt_target_param *par)
{
    const struct xt_trigger_info *info = par->targinfo;
    const struct iphdr *iph = ip_hdr(skb);

     DEBUGP("%s: type = %s\n", __FUNCTION__, 
	    (info->type == XT_TRIGGER_DNAT) ? "dnat" :
	    (info->type == XT_TRIGGER_IN) ? "in" : "out");

    /* The Port-trigger only supports TCP and UDP. */
    if ((iph->protocol != IPPROTO_TCP) && (iph->protocol != IPPROTO_UDP))
	return XT_CONTINUE;

    if (info->type == XT_TRIGGER_OUT)
	return trigger_out(&skb, par->hooknum, par->in, par->out, par->targinfo);
    else if (info->type == XT_TRIGGER_IN)
	return trigger_in(&skb, par->hooknum, par->in, par->out, par->targinfo);
    else if (info->type == XT_TRIGGER_DNAT)
    	return trigger_dnat(&skb, par->hooknum, par->in, par->out, par->targinfo);

    return XT_CONTINUE;
}

static bool trigger_check(const struct xt_tgchk_param *par)
{
	//const struct xt_entry *e = (struct xt_entry*)par->entryinfo;
	const struct xt_trigger_info *info = par->targinfo;
	struct list_head *cur_item, *tmp_item;

	if ((strcmp(par->table, "mangle") == 0)) {
		DEBUGP("trigger_check: bad table `%s'.\n", par->table);
		return 0;
	}
	/*chain cannot be INPUT, OUTPUT, POSTROUTING*/
	if (par->hook_mask & ~((1 << NF_INET_PRE_ROUTING) | (1 << NF_INET_FORWARD))) {
		DEBUGP("trigger_check: bad hooks %x.\n", par->hook_mask);
		return 0;
	}
	if (info->proto) {
	    if (info->proto != IPPROTO_TCP && info->proto != IPPROTO_UDP) {
		DEBUGP("trigger_check: bad proto %d.\n", info->proto);
		return 0;
	    }
	}
	if (info->type == XT_TRIGGER_OUT) {
		if (!info->mports.ports[0] || !info->rports.ports[0]) {
		DEBUGP("trigger_check: Try 'iptbles -j TRIGGER -h' for help.\n");
		return 0;
	    }
	}

	/* Empty the 'trigger_list' */
	list_for_each_safe(cur_item, tmp_item, &trigger_list) {
	    struct xt_trigger *trig = (void *)cur_item;

	    DEBUGP("%s: list_for_each_safe(): %p.\n", __FUNCTION__, trig);
	    del_timer(&trig->timeout);
	    __del_trigger(trig);
	}

	return 1;
}


static struct xt_target redirect_reg __read_mostly = { 
	.name 		= "TRIGGER",
	.family		= AF_INET,
	.target 	= trigger_target, 
	.targetsize	= sizeof(struct xt_trigger_info),
	.checkentry 	= trigger_check,
	.me 		= THIS_MODULE,
};

static int __init init(void)
{
	return xt_register_target(&redirect_reg);
}

static void __exit fini(void)
{
	xt_unregister_target(&redirect_reg);
}

module_init(init);
module_exit(fini);
