/*
 * TFTP library 
 * copyright (c) 2004 Vanden Berghen Frank  
 *
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */
 
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <syslog.h>
#include <stdio.h>   
#include <stdlib.h>
#include <memory.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h> 
#include <signal.h>
#include "tftp.h"

#ifdef CONFIG_MASS_PRODUCE
#include "public_util.h"
#endif	// CONFIG_MASS_PRODUCE

#define TFTP_DEBUG 1
#ifdef TFTP_DEBUG
#define DEBUG_MSG(fmt, arg...)       printf(fmt, ##arg)
#else
#define DEBUG_MSG(fmt, arg...) 
#endif

#include "fw_size.h" //Makefile create...

extern int create_socket(int type, int *p_port);
extern int TimeOut, NumberTimeOut;

/* dynamic receiving data display */
static int debug_count = 0;
static int debug_count_flag = 0;

static void debug_receive(void)
{
	if(debug_count < 70) {
		printf(".");
		debug_count++;
	} else {
		if(debug_count_flag >= 10)
			debug_count_flag = 0;

		printf("%d\n", debug_count_flag);
		debug_count_flag++;
		debug_count = 0;
	}
}


/* return 0 if no error.*/
char TFTPswrite(char *p_data,long p_data_amount,char first,void *f)
{
    int result;
    result = fwrite(p_data, p_data_amount,1,(FILE *)f);
    if(result < 0)
        return result;
    else
        return 0;
}

/* return 0 if no error.*/
char TFTPsread(char *p_data,long *p_data_amount,char first,void *f)
{
    *p_data_amount=fread(p_data,1,SEGSIZE,(FILE *)f);
	if(*p_data_amount < 0)
        return *p_data_amount; 
 	else
        return 0;    
}

void nak(int peer,struct sockaddr_in *p_to_addr,int error,char *p_error_msg)
{
	char buf[PKTSIZE];
	int length, status;
	size_t tolen=sizeof(struct sockaddr_in);
	struct tftphdr *p_tftp_pkt=(struct tftphdr *)&buf;

	printf("error mesg: %s \n", p_error_msg);
	p_tftp_pkt->th_opcode = htons((u_short)ERROR);
	p_tftp_pkt->th_code = htons((u_short)error);
	strcpy(p_tftp_pkt->th_msg, p_error_msg);
	length = strlen(p_tftp_pkt->th_msg);
	p_tftp_pkt->th_msg[length] = '\0';
	length += 5;

	status = sendto(peer, buf, length, 0,(struct sockaddr*)p_to_addr, tolen);
	if ( status!= length) 
	{
		printf("send nak failed: %d\n", errno);
	}

}

void tftp_free(void * ptr){
    if(ptr){
		free(ptr);
	}
}

extern char preamble_mac[];

int tftp_receive_ext(struct sockaddr_in *p_to_addr,char *p_name,char *p_mode,int InClient,                
					 char (*TFTPwrite)(char *,long ,char,void *),
                     void *argu,int vPKTSIZE)
{
	struct tftphdr *p_tftpdata_pkt,*p_tftpack_pkt;
	struct timeval tv;
	struct sockaddr_in from, to = *p_to_addr;
	size_t fromlen = sizeof(from), tolen = fromlen;
	fd_set lecture;
	char *p_data_pkt, *p_ack_pkt, *p_data_buf, *p_copy_buf;
	char *file_buffer, *copy_buffer;
	u_short nextBlockNumber;
	int result = 0, bytes;
	int image_length = 0;
	int size, ntimeout, peer, port_no;
	
	p_data_pkt = (char*)malloc(vPKTSIZE);
	p_ack_pkt = (char*)malloc(vPKTSIZE);
   
	file_buffer = (char*)malloc(FW_BUF_SIZE); /* Cameo: FW_BUF_SIZE define in fw_size.h */
  
    if(p_data_pkt == NULL || p_ack_pkt == NULL || file_buffer == NULL){
        DEBUG_MSG("TFTP: out of memory.\n");
        tftp_free(p_data_pkt); 
        tftp_free(p_ack_pkt);
		tftp_free(file_buffer); 
        return 255;
    }
	
    printf("TFTP receving..... \n");
    memset(file_buffer, 0, FW_BUF_SIZE);
    memset(p_data_pkt, 0, vPKTSIZE);
    memset(p_ack_pkt, 0, vPKTSIZE);
    copy_buffer = file_buffer;

    p_tftpdata_pkt = (struct tftphdr *)p_data_pkt;				
    p_tftpack_pkt = (struct tftphdr *)p_ack_pkt;			
    p_data_buf = (char*)p_tftpdata_pkt+4;   
    p_copy_buf = (char*)&p_tftpack_pkt->th_stuff[0];	

    port_no = 0;
    if ((peer = create_socket(SOCK_DGRAM, &port_no)) < 0){
        DEBUG_MSG("create socket failure %d:\n", errno);
        tftp_free(p_data_pkt); 
        tftp_free(p_ack_pkt);
       
        tftp_free(file_buffer); 
        return 255;
    }
          
    if (InClient){
        p_tftpack_pkt->th_opcode = htons((u_short)RRQ);
        strcpy(p_copy_buf, p_name);
        p_copy_buf += strlen(p_name);
        *p_copy_buf++ = '\0';
        strcpy(p_copy_buf, p_mode);
        p_copy_buf += strlen(p_mode);
        *p_copy_buf++ = '\0';
        size = (unsigned long)p_copy_buf-(unsigned long)p_ack_pkt;
    }else {
        p_tftpack_pkt->th_opcode = htons((u_short)ACK);
        p_tftpack_pkt->th_block = 0;
        size = 4;						
    }
    nextBlockNumber = 1;
    
    do{    
        /*next ACK's timeout */
        ntimeout = 0;
        do{
            if (ntimeout == NumberTimeOut){ 
                DEBUG_MSG("TFTP timeout");
                close(peer); 
                tftp_free(p_data_pkt); 
                tftp_free(p_ack_pkt);
                tftp_free(file_buffer); 
                return 255;
            }

            if (sendto(peer, p_tftpack_pkt, size, 0, (struct sockaddr *)&to, tolen) != size){
                DEBUG_MSG("tftp: write error : %d", errno);
                close(peer);  
                tftp_free(p_data_pkt); 
                tftp_free(p_ack_pkt);
                tftp_free(file_buffer); 
                return 255;
            }

			do{
				bytes = -1;
				FD_ZERO(&lecture);
				FD_SET(peer, &lecture); 
				tv.tv_sec = TimeOut; 
				tv.tv_usec = 0;

				if ((result = select(peer+1, &lecture, NULL, NULL, &tv)) == -1){
					DEBUG_MSG("tftp: select error : %d", errno);
					close(peer); 
					tftp_free(p_data_pkt); 
					tftp_free(p_ack_pkt);
					tftp_free(file_buffer); 
					return 255;
				};
				
				if (result > 0){
					memset(p_tftpdata_pkt, 0, vPKTSIZE);
					bytes = recvfrom(peer, p_tftpdata_pkt, vPKTSIZE, 0,(struct sockaddr *)&from, &fromlen);

          debug_receive();

				}
			} while ((bytes < 0)&&(result > 0));    //recv nothing but select>0, keep recv

			if (result > 0){
				to.sin_port=from.sin_port; 
				p_tftpdata_pkt->th_opcode = ntohs((u_short)p_tftpdata_pkt->th_opcode);            
				p_tftpdata_pkt->th_block = ntohs((u_short)p_tftpdata_pkt->th_block);
				
				if (p_tftpdata_pkt->th_opcode != DATA) {
					DEBUG_MSG("tftp: op code is not correct \n");
					close(peer); 
					tftp_free(p_data_pkt); 
					tftp_free(p_ack_pkt);
					tftp_free(file_buffer); 
					return 255;
				}
        
				if (p_tftpdata_pkt->th_block != nextBlockNumber){
					/* Re-synchronize with the other side */
					ioctl(peer, FIONREAD, &bytes); 
					while (bytes){
						recv(peer, p_tftpdata_pkt, vPKTSIZE, 0);
						ioctl(peer, FIONREAD, &bytes);
					};
					p_tftpdata_pkt->th_block = nextBlockNumber+1;                  
				}
			}
         debug_receive();
			ntimeout++;
		} while (p_tftpdata_pkt->th_block != nextBlockNumber);
       
        p_tftpack_pkt->th_block = htons(nextBlockNumber);
        nextBlockNumber++;

        /*only use in client */
        if (nextBlockNumber == 2){
            p_tftpack_pkt->th_opcode = htons((u_short)ACK); 
            size = 4;
        }     
		
        if (bytes-4 > 0){
            memcpy(copy_buffer, p_data_buf, bytes-4);
            memset(p_data_buf, 0, bytes-4);
            copy_buffer = copy_buffer + bytes-4;
			image_length = image_length + bytes-4;
			result = 0;
        }
 
    } while (bytes == vPKTSIZE);
	
    printf("TFTP receive successfully \n");
    /* send the "final" ack */
    sendto(peer, p_tftpack_pkt, 4, 0,(struct sockaddr *)&to, tolen);

#ifdef CONFIG_MASS_PRODUCE	
	// if the content of file_buffer is come from smac or not
	if (memcmp(preamble_mac, file_buffer, 6) == 0){	
		printf("[%s-%d] is smac command... \n", __FUNCTION__, __LINE__);
		execute_smac_cmds((file_buffer + 6), bytes - 6);
	}else{
		//save_upload_file(p_name, file_buffer, image_length);
		printf("[%s-%d] upload files... \n", __FUNCTION__, __LINE__);
		mp_upload_file(p_name, file_buffer, image_length);
	}
#endif	// CONFIG_MASS_PRODUCE
	
    close(peer); 
    tftp_free(p_data_pkt); 
    tftp_free(p_ack_pkt);
	tftp_free(file_buffer);
    return 0;
}

int tftp_receive(struct sockaddr_in *p_to_addr,char *p_name,char *p_mode,int InClient,                
                 char (*TFTPwrite)(char *,long ,char,void *),
                 void *argu)
{
    return tftp_receive_ext(p_to_addr, p_name, p_mode, InClient, TFTPwrite,argu, PKTSIZE);
}


int tftp_send_ext(struct sockaddr_in *p_to_addr,char *p_name,char *p_mode,int InClient,
                  char (*TFTPread)(char *,long *,char,void *),                
                  void *argu, int vPKTSIZE)
{
    char *p_data_pkt,*p_ack_pkt,*p_data_buf,*p_copy_buf;
    struct tftphdr *p_tftpdata_pkt,*p_tftpack_pkt;
    int result =0, bytes;
    int size,Oldsize=vPKTSIZE;
    int ntimeout,peer, port_no;
    ushort nextBlockNumber;
    struct timeval tv;
    fd_set lecture;
    struct sockaddr_in from,to=*p_to_addr;
    size_t fromlen=sizeof(from),tolen=fromlen;

    p_data_pkt=(char*)malloc(vPKTSIZE);
    if (p_data_pkt==NULL)
    {
        DEBUG_MSG("TFTP: out of memory.\n");
        return 255;
    }
    p_ack_pkt=(char*)malloc(vPKTSIZE);
    if (p_ack_pkt==NULL)
    {
        DEBUG_MSG("TFTP: out of memory.\n");
        tftp_free(p_data_pkt); return 255;
    }
    p_tftpdata_pkt=(struct tftphdr *)p_data_pkt;
    p_tftpack_pkt=(struct tftphdr *)p_ack_pkt;
    p_data_buf=(char*)&p_tftpdata_pkt->th_data[0];
    p_copy_buf=(char*)&p_tftpdata_pkt->th_stuff[0];

    port_no = 0;
    if ((peer = create_socket(SOCK_DGRAM, &port_no))<0)
    {
        DEBUG_MSG("create socket failure %d:\n", errno);
        tftp_free(p_data_pkt); tftp_free(p_ack_pkt);
        return 255;
    }
    
    if (InClient)
    {
        p_tftpdata_pkt->th_opcode=htons((u_short)WRQ);
        strcpy(p_copy_buf, p_name);
        p_copy_buf += strlen(p_name);
        *p_copy_buf++ = '\0';
        strcpy(p_copy_buf, p_mode);
        p_copy_buf += strlen(p_mode);
        *p_copy_buf++ = '\0';
        size=(unsigned long)p_copy_buf-(unsigned long)p_data_pkt;
        nextBlockNumber=0;
    } 
    else
    {
        p_tftpdata_pkt->th_opcode=htons((u_short)DATA);
        p_tftpdata_pkt->th_block=htons((ushort)1);
        /*The actual size read from file will be returned in "size" */
        if ((*TFTPread)(p_data_buf,(long*)(&size),1,argu)!=0) 
        {   

            DEBUG_MSG("TFTPread error : %d", errno);
            nak(peer, &to, errno,strerror(errno));
            close(peer); tftp_free(p_data_pkt); tftp_free(p_ack_pkt);
            return 255;
        } 
        size+=4;
        nextBlockNumber=1;
    }
    
    do 
    {  
        /*next ACK's timeout */
        ntimeout=0; 
        do
        {
            if (ntimeout==NumberTimeOut) { close(peer); tftp_free(p_data_pkt); tftp_free(p_ack_pkt); return 255;}

            if (sendto(peer,p_tftpdata_pkt,size,0,(struct sockaddr *)&to,tolen)!=size)
            {
                DEBUG_MSG("sendto failure %d", errno);
                close(peer); tftp_free(p_data_pkt); tftp_free(p_ack_pkt);
                return 255;
            }

        do
        {
            bytes = -1;
            FD_ZERO(&lecture);
            FD_SET(peer,&lecture); 
            tv.tv_sec=TimeOut; tv.tv_usec=0;

            if ((result=select(peer+1, &lecture, NULL, NULL, &tv))==-1)
            {
                DEBUG_MSG("tftp: select error : %d", errno);
                close(peer); tftp_free(p_data_pkt); tftp_free(p_ack_pkt);
                return 255;
            }
            if (result >0) bytes = recvfrom(peer, p_tftpack_pkt, vPKTSIZE, 0,(struct sockaddr *)&from, &fromlen);                
        } while ((bytes < 0)&&(result > 0)); //recv nothing but select>0, keep recv

         if (result > 0)
         {
            to.sin_port=from.sin_port;
            p_tftpack_pkt->th_opcode = ntohs((u_short)p_tftpack_pkt->th_opcode);
            p_tftpack_pkt->th_block = ntohs((u_short)p_tftpack_pkt->th_block);
            if (p_tftpack_pkt->th_opcode != ACK) 
            { 
                   DEBUG_MSG("opcode not correct");
                	close(peer); tftp_free(p_data_pkt); tftp_free(p_ack_pkt); 
                	return 255;
            }
            
            if (p_tftpack_pkt->th_block != nextBlockNumber)
            {
               /* Re-synchronize with the other side */
               ioctl(peer, FIONREAD, &bytes); 
               while (bytes)
               {
                  recv(peer, p_tftpack_pkt, vPKTSIZE, 0);
                  ioctl(peer, FIONREAD, &bytes);
               };
               p_tftpack_pkt->th_block=nextBlockNumber+1;  
            }
         }
         ntimeout++;
      } while (p_tftpack_pkt->th_block!=nextBlockNumber);

        if ((size<vPKTSIZE)&&(nextBlockNumber!=0)) break; 

        nextBlockNumber++;
        p_tftpdata_pkt->th_block=htons(nextBlockNumber);		
        
        /*only use in client*/
        if (nextBlockNumber==1)
        {
            p_tftpdata_pkt->th_opcode=htons((u_short)DATA);
            result=(*TFTPread)(p_data_buf,(long*)(&size),1,argu);
        } 
        else 
        {
            Oldsize=size;
            if (Oldsize==vPKTSIZE) result =(*TFTPread)(p_data_buf,(long*)(&size),0,argu);
            else result=0;
        }

        if (result < 0)
        {
            DEBUG_MSG("TFTPread error : %d", errno);
            nak(peer, &from, errno,strerror(errno));
            close(peer); tftp_free(p_data_pkt); tftp_free(p_ack_pkt);
            return result;
        }
        size+=4;
    } while (Oldsize==vPKTSIZE);
    printf("TFTP send successfully \n");
    close(peer); tftp_free(p_data_pkt); tftp_free(p_ack_pkt);
    return 0;
}

int tftp_send(struct sockaddr_in *p_to_addr,char *p_name,char *p_mode,int InClient,
              char (*TFTPread)(char *,long *,char,void *),                
              void *argu)
{
    return tftp_send_ext(p_to_addr, p_name, p_mode, InClient, TFTPread, argu, PKTSIZE);
}
