/* $Id: upnphttp.c,v 1.61 2011/06/27 11:05:59 nanard Exp $ */
/* Project :  miniupnp
 * Website :  http://miniupnp.free.fr/ or http://miniupnp.tuxfamily.org/
 * Author :   Thomas Bernard
 * Copyright (c) 2005-2011 Thomas Bernard
 * This software is subject to the conditions detailed in the
 * LICENCE file included in this distribution.
 * */
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <ctype.h>
#include "config.h"
#include "upnphttp.h"
#include "upnpdescgen.h"
#include "miniupnpdpath.h"
#include "upnpsoap.h"
#include "upnpevents.h"
#include "upnpglobalvars.h"

#define cprintf(fmt, args...) do { \
        FILE *cfp = fopen("/dev/console", "w"); \
        if (cfp) { \
                fprintf(cfp, fmt, ## args); \
                fclose(cfp); \
        } \
} while (0)

static const char *
findendheaders(const char * s, int len)
{
	while(len-->0)
	{
		if(s[0]=='\r' && s[1]=='\n' && s[2]=='\r' && s[3]=='\n')
			return s;
		s++;
	}
	return NULL;
}

struct upnphttp * 
New_upnphttp(int s)
{
	struct upnphttp * ret;
	if(s<0)
		return NULL;
	ret = (struct upnphttp *)malloc(sizeof(struct upnphttp));
	if(ret == NULL)
		return NULL;
	memset(ret, 0, sizeof(struct upnphttp));
	ret->socket = s;
	return ret;
}

void
CloseSocket_upnphttp(struct upnphttp * h)
{
//remark pipeline code: for fix customer report issue, sometimes see "recv (state0): Connection reset by peer" error message in log
//	if(h->req_ConnectionClose == 1){
		if(close(h->socket) < 0)
		{
			syslog(LOG_ERR, "CloseSocket_upnphttp: close(%d): %m", h->socket);
		}
		h->socket = -1;
		h->state = 100;
//		h->req_pipeline = 0;
//	}else{
//		//pipeline
//		char *endheaders = NULL;
//		char *headeroff;
//		int headerlen = 0;
//		char *pipebuf;
//		
//		headerlen = h->req_buflen - h->req_contentoff - h->req_contentlen;
//		
//		if(headerlen > 0){
//			headeroff = h->req_buf + h->req_contentoff + h->req_contentlen;
//			
//			endheaders = findendheaders(headeroff, headerlen);
//			if(endheaders)	//got pipeline request
//			{
//				pipebuf = malloc(headerlen+1);
//				if(pipebuf){
//					memcpy(pipebuf, headeroff, headerlen);
//					pipebuf[headerlen] = '\0';
//				}
//			}
//		}
	
//		//Luke: clean buffer for next use
//		free(h->req_buf);
//		free(h->res_buf);
//		memset(h, 0, upnphttp_cleansize);
		
		
//		//pipeline
//		if(endheaders){
//			h->req_buf = pipebuf;
//			h->req_buflen = headerlen;
//			h->req_pipeline = 1;
//		}
//	}
}

void
Delete_upnphttp(struct upnphttp * h)
{
	if(h)
	{
//		DPRINTF(E_DEBUG, L_HTTP, "Delete_upnphttp: close(%d)\n", h->socket);
		if(h->socket >= 0){
//			h->req_ConnectionClose = 1;
			CloseSocket_upnphttp(h);
		}
		if(h->req_buf)
			free(h->req_buf);
		if(h->res_buf)
			free(h->res_buf);
		free(h);
	}
}

/* parse HttpHeaders of the REQUEST */
static void
ParseHttpHeaders(struct upnphttp * h)
{
	char * line;
	char * colon;
	char * p;
	int n;
	line = h->req_buf;
	/* TODO : check if req_buf, contentoff are ok */
	while(line < (h->req_buf + h->req_contentoff))
	{
		colon = strchr(line, ':');
		if(colon)
		{
			if(strncasecmp(line, "Content-Length", 14)==0)
			{
				p = colon;
				while(*p < '0' || *p > '9')
					p++;
				h->req_contentlen = atoi(p);
				/*printf("*** Content-Lenght = %d ***\n", h->req_contentlen);
				printf("    readbufflen=%d contentoff = %d\n",
					h->req_buflen, h->req_contentoff);*/
			}
			else if(strncasecmp(line, "EXPECT", 6)==0)
			{
				/* Luke: support HTTP 1.1 100 Continue */
				p = colon + 1;
				while(isspace(*p))
					p++;
				if(strncasecmp(p, "100-Continue", 12)==0) {
					//current req_continue=0  -----> this is a new 100 Continue request, set req_continue=1 and wait follow-up data
					//current req_continue=1  -----> this is the follow-up data, set req_continue=0
					h->req_continue = !(h->req_continue);
				}
			}
			else if(strncasecmp(line, "Connection", 10)==0)
			{
				p = colon + 1;
				while(isspace(*p))
					p++;
				/* RFC 2616: HTTP/1.1 is that persistent connections are the DEFAULT behavior of any HTTP connection */
				if(strncasecmp(p, "Close", 5)==0) {
					h->req_ConnectionClose = 1;
				}
			}
			else if(strncasecmp(line, "SOAPAction", 10)==0)
			{
				p = colon;
				n = 0;
				while(*p == ':' || *p == ' ' || *p == '\t')
					p++;
				while(p[n]>=' ')
				{
					n++;
				}
				if((p[0] == '"' && p[n-1] == '"')
				  || (p[0] == '\'' && p[n-1] == '\''))
				{
					p++; n -= 2;
				}
				h->req_soapAction = p;
				h->req_soapActionLen = n;
			}
#ifdef ENABLE_EVENTS
			else if(strncasecmp(line, "Callback", 8)==0)
			{
				p = colon;
				while(*p != '<' && *p != '\r' )
					p++;
				n = 0;
				while(p[n] != '>' && p[n] != '\r' )
					n++;
				
				/* Luke: UPnP Device Architecture v1.1 said
						 One or more URLs each enclosed by angle brackets (< and >). 
						 Each URL MUST be an HTTP over TCP URL (prefixed by http://) */
				if((*p != '<') || (p[n] != '>') ||
				   (strncasecmp(p+1, "http://", 7) != 0)
				){
					h->req_Callback = 0;
					h->req_CallbackLen = 0;
				}else{
					h->req_Callback = p + 1;
					h->req_CallbackLen = MAX(0, n - 1);
				}
			}
			else if(strncasecmp(line, "SID", 3)==0)
			{
				//zqiu: fix bug for test 4.0.5
				//Skip extra headers like "SIDHEADER: xxxxxx xxx"
				for(p=line+3;p<colon;p++)
				{
					if(!isspace(*p))
					{
						p = NULL; //unexpected header
						break;
					}
				}
				if(p) {
					p = colon + 1;
					while(isspace(*p))
						p++;
					n = 0;
					while(!isspace(p[n]))
						n++;
					h->req_SID = p;
					h->req_SIDLen = n;
				}
			}
			/* Timeout: Seconds-nnnn */
/* TIMEOUT
Recommended. Requested duration until subscription expires,
either number of seconds or infinite. Recommendation
by a UPnP Forum working committee. Defined by UPnP vendor.
 Consists of the keyword "Second-" followed (without an
intervening space) by either an integer or the keyword "infinite". */
			else if(strncasecmp(line, "Timeout", 7)==0)
			{
				p = colon + 1;
				while(isspace(*p))
					p++;
				if(strncasecmp(p, "Second-", 7)==0) {
					h->req_Timeout = atoi(p+7);
					/* Luke: UPnP Device Architecture v1.1 said:
							"TIMEOUT SHOULD be greater than or equal to 1800 seconds" */
					if(h->req_Timeout < 1800){
						h->req_Timeout = 1800;
					}
				}
			}
#endif
			else if(strncasecmp(line, "ACCEPT-LANGUAGE", 15)==0)
			{
				p = colon + 1;
				while(isspace(*p))
					p++;			
				n = 0;
				while(!isspace(p[n]))
					n++;
				h->req_AcceptLang = p;
				h->req_AcceptLangLen = n;			
			}
			else if(strncasecmp(line, "NT", 2)==0)
			{
				p = colon + 1;
				while(isspace(*p))
					p++;			
				n = 0;
				while(!isspace(p[n]))
					n++;
				h->req_NT = p;
				h->req_NTLen = n;			
			}
			else if(strncasecmp(line, "Host", 4)==0)
			{
				int i;
				struct lan_addr_s * lan_addr;
				h->reqflags |= FLAG_HOST;
				p = colon + 1;
				while(isspace(*p))
					p++;
				for(i=0, lan_addr = lan_addrs.lh_first; lan_addr != NULL; lan_addr = lan_addr->list.le_next, i++)
				{
					for(i=0; lan_addr->str[i]; i++)
					{
						if(lan_addr->str[i] != p[i])
							break;
					}
					if(!lan_addr->str[i])
					{
						//h->iface = n;
						break;
					}
				}
			}
		}
		while(!(line[0] == '\r' && line[1] == '\n'))
			line++;
		line += 2;
	}

	//Luke: HTTP 1.0: some feature must not support in HTTP 1.0
	if(strcmp(h->HttpVer, "HTTP/1.0")==0){
		h->req_ConnectionClose = 1;
		h->req_continue = 0;
	}
}

/* very minimalistic 400 error message */
static void
Send400(struct upnphttp * h)
{
	static const char body400[] =
		"<HTML><HEAD><TITLE>400 Bad Request</TITLE></HEAD>"
		"<BODY><H1>Bad Request</H1>The request is invalid"
		" for this HTTP version.</BODY></HTML>\r\n";
	h->respflags = FLAG_HTML;
	BuildResp2_upnphttp(h, 400, "Bad Request",
	                    body400, sizeof(body400) - 1);
	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
}

/* very minimalistic 404 error message */
static void
Send404(struct upnphttp * h)
{
/*
	static const char error404[] = "HTTP/1.1 404 Not found\r\n"
		"Connection: close\r\n"
		"Content-type: text/html\r\n"
		"\r\n"
		"<HTML><HEAD><TITLE>404 Not Found</TITLE></HEAD>"
		"<BODY><H1>Not Found</H1>The requested URL was not found"
		" on this server.</BODY></HTML>\r\n";
	int n;
	n = send(h->socket, error404, sizeof(error404) - 1, 0);
	if(n < 0)
	{
		syslog(LOG_ERR, "Send404: send(http): %m");
	}*/
	static const char body404[] =
		"<HTML><HEAD><TITLE>404 Not Found</TITLE></HEAD>"
		"<BODY><H1>Not Found</H1>The requested URL was not found"
		" on this server.</BODY></HTML>\r\n";
	h->respflags = FLAG_HTML;
	BuildResp2_upnphttp(h, 404, "Not Found",
	                    body404, sizeof(body404) - 1);
	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
}

/* very minimalistic 500 error message */
void
Send500(struct upnphttp * h)
{
	static const char body500[] = 
		"<HTML><HEAD><TITLE>500 Internal Server Error</TITLE></HEAD>"
		"<BODY><H1>Internal Server Error</H1>Server encountered "
		"and Internal Error.</BODY></HTML>\r\n";
	h->respflags = FLAG_HTML;
	BuildResp2_upnphttp(h, 500, "Internal Server Errror",
	                    body500, sizeof(body500) - 1);
	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
}

/* very minimalistic 501 error message */
void
Send501(struct upnphttp * h)
{
/*
	static const char error501[] = "HTTP/1.1 501 Not Implemented\r\n"
		"Connection: close\r\n"
		"Content-type: text/html\r\n"
		"\r\n"
		"<HTML><HEAD><TITLE>501 Not Implemented</TITLE></HEAD>"
		"<BODY><H1>Not Implemented</H1>The HTTP Method "
		"is not implemented by this server.</BODY></HTML>\r\n";
	int n;
	n = send(h->socket, error501, sizeof(error501) - 1, 0);
	if(n < 0)
	{
		syslog(LOG_ERR, "Send501: send(http): %m");
	}
*/
	static const char body501[] = 
		"<HTML><HEAD><TITLE>501 Not Implemented</TITLE></HEAD>"
		"<BODY><H1>Not Implemented</H1>The HTTP Method "
		"is not implemented by this server.</BODY></HTML>\r\n";
	h->respflags = FLAG_HTML;
	BuildResp2_upnphttp(h, 501, "Not Implemented",
	                    body501, sizeof(body501) - 1);
	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
}

#ifdef HAS_DUMMY_SERVICE
static void
sendDummyDesc(struct upnphttp * h)
{
	static const char xml_desc[] = "<?xml version=\"1.0\"?>\r\n"
		"<scpd xmlns=\"urn:schemas-upnp-org:service-1-0\">"
		" <specVersion>"
		"    <major>1</major>"
		"    <minor>0</minor>"
		"  </specVersion>"
		"  <actionList />"
		"  <serviceStateTable />"
		"</scpd>\r\n";
	BuildResp_upnphttp(h, xml_desc, sizeof(xml_desc)-1);
	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
}
#endif

/* Sends the description generated by the parameter */
static void
sendXMLdesc(struct upnphttp * h, char * (f)(int *))
{
	char * desc;
	int len;
	desc = f(&len);
	if(!desc)
	{
		static const char error500[] = "<HTML><HEAD><TITLE>Error 500</TITLE>"
		   "</HEAD><BODY>Internal Server Error</BODY></HTML>\r\n";
		syslog(LOG_ERR, "Failed to generate XML description");
		h->respflags = FLAG_HTML;
		BuildResp2_upnphttp(h, 500, "Internal Server Error",
		                    error500, sizeof(error500)-1);
	}
	else
	{
		BuildResp_upnphttp(h, desc, len);
	}
	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
	free(desc);
}

//Luke: UPNP Device Architecture v1.1 defined that <presentationURL> MUST be a relative url
//		Send redirect HTML here
static void
SendHTML(struct upnphttp * h)
{
	char bodyHTML[256] = {};
	
	sprintf(bodyHTML, 
			"<HTML><script language=\"JavaScript\">\n"
			"function redirect(){"
			"var http=\"http://\";"
			"location.href = http+\"%s\";"
			"}\n"
			"</script>"
			"<BODY onLoad=\"redirect()\"></BODY></HTML>\r\n",
			lan_addrs.lh_first->str);
		
	h->respflags = FLAG_HTML;
	BuildResp2_upnphttp(h, 200, "OK",
	                    bodyHTML, sizeof(bodyHTML) - 1);
	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
}

/* ProcessHTTPPOST_upnphttp()
 * executes the SOAP query if it is possible */
static void
ProcessHTTPPOST_upnphttp(struct upnphttp * h)
{
	if((h->req_buflen - h->req_contentoff) >= h->req_contentlen)
	{
		if(h->req_soapAction)
		{
			/* we can process the request */
			syslog(LOG_INFO, "SOAPAction: %.*s",
		    	   h->req_soapActionLen, h->req_soapAction);
			ExecuteSoapAction(h, 
				h->req_soapAction,
				h->req_soapActionLen);
		}
		else
		{
			static const char err400str[] =
				"<html><body>Bad request</body></html>";
			syslog(LOG_INFO, "No SOAPAction in HTTP headers");
			h->respflags = FLAG_HTML;
			BuildResp2_upnphttp(h, 400, "Bad Request",
			                    err400str, sizeof(err400str) - 1);
			SendResp_upnphttp(h);
			CloseSocket_upnphttp(h);
		}
	}
	else
	{
		/* waiting for remaining data */
		h->state = 1;
	}
}

#ifdef ENABLE_EVENTS
/**
 * returns 0 if the callback header value is not valid
 * 1 if it is valid.
 */
static int
checkCallbackURL(struct upnphttp * h)
{
	char addrstr[48];
	int ipv6;
	const char * p;
	int i;

	if(!h->req_Callback || h->req_CallbackLen < 8)
		return 0;
	if(memcmp(h->req_Callback, "http://", 7) != 0)
		return 0;
	ipv6 = 0;
	i = 0;
	p = h->req_Callback + 7;
	if(*p == '[') {
		p++;
		ipv6 = 1;
		while(*p != ']' && i < (sizeof(addrstr)-1)
		      && p < (h->req_Callback + h->req_CallbackLen))
			addrstr[i++] = *(p++);
	} else {
		while(*p != '/' && *p != ':' && i < (sizeof(addrstr)-1)
		      && p < (h->req_Callback + h->req_CallbackLen))
			addrstr[i++] = *(p++);
	}
	addrstr[i] = '\0';
	if(ipv6) {
		struct in6_addr addr;
		if(inet_pton(AF_INET6, addrstr, &addr) <= 0)
			return 0;
#ifdef ENABLE_IPV6
		if(!h->ipv6
		  || (0!=memcmp(&addr, &(h->clientaddr_v6), sizeof(struct in6_addr))))
			return 0;
#else
		return 0;
#endif
	} else {
		struct in_addr addr;
		if(inet_pton(AF_INET, addrstr, &addr) <= 0)
			return 0;
#ifdef ENABLE_IPV6
		if(h->ipv6) {
			if(!IN6_IS_ADDR_V4MAPPED(&(h->clientaddr_v6)))
				return 0;
			if(0!=memcmp(&addr, ((const char *)&(h->clientaddr_v6) + 12), 4))
				return 0;
		} else {
			if(0!=memcmp(&addr, &(h->clientaddr), sizeof(struct in_addr)))
				return 0;
		}
#else
		if(0!=memcmp(&addr, &(h->clientaddr), sizeof(struct in_addr)))
			return 0;
#endif
	}
	return 1;
}

static void
ProcessHTTPSubscribe_upnphttp(struct upnphttp * h, const char * path)
{
	const char * sid;
	syslog(LOG_DEBUG, "ProcessHTTPSubscribe %s", path);
	syslog(LOG_DEBUG, "Callback '%.*s' Timeout=%d",
	       h->req_CallbackLen, h->req_Callback, h->req_Timeout);
	syslog(LOG_DEBUG, "SID '%.*s'", h->req_SIDLen, h->req_SID);
	if(h->req_SID && (h->req_Callback || h->req_NT)){
		/* Luke: UPnP Device Architecture v1.0 said 
				"An SID header field and one of NT or CALLBACK header fields are present." */
		Send400(h);
	} else if((!h->req_SID && !h->req_Callback) ||
			  (h->req_Callback && (strncasecmp(h->req_NT, "upnp:event", h->req_NTLen) != 0))) {
		/* Missing or invalid CALLBACK or not "NT:upnp:event": 412 Precondition Failed.
		 * If CALLBACK header is missing or does not contain a valid HTTP URL,
		 * the publisher must respond with HTTP error 412 Precondition Failed*/
		BuildResp2_upnphttp(h, 412, "Precondition Failed", 0, 0);
		SendResp_upnphttp(h);
		CloseSocket_upnphttp(h);
	} else {
	/* - add to the subscriber list
	 * - respond HTTP/x.x 200 OK 
	 * - Send the initial event message */
/* Server:, SID:; Timeout: Second-(xx|infinite) */
	/* Check that the callback URL is on the same IP as
	 * the request, and not on the internet, nor on ourself (DOS attack ?) */
		if(h->req_Callback) {
			if(checkCallbackURL(h)) {
				sid = upnpevents_addSubscriber(path, h->req_Callback,
				                               h->req_CallbackLen, h->req_Timeout);
				h->respflags = FLAG_TIMEOUT;
				if(sid) {
					syslog(LOG_DEBUG, "generated sid=%s", sid);
					h->respflags |= FLAG_SID;
					h->req_SID = sid;
					h->req_SIDLen = strlen(sid);
				}
				syslog(LOG_DEBUG, "BuildResp_upnphttp start");
				BuildResp_upnphttp(h, 0, 0);
				syslog(LOG_DEBUG, "BuildResp_upnphttp end");
			} else {
				syslog(LOG_WARNING, "Invalid Callback in SUBSCRIBE %.*s",
	       		       h->req_CallbackLen, h->req_Callback);
				BuildResp2_upnphttp(h, 412, "Precondition Failed", 0, 0);
			}
		} else {
			/* subscription renew */
			/* Invalid SID
			   412 Precondition Failed. If a SID does not correspond to a known,
			   un-expired subscription, the publisher must respond
			   with HTTP error 412 Precondition Failed. */
			if(renewSubscription(h->req_SID, h->req_SIDLen, h->req_Timeout) < 0) {
				BuildResp2_upnphttp(h, 412, "Precondition Failed", 0, 0);
			} else {
				h->respflags = FLAG_TIMEOUT;
				h->req_Timeout = 1800;	//Luke: UPnP Device Architecture v1.1 said TIMEOUT should be greater than 1800
				h->respflags |= FLAG_SID;
				BuildResp_upnphttp(h, 0, 0);
			}
		}
		syslog(LOG_DEBUG, "SendResp_upnphttp start");
		SendResp_upnphttp(h);
		syslog(LOG_DEBUG, "CloseSocket_upnphttp start");
		CloseSocket_upnphttp(h);
	}
	syslog(LOG_DEBUG, "ProcessHTTPSubscribe_upnphttp end");
}

static void
ProcessHTTPUnSubscribe_upnphttp(struct upnphttp * h, const char * path)
{
	syslog(LOG_DEBUG, "ProcessHTTPUnSubscribe %s", path);
	syslog(LOG_DEBUG, "SID '%.*s'", h->req_SIDLen, h->req_SID);
	/* Remove from the list */
	if(upnpevents_removeSubscriber(h->req_SID, h->req_SIDLen) < 0) {
		BuildResp2_upnphttp(h, 412, "Precondition Failed", 0, 0);
	} else {
		BuildResp_upnphttp(h, 0, 0);
	}
	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
}
#endif

/* Parse and process Http Query 
 * called once all the HTTP headers have been received. */
static void
ProcessHttpQuery_upnphttp(struct upnphttp * h)
{
	char HttpCommand[16];
	char HttpUrl[512];
	char * HttpVer;
	char * p;
	int i;
	p = h->req_buf;
	if(!p)
		return;
	for(i = 0; i<15 && *p != ' ' && *p != '\r'; i++)
		HttpCommand[i] = *(p++);
	HttpCommand[i] = '\0';
	while(*p==' ')
		p++;
	if(strncmp(p, "http://", 7) == 0)
	{
		p = p+7;
		while(*p!='/')
			p++;
	}
	for(i = 0; i<511 && *p != ' ' && *p != '\r'; i++)
		HttpUrl[i] = *(p++);
	HttpUrl[i] = '\0';
	while(*p==' ')
		p++;
	HttpVer = h->HttpVer;
	for(i = 0; i<15 && *p != '\r'; i++)
		HttpVer[i] = *(p++);
	HttpVer[i] = '\0';
	syslog(LOG_INFO, "HTTP REQUEST : %s %s (%s)",
	       HttpCommand, HttpUrl, HttpVer);
	ParseHttpHeaders(h);
	/* Luke: Don't process POST before get 100 Continue follow-up data */
	if((strcmp("POST", HttpCommand) == 0) && (h->req_continue == 0))
	{
		h->req_command = EPost;
		ProcessHTTPPOST_upnphttp(h);
	}
	else if(strcmp("GET", HttpCommand) == 0)
	{
		if( ((strcmp(h->HttpVer, "HTTP/1.1")==0) && !(h->reqflags & FLAG_HOST)) || (h->reqflags & FLAG_INVALID_REQ) )
		{
			syslog(LOG_WARNING, "Invalid request, responding ERROR 400.  (No Host specified in HTTP headers?)\n");
			Send400(h);
			return;
		}
		h->req_command = EGet;
		if(strcasecmp(ROOTDESC_PATH, HttpUrl) == 0)
		{
			sendXMLdesc(h, genRootDesc);
		}
		else if(strcasecmp(WANIPC_PATH, HttpUrl) == 0)
		{
			sendXMLdesc(h, genWANIPCn);
		}
		else if(strcasecmp(WANCFG_PATH, HttpUrl) == 0)
		{
			sendXMLdesc(h, genWANCfg);
		}
#ifdef HAS_DUMMY_SERVICE
		else if(strcasecmp(DUMMY_PATH, HttpUrl) == 0)
		{
			sendDummyDesc(h);
		}
#endif
#ifdef ENABLE_L3F_SERVICE
		else if(strcasecmp(L3F_PATH, HttpUrl) == 0)
		{
			sendXMLdesc(h, genL3F);
		}
#endif
#ifdef ENABLE_6FC_SERVICE
		else if(strcasecmp(WANIP6FC_PATH, HttpUrl) == 0)
		{
			sendXMLdesc(h, gen6FC);
		}
#endif
#ifdef ENABLE_DP_SERVICE
		else if(strcasecmp(DP_PATH, HttpUrl) == 0)
		{
			sendXMLdesc(h, genDP);
		}
#endif
		//Luke: UPNP Device Architecture v1.1 defined that <presentationURL> MUST be a relative url
		else if(strcasecmp("/index.htm", HttpUrl) == 0){
			SendHTML(h);
		}
		else
		{
			syslog(LOG_NOTICE, "%s not found, responding ERROR 404", HttpUrl);
			Send404(h);
		}
	}
#ifdef ENABLE_EVENTS
	else if(strcmp("SUBSCRIBE", HttpCommand) == 0)
	{
		h->req_command = ESubscribe;
		ProcessHTTPSubscribe_upnphttp(h, HttpUrl);
	}
	else if(strcmp("UNSUBSCRIBE", HttpCommand) == 0)
	{
		h->req_command = EUnSubscribe;
		ProcessHTTPUnSubscribe_upnphttp(h, HttpUrl);
	}
#else
	else if(strcmp("SUBSCRIBE", HttpCommand) == 0)
	{
		syslog(LOG_NOTICE, "SUBSCRIBE not implemented. ENABLE_EVENTS compile option disabled");
		Send501(h);
	}
#endif
	else if(h->req_continue)	//Luke: support HTTP 1.1 100 Continue
	{
		BuildResp2_upnphttp(h, 100, "Continue", 0, 0);
		SendResp_upnphttp(h);
	}
	else
	{
		syslog(LOG_NOTICE, "Unsupported HTTP Command %s", HttpCommand);
		Send501(h);
	}
}


void
Process_upnphttp(struct upnphttp * h)
{
	const char * endheaders;
	char buf[2048];
	int n;
	if(!h)
		return;
		
	switch(h->state)
	{
	case 0:
		n = recv(h->socket, buf, 2048, 0);
		
		if(n<0)
		{
			syslog(LOG_ERR, "recv (state0): %m");
			h->state = 100;
		}
		else if(n==0)
		{
			syslog(LOG_WARNING, "HTTP Connection closed inexpectedly");
			h->state = 100;
		}
		else
		{
			/* if 1st arg of realloc() is null,
			 * realloc behaves the same as malloc() */
			h->req_buf = (char *)realloc(h->req_buf, n + h->req_buflen + 1);
			memcpy(h->req_buf + h->req_buflen, buf, n);
			h->req_buflen += n;
			h->req_buf[h->req_buflen] = '\0';
			/* Luke: HTTP 1.1 pipeline support
					 By RFC 2616 session 4.4
					 "If a request contains a message-body and a Content-Length is not given, 
					 the server SHOULD respond with 400 (bad request) if it cannot determine the length of the message, 
					 or with 411 (length required) if it wishes to insist on receiving a valid Content-Length. "
					 But message-body maybe include the next request at pipeline. In this case, we cannot respond error.
					 So I determine Content-Length is zero if it is not given. */
//			do{
//				h->req_pipeline = 0;
				/* search for the string "\r\n\r\n" */
				endheaders = findendheaders(h->req_buf, h->req_buflen);
				if(endheaders)
				{
					h->req_contentoff = endheaders - h->req_buf + 4;
					ProcessHttpQuery_upnphttp(h);
				}
//			}while(h->req_pipeline);
		}
		break;
	case 1:
		n = recv(h->socket, buf, 2048, 0);
		if(n<0)
		{
			syslog(LOG_ERR, "recv (state1): %m");
			h->state = 100;
		}
		else if(n==0)
		{
			syslog(LOG_WARNING, "HTTP Connection closed inexpectedly");
			h->state = 100;
		}
		else
		{
			/*fwrite(buf, 1, n, stdout);*/	/* debug */
			h->req_buf = (char *)realloc(h->req_buf, n + h->req_buflen);
			memcpy(h->req_buf + h->req_buflen, buf, n);
			h->req_buflen += n;
			if((h->req_buflen - h->req_contentoff) >= h->req_contentlen)
			{
				ProcessHTTPPOST_upnphttp(h);
			}
			/* Luke: process next request if it is a pipeline connection */
//			while(h->req_pipeline){
//				h->req_pipeline = 0;
//				endheaders = findendheaders(h->req_buf, h->req_buflen);
//				if(endheaders)
//				{
//					h->req_contentoff = endheaders - h->req_buf + 4;		
//					ProcessHttpQuery_upnphttp(h);
//				}
//			};
		}
		break;
	default:
		syslog(LOG_WARNING, "Unexpected state: %d", h->state);
	}
}

static const char httpresphead[] =
	"%s %d %s\r\n"
	/*"Content-Type: text/xml; charset=\"utf-8\"\r\n"*/
	"Content-Type: %s\r\n"
	"%s"	//Connection: close\r\n
	"Content-Length: %d\r\n"
	"Server: " MINIUPNPD_SERVER_STRING "\r\n"
	;	/*"\r\n";*/
/*
		"<?xml version=\"1.0\"?>\n"
		"<s:Envelope xmlns:s=\"http://schemas.xmlsoap.org/soap/envelope/\" "
		"s:encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\">"
		"<s:Body>"

		"</s:Body>"
		"</s:Envelope>";
*/
/* with response code and response message
 * also allocate enough memory */

void
BuildHeader_upnphttp(struct upnphttp * h, int respcode,
                     const char * respmsg,
                     int bodylen)
{
	int templen;
	if(!h->res_buf)
	{
		templen = sizeof(httpresphead) + 192 + bodylen;
		h->res_buf = (char *)malloc(templen);
		if(!h->res_buf)
		{
			syslog(LOG_ERR, "malloc error in BuildHeader_upnphttp()");
			return;
		}
		memset(h->res_buf, 0, templen);
		h->res_buf_alloclen = templen;
	}
	/* Luke: 100-continue has no additional headers */
	if(respcode == 100){
		h->res_buflen = snprintf(h->res_buf, h->res_buf_alloclen, "HTTP/1.1 100 Continue\r\n");
		goto BuildHeader_done;
	}
	syslog(LOG_DEBUG, "BuildHeader 100-continue done");
	h->res_buflen = snprintf(h->res_buf, h->res_buf_alloclen,
	                         //httpresphead, h->HttpVer,
	                         httpresphead, "HTTP/1.1",
	                         respcode, respmsg,
	                         (h->respflags&FLAG_HTML)?"text/html":"text/xml; charset=\"utf-8\"",
							 (h->req_ConnectionClose == 0)?"":"Connection: close\r\n",
							 bodylen);
	/* Additional headers */
#ifdef ENABLE_EVENTS
	if(h->respflags & FLAG_TIMEOUT) {
		h->res_buflen += snprintf(h->res_buf + h->res_buflen,
		                          h->res_buf_alloclen - h->res_buflen,
		                          "Timeout: Second-");
		if(h->req_Timeout) {
			h->res_buflen += snprintf(h->res_buf + h->res_buflen,
			                          h->res_buf_alloclen - h->res_buflen,
			                          "%d\r\n", h->req_Timeout);
		} else {
			h->res_buflen += snprintf(h->res_buf + h->res_buflen,
			                          h->res_buf_alloclen - h->res_buflen,
			                          "1800\r\n");
						  /* Luke: UPnP Device Architecture v1.1 said:
							  "TIMEOUT SHOULD be greater than or equal to 1800 seconds" */
		}
	}
	if(h->respflags & FLAG_SID) {
		h->res_buflen += snprintf(h->res_buf + h->res_buflen,
		                          h->res_buf_alloclen - h->res_buflen,
		                          "SID: %.*s\r\n", h->req_SIDLen, h->req_SID);
	}
	if(h->req_AcceptLang){
		h->res_buflen += snprintf(h->res_buf + h->res_buflen,
		                          h->res_buf_alloclen - h->res_buflen,
		                          "CONTENT-LANGUAGE: %.*s\r\n", h->req_AcceptLangLen, h->req_AcceptLang);	
	}
	syslog(LOG_DEBUG, "BuildHeader DATE");
	/* Luke: For UPnP Device Architecture v1.1, DATE is RECOMMAND item */
	char   szTime[30];
	time_t curtime = time(NULL);
	strftime(szTime, 30,"%a, %d %b %Y %H:%M:%S GMT" , gmtime(&curtime));
	h->res_buflen += snprintf(h->res_buf + h->res_buflen,
	                          h->res_buf_alloclen - h->res_buflen,
	                          "Date: %s\r\n", szTime);
	h->res_buflen += snprintf(h->res_buf + h->res_buflen,
	                          h->res_buf_alloclen - h->res_buflen,
	                          "contentFeatures.dlna.org: \r\n");
	h->res_buflen += snprintf(h->res_buf + h->res_buflen,
	                          h->res_buf_alloclen - h->res_buflen,
	                          "EXT:\r\n");
#endif
BuildHeader_done:
	syslog(LOG_DEBUG, "goto BuildHeader_done: h->res_buf_alloclen=%d  h->res_buflen=%d   bodylen=%d ", h->res_buf_alloclen, h->res_buflen, bodylen);
	h->res_buf[h->res_buflen++] = '\r';
	h->res_buf[h->res_buflen++] = '\n';
	syslog(LOG_DEBUG, "if(h->res_buf_alloclen < (h->res_buflen + bodylen))");
	if(h->res_buf_alloclen < (h->res_buflen + bodylen))
	{
		char * tmp;
		tmp = (char *)realloc(h->res_buf, (h->res_buflen + bodylen));
		if(tmp)
		{
			syslog(LOG_DEBUG, "realloc");
			h->res_buf = tmp;
			h->res_buf_alloclen = h->res_buflen + bodylen;
		}
		else
		{
			syslog(LOG_ERR, "realloc error in BuildHeader_upnphttp()");
		}
	}
	syslog(LOG_DEBUG, "BuildHeader_done");
}

void
BuildResp2_upnphttp(struct upnphttp * h, int respcode,
                    const char * respmsg,
                    const char * body, int bodylen)
{
	syslog(LOG_DEBUG, "BuildHeader_upnphttp start");
	BuildHeader_upnphttp(h, respcode, respmsg, bodylen);
	syslog(LOG_DEBUG, "BuildHeader_upnphttp end");
	if(body)
		memcpy(h->res_buf + h->res_buflen, body, bodylen);
	h->res_buflen += bodylen;
}

/* responding 200 OK ! */
void
BuildResp_upnphttp(struct upnphttp * h,
                        const char * body, int bodylen)
{
	BuildResp2_upnphttp(h, 200, "OK", body, bodylen);
}

void
SendResp_upnphttp(struct upnphttp * h)
{
	char * p;
	ssize_t n;
	size_t len;
	p = h->res_buf;
	len = h->res_buflen;
	while (len > 0)
	{
		n = send(h->socket, p, len, 0);
		if(n<0)
		{
			syslog(LOG_ERR, "send(res_buf): %m");
			break;
		}
		else if(n == 0)
		{
			syslog(LOG_ERR, "send(res_buf): %zd bytes sent (out of %zu)",
							n, len);
			break;
		}
		else
		{
			p += n;
			len -= n;
		}
	}
}

