#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <mtd/mtd-user.h>
#include "nvram.h"

#include "nvram_reserve.h" //copy form www

static char *chop(char *io)
{
	char *p;

	p = io + strlen(io) - 1;
	while ( *p == '\n' || *p == '\r') {
		*p = '\0';
		p--;
	}
	return io;
}

static int str2nvram(char *s, struct nvram_tuple *nv)
{
	char *p;
	
	chop(s);
	if (*s == '#' || *s == ' ' || *s == '\t' || *s == '/')
		goto out;
	if ((p = strsep(&s, "=")) == NULL)
		goto out;
	if (s == NULL)
		goto out;
	if ((nv->name = strdup(p)) == NULL)
		goto out;
	if ((nv->value = strdup(s)) == NULL)
		goto nomem2;
	return 0;
nomem2:
	free(nv->name);
out:
	return -1;
}

struct nvram_tuple *init_nvram_tuples(const char *filename)
{
	FILE *fd;
	char buf[4096];
	struct nvram_tuple *np, *n = NULL;
#if 0 // in nvram_reserve.h
	int need_reserve=0;
	int reserve_item_count=0;
#endif	

	if ((fd = fopen(filename, "r")) == NULL)
		goto out;
		
	while (fgets(buf, sizeof(buf), fd) != NULL) {
		
		if ((np = malloc(sizeof(struct nvram_tuple))) == NULL)
			goto out;
			
		if (str2nvram(buf, np) == -1)
			continue;
			
#if 0
		reserve_item_count=0;
		
		while( nvram_reserve[reserve_item_count] != NULL ) 
		{
			if( strncmp(np->name, nvram_reserve[reserve_item_count], strlen(np->name)) == 0 )
			{
				need_reserve = 1;
				break;
			}	
			reserve_item_count ++;	
		}
		
		if(need_reserve)
		{
			printf("init_nvram_tuples: reserve %s value\n", nvram_reserve[reserve_item_count]);
			need_reserve = 0;
			continue;
		}	
#endif			
			
		np->next = n;
		n = np;
	}
out:
	if (fd != NULL)
		fclose(fd);
	return n;
}

void uninit_nvram_tuples(struct nvram_tuple *p)
{
	struct nvram_tuple *n;
	
	while (p != NULL) {
		n = p->next;
		free(p->name);
		free(p->value);
		free(p);
		p = n;
	}
	
}

static void fn_nvram_set(const char *k, const char *v, void *unused)
{
	nvram_set(k, v);
}
/**********************************
 * extension helper
 * ********************************/

int foreach_nvram_from(const char *file, void (*fn)(const char *key, const char *value, void *data), void *data)
{
	struct nvram_tuple *n, *h;
	
#if NVRAM_RESERVE // in nvram_reserve.h
	int need_reserve=0;
	int reserve_item_count=0;
	int enable_nvram_reserve=1;
	
	if( data!=NULL && strcmp(data, "rc_init_nvram")==0 )
		enable_nvram_reserve=0;
#endif	
	
	if ((h = init_nvram_tuples(file)) == NULL)
		return -1;
		
	fprintf(stderr, "openfile :%s\n", file);
	for (n = h;n != NULL; n = n->next) {
		/*
			Skip the checksum record we append in the config file when restore config via GUI
		*/
		if(strstr(n->name,"config_checksum")){
			continue;
		}
#if NVRAM_RESERVE
		if(enable_nvram_reserve==1)
		{
			reserve_item_count=0;
		
			while( nvram_reserve[reserve_item_count] != NULL ) {
				if( strncmp(n->name, nvram_reserve[reserve_item_count], strlen(n->name)) == 0 ){
					need_reserve = 1;
					break;
				}	
				reserve_item_count ++;	
			}
		
			if(need_reserve){
				printf("foreach_nvram_from: reserve %s value\n", nvram_reserve[reserve_item_count]);
				need_reserve = 0;
				continue;
			}
		}	
#endif		
		
		fn(n->name, n->value, data);
	}
	uninit_nvram_tuples(h);
	return 0;
}

int foreach_nvram_from_mtd(void (*fn)(const char *key, const char *value, void *data), void *data)
{
	char *name, *k, *v, *buf;
	int nvram_space, rev = -1;
	
	if ((nvram_space = nvram_get_nvramspace()) <= 0) {
		fprintf(stderr, "can not get space size from nvram device\n");
			return -1;
	}
	if ((buf = malloc(nvram_space)) == NULL) {
		fprintf(stderr, "can not malloc()\n");
			return -1;
	}

	//fprintf(stderr, "%s:%d, size: %d\n", __FUNCTION__, __LINE__, sizeof(buf));
	if (nvram_getall(buf, nvram_space) != 0)
		goto out;
	//for (name = buf; *name; name += strlen(name) + 1) {
	for (name = buf; *name; ) {
		v = name;
		name += strlen(name) + 1;	/* next key value */
		k = strsep(&v, "=");
		//fprintf(stderr, "%s:%d: [%s:%s]\n", __FUNCTION__, __LINE__, k, v);
		fn(k, v, data);
	}
	rev = 0;
out:
	free(buf);
	return rev;
}

/*************************************
 * public shared libraries
 * ***********************************/
/* @argv[0] := eraseall
 * @argv[1] := $(MTDDEV)
 * @argv[2] := NULL
 * */
int nvram_eraseall(char **argv)
{
	int i = 0, fd = -1, ret = -1;
	mtd_info_t mtd_info;
	erase_info_t erase_info;

	if (argv[1] == NULL)
		goto out;
	if (access(argv[1], F_OK) != 0)
		goto out;

	if ((fd = open(argv[1], O_SYNC | O_RDWR)) < 0)
		goto out;
	if (ioctl(fd, MEMGETINFO, &mtd_info) != 0)
		goto out;

	erase_info.start = 0;
	erase_info.length = mtd_info.erasesize;

	for (; i < mtd_info.size / mtd_info.erasesize; i++) {
		if (ioctl(fd, MEMERASE, &erase_info) != 0)
			goto out;
		erase_info.start += mtd_info.erasesize;
	}

	ret = 0;
out:
	printf("%s Size: %08X (%08X erased)\n",
		argv[1],
		mtd_info.size,
		erase_info.start);

	if (fd != -1) close(fd);
	if (errno) perror("eraseall");
	return ret;
}

int nvram_restore_default(void)
{
	int res = 0;

	printf("linux_nvram: restore default\n");
	if((res = foreach_nvram_from(NVRAM_DEFAULT, fn_nvram_set, "nvram_restore_default")) == 0)
		nvram_commit();
	return res;
}

int nvram_compatible_args(char **argv, int *rev)
{
	if (!strncmp(*argv, "restore_default", 15)) {
		*rev = nvram_restore_default();
	} else if (!strncmp(*argv, "eraseall", 8)) {
		nvram_eraseall(argv++);
	} else if (!strncmp(*argv, "upgrade", 7)) {
		nvram_upgrade();
		nvram_commit();
	} else
		return -1;	//no args processed.
	return 0;
}

int nvram_upgrade(void)
{
	int rev = -1;

	if ((rev = foreach_nvram_from(NVRAM_UPGRADE_TMP, fn_nvram_set, "nvram_upgrade")) == -1)
		rev = nvram_restore_default();
	else	/*XXX: The file should be removed by caller */
		if (rev == 0)
			remove(NVRAM_UPGRADE_TMP);
	
	/* XXX: reboot now? Has IE get the return count down page yet? */
	/* Don't reboot here, httpd will call reboot after configuration upload success */
#if 0
	if(rev == 0)
		kill(1, SIGTERM);
#endif
	return rev;
}

static void fn_nvram_download(const char *k, const char *v, void *data)
{
	char  *protection[] = {
		NULL,
		"wan_mac",
		"lan_mac",
		"lan_eth",
		"lan_bridge",
		"wan_eth",
		"wlan0_domain",
		"asp_temp_",
		"html_",
		NULL
	}, **p;
	FILE *fd = data;
	int protect = 0;
	
	//fprintf(fd, "%s=:%s\n", k, v == NULL ? "" : k);
	for (p = protection; *p != NULL; p++) {
		if (strcmp(k, *p) != 0)
			continue;
		protect = 1;
		break;
	}
	if (!protect)
		fprintf(fd, "%s=%s\n", k, v == NULL ? "" : v);
	else
		fprintf(stderr, "ignored : [%s:%s]\n", k, v);
}

int nvram_config2file(void)
{
	FILE *fd;
	int rev = -1;
	
	/* XXX: should not be hard code! */
	if ((fd = fopen(NVRAM_CONFIG_TMP, "a")) == NULL)
		return -1;
	rev = foreach_nvram_from_mtd(fn_nvram_download, fd);
	fclose(fd);
	return rev;
}
