#include "request.h"
#include "keyvalue.h"
#include "log.h"

#include <sys/stat.h>

#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#ifdef CONFIG_HNAP
#include "hnap.h"
#endif //CONFIG_HNAP

#ifdef CONFIG_SUPPORT_WEBFILE_ACCESS
char ws_action[20];
#endif // CONFIG_SUPPORT_WEBFILE_ACCESS

static int request_check_hostname(server *srv, connection *con, buffer *host) {
	enum { DOMAINLABEL, TOPLABEL } stage = TOPLABEL;
	size_t i;
	int label_len = 0;
	size_t host_len;
	char *colon;
	int is_ip = -1; /* -1 don't know yet, 0 no, 1 yes */
	int level = 0;
	struct in6_addr pf_in6;

	UNUSED(srv);
	UNUSED(con);

	/*
	 *       hostport      = host [ ":" port ]
	 *       host          = hostname | IPv4address | IPv6address
	 *       hostname      = *( domainlabel "." ) toplabel [ "." ]
	 *       domainlabel   = alphanum | alphanum *( alphanum | "-" ) alphanum
	 *       toplabel      = alpha | alpha *( alphanum | "-" ) alphanum
	 *       IPv4address   = 1*digit "." 1*digit "." 1*digit "." 1*digit
	 *       IPv6address   = "[" ... "]"
	 *       port          = *digit
	 */

	/* no Host: */
	if (!host || host->used == 0) return 0;

	host_len = host->used - 1;

	colon = memchr(host->ptr, ':', host_len);
	
	/* Luke: fix IPv6 GUI fail
			 Host which sending by Apple safari is not include [] , I check it */
	if((host->ptr[0] != '[') && (colon != NULL)){
		if(inet_pton(AF_INET6, host->ptr, &pf_in6)){	//inet_pton return 1 if host->ptr is IPv6 address
			//IPv6 address
			return 0;
		}
	}

	/* IPv6 adress */
	if (host->ptr[0] == '[') {
		char *c = host->ptr + 1;
		int colon_cnt = 0;

		/* check portnumber */
		for (; *c && *c != ']'; c++) {
			if (*c == ':') {
				if (++colon_cnt > 7) {
					return -1;
				}
			} else if (!light_isxdigit(*c)) {
				return -1;
			}
		}

		/* missing ] */
		if (!*c) {
			return -1;
		}

		/* check port */
		if (*(c+1) == ':') {
			for (c += 2; *c; c++) {
				if (!light_isdigit(*c)) {
					return -1;
				}
			}
		}
		return 0;
	}

	if (NULL != colon) {
		char *c = colon + 1;

		/* check portnumber */
		for (; *c; c++) {
			if (!light_isdigit(*c)) return -1;
		}

		/* remove the port from the host-len */
		host_len = colon - host->ptr;
	}

	/* Host is empty */
	if (host_len == 0) return -1;

	/* if the hostname ends in a "." strip it */
	if (host->ptr[host_len-1] == '.') {
		/* shift port info one left */
		if (NULL != colon) memmove(colon-1, colon, host->used - host_len);
		else host->ptr[host_len-1] = '\0';
		host_len -= 1;
		host->used -= 1;
	}

	if (host_len == 0) return -1;

	/* scan from the right and skip the \0 */
	for (i = host_len; i-- > 0; ) {
		const char c = host->ptr[i];

		switch (stage) {
		case TOPLABEL:
			if (c == '.') {
				/* only switch stage, if this is not the last character */
				if (i != host_len - 1) {
					if (label_len == 0) {
						return -1;
					}

					/* check the first character at right of the dot */
					if (is_ip == 0) {
						if (!light_isalnum(host->ptr[i+1])) {
							return -1;
						}
					} else if (!light_isdigit(host->ptr[i+1])) {
						is_ip = 0;
					} else if ('-' == host->ptr[i+1]) {
						return -1;
					} else {
						/* just digits */
						is_ip = 1;
					}

					stage = DOMAINLABEL;

					label_len = 0;
					level++;
				} else if (i == 0) {
					/* just a dot and nothing else is evil */
					return -1;
				}
			} else if (i == 0) {
				/* the first character of the hostname */
				if (!light_isalnum(c)) {
					return -1;
				}
				label_len++;
			} else {
				if (c != '-' && !light_isalnum(c)) {
					return -1;
				}
				if (is_ip == -1) {
					if (!light_isdigit(c)) is_ip = 0;
				}
				label_len++;
			}

			break;
		case DOMAINLABEL:
			if (is_ip == 1) {
				if (c == '.') {
					if (label_len == 0) {
						return -1;
					}

					label_len = 0;
					level++;
				} else if (!light_isdigit(c)) {
					return -1;
				} else {
					label_len++;
				}
			} else {
				if (c == '.') {
					if (label_len == 0) {
						return -1;
					}

					/* c is either - or alphanum here */
					if ('-' == host->ptr[i+1]) {
						return -1;
					}

					label_len = 0;
					level++;
				} else if (i == 0) {
					if (!light_isalnum(c)) {
						return -1;
					}
					label_len++;
				} else {
					if (c != '-' && !light_isalnum(c)) {
						return -1;
					}
					label_len++;
				}
			}

			break;
		}
	}

	/* a IP has to consist of 4 parts */
	if (is_ip == 1 && level != 3) {
		return -1;
	}

	if (label_len == 0) {
		return -1;
	}

	return 0;
}

#if 0
#define DUMP_HEADER
#endif

static int http_request_split_value(array *vals, buffer *b) {
	char *s;
	size_t i;
	int state = 0;
	/*
	 * parse
	 *
	 * val1, val2, val3, val4
	 *
	 * into a array (more or less a explode() incl. striping of whitespaces
	 */

	if (b->used == 0) return 0;

	s = b->ptr;

	for (i =0; i < b->used - 1; ) {
		char *start = NULL, *end = NULL;
		data_string *ds;

		switch (state) {
		case 0: /* ws */

			/* skip ws */
			for (; (*s == ' ' || *s == '\t') && i < b->used - 1; i++, s++);


			state = 1;
			break;
		case 1: /* value */
			start = s;

			for (; *s != ',' && i < b->used - 1; i++, s++);
			end = s - 1;

			for (; (*end == ' ' || *end == '\t') && end > start; end--);

			if (NULL == (ds = (data_string *)array_get_unused_element(vals, TYPE_STRING))) {
				ds = data_string_init();
			}

			buffer_copy_string_len(ds->value, start, end-start+1);
			array_insert_unique(vals, (data_unset *)ds);

			if (*s == ',') {
				state = 0;
				i++;
				s++;
			} else {
				/* end of string */

				state = 2;
			}
			break;
		default:
			i++;
			break;
		}
	}
	return 0;
}

static int request_uri_is_valid_char(unsigned char c) {
	if (c <= 32) return 0;
	if (c == 127) return 0;
	if (c == 255) return 0;

	return 1;
}

#ifdef CONFIG_HNAP
char auth_state;		// 0: authorized, 1: Unauthorized
char soap_action[32];

const static int b64_decode_table[256] = {
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 00-0F */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 10-1F */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,62,-1,-1,-1,63,  /* 20-2F */
52,53,54,55,56,57,58,59,60,61,-1,-1,-1,-1,-1,-1,  /* 30-3F */
-1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,  /* 40-4F */
15,16,17,18,19,20,21,22,23,24,25,-1,-1,-1,-1,-1,  /* 50-5F */
-1,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,  /* 60-6F */
41,42,43,44,45,46,47,48,49,50,51,-1,-1,-1,-1,-1,  /* 70-7F */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 80-8F */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 90-9F */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* A0-AF */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* B0-BF */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* C0-CF */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* D0-DF */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* E0-EF */
-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1   /* F0-FF */
};

/* Do base-64 decoding on a string.  Ignore any non-base64 bytes.
** Return the actual number of bytes generated.  The decoded size will
** be at most 3/4 the size of the encoded, and may be smaller if there
** are padding characters (blanks, newlines).
*/
int b64_decode( const char* str, unsigned char* space, int size ){
	const char* cp;
	int space_idx, phase;
	int d, prev_d=0;
	unsigned char c;
	
	space_idx = 0;
	phase = 0;
	if(str==NULL)
		return space_idx;
	
	for ( cp = str; *cp != '\0'; ++cp )
	{
		d = b64_decode_table[(int)*cp];
		if ( d != -1 )
		{
			switch ( phase )
			{
				case 0:
					++phase;
					break;
				case 1:
					c = ( ( prev_d << 2 ) | ( ( d & 0x30 ) >> 4 ) );
					if ( space_idx < size )
						space[space_idx++] = c;
					++phase;
					break;
				case 2:
					c = ( ( ( prev_d & 0xf ) << 4 ) | ( ( d & 0x3c ) >> 2 ) );
					if ( space_idx < size )
						space[space_idx++] = c;
					++phase;
					break;
				case 3:
					c = ( ( ( prev_d & 0x03 ) << 6 ) | d );
					if ( space_idx < size )
						space[space_idx++] = c;
					phase = 0;
					break;
			}
			prev_d = d;
		}
	}
	return space_idx;
}

int hnap_auth_check(char* authorization){
	char authinfo[500];
	char auth_user[100];
	char* auth_pass = NULL;
	char* admin_passwd = NULL;
	int l;
	FILE *fp;
	char buf[256];
	char *delim = ":";
	
	/* Basic authorization info? */
	if ( authorization == NULL || strncmp( authorization, "Basic ", 6 ) != 0 ) {
		return 0;
	}

	/* Decode it. */
	memset(authinfo, 0, sizeof(authinfo));
	l = b64_decode( &(authorization[6]), authinfo, sizeof(authinfo) );
	authinfo[l] = '\0';

	/* Split into user and password. */
	auth_pass = strchr(authinfo, ':');
	*auth_pass++ = '\0';

	/* check auth user name */
	if(strlen(authinfo) == 0){
		return 0;
	}
		
	memset(auth_user, 0, sizeof(auth_user));
	strcpy(auth_user, strtok(authinfo, delim));
	
	/* get username & password from /etc/lighttpd/lighttpd.user */
	memset(buf, 0, sizeof(buf));
	fp = fopen("/etc/lighttpd/lighttpd.user", "r");
	if(fp != NULL){
		if(fgets(buf, sizeof(buf), fp) != NULL){
			
			admin_passwd = strchr(buf, ':');
			*admin_passwd++ = '\0';
			
		}		
		fclose(fp);
	}	

	/* check password */
	if (strcmp(admin_passwd, auth_pass) == 0) {
		return 1;
	}

	return 0;
}


int hnap_support_list_check(char* soapaction){
	int k = 0;
	int return_val = 0;
	
	if(strstr(soapaction, "Method")){
		return 0;
	}
	
	for(k = 0; k < HNAP_SUPPORT_LIST_MAX; k++){
		if(strstr(soapaction, hnap_support_list[k])){						
			return_val = 1;
			strcpy(soap_action, hnap_support_list[k]);
			break;
		} 
	}	
		
	return return_val;
}

#endif //CONFIG_HNAP

int check_mp_file(){
	FILE *mp_file = NULL;
	char buf[5];
	int byte_read;
	int mode = 0;
	
	mp_file = fopen("/tmp/test_mode", "r");	// the file path must be as same as MP_TEST_FILE in public_util.h
	if(mp_file != NULL){
		memset(buf, 0, sizeof(buf));			
		byte_read = fread(buf, sizeof(char), sizeof(buf), mp_file);
		
		if (byte_read > 0){ 
			mode = atoi(buf);
		}
		
		fclose(mp_file);
	}
	
	return mode;	
}

#ifdef CONFIG_SUPPORT_WEBFILE_ACCESS
void set_ws_action(char *src){
	char *ptr = strchr(src, '?');
	int len = 0;
			
	if (ptr != NULL){
		if (ptr > src){
			len = ptr - src;
		}else{
			len = src - ptr;
		}
			
		memcpy(ws_action, src, len);		
	}else if (strncmp(src, "Login", 5) == 0){
		strcpy(ws_action, "Login");
	}
}

#define ENCODING_LEN		25
char *encoding_code[] = {"20", "22", "23", "24", "25", "26", "2B", "2C", "2F", "3A", "3B", "3C", "3D", "3E", "3F", "40", "5B", "5C", "5D", "5E", "60", "7B", "7C", "7D", "7E"};
char encoding_char[] = {' ', '"', '#', '$', '%', '&', '+', ',', '/', ':', ';', '<', '=', '>', '?', '@', '[', '\\', ']', '^', '`', '{', '|', '}', '~'};

#define UID_COUNT_FILE		"/var/tmp/uid_count"
#define UID_INFO_FILE		"/var/tmp/uid_info"
#define WS_LOGIN_INFO_FILE	"/var/tmp/ws_login_info"

#define MAX_NODE_NUM		5
#define MAX_USER_NUM		10
#define MAX_SESSION_NUM		4
#define MAX_PATH_LEN 		1024
#define MAX_SESSION_LEN		8
#define MAX_USER_NAME_LEN	15		// must match the maxlength of storage_name on GUI

typedef struct{		
    char name[MAX_USER_NAME_LEN+1];   
} login_account;

typedef struct{		
	char uid[MAX_SESSION_LEN+1];
	unsigned long login_time;
}uid_info;

typedef struct{		
	char path[MAX_PATH_LEN];
	char permission[3];		// ro: Read Only, rw: Read/Write
}path_info;

typedef struct{	
	int level;		// 0: admin, 1: Guest, 2: other users
	int path_num;
	uid_info session_uid[MAX_SESSION_NUM];
	path_info access_path[MAX_NODE_NUM];
}login_user;

int check_access(server *srv, char *which_path, login_user *which_user){
	char path[MAX_PATH_LEN];
	char check_path[MAX_PATH_LEN];
	int len = 0;
	int result = 0;
	int i;
		
	for (i = 0; i < strlen(which_path); i++){	
		if (which_path[i] == '/'){	// get the last position of '/'
			len = i;
		}
	}	
	
	memset(path, 0, sizeof(path));
	memcpy(path, which_path, len);
		
	for (i = 0; i < which_user->path_num; i++){
		memset(check_path, 0, sizeof(MAX_PATH_LEN));
		sprintf(check_path, "%s", which_user->access_path[i].path);
			
		if (strlen(path) >= strlen(check_path)){		
			if (strstr(path, check_path)){
				result = 1;
				break;
			}
		}
	}	
	
	return result;
}

inline int ishex(int x){	
	return	(x >= '0' && x <= '9')	||	(x >= 'a' && x <= 'f')	||	(x >= 'A' && x <= 'F');
			}			
							
int decode(char *s, char *dec){	
	char *o, *end = s + strlen(s);	
	int c;
 	
	for (o = dec; s <= end; o++) {		
		c = *s++;
		if (c == '%20'){
			c = ' ';
		}else if (c == '%' && (!ishex(*s++)	|| !ishex(*s++)|| !sscanf(s - 2, "%2x", &c))){
			return -1; 		
		}
		if (dec){
			*o = c;	
	}
	}
	return o - dec;
}

int check_dws_cookie(server *srv, connection *con){	
	FILE *login_file = NULL;
	FILE *uid_file = NULL;
	login_user current_login_user;
	login_account users[MAX_USER_NUM];
	char user_name[MAX_USER_NAME_LEN+1];
	data_string *ds;
	char *addr;
	char *cookie_ptr = NULL;
	char path[MAX_PATH_LEN];
	char decode_path[MAX_PATH_LEN];
	char file_name[50];
	char cookie[13];	
	int auth = 0;
	int count = 0;
	int i, j;
	
#ifdef HAVE_IPV6
	char b2[INET6_ADDRSTRLEN + 1];
#endif
	
	if (strncmp(con->request.uri->ptr, "/usb_dev", 8) == 0){	
		memset(path, 0, sizeof(path));
		strcpy(path, con->request.uri->ptr);
		decode(&path, &decode_path);
		
			memset(cookie, 0, sizeof(cookie));
		cookie_ptr = strstr(decode_path, "?uid=");
					
		if (cookie_ptr != NULL){				
			strcpy(cookie, (cookie_ptr+1));
		}else if (NULL != (ds = (data_string *)array_get_element(con->request.headers, "Cookie"))){			
			memcpy(cookie, ds->value->ptr, sizeof(cookie) - 1);				
		}
		
		if (strlen(cookie) > 0){
				
			switch (con->dst_addr.plain.sa_family) {
#ifdef HAVE_IPV6
				case AF_INET6:
					addr = inet_ntop(con->dst_addr.plain.sa_family,
							  (const void *) &(con->dst_addr.ipv6.sin6_addr),
							  b2, sizeof(b2)-1);
					break;
				case AF_INET:
					addr = inet_ntop(con->dst_addr.plain.sa_family,
							  (const void *) &(con->dst_addr.ipv4.sin_addr),
							  b2, sizeof(b2)-1);
					break;
#else
				case AF_INET:
					addr = inet_ntoa(con->dst_addr.ipv4.sin_addr);
					break;
#endif
				default:
					addr = "";
					break;
			}
					
			if (strlen(addr) > 0){
				login_file = fopen(WS_LOGIN_INFO_FILE, "r");
				if (login_file){
					memset(user_name, 0, MAX_USER_NAME_LEN+1);
					while(fgets(user_name, MAX_USER_NAME_LEN, login_file) != NULL){
						user_name[strlen(user_name)-1] = '\0'; // remove the last char:'\n'
						strcpy(users[count].name, user_name);
						count = count+1;
					}
					fclose(login_file);
				}else{	//keyin the URL to get GUI, but not login
					return auth;
				}
				
				for (i = 0; i < count; i++){
					memset(file_name, 0, 50);
					sprintf(file_name, "%s_%s", UID_INFO_FILE, users[i].name);
					
					uid_file = fopen(file_name, "r");
					if (uid_file){
						fread(&current_login_user, sizeof(login_user), 1, uid_file);
						fclose(uid_file);
						for (j = 0; j < MAX_SESSION_NUM; j++){
							if (strcmp(current_login_user.session_uid[j].uid, "") != 0){
								if (strncmp(&cookie[4], current_login_user.session_uid[j].uid, MAX_SESSION_LEN) == 0){
									if (current_login_user.level != 0){
										auth = check_access(srv, &decode_path[8], &current_login_user);
							}else{
								auth = 1;
							}
									break;
								}
							}
						}
					}
											
					if(auth){
						break;
					}
				}
			}
		}
	}else{	// when users are not trying to access files in the usb devices
		auth = 1;
	}
	
	return auth;
}
#endif	// CONFIG_SUPPORT_WEBFILE_ACCESS

int http_request_parse(server *srv, connection *con) {
	char *uri = NULL, *proto = NULL, *method = NULL, con_length_set;
	int is_key = 1, key_len = 0, is_ws_after_key = 0, in_folding;
	char *value = NULL, *key = NULL;
	char *reqline_host = NULL;
	int reqline_hostlen = 0;

	enum { HTTP_CONNECTION_UNSET, HTTP_CONNECTION_KEEPALIVE, HTTP_CONNECTION_CLOSE } keep_alive_set = HTTP_CONNECTION_UNSET;

	int line = 0;

	int request_line_stage = 0;
	size_t i, first;

	int done = 0;

#ifdef CONFIG_HNAP	
	char soapaction[256];
	char tmp_authorization[512];
	int is_soap = 0;
	int is_get_hnap = 0;
	int auth_found = 0;
	int ignore_password = 0;
		
	memset(soap_action, 0, sizeof(soap_action));	
	memset(soapaction, 0, sizeof(soapaction));
	memset(tmp_authorization, 0, sizeof(tmp_authorization));
	
	auth_state = 0;
#endif

	/*
	 * Request: "^(GET|POST|HEAD) ([^ ]+(\\?[^ ]+|)) (HTTP/1\\.[01])$"
	 * Option : "^([-a-zA-Z]+): (.+)$"
	 * End    : "^$"
	 */

	if (con->conf.log_request_header) {
		log_error_write(srv, __FILE__, __LINE__, "sdsdSb",
				"fd:", con->fd,
				"request-len:", con->request.request->used,
				"\n", con->request.request);
	}

	if (con->request_count > 1 &&
	    con->request.request->ptr[0] == '\r' &&
	    con->request.request->ptr[1] == '\n') {
		/* we are in keep-alive and might get \r\n after a previous POST request.*/

		buffer_copy_string_len(con->parse_request, con->request.request->ptr + 2, con->request.request->used - 1 - 2);
	} else {
		/* fill the local request buffer */
		buffer_copy_string_buffer(con->parse_request, con->request.request);
	}

	keep_alive_set = 0;
	con_length_set = 0;

	/* parse the first line of the request
	 *
	 * should be:
	 *
	 * <method> <uri> <protocol>\r\n
	 * */
	for (i = 0, first = 0; i < con->parse_request->used && line == 0; i++) {
		char *cur = con->parse_request->ptr + i;

		switch(*cur) {
		case '\r':
			if (con->parse_request->ptr[i+1] == '\n') {
				http_method_t r;
				char *nuri = NULL;
				size_t j;

				/* \r\n -> \0\0 */
				con->parse_request->ptr[i] = '\0';
				con->parse_request->ptr[i+1] = '\0';

				buffer_copy_string_len(con->request.request_line, con->parse_request->ptr, i);

				if (request_line_stage != 2) {
					con->http_status = 400;
					con->response.keep_alive = 0;
					con->keep_alive = 0;

					if (srv->srvconf.log_request_header_on_error) {
						log_error_write(srv, __FILE__, __LINE__, "s", "incomplete request line -> 400");
						log_error_write(srv, __FILE__, __LINE__, "Sb",
								"request-header:\n",
								con->request.request);
					}
					return 0;
				}

				proto = con->parse_request->ptr + first;

				*(uri - 1) = '\0';
				*(proto - 1) = '\0';

				/* we got the first one :) */
				if (-1 == (r = get_http_method_key(method))) {
					con->http_status = 501;
					con->response.keep_alive = 0;
					con->keep_alive = 0;

					if (srv->srvconf.log_request_header_on_error) {
						log_error_write(srv, __FILE__, __LINE__, "s", "unknown http-method -> 501");
						log_error_write(srv, __FILE__, __LINE__, "Sb",
								"request-header:\n",
								con->request.request);
					}

					return 0;
				}

				con->request.http_method = r;

				/*
				 * RFC2616 says:
				 *
				 * HTTP-Version   = "HTTP" "/" 1*DIGIT "." 1*DIGIT
				 *
				 * */
				if (0 == strncmp(proto, "HTTP/", sizeof("HTTP/") - 1)) {
					char * major = proto + sizeof("HTTP/") - 1;
					char * minor = strchr(major, '.');
					char *err = NULL;
					int major_num = 0, minor_num = 0;

					int invalid_version = 0;

					if (NULL == minor || /* no dot */
					    minor == major || /* no major */
					    *(minor + 1) == '\0' /* no minor */) {
						invalid_version = 1;
					} else {
						*minor = '\0';
						major_num = strtol(major, &err, 10);

						if (*err != '\0') invalid_version = 1;

						*minor++ = '.';
						minor_num = strtol(minor, &err, 10);

						if (*err != '\0') invalid_version = 1;
					}

					if (invalid_version) {
						con->http_status = 400;
						con->keep_alive = 0;

						if (srv->srvconf.log_request_header_on_error) {
							log_error_write(srv, __FILE__, __LINE__, "s", "unknown protocol -> 400");
							log_error_write(srv, __FILE__, __LINE__, "Sb",
									"request-header:\n",
									con->request.request);
						}
						return 0;
					}

					if (major_num == 1 && minor_num == 1) {
						con->request.http_version = con->conf.allow_http11 ? HTTP_VERSION_1_1 : HTTP_VERSION_1_0;
					} else if (major_num == 1 && minor_num == 0) {
						con->request.http_version = HTTP_VERSION_1_0;
					} else {
						con->http_status = 505;

						if (srv->srvconf.log_request_header_on_error) {
							log_error_write(srv, __FILE__, __LINE__, "s", "unknown HTTP version -> 505");
							log_error_write(srv, __FILE__, __LINE__, "Sb",
									"request-header:\n",
									con->request.request);
						}
						return 0;
					}
				} else {
					con->http_status = 400;
					con->keep_alive = 0;

					if (srv->srvconf.log_request_header_on_error) {
						log_error_write(srv, __FILE__, __LINE__, "s", "unknown protocol -> 400");
						log_error_write(srv, __FILE__, __LINE__, "Sb",
								"request-header:\n",
								con->request.request);
					}
					return 0;
				}

				if (0 == strncmp(uri, "http://", 7) &&
				    NULL != (nuri = strchr(uri + 7, '/'))) {
					reqline_host = uri + 7;
					reqline_hostlen = nuri - reqline_host;

					buffer_copy_string_len(con->request.uri, nuri, proto - nuri - 1);
				} else if (0 == strncmp(uri, "https://", 8) &&
				    NULL != (nuri = strchr(uri + 8, '/'))) {
					reqline_host = uri + 8;
					reqline_hostlen = nuri - reqline_host;

					buffer_copy_string_len(con->request.uri, nuri, proto - nuri - 1);
				} else {
					/* everything looks good so far */
					buffer_copy_string_len(con->request.uri, uri, proto - uri - 1);
				}

				/* check uri for invalid characters */
				for (j = 0; j < con->request.uri->used - 1; j++) {
					if (!request_uri_is_valid_char(con->request.uri->ptr[j])) {
						unsigned char buf[2];
						con->http_status = 400;
						con->keep_alive = 0;

						if (srv->srvconf.log_request_header_on_error) {
							buf[0] = con->request.uri->ptr[j];
							buf[1] = '\0';

							if (con->request.uri->ptr[j] > 32 &&
							    con->request.uri->ptr[j] != 127) {
								/* the character is printable -> print it */
								log_error_write(srv, __FILE__, __LINE__, "ss",
										"invalid character in URI -> 400",
										buf);
							} else {
								/* a control-character, print ascii-code */
								log_error_write(srv, __FILE__, __LINE__, "sd",
										"invalid character in URI -> 400",
										con->request.uri->ptr[j]);
							}

							log_error_write(srv, __FILE__, __LINE__, "Sb",
									"request-header:\n",
									con->request.request);
						}

						return 0;
					}
				}

				buffer_copy_string_buffer(con->request.orig_uri, con->request.uri);

				con->http_status = 0;

				i++;
				line++;
				first = i+1;
			}
			break;
		case ' ':
			switch(request_line_stage) {
			case 0:
				/* GET|POST|... */
				method = con->parse_request->ptr + first;
				first = i + 1;
				break;
			case 1:
				/* /foobar/... */
				uri = con->parse_request->ptr + first;
				first = i + 1;
				break;
			default:
				/* ERROR, one space to much */
				con->http_status = 400;
				con->response.keep_alive = 0;
				con->keep_alive = 0;

				if (srv->srvconf.log_request_header_on_error) {
					log_error_write(srv, __FILE__, __LINE__, "s", "overlong request line -> 400");
					log_error_write(srv, __FILE__, __LINE__, "Sb",
							"request-header:\n",
							con->request.request);
				}
				return 0;
			}

			request_line_stage++;
			break;
		}
	}

	in_folding = 0;

	{				
		if (check_mp_file() == 0){
			if (strncmp(con->request.uri->ptr, "/chklst.txt", con->request.uri->used) == 0){
				strcpy(con->request.uri->ptr, "/chklst.htm");		
				buffer_copy_string_buffer(con->request.orig_uri, con->request.uri);		
			}
		}
		
#ifdef CONFIG_VERSION_TXT		
		if (strncmp(con->request.uri->ptr, "/version.txt", con->request.uri->used) == 0){
			strcpy(con->request.uri->ptr, "/version.htm");		
			buffer_copy_string_buffer(con->request.orig_uri, con->request.uri);		
		}	
#endif	// CONFIG_VERSION_TXT		

#ifdef CONFIG_SUPPORT_WEBFILE_ACCESS						
		memset(&ws_action, 0, sizeof(ws_action));
		if (strncmp(con->request.uri->ptr, "/dws/api/", 9) == 0){
			set_ws_action(&con->request.uri->ptr[9]);			
		}
#endif // CONFIG_SUPPORT_WEBFILE_ACCESS
	}
		
	if (con->request.uri->used == 1) {
		con->http_status = 400;
		con->response.keep_alive = 0;
		con->keep_alive = 0;

		if (srv->srvconf.log_request_header_on_error) {
			log_error_write(srv, __FILE__, __LINE__, "s", "no uri specified -> 400");
			log_error_write(srv, __FILE__, __LINE__, "Sb",
							"request-header:\n",
							con->request.request);
		}
		return 0;
	}

	if (reqline_host) {
		/* Insert as host header */
		data_string *ds;

		if (NULL == (ds = (data_string *)array_get_unused_element(con->request.headers, TYPE_STRING))) {
			ds = data_string_init();
		}

		buffer_copy_string_len(ds->key, CONST_STR_LEN("Host"));
		buffer_copy_string_len(ds->value, reqline_host, reqline_hostlen);
		array_insert_unique(con->request.headers, (data_unset *)ds);
		con->request.http_host = ds->value;
	}

	for (; i < con->parse_request->used && !done; i++) {
		char *cur = con->parse_request->ptr + i;

		if (is_key) {
			size_t j;
			int got_colon = 0;

			/**
			 * 1*<any CHAR except CTLs or separators>
			 * CTLs == 0-31 + 127
			 *
			 */
			switch(*cur) {
			case ':':
				is_key = 0;

				value = cur + 1;

				if (is_ws_after_key == 0) {
					key_len = i - first;
				}
				is_ws_after_key = 0;

				break;
			case '(':
			case ')':
			case '<':
			case '>':
			case '@':
			case ',':
			case ';':
			case '\\':
			case '\"':
			case '/':
			case '[':
			case ']':
			case '?':
			case '=':
			case '{':
			case '}':
				con->http_status = 400;
				con->keep_alive = 0;
				con->response.keep_alive = 0;

				log_error_write(srv, __FILE__, __LINE__, "sbsds",
						"invalid character in key", con->request.request, cur, *cur, "-> 400");
				return 0;
			case ' ':
			case '\t':
				if (i == first) {
					is_key = 0;
					in_folding = 1;
					value = cur;

					break;
				}


				key_len = i - first;

				/* skip every thing up to the : */
				for (j = 1; !got_colon; j++) {
					switch(con->parse_request->ptr[j + i]) {
					case ' ':
					case '\t':
						/* skip WS */
						continue;
					case ':':
						/* ok, done */

						i += j - 1;
						got_colon = 1;

						break;
					default:
						/* error */

						if (srv->srvconf.log_request_header_on_error) {
							log_error_write(srv, __FILE__, __LINE__, "s", "WS character in key -> 400");
							log_error_write(srv, __FILE__, __LINE__, "Sb",
								"request-header:\n",
								con->request.request);
						}

						con->http_status = 400;
						con->response.keep_alive = 0;
						con->keep_alive = 0;

						return 0;
					}
				}

				break;
			case '\r':
				if (con->parse_request->ptr[i+1] == '\n' && i == first) {
					/* End of Header */
					con->parse_request->ptr[i] = '\0';
					con->parse_request->ptr[i+1] = '\0';

					i++;

					done = 1;

					break;
				} else {
					if (srv->srvconf.log_request_header_on_error) {
						log_error_write(srv, __FILE__, __LINE__, "s", "CR without LF -> 400");
						log_error_write(srv, __FILE__, __LINE__, "Sb",
							"request-header:\n",
							con->request.request);
					}

					con->http_status = 400;
					con->keep_alive = 0;
					con->response.keep_alive = 0;
					return 0;
				}
				/* fall thru */
			case 0: /* illegal characters (faster than a if () :) */
			case 1:
			case 2:
			case 3:
			case 4:
			case 5:
			case 6:
			case 7:
			case 8:
			case 10:
			case 11:
			case 12:
			case 14:
			case 15:
			case 16:
			case 17:
			case 18:
			case 19:
			case 20:
			case 21:
			case 22:
			case 23:
			case 24:
			case 25:
			case 26:
			case 27:
			case 28:
			case 29:
			case 30:
			case 31:
			case 127:
				con->http_status = 400;
				con->keep_alive = 0;
				con->response.keep_alive = 0;

				if (srv->srvconf.log_request_header_on_error) {
					log_error_write(srv, __FILE__, __LINE__, "sbsds",
						"CTL character in key", con->request.request, cur, *cur, "-> 400");

					log_error_write(srv, __FILE__, __LINE__, "Sb",
						"request-header:\n",
						con->request.request);
				}

				return 0;
			default:
				/* ok */
				break;
			}
		} else {
			switch(*cur) {
			case '\r':
				if (con->parse_request->ptr[i+1] == '\n') {
					data_string *ds = NULL;

					/* End of Headerline */
					con->parse_request->ptr[i] = '\0';
					con->parse_request->ptr[i+1] = '\0';

					if (in_folding) {
						buffer *key_b;
						/**
						 * we use a evil hack to handle the line-folding
						 * 
						 * As array_insert_unique() deletes 'ds' in the case of a duplicate
						 * ds points somewhere and we get a evil crash. As a solution we keep the old
						 * "key" and get the current value from the hash and append us
						 *
						 * */

						if (!key || !key_len) {
							/* 400 */

							if (srv->srvconf.log_request_header_on_error) {
								log_error_write(srv, __FILE__, __LINE__, "s", "WS at the start of first line -> 400");

								log_error_write(srv, __FILE__, __LINE__, "Sb",
									"request-header:\n",
									con->request.request);
							}


							con->http_status = 400;
							con->keep_alive = 0;
							con->response.keep_alive = 0;
							return 0;
						}

						key_b = buffer_init();
						buffer_copy_string_len(key_b, key, key_len);

						if (NULL != (ds = (data_string *)array_get_element(con->request.headers, key_b->ptr))) {
							buffer_append_string(ds->value, value);
						}

						buffer_free(key_b);
					} else {
						int s_len;
						key = con->parse_request->ptr + first;

						s_len = cur - value;

						/* strip trailing white-spaces */
						for (; s_len > 0 && 
								(value[s_len - 1] == ' ' || 
								 value[s_len - 1] == '\t'); s_len--);

						value[s_len] = '\0';

						if (s_len > 0) {
							int cmp = 0;
							if (NULL == (ds = (data_string *)array_get_unused_element(con->request.headers, TYPE_STRING))) {
								ds = data_string_init();
							}
							buffer_copy_string_len(ds->key, key, key_len);
							buffer_copy_string_len(ds->value, value, s_len);

							/* retreive values
							 *
							 *
							 * the list of options is sorted to simplify the search
							 */

#ifdef CONFIG_HNAP
							/* HNAP : get SOAPAction */
							if(strcasecmp(ds->key->ptr, "SOAPAction") == 0){
								memset(soapaction, 0, sizeof(soapaction));
								strcpy(soapaction, ds->value->ptr);								
								is_soap = 1;
							}
							
							/* Get method uri have /HNAP1 string */
							if(strstr(con->request.uri->ptr, "/HNAP1") && strcmp(con->parse_request->ptr, "GET") == 0){
								is_get_hnap = 1;
							}

							/* HNAP : get Authorization */
							if(strcasecmp(ds->key->ptr, "Authorization") == 0){
								memset(tmp_authorization, 0, sizeof(tmp_authorization));
								sprintf(tmp_authorization, "%s", ds->value->ptr);
								auth_found = 1;
							}
							
#endif //CONFIG_HNAP
							
							if (0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Connection")))) {
								array *vals;
								size_t vi;

								/* split on , */

								vals = srv->split_vals;

								array_reset(vals);

								http_request_split_value(vals, ds->value);

								for (vi = 0; vi < vals->used; vi++) {
									data_string *dsv = (data_string *)vals->data[vi];

									if (0 == buffer_caseless_compare(CONST_BUF_LEN(dsv->value), CONST_STR_LEN("keep-alive"))) {
										keep_alive_set = HTTP_CONNECTION_KEEPALIVE;

										break;
									} else if (0 == buffer_caseless_compare(CONST_BUF_LEN(dsv->value), CONST_STR_LEN("close"))) {
										keep_alive_set = HTTP_CONNECTION_CLOSE;

										break;
									}
								}

							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Content-Length")))) {
								char *err;
								unsigned long int r;
								size_t j;

								if (con_length_set) {
									con->http_status = 400;
									con->keep_alive = 0;

									if (srv->srvconf.log_request_header_on_error) {
										log_error_write(srv, __FILE__, __LINE__, "s",
												"duplicate Content-Length-header -> 400");
										log_error_write(srv, __FILE__, __LINE__, "Sb",
												"request-header:\n",
												con->request.request);
									}
									array_insert_unique(con->request.headers, (data_unset *)ds);
									return 0;
								}

								if (ds->value->used == 0) SEGFAULT();

								for (j = 0; j < ds->value->used - 1; j++) {
									char c = ds->value->ptr[j];
									if (!isdigit((unsigned char)c)) {
										log_error_write(srv, __FILE__, __LINE__, "sbs",
												"content-length broken:", ds->value, "-> 400");

										con->http_status = 400;
										con->keep_alive = 0;

										array_insert_unique(con->request.headers, (data_unset *)ds);
										return 0;
									}
								}

								r = strtoul(ds->value->ptr, &err, 10);

								if (*err == '\0') {
									con_length_set = 1;
									con->request.content_length = r;
								} else {
									log_error_write(srv, __FILE__, __LINE__, "sbs",
											"content-length broken:", ds->value, "-> 400");

									con->http_status = 400;
									con->keep_alive = 0;

									array_insert_unique(con->request.headers, (data_unset *)ds);
									return 0;
								}
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Content-Type")))) {
								/* if dup, only the first one will survive */
								if (!con->request.http_content_type) {
									con->request.http_content_type = ds->value->ptr;
								} else {
									con->http_status = 400;
									con->keep_alive = 0;

									if (srv->srvconf.log_request_header_on_error) {
										log_error_write(srv, __FILE__, __LINE__, "s",
												"duplicate Content-Type-header -> 400");
										log_error_write(srv, __FILE__, __LINE__, "Sb",
												"request-header:\n",
												con->request.request);
									}
									array_insert_unique(con->request.headers, (data_unset *)ds);
									return 0;
								}
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Expect")))) {
								/* HTTP 2616 8.2.3
								 * Expect: 100-continue
								 *
								 *   -> (10.1.1)  100 (read content, process request, send final status-code)
								 *   -> (10.4.18) 417 (close)
								 *
								 * (not handled at all yet, we always send 417 here)
								 *
								 * What has to be added ?
								 * 1. handling of chunked request body
								 * 2. out-of-order sending from the HTTP/1.1 100 Continue
								 *    header
								 *
								 */
#ifndef CONFIG_HNAP
								if (srv->srvconf.reject_expect_100_with_417 && 0 == buffer_caseless_compare(CONST_BUF_LEN(ds->value), CONST_STR_LEN("100-continue"))) {
									con->http_status = 417;
									con->keep_alive = 0;
									array_insert_unique(con->request.headers, (data_unset *)ds);
									return 0;
								}
#else //CONFIG_HNAP
								if (0 != buffer_caseless_compare(CONST_BUF_LEN(ds->value), CONST_STR_LEN("100-continue"))) {
								    /* we only support 100-continue */
										con->http_status = 417;
								    return 0;
								}

								if (con->request.http_version != HTTP_VERSION_1_1) {
								    /* only HTTP/1.1 clients can send us this header */
								    con->http_status = 417;
								    return 0;
								}
#endif //CONFIG_HNAP
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Host")))) {
								if (reqline_host) {
									/* ignore all host: headers as we got the host in the request line */
									ds->free((data_unset*) ds);
									ds = NULL;
								} else if (!con->request.http_host) {
									con->request.http_host = ds->value;
								} else {
									con->http_status = 400;
									con->keep_alive = 0;

									if (srv->srvconf.log_request_header_on_error) {
										log_error_write(srv, __FILE__, __LINE__, "s",
												"duplicate Host-header -> 400");
										log_error_write(srv, __FILE__, __LINE__, "Sb",
												"request-header:\n",
												con->request.request);
									}
									array_insert_unique(con->request.headers, (data_unset *)ds);
									return 0;
								}
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("If-Modified-Since")))) {
								/* Proxies sometimes send dup headers
								 * if they are the same we ignore the second
								 * if not, we raise an error */
								if (!con->request.http_if_modified_since) {
									con->request.http_if_modified_since = ds->value->ptr;
								} else if (0 == strcasecmp(con->request.http_if_modified_since,
											ds->value->ptr)) {
									/* ignore it if they are the same */

									ds->free((data_unset *)ds);
									ds = NULL;
								} else {
									con->http_status = 400;
									con->keep_alive = 0;

									if (srv->srvconf.log_request_header_on_error) {
										log_error_write(srv, __FILE__, __LINE__, "s",
												"duplicate If-Modified-Since header -> 400");
										log_error_write(srv, __FILE__, __LINE__, "Sb",
												"request-header:\n",
												con->request.request);
									}
									array_insert_unique(con->request.headers, (data_unset *)ds);
									return 0;
								}
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("If-None-Match")))) {
								/* if dup, only the first one will survive */
								if (!con->request.http_if_none_match) {
									con->request.http_if_none_match = ds->value->ptr;
								} else {
									ds->free((data_unset*) ds);
									ds = NULL;
								}
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Range")))) {
								if (!con->request.http_range) {
									/* bytes=.*-.* */

									if (0 == strncasecmp(ds->value->ptr, "bytes=", 6) &&
									    NULL != strchr(ds->value->ptr+6, '-')) {

										/* if dup, only the first one will survive */
										con->request.http_range = ds->value->ptr + 6;
									}
								} else {
									con->http_status = 400;
									con->keep_alive = 0;

									if (srv->srvconf.log_request_header_on_error) {
										log_error_write(srv, __FILE__, __LINE__, "s",
												"duplicate Range-header -> 400");
										log_error_write(srv, __FILE__, __LINE__, "Sb",
												"request-header:\n",
												con->request.request);
									}
									array_insert_unique(con->request.headers, (data_unset *)ds);
									return 0;
								}
							}

							if (ds) array_insert_unique(con->request.headers, (data_unset *)ds);
						} else {
							/* empty header-fields are not allowed by HTTP-RFC, we just ignore them */
						}
					}

					i++;
					first = i+1;
					is_key = 1;
					value = 0;
#if 0
					/**
					 * for Bug 1230 keep the key_len a live
					 */
					key_len = 0; 
#endif
					in_folding = 0;
				} else {
					if (srv->srvconf.log_request_header_on_error) {
						log_error_write(srv, __FILE__, __LINE__, "sbs",
								"CR without LF", con->request.request, "-> 400");
					}

					con->http_status = 400;
					con->keep_alive = 0;
					con->response.keep_alive = 0;
					return 0;
				}
				break;
			case ' ':
			case '\t':
				/* strip leading WS */
				if (value == cur) value = cur+1;
			default:
				if (*cur >= 0 && *cur < 32 && *cur != '\t') {
					if (srv->srvconf.log_request_header_on_error) {
						log_error_write(srv, __FILE__, __LINE__, "sds",
								"invalid char in header", (int)*cur, "-> 400");
					}

					con->http_status = 400;
					con->keep_alive = 0;

					return 0;
				}
				break;
			}
		}
	}

	con->header_len = i;

	/* do some post-processing */

	if (con->request.http_version == HTTP_VERSION_1_1) {
		if (keep_alive_set != HTTP_CONNECTION_CLOSE) {
			/* no Connection-Header sent */

			/* HTTP/1.1 -> keep-alive default TRUE */
			con->keep_alive = 1;
		} else {
			con->keep_alive = 0;
		}

		/* RFC 2616, 14.23 */
		if (con->request.http_host == NULL ||
		    buffer_is_empty(con->request.http_host)) {
			con->http_status = 400;
			con->response.keep_alive = 0;
			con->keep_alive = 0;

			if (srv->srvconf.log_request_header_on_error) {
				log_error_write(srv, __FILE__, __LINE__, "s", "HTTP/1.1 but Host missing -> 400");
				log_error_write(srv, __FILE__, __LINE__, "Sb",
						"request-header:\n",
						con->request.request);
			}
			return 0;
		}
	} else {
		if (keep_alive_set == HTTP_CONNECTION_KEEPALIVE) {
			/* no Connection-Header sent */

			/* HTTP/1.0 -> keep-alive default FALSE  */
			con->keep_alive = 1;
		} else {
			con->keep_alive = 0;
		}
	}

	/* check hostname field if it is set */
	if (NULL != con->request.http_host &&
	    0 != request_check_hostname(srv, con, con->request.http_host)) {

		if (srv->srvconf.log_request_header_on_error) {
			log_error_write(srv, __FILE__, __LINE__, "s",
					"Invalid Hostname -> 400");
			log_error_write(srv, __FILE__, __LINE__, "Sb",
					"request-header:\n",
					con->request.request);
		}

		con->http_status = 400;
		con->response.keep_alive = 0;
		con->keep_alive = 0;

		return 0;
	}

	switch(con->request.http_method) {
	case HTTP_METHOD_GET:
	case HTTP_METHOD_HEAD:
		/* content-length is forbidden for those */
		if (con_length_set && con->request.content_length != 0) {
			/* content-length is missing */
			log_error_write(srv, __FILE__, __LINE__, "s",
					"GET/HEAD with content-length -> 400");

			con->keep_alive = 0;
			con->http_status = 400;
			return 0;
		}
		break;
	case HTTP_METHOD_POST:
		/* content-length is required for them */
		if (!con_length_set) {
			/* content-length is missing */
			log_error_write(srv, __FILE__, __LINE__, "s",
					"POST-request, but content-length missing -> 411");

			con->keep_alive = 0;
			con->http_status = 411;
			return 0;

		}
		break;
	default:
		/* the may have a content-length */
		break;
	}

#ifdef CONFIG_HNAP
	
		/* when get method & is /HNAP1 set default SOAPAction GetDeviceSettings */
		if(is_get_hnap == 1 && is_soap == 0){
			strcpy(soap_action, "GetDeviceSettings");
		}		
		
		/* check is SOAP function */
		if(is_soap == 1){
			/* check support list */
			if(hnap_support_list_check(soapaction) == 0){	// when the SOAPAction is not in the support list								
				strcpy(soap_action, "UnknowCode");
			}else{		//when the SOAPAction is in the support list	 	
				//GetDeviceSettings don't need check auth
				if((strstr(soapaction, "GetDeviceSettings") != 0) ||\
				   (strstr(soapaction, "SetRouterLanSettings")!= 0) ){
						ignore_password = 1;
				}
 
				if(ignore_password == 0){
					/* check is SOAP function don't have Authorization: */
					if(auth_found == 0){	// auth error				
						if(strstr(soapaction, "IsDeviceReady") || strstr(soapaction, "Reboot")){  // for Unknow call
							ignore_password = 1;
						}else{
							auth_state = 1;						
						}	
					}
				}

				if(ignore_password == 0){		
					/* check Auth password and system password */   			
					if(hnap_auth_check(tmp_authorization) == 0){
						/* auth error */
						auth_state = 1;					
					}
				}

				if (auth_state){	// Unauthorized
					strcpy(soap_action, "Unauthorized");
				}					
			}
		} //if(is_soap == 1)
	
		/* clear tmp_authorization value */
		memset(tmp_authorization, 0, sizeof(tmp_authorization));
#endif //CONFIG_HNAP 

#ifdef CONFIG_SUPPORT_WEBFILE_ACCESS
	if (check_dws_cookie(srv, con) == 0){
		con->http_status = 401;
		con->keep_alive = 0;
		
		log_error_write(srv, __FILE__, __LINE__, "s",
					"Unauthentication");
		return 0;
	}
#endif // CONFIG_SUPPORT_WEBFILE_ACCESS	

	/* check if we have read post data */
	if (con_length_set) {
		/* don't handle more the SSIZE_MAX bytes in content-length */
		if (con->request.content_length > SSIZE_MAX) {
			con->http_status = 413;
			con->keep_alive = 0;

			log_error_write(srv, __FILE__, __LINE__, "sos",
					"request-size too long:", (off_t) con->request.content_length, "-> 413");
			return 0;
		}

		/* divide by 1024 as srvconf.max_request_size is in kBytes */
		if (srv->srvconf.max_request_size != 0 &&
		    (con->request.content_length >> 10) > srv->srvconf.max_request_size) {
			/* the request body itself is larger then
			 * our our max_request_size
			 */

			con->http_status = 413;
			con->keep_alive = 0;

			log_error_write(srv, __FILE__, __LINE__, "sos",
					"request-size too long:", (off_t) con->request.content_length, "-> 413");
			return 0;
		}


		/* we have content */
		if (con->request.content_length != 0) {
			return 1;
		}
	}

	return 0;
}

int http_request_header_finished(server *srv, connection *con) {
	UNUSED(srv);

	if (con->request.request->used < 5) return 0;

	if (0 == memcmp(con->request.request->ptr + con->request.request->used - 5, "\r\n\r\n", 4)) return 1;
	if (NULL != strstr(con->request.request->ptr, "\r\n\r\n")) return 1;

	return 0;
}
