#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "base.h"
#include "log.h"
#include "buffer.h"

#include "plugin.h"
#include "response.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

extern int enable_404_redirect;
char w4r_device_name[256] = {};
char w4r_device_name_mac[256] = {};
char w4r_ipv4_lan_ip[16] = {};
char w4r_ipv4_wan_ip[16] = {};

#ifdef CONFIG_DDNS
char ddns_enable = 0;
char ddns_host_name[61];	// the variable size must match the maxlength of ddns host name on GUI
#endif	// CONFIG_DDNS

#ifdef CONFIG_IPV6_SUPPORT
char w4r_ipv6_lan_link_local_ip[40] = {};
char w4r_ipv6_lan_ip[40] = {};
#endif // CONFIG_IPV6_SUPPORT

#ifdef CONFIG_SUPPORT_MDNS
char mdns_name[30];
char mdns_name_mac[30];
#endif	// CONFIG_SUPPORT_MDNS

int factory_default = 0;

typedef struct {
	pcre_keyvalue_buffer *redirect;
	data_config *context; /* to which apply me */
} plugin_config;

typedef struct {
	PLUGIN_DATA;
	buffer *match_buf;
	buffer *location;

	plugin_config **config_storage;

	plugin_config conf;
} plugin_data;

void read_404_info(){
	FILE* fp;
	char buf[512] = {};
	char *start,*end;
	char temp_factory_default[2];
	
#ifdef CONFIG_DDNS
	char temp_ddns_enable[2];	
#endif	// CONFIG_DDNS	
	
	memset(w4r_device_name, 0, sizeof(w4r_device_name));
	memset(w4r_device_name_mac, 0, sizeof(w4r_device_name_mac));
	memset(w4r_ipv4_lan_ip, 0, sizeof(w4r_ipv4_lan_ip));
	memset(w4r_ipv4_wan_ip, 0, sizeof(w4r_ipv4_wan_ip));
	memset(temp_factory_default, 0, sizeof(temp_factory_default));

#ifdef CONFIG_IPV6_SUPPORT	
	memset(w4r_ipv6_lan_link_local_ip, 0, sizeof(w4r_ipv6_lan_link_local_ip));
	memset(w4r_ipv6_lan_ip, 0, sizeof(w4r_ipv6_lan_ip));
#endif // CONFIG_IPV6_SUPPORT
	
#ifdef CONFIG_SUPPORT_MDNS
	memset(mdns_name, 0, sizeof(mdns_name));
#endif	// CONFIG_SUPPORT_MDNS
	
	fp = fopen("/var/tmp/lighttpd_404_info.txt","r");
	
	if(fp)
	{
		while(start = fgets(buf,sizeof(buf),fp))
		{		
			if(start = strstr(buf,"device_name")){
				start += 12;
				end = strchr(start,'\n');
				strncpy(w4r_device_name,start,end - start);
#ifdef CONFIG_SUPPORT_MDNS
				sprintf(mdns_name, "%s.local", w4r_device_name);
#endif	// CONFIG_SUPPORT_MDNS				
			}else if(start = strstr(buf,"device_mac_name")){
				start += 16;
				end = strchr(start,'\n');
				strncpy(w4r_device_name_mac,start,end - start);
#ifdef CONFIG_SUPPORT_MDNS
				sprintf(mdns_name_mac, "%s.local", w4r_device_name_mac);
#endif	// CONFIG_SUPPORT_MDNS
			}else if(start = strstr(buf,"ipv4_lan_ip")){
				start += 12;
				end = strchr(start,'\n');
				strncpy(w4r_ipv4_lan_ip,start,end - start);
			}else if(start = strstr(buf,"ipv4_wan_ip")){
				start += 12;
				end = strchr(start,'\n');
				strncpy(w4r_ipv4_wan_ip,start,end - start);			
			}else if(start = strstr(buf,"factory_default")){
				start += 16;
				end = strchr(start,'\n');
				strncpy(temp_factory_default, start, end - start);
				factory_default = atoi(temp_factory_default);
#ifdef CONFIG_DDNS
			}else if(start = strstr(buf,"ddns_enable")){
				start += 12;
				end = strchr(start,'\n');
				memset(temp_ddns_enable, 0, sizeof(temp_ddns_enable));
				strncpy(temp_ddns_enable, start, end - start);
				ddns_enable = atoi(temp_ddns_enable);
			}else if(start = strstr(buf,"ddns_host_name")){
				start += 15;
				end = strchr(start,'\n');
				memset(ddns_host_name, 0, sizeof(ddns_host_name));
				strncpy(ddns_host_name,start,end - start);				
#endif	// CONFIG_DDNS

#ifdef CONFIG_IPV6_SUPPORT	
		   }else if(start = strstr(buf,"ipv6_lan_link_local_ip")){
				start += 23;				
				end = strchr(start,'/');	//remove prefix length info
				if(end){
					*end = ']';	//host format:   [ipv6 address]
					strncpy(w4r_ipv6_lan_link_local_ip+1,start,end - start+1);
					w4r_ipv6_lan_link_local_ip[0] = '[';
				}
			}else if(start = strstr(buf,"ipv6_lan_ip")){
				start += 12;				
				end = strchr(start,'/');	//remove prefix length info
				if(end){
					*end = ']';
					strncpy(w4r_ipv6_lan_ip+1,start,end - start+1);
					w4r_ipv6_lan_ip[0] = '[';
				}
#endif // CONFIG_IPV6_SUPPORT
			}
			
			memset(buf,0,sizeof(buf));
		}
		fclose(fp);
	}

}

INIT_FUNC(mod_404redirect_init) {
	plugin_data *p;

	p = calloc(1, sizeof(*p));

	p->match_buf = buffer_init();
	p->location = buffer_init();

	return p;
}

FREE_FUNC(mod_404redirect_free) {
	plugin_data *p = p_d;

	if (!p) return HANDLER_GO_ON;

	if (p->config_storage) {
		size_t i;
		for (i = 0; i < srv->config_context->used; i++) {
			plugin_config *s = p->config_storage[i];

			pcre_keyvalue_buffer_free(s->redirect);

			free(s);
		}
		free(p->config_storage);
	}


	buffer_free(p->match_buf);
	buffer_free(p->location);

	free(p);

	return HANDLER_GO_ON;
}

/* handle plugin config and check values */

SETDEFAULTS_FUNC(mod_404redirect_set_defaults) {
	return HANDLER_GO_ON;
}

static handler_t mod_404redirect_uri_handler(server *srv, connection *con, void *p_data) {
	plugin_data *p = p_data;
	char *port = NULL;
	int ip_len = 0;

	//if lan settings changed, read 404 info again	
	if(enable_404_redirect == 1){
		read_404_info();
		enable_404_redirect = 0;
	}

	port = strchr(con->request.http_host->ptr, ':');
	if(port > 0){
		ip_len = port - con->request.http_host->ptr;
	}else{
		ip_len = strlen(con->request.http_host->ptr);
	}

#ifdef CONFIG_DDNS
	if ((ddns_enable == 1) && (strncasecmp(ddns_host_name, con->request.http_host->ptr, ip_len) == 0)){		
		return HANDLER_GO_ON;
	}
#endif	// 	CONFIG_DDNS
	
#ifdef CONFIG_DNS_HIJACK
	if (strcasecmp(con->request.http_host->ptr, HIJACK_DNS_ADDR) == 0){		
		return HANDLER_GO_ON;
	}
#endif	// 	CONFIG_DNS_HIJACK
	
	if(strcasecmp(w4r_device_name,con->request.http_host->ptr) != 0 &&
		strcasecmp(w4r_device_name_mac,con->request.http_host->ptr) != 0 &&
		strcmp(w4r_ipv4_lan_ip,con->request.http_host->ptr) != 0 &&
		strcmp("127.0.0.1",con->request.http_host->ptr) != 0 && //Loopback IP
		strncmp(w4r_ipv4_wan_ip,con->request.http_host->ptr, ip_len) != 0
#ifdef CONFIG_IPV6_SUPPORT			
		&& strcasecmp(w4r_ipv6_lan_link_local_ip,con->request.http_host->ptr) != 0 
		&& strcasecmp(w4r_ipv6_lan_ip,con->request.http_host->ptr) != 0 
#endif // CONFIG_IPV6_SUPPORT		

#ifdef CONFIG_SUPPORT_MDNS
		&& strcasecmp(mdns_name, con->request.http_host->ptr) != 0 
		&& strcasecmp(mdns_name_mac, con->request.http_host->ptr) != 0
#endif	// CONFIG_SUPPORT_MDNS
		)
	{
		char loc[256];
				
		if (factory_default == 0){
			sprintf(loc, "http://%s/error-404.htm",w4r_device_name);
		}else{
			sprintf(loc, "http://%s/login.htm",w4r_device_name);
		}
		
		buffer_reset(p->location);
		buffer_append_string_len(p->location, loc, strlen(loc));
	
		response_header_insert(srv, con, CONST_STR_LEN("Location"), CONST_BUF_LEN(p->location));
		con->http_status = 307;
		con->file_finished = 1;

		return HANDLER_FINISHED;
	}
	
	return HANDLER_GO_ON;
}


int mod_404redirect_plugin_init(plugin *p) {
	p->version     = LIGHTTPD_VERSION_ID;
	p->name        = buffer_init_string("404redirect");

	p->init        = mod_404redirect_init;
	p->handle_uri_clean  = mod_404redirect_uri_handler;
	p->set_defaults  = mod_404redirect_set_defaults;
	p->cleanup     = mod_404redirect_free;

	p->data        = NULL;

	return 0;
}
