/* Port-triggering target. 
 *
 * Copyright (C) 2003, CyberTAN Corporation
 * All Rights Reserved.
 * 
 * Luke 2011/6/9: support multiport
 */

/* Shared library add-on to iptables to add port-trigger support. */

#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <iptables.h>
#include <linux/netfilter_ipv4/ip_tables.h>
//#include <linux/netfilter_ipv4/ip_nat_rule.h>
#include <net/netfilter/nf_nat.h>
#include <linux/netfilter/xt_TRIGGER.h>

/* Function which prints out usage message. */
static void TRIGGER_help(void)
{
	printf(
"TRIGGER v%s options:\n"
" --trigger-type (dnat|in|out)\n"
"				Trigger type\n"
" --trigger-proto proto\n"
"				Trigger protocol\n"
" --trigger-match port[,port:port,port]\n"
"				Trigger destination port range\n"
" --trigger-relate port[,port:port,port]\n"
"				Port range to map related destination port range to.\n\n",
XTABLES_VERSION);
}

static const struct option TRIGGER_opts[] = {
	{ "trigger-type", 1, 0, '1' },
	{ "trigger-proto", 1, 0, '2' },
	{ "trigger-match", 1, 0, '3' },
	{ "trigger-relate", 1, 0, '4' },
	{ 0 }
};

static char *
proto_to_name(u_int8_t proto)
{
	switch (proto) {
	case IPPROTO_TCP:
		return "tcp";
	case IPPROTO_UDP:
		return "udp";
	case IPPROTO_UDPLITE:
		return "udplite";
	case IPPROTO_SCTP:
		return "sctp";
	case IPPROTO_DCCP:
		return "dccp";
	default:
		return NULL;
	}
}

/* Initialize the target. */
//static void TRIGGER_init(struct xt_entry_target *t, unsigned int *nfcache)
//{
//	/* Can't cache this */
//	*nfcache |= NFC_UNKNOWN;
//}

/* Parses ports */
static void
parse_multi_ports_v1(const char *portstring, 
		     struct xt_trigger_ports *multiinfo,
		     char *proto)
{
	char *buffer, *cp, *next, *range;
	unsigned int i;
	u_int16_t m;

	buffer = strdup(portstring);
	if (!buffer) xtables_error(OTHER_PROBLEM, "strdup failed");

	for (i=0; i<XT_MULTI_PORTS; i++)
		multiinfo->pflags[i] = 0;
 
	for (cp=buffer, i=0; cp && i<XT_MULTI_PORTS; cp=next, i++) {
		next=strchr(cp, ',');
 		if (next) *next++='\0';
		range = strchr(cp, ':');
		if (range) {
			if (i == XT_MULTI_PORTS-1)
				xtables_error(PARAMETER_PROBLEM,
					   "too many ports specified");
			*range++ = '\0';
		}
		multiinfo->ports[i] = xtables_parse_port(cp, proto);
		if (range) {
			multiinfo->pflags[i] = 1;
			multiinfo->ports[++i] = xtables_parse_port(range, proto);
			if (multiinfo->ports[i-1] >= multiinfo->ports[i])
				xtables_error(PARAMETER_PROBLEM,
					   "invalid portrange specified");
			m <<= 1;
		}
 	}
	multiinfo->count = i;
	if (cp) xtables_error(PARAMETER_PROBLEM, "too many ports specified");
 	free(buffer);
}


/* Function which parses command options; returns true if it
   ate an option */
static int TRIGGER_parse(int c, char **argv, int invert, unsigned int *flags,
      const void *entry,
      struct xt_entry_target **target)
{
	struct xt_trigger_info *info = (struct xt_trigger_info *)(*target)->data;

	switch (c) {
	case '1':
		if (!strcasecmp(optarg, "dnat"))
			info->type = XT_TRIGGER_DNAT;
		else if (!strcasecmp(optarg, "in"))
			info->type = XT_TRIGGER_IN;
		else if (!strcasecmp(optarg, "out"))
			info->type = XT_TRIGGER_OUT;
		else
			xtables_error(PARAMETER_PROBLEM,
				   "unknown type `%s' specified", optarg);
		return 1;

	case '2':
		if (!strcasecmp(optarg, "tcp"))
			info->proto = IPPROTO_TCP;
		else if (!strcasecmp(optarg, "udp"))
			info->proto = IPPROTO_UDP;
		else if (!strcasecmp(optarg, "all"))
			info->proto = 0;
		else
			xtables_error(PARAMETER_PROBLEM,
				   "unknown protocol `%s' specified", optarg);
		return 1;

	case '3':
		if (xtables_check_inverse(optarg, &invert, &optind, 0))
			xtables_error(PARAMETER_PROBLEM,
				   "Unexpected `!' after --trigger-match");

		parse_multi_ports_v1(optarg, &(info->mports), proto_to_name(info->proto));
		return 1;

	case '4':
		if (xtables_check_inverse(optarg, &invert, &optind, 0))
			xtables_error(PARAMETER_PROBLEM,
				   "Unexpected `!' after --trigger-relate");

		parse_multi_ports_v1(optarg, &(info->rports), proto_to_name(info->proto));
		*flags |= IP_NAT_RANGE_PROTO_SPECIFIED;
		return 1;

	default:
		return 0;
	}
}

/* Final check; don't care. */
static void TRIGGER_final_check(unsigned int flags)
{
	//printf("change port 1901--->1900\n");
}

/* Prints out the targinfo. */
static void TRIGGER_print(const void *ip,
      const struct xt_entry_target *target,
      int numeric)
{
	struct xt_trigger_info *info = (const void *)target->data;
	unsigned int i;

	printf("TRIGGER ");
	if (info->type == XT_TRIGGER_DNAT)
		printf("type:dnat ");
	else if (info->type == XT_TRIGGER_IN)
		printf("type:in ");
	else if (info->type == XT_TRIGGER_OUT)
		printf("type:out ");

	if (info->proto == IPPROTO_TCP)
		printf("tcp ");
	else if (info->proto == IPPROTO_UDP)
		printf("udp ");

		
	if (info->type == XT_TRIGGER_OUT){
		printf("match:");
		for (i=0; i < info->mports.count; i++) {
			printf("%s", i ? "," : "");
			printf("%hu", info->mports.ports[i]);
			if (info->mports.pflags[i]) {
				printf(":");
				printf("%hu", info->mports.ports[++i]);
			}
		}	
		printf(" ");	
			
		printf("relate:");
		for (i=0; i < info->rports.count; i++) {
			printf("%s", i ? "," : "");
			printf("%hu", info->rports.ports[i]);
			if (info->rports.pflags[i]) {
				printf(":");
				printf("%hu", info->rports.ports[++i]);
			}
		}	
		printf(" ");
	}
}

/* Saves the union ipt_targinfo in parsable form to stdout. */
/* Luke: TRIGGER_save is unused */
/*static void TRIGGER_save(const void *ip, const struct xt_entry_target *target)
{
	const struct xt_trigger_info *info = (const void *)target->data;
	unsigned int i;

	printf("--trigger-proto ");
	if (info->proto == IPPROTO_TCP)
		printf("tcp ");
	else if (info->proto == IPPROTO_UDP)
		printf("udp ");	
		
	printf("--trigger-match %hu:%hu ", info->mports.ports[0], info->mports.ports[1]);
	printf("--trigger-relate %hu:%hu ", info->rports.ports[0], info->rports.ports[1]);	
}*/

static struct xtables_target trigger 
= {	//.next		= NULL,
	.name		= "TRIGGER",
	.version	= XTABLES_VERSION,
	.size		= XT_ALIGN(sizeof(struct xt_trigger_info)),
	.userspacesize	= XT_ALIGN(sizeof(struct xt_trigger_info)),
	.help		= TRIGGER_help,
	//.init		= TRIGGER_init,
	.parse		= TRIGGER_parse,
	.final_check	= TRIGGER_final_check,
	.print		= TRIGGER_print,
	//.save		= TRIGGER_save,
	.extra_opts	= TRIGGER_opts
};

void _init(void)
{
	xtables_register_target(&trigger);
}
