/*
**  igmpproxy - IGMP proxy based multicast router 
**  Copyright (C) 2005 Johnny Egeland <johnny@rlo.org>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
**----------------------------------------------------------------------------
**
**  This software is derived work from the following software. The original
**  source code has been modified from it's original state by the author
**  of igmpproxy.
**
**  smcroute 0.92 - Copyright (C) 2001 Carsten Schill <carsten@cschill.de>
**  - Licensed under the GNU General Public License, version 2
**  
**  mrouted 3.9-beta3 - COPYRIGHT 1989 by The Board of Trustees of 
**  Leland Stanford Junior University.
**  - Original license can be found in the "doc/mrouted-LINCESE" file.
**
*/
/**
*   igmp.h - Recieves IGMP requests, and handle them 
*            appropriately...
*/

#include "defs.h"

#ifdef MV88F6281
#include <sys/ioctl.h>
#include <linux/if_bridge.h>
//#include <linux/ip.h>
//#include <linux/igmp.h>
#include <linux/sockios.h>
#endif
 
#ifdef CONFIG_ATHEROS_HEADER_EN
extern int isAdressValidForIf( struct IfDesc* intrface, uint32 ipaddr );
extern int inetChksum(u_short *addr, u_int len);
#endif

// Globals                  
uint32     allhosts_group;          /* All hosts addr in net order */
uint32     allrouters_group;          /* All hosts addr in net order */
              
extern int MRouterFD;

/*
 * Open and initialize the igmp socket, and fill in the non-changing
 * IP header fields in the output packet buffer.
 */
void initIgmp() {
    struct ip *ip;

    recv_buf = malloc(RECV_BUF_SIZE);
    send_buf = malloc(RECV_BUF_SIZE);

    k_hdr_include(TRUE);    /* include IP header when sending */
    k_set_rcvbuf(256*1024,48*1024); /* lots of input buffering        */
    k_set_ttl(1);       /* restrict multicasts to one hop */
    k_set_loop(FALSE);      /* disable multicast loopback     */

    ip         = (struct ip *)send_buf;
    bzero(ip, sizeof(struct ip));
    /*
     * Fields zeroed that aren't filled in later:
     * - IP ID (let the kernel fill it in)
     * - Offset (we don't send fragments)
     * - Checksum (let the kernel fill it in)
     */
    ip->ip_v   = IPVERSION;
    ip->ip_hl  = sizeof(struct ip) >> 2;
    ip->ip_tos = 0xc0;      /* Internet Control */
    ip->ip_ttl = MAXTTL;    /* applies to unicasts only */
    ip->ip_p   = IPPROTO_IGMP;

    allhosts_group   = htonl(INADDR_ALLHOSTS_GROUP);
    allrouters_group = htonl(INADDR_ALLRTRS_GROUP);
#ifdef CONFIG_ATHEROS_HEADER_EN

    //force the igmp version to 2
    FILE *fp = fopen("/proc/sys/net/ipv4/conf/all/force_igmp_version", "w");
    if(fp)
        fprintf(fp, "2");
    fclose(fp);
#endif
}

/**
*   Finds the textual name of the supplied IGMP request.
*/
char *igmpPacketKind(u_int type, u_int code) {
    static char unknown[20];

    switch (type) {
    case IGMP_MEMBERSHIP_QUERY:     return  "Membership query  ";
    case IGMP_V1_MEMBERSHIP_REPORT:  return "V1 member report  ";
    case IGMP_V2_MEMBERSHIP_REPORT:  return "V2 member report  ";
    case IGMP_V2_LEAVE_GROUP:        return "Leave message     ";
    
    default:
        sprintf(unknown, "unk: 0x%02x/0x%02x    ", type, code);
        return unknown;
    }
}

#ifdef MV88F6281

#define MAX( a, b ) ((a) < (b) ? (b) : (a))

typedef unsigned short u16;
typedef unsigned long u32;

u16 ip_sum_calc(u16 len_ip_header, char buff[])
{
	u16 word16;
	u32 sum=0;
	u16 i;
    
	for (i=0; i < len_ip_header; i= i + 2) {
		word16 =((buff[i] << 8) & 0xFF00) +(buff[i + 1] & 0xFF);
		sum = sum + (u32) word16;	
	}
	 								
	while (sum>>16)
	  sum = (sum & 0xFFFF) + (sum >> 16);
	sum = ~sum;
							  			
	return ((u16) sum);
}

int op_sock(const char * iff, int len) {

	int fd;

	if ((fd = socket( AF_INET, SOCK_RAW, IPPROTO_IGMP )) < 0 ) {
		syslog(LOG_WARNING, "igmpd: socket open failed\n");
		return -1;
	}

	if (setsockopt(fd, SOL_SOCKET, SO_BINDTODEVICE, iff, len)) {
		syslog(LOG_WARNING, "igmpd: setsockopt failed\n");
		return -1;
	}

	return fd;
}

int get_smac(struct iphdr *iph, char *smac) {

	FILE *fp;
	char cmd[1024];
	int a[6];

	sprintf(cmd, "arping -f -c 3 -I br0 %d.%d.%d.%d", iph->saddr & 0xff,
							(iph->saddr >> 8) & 0xff,
							(iph->saddr >> 16) & 0xff,
							(iph->saddr >> 24) & 0xff);

	if (!(fp = popen(cmd, "r")))
		return -1;
	fgets(cmd, 1024, fp);
	if (!fgets(cmd, 1024, fp))
		return -1;
	pclose(fp);

	sscanf(cmd, "%*s %*s %*s %*s [%x:%x:%x:%x:%x:%x] %*s",&a[0],&a[1],&a[2],&a[3],&a[4],&a[5]);
	sprintf(smac, "%0.2x:%0.2x:%0.2x:%0.2x:%0.2x:%0.2x",a[0],a[1],a[2],a[3],a[4],a[5]);

	return 0;
}

int get_br_port(char *smac, int *port) {

	FILE *fp;
	char cmd[1024];
	char mac[18];

	if (!(fp = popen("brctl showmacs br0", "r"))) 
		return -1;
	fgets(cmd, 1024, fp);
	while (fgets(cmd, 1024, fp)) {
		sscanf(cmd, "%d %s %*s %*s", port, mac);
		if (!strcmp(mac, smac)) {
			pclose(fp);
			return 0;
		}
	}		
	pclose(fp);
	return -1;
}

void br_update(struct iphdr *iph, unsigned int mca, int type) {
	char smac[18];
	int fd;
	int port;
	struct ifreq ifr;	
	unsigned long args[4];

	/* get src mac */				
	if (get_smac(iph, smac) < 0) {
	        log_info(LOG_WARNING, 0, "igmpd: error getting smac\n");
		return;
	}

	/* get br port */
	if (get_br_port(smac, &port) < 0) {
	        log_info(LOG_WARNING, 0, "igmpd: error getting br port\n");
		return;
	}

	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	        log_info(LOG_WARNING, 0, "igmpd: error opring socket\n");
		return;
	}

	strcpy(ifr.ifr_ifrn.ifrn_name, "br0");

	args[0] = 19;
	args[1] = port;
	args[2] = mca;
	args[3] = type;
	ifr.ifr_data = (char *)&args;
	ioctl(fd, SIOCDEVPRIVATE, &ifr);
}

/*
void process_pkt(const char *inif, const char *outif,
									char *buf, int len) {

	int i, cnt;
	char cmd[1024];
	struct iphdr *iph = (struct iphdr *)buf;
	struct igmphdr *ih = (struct igmphdr*)((char*)iph + iph->ihl*4);
	struct igmpv3_report *ih3;
	struct igmpv3_grec *grec;

	i = ntohs(ih->csum);
	ih->csum = 0;
	if (ip_sum_calc(ntohs(iph->tot_len) - iph->ihl*4, (char *)ih) != i)
		return;

	switch (ih->type) {

		case IGMPV2_HOST_MEMBERSHIP_REPORT:
			sprintf(cmd, "/sbin/smcroute -a %s 0.0.0.0 %d.%d.%d.%d %s",
													inif,
													ih->group & 0xff,
													(ih->group >> 8) & 0xff,
													(ih->group >> 16) & 0xff,
													(ih->group >> 24) & 0xff,
													outif);
			system(cmd);
			if(!strcmp(outif, "br0")) 
				br_update(iph, ih->group, ih->type);	
			break;

		case IGMP_HOST_LEAVE_MESSAGE:
			sprintf(cmd, "/sbin/smcroute -r %s 0.0.0.0 %d.%d.%d.%d",
													inif,
													ih->group & 0xff,
													(ih->group >> 8) & 0xff,
													(ih->group >> 16) & 0xff,
													(ih->group >> 24) & 0xff);
			system(cmd);
			if(!strcmp(outif, "br0")) 
				br_update(iph, ih->group, ih->type);	
			break;

		case IGMPV3_HOST_MEMBERSHIP_REPORT:
			ih3 = (struct igmpv3_report *)ih;

    		for (i = 0; i < ntohs(ih3->ngrec); i++) {
                	grec = &ih3->grec[i];
					if (grec->grec_type == IGMPV3_CHANGE_TO_INCLUDE)
						sprintf(cmd, "/sbin/smcroute -r %s 0.0.0.0 %d.%d.%d.%d",
													inif,
													grec->grec_mca & 0xff,
													(grec->grec_mca >> 8) & 0xff,
													(grec->grec_mca >> 16) & 0xff,
													(grec->grec_mca >> 24) & 0xff);
					else if (grec->grec_type == IGMPV3_CHANGE_TO_EXCLUDE)
						sprintf(cmd, "/sbin/smcroute -a %s 0.0.0.0 %d.%d.%d.%d %s",
													inif,
													grec->grec_mca & 0xff,
													(grec->grec_mca >> 8) & 0xff,
													(grec->grec_mca >> 16) & 0xff,
													(grec->grec_mca >> 24) & 0xff,
													outif);
				system(cmd);

				if(!strcmp(outif, "br0")) 
					br_update(iph, grec->grec_mca, grec->grec_type);	
			}
			break;
	}
}
*/
#endif

/**
 * Process a newly received IGMP packet that is sitting in the input
 * packet buffer.
 */
void acceptIgmp(int recvlen) {
    register uint32 src, dst, group;
    struct ip *ip;
    struct igmp *igmp;
#ifdef CONFIG_ATHEROS_HEADER_EN
    int port = 0;
#endif
    int ipdatalen, iphdrlen, igmpdatalen;

#ifdef MV88F6281
    struct iphdr *iph;
    struct IfDesc *srcVif;
#endif

    if (recvlen < sizeof(struct ip)) {
        log_msg(LOG_WARNING, 0,
            "received packet too short (%u bytes) for IP header", recvlen);
        return;
    }

    ip        = (struct ip *)recv_buf;
    src       = ip->ip_src.s_addr;
    dst       = ip->ip_dst.s_addr;

#ifdef MV88F6281
    iph       = (struct iphdr *)recv_buf;
#endif

#ifdef CONFIG_ATHEROS_HEADER_EN
    // Skip 239.255.255.0 packet 
    if((dst == htonl(0xEFFFFFFA)) || (src == htonl(0xEFFFFFFA))) {
	return;
    }

#endif
    //IF_DEBUG log_msg(LOG_DEBUG, 0, "Got a IGMP request to process...");

    /* 
     * this is most likely a message from the kernel indicating that
     * a new src grp pair message has arrived and so, it would be 
     * necessary to install a route into the kernel for this.
     */
    if (ip->ip_p == 0) {
        if (src == 0 || dst == 0) {
            log_msg(LOG_WARNING, 0, "kernel request not accurate");
        }
        else {
#ifdef CONFIG_ATHEROS_HEADER_EN
            struct IfDesc *checkVIF;
	    		// if snooping mode,return.
            if(commonConfig.mode==1) return;

            // Check if the source address matches a valid address on upstream vif.
            checkVIF = getIfByIx( upStreamVif );
            if(checkVIF == 0) {
                log_msg(LOG_ERR, 0, "Upstream VIF was null.");
                return;
            } 
            else if(src == checkVIF->InAdr.s_addr) {
                log_msg(LOG_NOTICE, 0, "Route activation request from %s for %s is from myself. Ignoring.",
                    inetFmt(src, s1), inetFmt(dst, s2));
                return;
            }
            else if(!isAdressValidForIf(checkVIF, src)) {
                log_msg(LOG_WARNING, 0, "The source address %s for group %s, is not in any valid net for upstream VIF.",
                    inetFmt(src, s1), inetFmt(dst, s2));
                //return;
            }
#else // #ifdef CONFIG_ATHEROS_HEADER_EN
/* jimmy added 20080829 */
/*
When client join Multicast Group, igmpproxy will record 
[WAN_Mcast_PC_src -> Mcast_group] in his own internel route records 
and add to kernel route cache .

And here will update igmpproxy internel route records 
[WAN_Mcast_PC_src -> Mcast_group] to [router_WAN -> Mcast_group]
and additionally add another route cache pair [router_WAN -> Mcast_group]
to kernel

Thus, when client leave the group, igmpproxy will use his 
internel records [router_WAN : Mcast_group] to remove kernel's route cache
without removing [WAN_Mcast_PC_src : Mcast_group],
so kernel is continuously forwarding UDP packets

To fix this, when receive "Route activate request" , if src is our router's wan ip,
ignore it !
*/
			struct IfDesc *upstrIf;
			int i;
			for(i=0;i<2;i++)
			{
				if(upStreamVif[i] == -1){
					continue;
				}
				
				upstrIf = getIfByIx( upStreamVif[i] );
				if(upstrIf != NULL) {
					if(strcmp(inetFmt(upstrIf->InAdr.s_addr,s1),inetFmt(src,s2)) == 0){
						log_msg(LOG_DEBUG,0,"Route activate request %s to %s is from myself, Ignoring it !",inetFmt(src,s1),inetFmt(dst,s2));
						log_msg(LOG_DEBUG,0,"Upstream : %s",inetFmt(upstrIf->InAdr.s_addr,s1));
						return ;
					}
				}
			}
/* ------------------------------------------------------------ */
#if 0    /*remove for cd-router test wan remoteHostIp is different subnet with wan mask*/          
            struct IfDesc *checkVIF;
            
            // Check if the source address matches a valid address on upstream vif.
            checkVIF = getIfByIx( upStreamVif );
            
            if(checkVIF == 0) {
                log_msg(LOG_ERR, 0, "Upstream VIF was null.");
                return;
            } 
            else if(src == checkVIF->InAdr.s_addr) {
                log_msg(LOG_NOTICE, 0, "Route activation request from %s for %s is from myself. Ignoring.",
                    inetFmt(src, s1), inetFmt(dst, s2));
                return;
            }
            else if(!isAdressValidForIf(checkVIF, src)) {
                log_msg(LOG_WARNING, 0, "The source address %s for group %s, is not in any valid net for upstream VIF.",
                    inetFmt(src, s1), inetFmt(dst, s2));
                return;
            }
            
#else
#ifdef CONFIG_SUPPORT_PRIVACY_ADDRESS
	    if ((dst & LOCAL_PRIVATE_NETMASK) == LOCAL_PRIVATE_NETMASK){
	    	IF_DEBUG log_msg(LOG_DEBUG, 0, "Got private dst multicast address %s passthrough return ",
                         	inetFmt(dst,s2));
                return;
            }             	                
#endif            
#endif // #if 0
#endif // #ifdef CONFIG_ATHEROS_HEADER_EN
            // Activate the route.
            IF_DEBUG log_msg(LOG_DEBUG, 0, "Route activate request from %s to %s",
                         inetFmt(src,s1), inetFmt(dst,s2));
            activateRoute(dst, src);
            

        }
        return;
    }

    iphdrlen  = ip->ip_hl << 2;
    ipdatalen = ntohs(ip->ip_len) - iphdrlen;

    if (iphdrlen + ipdatalen != recvlen) {
        log_msg(LOG_WARNING, 0,
            "received packet from %s shorter (%u bytes) than hdr+data length (%u+%u)",
            inetFmt(src, s1), recvlen, iphdrlen, ipdatalen);
        return;
    }

    igmp        = (struct igmp *)(recv_buf + iphdrlen);
    group       = igmp->igmp_group.s_addr;
#ifdef CONFIG_ATHEROS_HEADER_EN

    if ((((char *) (&igmp->igmp_cksum))[0] == 0x7d) 
          && ((((char *) (&igmp->igmp_cksum))[1] & 0xf0) == 0x50))
        port = (uint32) ((char *) (&igmp->igmp_cksum))[1] & 0xf;

#endif
    igmpdatalen = ipdatalen - IGMP_MINLEN;
    if (igmpdatalen < 0) {
        log_msg(LOG_WARNING, 0,
            "received IP data field too short (%u bytes) for IGMP, from %s",
            ipdatalen, inetFmt(src, s1));
        return;
    }
#ifdef CONFIG_ATHEROS_HEADER_EN
    log_msg(LOG_NOTICE, 0, "RECV %s from %-15s to %s at port %d",
#else
    log_msg(LOG_NOTICE, 0, "RECV %s from %-15s to %s",
#endif
        igmpPacketKind(igmp->igmp_type, igmp->igmp_code),
#ifdef CONFIG_ATHEROS_HEADER_EN
        inetFmt(src, s1), inetFmt(dst, s2),port );
#else
        inetFmt(src, s1), inetFmt(dst, s2) );
#endif

    switch (igmp->igmp_type) {
    case IGMP_V1_MEMBERSHIP_REPORT:
    case IGMP_V2_MEMBERSHIP_REPORT:
#ifdef MV88F6281
	srcVif = getIfByAddress(src);
	if(!strcmp(srcVif->Name, "br0")) 
		br_update(iph, group, igmp->igmp_type);
#endif
#ifdef CONFIG_ATHEROS_HEADER_EN
        acceptGroupReport(port,src, group, igmp->igmp_type);
#else
        acceptGroupReport(src, group, igmp->igmp_type);
#endif
        return;
    
    case IGMP_V2_LEAVE_GROUP:
#ifdef MV88F6281
	srcVif = getIfByAddress(src);
	if(!strcmp(srcVif->Name, "br0")) 
		br_update(iph, group, igmp->igmp_type);
#endif
        acceptLeaveMessage(src, group);
        return;
    
    /*
    case IGMP_MEMBERSHIP_QUERY:
        //accept_membership_query(src, dst, group, igmp->igmp_code);
        return;

    */

    default:
        log_msg(LOG_INFO, 0,
            "ignoring unknown IGMP message type %x from %s to %s",
            igmp->igmp_type, inetFmt(src, s1),
            inetFmt(dst, s2));
        return;
    }
}


/*
 * Construct an IGMP message in the output packet buffer.  The caller may
 * have already placed data in that buffer, of length 'datalen'.
 */
void buildIgmp(uint32 src, uint32 dst, int type, int code, uint32 group, int datalen) {
    struct ip *ip;
    struct igmp *igmp;
    extern int curttl;

    ip                      = (struct ip *)send_buf;
    ip->ip_src.s_addr       = src;
    ip->ip_dst.s_addr       = dst;
    ip->ip_len              = MIN_IP_HEADER_LEN + IGMP_MINLEN + datalen;
    ip->ip_len              = htons(ip->ip_len);

    if (IN_MULTICAST(ntohl(dst))) {
        ip->ip_ttl = curttl;
    } else {
        ip->ip_ttl = MAXTTL;
    }

    igmp                    = (struct igmp *)(send_buf + MIN_IP_HEADER_LEN);
    igmp->igmp_type         = type;
    igmp->igmp_code         = code;
    igmp->igmp_group.s_addr = group;
    igmp->igmp_cksum        = 0;
    igmp->igmp_cksum        = inetChksum((u_short *)igmp,
                                         IGMP_MINLEN + datalen);
}

/* 
 * Call build_igmp() to build an IGMP message in the output packet buffer.
 * Then send the message from the interface with IP address 'src' to
 * destination 'dst'.
 */
void sendIgmp(uint32 src, uint32 dst, int type, int code, uint32 group, int datalen) {
    struct sockaddr_in sdst;
    int setloop = 0, setigmpsource = 0;

    buildIgmp(src, dst, type, code, group, datalen);

    if (IN_MULTICAST(ntohl(dst))) {
        k_set_if(src);
        setigmpsource = 1;
        if (type != IGMP_DVMRP || dst == allhosts_group) {
            setloop = 1;
            k_set_loop(TRUE);
        }
    }

    bzero(&sdst, sizeof(sdst));
    sdst.sin_family = AF_INET;
#ifdef HAVE_SA_LEN
    sdst.sin_len = sizeof(sdst);
#endif
    sdst.sin_addr.s_addr = dst;
    if (sendto(MRouterFD, send_buf,
               MIN_IP_HEADER_LEN + IGMP_MINLEN + datalen, 0,
               (struct sockaddr *)&sdst, sizeof(sdst)) < 0) {
        if (errno == ENETDOWN)
            log_msg(LOG_ERR, errno, "Sender VIF was down.");
        else
            log_msg(LOG_INFO, errno,
                "sendto to %s on %s",
                inetFmt(dst, s1), inetFmt(src, s2));
    }

    if(setigmpsource) {
        if (setloop) {
            k_set_loop(FALSE);
        }
        // Restore original...
        k_set_if(INADDR_ANY);
    }

    IF_DEBUG log_msg(LOG_DEBUG, 0, "SENT %s from %-15s to %s",
        igmpPacketKind(type, code), src == INADDR_ANY ? "INADDR_ANY" :
        inetFmt(src, s1), inetFmt(dst, s2));
}

