/* Shared library add-on to iptables to add MARK target support. */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>

#include <iptables.h>
#include <linux/netfilter_ipv4/ip_tables.h>
/* For 64bit kernel / 32bit userspace */
#include "../include/linux/netfilter_ipv4/ipt_MARK.h"

#if HAS_IPTABLES_HELP /* cfho 2008-0827 */
/* Function which prints out usage message. */
static void
help(void)
{
	printf(
"MARK target v%s options:\n"
"  --set-mark value                   Set nfmark value\n"
"  --and-mark value                   Binary AND the nfmark with value\n"
"  --or-mark  value                   Binary OR  the nfmark with value\n"
"\n",
IPTABLES_VERSION);
}
#endif

static struct option opts[] = {
	{ "set-mark", 1, 0, '1' },
	{ "and-mark", 1, 0, '2' },
	{ "or-mark", 1, 0, '3' },
	{ 0 }
};

/* Initialize the target. */
static void
init(struct ipt_entry_target *t, unsigned int *nfcache)
{
}

#if HAS_IPTABLES_COMPAT_REVISION /* mook 2009-0319 */
/* Function which parses command options; returns true if it
   ate an option */
static int
parse_v0(int c, char **argv, int invert, unsigned int *flags,
	 const struct ipt_entry *entry,
	 struct ipt_entry_target **target)
{
	struct ipt_mark_target_info *markinfo
		= (struct ipt_mark_target_info *)(*target)->data;

	switch (c) {
	case '1':
#ifdef KERNEL_64_USERSPACE_32
		if (string_to_number_ll(optarg, 0, 0, 
				     &markinfo->mark))
#else
		if (string_to_number_l(optarg, 0, 0, 
				     &markinfo->mark))
#endif
			exit_error(PARAMETER_PROBLEM, "Bad MARK value `%s'", optarg);
		if (*flags)
			exit_error(PARAMETER_PROBLEM,
			           "MARK target: Can't specify --set-mark twice");
		*flags = 1;
		break;
	case '2':
		exit_error(PARAMETER_PROBLEM,
			   "MARK target: kernel too old for --and-mark");
	case '3':
		exit_error(PARAMETER_PROBLEM,
			   "MARK target: kernel too old for --or-mark");
	default:
		return 0;
	}

	return 1;
}
#endif /* HAS_IPTABLES_COMPAT_REVISION */

static void
final_check(unsigned int flags)
{
	if (!flags)
		exit_error(PARAMETER_PROBLEM,
		           "MARK target: Parameter --set/and/or-mark"
			   " is required");
}

/* Function which parses command options; returns true if it
   ate an option */
static int
parse_v1(int c, char **argv, int invert, unsigned int *flags,
	 const struct ipt_entry *entry,
	 struct ipt_entry_target **target)
{
	struct ipt_mark_target_info_v1 *markinfo
		= (struct ipt_mark_target_info_v1 *)(*target)->data;

	switch (c) {
	case '1':
	        markinfo->mode = IPT_MARK_SET;
		break;
	case '2':
	        markinfo->mode = IPT_MARK_AND;
		break;
	case '3':
	        markinfo->mode = IPT_MARK_OR;
		break;
	default:
		return 0;
	}

#ifdef KERNEL_64_USERSPACE_32
	if (string_to_number_ll(optarg, 0, 0,  &markinfo->mark))
#else
	if (string_to_number_l(optarg, 0, 0, &markinfo->mark))
#endif
		exit_error(PARAMETER_PROBLEM, "Bad MARK value `%s'", optarg);

	if (*flags)
		exit_error(PARAMETER_PROBLEM,
			   "MARK target: Can't specify --set-mark twice");

	*flags = 1;
	return 1;
}

#if HAS_IPTABLES_PRINT || HAS_IPTABLES_SAVE
#ifdef KERNEL_64_USERSPACE_32
static void
print_mark(unsigned long long mark)
{
	printf("0x%llx ", mark);
}
#else
static void
print_mark(unsigned long mark)
{
	printf("0x%lx ", mark);
}
#endif
#endif

#if HAS_IPTABLES_COMPAT_REVISION /* mook 2009-0319 */
#if HAS_IPTABLES_PRINT
/* Prints out the targinfo. */
static void
print_v0(const struct ipt_ip *ip,
	 const struct ipt_entry_target *target,
	 int numeric)
{
	const struct ipt_mark_target_info *markinfo =
		(const struct ipt_mark_target_info *)target->data;
	printf("MARK set ");
	print_mark(markinfo->mark);
}
#endif

#if HAS_IPTABLES_SAVE /* cfho 2008-0827 */
/* Saves the union ipt_targinfo in parsable form to stdout. */
static void
save_v0(const struct ipt_ip *ip, const struct ipt_entry_target *target)
{
	const struct ipt_mark_target_info *markinfo =
		(const struct ipt_mark_target_info *)target->data;

	printf("--set-mark ");
	print_mark(markinfo->mark);
}
#endif
#endif /* HAS_IPTABLES_COMPAT_REVISION */

#if HAS_IPTABLES_PRINT
/* Prints out the targinfo. */
static void
print_v1(const struct ipt_ip *ip,
	 const struct ipt_entry_target *target,
	 int numeric)
{
	const struct ipt_mark_target_info_v1 *markinfo =
		(const struct ipt_mark_target_info_v1 *)target->data;

	switch (markinfo->mode) {
	case IPT_MARK_SET:
		printf("MARK set ");
		break;
	case IPT_MARK_AND:
		printf("MARK and ");
		break;
	case IPT_MARK_OR: 
		printf("MARK or ");
		break;
	}
	print_mark(markinfo->mark);
}
#endif

#if HAS_IPTABLES_SAVE /* cfho 2008-0827 */
/* Saves the union ipt_targinfo in parsable form to stdout. */
static void
save_v1(const struct ipt_ip *ip, const struct ipt_entry_target *target)
{
	const struct ipt_mark_target_info_v1 *markinfo =
		(const struct ipt_mark_target_info_v1 *)target->data;

	switch (markinfo->mode) {
	case IPT_MARK_SET:
		printf("--set-mark ");
		break;
	case IPT_MARK_AND:
		printf("--and-mark ");
		break;
	case IPT_MARK_OR: 
		printf("--or-mark ");
		break;
	}
	print_mark(markinfo->mark);
}
#endif

#if HAS_IPTABLES_COMPAT_REVISION /* mook 2009-0319 */
static
struct iptables_target mark_v0 = {
	.next		= NULL,
	.name		= "MARK",
	.version	= IPTABLES_VERSION,
	.revision	= 0,
	.size		= IPT_ALIGN(sizeof(struct ipt_mark_target_info)),
	.userspacesize	= IPT_ALIGN(sizeof(struct ipt_mark_target_info)),
#if HAS_IPTABLES_HELP /* cfho 2008-0827 */
        .help          = &help,
#else
        .help          = 0,
#endif
	.init		= &init,
	.parse		= &parse_v0,
	.final_check	= &final_check,
#if HAS_IPTABLES_PRINT
	.print		= &print_v0,
#else
	.print		= 0,
#endif
#if HAS_IPTABLES_SAVE /* cfho 2008-0827 */
        .save           = &save_v0,
#else
	.save		= 0,
#endif
	.extra_opts	= opts
};
#endif /* HAS_IPTABLES_COMPAT_REVISION */

static
struct iptables_target mark_v1 = {
	.next		= NULL,
	.name		= "MARK",
	.version	= IPTABLES_VERSION,
	.revision	= 1,
	.size		= IPT_ALIGN(sizeof(struct ipt_mark_target_info_v1)),
	.userspacesize	= IPT_ALIGN(sizeof(struct ipt_mark_target_info_v1)),
#if HAS_IPTABLES_HELP /* cfho 2008-0827 */
        .help          = &help,
#else
        .help          = 0,
#endif
	.init		= &init,
	.parse		= &parse_v1,
	.final_check	= &final_check,
#if HAS_IPTABLES_PRINT
	.print		= &print_v1,
#else
	.print		= 0,
#endif
#if HAS_IPTABLES_SAVE /* cfho 2008-0827 */
        .save           = &save_v1,
#else
	.save		= 0,
#endif
	.extra_opts	= opts
};

void _init(void)
{
#if HAS_IPTABLES_COMPAT_REVISION /* mook 2009-0319 */
	register_target(&mark_v0);
#endif
	register_target(&mark_v1);
}
