/*
PCM chardriver. The ALSA infrastructure is too large for the purpose,
so we use a chardriver interface.

The driver supports two channels, represented by the minor numbers 0 & 1.

Settings are set via IOCTL, see below for definitions.

Only the the first channel (minor number 0) is allowed to change global
settings. Each channel can change its own private settings.

Channel sample data flows through the read/write methods. Both functions
may be read/written in blocking or non-blocking mode. select/poll is
also implemented. In case of read data overflow, data gets lost. In
case of a write underflow, samples may be repeated.

The default settings are master mode, 8KHz frame rate, 32 bits/frame,
with channel 0 starting at bit 0 and channel 1 starting at bit 16.
The purpose of this is to allow simple scripting of a loopback test,
using shell 'cat' commands (e.g. cat /dev/pcm0 >/dev/pcm1).

The /proc/pcm entry shows current settings and state plus some statistics.

Iwo Mergler <Iwo@call-direct.com.au>
*/
#ifndef _RT305X_PCM_H
#define _RT305X_PCM_H

#include <linux/ioctl.h>

#ifdef __KERNEL__
#include <asm/rt2880/rt_mmap.h>

#define INT_REFCLK 15625000

#define SYSREG_CCR        ((volatile void __iomem *)RALINK_SYSCTL_BASE+0x30)

#define PCM_BASE ((volatile void __iomem *)RALINK_PCM_BASE)

#define PCMREG_GLB_CFG    (PCM_BASE+0x00)
#define PCMREG_CFG        (PCM_BASE+0x04)
#define PCMREG_INT_STATUS (PCM_BASE+0x08)
#define PCMREG_INT_EN     (PCM_BASE+0x0c)
#define PCMREG_FF_STATUS  (PCM_BASE+0x10)
#define PCMREG_CH_CFG(n)  (PCM_BASE+((n==0)?0x20:0x24))
#define PCMREG_SPARE      (PCM_BASE+0x30)
#define PCMREG_CH_FIFO(n) (PCM_BASE+((n==0)?0x80:0x84))

#endif /* __KERNEL__ */

/* Number of bytes to buffer in driver. Multiple of 32 bits. */
#define PCM_BUFFER 4096

#define PCM_ENABLE  1
#define PCM_DISABLE 0

#define USE_FIXED_CLK_DIVIDER

/* Global PCM settings (affects both channels) */
struct pcm_settings {
	/* MASTER/SLAVE selection */
	unsigned mode;
#define PCM_MODE_OFF    0
#define PCM_MODE_MASTER 1
#define PCM_MODE_SLAVE  2

	/* Internal PCM clock, master mode only. When setting this value,
	   the driver may not achieve it exactly. In either case, it returns
	   the exact bitrate in this field. */
	unsigned clk_speed_hz;
#define	PCLK_256KHZ 	256000
#define	PCLK_512KHZ		512000
#define	PCLK_1P024MHZ	1024000
#define	PCLK_2P048MHZ	2048000
#define	PCLK_4P096MHZ	4096000
#define	PCLK_8P192MHZ	8192000

	/* PCM frame size - bits */
	unsigned framesize;
#define PCM_FRAME_32   0
#define PCM_FRAME_64   1
#define PCM_FRAME_128  2
#define PCM_FRAME_256  3
#define PCM_FRAME_512  4
#define PCM_FRAME_1024 5

#ifdef USE_FIXED_CLK_DIVIDER
	unsigned clk_divider;
#endif

};

struct pcm_clk_table_type {
	unsigned clk_speed_hz;
	unsigned framesize;
	unsigned clk_divider;
};

struct pcm_clk_table_type pcm_clk_table[6] = {
		{ PCLK_256KHZ, 		PCM_FRAME_32,		60 },
		{ PCLK_512KHZ, 		PCM_FRAME_64,		29 },	// or 30
		{ PCLK_1P024MHZ, 	PCM_FRAME_128,		15 },
		{ PCLK_2P048MHZ, 	PCM_FRAME_256,		6  },	// or 7
		{ PCLK_4P096MHZ, 	PCM_FRAME_512,		3  },
		{ PCLK_8P192MHZ, 	PCM_FRAME_1024, 	1  } };

typedef enum {
	PCM_CLK_256 = 0,
	PCM_CLK_512,
	PCM_CLK_1024,
	PCM_CLK_2048,
	PCM_CLK_4096,
	PCM_CLK_8192
} pcm_clk_enum_type;

typedef enum {
	PCM_CLK_CHANGE = 0,
	PCM_LOOPBACK_MODE
} pcm_drv_cmd_enum_type;

/* Per-channel PCM settings */
struct pcm_channel_settings {
	/* Loopback switch (modes can be OR-ed together) */
	unsigned loopback;
#define PCM_LOOPBACK_NONE 0  /* Normal operation */
#define PCM_LOOPBACK_FIFO 1  /* TXFIFO -> RXFIFO */
#define PCM_LOOPBACK_EXT  2  /* DRX->DTX */

	/* Sample mode setting */
	unsigned samples;
#define PCM_SAMPLE_16   0 /* 16 bits / sample, e.g. linear PCM16 */
#define PCM_SAMPLE_8    1 /* 8 bits / sample, e.g. x-law encoded */
#define PCM_SAMPLE_ULAW 2 /* hardware conversion: PCM16 -> U-law */
#define PCM_SAMPLE_ALAW 3 /* hardware conversion: PCM16 -> A-law */

	/* Bit number of first bit in timeslot */
	unsigned slot_start;
};

#define PCM_IOCTL_MAGIC 0xF5

#define PCM_GETGLOBAL _IOR(PCM_IOCTL_MAGIC,0,struct pcm_settings)
#define PCM_SETGLOBAL _IORW(PCM_IOCTL_MAGIC,1,struct pcm_settings)

#define PCM_GET       _IOR(PCM_IOCTL_MAGIC,2,struct pcm_channel_settings)
#define PCM_SET       _IOW(PCM_IOCTL_MAGIC,3,struct pcm_channel_settings)

#endif /* _RT305X_PCM_H */
