/*
GPIO support for Ralink RT3052 / RT3050

Iwo Mergler <Iwo@call-direct.com.au>
*/
#ifndef _ARCH_RT2880_GPIO_H
#define _ARCH_RT2880_GPIO_H

#include <asm/errno.h>

#include <asm/rt2880/surfboardint.h>
#include <asm/rt2880/rt_mmap.h>
#define CONFIG_RALINK_PARANOID_GPIO
/*
 *  GPIO IRQ base. The processor has 6 interrupts + 32 interrupts in main interrupt controller
 *  MIPS default maximum is 128. So we use an offset of 64.
 */

#define RALINK_GPIO_INTBASE 64

/*
 * Base addresses
 */

#define RALINK_PRGIO_ADDR        ((volatile void __iomem *)RALINK_PIO_BASE)    // Programmable I/O

/* The register banks have a gratuitous jump in the layout. */
#define GPIOREG_OFFSET(pin,bank0,bank1,bank2) (((pin)<24)?(bank0):((pin)<40)?(bank1):(bank2))
#define GPIOREG_PIOINT(pin) (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x00,0x38,0x60))
#define GPIOREG_EDGE(pin)   (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x04,0x3c,0x64))
#define GPIOREG_RENA(pin)   (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x08,0x40,0x68))
#define GPIOREG_FENA(pin)   (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x0c,0x44,0x6c))
#define GPIOREG_DATA(pin)   (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x20,0x48,0x70))
#define GPIOREG_DIR(pin)    (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x24,0x4c,0x74))
#define GPIOREG_POL(pin)    (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x28,0x50,0x78))
#define GPIOREG_SET(pin)    (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x2c,0x54,0x7c))
#define GPIOREG_RESET(pin)  (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x30,0x58,0x80))
#define GPIOREG_TOG(pin)    (RALINK_PRGIO_ADDR+GPIOREG_OFFSET(pin,0x34,0x5c,0x84))

#define GPIO_BANKS 3

/* Returns first pin in bank */
#define BANK(n) (n==0?0:n==1?24:40)

/* Location of pin in bank */
#define GPIO_SHIFT(pin)     (((pin)<24)?(pin):((pin)<40)?(pin)-24:(pin-40))

/* Number of possible GPIOS, 0 - 51 */
#define GPIO_PINS  52
#define GPIO_PINVALID(pin) ((unsigned int)(pin) < GPIO_PINS)

/* Multiplexer constants, for gpio_muxmode() below. Pins are switched in groups.
   There are further restrictions for the pins 7-14 range. */
enum gpiomuxmode {
	GPIO,                           /* All pins */
	MUX,                            /* Pins 1-6 & 15-51 only */
	UARTF, MODEM, I2S, PCM, REFCLK, /* Pins 7-14 only */
	DONTCARE                        /* Internal use only */
};


#ifndef CONFIG_RALINK_PARANOID_GPIO /* Normal operations without checking */

/*
Call this before any other GPIO operation. Returns !0 if already in use.
*/
extern int __must_check gpio_request(unsigned gpio, const char *label);

/*
Call this at the end of the GPIO operations. The state of the pin
is unchanged.
*/
extern void gpio_free(unsigned gpio);

/*
Switch pin to input mode. Also switches muxmode to GPIO.
*/
extern int gpio_direction_input(unsigned gpio);

/*
Switch pin to output mode and given value. Also switches muxmode to GPIO.
*/
extern int gpio_direction_output(unsigned gpio, int value);

/*
Read the current multiplexer mode of the pin.
*/
extern enum gpiomuxmode gpio_get_muxmode(unsigned gpio);

/*
Switch pin mux mode. RT305x - specific, operates on groups of pins.
*/
extern int gpio_set_muxmode(unsigned gpio, enum gpiomuxmode mode);

/*
Returns the value of a GPIO pin (0/!0). This also works in MUX or output mode.
Please use gpio_get_value_cansleep() outside atomic/interrupt context.
*/
static inline int gpio_get_value(unsigned gpio)
{
	return (readl(GPIOREG_DATA(gpio)) & (1<<GPIO_SHIFT(gpio)));
}

/*
Sets the value of a pin (0/1). If the pin is not in output mode, the behaviour
is undefined. Please use gpio_set_value_cansleep() outside atomic/interrupt context.
*/
static inline void gpio_set_value(unsigned gpio, int value)
{
	if (value)
		writel(1<<GPIO_SHIFT(gpio),GPIOREG_SET(gpio));
	else
		writel(1<<GPIO_SHIFT(gpio),GPIOREG_RESET(gpio));
}

/*
Retrieve the irq number attached to the specified pin. You must then call
request_interrupt() to register a handler. The RT 305x GPIOs only support
edge-triggered interrupts on rising and falling edges.
*/
static inline int gpio_to_irq(unsigned gpio)
{
	return(gpio + RALINK_GPIO_INTBASE);
}

/*
Translates irq number back to the GPIO pin. This may be
used inside an interrupt handler function.
*/
static inline int irq_to_gpio(unsigned irq)
{
	return (irq - RALINK_GPIO_INTBASE);
}

#else  /* CONFIG_RALINK_PARANOID_GPIO */

/* The following macros and prototypes allow caller tracking. See gpio.c
for details. */

#define gpio_request(gpio, label) _gpio_request(gpio,label, __func__, __LINE__)
extern int __must_check _gpio_request(unsigned gpio, const char *label, const char *caller, int line);
#define gpio_free(gpio) _gpio_free(gpio, __func__, __LINE__)
extern void _gpio_free(unsigned gpio, const char *caller, int line);

#define gpio_direction_input(gpio) _gpio_direction_input(gpio, __func__, __LINE__)
extern int _gpio_direction_input(unsigned gpio, const char *caller, int line);
#define gpio_direction_output(gpio, value) _gpio_direction_output(gpio, value, __func__, __LINE__)
extern int _gpio_direction_output(unsigned gpio, int value, const char *caller, int line);

#define gpio_get_muxmode(gpio) _gpio_get_muxmode(gpio, __func__, __LINE__)
extern enum gpiomuxmode _gpio_get_muxmode(unsigned gpio, const char *caller, int line);
#define gpio_set_muxmode(gpio,mode) _gpio_set_muxmode(gpio, mode, __func__, __LINE__)
extern int _gpio_set_muxmode(unsigned gpio, enum gpiomuxmode mode, const char *caller, int line);

#define gpio_get_value(gpio) _gpio_get_value(gpio, __func__, __LINE__)
extern int _gpio_get_value(unsigned gpio, const char *caller, int line);
#define gpio_set_value(gpio, value) _gpio_set_value(gpio, value, __func__, __LINE__)
extern void _gpio_set_value(unsigned gpio, int value, const char *caller, int line);

#define gpio_to_irq(gpio) _gpio_to_irq(gpio, __func__, __LINE__)
extern int _gpio_to_irq(unsigned gpio, const char *caller, int line);
#define irq_to_gpio(irq) _irq_to_gpio(irq, __func__, __LINE__)
extern int _irq_to_gpio(unsigned irq, const char *caller, int line);

#endif /* CONFIG_RALINK_PARANOID_GPIO */

#include <asm-generic/gpio.h> /* For *_cansleep functions */

#endif /* _ARCH_RT2880_GPIO_H */
