/*
  Mode switching tool for controlling flip flop (multiple device) USB gear
  Version 0.9.7beta, 2009/03/14

  Copyright (C) 2007, 2008, 2009 Josua Dietze (mail to "usb_admin" at the
  domain from the README; please do not post the complete address to The Net!
  Or write a personal message through the forum to "Josh")

  Command line parsing, decent usage/config output/handling, bugfixes and advanced
  options added by:
    Joakim Wennergren (jokedst) (gmail.com)

  TargetClass parameter implementation to support new Option devices/firmware:
    Paul Hardwick (http://www.pharscape.org)

  Created with initial help from:
    "usbsnoop2libusb.pl" by Timo Lindfors (http://iki.fi/lindi/usb/usbsnoop2libusb.pl)

  Config file parsing stuff borrowed from:
    Guillaume Dargaud (http://www.gdargaud.net/Hack/SourceCode.html)

  Hexstr2bin function borrowed from:
    Jouni Malinen (http://hostap.epitest.fi/wpa_supplicant, from "common.c")

  Code, fixes and ideas from:
    Aki Makkonen
    Denis Sutter 
    Lucas Benedicic 
    Roman Laube 
	Vincent Teoh
	Tommy Cheng
	Daniel Cooper

  More contributors are listed in the config file.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details:

  http://www.gnu.org/licenses/gpl.txt

*/

/* Recommended tab size: 4 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <signal.h>
#include <ctype.h>
#include <getopt.h>
#include <usb.h>
#include "usb_modeswitch.h"

#define LINE_DIM 1024
#define BUF_SIZE 4096

int write_bulk(int endpoint, char *message, int length);
int read_bulk(int endpoint, char *buffer, int length);

char *TempPP=NULL;

struct usb_dev_handle *devh;

int DefaultVendor=0, DefaultProduct=0, TargetVendor=0, TargetProduct=0, TargetClass=0;
int MessageEndpoint=0, ResponseEndpoint=-1;
int targetDeviceCount=0;
int ret;
char DetachStorageOnly=0, HuaweiMode=0, SierraMode=0, SonyMode=0, QisdaMode=0, DLinkMode=0;
char verbose=0, show_progress=1, ResetUSB=0, CheckSuccess=0, config_read=0;
char MessageContent[LINE_DIM];
char ByteString[LINE_DIM/2];
char buf[BUF_SIZE];

// Settable Interface and Configuration (for debugging mostly) (jmw)
int Interface = 0, Configuration = -1, AltSetting = -1;

static struct option long_options[] =
{
	{"help",				no_argument, 0, 'h'},
	{"DefaultVendor",		required_argument, 0, 'v'},
	{"DefaultProduct",		required_argument, 0, 'p'},
	{"TargetVendor",		required_argument, 0, 'V'},
	{"TargetProduct",		required_argument, 0, 'P'},
	{"TargetClass",			required_argument, 0, 'C'},
	{"MessageEndpoint",		required_argument, 0, 'm'},
	{"MessageContent",		required_argument, 0, 'M'},
	{"ResponseEndpoint",	required_argument, 0, 'r'},
	{"DetachStorageOnly",	required_argument, 0, 'd'},
	{"DLinkMode",			required_argument, 0, 'D'},
	{"HuaweiMode",			required_argument, 0, 'H'},
	{"QisdaMode",			required_argument, 0, 'B'},
	{"SierraMode",			required_argument, 0, 'S'},
	{"SonyMode",			required_argument, 0, 'O'},
	{"NeedResponse",		required_argument, 0, 'n'},
	{"ResetUSB",			required_argument, 0, 'R'},
	{"config",				required_argument, 0, 'c'},
	{"verbose",				no_argument, 0, 'W'},
	{"quiet",				no_argument, 0, 'Q'},
	{"success",				required_argument, 0, 's'},
	{"Interface",			required_argument, 0, 'i'},
	{"Configuration",		required_argument, 0, 'u'},
	{"AltSetting",			required_argument, 0, 'a'},
	{0, 0, 0, 0}
};

void readConfigFile(const char *configFilename)
{
	if(verbose) printf("Reading config file: %s\n", configFilename);
	ParseParamHex(configFilename, TargetVendor);
	ParseParamHex(configFilename, TargetProduct);
	ParseParamHex(configFilename, TargetClass);
	ParseParamHex(configFilename, DefaultVendor);
	ParseParamHex(configFilename, DefaultProduct);
	ParseParamBool(configFilename, DetachStorageOnly);
	ParseParamBool(configFilename, HuaweiMode);
	ParseParamBool(configFilename, QisdaMode);
	ParseParamBool(configFilename, SierraMode);
	ParseParamBool(configFilename, SonyMode);
	ParseParamBool(configFilename, DLinkMode);
	ParseParamHex(configFilename, MessageEndpoint);
	ParseParamString(configFilename, MessageContent);
//    ParseParamHex(configFilename, NeedResponse);
	ParseParamHex(configFilename, ResponseEndpoint);
	ParseParamHex(configFilename, ResetUSB);
	ParseParamHex(configFilename, CheckSuccess);
	ParseParamHex(configFilename, Interface);
	ParseParamHex(configFilename, Configuration);
	ParseParamHex(configFilename, AltSetting);
	config_read = 1;
}

void printConfig()
{
	printf ("DefaultVendor=0x%x\n",		DefaultVendor);
	printf ("DefaultProduct=0x%x\n",	DefaultProduct);
	printf ("TargetVendor=0x%x\n",		TargetVendor);
	printf ("TargetProduct=0x%x\n",		TargetProduct);
	printf ("TargetClass=0x%x\n",		TargetClass);
	printf ("DetachStorageOnly=%i\n",	(int)DetachStorageOnly);
	printf ("DLinkMode=%i\n",			(int)DLinkMode);
	printf ("HuaweiMode=%i\n",			(int)HuaweiMode);
	printf ("QisdaMode=%i\n",			(int)QisdaMode);
	printf ("SierraMode=%i\n",			(int)SierraMode);
	printf ("SonyMode=%i\n",			(int)SonyMode);
	printf ("MessageEndpoint=0x%x\n",	MessageEndpoint);
	printf ("MessageContent=\"%s\"\n",	MessageContent);
//	printf ("NeedResponse=%i\n",		(int)NeedResponse);
	if ( ResponseEndpoint > -1 )
		printf ("ResponseEndpoint=0x%x\n",	ResponseEndpoint);
	printf ("Interface=0x%x\n",			Interface);
	if ( Configuration > -1 )
		printf ("Configuration=0x%x\n",	Configuration);
	if ( AltSetting > -1 )
		printf ("AltSetting=0x%x\n",	AltSetting);
	if ( CheckSuccess )
		printf ("\nSuccess check enabled, settle time %d seconds\n", CheckSuccess);
	else
		printf ("\nSuccess check disabled\n");
	printf ("\n");
}

int readArguments(int argc, char **argv)
{
	int c, option_index = 0, count=0;
	if(argc==1) return 0;

	while (1)
	{
		c = getopt_long (argc, argv, "hWQs:v:p:V:P:C:m:M:r:d:H:B:D:S:O:n:c:R:i:u:a:",
						long_options, &option_index);
	
		/* Detect the end of the options. */
		if (c == -1)
			break;
		count++;
		switch (c)
		{
			case 'R': ResetUSB = strtol(optarg, NULL, 16); break;
			case 'v': DefaultVendor = strtol(optarg, NULL, 16); break;
			case 'p': DefaultProduct = strtol(optarg, NULL, 16); break;
			case 'V': TargetVendor = strtol(optarg, NULL, 16); break;
			case 'P': TargetProduct = strtol(optarg, NULL, 16); break;
			case 'C': TargetClass = strtol(optarg, NULL, 16); break;
			case 'm': MessageEndpoint = strtol(optarg, NULL, 16); break;
			case 'M': strcpy(MessageContent, optarg); break;
			case 'r': ResponseEndpoint = strtol(optarg, NULL, 16); break;
			case 'd': DetachStorageOnly = (toupper(optarg[0])=='Y' || toupper(optarg[0])=='T'|| optarg[0]=='1'); break;
			case 'D': DLinkMode = (toupper(optarg[0])=='Y' || toupper(optarg[0])=='T'|| optarg[0]=='1'); break;
			case 'H': HuaweiMode = (toupper(optarg[0])=='Y' || toupper(optarg[0])=='T'|| optarg[0]=='1'); break;
			case 'B': QisdaMode = (toupper(optarg[0])=='Y' || toupper(optarg[0])=='T'|| optarg[0]=='1'); break;
			case 'S': SierraMode = (toupper(optarg[0])=='Y' || toupper(optarg[0])=='T'|| optarg[0]=='1'); break;
			case 'O': SonyMode = (toupper(optarg[0])=='Y' || toupper(optarg[0])=='T'|| optarg[0]=='1'); break;
			case 'n': break;
			case 'c': readConfigFile(optarg); break;
			case 'W': verbose = 1; show_progress = 1; count--; break;
			case 'Q': show_progress = 0; verbose = 0; count--; break;
			case 's': CheckSuccess= strtol(optarg, NULL, 16); count--; break;

			case 'i': Interface = strtol(optarg, NULL, 16); break;
			case 'u': Configuration = strtol(optarg, NULL, 16); break;
			case 'a': AltSetting = strtol(optarg, NULL, 16); break;
	
			case 'h':
				printf ("Usage: usb_modeswitch [-hvpVPmMrdHn] [-c filename]\n\n");
				printf (" -h, --help                    this help\n");
				printf (" -v, --DefaultVendor [nr]      set vendor number to look for\n");
				printf (" -p, --DefaultProduct [nr]     set model number to look for\n");
				printf (" -V, --TargetVendor [nr]       target vendor; needed for success check\n");
				printf (" -P, --TargetProduct [nr]      target model; needed for success check\n");
				printf (" -C, --TargetClass [nr]        target device class\n");
				printf (" -m, --MessageEndpoint [nr]    where to direct command sequence\n");
				printf (" -M, --MessageContent [nr]     command sequence to send\n");
//				printf (" -n, --NeedResponse [1|0]      whether to try to read a response - OBSOLETE\n");
				printf (" -r, --ResponseEndpoint [nr]   if given, read response from there\n");
				printf (" -d, --DetachStorageOnly [1|0] whether to just detach the storage driver\n");
				printf (" -D, --DLinkMode [1|0]         whether to just apply a special sequence\n");
				printf (" -H, --HuaweiMode [1|0]        whether to just apply a special sequence\n");
				printf (" -B, --QisdaMode [1|0]         whether to just apply a special sequence\n");
				printf (" -S, --SierraMode [1|0]        whether to just apply a special sequence\n");
				printf (" -O, --SonyMode [1|0]          whether to just apply a special sequence\n");
				printf (" -R, --ResetUSB [1|0]          whether to reset the device in the end\n");
				printf (" -c, --config [filename]       load different config file\n");
				printf (" -Q, --quiet                   don't show progress or error messages\n");
				printf (" -W, --verbose                 print all settings before running\n");
				printf (" -s, --success [nr]            check switching result after [nr] secs\n\n");
				printf (" -i, --Interface               select initial USB interface (default 0)\n");
				printf (" -u, --Configuration           select USB configuration\n");
				printf (" -a, --AltSetting              select alternative USB interface setting\n\n");
				exit(0);
				break;
		
			default: //Unsupported - error message has already been printed
				printf ("\n");
				exit(1);
		}
	}

	return count;
}


int main(int argc, char **argv) {
	int numDefaults = 0, specialMode = 0;

	struct usb_device *dev;

	printf("\n * usb_modeswitch: tool for controlling \"flip flop\" mode USB devices\n");
   	printf(" * Version 0.9.7beta (C) Josua Dietze 2009\n");
   	printf(" * Works with libusb 0.1.12 and probably other versions\n\n");


	/*
	 * Parameter parsing, USB preparation/diagnosis, plausibility checks
	 */

	// Check command arguments, use params instead of config file when given
	switch (readArguments(argc, argv)) {
		case 0:						// no argument or -W, -q or -s
			readConfigFile("/etc/usb_modeswitch.conf");
			break;
		default:					// one or more arguments except -W, -q or -s 
			if (!config_read)		// if arguments contain -c, the config file was already processed
				if(verbose) printf("Taking all parameters from the command line\n\n");
	}

	if(verbose)
		printConfig();

	// libusb initialization
	usb_init();

	if (verbose)
		usb_set_debug(15);

	usb_find_busses();
	usb_find_devices();

	if (verbose)
		printf("\n");

	// Plausibility checks. The default IDs are mandatory
	if (!(DefaultVendor && DefaultProduct)) {
		if (show_progress) printf("No default vendor/product ID given. Aborting.\n\n");
		exit(1);
	}
	if (strlen(MessageContent)) {
		if (strlen(MessageContent) % 2 != 0) {
			if (show_progress) fprintf(stderr, "Error: MessageContent hex string has uneven length. Aborting.\n\n");
			exit(1);
		}
		if (!MessageEndpoint) {
			if (show_progress) fprintf(stderr, "Error: no MessageEndpoint given. Aborting.\n\n");
			exit(1);
		}
		if ( hexstr2bin(MessageContent, ByteString, strlen(MessageContent)/2) == -1) {
			if (show_progress) fprintf(stderr, "Error: MessageContent %s\n is not a hex string. Aborting.\n\n", MessageContent);
			exit(1);
		}
	}
	if (show_progress)
		if (CheckSuccess && !(TargetVendor && TargetProduct) && !TargetClass)
			printf("Note: target parameter missing; success check limited\n");

	// Count existing target devices (remember for success check)
	if ((TargetVendor && TargetProduct) || TargetClass) {
		if (show_progress) printf("Looking for target devices ...\n");
		search_devices(&targetDeviceCount, TargetVendor, TargetProduct, TargetClass);
		if (targetDeviceCount) {
			if (show_progress) printf(" Found devices in target mode or class (%d)\n", targetDeviceCount);
		} else {
			if (show_progress) printf(" No devices in target mode or class found\n");
		}
	}

	// Count default devices, return the last one found
	if (show_progress) printf("Looking for default devices ...\n");
	dev = search_devices(&numDefaults, DefaultVendor, DefaultProduct, TargetClass);
	if (numDefaults) {
		if (show_progress) printf(" Found default devices (%d)\n", numDefaults);
		if (TargetClass && !(TargetVendor && TargetProduct)) {
			if ( dev != NULL ) {
				if (show_progress) printf(" Found a default device NOT in target class mode\n");
			} else {
				if (show_progress) printf(" All devices in target class mode. Nothing to do. Bye!\n\n");
				exit(0);
			}
		}
	}
	if (dev != NULL) {
		if (show_progress) printf("Prepare switching, accessing device %03d on bus %03d ...\n", dev->devnum, strtol(dev->bus->dirname,NULL,10));
		devh = usb_open(dev);
	} else {
		if (show_progress) printf(" No default device found. Is it connected? Bye!\n\n");
		exit(0);
	}

	// Interface different from default, test it
	if (Interface > 0) {
		ret = usb_claim_interface(devh, Interface);
		if (ret != 0) {
			if (show_progress) printf("Could not claim interface %d (error %d). Aborting.\n\n", Interface, ret);
			exit(1);
		}
	}

	// Some scenarios are exclusive, so check for unwanted combinations
	specialMode = DetachStorageOnly + HuaweiMode + SierraMode + SonyMode + QisdaMode + DLinkMode;
	if ( specialMode > 1 ) {
		if (show_progress) printf("Invalid mode combination. Check your configuration. Aborting.\n\n");
		exit(1);
	}

	/*
	 * The switching actions
	 */

	if (DetachStorageOnly) {
		if (show_progress) printf("Only detaching storage driver for switching ...\n");
		ret = detachDriver();
		if (ret == 2) {
			if (show_progress) printf(" You may want to remove the storage driver manually. Bye\n\n");
//			exit(0);
		}
	}

	if (DLinkMode) {
		switchDLinkMode();
	}
	if (HuaweiMode) {
		switchHuaweiMode();
	}
	if (QisdaMode) {
		switchQisdaMode();
	}
	if (SierraMode) {
		switchSierraMode();
	}
	if (SonyMode) {
		switchSonyMode();
	}

	if (strlen(MessageContent) && MessageEndpoint) {
		if (specialMode == 0) {
			detachDriver();
			ret = switchSendMessage();
			if (ret && (ResponseEndpoint != -1))
				switchReadResponse();
		} else {
			if (show_progress) printf("Note: ignoring MessageContent. Can't combine with special mode\n");
		}
	}

	if (Configuration != -1) {
		switchConfiguration ();
	}

	if (AltSetting != -1) {
		switchAltSetting();
	}

	if (ResetUSB) {
		resetUSB();
	}

	if (devh)
		usb_close(devh);

	if (CheckSuccess) {
		signal(SIGTERM, release_usb_device);
		if (checkSuccess(dev))
			exit(0);
		else
			exit(1);
	} else {
		if (show_progress) printf("-> Run lsusb to note any changes. Bye\n\n");
		exit(0);
	}
}


int resetUSB () {
	int success;
	int bpoint = 0;

	if (show_progress) printf("Resetting usb device ");

	sleep( 1 );
	do {
		success = usb_reset(devh);
		if (show_progress) printf(".");
		bpoint++;
		if (bpoint > 100)
			success = 1;
	} while (success < 0);

	if ( success ) {
		if (show_progress) printf("\n Reset failed. Can be ignored if device switched OK.\n");
	} else {
		if (show_progress) printf("\n OK, device was reset\n");
	}
}

int switchSendMessage () {
	int message_length;

	if (show_progress) printf("Setting up communication with interface %d ...\n", Interface);
	ret = usb_claim_interface(devh, Interface);
	if (ret != 0) {
		if (show_progress) printf(" Could not claim interface (error %d). Skipping message sending\n", ret);
		return 0;
	}
	if (show_progress) printf("Trying to send the message to endpoint 0x%02x ...\n", MessageEndpoint);
	message_length = strlen(MessageContent) / 2;
	usb_clear_halt(devh, MessageEndpoint);
	write_bulk(MessageEndpoint, ByteString, message_length);

	usb_clear_halt(devh, MessageEndpoint);
	usb_release_interface(devh, Interface);
}

int switchReadResponse () {
	if (show_progress) printf("Reading the response to the message ...\n");
	return read_bulk(ResponseEndpoint, ByteString, LINE_DIM/2);
}

int switchConfiguration () {

	if (show_progress) printf("Changing configuration to %i ...\n", Configuration);
	ret = usb_set_configuration(devh, Configuration);
	if (ret == 0 ) {
		if (show_progress) printf(" OK, configuration set\n");
		return 1;
	} else {
		if (show_progress) printf(" Setting the configuration returned error %d. Trying to continue\n", ret);
		return 0;
	}
}

int switchAltSetting () {

	if (show_progress) printf("Changing to alt setting %i ...\n", AltSetting);
	ret = usb_claim_interface(devh, Interface);
	ret = usb_set_altinterface(devh, AltSetting);
	usb_release_interface(devh, Interface);
	if (ret != 0) {
		if (show_progress) fprintf(stderr, " Changing to alt setting returned error %d. Trying to continue\n", ret);
		return 0;
	} else {
		if (show_progress) printf(" OK, changed to alt setting\n");
		return 1;
	}
}

int switchDLinkMode () {

	if (show_progress)
		printf("Sending DLink control message ...\n");
	detachDriver();

	memcpy(buf, "\x55\x53\x42\x43\xe8\xb5\xce\x82\x00\x02\x00\x00\x00"
		    "\x00\x0a\x2a\x08\x00\x00\x00\x00\x00\x00\x01\x00\x00"
		    "\x00\x00\x00\x00\x00", 0x000001f);
	ret = usb_bulk_write(devh, 0x00000001, buf, 0x000001f, 1000);
	if (show_progress)
		printf(" usb bulk write: returned %d\n", ret);
	usleep(400);
	usleep(1000);

	memcpy(buf, "\xeb\x3c\x90\x4d\x53\x44\x4f\x53\x35\x2e\x30\x00\x02"
		    "\x01\x06\x00\x02\x00\x02\x00\xee\xf8\xed\x00\x3f\x00"
		    "\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x29"
		    "\x7c\xac\x14\x5c\x4e\x4f\x20\x4e\x41\x4d\x45\x20\x20"
		    "\x20\x20\x46\x41\x54\x31\x36\x20\x20\x20\x33\xc9\x8e"
		    "\xd1\xbc\xf0\x7b\x8e\xd9\xb8\x00\x20\x8e\xc0\xfc\xbd"
		    "\x00\x7c\x38\x4e\x24\x7d\x24\x8b\xc1\x99\xe8\x3c\x01"
		    "\x72\x1c\x83\xeb\x3a\x66\xa1\x1c\x7c\x26\x66\x3b\x07"
		    "\x26\x8a\x57\xfc\x75\x06\x80\xca\x02\x88\x56\x02\x80"
		    "\xc3\x10\x73\xeb\x33\xc9\x8a\x46\x10\x98\xf7\x66\x16"
		    "\x03\x46\x1c\x13\x56\x1e\x03\x46\x0e\x13\xd1\x8b\x76"
		    "\x11\x60\x89\x46\xfc\x89\x56\xfe\xb8\x20\x00\xf7\xe6"
		    "\x8b\x5e\x0b\x03\xc3\x48\xf7\xf3\x01\x46\xfc\x11\x4e"
		    "\xfe\x61\xbf\x00\x00\xe8\xe6\x00\x72\x39\x26\x38\x2d"
		    "\x74\x17\x60\xb1\x0b\xbe\xa1\x7d\xf3\xa6\x61\x74\x32"
		    "\x4e\x74\x09\x83\xc7\x20\x3b\xfb\x72\xe6\xeb\xdc\xa0"
		    "\xfb\x7d\xb4\x7d\x8b\xf0\xac\x98\x40\x74\x0c\x48\x74"
		    "\x13\xb4\x0e\xbb\x07\x00\xcd\x10\xeb\xef\xa0\xfd\x7d"
		    "\xeb\xe6\xa0\xfc\x7d\xeb\xe1\xcd\x16\xcd\x19\x26\x8b"
		    "\x55\x1a\x52\xb0\x01\xbb\x00\x00\xe8\x3b\x00\x72\xe8"
		    "\x5b\x8a\x56\x24\xbe\x0b\x7c\x8b\xfc\xc7\x46\xf0\x3d"
		    "\x7d\xc7\x46\xf4\x29\x7d\x8c\xd9\x89\x4e\xf2\x89\x4e"
		    "\xf6\xc6\x06\x96\x7d\xcb\xea\x03\x00\x00\x20\x0f\xb6"
		    "\xc8\x66\x8b\x46\xf8\x66\x03\x46\x1c\x66\x8b\xd0\x66"
		    "\xc1\xea\x10\xeb\x5e\x0f\xb6\xc8\x4a\x4a\x8a\x46\x0d"
		    "\x32\xe4\xf7\xe2\x03\x46\xfc\x13\x56\xfe\xeb\x4a\x52"
		    "\x50\x06\x53\x6a\x01\x6a\x10\x91\x8b\x46\x18\x96\x92"
		    "\x33\xd2\xf7\xf6\x91\xf7\xf6\x42\x87\xca\xf7\x76\x1a"
		    "\x8a\xf2\x8a\xe8\xc0\xcc\x02\x0a\xcc\xb8\x01\x02\x80"
		    "\x7e\x02\x0e\x75\x04\xb4\x42\x8b\xf4\x8a\x56\x24\xcd"
		    "\x13\x61\x61\x72\x0b\x40\x75\x01\x42\x03\x5e\x0b\x49"
		    "\x75\x06\xf8\xc3\x41\xbb\x00\x00\x60\x66\x6a\x00\xeb"
		    "\xb0\x4e\x54\x4c\x44\x52\x20\x20\x20\x20\x20\x20\x0d"
		    "\x0a\x52\x65\x6d\x6f\x76\x65\x20\x64\x69\x73\x6b\x73"
		    "\x20\x6f\x72\x20\x6f\x74\x68\x65\x72\x20\x6d\x65\x64"
		    "\x69\x61\x2e\xff\x0d\x0a\x44\x69\x73\x6b\x20\x65\x72"
		    "\x72\x6f\x72\xff\x0d\x0a\x50\x72\x65\x73\x73\x20\x61"
		    "\x6e\x79\x20\x6b\x65\x79\x20\x74\x6f\x20\x72\x65\x73"
		    "\x74\x61\x72\x74\x0d\x0a\x00\x00\x00\x00\x00\x00\x00"
		    "\xac\xcb\xd8\x55\xaa", 0x0000200);
	ret = usb_bulk_write(devh, 0x00000001, buf, 0x0000200, 1030);
	if (show_progress)
		printf(" usb bulk write: returned %d\n", ret);
	usleep(6000);
	usleep(1000);

	ret = usb_bulk_read(devh, 0x00000081, buf, 0x0000200, 1030);
	if (show_progress)
		printf(" usb bulk read: returned %d\n", ret);
	usleep(200);
	usleep(1000);

	memcpy(buf, "\x55\x53\x42\x43\xe8\xb5\xce\x82\x00\x10\x00\x00\x00"
		    "\x00\x0a\x2a\x08\x00\x00\x01\xe0\x00\x00\x08\x00\x00"
		    "\x00\x00\x00\x00\x00", 0x000001f);
	ret = usb_bulk_write(devh, 0x00000001, buf, 0x000001f, 1000);
	if (show_progress)
		printf(" usb bulk write: returned %d\n", ret);
	usleep(400);
	usleep(1000);

	memcpy(buf, "\x43\x4f\x4e\x4e\x4d\x47\x52\x20\x20\x20\x20\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xa4\x8c\x35\x3b\x00\x00\x00\x00\x00\x00\x41\x55\x54\x4f\x52\x55\x4e\x20\x49\x4e\x46\x20\x00\x52\xa3\x8c\x35\x3b\x5e\x3b\x00\x00\x10\x7d\x74\x3a\x02\x00\x3b\x00\x00\x00\x41\x4d\x00\x61\x00\x63\x00\x00\x00\xff\xff\x0f\x00\xf5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\xff\xff\xff\xff\x4d\x41\x43\x20\x20\x20\x20\x20\x20\x20\x20\x10\x00\x59\xa3\x8c\x35\x3b\x35\x3b\x00\x00\xa4\x8c\x35\x3b\x03\x00\x00\x00\x00\x00\x51\x53\x47\x20\x20\x20\x20\x20\x50\x44\x46\x20\x10\xc6\xa5\x8c\x35\x3b\x35\x3b\x00\x00\xc3\x93\x97\x3a\x12\x2c\xc7\x8c\x0a\x00\x53\x45\x54\x55\x50\x20\x20\x20\x20\x20\x20\x10\x08\x3c\xa6\x8c\x35\x3b\x35\x3b\x00\x00\xa7\x8c\x35\x3b\x59\x31\x00\x00\x00\x00\x53\x45\x54\x55\x50\x20\x20\x20\x45\x58\x45\x20\x18\x72\xae\x8c\x35\x3b\x35\x3b\x00\x00\xf7\x94\xf0\x3a\x55\xb4\x40\x27\x0a\x00\x42\x58\x00\x2e\x00\x63\x00\x66\x00\x67\x00\x0f\x00\x16\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\xff\xff\xff\xff\x01\x56\x00\x49\x00\x44\x00\x30\x00\x37\x00\x0f\x00\x16\x44\x00\x31\x00\x50\x00\x49\x00\x44\x00\x33\x00\x00\x00\x45\x00\x30\x00\x56\x49\x44\x30\x37\x44\x7e\x31\x43\x46\x47\x20\x00\xaf\xae\x8c\x35\x3b\x35\x3b\x00\x00\x4c\x90\xf4\x3a\x69\xb9\x6b\x00\x00\x00\x57\x43\x44\x4d\x41\x20\x20\x20\x43\x46\x47\x20\x18\x8c\x38\x8d\x5e\x3b\x5e\x3b\x00\x00\x39\x8d\x5e\x3b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 0x0001000);
	ret = usb_bulk_write(devh, 0x00000001, buf, 0x0001000, 1245);
	if (show_progress)
		printf(" usb bulk write: returned %d\n", ret);
	sleep(2);
}

int switchHuaweiMode () {

	if (show_progress) printf("Sending Huawei control message ...\n");
	ret = usb_control_msg(devh, USB_TYPE_STANDARD + USB_RECIP_DEVICE, USB_REQ_SET_FEATURE, 00000001, 0, buf, 0, 1000);
	if (ret != 0) {
		if (show_progress) fprintf(stderr, "Error: sending Huawei control message failed (error %d). Aborting.\n\n", ret);
		exit(1);
	} else
		if (show_progress) printf(" OK, Huawei control message sent\n");
}

int switchQisdaMode () {

	if (show_progress) printf("Sending Qisda control message ...\n");
	memset(buf, 0, sizeof(buf));
	buf[9] = 0x01;
	ret = usb_control_msg(devh, USB_TYPE_VENDOR + USB_RECIP_DEVICE, 0x04, 0x00, 0x00, buf, 16, 1000);
	if (ret != 0) {
		if (show_progress) fprintf(stderr, "Error: sending Qisda control message failed (error %d). Aborting.\n\n", ret);
		exit(1);
	} else
		if (show_progress) printf(" OK, Qisda control message sent\n");
}

int switchSierraMode () {

	if (show_progress) printf("Trying to send Sierra control message\n");
	ret = usb_control_msg(devh, 0x40, 0x0b, 00000001, 0, buf, 0, 1000);
	if (ret != 0) {
		if (show_progress) fprintf(stderr, "Error: sending Sierra control message failed (error %d). Aborting.\n\n", ret);
	    exit(1);
	} else
		if (show_progress) printf(" OK, Sierra control message sent\n");
}

int switchSonyMode () {

	if (show_progress) printf("Trying to send Sony control message\n");
	buf[0] = 0x5a;
	buf[1] = 0x11;
	buf[2] = 0x02;
	ret = usb_control_msg(devh, 0xc0, 0x11, 00000002, 0, buf, 3, 1000);
	if (ret < 0) {
		if (show_progress) fprintf(stderr, "Error: sending Sony control message failed (error %d). Aborting.\n\n", ret);
		exit(1);
	} else
		if (show_progress) printf(" OK, Sony control message sent\n");
}


// Detach driver either as the main action or as preparation for other modes
int detachDriver() {

#ifndef LIBUSB_HAS_GET_DRIVER_NP
	printf(" Cant't do driver detection and detaching on this platform.\n");
	return 2;
#endif

	if (show_progress) printf("Looking for active driver ...\n");
	ret = usb_get_driver_np(devh, Interface, buf, BUF_SIZE);
	if (ret != 0) {
		if (show_progress) printf(" No driver found. Either detached before or never attached\n");
		return 1;
	}
	if (show_progress) printf(" OK, driver found (\"%s\")\n", buf);
	if (DetachStorageOnly && strcmp(buf,"usb-storage")) {
		if (show_progress) printf(" Driver is not usb-storage, leaving it alone\n");
		return 1;
	}

#ifndef LIBUSB_HAS_DETACH_KERNEL_DRIVER_NP
	if (show_progress) printf(" Can't do driver detaching on this platform\n");
	return 2;
#endif

	ret = usb_detach_kernel_driver_np(devh, Interface);
	if (ret == 0) {
		if (show_progress) printf(" OK, driver \"%s\" detached\n", buf);
	} else {
		if (show_progress) printf(" Driver \"%s\" detach failed with error %d. Trying to continue\n", buf, ret);
	}
	return 1;
}

int checkSuccess(struct usb_device *dev) {
	
	if (show_progress) printf("Checking for mode switch after %d seconds settling time ...\n", CheckSuccess);
	sleep(CheckSuccess);

	// Test if default device still can be accessed; positive result does
    // not necessarily mean failure
	devh = usb_open(dev);
	ret = usb_claim_interface(devh, Interface);
	if (ret < 0) {
		if (show_progress) printf(" Original device can't be accessed anymore. Good.\n");
	} else {
		if (show_progress) printf(" Original device can still be accessed. Hmm.\n");
		usb_release_interface(devh, Interface);
	}

	// Recount target devices (compare with previous count) if target data is given
	int newTargetCount;
	if ((TargetVendor && TargetProduct) || TargetClass) {
		usb_find_devices();
		search_devices(&newTargetCount, TargetVendor, TargetProduct, TargetClass);
		if (newTargetCount > targetDeviceCount) {
			if (show_progress) printf(" Found a new device in target mode or class\n\nMode switch was successful. Bye!\n\n");
			return 1;
		} else {
			if (show_progress) printf(" No new devices in target mode or class found\n\nMode switch seems to have failed. Bye!\n\n");
			return 0;
		}
	} else {
		if (show_progress) printf("For a better success check provide target IDs or class. Bye!\n\n");
		if (ret < 0)
			return 1;
		else
			return 0;
	}
}

int write_bulk(int endpoint, char *message, int length) {
	int ret;
	ret = usb_bulk_write(devh, endpoint, message, length, 1000);
	if (ret >= 0 ) {
		if (show_progress) printf(" OK, message successfully sent\n");
		return 1;
	} else {
		if (show_progress) printf(" Sending the message returned error %d. Trying to continue\n", ret);
		return 0;
	}
}

int read_bulk(int endpoint, char *buffer, int length) {
	int ret;
	ret = usb_bulk_read(devh, endpoint, buffer, length, 1000);
	if (ret >= 0 ) {
		if (show_progress) printf(" OK, response successfully read (%d bytes).\n", ret);
	} else {
		if (show_progress) printf(" Reading the response returned error %d, trying to continue\n", ret);
	}
	return ret;
}

void release_usb_device(int dummy) {
	int ret;
	if (show_progress) printf("Program cancelled by system. Bye!\n\n");
	usb_release_interface(devh, Interface);
	usb_close(devh);
	exit(0);
}


// iterates over busses and devices, counting the found and returning the last one of them

struct usb_device* search_devices( int *numFound, int vendor, int product, int targetClass) {
	struct usb_bus *bus;
	int devClass;
	struct usb_device* right_dev = NULL;
	
	if ( targetClass && !(vendor && product) ) {
		vendor = DefaultVendor;
		product = DefaultProduct;
	}
	*numFound = 0;
	for (bus = usb_get_busses(); bus; bus = bus->next) {
		struct usb_device *dev;
		for (dev = bus->devices; dev; dev = dev->next) {
			if (dev->descriptor.idVendor == vendor && dev->descriptor.idProduct == product) {
				(*numFound)++;
				devClass = dev->descriptor.bDeviceClass;
				if (devClass == 0)
					devClass = dev->config[0].interface[0].altsetting[0].bInterfaceClass;
				if (devClass != targetClass || targetClass == 0)
					right_dev = dev;
			}
		}
	}
	return right_dev;
}


// the parameter parsing stuff

char* ReadParseParam(const char* FileName, char *VariableName) {
	static char Str[LINE_DIM];
	char *VarName, *Comment=NULL, *Equal=NULL;
	char *FirstQuote, *LastQuote, *P1, *P2;
	int Line=0, Len=0, Pos=0;
	FILE *file=fopen(FileName, "r");
	
	if (file==NULL) {
		if (show_progress) fprintf(stderr, "Error: Could not find file %s\n\n", FileName);
		exit(1);
	}

	while (fgets(Str, LINE_DIM-1, file) != NULL) {
		Line++;
		Len=strlen(Str);
		if (Len==0) goto Next;
		if (Str[Len-1]=='\n' or Str[Len-1]=='\r') Str[--Len]='\0';
		Equal = strchr (Str, '=');			// search for equal sign
		Pos = strcspn (Str, ";#!");			// search for comment
		Comment = (Pos==Len) ? NULL : Str+Pos;
		if (Equal==NULL or ( Comment!=NULL and Comment<=Equal)) goto Next;	// Only comment
		*Equal++ = '\0';
		if (Comment!=NULL) *Comment='\0';

		// String
		FirstQuote=strchr (Equal, '"');		// search for double quote char
		LastQuote=strrchr (Equal, '"');
		if (FirstQuote!=NULL) {
			if (LastQuote==NULL) {
				if (show_progress) fprintf(stderr, "Error reading parameter file %s line %d - Missing end quote.\n", FileName, Line);
				goto Next;
			}
			*FirstQuote=*LastQuote='\0';
			Equal=FirstQuote+1;
		}
		
		// removes leading/trailing spaces
		Pos=strspn (Str, " \t");
		if (Pos==strlen(Str)) {
			if (show_progress) fprintf(stderr, "Error reading parameter file %s line %d - Missing variable name.\n", FileName, Line);
			goto Next;		// No function name
		}
		while ((P1=strrchr(Str, ' '))!=NULL or (P2=strrchr(Str, '\t'))!=NULL)
			if (P1!=NULL) *P1='\0';
			else if (P2!=NULL) *P2='\0';
		VarName=Str+Pos;
		//while (strspn(VarName, " \t")==strlen(VarName)) VarName++;

		Pos=strspn (Equal, " \t");
		if (Pos==strlen(Equal)) {
			if (show_progress) fprintf(stderr, "Error reading parameter file %s line %d - Missing value.\n", FileName, Line);
			goto Next;		// No function name
		}
		Equal+=Pos;

//		printf("%s=%s\n", VarName, Equal);
		if (strcmp(VarName, VariableName)==0) {		// Found it
			fclose(file);
			return Equal;
		}
		Next:;
	}
	
	// not found
//	fprintf(stderr, "Error reading parameter file %s - Variable %s not found.", 
//				FileName, VariableName);
	fclose(file);
	return NULL;
}

int hex2num(char c)
{
	if (c >= '0' && c <= '9')
	return c - '0';
	if (c >= 'a' && c <= 'f')
	return c - 'a' + 10;
	if (c >= 'A' && c <= 'F')
	return c - 'A' + 10;
	return -1;
}


int hex2byte(const char *hex)
{
	int a, b;
	a = hex2num(*hex++);
	if (a < 0)
	return -1;
	b = hex2num(*hex++);
	if (b < 0)
	return -1;
	return (a << 4) | b;
}

int hexstr2bin(const char *hex, char *buf, int len)
{
	int i;
	int a;
	const char *ipos = hex;
	char *opos = buf;
//    printf("Debug: hexstr2bin bytestring is ");

	for (i = 0; i < len; i++) {
	a = hex2byte(ipos);
//        printf("%02X", a);
	if (a < 0)
		return -1;
	*opos++ = a;
	ipos += 2;
	}
//    printf(" \n");
	return 0;
}

