
#include <stdio.h>
#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/agent/net-snmp-agent-includes.h>
#include "senao_wlan.h"
#include "apinfo_client.h"

#define SENAO_STR_LEN 300
#define SEANO_STR_DEFAULT "Senao Networks SNMP WLAN MIB"

struct variable2 senao_wlan_variables[] = {
    {SENAO_STRING,          ASN_OCTET_STR, RWRITE, var_senao_wlan, 1, {1}},
    {SENAO_INTEGER,         ASN_INTEGER,   RWRITE, var_senao_wlan, 2, {2, 1}},
//    {SENAO_OBJECTID,     ASN_OBJECT_ID, RWRITE, var_senao_wlan, 2, {2, 2}},
    {SENAO_TIMETICKS,       ASN_TIMETICKS, RWRITE, var_senao_wlan, 1, {3}},
    {SENAO_IPADDRESS,       ASN_IPADDRESS, RWRITE, var_senao_wlan, 1, {4}},
    {SENAO_COUNTER,         ASN_COUNTER,   RWRITE, var_senao_wlan, 1, {5}},
//    {SENAO_GAUGE,           ASN_GAUGE,     RWRITE, var_senao_wlan, 1, {6, 1}},
    {APINFO_CHANNEL_NUMBER, ASN_INTEGER,   RWRITE, var_senao_wlan, 2, {2, 2}},
    {APINFO_SSID,           ASN_OCTET_STR, RWRITE, var_senao_wlan, 2, {2, 3}},

//    {SENAO_TRIGGERTRAP,  ASN_INTEGER,   RONLY,  var_senao_wlan, 1, {7}},
//    {SENAO_TRIGGERTRAP2, ASN_INTEGER,   RWRITE, var_senao_wlan, 1, {8}}

};


oid  senao_wlan_variables_oid[] = { 1, 3, 6, 1, 4, 1, 14125, 6 };

static long senao_long;
static char senao_str[SENAO_STR_LEN];
static int  senao_sd;
static char senao_data[APINFO_MAX_LEN];

void
init_senao_wlan(void)
{
    REGISTER_MIB("senao_wlan", senao_wlan_variables, variable2,
                 senao_wlan_variables_oid);
    strncpy(senao_str, SEANO_STR_DEFAULT, SENAO_STR_LEN);
    senao_sd = apinfo_open();
}

u_char *
var_senao_wlan(struct variable *vp,
           	  oid * name,
	          size_t * length,
        	  int exact, size_t * var_len, WriteMethod ** write_method)
{


    DEBUGMSGTL(("senao_wlan", "var_senao_wlan entered\n"));
    if (header_generic(vp, name, length, exact, var_len, write_method) ==
        MATCH_FAILED)
        return NULL;

    switch (vp->magic) {
    case SENAO_STRING:
        *var_len = strlen(senao_str);
	*write_method = write_senao_wlan_str;
        return (u_char *) senao_str;

    case SENAO_INTEGER:
	senao_long = 1;
        *write_method = write_senao_wlan_int;
        return (u_char *) & senao_long;
/*
    case SENAO_OBJECTID:

	return;
*/
    case SENAO_TIMETICKS:
        senao_long = 363136200;   /* 42 days, 42 minutes and 42.0 seconds */
        return (u_char *) & senao_long;

    case SENAO_IPADDRESS:
        senao_long = ntohl(INADDR_LOOPBACK);
        return (u_char *) & senao_long;

    case SENAO_COUNTER:
        senao_long = 42;
        return (u_char *) & senao_long;

    case SENAO_GAUGE:
        senao_long = 42;          /* Do we detect a theme running through these answers? */
        return (u_char *) & senao_long;
/*    
    case SENAO_TRIGGERTRAP:
        long_ret = 0;
        *write_method = write_senao_wlan_trap;
        return (u_char *) & long_ret;

    case SENAO_TRIGGERTRAP2:
        long_ret = 0;
        *write_method = write_senao_wlan_trap2;
        return (u_char *) & long_ret;
*/

    case APINFO_CHANNEL_NUMBER:
        apinfo_read(senao_sd, APINFO_CHANNEL_NUMBER, senao_data, sizeof(int));
	*write_method = write_senao_wlan_channel_number;
        return (u_char *)senao_data;


    case APINFO_SSID:
        apinfo_read(senao_sd, APINFO_SSID, senao_data, sizeof(senao_data) );
	*var_len = strlen(senao_data);
	*write_method = write_senao_wlan_ssid;
        return (u_char *)senao_data;

    default:
        DEBUGMSGTL(("snmpd", "unknown sub-id %d in senao/var_senao_wlan\n",
                    vp->magic));


	}
    return NULL;
}

int
write_senao_wlan_channel_number( int action,
		                 u_char * var_val,
                		 u_char var_val_type,
		                 size_t var_val_len,
                		 u_char * statP, oid * name, size_t name_len)
{
	apinfo_write(senao_sd, APINFO_CHANNEL_NUMBER, var_val, var_val_len);
}

int
write_senao_wlan_ssid( int action,
                       u_char * var_val,
                       u_char var_val_type,
                       size_t var_val_len,
                       u_char * statP, oid * name, size_t name_len)
{
        apinfo_write(senao_sd, APINFO_SSID, var_val, var_val_len);
}

int
write_senao_wlan_int(int action,
                 u_char * var_val,
                 u_char var_val_type,
                 size_t var_val_len,
                 u_char * statP, oid * name, size_t name_len)
{
#define MAX_SENAO_INT 100
    static long     intval;
    static long     old_intval;

    switch (action) {
    case RESERVE1:
        /*
         *  Check that the value being set is acceptable
         */
        if (var_val_type != ASN_INTEGER) {
            DEBUGMSGTL(("senao_wlan", "%x not integer type", var_val_type));
            return SNMP_ERR_WRONGTYPE;
        }
        if (var_val_len > sizeof(long)) {
            DEBUGMSGTL(("senao_wlan", "wrong length %x", var_val_len));
            return SNMP_ERR_WRONGLENGTH;
        }

        intval = *((long *) var_val);
        if (intval > MAX_SENAO_INT) {
            DEBUGMSGTL(("senao_wlan", "wrong value %x", intval));
            return SNMP_ERR_WRONGVALUE;
        }
	printf("write_senao_wlan_int: set integer = %d\n", intval);
        break;

    case RESERVE2:
        /*
         *  This is conventially where any necesary
         *   resources are allocated (e.g. calls to malloc)
         *  Here, we are using static variables
         *   so don't need to worry about this.
         */
        break;

    case FREE:
        /*
         *  This is where any of the above resources
         *   are freed again (because one of the other
         *   values being SET failed for some reason).
         *  Again, since we are using static variables
         *   we don't need to worry about this either.
         */
        break;

    case ACTION:
        /*
         *  Set the variable as requested.
         *   Note that this may need to be reversed,
         *   so save any information needed to do this.
         */
        old_intval = senao_long;
        senao_long = intval;
        break;

    case UNDO:
        /*
         *  Something failed, so re-set the
         *   variable to its previous value
         *  (and free any allocated resources).
         */
        senao_long = old_intval;
        break;

    case COMMIT:
        /*
         *  Everything worked, so we can discard any
         *   saved information, and make the change
         *   permanent (e.g. write to the config file).
         *  We also free any allocated resources.
         *
         *  In this case, there's nothing to do.
         */
        break;

    }
    return SNMP_ERR_NOERROR;
}

static char str_buf[SENAO_STR_LEN];
static char old_buf[SENAO_STR_LEN];

int
write_senao_wlan_str(int action,
                 u_char * var_val,
                 u_char var_val_type,
                 size_t var_val_len,
                 u_char * statP, oid * name, size_t name_len)
{

    switch (action) {
    case RESERVE1:
        /*
         *  Check that the value being set is acceptable
         */
        if (var_val_type != ASN_OCTET_STR) {
            DEBUGMSGTL(("senao_wlan", "%x not string type", var_val_type));
            return SNMP_ERR_WRONGTYPE;
        }
        if (var_val_len > SENAO_STR_LEN) {
            DEBUGMSGTL(("senao_wlan", "wrong length %x", var_val_len));
            return SNMP_ERR_WRONGLENGTH;
        }

	memcpy(str_buf, var_val, SENAO_STR_LEN);
/*
        if (intval > MAX_SENAO_INT) {
            DEBUGMSGTL(("senao_wlan", "wrong value %x", intval));
            return SNMP_ERR_WRONGVALUE;
        }
*/
        printf("write_senao_wlan_int: set string = %s\n", str_buf);
        break;

    case RESERVE2:
        /*
         *  This is conventially where any necesary
         *   resources are allocated (e.g. calls to malloc)
         *  Here, we are using static variables
         *   so don't need to worry about this.
         */
        break;

    case FREE:
        /*
         *  This is where any of the above resources
         *   are freed again (because one of the other
         *   values being SET failed for some reason).
         *  Again, since we are using static variables
         *   we don't need to worry about this either.
         */
        break;

    case ACTION:
        /*
         *  Set the variable as requested.
         *   Note that this may need to be reversed,
         *   so save any information needed to do this.
         */
        memcpy(old_buf, senao_str, SENAO_STR_LEN);
	memcpy(senao_str, str_buf, SENAO_STR_LEN);
        break;

    case UNDO:
        /*
         *  Something failed, so re-set the
         *   variable to its previous value
         *  (and free any allocated resources).
         */
	memcpy(senao_str, old_buf, SENAO_STR_LEN);
        break;

    case COMMIT:
        /*
         *  Everything worked, so we can discard any
         *   saved information, and make the change
         *   permanent (e.g. write to the config file).
         *  We also free any allocated resources.
         *
         *  In this case, there's nothing to do.
         */
        break;

    }
    return SNMP_ERR_NOERROR;
}


