#ifndef _XAT_H_
#define _XAT_H_

#include "rtmp.h"
#include "rtmp_type.h"
#include "rt_linux.h"

#if 1 //def __LINUX__
#include <linux/if_ether.h>
#include <linux/if_arp.h>
#include <linux/ip.h>
#else


//Currently support upper layer protocols
#define ETH_P_IP        0x0800          /* Internet Protocol packet     */
#define ETH_P_ARP       0x0806          /* Address Resolution packet    */
#define ETH_P_PPP_DISC  0x8863          /* PPPoE discovery messages     */
#define ETH_P_PPP_SES   0x8864          /* PPPoE session messages       */


/* ARP protocol HARDWARE identifiers. */
#define ARPHRD_ETHER    1               /* Ethernet 10Mbps              */

/* ARP protocol opcodes. */
#define ARPOP_REQUEST   1               /* ARP request                  */
#define ARPOP_REPLY     2               /* ARP reply                    */

struct arphdr{
    unsigned short  ar_hrd;         /* format of hardware address   */
    unsigned short  ar_pro;         /* format of protocol address   */
    unsigned char   ar_hln;         /* length of hardware address   */
    unsigned char   ar_pln;         /* length of protocol address   */
    unsigned short  ar_op;          /* ARP opcode (command)         */
};

struct iphdr{
#ifndef BIG_ENDIAN
    UCHAR   ihl:4,
            version:4;
#else
    UCHAR   version:4,
            ihl:4;
#endif
    UCHAR   tos;
    UINT16  tot_len;
    UINT16  id;
    UINT16  frag_off;
    UCHAR   ttl;
    UCHAR   protocol;
    UINT16  check;
    UINT16  saddr;
    UINT32  daddr;
}; 
#endif  //endif of __LINUX__

#ifndef XAT_SUPPORT
#error "You should define XAT_SUPPORT if you want to compile XAT related functions!"
#endif


/* XAT relate definition */
#define XAT_MAX_HASH_ENTRY_SUPPORT      64
#define XAT_TB_ENTRY_AGEOUT_TIME            (5 * 60 * OS_HZ)    // 30000, 5min. XAT convert table entry age-out time interval. now set it as 5min.


/* 802.3 Ethernet related definition */
#define XAT_ETHER_HDR_LEN       14      // dstMac(6) + srcMac(6) + protoType(2)
#define XAT_VLAN_ETH_HDR_LEN    (XAT_ETHER_HDR_LEN + 4)     // 4 for h_vlan_TCI and h_vlan_encapsulated_proto

#define XAT_MAC_ADDR_HASH(Addr)       (Addr[0] ^ Addr[1] ^ Addr[2] ^ Addr[3] ^ Addr[4] ^ Addr[5])
#define XAT_MAC_ADDR_HASH_INDEX(Addr) (XAT_MAC_ADDR_HASH(Addr) % XAT_MAX_HASH_ENTRY_SUPPORT)

#define isMcastEtherAddr(addr)  (addr[0] & 0x1)
#define isBcastEtherAddr(addr)  ((addr[0] & addr[1] & addr[2] & addr[3] & addr[4] & addr[5])== 0xff)
#define isZeroEtherAddr(addr)   (!(addr[0] | addr[1] | addr[2] | addr[3] | addr[4] | addr[5]))

#define IS_GROUP_MAC(Addr)      ((Addr[0]) & 0x01)
#define IS_UCAST_MAC(addr)      (!(isMcastEtherAddr(addr) || isZeroEtherAddr(addr))) // isUcastMac = !(00:00:00:00:00:00 || mcastMac);
#define IS_EQUAL_MAC(a, b)      (((a[0] ^ b[0]) | (a[1] ^ b[1]) | (a[2] ^ b[2]) | (a[3] ^ b[3]) | (a[4] ^ b[4]) | (a[5] ^ b[5])) == 0)

#define IS_VLAN_PACKET(pkt)     ((((pkt)[12] << 8) | (pkt)[13]) == 0x8100)

/* IPv4 related definition */
#define IPMAC_TB_HASH_ENTRY_NUM         (XAT_MAX_HASH_ENTRY_SUPPORT+1)  // One entry for broadcast address
#define IPMAC_TB_HASH_INDEX_OF_BCAST    XAT_MAX_HASH_ENTRY_SUPPORT      // cause hash index start from 0.

#define XAT_IP_ADDR_HASH(Addr)      (((Addr>>24)&0xff)^((Addr>>16) & 0xff) ^((Addr>>8) & 0xff) ^ (Addr & 0xff))
#define XAT_IP_ADDR_HASH_INDEX(Addr)    (XAT_IP_ADDR_HASH(Addr) % XAT_MAX_HASH_ENTRY_SUPPORT)

#define IS_GOOD_IP(IP)  (IP!= 0)


/* IPv6 related definition */
#define IPV6MAC_TB_HASH_ENTRY_NUM       (XAT_MAX_HASH_ENTRY_SUPPORT+1)  // One entry for broadcast address
#define IPV6MAC_TB_HASH_INDEX_OF_BCAST  XAT_MAX_HASH_ENTRY_SUPPORT      // cause hash index start from 0.

//We just use byte 10,13,14,15 to calculate the IPv6 hash, because the byte 11,12 usually are 0xff, 0xfe for link-local address.
#define XAT_IPV6_ADDR_HASH(Addr)    ((Addr[10]&0xff) ^ (Addr[13] & 0xff) ^(Addr[14] & 0xff) ^ (Addr[15] & 0xff))
#define XAT_IPV6_ADDR_HASH_INDEX(Addr)  (XAT_IPV6_ADDR_HASH(Addr) % XAT_MAX_HASH_ENTRY_SUPPORT)

#define IS_UNSPECIFIED_IPV6_ADDR(_addr) \
        (!((_addr).ipv6_addr32[0] | (_addr).ipv6_addr32[1] | (_addr).ipv6_addr32[2] | (_addr).ipv6_addr32[3]))
#define IS_LOOPBACK_IPV6_ADDR(_addr) \
        (NdisEqualMemory((UCHAR *)(&((_addr).ipv6_addr[0])), &IPV6_LOOPBACKADDR[0], IPV6_ADDR_LEN))
#define IS_MULTICAST_IPV6_ADDR(_addr) \
        (((_addr).ipv6_addr[0] & 0xff) == 0xff)


// The XAT_TABLE used for MacAddress <-> UpperLayer Address Translation.
typedef struct _XAT_TABLE_
{
    VOID    *IPMacTable;        // IPv4 Address, Used for IP, ARP protocol
    VOID    *IPv6MacTable;  // IPv6 Address, Used for IPv6 related protocols
    VOID    *SesMacTable;       // PPPoE Session
    VOID    *UidMacTable;       // PPPoE Discovery
}XAT_TABLE, *PXAT_TABLE;


typedef enum _XAT_ENGINE_STATUS_
{
    XAT_ENGINE_STAT_UNKNOWN = 0,
    XAT_ENGINE_STAT_INITED = 1,
    XAT_ENGINE_STAT_EXITED = 2,
}XAT_ENGINE_STATUS;


typedef struct _XAT_STRUCT_
{
    XAT_ENGINE_STATUS   status;
    NDIS_SPIN_LOCK      XATDBLock;
    XAT_TABLE           MatTableSet;
#ifdef KMALLOC_BATCH
    UCHAR               *pXATNodeEntryPoll;
#endif
    UINT32              nodeCount;      // the number of nodes which connect to Internet via us.
    VOID                *pPriv;
}XAT_STRUCT;


typedef struct _XATProtoOps
{
    NDIS_STATUS     (*init)(XAT_STRUCT *pMatCfg);
    PUCHAR          (*tx)(XAT_STRUCT *pMatCfg, PNDIS_PACKET pSkb, PUCHAR pLayerHdr, PUCHAR pMacAddr);
    PUCHAR          (*rx)(XAT_STRUCT *pMatCfg, PNDIS_PACKET pSkb, PUCHAR pLayerHdr, PUCHAR pMacAddr);
    NDIS_STATUS     (*exit)(XAT_STRUCT *pMatCfg);
}XATProtoOps, *PXATProtoOps;


typedef struct _XATProtoTable
{
    USHORT          protocol;
    XATProtoOps     *pHandle;
}XATProtoTable, *PXATProtoTable;


VOID dumpPkt(PUCHAR pHeader, int len);

//#define KMALLOC_BATCH

PUCHAR XATDBEntryAlloc(
    IN XAT_STRUCT   *pMatStruct, 
    IN UINT32       size);

NDIS_STATUS XATDBEntryFree(
    IN XAT_STRUCT   *pMatStruct, 
    IN PUCHAR       NodeEntry);




typedef struct _RTMP_ADAPTER
{   
    UCHAR     CurrentAddress[MAC_ADDR_LEN];      // User changed MAC address
    XAT_STRUCT MatCfg;
} RTMP_ADAPTER, *PRTMP_ADAPTER;
  

NDIS_STATUS XATEngineInit(IN RTMP_ADAPTER *pAd);
NDIS_STATUS XATEngineExit(IN RTMP_ADAPTER *pAd);
#if 0
PUCHAR XAT(IN PRTMP_ADAPTER    pAd,
                        IN PNDIS_PACKET     pPkt,
                        IN UINT             infIdx);
PUCHAR XATEngineTxHandle(IN PRTMP_ADAPTER    pAd,
                         IN PNDIS_PACKET     pPkt,
                         IN UINT             ifIdx);
#endif
PUCHAR  getBROADCAST_ADDR(void);
//PUCHAR  getStaMac_ADDR(void);
RTMP_ADAPTER *getAd(void);
VOID dumpPkt(PUCHAR pHeader, int len);
#endif /* _XAT_H_ */

