/*
 * tc_red.c		RED maintanance routines.
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Authors:	Alexey Kuznetsov, <kuznet@ms2.inr.ac.ru>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <fcntl.h>
#include <math.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>

#include "tc_core.h"
#include "tc_red.h"

/*
   Plog = log(prob/(qmax - qmin))
 */
int tc_red_eval_P(unsigned qmin, unsigned qmax, double prob)
{
	int i = qmax - qmin;

	if (i <= 0)
		return -1;

	prob /= i;

	for (i=0; i<32; i++) {
		if (prob > 1.0)
			break;
		prob *= 2;
	}
	if (i>=32)
		return -1;
	return i;
}

/*
   burst + 1 - qmin/avpkt < (1-(1-W)^burst)/W
 */

int tc_red_eval_ewma(unsigned qmin, unsigned burst, unsigned avpkt)
{
	int wlog = 1;
	double W = 0.5;
	double a = (double)burst + 1 - (double)qmin/avpkt;

	if (a < 1.0)
		return -1;
	for (wlog=1; wlog<32; wlog++, W /= 2) {
#ifdef WE_NOLIBM
		{
			int cnt;
			double res = 1;

			for(cnt = 0; cnt < burst; ++cnt)
			{
				res *= (1-W);
			}
			if (a <= (1 - res)/W)
				return wlog;
		}
#else
		if (a <= (1 - pow(1-W, burst))/W)
			return wlog;
#endif
	}
	return -1;
}

/*
   Stab[t>>Scell_log] = -log(1-W) * t/xmit_time
 */
#ifdef WE_NOLIBM
static double frexp(double value, int *eptr)
{
	union {
		double v;
		struct {
			u_int u_mant2 : 32;
			u_int u_mant1 : 20;
			u_int   u_exp : 11;
			u_int  u_sign :  1;
		} s;
	} u;

	if (value) {
		u.v = value;
		*eptr = u.s.u_exp - 1022;
		u.s.u_exp = 1022;
		return(u.v);
	} else {
		*eptr = 0;
		return((double)0);
	}
}

static double log(double arg)
{
	double	log2	= 0.693147180559945309e0;
	double	ln10	= 2.302585092994045684;
	double	sqrto2	= 0.707106781186547524e0;
	double	p0	= -.240139179559210510e2;
	double	p1	= 0.309572928215376501e2;
	double	p2	= -.963769093368686593e1;
	double	p3	= 0.421087371217979714e0;
	double	q0	= -.120069589779605255e2;
	double	q1	= 0.194809660700889731e2;
	double	q2	= -.891110902798312337e1;
	double x,z, zsq, temp;
	int exp;

	if(arg <= 0.) {
		//errno = EDOM;
		return(-HUGE);
	}
	x = frexp(arg,&exp);
	while(x<0.5) {
		x = x*2;
		exp = exp-1;
	}
	if(x<sqrto2) {
		x = 2*x;
		exp = exp-1;
	}

	z = (x-1)/(x+1);
	zsq = z*z;

	temp = ((p3*zsq + p2)*zsq + p1)*zsq + p0;
	temp = temp/(((1.0*zsq + q2)*zsq + q1)*zsq + q0);
	temp = temp*z + exp*log2;
	return(temp);
}
#endif

int tc_red_eval_idle_damping(int Wlog, unsigned avpkt, unsigned bps, __u8 *sbuf)
{
	double xmit_time = tc_core_usec2tick(1000000*(double)avpkt/bps);
	double lW = -log(1.0 - 1.0/(1<<Wlog))/xmit_time;
	double maxtime = 31/lW;
	int clog;
	int i;
	double tmp;

	tmp = maxtime;
	for (clog=0; clog<32; clog++) {
		if (maxtime/(1<<clog) < 512)
			break;
	}
	if (clog >= 32)
		return -1;

	sbuf[0] = 0;
	for (i=1; i<255; i++) {
		sbuf[i] = (i<<clog)*lW;
		if (sbuf[i] > 31)
			sbuf[i] = 31;
	}
	sbuf[255] = 31;
	return clog;
}
