/*
 * tc_core.c		TC core library.
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Authors:	Alexey Kuznetsov, <kuznet@ms2.inr.ac.ru>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <fcntl.h>
#include <math.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>

#include "tc_core.h"

#ifdef WE_NOLIBM
static double frexp(double value, int *eptr)
{
	union {
		double v;
		struct {
			u_int u_mant2 : 32;
			u_int u_mant1 : 20;
			u_int   u_exp : 11;
			u_int  u_sign :  1;
		} s;
	} u;

	if (value) {
		u.v = value;
		*eptr = u.s.u_exp - 1022;
		u.s.u_exp = 1022;
		return(u.v);
	} else {
		*eptr = 0;
		return((double)0);
	}
}

static double log(double arg)
{
	double	log2	= 0.693147180559945309e0;
	double	ln10	= 2.302585092994045684;
	double	sqrto2	= 0.707106781186547524e0;
	double	p0	= -.240139179559210510e2;
	double	p1	= 0.309572928215376501e2;
	double	p2	= -.963769093368686593e1;
	double	p3	= 0.421087371217979714e0;
	double	q0	= -.120069589779605255e2;
	double	q1	= 0.194809660700889731e2;
	double	q2	= -.891110902798312337e1;
	double x,z, zsq, temp;
	int exp;

	if(arg <= 0.) {
		//errno = EDOM;
		return(-HUGE);
	}
	x = frexp(arg,&exp);
	while(x<0.5) {
		x = x*2;
		exp = exp-1;
	}
	if(x<sqrto2) {
		x = 2*x;
		exp = exp-1;
	}

	z = (x-1)/(x+1);
	zsq = z*z;

	temp = ((p3*zsq + p2)*zsq + p1)*zsq + p0;
	temp = temp/(((1.0*zsq + q2)*zsq + q1)*zsq + q0);
	temp = temp*z + exp*log2;
	return(temp);
}
#endif

int tc_setup_estimator(unsigned A, unsigned time_const, struct tc_estimator *est)
{
	for (est->interval=0; est->interval<=5; est->interval++) {
		if (A <= (1<<est->interval)*(1000000/4))
			break;
	}
	if (est->interval > 5)
		return -1;
	est->interval -= 2;
	for (est->ewma_log=1; est->ewma_log<32; est->ewma_log++) {
		double w = 1.0 - 1.0/(1<<est->ewma_log);
		if (A/(-log(w)) > time_const)
			break;
	}
	est->ewma_log--;
	if (est->ewma_log==0 || est->ewma_log >= 31)
		return -1;
	return 0;
}
