#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>
//#include <inet_sup.h>
//#include <knlintf.h>
#include <dns_proxy.h>
#include <net/if.h> /* contains the struct ifreq definition */
#include <sys/ioctl.h>
#include <linux/if_ether.h> /* out of habit, I use a SOCK_PACKET socket*/
#include <signal.h>
#include "apcfg.h"
//#include "config.h"
#include "tokens.h"

static void checkWanDnsChanged();

/**********************************
 Definitions
***********************************/
#define	DNS_PROXY_ENTRY_MAX         32

#define DNS_SERVICE                 53
#define DNS_PROXY_TIMEOUT           (5 * 50)
#define DNS_PROXY_BUF_SIZE          512

#define DEFAULT_DNS_IP      0xa85f0101 //"168.95.1.1"
#define IF_NAME             BRG_DEV

typedef  struct DNS_PROXY_IF_LIST_S
{
    char    *pIfName;
    UINT32  if_addr;
    int     sock;
    struct  DNS_PROXY_IF_LIST_S *next;
}
DNS_PROXY_IF_LIST_T;


typedef  struct DNS_PROXY_DNS_LIST_S
{
    struct sockaddr_in	dns_addr;
    struct DNS_PROXY_DNS_LIST_S *next;
}
DNS_PROXY_DNS_LIST_T;


typedef struct DNS_PROXY_ENTRY_S
{
    INT32   used;
    UINT32  count;
    UINT32  time;
    int     in_sock;
    int     out_sock;
    struct sockaddr_in	cAddr;
}
DNS_PROXY_ENTRY_T;

/**********************************
 Variables
***********************************/
//static  KNL_MSGQ_ID             DnsProxy_MsgQ = 0;
static  DNS_PROXY_IF_LIST_T     *pDnsProxy_IfList = 0;
static  DNS_PROXY_DNS_LIST_T    *pDnsProxy_DnsList = 0;
static  DNS_PROXY_ENTRY_T       *pDnsProxy_Table = 0;
static  char                    *pDnsProxy_Buffer = 0;

static  UINT32  apCfgWanDns[MAX_DYN_DNS];

static  struct  sockaddr_in     known_dns;


//*************************
// Functions
//*************************
typedef struct __DNS_PROXY_CMD_T
{
    int op;
    union msg
    {
        UINT32  dns_ip;
        char    ifname[8];
    }
    data;
}
DNS_PROXY_CMD_T;

#define DNS_PROXY_SET_WELL_KNOWN_DNS    1
#define DNS_PROXY_CLEAN_DNS             2
#define DNS_PROXY_ADD_DNS               3
#define DNS_PROXY_DEL_DNS               4
#define DNS_PROXY_ADD_IF                5
#define DNS_PROXY_DEL_IF                6
#define DNS_PROXY_SHUTDOWN              7

/******************************************************************/
static void getApcfgWanCurDns()
{
    // int i;
    // char tokenBuf[MAX_AP_TOKEN_LEN];

    initApShareMem();
#if 1 /* Linux, we parse the /etc/reslove.conf*/

#else
    for(i=1;i<=MAX_DYN_DNS;i++)
    {
        sprintf(tokenBuf,"%s%d",WAN_CUR_DNS_TOK,i);

        apCfgWanDns[i-1]=apCfgGetIntValue(tokenBuf);
    }    
#endif
    destroyApShareMem();
}

/*-----------------------------------------------------------------
* ROUTINE NAME - DnsProxy_TableTimedOut
*------------------------------------------------------------------
* FUNCTION: 
* INPUT:    cAddr, the address of client which issue the dns request
*           outsock, for returning the new socket 
* OUTPUT:   the index number of DnsProxy_Table entry 
---------------------------------------------------------------*/
static  INT32   DnsProxy_TableTimedOut()
{

    int i;

    for (i=0; i<DNS_PROXY_ENTRY_MAX; i++)
    {
        if (pDnsProxy_Table[i].used == 1)
        {
            if (++pDnsProxy_Table[i].time > DNS_PROXY_TIMEOUT)
            {
                pDnsProxy_Table[i].used = 0;    // Clear this one

                close(pDnsProxy_Table[i].out_sock);
            }
        }
    }

    return 0;
}


/*-----------------------------------------------------------------
* ROUTINE NAME - DnsProxy_GetClientAddr
*------------------------------------------------------------------
* FUNCTION: This routine be called when getting dns response. It 
*           finds the address of client that issued the dns request
*           by using socket as key.
* INPUT:    cAddr, for return the address of client 
*           sd, as a key for finding the client address 
* OUTPUT:   1, when find the corresponding address to the input socket
*           0 (ERROR), otherwise.
---------------------------------------------------------------*/
static  DNS_PROXY_ENTRY_T   *DnsProxy_GetClientEntry(int sd)
{
    int i;

    for (i=0; i<DNS_PROXY_ENTRY_MAX; i++)
    {
        if (pDnsProxy_Table[i].used == 1 &&
            pDnsProxy_Table[i].out_sock == sd)
        {
            return (pDnsProxy_Table+i);
        }
    }

    return 0;
}


/*-----------------------------------------------------------------
* ROUTINE NAME - DnsProxy_AddEntry
*------------------------------------------------------------------
* FUNCTION: This routine be called for geting new socket to connect 
*           dns server. It also find an available entry of 
*           DnsProxy_Table for storing the client address and the new
*           socket by running a last-update algorithm.
* INPUT:    cAddr, the address of client which issue the dns request
*           outsock, for returning the new socket 
* OUTPUT:   the index number of DnsProxy_Table entry 
---------------------------------------------------------------*/
static void DnsProxy_EntryAdd(struct sockaddr_in cAddr,
                              int in_sock,
                              int out_sock)
{
    int i;
    int found = 0;
    int entry = 0;

    for (i=0; i<DNS_PROXY_ENTRY_MAX; i++)
    {
        //
        // Increase count anyway
        //
        if (++pDnsProxy_Table[i].count == 0)
            pDnsProxy_Table[i].count--;

            // 
            // Find an empty entry
            //
            if (pDnsProxy_Table[i].used == 0)
            {
                found = 1;
                entry = i;
            }
            else if (found == 0)
            {
                //
                // An entry older than current most oldest one
                // Take this one as the oldest one
                //
                if (pDnsProxy_Table[i].count >= pDnsProxy_Table[entry].count)
                {
                    entry = i;
                }
        }
    }

    //
    // Not found, close the oldest entry's socket
    //
    if (found == 0)
    {
        close(pDnsProxy_Table[entry].out_sock);
    }

    //
    // add this entry to it
    //
    pDnsProxy_Table[entry].used = 1;
    pDnsProxy_Table[entry].count = 0;
    pDnsProxy_Table[entry].time = 0;
    pDnsProxy_Table[entry].in_sock = in_sock;
    pDnsProxy_Table[entry].out_sock = out_sock;
    memcpy(&pDnsProxy_Table[entry].cAddr, &cAddr, sizeof(struct sockaddr_in));
}


/*-----------------------------------------------------------------
* ROUTINE NAME - DnsProxy_DelEntry
*------------------------------------------------------------------
* FUNCTION: 
* INPUT:    cAddr, the address of client which issue the dns request
*           outsock, for returning the new socket 
* OUTPUT:   the index number of DnsProxy_Table entry 
---------------------------------------------------------------*/
static INT32  DnsProxy_EntryDelete(int sd)
{
    int i;

    for (i=0; i<DNS_PROXY_ENTRY_MAX; i++)
    {
        if (pDnsProxy_Table[i].used == 1 &&
            pDnsProxy_Table[i].out_sock == sd)
        {
            pDnsProxy_Table[i].used = 0;    // Clear this one

            close(sd);
            return i;
        }
    }

    return -1;
}


/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_IfIpAddr
*---------------------------------------------------------------
* FUNCTION: This routine creates dns relay task
*
* INPUT:    None
---------------------------------------------------------------*/
static UINT32   DnsProxy_IfIpAddr(char *pIfName)
{
    int so;
    int status;
    u_long ipaddr;
    struct ifreq  ifr;

    // Open a raw socket
    if ((so = socket(PF_PACKET,SOCK_PACKET,htons(ETH_P_ALL))) < 0)
	    return 0;

    // Prepare the interface request structure to get address
    strncpy (ifr.ifr_name, pIfName, sizeof (ifr.ifr_name));

    // Do ioctl to get interface address
    status = ioctl (so, SIOCGIFADDR, (int)&ifr);
    close (so);

    if (status != 0)
        return 0;

    // Retrieve the IP address from ifreq
    ipaddr = ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr;

    return ntohl(ipaddr);
}




//###############################################################
// Message Queue process routeine
//###############################################################

/*--------------------------------------------------------------
* ROUTINE NAME - set_well_known_dns
*---------------------------------------------------------------
* FUNCTION: This routine set the default dns
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
static  int set_well_known_dns(dns_ip)
{
    memset(&known_dns, 0, sizeof(known_dns));
                                   
    // Prepare a socket to a well-known dns server

    known_dns.sin_family = AF_INET;
    known_dns.sin_port = htons(DNS_SERVICE);

    if (dns_ip == 0)
        known_dns.sin_addr.s_addr = htonl(DEFAULT_DNS_IP);
    else
        known_dns.sin_addr.s_addr = htonl(dns_ip);

    return 0;
}
#endif

/*--------------------------------------------------------------
* ROUTINE NAME - cleanup_dns
*---------------------------------------------------------------
* FUNCTION: This routine cleans up all the dns entry
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
static  int cleanup_dns()
{
    DNS_PROXY_DNS_LIST_T *pTemp;

    while (pDnsProxy_DnsList != 0)
    {
        pTemp = pDnsProxy_DnsList;
        pDnsProxy_DnsList = pDnsProxy_DnsList->next;

        // Free this entry	
        free(pTemp);
    }

    return 0;
}
#endif

/*--------------------------------------------------------------
* ROUTINE NAME - add_dns
*---------------------------------------------------------------
* FUNCTION: This routine added a dns entry in the list
*
* INPUT:    None
---------------------------------------------------------------*/
static  int add_dns(UINT32 new_dns_ip)
{

    DNS_PROXY_DNS_LIST_T    *pPrev, *pTemp, *pNewEntry;

#if 0    
    if (new_dns_ip == 0)
    {
        return ERROR;
    }
#endif

    if ((pNewEntry = (DNS_PROXY_DNS_LIST_T *)calloc(1, sizeof(DNS_PROXY_DNS_LIST_T))) == 0)
    {
        return ERROR;
    }

    //
    // Initiate the new DNS
    //
    pNewEntry->dns_addr.sin_family = AF_INET;
    pNewEntry->dns_addr.sin_port = htons(DNS_SERVICE);
    pNewEntry->dns_addr.sin_addr.s_addr = htonl(new_dns_ip);

    pNewEntry->next = 0;


    //
    // Check duplication
    //

    pTemp = pDnsProxy_DnsList;
    pPrev = 0;
    while (pTemp != 0)
    {
        if (pTemp->dns_addr.sin_addr.s_addr == htonl(new_dns_ip))
        {
            free(pNewEntry);
            return ERROR;
        }

        pPrev = pTemp;
        pTemp = pTemp->next;
    }

    //
    // Append the DNS to the end
    //
    if (pPrev == 0)
        pDnsProxy_DnsList = pNewEntry;
    else
        pPrev->next = pNewEntry;

    return OK;
}



/*--------------------------------------------------------------
* ROUTINE NAME - del_dns
*---------------------------------------------------------------
* FUNCTION: This routine delete a dns entry from list
*
* INPUT:    None
---------------------------------------------------------------*/
static int del_dns(UINT32 old_dns_ip)
{
    DNS_PROXY_DNS_LIST_T *pTemp, *pPrev;

    if (old_dns_ip == 0)
    {
        return ERROR;
    }

    pTemp = pDnsProxy_DnsList;
    pPrev = 0;
    while (pTemp != 0)
    {
        if (pTemp->dns_addr.sin_addr.s_addr == htonl(old_dns_ip))
        {
            if (pPrev == 0)
            {
                pDnsProxy_DnsList = pTemp->next;
            }
            else
            {
                pPrev->next = pTemp->next;
            }

            // Free this entry	
            free(pTemp);
            return OK;
        }
        else
        {
            pPrev = pTemp;
            pTemp = pTemp->next;			
        }
    }

    return ERROR;
}


/*--------------------------------------------------------------
* ROUTINE NAME - add_if
*---------------------------------------------------------------
* FUNCTION: This routine adds an interface to the dns if list
*
* INPUT:    None
---------------------------------------------------------------*/
static int add_if(char *pIfName)
{
    DNS_PROXY_IF_LIST_T  *pTemp, *pNewEntry;

    char    *pName;
    UINT32  if_ip;

    struct  sockaddr_in	if_saddr;
    int     sd = -1;

    //
    // Get the interface name, and bind the interface IP
    // to the DNS socket
    //

    if (pIfName == 0)
        return ERROR;

    if ((if_ip = DnsProxy_IfIpAddr(pIfName)) == 0)
        return ERROR;

    if ((sd = socket(PF_INET, SOCK_DGRAM, 0)) == ERROR)
        return ERROR;


    memset(&if_saddr, 0, sizeof(if_saddr));
    if_saddr.sin_family = AF_INET;

    if_saddr.sin_port = htons(DNS_SERVICE);
    if_saddr.sin_addr.s_addr = htons(INADDR_ANY);/*htonl(if_ip)*/

    if (bind(sd, (struct sockaddr *) &if_saddr, sizeof(if_saddr))<0)
    {
        perror("bind");
        close(sd);
        return ERROR;
    }

    listen(sd, 1);

    /****************************
      Don't insert a existent one
     ****************************/
    pTemp = pDnsProxy_IfList;
    while (pTemp != 0)
    {
        if (strcmp(pTemp->pIfName, pIfName) == 0)
        {
            goto if_add_clean1;

        }
        pTemp = pTemp->next;
    }

    if ((pName = (char *)calloc(1, strlen(pIfName)+1)) == 0)
    {
        goto if_add_clean1;
    }


    if ((pNewEntry = (DNS_PROXY_IF_LIST_T *)calloc(1, sizeof(DNS_PROXY_IF_LIST_T))) == 0)
    {
        goto if_add_clean2;
    }

    /***************************
      Insert the new one
     ***************************/
    strcpy(pName, pIfName);

    pNewEntry->pIfName = pName;
    pNewEntry->if_addr = htons(INADDR_ANY);//if_ip;
    pNewEntry->sock = sd;

    pNewEntry->next = pDnsProxy_IfList;
    pDnsProxy_IfList = pNewEntry;

    return OK;
#if 0
if_add_clean3:
    if (pNewEntry)
        free(pNewEntry);
#endif
if_add_clean2:
    if (pName)
    {
        free(pName);
    }
if_add_clean1:
    if (sd != -1)
    {
        close(sd);
    }
    return ERROR;
}


/*--------------------------------------------------------------
* ROUTINE NAME - del_if
*---------------------------------------------------------------
* FUNCTION: This routine deletes an interface from the dns if list
*
* INPUT:    None
---------------------------------------------------------------*/
static int del_if(char *pIfName)
{
    DNS_PROXY_IF_LIST_T  *pTemp, *pPrev;
    int sock;
    int i;

    //
    // Check the interface name
    //
    if (pIfName == 0)
        return ERROR;

    //
    // Wait semaphore
    //
    pTemp = pDnsProxy_IfList;
    pPrev = 0;
    while (pTemp != 0)
    {
        if (strcmp(pTemp->pIfName, pIfName) == 0)
        {
            if (pPrev == 0)
            {
                pDnsProxy_IfList = pTemp->next;
            }
            else
            {
                pPrev->next = pTemp->next;
            }

            // Clear all waiting reply socket
            sock = pTemp->sock;		
            for (i=0; i<DNS_PROXY_ENTRY_MAX; i++)
            {
                if (pDnsProxy_Table[i].used == 1 &&
                    pDnsProxy_Table[i].in_sock == sock)
                {
                    pDnsProxy_Table[i].used = 0;
                    close(pDnsProxy_Table[i].out_sock);
                }
            }

            close (sock);
            free(pTemp->pIfName);
            free(pTemp);

            return OK;
        }
        else
        {
            pPrev = pTemp;
            pTemp = pTemp->next;
        }
    }

    return ERROR;
}


/*--------------------------------------------------------------

* ROUTINE NAME - dns_shutdown
*---------------------------------------------------------------
* FUNCTION: This routine shutdown the dns task
*           After this function is executed, the task should
*           call KNL_EXIT to end to task.
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
static void dns_shutdown()
{
    INT32   i;

    //
    // Free DNS receive buffer
    //
    free (pDnsProxy_Buffer);

    //
    // Free DNS proxy table
    //

    for (i=0; i<DNS_PROXY_ENTRY_MAX; i++)
    {
        if (pDnsProxy_Table[i].used == 1)
        {
            close(pDnsProxy_Table[i].out_sock);
        }
    }

    free (pDnsProxy_Table);

    //
    // Free interface list
    //
    while (pDnsProxy_IfList != 0)
    {
        DNS_PROXY_IF_LIST_T  	*pIfTemp;

        pIfTemp = pDnsProxy_IfList;

        pDnsProxy_IfList = pDnsProxy_IfList->next;

        close (pIfTemp->sock);

        if (pIfTemp->pIfName)
            free(pIfTemp->pIfName);

        free (pIfTemp);
    }

    //
    // Free DNS IP list
    //
    while (pDnsProxy_DnsList != 0)
    {
        DNS_PROXY_DNS_LIST_T    *pDnsTemp;


        pDnsTemp = pDnsProxy_DnsList;

        pDnsProxy_DnsList = pDnsProxy_DnsList->next;
        free (pDnsTemp);
    }

//    KNL_MSGQ_DELETE(DnsProxy_MsgQ);
    DnsProxy_MsgQ = 0;

    // Exit this task
//    KNL_TASK_EXIT();
}
#endif

/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_ProcessMsg
*---------------------------------------------------------------
* FUNCTION: This routine processes the message in the message Q
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
static  int DnsProxy_ProcessMsg()
{
//    DNS_PROXY_CMD_T cmd;

#if 0
    if (KNL_MSGQ_RECV(DnsProxy_MsgQ, &cmd, sizeof(cmd), KNL_NOWAIT) != ERROR)
    {
        switch (cmd.op)
        {
        case DNS_PROXY_SET_WELL_KNOWN_DNS:
            set_well_known_dns(cmd.data.dns_ip);
            break;

        case DNS_PROXY_CLEAN_DNS:
            cleanup_dns();
            break;
    
        case DNS_PROXY_ADD_DNS:
            add_dns(cmd.data.dns_ip);
            break;

        case DNS_PROXY_DEL_DNS:
            del_dns(cmd.data.dns_ip);
            break;

        case DNS_PROXY_ADD_IF:
            add_if(cmd.data.ifname);
            break;

        case DNS_PROXY_DEL_IF:
            del_if(cmd.data.ifname);
            break;

        case DNS_PROXY_SHUTDOWN:
            dns_shutdown();
            break;

        default:
            break;
        }
    }
#endif
    return 0;
}
#endif


/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_Init
*---------------------------------------------------------------
* FUNCTION: This routine Init the dns task variables
*
* INPUT:    None
---------------------------------------------------------------*/
static  int DnsProxy_Init()
{
    //

    // Allocate buffer for proxy buffer and proxy table
    //

    if ((pDnsProxy_Buffer = (char *)malloc(DNS_PROXY_BUF_SIZE)) == 0)
    {
        return -1;
    }

    pDnsProxy_Table = (DNS_PROXY_ENTRY_T *)calloc(1, DNS_PROXY_ENTRY_MAX * sizeof(DNS_PROXY_ENTRY_T));
    if (pDnsProxy_Table == 0) 
    {
        free(pDnsProxy_Buffer);
        return -1;
    }

    pDnsProxy_IfList = 0;
    if(add_if(IF_NAME)==ERROR)
        return -1;

    pDnsProxy_DnsList = 0;
    {
        int i;
        for(i=0;i<MAX_DYN_DNS;i++)
        {
            if(apCfgWanDns[i]==0)
                continue;    
            add_dns(apCfgWanDns[i]);
        }        
    }

    // Do well known address
    memset(&known_dns, 0, sizeof(known_dns));

    // Prepare a socket to a well-known dns server
    known_dns.sin_family = AF_INET;
    known_dns.sin_port = htons(DNS_SERVICE);
    known_dns.sin_addr.s_addr = htonl(DEFAULT_DNS_IP);    // inet_addr return network order

    return 0;
}


/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_Daemon
*---------------------------------------------------------------
* FUNCTION: The main routine of dns relay agent.
*           It listens dns request on lan port, relays request
*           to real dns server whatever interface which can reach
*           dns server. Then wait for response from server.
*           when receive response from server, it relay the response
*           back to client.
* INPUT:    None
* OUTPUT:   None
---------------------------------------------------------------*/
static  void DnsProxy_Daemon()

{
    int     out_sock;
    struct  sockaddr_in clientAddr;
    struct  sockaddr_in dnsAddr;
    int     addrlen;
    int     i, /*m,*/ n;

    struct  timeval wait_time;
    fd_set  rcv_fds;

    DNS_PROXY_IF_LIST_T *pTemp;
    DNS_PROXY_ENTRY_T   *client;

    //
    // Do DNS proxy initialization
    //

    if (DnsProxy_Init() != 0)
    {

        return;
    }

    //
    // Initialization completed
    //
    wait_time.tv_sec = 0;           /*  no.  of seconds  */
    wait_time.tv_usec = 20000;      /*  20 mini-second  */

    //
    // Main loop to process the DNS proxy packet
    //
    while (1)
    {
               
        FD_ZERO(&rcv_fds);

        usleep(300); //0.3sec
        
        checkWanDnsChanged();

        
        /***********************************
         Process message
         ***********************************/
      //  DnsProxy_ProcessMsg();

        /***********************************
         Bind all listen socket
         ***********************************/
        // Listen proxy interface ip sockets
        pTemp = pDnsProxy_IfList;


        while (pTemp != 0)
        {
            usleep(300); //0.3sec
            FD_SET(pTemp->sock, &rcv_fds);
            pTemp = pTemp->next;
        }

      
        // Listen reply packet sockets
        for (i=0; i<DNS_PROXY_ENTRY_MAX; i++)
        {

            if (pDnsProxy_Table[i].used == 1)
            {
                FD_SET(pDnsProxy_Table[i].out_sock, &rcv_fds);
            }
        }


        /***************************************
         Check if there are any packet coming
        ****************************************/

        n = select(FD_SETSIZE, &rcv_fds, (fd_set *)NULL, 
                   (fd_set *)NULL, &wait_time);
        if (n <= 0) 
        {
			DnsProxy_TableTimedOut();
            continue;
        }


    
        /***********************************************
         Read the packets comming from private network
         ***********************************************/
        pTemp = pDnsProxy_IfList;
        while (pTemp != 0)
        {
            usleep(300); //0.3sec
            if (FD_ISSET(pTemp->sock, &rcv_fds))
            {   
                //
                // Read from the private network if dns request comming
                //
                addrlen = sizeof(clientAddr);
                
                if ((n = recvfrom(pTemp->sock, pDnsProxy_Buffer, DNS_PROXY_BUF_SIZE, 0,
                                  (struct sockaddr *)&clientAddr, &addrlen)) < 0)
                {
                    continue;
                }
                //
                // Relay the DNS request to the DNS server
                //
                if (pDnsProxy_DnsList != 0)
                {
                    DNS_PROXY_DNS_LIST_T *pTempDns;

                    out_sock = socket(AF_INET,SOCK_DGRAM,0);

                    // Send to all dns ...
                    pTempDns = pDnsProxy_DnsList;
                    while (pTempDns != 0)
                    {   
                        usleep(300); //0.3sec
                        sendto(out_sock, pDnsProxy_Buffer, n, 0,
                               (struct sockaddr *)&(pTempDns->dns_addr), sizeof(struct sockaddr_in));

                        pTempDns = pTempDns->next;
                       
                    }
                    

                    DnsProxy_EntryAdd(clientAddr, pTemp->sock, out_sock);
                }
                else
                { 
                    // Simon for pppoe connect on demand used
                    // Send to 168.95.1.1
                    out_sock = socket(AF_INET,SOCK_DGRAM,0);

                    // Send to it
                    sendto(out_sock, pDnsProxy_Buffer, n, 0, (struct sockaddr *)&known_dns, sizeof(struct sockaddr_in));

                    // Simon ++ 2000.09.26 
                    DnsProxy_EntryAdd(clientAddr, pTemp->sock, out_sock);
                }
            }

            FD_CLR(pTemp->sock, &rcv_fds); 
            pTemp = pTemp->next;
        }
        
        /********************************************
         Poll all the sockets created for DNS relay
         ********************************************/
        for (i=0; i<FD_SETSIZE; i++)
        {
            if (FD_ISSET(i, &rcv_fds))
            {
                if ((client = DnsProxy_GetClientEntry(i)) != 0)
                {
                    addrlen = sizeof(dnsAddr);
                    if ((n = recvfrom(i, pDnsProxy_Buffer, DNS_PROXY_BUF_SIZE, 0,
                                      (struct sockaddr *)&dnsAddr, &addrlen)) >= 0)
                    {
                        sendto(client->in_sock, pDnsProxy_Buffer, n, 0, 
                               (struct sockaddr *)&(client->cAddr), sizeof(struct sockaddr_in));
                    }

                    DnsProxy_EntryDelete(i);
                }
                else
                {
                    close(i);
                }
            }

        } /* End for */

    } /* End while */
} 


/* ############################################################################
  Configuration function access by outside function, need to do mutual 
  exculsive semaphore ...
  ############################################################################*/
/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_SetWellKnownDns
*---------------------------------------------------------------
* FUNCTION: This routine creates dns relay task
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
INT32	DnsProxy_SetWellKnownDns(UINT32 dns_ip)
{
    DNS_PROXY_CMD_T cmd;

    cmd.op = DNS_PROXY_SET_WELL_KNOWN_DNS;
    cmd.data.dns_ip = dns_ip;

    return 1;//KNL_MSGQ_SEND(DnsProxy_MsgQ, &cmd, sizeof(cmd), KNL_WAIT);
}
#endif


/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_DnsClean
*---------------------------------------------------------------
* FUNCTION: This routine creates dns relay task
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
INT32	DnsProxy_DnsClean()
{
    DNS_PROXY_CMD_T cmd;

    cmd.op = DNS_PROXY_CLEAN_DNS;

    return 1;//KNL_MSGQ_SEND(DnsProxy_MsgQ, &cmd, sizeof(cmd), KNL_WAIT);
}
#endif


/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_DnsAdd
*---------------------------------------------------------------
* FUNCTION: This routine creates dns relay task
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
INT32   DnsProxy_DnsAdd(UINT32 new_dns_ip)
{
    DNS_PROXY_CMD_T cmd;

    cmd.op = DNS_PROXY_ADD_DNS;
    cmd.data.dns_ip = new_dns_ip;

    return 1;//KNL_MSGQ_SEND(DnsProxy_MsgQ, &cmd, sizeof(cmd), KNL_WAIT);
}
#endif

/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_DnsDelete
*---------------------------------------------------------------
* FUNCTION: This routine creates dns relay task
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
INT32   DnsProxy_DnsDelete(UINT32 old_dns_ip)
{
    DNS_PROXY_CMD_T cmd;

    cmd.op = DNS_PROXY_DEL_DNS;
    cmd.data.dns_ip = old_dns_ip;

    return 1;//KNL_MSGQ_SEND(DnsProxy_MsgQ, &cmd, sizeof(cmd), KNL_WAIT);
}
#endif

/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_IfAdd
*---------------------------------------------------------------
* FUNCTION: This routine creates dns relay task
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
INT32   DnsProxy_IfAdd(char *pIfName)
{
    DNS_PROXY_CMD_T cmd;

    cmd.op = DNS_PROXY_ADD_IF;

    strncpy(cmd.data.ifname, pIfName, sizeof(cmd.data.ifname)-1);
    cmd.data.ifname[sizeof(cmd.data.ifname)-1] = 0;

    return 1;//KNL_MSGQ_SEND(DnsProxy_MsgQ, &cmd, sizeof(cmd), KNL_WAIT);
}
#endif

/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_IfDelete
*---------------------------------------------------------------
* FUNCTION: 
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
INT32   DnsProxy_IfDelete(char *pIfName)
{
    DNS_PROXY_CMD_T cmd;


    cmd.op = DNS_PROXY_DEL_IF;

    strncpy(cmd.data.ifname, pIfName, sizeof(cmd.data.ifname)-1);

    cmd.data.ifname[sizeof(cmd.data.ifname)-1] = 0;

    return 1;//KNL_MSGQ_SEND(DnsProxy_MsgQ, &cmd, sizeof(cmd), KNL_WAIT);
}
#endif

/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_Stop
*---------------------------------------------------------------
* FUNCTION: This routine set the DNS proxy flag to false, let the
*           DnsProxy_Daemo to kill itself.
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
INT32 	DnsProxy_Stop()
{
    DNS_PROXY_CMD_T cmd;

    cmd.op = DNS_PROXY_SHUTDOWN;

    return 1;//KNL_MSGQ_SEND(DnsProxy_MsgQ, &cmd, sizeof(cmd), KNL_WAIT);
}

#endif
/*--------------------------------------------------------------
* ROUTINE NAME - DnsProxy_Start
*---------------------------------------------------------------
* FUNCTION: This routine creates dns proxy task
*
* INPUT:    None
---------------------------------------------------------------*/
#if 0
INT32	DnsProxy_Start()
{
//    KNL_TASK_ID tid;


    if (DnsProxy_MsgQ != 0)
        return OK;

    //
    // Create message queue
    //
#if 0    
    if (KNL_MSGQ_CREATE("dnsproxy", &DnsProxy_MsgQ, 1, sizeof(DNS_PROXY_CMD_T)) != OK)
    {
        DnsProxy_MsgQ = 0;
        return ERROR;
    }
#endif    

    // 
    // Run DnsProxy_Agent
    //
#if 0    
    if (KNL_TASK_CREATE("tDnsProxy", &tid, 0, (FUNCPTR)DnsProxy_Daemon, 0, 0) != OK)
    {
        KNL_MSGQ_DELETE(DnsProxy_MsgQ);
        DnsProxy_MsgQ = 0;
    }

#endif
    return OK;
}
#endif

static int DnsProxy_Destroy()
{

    free(pDnsProxy_Buffer);

    free(pDnsProxy_Table);

    if(del_if(IF_NAME)==ERROR)
        return -1;  
    
    {
        int i;
        for(i=0;i<MAX_DYN_DNS;i++)
        {
            if(apCfgWanDns[i]==0)
                continue;
            del_dns(apCfgWanDns[i]);
        }
    }

    memset(&known_dns, 0, sizeof(known_dns));

    return 0;
}

static void resetDnsProxy()
{
    //getApcfgWanCurDns();
    DnsProxy_Destroy();
    DnsProxy_Init();
}

static void checkWanDnsChanged()
{
    struct sigaction act;
    act.sa_handler = resetDnsProxy;
    sigemptyset(&act.sa_mask);
    act.sa_flags = 0;
    /*kill -16 pid for IDT linux os platform*/
    sigaction(SIGUSR1,&act,0);
}

int main(int agrc , char **argv)
{

    //getApcfgWanCurDns();
    DnsProxy_Daemon();
    return 0;
}
