/**
 * \addtogroup httpd
 * @{
 */

/**
 * \file
 * HTTP server script language C functions file.
 * \author Adam Dunkels <adam@dunkels.com>
 *
 * This file contains functions that are called by the web server
 * scripts. The functions takes one argument, and the return value is
 * interpreted as follows. A zero means that the function did not
 * complete and should be invoked for the next packet as well. A
 * non-zero value indicates that the function has completed and that
 * the web server should move along to the next script line.
 *
 */

/*
 * Copyright (c) 2001, Adam Dunkels.
 * All rights reserved. 
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met: 
 * 1. Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright 
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.  
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.  
 *
 * This file is part of the uIP TCP/IP stack.
 *
 * $Id: cgi.c,v 1.3 2009/09/10 03:28:05 jackey Exp $
 *
 */

#include "uip.h"
#include "cgi.h"
#include "httpd.h"
#include "fs.h"
#include "bsp.h"

#include <stdio.h>
#include <string.h>
#include "../init/rtk.h"

#define HWID_CHECK_LEN	        22
#define HWID_LEN                24

int content_length = 0;

// Predefine function
char *get_request_line(int *request_len, int *request_idx);
u8_t upgrade_cgi(u8_t next);

cgifunction cgitab[] = {
	upgrade_cgi /* CGI function "c" */
};
/* Firmware image file header */
#define IMG_SIGNATURE_LEN			4

static const char closed[] =   /*  "CLOSED",*/
{0x43, 0x4c, 0x4f, 0x53, 0x45, 0x44, 0};
static const char syn_rcvd[] = /*  "SYN-RCVD",*/
{0x53, 0x59, 0x4e, 0x2d, 0x52, 0x43, 0x56, 
	0x44,  0};
static const char syn_sent[] = /*  "SYN-SENT",*/
{0x53, 0x59, 0x4e, 0x2d, 0x53, 0x45, 0x4e, 
	0x54,  0};
static const char established[] = /*  "ESTABLISHED",*/
{0x45, 0x53, 0x54, 0x41, 0x42, 0x4c, 0x49, 0x53, 0x48, 
	0x45, 0x44, 0};
static const char fin_wait_1[] = /*  "FIN-WAIT-1",*/
{0x46, 0x49, 0x4e, 0x2d, 0x57, 0x41, 0x49, 
	0x54, 0x2d, 0x31, 0};
static const char fin_wait_2[] = /*  "FIN-WAIT-2",*/
{0x46, 0x49, 0x4e, 0x2d, 0x57, 0x41, 0x49, 
	0x54, 0x2d, 0x32, 0};
static const char closing[] = /*  "CLOSING",*/
{0x43, 0x4c, 0x4f, 0x53, 0x49, 
	0x4e, 0x47, 0};
static const char time_wait[] = /*  "TIME-WAIT,"*/
{0x54, 0x49, 0x4d, 0x45, 0x2d, 0x57, 0x41, 
	0x49, 0x54, 0};
static const char last_ack[] = /*  "LAST-ACK"*/
{0x4c, 0x41, 0x53, 0x54, 0x2d, 0x41, 0x43, 
	0x4b, 0};

unsigned long total_size=0;
unsigned long total_filesize=0;
unsigned long total_packets=0;

#define PRODUCT_NAME	"G700APNB"
#define FILESTART 0x80300000
#define BUFF_LOCATION FILESTART
#define LOADER_LOCATION (0x10000000)
#define KERNEL_LOCATION (0x10050000)
#define BUFFER_SIZE (3*1024*1024)		// 1 MB

#define KERNEL_LENGTH 0xE0000
#define ROOTFS_LENGTH 0x2F0000

int buffer_ptr=0;

static void cgi_get_flash_buf(u8_t **bufptr, int *bufsize)
{
	int size = BUFFER_SIZE;

	if (size > 0) {
		*bufptr = (u8_t *) BUFF_LOCATION;
		*bufsize = size;
	}
}


char *get_request_line(int *request_len, int *request_idx)
{
	int i;
	char c;

	for ( i = *request_idx; *request_idx < *request_len; ++(*request_idx) )
	{
		c = uip_appdata[*request_idx];
		if ( c == '\n' || c == '\r' )
		{
			uip_appdata[*request_idx] = '\0';
			++(*request_idx);
			if ( c == '\r' && *request_idx < *request_len && uip_appdata[*request_idx] == '\n' )
			{
				uip_appdata[*request_idx] = '\0';
				++(*request_idx);
			}
			return (char *)&(uip_appdata[i]);
		}
	}
	return (char*) 0;
}

u8_t upgrade_cgi(u8_t next)
{
	char *line;
	char *cp;
	unsigned long dest, count,addr;	
	unsigned long burnlen;
	u8_t *phwid;
	char imageHWID[HWID_LEN];
	int x;
	int request_len,request_idx;
	int len;
	IMG_HEADER_Tp pHeader;
	u8_t *ptr;
	int bufsize = 0;
	int flag=0;
	int imglen=0;
	cgi_get_flash_buf(&ptr, &bufsize);

	if(next == 1) 
		return 1;

	request_len = uip_datalen();
	request_idx = 0;

	
	if (total_size == 0)	{
		while ((line = get_request_line(&request_len, &request_idx) ) != (char*) 0 ) {
			if( line[0] == '\0')
			 break;
			/*	Get "POST /cgi/index"	and reset all 	*/
			else if(strncmp(line, "POST /cgi/index", 15) ==0)	{
				total_size=0;
				total_packets = 1;
				total_filesize=0;
			}


			/*	Get Content-Length	*/
			else if(strncmp(line, "Content-Length:", 15) ==0 )	{
				cp=&line[16];
				total_filesize = simple_strtol(cp,NULL,0);
				total_packets++;
			}
      
      			else if( strncmp( line, "-----------------------------", 29 ) == 0 ) {
				content_length = strlen(line);	
			}
			/*	Start to receive firmware data	*/
			else if (strncmp(line, "Content-Type: application/octet-stream", 25) ==0) {
				//prom_printf("<<< upgrade_cgi >>> Start to receive firmware data\n");
	    		/*	deduct the attachment in front of HTTP Data	*/
				cp = (char *)&uip_appdata[request_idx+2];
				if(uip_datalen() != 1446) {
   				total_filesize = total_filesize - (request_idx+2) - content_length - 6;
				} else {
   				total_filesize = total_filesize - (request_idx+2);
				} 
				len = uip_datalen() - (request_idx+2);
				total_size = len;
				memcpy(ptr,cp,len);
				buffer_ptr=len;
                		total_packets++;
				//prom_printf("total_filesize is %d\n",total_filesize);
			}
		}
	} else if(total_size != 0 )	{
		len = request_len;
		total_size += len;
		prom_printf(".");
        	cp = (char *)&uip_appdata[0];
		if(total_size < total_filesize)	
		{
			memcpy( (ptr+buffer_ptr), cp, len );
			buffer_ptr += len;
		} 
		else 
		{
			prom_printf("Check HWID information !!\n");
			memcpy( (ptr+buffer_ptr), cp, len );
			buffer_ptr += len;
			phwid = (void *) (ptr + total_filesize - HWID_LEN);
			memcpy(imageHWID,phwid,HWID_LEN);
			imageHWID[HWID_LEN] = '\0';
			prom_printf("imageHWID = %s,HWID = %s\n",imageHWID,HWID);			
			if(strncmp(imageHWID,HWID,HWID_CHECK_LEN)) {
			        prom_printf("\n Loader Hardware ID Check Error\n");
				total_size=0;
				total_filesize=0;
				total_packets=0;
				return 2;
			}
			if(total_size < 26 * 1024) { //Loader Upgrade Checking
				prom_printf("Upgrade Loader \n");
				pHeader = (IMG_HEADER_Tp) &ptr[0];
				imglen = pHeader->len;
				if(checkAutoFlashing(BUFF_LOCATION, total_filesize - HWID_LEN, 0)){
					prom_printf("Realtek Checksum OK and Upgrade Success!!\n");
					return 1;
				} else {
					prom_printf("Realtek Checksum or Upgrade Fail !!\n");
					return 2;
				}
			} else { 	//Image Upgrade Checking
				pHeader = (IMG_HEADER_Tp) &ptr[0];
				imglen = pHeader->len;
				prom_printf("Upgrade kernel system !! \n");
				if(checkAutoFlashing(BUFF_LOCATION, total_filesize - ROOTFS_LENGTH - HWID_LEN , 0)){
				        prom_printf("Realtek Checksum OK and Upgrade Success!!\n");
				} else {
					prom_printf("Realtek Checksum or Upgrade Fail !!\n");
					return 2;
				}
				ptr = ptr + KERNEL_LENGTH; //Rootfs image header address
				pHeader = (IMG_HEADER_Tp) &ptr[0];
				imglen = pHeader->len;
				if(checkAutoFlashing(BUFF_LOCATION + KERNEL_LENGTH, total_filesize - KERNEL_LENGTH - HWID_LEN, 0)){
					prom_printf("Realtek Checksum OK and Upgrade Success!!\n");
					return 1;
				} else {
				        prom_printf("Realtek Checksum or Upgrade Fail !!\n");
					return 2;
				}

			}	
		}
	}
	return 0;
}
