
/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************/

/*****************************************************************
 *
 * $Workfile: ieee802_1d.h $
 * Description: Here are the types and some function prototypes from 
 *              code supplied in IEEE 802.1D. For Spanning tree protocol.
 *              Minor adaptions of the 802.1D code is made.
 *
 * $Date: 2009/02/17 09:44:45 $
 * $Revision: 1.1.1.1 $
 *
 * $Nokeywords: $
 ********************************************************************/
#ifndef IEEE802_1_H
#define IEEE802_1_H


/***************************************************************************
* DEFINED CONSTANTS
**************************************************************************/

#define Zero    0
#define One     1
#define False   0
#define True    1

/** port states. **/

#define Disabled     0        /* (8.4.5) */
#define Listening    1        /* (8.4.2) */
#define Learning     2        /* (8.4.3) */
#define Forwarding   3        /* (8.4.4) */
#define Blocking     4        /* (8.4.1) */

/** BPDU type constants **/

#define Config_bpdu_type    0
#define Tcn_bpdu_type       128

/** implementation constants. **/
#include <krndef.h>
#define Max_possible_no_of_ports         MAX_LOGIC_PORT 
/* Maximum number of ports the software may need to handle */

#define All_ports             (Max_possible_no_of_ports+1)  
/* ports start at 1, arrays in C start at 0. For allocating arrays */

#define Default_path_cost     10
/* arbitrary */

#define Message_age_increment  1
/* minimum increment possible to avoid underestimating age, allows
   for BPDU transmission time */

#define No_port                0
/* reserved value for Bridge's root port parameter indicating no
   root port, used when Bridge is the root */


/***************************************************************************
* TYPEDEFS, STRUCTURES, AND UNION DECLARATIONS
**************************************************************************/

/** basic types. **/

typedef int Int;    /* to align with convention used here for use of
                       case. Types and defined constants have their
                       initial letters capitalized. */

typedef Int STP_Boolean;     /* : (True, False) */

typedef Int State;         /* : (Disabled, Listening, Learning,
                                 Forwarding, Blocking) */

/** IEEE 802.1D: BPDU encoding types defined in Clause 9, 
"Encoding of Bridge Protocol Data Units" are:

Protocol_version (9.2.2)
Bpdu_type (9.2.3)
Flag (9.2.4)
Identifier (9.2.5)
Cost (9.2.6)
Port_id (9.2.7)
Time (9.2.8)
**/

typedef unsigned char Protocol_version ;

typedef unsigned char Bpdu_type;

typedef STP_Boolean Flag;

typedef struct 
{
    unsigned long  four_octets[2]; 
} Identifier;    /* 8 octets */

typedef int Cost; /* 4 octets */

typedef unsigned short Port_id;  /* 2 octets */

typedef unsigned short Time;  /* 2 octets */
                            


/** Configuration BPDU Parameters (8.5.1) **/

typedef struct
{
    Bpdu_type   type;
    Identifier  root_id;         /* (8.5.1.1) */
    Cost        root_path_cost;  /* (8.5.1.2) */
    Identifier  bridge_id;       /* (8.5.1.3) */
    Port_id     port_id;         /* (8.5.1.4) */
    Time        message_age;     /* (8.5.1.5) */
    Time        max_age;         /* (8.5.1.6) */
    Time        hello_time;      /* (8.5.1.7) */
    Time        forward_delay;   /* (8.5.1.8) */
    Flag        topology_change_acknowledgment;  /* (8.5.1.9) */
    Flag        topology_change; /* (8.5.1.10) */
} Config_bpdu;

/** Topology Change Notification BPDU Parameters (8.5.2) **/

typedef struct
{
    Bpdu_type    type;
} Tcn_bpdu;

/** Bridge Parameters (8.5.3) **/

typedef struct
{
    Identifier      designated_root;          /* (8.5.3.1) */
    Cost            root_path_cost;           /* (8.5.3.2) */
    Int             root_port;                /* (8.5.3.3) */
    Time            max_age;                  /* (8.5.3.4) */
    Time            hello_time;               /* (8.5.3.5) */
    Time            forward_delay;            /* (8.5.3.6) */
    Identifier      bridge_id;                /* (8.5.3.7) */
    Time            bridge_max_age;           /* (8.5.3.8) */
    Time            bridge_hello_time;        /* (8.5.3.9) */
    Time            bridge_forward_delay;     /* (8.5.3.10) */
    STP_Boolean         topology_change_detected; /* (8.5.3.11) */
    STP_Boolean         topology_change;          /* (8.5.3.12) */
    Time            topology_change_time;     /* (8.5.3.13) */
    Time            hold_time;                /* (8.5.3.14) */
} Bridge_data;

/** Port Parameters (8.5.5) **/

typedef struct
{
    Port_id        port_id;                    /* (8.5.5.1) */
    State          state;                      /* (8.5.5.2) */
    Int            path_cost;                  /* (8.5.5.3) */
    Identifier     designated_root;            /* (8.5.5.4) */
    Int            designated_cost;            /* (8.5.5.5) */
    Identifier     designated_bridge;          /* (8.5.5.6) */
    Port_id        designated_port;            /* (8.5.5.7) */
    STP_Boolean        topology_change_acknowledge;/* (8.5.5.8) */
    STP_Boolean        config_pending;             /* (8.5.5.9) */
    STP_Boolean        change_detection_enabled;   /* (8.5.5.10) */
} Port_data;

/** types to support timers for this pseudo-implementation. **/

typedef struct
{
    STP_Boolean     active;       /* timer in use. */
    Time        value;        /* current value of timer, counting up. */
} Timer;




/* Function prototypes */
int id_cmp(Identifier *a, Identifier *b);
void transmit_config(Int port_no); /* (8.6.1) */

STP_Boolean root_bridge(void);
STP_Boolean supersedes_port_info(Int port_no, Config_bpdu *config); /* (8.6.2.2) */
void record_config_information(Int port_no, Config_bpdu *config); /* (8.6.2) */
void record_config_timeout_values(Config_bpdu *config); /* (8.6.3) */
void config_bpdu_generation(void); /* (8.6.4) */
STP_Boolean designated_port(Int port_no);
void reply(Int port_no); /* (8.6.5) */
void transmit_tcn(void); /* (8.6.6) */
void configuration_update(void); /* (8.6.7) */
void root_selection(void); /* (8.6.8) */
void designated_port_selection(void); /* (8.6.9) */
void become_designated_port(Int port_no); /* (8.6.10) */
void port_state_selection(void); /* (8.6.11) */
void make_forwarding(Int port_no); /* (8.6.12) */
void make_blocking(Int port_no); /* (8.6.13) */
void set_port_state(Int port_no, State state);
void topology_change_detection(void); /* (8.6.14) */
void topology_change_acknowledged(void); /* (8.6.15) */
void acknowledge_topology_change(Int port_no); /* (8.6.16) */
void received_config_bpdu(Int port_no, Config_bpdu *config); /* (8.7.1) */
void received_tcn_bpdu(Int port_no, Tcn_bpdu *tcn); /* (8.7.2) */
void hello_timer_expiry(void); /* (8.7.3) */
void message_age_timer_expiry(Int port_no); /* (8.7.4) */
void forward_delay_timer_expiry(Int port_no); /* (8.7.5) */
STP_Boolean designated_for_some_port(void);
void tcn_timer_expiry(void); /* (8.7.6) */
void topology_change_timer_expiry(void); /* (8.7.7) */
void hold_timer_expiry(Int port_no); /* (8.7.8) */
void initialisation(void); /* (8.8.1) */
void initialize_port(Int port_no);
void enable_port(Int port_no); /* (8.8.2) */
void disable_port(Int port_no); /* (8.8.3) */
void set_bridge_priority(Identifier new_bridge_id); /* (8.8.4) */
void set_port_priority(Int port_no, Port_id new_port_id); /* (8.8.5) */
void set_path_cost(Int port_no, Cost path_cost); /* (8.8.6) */
void enable_change_detection(Int port_no); /* (8.8.7) */
void disable_change_detection(Int port_no); /* (8.8.8) */
void STP_tick(void);
void start_hello_timer(void);
void stop_hello_timer(void);
STP_Boolean hello_timer_expired(void);
void start_tcn_timer(void);
void stop_tcn_timer(void);
STP_Boolean tcn_timer_expired(void);
void start_topology_change_timer(void);
void stop_topology_change_timer(void);
STP_Boolean topology_change_timer_expired(void);
void start_message_age_timer(Int port_no, Time message_age);
void stop_message_age_timer(Int port_no);
STP_Boolean message_age_timer_expired(Int port_no);
void start_forward_delay_timer(Int port_no);
void stop_forward_delay_timer(Int port_no);
STP_Boolean forward_delay_timer_expired(Int port_no);
void start_hold_timer(Int port_no);
void stop_hold_timer(Int port_no);
STP_Boolean hold_timer_expired(Int port_no);


void send_config_bpdu(Int port_no, Config_bpdu *bpdu);
/*
is a pseudo-implementation specific routine that transmits
the bpdu on the specified port within the specified time.
*/

void send_tcn_bpdu(Int port_no, Tcn_bpdu *bpdu);
/*
is a pseudo-implementation-specific routine that transmits
the bpdu on the specified port within the specified time.
*/


#endif /* IEEE802_1_H */
