/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************/

#include "ieee802_1d.h" 
#include "stp.h"

void CCLSendConfigBpdu(Int port_no, Config_bpdu *bpdu);
void CCLSendTcnBpdu(Int port_no, Tcn_bpdu *bpdu);
void CCLSetPortState(Int port_no, State state);


/*************************************************************************
*
* SPANNING TREE ALGORITHM AND PROTOCOL
*
**************************************************************************/



/***************************************************************************
* MACROS
**************************************************************************/

#define ID_CMP(a, b)  (id_cmp(&a, &b))


/***************************************************************************
* STATIC STORAGE ALLOCATION
**************************************************************************/

Bridge_data bridge_info; /* (8.5.3) */
Port_data port_info[All_ports]; /* (8.5.5) */
Config_bpdu config_bpdu[All_ports];
Tcn_bpdu tcn_bpdu[All_ports];
Timer hello_timer; /* (8.5.4.1) */
Timer tcn_timer; /* (8.5.4.2) */
Timer topology_change_timer; /* (8.5.4.3) */
Timer message_age_timer[All_ports]; /* (8.5.6.1) */
Timer forward_delay_timer[All_ports]; /* (8.5.6.2) */
Timer hold_timer[All_ports]; /* (8.5.6.3) */



/***************************************************************************
* CODE
**************************************************************************/


int id_cmp(Identifier *a, Identifier *b)
{    
    int i;  
    for (i=0; i<2; i++) 
    {
        if (a->four_octets[i] < b->four_octets[i])
            return(-1);
        if (a->four_octets[i] > b->four_octets[i])
            return(1);
    }
    return(0);
}
                 
            
/** Elements of Procedure (8.6) **/


void transmit_config(Int port_no) /* (8.6.1) */
{
#if 0
    if(port_info[port_no].state == Blocking) //ANVL test case 4.12
    {
     return;
    }
    else if (hold_timer[port_no].active) /* (8.6.1.3.1) */
#else
    if (hold_timer[port_no].active) /* (8.6.1.3.1) */
#endif
    {
        port_info[port_no].config_pending = True; /* (8.6.1.3.1) */
    }
    else /* (8.6.1.3.2) */
    {
//    printk("transmit_config in %d \n", port_no);
        config_bpdu[port_no].type = Config_bpdu_type;
        config_bpdu[port_no].root_id = bridge_info.designated_root;
        /* (8.6.1.3.2(a))*/
        config_bpdu[port_no].root_path_cost = bridge_info.root_path_cost;
        /* (8.6.1.3.2(b))*/
        config_bpdu[port_no].bridge_id = bridge_info.bridge_id;
        /* (8.6.1.3.2(c))*/
        config_bpdu[port_no].port_id = port_info[port_no].port_id;
        /* (8.6.1.3.2(d))*/
        if (root_bridge())
        {
            config_bpdu[port_no].message_age = Zero; /* (8.6.1.3.2(e))*/
        }
        else
        {
            config_bpdu[port_no].message_age
                = (Time) (message_age_timer[bridge_info.root_port].value
                          + Message_age_increment); /* (8.6.1.3.2(f))*/
        }
        config_bpdu[port_no].max_age = bridge_info.max_age; /* (8.6.1.3.2(g))*/
        config_bpdu[port_no].hello_time = bridge_info.hello_time;
        config_bpdu[port_no].forward_delay = bridge_info.forward_delay;
        config_bpdu[port_no].topology_change_acknowledgment
            = port_info[port_no].topology_change_acknowledge; /* (8.6.1.3.2(h)) */
        config_bpdu[port_no].topology_change
            = bridge_info.topology_change; /* (8.6.1.3.2(i)) */
        if (config_bpdu[port_no].message_age < bridge_info.max_age)
        {
            port_info[port_no].topology_change_acknowledge = False;  /* (8.6.1.3.3) */
            port_info[port_no].config_pending = False; /* (8.6.1.3.3)*/
            send_config_bpdu(port_no, &config_bpdu[port_no]);
            start_hold_timer(port_no); /* (8.6.3.3(b))*/
        }
    }
}

void send_config_bpdu(Int port_no, Config_bpdu *bpdu)
{
    /* Calling VIA driver code */
    CCLSendConfigBpdu(port_no, bpdu);
}


void send_tcn_bpdu(Int port_no, Tcn_bpdu *bpdu)
{
    /* Calling VIA driver code */
    CCLSendTcnBpdu(port_no, bpdu);
}

STP_Boolean root_bridge(void)
{
    return(ID_CMP(bridge_info.designated_root, bridge_info.bridge_id) == 0);
}

STP_Boolean supersedes_port_info(Int port_no, Config_bpdu *config) /* (8.6.2.2) */
{
    return (
             ( ID_CMP(config->root_id,
                      port_info[port_no].designated_root) < 0 /* (8.6.2.2 a) */
             )
             ||
             ( ( ID_CMP( config->root_id,
                         port_info[port_no].designated_root) == 0 
               )
               &&
               ( ( config->root_path_cost
                   < port_info[port_no].designated_cost /* (8.6.2.2 b) */
                 )
                 ||
                 ( ( config->root_path_cost
                     == port_info[port_no].designated_cost 
                   )
                   &&
                   ( ( ID_CMP(config->bridge_id,
                              port_info[port_no].designated_bridge) < 0 /* (8.6.2.2 c) */
                     )
                     ||
                     ( ( ID_CMP(config->bridge_id,
                                  port_info[port_no].designated_bridge) == 0
                       ) /* (8.6.2.2 d) */
                       &&
                       ( ( ID_CMP(config->bridge_id,bridge_info.bridge_id) != 0
                         ) /* (8.6.2.2 d1) */
                         ||
                         ( config->port_id
                           <= port_info[port_no].designated_port
                         ) /* (8.6.2.2 d2) */
                       ) 
                     ) 
                   ) 
                 ) 
               ) 
             )
           );
}

void record_config_information(Int port_no, Config_bpdu *config) /* (8.6.2) */
{
    port_info[port_no].designated_root = config->root_id; /* (8.6.2.3.1) */
    port_info[port_no].designated_cost = config->root_path_cost;
    port_info[port_no].designated_bridge = config->bridge_id;
    port_info[port_no].designated_port = config->port_id;
    start_message_age_timer(port_no, config->message_age); /* (8.6.2.3.2) */
}

void record_config_timeout_values(Config_bpdu *config) /* (8.6.3) */
{
    bridge_info.max_age = config->max_age; /* (8.6.3.3) */
    bridge_info.hello_time = config->hello_time;
    bridge_info.forward_delay = config->forward_delay;
    bridge_info.topology_change = config->topology_change;
}

void config_bpdu_generation(void) /* (8.6.4) */
{
    Int port_no;

//    printk("No_of_ports = %d \n", No_of_ports);
    for (port_no = One; port_no <= No_of_ports; port_no++) /* (8.6.4.3) */
    {
        if ( designated_port(port_no) /* (8.6.4.3) */
             &&
             (port_info[port_no].state != Disabled))
        {
//    printk("transmit_config %d \n", port_no);
            transmit_config(port_no); /* (8.6.4.3) */
        } /* (8.6.1.2) */
    }
}

/* where */
STP_Boolean designated_port(Int port_no)
{
    return ( ( ID_CMP(port_info[port_no].designated_bridge, bridge_info.bridge_id) == 0
             )
             &&
             ( port_info[port_no].designated_port
               == port_info[port_no].port_id
             )
           );
}

void reply(Int port_no) /* (8.6.5) */
{
    transmit_config(port_no); /* (8.6.5.3) */
}

void transmit_tcn(void) /* (8.6.6) */
{
    Int port_no;
    port_no = bridge_info.root_port;
    tcn_bpdu[port_no].type = Tcn_bpdu_type;
    send_tcn_bpdu(port_no, &tcn_bpdu[bridge_info.root_port]);/* (8.6.6.3) */
}


/* where
send_tcn_bpdu(port_no,bpdu)
Int port_no;
Tcn_bpdu *bpdu;
is a pseudo-implementation-specific routine that transmits
the bpdu on the specified port within the specified time.
*/

void configuration_update(void) /* (8.6.7) */
{
    root_selection(); /* (8.6.7.3.1) */
    /* (8.6.8.2) */
    designated_port_selection(); /* (8.6.7.3.2) */
    /* (8.6.9.2) */
}

void root_selection(void) /* (8.6.8) */
{
    Int root_port;
    Int port_no;
    root_port = No_port;
    for (port_no = One; port_no <= No_of_ports; port_no++) /* (8.6.8.3.1) */
    {
        if ( ( (!designated_port(port_no))
               &&
               (port_info[port_no].state != Disabled)
               &&
               (ID_CMP(port_info[port_no].designated_root, bridge_info.bridge_id) < 0)
             )
             &&
             ( (root_port == No_port)
               ||
               ( ID_CMP(port_info[port_no].designated_root, 
                        port_info[root_port].designated_root) < 0 /* (8.6.8.3.1(a)) */
               )
               ||
               ( ( ID_CMP(port_info[port_no].designated_root,
                          port_info[root_port].designated_root) == 0
                 )
                 &&
                 ( ( ( port_info[port_no].designated_cost
                       + port_info[port_no].path_cost
                     )
                     <
                     ( port_info[root_port].designated_cost
                       + port_info[root_port].path_cost
                     ) /* (8.6.8.3.1(b)) */
                   )
                   ||
                   ( ( ( port_info[port_no].designated_cost
                         + port_info[port_no].path_cost
                       )
                       ==
                       ( port_info[root_port].designated_cost
                         + port_info[root_port].path_cost
                       )
                     )
                     &&
                     ( ( ID_CMP(port_info[port_no].designated_bridge,
                                port_info[root_port].designated_bridge) < 0
                       ) /* (8.6.8.3.1(c)) */
                       ||
                       ( ( ID_CMP(port_info[port_no].designated_bridge,
                                  port_info[root_port].designated_bridge) == 0
                         )
                         &&
                         ( ( port_info[port_no].designated_port
                             < port_info[root_port].designated_port
                           ) /* (8.6.8.3.1(d)) */
                           ||
                           ( ( port_info[port_no].designated_port
                               == port_info[root_port].designated_port
                             )
                             &&
                             ( port_info[port_no].port_id
                               < port_info[root_port].port_id
                             ) /* (8.6.8.3.1(e)) */
                           ) 
                         ) 
                       ) 
                     ) 
                   ) 
                 ) 
               ) 
             ) 
           )
        {
            root_port = port_no;
        }
    }
    bridge_info.root_port = root_port; /* (8.6.8.3.1) */
    if (root_port == No_port) /* (8.6.8.3.2) */
    {
        bridge_info.designated_root = bridge_info.bridge_id;
        /* (8.6.8.3.2(a)) */
        bridge_info.root_path_cost = Zero; /* (8.6.8.3.2(b)) */
    }
    else /* (8.6.8.3.3) */
    {
        bridge_info.designated_root = port_info[root_port].designated_root;
        /* (8.6.8.3.3(a)) */
        bridge_info.root_path_cost = ( port_info[root_port].designated_cost
                     + port_info[root_port].path_cost); /* (8.6.8.3.3(b)) */
    }
}

void designated_port_selection(void) /* (8.6.9) */
{
    Int port_no;
    for (port_no = One; port_no <= No_of_ports; port_no++) /* (8.6.9.3) */
    {
        if ( designated_port(port_no) /* (8.6.9.3 a) */
             ||
             (
               ID_CMP(port_info[port_no].designated_root,
                      bridge_info.designated_root) != 0 /* (8.6.9.3 b) */
             )
             ||
             ( bridge_info.root_path_cost
               < port_info[port_no].designated_cost
             ) /* (8.6.9.3 c) */
             ||
             ( ( bridge_info.root_path_cost
                 == port_info[port_no].designated_cost
               )
               &&
               ( ( ID_CMP(bridge_info.bridge_id,
                          port_info[port_no].designated_bridge) < 0
                 ) /* (8.6.9.3 d) */
                 ||
                 ( ( ID_CMP(bridge_info.bridge_id,
                     port_info[port_no].designated_bridge) == 0
                   )
                   &&
                   ( port_info[port_no].port_id
                     <= port_info[port_no].designated_port
                   ) /* (8.6.9.3 e) */
                 ) 
               ) 
             ) 
           )
        {
            become_designated_port(port_no); /* (8.6.10.2 a) */
        }   
    }
}

void become_designated_port(Int port_no) /* (8.6.10) */
{
    port_info[port_no].designated_root = bridge_info.designated_root;
    /* (8.6.10.3 a) */
    port_info[port_no].designated_cost = bridge_info.root_path_cost;
    /* (8.6.10.3 b) */
    port_info[port_no].designated_bridge = bridge_info.bridge_id;
    /* (8.6.10.3 c) */
    port_info[port_no].designated_port = port_info[port_no].port_id;
    /* (8.6.10.3 d) */
}

void port_state_selection(void) /* (8.6.11) */
{
    Int port_no;
    for (port_no = One; port_no <= No_of_ports; port_no++)
    {
        if (port_no == bridge_info.root_port) /* (8.6.11.3 a) */
        {
            port_info[port_no].config_pending = False; /* (8.6.11.3 a1)*/
            port_info[port_no].topology_change_acknowledge = False;
            make_forwarding(port_no); /* (8.6.11.3 a2)*/
        }
        else if (designated_port(port_no)) /* (8.6.11.3 b) */
        {
            stop_message_age_timer(port_no); /* (8.6.11.3 b1)*/
            make_forwarding(port_no); /* (8.6.11.3 b2)*/
        }
        else /* (8.6.11.3 c) */
        {   
            port_info[port_no].config_pending = False; /* (8.6.11.3 c1)*/
            port_info[port_no].topology_change_acknowledge = False;
            make_blocking(port_no); /* (8.6.11.3 c2)*/
        }
    }
}

void make_forwarding(Int port_no) /* (8.6.12) */
{
    if (port_info[port_no].state == Blocking) /* (8.6.12.3) */
    {
        set_port_state(port_no, Listening); /* (8.6.12.3 a) */
        start_forward_delay_timer(port_no); /* (8.6.12.3 b) */
    }
}

void make_blocking(Int port_no) /* (8.6.13) */
{
    if ( (port_info[port_no].state != Disabled)
         &&
         (port_info[port_no].state != Blocking)
       ) /* (8.6.13.3) */
    {
        if ( (port_info[port_no].state == Forwarding)
             ||
             (port_info[port_no].state == Learning)
           )
        {
            if (port_info[port_no].change_detection_enabled == True)
                /* (8.5.5.10) */
            {
                topology_change_detection(); /* (8.6.13.3 a) */
            } /* (8.6.14.2.3) */
        }
        set_port_state(port_no, Blocking); /* (8.6.13.3 b) */
        stop_forward_delay_timer(port_no); /* (8.6.13.3 c) */
    }
}

/* where */
void set_port_state(Int port_no, State state)
{
    /* Calling VIA driver code for setting the port state. */
    CCLSetPortState(port_no, state);

    port_info[port_no].state = state;
}

void topology_change_detection(void) /* (8.6.14) */
{
    if (root_bridge()) /* (8.6.14.3 a) */
    {
        bridge_info.topology_change = True; /* (8.6.14.3 a1)*/
        start_topology_change_timer(); /* (8.6.14.3 a2)*/
    }
    else if (bridge_info.topology_change_detected == False) /* (8.6.14.3 b) */
    {
        transmit_tcn(); /* (8.6.14.3 b1)*/
        start_tcn_timer(); /* (8.6.14.3 b2)*/
    }
    bridge_info.topology_change_detected = True; /* (8.6.14.3 c) */
}

void topology_change_acknowledged(void) /* (8.6.15) */
{
    bridge_info.topology_change_detected = False; /* (8.6.15.3 a) */
    stop_tcn_timer(); /* (8.6.15.3 b) */
}

void acknowledge_topology_change(Int port_no) /* (8.6.16) */
{
    port_info[port_no].topology_change_acknowledge = True; /* (8.6.16.3 a) */
    transmit_config(port_no); /* (8.6.16.3 b) */
}

/** Operation of the Protocol (8.7) **/
void received_config_bpdu(Int port_no, Config_bpdu * config) /* (8.7.1) */
{
    STP_Boolean root;
    root = root_bridge();
    if (port_info[port_no].state != Disabled)
    {
//     printk("Port %d is not disabled %08X %08X, %08X %08X\n",port_no,config->root_id.four_octets[0],config->root_id.four_octets[1],port_info[port_no].designated_root.four_octets[0],port_info[port_no].designated_root.four_octets[1]);
        if (supersedes_port_info(port_no, config)) /* (8.7.1.1) */
        { /* (8.6.2.2) */
//     printk("Port %d is not disabled and supersedes\n",port_no);
            record_config_information(port_no, config); /* (8.7.1.1 a) */
//    printk("designated_root %d\n",port_info[port_no].designated_root);
//    printk("designated_cost %d\n",port_info[port_no].designated_cost);
//    printk("designated_bridge %d\n",port_info[port_no].designated_bridge); 
//    printk("designated_port %d\n",port_info[port_no].designated_port); 
            /* (8.6.2.2) */
            configuration_update(); /* (8.7.1.1 b) */
            /* (8.6.7.2 a) */
            port_state_selection(); /* (8.7.1.1 c) */
            /* (8.6.11.2 a) */
            if ((!root_bridge()) && root) /* (8.7.1.1 d) */
            {
                stop_hello_timer();
                if (bridge_info.topology_change_detected) /* (8.7.1.1 e) */
                {
                    stop_topology_change_timer();
                    transmit_tcn(); /* (8.6.6.1) */
                    start_tcn_timer();
                }
            }
            if (port_no == bridge_info.root_port)
            {
                record_config_timeout_values(config); /* (8.7.1.1 e) */
                /* (8.6.3.2) */
                config_bpdu_generation(); /* (8.6.4.2 a) */
                if (config->topology_change_acknowledgment) /* (8.7.1.1 g) */
                {
                    topology_change_acknowledged(); /* (8.6.15.2) */
                }
            }
        }
        else if (designated_port(port_no)) /* (8.7.1.2) */
        {
//     printk("Port %d is not disabled and not supersedes and designed port\n",port_no);
            reply(port_no); /* (8.7.1.2) */
            /* (8.6.5.2) */
        }
    }
}

void received_tcn_bpdu(Int port_no, Tcn_bpdu *tcn) /* (8.7.2) */
{
    if (port_info[port_no].state != Disabled)
    {
        if (designated_port(port_no))
        {
            topology_change_detection(); /* (8.7.2 a) */
            /* (8.6.14.2.1) */
            acknowledge_topology_change(port_no); /* (8.7.2 b) */
        } /* (8.6.16.2) */
    }
}

void hello_timer_expiry(void) /* (8.7.3) */
{
    config_bpdu_generation(); /* (8.6.4.2 b) */
    start_hello_timer();
}

void message_age_timer_expiry(Int port_no) /* (8.7.4) */
{
    STP_Boolean root;
    root = root_bridge();
    become_designated_port(port_no); /* (8.7.4 a) */
    /* (8.6.10.2 b) */
    configuration_update(); /* (8.7.4 b) */
    /* (8.6.7.2 b) */
    port_state_selection(); /* (8.7.4 c) */
    /* (8.6.11.2 b) */
    if ((root_bridge()) && (!root)) /* (8.7.4 d) */
    {
        bridge_info.max_age = bridge_info.bridge_max_age; /* (8.7.4 d1) */
        bridge_info.hello_time = bridge_info.bridge_hello_time;
        bridge_info.forward_delay = bridge_info.bridge_forward_delay;
        topology_change_detection(); /* (8.7.4 d2) */
        /* (8.6.14.2.4) */
        stop_tcn_timer(); /* (8.7.4 d3) */
        config_bpdu_generation(); /* (8.7.4 d4) */
        start_hello_timer();
    }
}

void forward_delay_timer_expiry(Int port_no) /* (8.7.5) */
{
    if (port_info[port_no].state == Listening) /* (8.7.5 a) */
    {
        set_port_state(port_no, Learning); /* (8.7.5 a1) */
        start_forward_delay_timer(port_no); /* (8.7.5 a2) */
    }
    else if (port_info[port_no].state == Learning) /* (8.7.5 b) */
    {
        set_port_state(port_no, Forwarding); /* (8.7.5 b1) */
        if (designated_for_some_port()) /* (8.7.5 b2) */
        {
            if (port_info[port_no].change_detection_enabled == True)
            /* (8.5.5.10) */
            {
                topology_change_detection(); /* (8.6.14.2.2) */
            }
        }   
    }
}

/* where */
STP_Boolean designated_for_some_port(void)
{
    Int port_no;
    for (port_no = One; port_no <= No_of_ports; port_no++)
    {
        if ( ID_CMP(port_info[port_no].designated_bridge,
                    bridge_info.bridge_id) == 0
           )
        {
            return(True);
        }
    }
    return(False);
}

void tcn_timer_expiry(void) /* (8.7.6) */
{
    transmit_tcn(); /* (8.7.6 a) */
    start_tcn_timer(); /* (8.7.6 b) */
}

void topology_change_timer_expiry(void) /* (8.7.7) */
{
    bridge_info.topology_change_detected = False; /* (8.7.7 a) */
    bridge_info.topology_change = False; /* (8.7.7 b) */
}

void hold_timer_expiry(Int port_no) /* (8.7.8) */
{
    if (port_info[port_no].config_pending)
    {
        transmit_config(port_no); /* (8.6.1.2) */
    }
}

/** Management of the Bridge Protocol Entity (8.8) **/
void initialisation(void) /* (8.8.1) */
{
    Int port_no;
    bridge_info.designated_root = bridge_info.bridge_id; /* (8.8.1 a) */
    bridge_info.root_path_cost = Zero;
    bridge_info.root_port = No_port;
    bridge_info.max_age = bridge_info.bridge_max_age; /* (8.8.1 b) */
    bridge_info.hello_time = bridge_info.bridge_hello_time;
    bridge_info.forward_delay = bridge_info.bridge_forward_delay;
    bridge_info.topology_change_detected = False; /* (8.8.1 c) */
    bridge_info.topology_change = False;
    stop_tcn_timer();
    stop_topology_change_timer();
    for (port_no = One; port_no <= No_of_ports; port_no++) /* (8.8.1 d) */
    {
        initialize_port(port_no);
    }
    port_state_selection(); /* (8.8.1 e) */
    config_bpdu_generation(); /* (8.8.1 f) */
    start_hello_timer();
}

/*
*/
void initialize_port(Int port_no)
{
    become_designated_port(port_no); /* (8.8.1 d1) */
    set_port_state(port_no, Blocking); /* (8.8.1 d2) */
    port_info[port_no].topology_change_acknowledge = False;
    /* (8.8.1 d3) */
    port_info[port_no].config_pending = False; /* (8.8.1 d4) */
    port_info[port_no].change_detection_enabled = True; /* (8.8.1 d8) */
    port_info[port_no].path_cost = DFLT_PATH_COST; /* (8.8.6 a) */
    stop_message_age_timer(port_no); /* (8.8.1 d5) */
    stop_forward_delay_timer(port_no); /* (8.8.1 d6) */
    stop_hold_timer(port_no); /* (8.8.1 d7) */
}

void enable_port(Int port_no) /* (8.8.2) */
{
    initialize_port(port_no);
    port_state_selection(); /* (8.8.2 g) */
}

/*
*/
void disable_port(Int port_no) /* (8.8.3) */
{
    STP_Boolean root;
    root = root_bridge();
    become_designated_port(port_no); /* (8.8.3 a) */
    set_port_state(port_no, Disabled); /* (8.8.3 b) */
    port_info[port_no].topology_change_acknowledge = False; /* (8.8.3 c) */
    port_info[port_no].config_pending = False; /* (8.8.3 d) */
    port_info[port_no].path_cost = DFLT_PATH_COST; /* (8.8.3 d) */
    stop_message_age_timer(port_no); /* (8.8.3 e) */
    stop_forward_delay_timer(port_no); /* (8.8.3 f) */
    configuration_update(); /* (8.8.3 g) */
    port_state_selection(); /* (8.8.3 h) */
    if ((root_bridge()) && (!root)) /* (8.8.3 i) */
    {
        bridge_info.max_age = bridge_info.bridge_max_age; /* (8.8.3 i1) */
        bridge_info.hello_time = bridge_info.bridge_hello_time;
        bridge_info.forward_delay = bridge_info.bridge_forward_delay;
        topology_change_detection(); /* (8.8.3 i2) */
        stop_tcn_timer(); /* (8.8.3 i3) */
        config_bpdu_generation(); /* (8.8.3 i4) */
        start_hello_timer();
    }
}

void set_bridge_priority(Identifier new_bridge_id) /* (8.8.4) */
{
    STP_Boolean root;
    Int port_no;
    root = root_bridge();
    for (port_no = One; port_no <= No_of_ports; port_no++) /* (8.8.4 b) */
    {
        if (designated_port(port_no))
        {
            port_info[port_no].designated_bridge = new_bridge_id;
        }
    }
    bridge_info.bridge_id = new_bridge_id; /* (8.8.4 c) */
    configuration_update(); /* (8.8.4 d) */
    port_state_selection(); /* (8.8.4 e) */
    if ((root_bridge()) && (!root)) /* (8.8.4 f) */
    {
        bridge_info.max_age = bridge_info.bridge_max_age; /* (8.8.4 f1) */
        bridge_info.hello_time = bridge_info.bridge_hello_time;
        bridge_info.forward_delay = bridge_info.bridge_forward_delay;
        topology_change_detection(); /* (8.8.4 f2) */
        stop_tcn_timer(); /* (8.8.4 f3) */
        config_bpdu_generation(); /* (8.8.4 f4) */
        start_hello_timer();
    }
}

void set_port_priority(Int port_no, Port_id new_port_id) /* (8.8.5) */
{
    if (designated_port(port_no)) /* (8.8.5 b) */
    {
        port_info[port_no].designated_port = new_port_id;
    }
    port_info[port_no].port_id = new_port_id; /* (8.8.5 c) */
    if ( ( ID_CMP(bridge_info.bridge_id, /* (8.8.5 d) */
                  port_info[port_no].designated_bridge) == 0
         )
         &&
         ( port_info[port_no].port_id
           < port_info[port_no].designated_port
         )
       )
    {
        become_designated_port(port_no); /* (8.8.5 d1) */
        port_state_selection(); /* (8.8.5 d2) */
    }
}

/*
*/
void set_path_cost(Int port_no, Cost path_cost) /* (8.8.6) */
{
    port_info[port_no].path_cost = path_cost; /* (8.8.6 a) */
    configuration_update(); /* (8.8.6 b) */
    port_state_selection(); /* (8.8.6 c) */
}

/*
*/
void enable_change_detection(Int port_no) /* (8.8.7) */
{
    port_info[port_no].change_detection_enabled = True;
}

/*
*/
void disable_change_detection(Int port_no) /* (8.8.8) */
{
    port_info[port_no].change_detection_enabled = False;
}


/** pseudo-implementation-specific timer running support **/
void STP_tick(void)
{
    Int port_no;
    if (hello_timer_expired())
    { 
        hello_timer_expiry();
    }
    if (tcn_timer_expired())
    { 
        tcn_timer_expiry();
    }
    if (topology_change_timer_expired())
    { 
        topology_change_timer_expiry();
    }
    for (port_no = One; port_no <= No_of_ports; port_no++)
    {
        if (message_age_timer_expired(port_no))
        {
            message_age_timer_expiry(port_no);
        }
    }
    for (port_no = One; port_no <= No_of_ports; port_no++)
    {
        if (forward_delay_timer_expired(port_no))
        {
            forward_delay_timer_expiry(port_no);
        }
        if (hold_timer_expired(port_no))
        {
            hold_timer_expiry(port_no);
        }
    }
}

/* where */
void start_hello_timer(void)
{ 
    hello_timer.value = (Time) Zero;
    hello_timer.active = True;
}

void stop_hello_timer(void)
{ 
    hello_timer.active = False;
}

STP_Boolean hello_timer_expired(void)
{ 
    if (hello_timer.active)
    {
        hello_timer.value += ulSpanTreeTickScaling;

        if (hello_timer.value >= bridge_info.hello_time)
        { 
            hello_timer.active = False;
            return(True);
        }
    }
    return(False);
}

void start_tcn_timer(void)
{ 
    tcn_timer.value = (Time) Zero;
    tcn_timer.active = True;
}

void stop_tcn_timer(void)
{ 
    tcn_timer.active = False;
}

STP_Boolean tcn_timer_expired(void)
{ 
    if (tcn_timer.active)
    {
        tcn_timer.value += ulSpanTreeTickScaling;

        if (tcn_timer.value >= bridge_info.bridge_hello_time)
        { 
            tcn_timer.active = False;
            return(True);
        }
    }
    return(False);
}

void start_topology_change_timer(void)
{ 
    topology_change_timer.value = (Time) Zero;
    topology_change_timer.active = True;
}

void stop_topology_change_timer(void)
{ 
    topology_change_timer.active = False;
}

STP_Boolean topology_change_timer_expired(void)
{ 
    if ( topology_change_timer.active)
    {
        topology_change_timer.value += ulSpanTreeTickScaling;

        if ( ++topology_change_timer.value
            >= bridge_info.topology_change_time)
        { 
            topology_change_timer.active = False;
            return(True);
        }
    }
    return(False);
}

void start_message_age_timer(Int port_no, Time message_age)
{ 
    message_age_timer[port_no].value = message_age;
    message_age_timer[port_no].active = True;
}

void stop_message_age_timer(Int port_no)
{ 
    message_age_timer[port_no].active = False;
}

STP_Boolean message_age_timer_expired(Int port_no)
{ 
    if (message_age_timer[port_no].active)
    {
        message_age_timer[port_no].value += ulSpanTreeTickScaling;
        if (message_age_timer[port_no].value >= bridge_info.max_age)
        { 
            message_age_timer[port_no].active = False;
            return(True);
        }
    }
    return(False);
}

void start_forward_delay_timer(Int port_no)
{ 
    forward_delay_timer[port_no].value = Zero;
    forward_delay_timer[port_no].active = True;
}

void stop_forward_delay_timer(Int port_no)
{ 
    forward_delay_timer[port_no].active = False;
}

STP_Boolean forward_delay_timer_expired(Int port_no)
{ 
    if (forward_delay_timer[port_no].active)
    {
        forward_delay_timer[port_no].value += ulSpanTreeTickScaling;
        
        if (forward_delay_timer[port_no].value >= bridge_info.forward_delay)
        { 
            forward_delay_timer[port_no].active = False;
            return(True);
        }
    }
    return(False);
}

void start_hold_timer(Int port_no)
{ 
    hold_timer[port_no].value = Zero;
    hold_timer[port_no].active = True;
}

void stop_hold_timer(Int port_no)
{ 
    hold_timer[port_no].active = False;
}

STP_Boolean hold_timer_expired(Int port_no)
{ 
    if (hold_timer[port_no].active)
    {
        hold_timer[port_no].value += ulSpanTreeTickScaling;
        if (hold_timer[port_no].value >= bridge_info.hold_time)
        { 
            hold_timer[port_no].active = False;
            return(True);
        }
    }
    return(False);
}

/** pseudo-implementation specific transmit routines **/
// TODO: #include "transmit.c"
