/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Topolgy Change state machine : 17.25 */

#include "krnportmask.h"

#include "base.h"
#include "stpm.h"
#include "stp_to.h"		/* for STP_OUT_flush_lt */

#define STATES { \
  CHOOSE(INIT),             \
  CHOOSE(INACTIVE),         \
  CHOOSE(TCACTIVE),         \
  CHOOSE(DETECTED),         \
  CHOOSE(NOTIFIED_TC),      \
  CHOOSE(PROPAGATING),      \
  CHOOSE(ACKNOWLEDGED),     \
  CHOOSE(NOTIFIED_TCN),     \
  CHOOSE(TCFLUSH),          \
}

#define GET_STATE_NAME STP_topoch_get_state_name
#include "choose.h"

static Bool
flush (STATE_MACH_T * this, char *reason)
{				/* 17.19.9 */
  register PORT_T *port = this->owner.port;

  if (port->operEdge)
    return False;
  if (!port->adminEnable) {	/* we have just flushed it in stp_in_enable_port */
    return False;
  }
#if STP_DBG
  if (this->debug || port->sttrans->debug) {
    stp_trace ("%s (%s, %s, %s, '%s')",
	       "flush", port->port_name, port->owner->name, "this port",
	       reason);
  }
#endif

  STP_OUT_flush_single_lt (port->port_index, reason);
  return True;
}
#ifndef STRONGLY_SPEC_802_1W
/* 
 * In many kinds of hardware the function
 * STP_OUT_flush_lt is a) is very hard and b) cannot
 * delete learning emtries per port. The alternate
 * method may be used: we don't care operEdge flag here,
 * but clean learning table once for TopologyChange
 * for all ports, except the received port. I am ready to discuss :(
 * See below word STRONGLY_SPEC_802_1W
 */
#endif

static void
setTcPropBridge (STATE_MACH_T * this, char *reason)
{				/* 17.19.14 */
  register PORT_T *port = this->owner.port;
  register PORT_T *tmp;
#ifndef STRONGLY_SPEC_802_1W
  TstLPortMask ports_to_flush;

  K_LPortMaskCopy (&ports_to_flush, port->owner->portmap);
  K_LPortMaskClrPort (&ports_to_flush, port->port_index - 1);
#endif

  for (tmp = port->owner->ports; tmp; tmp = tmp->next) {
    if (tmp->port_index != port->port_index) {
      tmp->tcProp = True;
    }
#ifndef STRONGLY_SPEC_802_1W
    if (tmp->operEdge) {
      K_LPortMaskClrPort (&ports_to_flush, tmp->port_index - 1);
    }
#endif
  }

#ifndef STRONGLY_SPEC_802_1W

#if STP_DBG
  if (this->debug || port->sttrans->debug /* || port->edge->debug */ ) {
    stp_trace ("%s (%s, %s, %s, '%s')", "clearFDB", port->port_name,
	       port->owner->name, "other ports", reason);
  }
#endif

  if (!K_LPortMaskIsZero (&ports_to_flush)) {
    STP_OUT_flush_lt (&ports_to_flush, reason);
  }
#endif
}

static unsigned int
newTcWhile (STATE_MACH_T * this, char *reason)
{				/* 17.19.7 */
  register PORT_T *port = this->owner.port;
  unsigned long newValue;

  if (port->sendRSTP && port->operPointToPointMac) {
    newValue = 2 * port->owner->rootTimes.HelloTime;
  } else {
    newValue = port->owner->rootTimes.MaxAge;
  }
#if 0
  if (port->edge->debug)
    stp_trace ("newTcWhile(%s) for port %s returns %ld",
	       reason, port->port_name, newValue);
#endif
  return newValue;
}

void
STP_topoch_enter_state (STATE_MACH_T * this)
{
  register PORT_T *port = this->owner.port;

  switch (this->State) {
    case BEGIN:
    case INIT:
#ifdef STRONGLY_SPEC_802_1W
    flush (this, "topoch INIT");
#endif
    port->tcWhile = 0;
    port->tc = port->tcProp = port->tcAck = False;
    break;
    case INACTIVE:
    port->rcvdTc =
      port->rcvdTcn = port->rcvdTcAck = port->tc = port->tcProp = False;
    break;
    case TCACTIVE:
    break;
    case DETECTED:
    port->tcWhile = newTcWhile (this, "DETECTED");
    setTcPropBridge (this, "DETECTED");
    port->tc = False;
    port->newInfo = True;
    break;
    case NOTIFIED_TC:
    port->rcvdTcn = port->rcvdTc = False;
    if (port->role == DesignatedPort) {
      port->tcAck = True;
    }
    setTcPropBridge (this, "NOTIFIED_TC");
    break;
    case PROPAGATING:
    port->tcWhile = newTcWhile (this, "PROPAGATING");
#ifdef STRONGLY_SPEC_802_1W
    flush (this, "topoch PROPAGATING");
#endif
    port->tcProp = False;
    port->newInfo = True;
    break;
    case ACKNOWLEDGED:
    port->tcWhile = 0;
    port->rcvdTcAck = False;
    break;
    case NOTIFIED_TCN:
    port->tcWhile = newTcWhile (this, "PROPAGATING");
    break;
    case TCFLUSH:
#ifndef STRONGLY_SPEC_802_1W
    flush (this, "TCFLUSH");
#endif
    break;
  };
}

Bool
STP_topoch_check_conditions (STATE_MACH_T * this)
{
  register PORT_T *port = this->owner.port;

  if (BEGIN == this->State) {
    return STP_hop_2_state (this, INIT);
  }

  switch (this->State) {
    case INIT:
    return STP_hop_2_state (this, INACTIVE);
    case INACTIVE:
    if (port->role == RootPort || port->role == DesignatedPort)
      return STP_hop_2_state (this, TCACTIVE);
    if (port->rcvdTc || port->rcvdTcn || port->rcvdTcAck ||
	port->tc || port->tcProp) return STP_hop_2_state (this, INACTIVE);
    break;
    case TCACTIVE:
    if (port->role != RootPort && (port->role != DesignatedPort))
#ifdef STRONGLY_SPEC_802_1W
      return STP_hop_2_state (this, INIT);
#else
      return STP_hop_2_state (this, TCFLUSH);
#endif

    if (port->tc)
      return STP_hop_2_state (this, DETECTED);
    if (port->rcvdTcn)
      return STP_hop_2_state (this, NOTIFIED_TCN);
    if (port->rcvdTc)
      return STP_hop_2_state (this, NOTIFIED_TC);
    if (port->tcProp && !port->operEdge)
      return STP_hop_2_state (this, PROPAGATING);
    if (port->rcvdTcAck)
      return STP_hop_2_state (this, ACKNOWLEDGED);
    break;
    case DETECTED:
    return STP_hop_2_state (this, TCACTIVE);
    case NOTIFIED_TC:
    return STP_hop_2_state (this, TCACTIVE);
    case PROPAGATING:
    return STP_hop_2_state (this, TCACTIVE);
    case ACKNOWLEDGED:
    return STP_hop_2_state (this, TCACTIVE);
    case NOTIFIED_TCN:
    return STP_hop_2_state (this, NOTIFIED_TC);
    case TCFLUSH:
    return STP_hop_2_state (this, INIT);
  };
  return False;
}
