/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Port State Transition state machine : 17.24 */

#include "base.h"
#include "stpm.h"
#include "stp_to.h"

#define STATES { \
  CHOOSE(DISCARDING),   \
  CHOOSE(LEARNING), \
  CHOOSE(FORWARDING),   \
}

#define GET_STATE_NAME STP_sttrans_get_state_name
#include "choose.h"


#ifdef STRONGLY_SPEC_802_1W
static Bool
disableLearning (STATE_MACH_T * this)
{
  register PORT_T *port = this->owner.port;

  return STP_OUT_set_learning (port->port_index, False);
}

static Bool
enableLearning (STATE_MACH_T * this)
{
  register PORT_T *port = this->owner.port;

  return STP_OUT_set_learning (port->port_index, True);
}

static Bool
disableForwarding (STATE_MACH_T * this)
{
  register PORT_T *port = this->owner.port;

  return STP_OUT_set_forwarding (port->port_index, False);
}

static Bool
enableForwarding (STATE_MACH_T * this)
{
  register PORT_T *port = this->owner.port;

  return STP_OUT_set_forwarding (port->port_index, True);
}
#endif

void
STP_sttrans_enter_state (STATE_MACH_T * this)
{
  register PORT_T *port = this->owner.port;

  switch (this->State) {
    case BEGIN:
    case DISCARDING:
    port->learning = False;
    port->forwarding = False;
#ifdef STRONGLY_SPEC_802_1W
    disableLearning (this);
    disableForwarding (this);
#else
 #ifdef STP_DBG
    if (port->p2p->debug)
      stp_trace ("port %s => Discard !!!!!!!!!!!!!!!!!!!!!!!!!",
		 port->port_name);
 #endif
    STP_OUT_set_port_state (port->port_index, UID_PORT_DISCARDING);
#endif
    break;
    case LEARNING:
    port->learning = True;
#ifdef STRONGLY_SPEC_802_1W
    enableLearning (this);
#else
    STP_OUT_set_port_state (port->port_index, UID_PORT_LEARNING);
#endif
    break;
    case FORWARDING:
    port->tc = !port->operEdge;
    port->forwarding = True;
#ifdef STRONGLY_SPEC_802_1W
    enableForwarding (this);
#else
    port->proposing = False;
    /*
     * winfred: not in standard, but a bug: If it is in RSTP mode and p2p is
     * false, this port will not stop sending bpdu with proposal flag set.
     * The reson is that rcvBpdu() will return OtherMsg, and port information
     * state machine will never enter AGREEMENT state.
     * No further proposal is needed after this port transits to forwarding.
     */
 #ifdef STP_DBG
    if (port->p2p->debug || port->edge->debug) {
      stp_trace ("port %s => Forward !!!!!!!!!!!!!!!!!!!!!!!!! tc=%d",
		 port->port_name, (int) port->tc);
    }
 #endif
    STP_OUT_set_port_state (port->port_index, UID_PORT_FORWARDING);
#endif
    break;
  }

}

Bool
STP_sttrans_check_conditions (STATE_MACH_T * this)
{
  register PORT_T *port = this->owner.port;

  if (BEGIN == this->State) {
    return STP_hop_2_state (this, DISCARDING);
  }

  switch (this->State) {
    case DISCARDING:
    if (port->learn) {
      return STP_hop_2_state (this, LEARNING);
    }
    break;
    case LEARNING:
    if (port->forward) {
      port->snmp_topo_change = !port->operEdge;
      return STP_hop_2_state (this, FORWARDING);
    }
    if (!port->learn) {
      return STP_hop_2_state (this, DISCARDING);
    }
    break;
    case FORWARDING:
    if (!port->forward) {
      port->snmp_topo_change = !port->operEdge;
      return STP_hop_2_state (this, DISCARDING);
    }
    break;
  }

  return False;
}
