/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* This file contains system dependent API
   from the RStp to a operation system (see stp_to.h) */

/* stp_to API for Linux */

#include <stdarg.h>
#include <linux/spinlock.h>

#include "krnmac.h"
#include "krnportmask.h"

#include "base.h"
#include "stpm.h"
#include "stp_in.h"
#include "stp_to.h"

static spinlock_t rstp_OUT_inner_sem;
Bool   rstp_debug = False;

extern TstLPortMask enabled_ports;
extern unsigned char rstp_bridge_mac[6];

void
stp_trace (const char *format, ...)
{
#define MAX_MSG_LEN  128
  char msg[MAX_MSG_LEN];
  va_list args;

  if (!rstp_debug)
    return;
  va_start (args, format);
  vsnprintf (msg, MAX_MSG_LEN - 1, format, args);
  Print ("RSTP:%s:%s\n", STP_IN_get_stamp (), msg);
  va_end (args);
}

#ifdef STRONGLY_SPEC_802_1W
int
STP_OUT_set_learning (int port_index, int enable)
{
  if (enable)
    STP_OUT_set_port_state (port_index, UID_PORT_LEARNING);
  else
    STP_OUT_set_port_state (port_index, UID_PORT_DISCARDING);
  return STP_OK;
}

int
STP_OUT_set_forwarding (int port_index, int enable)
{
  if (enable)
    STP_OUT_set_port_state (port_index, UID_PORT_FORWARDING);
  else
    STP_OUT_set_port_state (port_index, UID_PORT_DISCARDING);
  return STP_OK;
}
#else
/* 
 * In many kinds of hardware the state of ports may
 * be changed with another method
 */
int
STP_OUT_set_port_state (IN int port_index, IN RSTP_PORT_STATE state)
{
  void CCL_RSTP_SetPortState(int, IN RSTP_PORT_STATE);
  CCL_RSTP_SetPortState(port_index, state);
  return STP_OK;
  //return AR_INT_STP_set_port_state (port_index, state);
}
#endif


void
STP_OUT_get_port_mac (int port_index, unsigned char *mac)
{
  memcpy (mac, rstp_bridge_mac, 6);
  //memcpy (mac, STP_MAIN_get_port_mac (port_index), 6);
}

int				/* 1- Up, 0- Down */
STP_OUT_get_port_link_status (int port_index)
{
  if (K_LPortMaskGetPort (&enabled_ports, (port_index - 1)))
    return 1;
  return 0;
}

int
STP_OUT_flush_lt (IN TstLPortMask * port_map, char *reason)
{
#ifdef __CCL_TUNING_SLOW_MAC_TBL__
  //2^31 (second) / 60 (minute) / 60 (hour) / 24 (day) / 365 (year) = 68.09
  static	long time_last	= -6;
		long time_now	= sys_time(NULL);

  if (time_now - time_last > 5)	{ // about 5 seconds are required!
    time_last = time_now;
#endif
    K_MacClrTbl (MAC_CLR_DYNMC);
#ifdef __CCL_TUNING_SLOW_MAC_TBL__
  }
#endif

  return STP_OK;
}

int
STP_OUT_flush_single_lt (IN int port_index, char *reason)
{
  TstLPortMask ports_to_flush;

  K_LPortMaskClrAll (&ports_to_flush);
  K_LPortMaskSetPort (&ports_to_flush, port_index - 1);
  return STP_OUT_flush_lt (&ports_to_flush, reason);
}

int
STP_OUT_set_hardware_mode (UID_STP_MODE_T mode)
{
  return STP_OK;
  //return AR_INT_STP_set_mode (mode);
}


int
STP_OUT_tx_bpdu (int port_index, unsigned char *bpdu, size_t bpdu_len)
{
  extern int ccl_rstp_tx_bpdu (int port_index, unsigned char *bpdu,
			     size_t bpdu_len);
  return ccl_rstp_tx_bpdu (port_index, bpdu, bpdu_len);
}

const char *
STP_OUT_get_port_name (IN int port_index)
{
  static char tmp[7];
  sprintf (tmp, "port%02d", (int)port_index);
  return tmp;
  //return port2str (port_index, &sys_config);
}

unsigned long
STP_OUT_get_deafult_port_path_cost (IN unsigned int portNo)
{
  return 200000;
}

static PORT_T *
my_stpapi_port_find (STPM_T * this, int port_index)
{
  register PORT_T *port;

  for (port = this->ports; port; port = port->next)
    if (port_index == port->port_index) {
      return port;
    }
  return NULL;
}

unsigned long
STP_OUT_get_port_oper_speed (unsigned int port_index)
{
#if 0
  register STPM_T *stpm;
  register PORT_T *port;
  long speed = 0;

  RSTP_CRITICAL_PATH_START;

  for (stpm = STP_stpm_get_the_list (); stpm; stpm = stpm->next) {
    if (STP_ENABLED != stpm->admin_state)
      continue;
    port = my_stpapi_port_find (stpm, port_index);
    if (!port)
      continue;
    speed = port->operSpeed;
  }
  RSTP_CRITICAL_PATH_END;
  return speed;
#endif
  long CCL_RSTP_GetSpeed (int);
  return CCL_RSTP_GetSpeed (port_index);
}

int				/* 1- Full, 0- Half */
STP_OUT_get_duplex (IN int port_index)
{
  int CCL_RSTP_GetDuplex (int);
  return CCL_RSTP_GetDuplex (port_index);
}

int
STP_OUT_get_init_stpm_cfg (IN int stpm_id, INOUT UID_STP_CFG_T * cfg)
{
  cfg->bridge_priority = DEF_BR_PRIO;
  cfg->max_age = DEF_BR_MAXAGE;
  cfg->hello_time = DEF_BR_HELLOT;
  cfg->forward_delay = DEF_BR_FWDELAY;
  cfg->force_version = NORMAL_RSTP;

  return STP_OK;
}


int
STP_OUT_get_init_port_cfg (int stpm_id,
			   int port_index, UID_STP_PORT_CFG_T * cfg)
{
  cfg->port_priority = DEF_PORT_PRIO;
  cfg->admin_non_stp = DEF_ADMIN_NON_STP;
  cfg->admin_edge = DEF_ADMIN_EDGE;
  cfg->admin_port_path_cost = ADMIN_PORT_PATH_COST_AUTO;
  cfg->admin_point2point = DEF_P2P;

  return 0;
}

static Bool trace_snmp_traps = False;

int
STP_OUT_trace_snmp_traps (Bool newValue)
{
  trace_snmp_traps = newValue;
  return 0;
}

int
STP_OUT_set_new_root_trap (IN int stpm_id)
{
  if (trace_snmp_traps)
    stp_trace ("   >>> SNMP newRoot trap");
  return 0;
}

int
STP_OUT_set_topology_change_trap (IN int stpm_id,
				  IN int port_index,
				  IN char *port_name, IN int forward)
{
  if (trace_snmp_traps)
    stp_trace ("   >>> SNMP topologyChange trap port %s => %s",
	       port_name, forward ? "Fwd" : "Dsc");
  return 0;
}

void STP_OUT_psos_init_semaphore (void)
{
  spin_lock_init(&rstp_OUT_inner_sem);   
}

void STP_OUT_psos_close_semaphore (void)
{
  spin_lock(&rstp_OUT_inner_sem);
}

void STP_OUT_psos_open_semaphore (void)
{
  spin_unlock(&rstp_OUT_inner_sem);
}


