/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* This file contains API from an operation system to the RSTP library */

#include "krnportmask.h"

#include "base.h"
#include "stpm.h"
#include "stp_in.h"
#include "stp_to.h"

int max_port = 0;

#define INCR100(nev) { nev++; if (nev > 99) nev = 0;}
typedef enum
{
  RSTP_PORT_EN_T,
  RSTP_PORT_DIS_T,
  RSTP_PORT_SPEED_T,
  RSTP_PORT_DPLEX_T,
  RSTP_PORT_RX_T,
  RSTP_PORT_TIME_T,
  RSTP_PORT_MNGR_T,
  RSTP_EVENT_LAST_DUMMY
}
RSTP_EVENT_T;

RSTP_EVENT_T tev = RSTP_EVENT_LAST_DUMMY;
int nev = 0;

static void _stp_in_dbg_stt_sniffer (int stpd, char *port_name, BPDU_T * bpdu,
				     size_t len);

int
STP_IN_stpm_delete (int stpm_id)
{
  register STPM_T *this;
  int iret = 0;

  RSTP_CRITICAL_PATH_START;
  this = stpapi_stpm_find (stpm_id);

  if (!this) {			/* it had not yet been created :( */
    iret = STP_Had_Not_Yet_Been_Created;
  } else {

    if (STP_ENABLED == this->admin_state) {
      if (0 != STP_stpm_enable (this, STP_DISABLED)) {	/* can't disable :( */
	iret = STP_Another_Error;
      } else
	STP_OUT_set_hardware_mode (STP_DISABLED);
    }

    if (0 == iret) {
      STP_stpm_delete (this);
    }
  }
  RSTP_CRITICAL_PATH_END;
  return iret;
}

void *
stp_in_stpm_create (int stpm_id, char *name, TstLPortMask * port_bmp,
		    int *err_code)
{
  int port_index;
  register STPM_T *this;

  /* stp_trace ("stp_in_stpm_create(%s)", name); */
  this = stpapi_stpm_find (stpm_id);
  if (this) {			/* it had just been created :( */
    *err_code = STP_Nothing_To_Do;
    return this;
  }

  this = STP_stpm_create (stpm_id, name);
  if (!this) {			/* can't create stpm :( */
    *err_code = STP_Cannot_Create_Instance;
    return NULL;
  }
  K_LPortMaskCopy (this->portmap, port_bmp);
  this->number_of_ports = 0;

  for (port_index = max_port; port_index > 0; port_index--) {
    if (K_LPortMaskGetPort (port_bmp, (port_index - 1))) {
      if (!STP_port_create (this, port_index)) {
	/* can't add port :( */
	stp_trace ("can't create port %d", (int) port_index);
	STP_stpm_delete (this);
	*err_code = STP_Cannot_Create_Instance_For_Port;
	return NULL;
      } else {
	this->number_of_ports++;
      }
    }
  }

  *err_code = STP_OK;
  return this;
}

int
_stp_in_stpm_enable (int stpm_id, char *name,
		     TstLPortMask * port_bmp, UID_STP_MODE_T admin_state)
{
  register STPM_T *this;
  Bool created_here = False;
  int rc, err_code;

/*  stp_trace ("_stp_in_stpm_enable(%s)", name); */
  this = stpapi_stpm_find (stpm_id);

  if (STP_DISABLED != admin_state) {
    if (!stpm_id) {
      register STPM_T *stpm;

      for (stpm = STP_stpm_get_the_list (); stpm; stpm = stpm->next) {
	if (STP_DISABLED != stpm->admin_state) {
	  STP_OUT_set_hardware_mode (STP_DISABLED);
	  STP_stpm_enable (stpm, STP_DISABLED);
	}
      }
    }
  }

  if (!this) {			/* it had not yet been created */
    if (STP_ENABLED == admin_state) {	/* try to create it */
      stp_trace ("implicit create to stpm '%s'", name);
      this = stp_in_stpm_create (stpm_id, name, port_bmp, &err_code);
      if (!this) {
	stp_trace ("implicit create to stpm '%s' failed", name);
	return STP_Imlicite_Instance_Create_Failed;
      }
      created_here = True;
    } else {			/* try to disable nothing ? */
      return 0;
    }
  }

  if (this->admin_state == admin_state) {	/* nothing to do :) */
    return 0;
  }

  rc = STP_stpm_enable (this, admin_state);
  if (!rc) {
    STP_OUT_set_hardware_mode (admin_state);
  }

  if (rc && created_here) {
    STP_stpm_delete (this);
  }

  return rc;
}


STPM_T *
stpapi_stpm_find (int stpm_id)
{
  register STPM_T *this;

  for (this = STP_stpm_get_the_list (); this; this = this->next)
    if (stpm_id == this->stpm_id)
      return this;

  return NULL;
}

static PORT_T *
_stpapi_port_find (STPM_T * this, int port_index)
{
  register PORT_T *port;

  for (port = this->ports; port; port = port->next)
    if (port_index == port->port_index) {
      return port;
    }

  return NULL;
}


static void
_conv_br_id_2_uid (IN BRIDGE_ID * f, OUT UID_BRIDGE_ID_T * t)
{
  memcpy (t, f, sizeof (UID_BRIDGE_ID_T));
}

static int
_check_stpm_config (IN UID_STP_CFG_T * uid_cfg)
{
  if (uid_cfg->bridge_priority < MIN_BR_PRIO) {
    stp_trace ("%d bridge_priority small", (int) uid_cfg->bridge_priority);
    return STP_Small_Bridge_Priority;
  }

  if (uid_cfg->bridge_priority > MAX_BR_PRIO) {
    stp_trace ("%d bridge_priority large", (int) uid_cfg->bridge_priority);
    return STP_Large_Bridge_Priority;
  }

  if (0 != (uid_cfg->bridge_priority % BR_PRIO_STEP)) {
    stp_trace ("%d bridge_priority granularity (%d, %d)",
	       (int) uid_cfg->bridge_priority,
	       (int) ((uid_cfg->bridge_priority / BR_PRIO_STEP) *
		      BR_PRIO_STEP),
	       (int) (((uid_cfg->bridge_priority / BR_PRIO_STEP) + 1) *
		      BR_PRIO_STEP));
    return STP_Bridge_Priority_Granularity;
  }

  if (uid_cfg->hello_time < MIN_BR_HELLOT) {
    stp_trace ("%d hello_time small", (int) uid_cfg->hello_time);
    return STP_Small_Hello_Time;
  }

  if (uid_cfg->hello_time > MAX_BR_HELLOT) {
    stp_trace ("%d hello_time large", (int) uid_cfg->hello_time);
    return STP_Large_Hello_Time;
  }

  if (uid_cfg->max_age < MIN_BR_MAXAGE) {
    stp_trace ("%d max_age small", (int) uid_cfg->max_age);
    return STP_Small_Max_Age;
  }

  if (uid_cfg->max_age > MAX_BR_MAXAGE) {
    stp_trace ("%d max_age large", (int) uid_cfg->max_age);
    return STP_Large_Max_Age;
  }

  if (uid_cfg->forward_delay < MIN_BR_FWDELAY) {
    stp_trace ("%d forward_delay small", (int) uid_cfg->forward_delay);
    return STP_Small_Forward_Delay;
  }

  if (uid_cfg->forward_delay > MAX_BR_FWDELAY) {
    stp_trace ("%d forward_delay large", (int) uid_cfg->forward_delay);
    return STP_Large_Forward_Delay;
  }

  /* IEEE Std 802.1w-2001, 17.28.2, p.76 */
  if (2 * (uid_cfg->forward_delay - 1) < uid_cfg->max_age) {
    stp_trace ("forward_delay & max_age");
    return STP_Forward_Delay_And_Max_Age_Are_Inconsistent;
  }

  /* IEEE Std 802.1w-2001, B.4.6, p.100 */
  if (uid_cfg->max_age < 2 * (uid_cfg->hello_time + 1)) {
    stp_trace ("hello_time & max_age");
    return STP_Hello_Time_And_Max_Age_Are_Inconsistent;
  }

  return 0;
}

static void
_stp_in_enable_port_on_stpm (STPM_T * stpm, int port_index, Bool enable)
{
  register PORT_T *port;

  port = _stpapi_port_find (stpm, port_index);
  if (!port)
    return;
  if (port->portEnabled == enable) {	/* nothing to do :) */
    return;
  }
  port->uptime = 0;
  if (enable) {			/* clear port statistics */
    port->rx_cfg_bpdu_cnt =
      port->rx_rstp_bpdu_cnt = port->rx_tcn_bpdu_cnt = 0;
    port->tx_cfg_bpdu_cnt =
      port->tx_rstp_bpdu_cnt = port->tx_tcn_bpdu_cnt = 0;
  }
#ifdef STP_DBG
  if (port->edge->debug) {
    stp_trace ("Port %s became '%s' adminEdge=%c",
	       port->port_name, enable ? "enable" : "disable",
	       port->adminEdge ? 'Y' : 'N');
  }
#endif

  port->adminEnable = enable;
  if (STP_ENABLED != stpm->admin_state) {
    port->portEnabled =
      port->forwarding = port->forward =
      port->learning = port->learn = enable;
    port->role = enable ? NonStpPort : DisabledPort;
    return;
  }

  STP_port_init (port, stpm, False);
}

void
STP_IN_init (int max_port_index)
{
  max_port = max_port_index;
  RSTP_INIT_CRITICAL_PATH_PROTECTIO;
}

int
STP_IN_stpm_get_cfg (IN int stpm_id, OUT UID_STP_CFG_T * uid_cfg)
{
  register STPM_T *this;

  uid_cfg->field_mask = 0;

  RSTP_CRITICAL_PATH_START;
  this = stpapi_stpm_find (stpm_id);

  if (!this) {			/* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
  }

  uid_cfg->number_of_ports = this->number_of_ports;
  K_LPortMaskCopy (&uid_cfg->ports, this->portmap);

  if (this->admin_state != STP_DISABLED) {
    uid_cfg->field_mask |= BR_CFG_STATE;
  }
  uid_cfg->stp_enabled = this->admin_state;

  if (this->ForceVersion != 2) {
    uid_cfg->field_mask |= BR_CFG_FORCE_VER;
  }
  uid_cfg->force_version = this->ForceVersion;

  if (this->BrId.prio != DEF_BR_PRIO) {
    uid_cfg->field_mask |= BR_CFG_PRIO;
  }
  uid_cfg->bridge_priority = this->BrId.prio;

  if (this->BrTimes.MaxAge != DEF_BR_MAXAGE) {
    uid_cfg->field_mask |= BR_CFG_AGE;
  }
  uid_cfg->max_age = this->BrTimes.MaxAge;

  if (this->BrTimes.HelloTime != DEF_BR_HELLOT) {
    uid_cfg->field_mask |= BR_CFG_HELLO;
  }
  uid_cfg->hello_time = this->BrTimes.HelloTime;

  if (this->BrTimes.ForwardDelay != DEF_BR_FWDELAY) {
    uid_cfg->field_mask |= BR_CFG_DELAY;
  }
  uid_cfg->forward_delay = this->BrTimes.ForwardDelay;

  uid_cfg->hold_time = TxHoldCount;

  RSTP_CRITICAL_PATH_END;
  return 0;
}

int
STP_IN_port_get_cfg (int stpm_id, int port_index,
		     UID_STP_PORT_CFG_T * uid_cfg)
{
  register STPM_T *this;
  register PORT_T *port;

  RSTP_CRITICAL_PATH_START;
  this = stpapi_stpm_find (stpm_id);

  if (!this) {			/* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
  }

  port = _stpapi_port_find (this, port_index);
  if (!port) {			/* port is absent in the stpm :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Port_Is_Absent_In_The_Stp;
  }

  uid_cfg->field_mask = 0;

  uid_cfg->port_priority = port->port_id >> 8;
  if (uid_cfg->port_priority != DEF_PORT_PRIO)
    uid_cfg->field_mask |= PT_CFG_PRIO;

  uid_cfg->admin_port_path_cost = port->adminPCost;
  if (uid_cfg->admin_port_path_cost != ADMIN_PORT_PATH_COST_AUTO)
    uid_cfg->field_mask |= PT_CFG_COST;

  uid_cfg->admin_point2point = port->adminPointToPointMac;
  if (uid_cfg->admin_point2point != DEF_P2P)
    uid_cfg->field_mask |= PT_CFG_P2P;

  uid_cfg->admin_edge = port->adminEdge;
  if (uid_cfg->admin_edge != DEF_ADMIN_EDGE)
    uid_cfg->field_mask |= PT_CFG_EDGE;


  RSTP_CRITICAL_PATH_END;
  return 0;
}

int
STP_IN_port_get_state (IN int stpm_id, INOUT UID_STP_PORT_STATE_T * entry)
{
  register STPM_T *this;
  register PORT_T *port;
  PORT_ROLE_T role_2_show;

  RSTP_CRITICAL_PATH_START;
  this = stpapi_stpm_find (stpm_id);

  if (!this) {			/* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
  }

  port = _stpapi_port_find (this, entry->port_no);
  if (!port) {			/* port is absent in the stpm :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Port_Is_Absent_In_The_Stp;
  }

  entry->port_id = port->port_id;
  if (port->admin_non_stp) {
    entry->state = UID_PORT_NON_STP;
  }
  /*else if (DisabledPort == port->role) {
    if (STP_TEST == this->admin_state) {
      entry->state = port->adminEnable ?
	UID_PORT_FORWARDING : UID_PORT_DISABLED;
    } else
      entry->state = UID_PORT_DISABLED;
  }*/
  else if (!port->forward && !port->learn) {
    entry->state = UID_PORT_DISCARDING;
  } else if (!port->forward && port->learn) {
    entry->state = UID_PORT_LEARNING;
  } else {
    entry->state = UID_PORT_FORWARDING;
  }

  entry->uptime = port->uptime;
  entry->path_cost = port->operPCost;
  _conv_br_id_2_uid (&port->portPrio.root_bridge, &entry->designated_root);
  entry->designated_cost = port->portPrio.root_path_cost;
  _conv_br_id_2_uid (&port->portPrio.design_bridge,
		     &entry->designated_bridge);
  entry->designated_port = port->portPrio.design_port;

  if (STP_TEST != this->admin_state && NonStpPort != port->selectedRole)
    role_2_show = port->role;
  else
    role_2_show = port->selectedRole;

  switch (role_2_show) {
    case DisabledPort:
    entry->role = ' ';
    break;
    case AlternatePort:
    entry->role = 'A';
    break;
    case BackupPort:
    entry->role = 'B';
    break;
    case RootPort:
    entry->role = 'R';
    break;
    case DesignatedPort:
    entry->role = 'D';
    break;
    case NonStpPort:
    entry->role = '-';
    break;
    default:
    entry->role = '?';
    break;
  }

  if (DisabledPort == port->role || NonStpPort == port->role) {
    memset (&entry->designated_root, 0, sizeof (UID_BRIDGE_ID_T));
    memset (&entry->designated_bridge, 0, sizeof (UID_BRIDGE_ID_T));
    entry->designated_cost = 0;
    entry->designated_port = port->port_id;
  }

  if (DisabledPort == port->role) {
    entry->oper_point2point =
      (P2P_FORCE_FALSE == port->adminPointToPointMac) ? 0 : 1;
    entry->oper_edge = port->adminEdge;
    entry->oper_stp_neigb = 0;
  } else {
    entry->oper_point2point = port->operPointToPointMac ? 1 : 0;
    entry->oper_edge = port->operEdge ? 1 : 0;
    entry->oper_stp_neigb = port->sendRSTP ? 0 : 1;
  }
  entry->oper_port_path_cost = port->operPCost;

  entry->rx_cfg_bpdu_cnt = port->rx_cfg_bpdu_cnt;
  entry->rx_rstp_bpdu_cnt = port->rx_rstp_bpdu_cnt;
  entry->rx_tcn_bpdu_cnt = port->rx_tcn_bpdu_cnt;
  entry->tx_cfg_bpdu_cnt = port->tx_cfg_bpdu_cnt;
  entry->tx_rstp_bpdu_cnt = port->tx_rstp_bpdu_cnt;
  entry->tx_tcn_bpdu_cnt = port->tx_tcn_bpdu_cnt;

  entry->fdWhile = port->fdWhile;	/* 17.15.1 */
  entry->helloWhen = port->helloWhen;	/* 17.15.2 */
  entry->mdelayWhile = port->mdelayWhile;	/* 17.15.3 */
  entry->rbWhile = port->rbWhile;	/* 17.15.4 */
  entry->rcvdInfoWhile = port->rcvdInfoWhile;	/* 17.15.5 */
  entry->rrWhile = port->rrWhile;	/* 17.15.6 */
  entry->tcWhile = port->tcWhile;	/* 17.15.7 */
  entry->txCount = port->txCount;	/* 17.18.40 */
  entry->lnkWhile = port->lnkWhile;
  entry->infoIs = (int) port->infoIs;

  entry->rcvdInfoWhile = port->rcvdInfoWhile;
  entry->top_change_ack = port->tcAck;
  entry->tc = port->tc;

  RSTP_CRITICAL_PATH_END;
  return 0;
}

int
STP_IN_stpm_get_state (IN int stpm_id, OUT UID_STP_STATE_T * entry)
{
  register STPM_T *this;

  RSTP_CRITICAL_PATH_START;
  this = stpapi_stpm_find (stpm_id);

  if (!this) {			/* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
  }

  strncpy (entry->name, this->name, NAME_LEN);
  entry->stpm_id = this->stpm_id;
  _conv_br_id_2_uid (&this->rootPrio.root_bridge, &entry->designated_root);
  entry->root_path_cost = this->rootPrio.root_path_cost;
  entry->root_port = this->rootPortId;
  entry->max_age = this->rootTimes.MaxAge;
  entry->forward_delay = this->rootTimes.ForwardDelay;
  entry->hello_time = this->rootTimes.HelloTime;

  _conv_br_id_2_uid (&this->BrId, &entry->bridge_id);

  entry->stp_enabled = this->admin_state;

  entry->timeSince_Topo_Change = this->timeSince_Topo_Change;
  entry->Topo_Change_Count = this->Topo_Change_Count;
  entry->Topo_Change = this->Topo_Change;

  RSTP_CRITICAL_PATH_END;
  return 0;
}

int
STP_IN_stpm_get_name_by_stpm_id (int stpm_id, char *name, size_t buffsize)
{
  register STPM_T *stpm;
  int iret = -1;

  RSTP_CRITICAL_PATH_START;
  for (stpm = STP_stpm_get_the_list (); stpm; stpm = stpm->next) {
    if (stpm_id == stpm->stpm_id) {
      if (stpm->name)
	strncpy (name, stpm->name, buffsize);
      else
	memset (name, 0, buffsize);
      iret = 0;
      break;
    }
  }
  RSTP_CRITICAL_PATH_END;
  return iret;
}

int				/* call it, when link Up/Down */
STP_IN_enable_port (int port_index, Bool enable)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;
  tev = enable ? RSTP_PORT_EN_T : RSTP_PORT_DIS_T;
  INCR100 (nev);

#ifdef STP_DBG
  stp_trace ("port p%02d => %sABLE", (int) port_index, enable ? "EN" : "DIS");
#endif
  if (!enable) {
#if 0	//def STP_DBG
    stp_trace ("%s (p%02d, %s, '%s')",
	       "clearFDB", (int) port_index, "this port", "disable port");
#endif
    /* STP_OUT_flush_single_lt (port_index, "disable port");
     * don't flush mac table, it takes too much time */
  }

  for (stpm = STP_stpm_get_the_list (); stpm; stpm = stpm->next) {
    _stp_in_enable_port_on_stpm (stpm, port_index, enable);
  }
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int
STP_IN_set_forward_all_port (void)
{
  register STPM_T *stpm;
  register PORT_T *port;

  RSTP_CRITICAL_PATH_START;
  for (stpm = STP_stpm_get_the_list (); stpm; stpm = stpm->next) {
    for (port = stpm->ports; port; port = port->next) {
      port->adminEnable = True;
      port->role = NonStpPort;
    }
  }
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int	/* call it, when port speed has been changed, speed in Kb/s  */
STP_IN_changed_port_speed (int port_index, long speed)
{
  register STPM_T *stpm;
  register PORT_T *port;

  RSTP_CRITICAL_PATH_START;
  tev = RSTP_PORT_SPEED_T;
  INCR100 (nev);

  for (stpm = STP_stpm_get_the_list (); stpm; stpm = stpm->next) {
    if (STP_ENABLED != stpm->admin_state)
      continue;
    port = _stpapi_port_find (stpm, port_index);
    if (!port || port->operSpeed == speed)
      continue;
    port->operSpeed = speed;
#ifdef STP_DBG
    if (port->pcost->debug) {
      stp_trace ("changed operSpeed=%lu", port->operSpeed);
    }
#endif

    port->reselect = True;
    port->selected = False;
  }
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int	/* call it, when port duplex mode has been changed  */
STP_IN_changed_port_duplex (int port_index)
{
  register STPM_T *stpm;
  register PORT_T *port;

  RSTP_CRITICAL_PATH_START;
  tev = RSTP_PORT_DPLEX_T;
  INCR100 (nev);

  for (stpm = STP_stpm_get_the_list (); stpm; stpm = stpm->next) {
    if (STP_ENABLED != stpm->admin_state)
      continue;

    port = _stpapi_port_find (stpm, port_index);
    if (!port)
      continue;
#ifdef STP_DBG
    if (port->p2p->debug) {
      stp_trace ("STP_IN_changed_port_duplex(%s)", port->port_name);
    }
#endif
    port->p2p_recompute = True;
    port->reselect = True;
    port->selected = False;
  }
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int
STP_IN_check_bpdu_header (BPDU_T * bpdu, size_t len)
{
  unsigned short len8023;

  len8023 = ntohs (*(unsigned short *) bpdu->eth.len8023);
  if (len8023 > 1500) {		/* big len8023 format :( */
    return STP_Big_len8023_Format;
  }

  if (len8023 < MIN_BPDU) {	/* small len8023 format :( */
    return STP_Small_len8023_Format;
  }

  if (len8023 + 14 > len) {	/* len8023 format gt len :( */
    return STP_len8023_Format_Gt_Len;
  }

  if (bpdu->eth.dsap != BPDU_L_SAP ||
      bpdu->eth.ssap != BPDU_L_SAP || bpdu->eth.llc != LLC_UI) {
    /* this is not a proper 802.3 pkt! :( */
    return STP_Not_Proper_802_3_Packet;
  }

  if (bpdu->hdr.protocol[0] || bpdu->hdr.protocol[1]) {
    return STP_Invalid_Protocol;
  }
#if 0
  if (bpdu->hdr.version != BPDU_VERSION_ID) {
    return STP_Invalid_Version;
  }
#endif
  /* see also 9.3.4: think & TBD :( */
  return 0;
}

int
STP_IN_rx_bpdu (int port_index, BPDU_T * bpdu, size_t len)
{
  register PORT_T *port;
  register STPM_T *this;
  int iret;

  RSTP_CRITICAL_PATH_START;
  tev = RSTP_PORT_RX_T;
  INCR100 (nev);

  //this = stpapi_stpm_find (stpm_id);
  this = STP_stpm_get_the_list ();
  if (!this) {			/*  the stpm had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
  }

  port = _stpapi_port_find (this, port_index);
  if (!port) {			/* port is absent in the stpm :( */
    stp_trace ("RX bpdu stpm_id=%d port=%d port is absent in the stpm :(",
	       (int) this->stpm_id, (int) port_index);
    RSTP_CRITICAL_PATH_END;
    return STP_Port_Is_Absent_In_The_Stp;
  }

  if (!port->portEnabled) {	/* port link change indication will come later :( */
    _stp_in_enable_port_on_stpm (this, port->port_index, True);
  }
  if (STP_ENABLED != this->admin_state) {	/* the stpm had not yet been enabled :( */
#ifdef STP_DBG
    if (port->info->debug)
      _stp_in_dbg_stt_sniffer (0, port->port_name, bpdu, len);
#endif
    RSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Enabled;
  }
#ifdef STP_DBG
  if (port->edge->debug && port->operEdge) {
    stp_trace ("port %s not operEdge !", port->port_name);
  }
#endif
  port->operEdge = False;
  port->wasInitBpdu = True;

  iret = STP_port_rx_bpdu (port, bpdu, len);
  STP_stpm_update (this);
  RSTP_CRITICAL_PATH_END;

  return iret;
}

int
STP_IN_one_second (void)
{
  register STPM_T *stpm;
  register int dbg_cnt = 0;

  RSTP_CRITICAL_PATH_START;
  tev = RSTP_PORT_TIME_T;
  INCR100 (nev);

  for (stpm = STP_stpm_get_the_list (); stpm; stpm = stpm->next) {
    switch (stpm->admin_state) {
      case STP_ENABLED:
      /* stp_trace ("STP_IN_one_second stpm_id=%d", (int) stpm->stpm_id); */
      STP_stpm_one_second (stpm);
      dbg_cnt++;
      break;
      case STP_TEST:
      if (!stpm->sttx_numbf) {
	break;
      }
      if (stpm->sttx_timet < 0) {
	/* stp_trace ("sttx_timet=%d < 0", stpm->sttx_timet); */
	break;
      }
      if (!--stpm->sttx_timet) {
	stp_trace ("sttx(%s) transmit sttx_numbf=%d sttx_timet=%d",
		   stpm->name, stpm->sttx_numbf, (int) stpm->sttx_timet);
	STP_stpm_dbg_sttx (stpm, stpm->sttx_type, &stpm->sttx_bmp);
	stpm->sttx_timet = stpm->sttx_timef;
	if (stpm->sttx_numbf > 0)
	  stpm->sttx_numbf--;
	dbg_cnt++;
      } else {
	;			/* stp_trace ("sttx_timet=%d > 0", stpm->sttx_timet); */
      }
      break;
      default:
      break;
    }
  }

  RSTP_CRITICAL_PATH_END;

  return dbg_cnt;
}

extern Bool STP_rolesel_update_root_priority (STPM_T * stpm);

int
STP_IN_stpm_set_cfg (IN int stpm_id, IN UID_STP_CFG_T * uid_cfg)
{
  int rc = 0, prev_prio, err_code;
  Bool created_here, enabled_here;
  register STPM_T *this;
  UID_STP_CFG_T old;

  /* stp_trace ("STP_IN_stpm_set_cfg"); */
  if (0 != STP_IN_stpm_get_cfg (stpm_id, &old)) {
    STP_OUT_get_init_stpm_cfg (stpm_id, &old);
  }

  RSTP_CRITICAL_PATH_START;
  tev = RSTP_PORT_MNGR_T;
  INCR100 (nev);
  if (BR_CFG_PRIO & uid_cfg->field_mask) {
    old.bridge_priority = uid_cfg->bridge_priority;
  }

  if (BR_CFG_AGE & uid_cfg->field_mask) {
    old.max_age = uid_cfg->max_age;
  }

  if (BR_CFG_HELLO & uid_cfg->field_mask) {
    old.hello_time = uid_cfg->hello_time;
  }

  if (BR_CFG_DELAY & uid_cfg->field_mask) {
    old.forward_delay = uid_cfg->forward_delay;
  }

  if (BR_CFG_FORCE_VER & uid_cfg->field_mask) {
    old.force_version = uid_cfg->force_version;
  }

  rc = _check_stpm_config (&old);
  if (0 != rc) {
    stp_trace ("_check_stpm_config failed %d", (int) rc);
    RSTP_CRITICAL_PATH_END;
    return rc;
  }

  if ((BR_CFG_STATE & uid_cfg->field_mask) &&
      (STP_DISABLED == uid_cfg->stp_enabled)) {
    rc = _stp_in_stpm_enable (stpm_id, uid_cfg->name,
			      &uid_cfg->ports, STP_DISABLED);
    if (0 != rc) {
      stp_trace ("can't disable rc=%d", (int) rc);
      RSTP_CRITICAL_PATH_END;
      return rc;
    }
    uid_cfg->field_mask &= !BR_CFG_STATE;
    if (!uid_cfg->field_mask) {
      RSTP_CRITICAL_PATH_END;
      return 0;
    }
  }

  /* get current state */
  this = stpapi_stpm_find (stpm_id);
  created_here = False;
  enabled_here = False;
  if (!this) {			/* it had not yet been created */
    this = stp_in_stpm_create (stpm_id, uid_cfg->name, &uid_cfg->ports, &err_code);	/*STP_IN_stpm_set_cfg */
    if (!this) {
      stp_trace ("can't create");
      RSTP_CRITICAL_PATH_END;
      return err_code;
    }
  }

  prev_prio = this->BrId.prio;
  this->BrId.prio = old.bridge_priority;
  if (STP_ENABLED == this->admin_state) {
    if (0 != STP_stpm_check_bridge_priority (this)) {
      this->BrId.prio = prev_prio;
      stp_trace ("STP_stpm_check_bridge_priority failed");
      RSTP_CRITICAL_PATH_END;
      return STP_Invalid_Bridge_Priority;
    }

    /*stp_trace ("prev_prio=%lx BrId.prio=%lx rootPortId=%lx",
       (unsigned long) prev_prio,
       (unsigned long) this->BrId.prio,
       (unsigned long) this->rootPortId); */

    STP_rolesel_update_root_priority (this);
  }

  this->BrTimes.MaxAge = old.max_age;
  this->BrTimes.HelloTime = old.hello_time;
  this->BrTimes.ForwardDelay = old.forward_delay;
  this->ForceVersion = (PROTOCOL_VERSION_T) old.force_version;

  if ((BR_CFG_STATE & uid_cfg->field_mask) &&
      STP_DISABLED != uid_cfg->stp_enabled &&
      uid_cfg->stp_enabled != this->admin_state) {
    rc =
      _stp_in_stpm_enable (stpm_id, uid_cfg->name, &uid_cfg->ports,
			   uid_cfg->stp_enabled);
    if (0 != rc) {
      stp_trace ("cannot enable");
      if (created_here) {
	STP_stpm_delete (this);
      }
      RSTP_CRITICAL_PATH_END;
      return rc;
    }
    enabled_here = True;
  }

  if (!enabled_here &&
      STP_TEST != this->admin_state && STP_DISABLED != this->admin_state) {
    STP_stpm_update_after_bridge_management (this);
  }
  RSTP_CRITICAL_PATH_END;
  return 0;
}

int
STP_IN_set_port_cfg (IN int stpm_id, IN UID_STP_PORT_CFG_T * uid_cfg)
{
  register STPM_T *this;
  register PORT_T *port;
  register int port_no;

  /* Validation */
  if (PT_CFG_PRIO & uid_cfg->field_mask) {

    if (uid_cfg->port_priority > MAX_PORT_PRIO) {
      stp_trace ("%d port_priority large", (int) uid_cfg->port_priority);
      return STP_Large_Port_Priority;
    }

    if (0 != (uid_cfg->port_priority % PORT_PRIO_STEP)) {
      stp_trace ("%d port_priority granularity (%d, %d)",
		 (int) uid_cfg->port_priority,
		 (int) ((uid_cfg->port_priority / PORT_PRIO_STEP) *
			PORT_PRIO_STEP),
		 (int) (((uid_cfg->port_priority / PORT_PRIO_STEP) + 1) *
			PORT_PRIO_STEP));
      return STP_Port_Priority_Granularity;
    }
  }

  if (PT_CFG_COST & uid_cfg->field_mask) {
    if (uid_cfg->admin_port_path_cost > MAX_ADMIN_PORT_PATH_COST) {
      stp_trace ("%ld admin_port_path_cost large (> %ld)",
		 (long) uid_cfg->admin_port_path_cost,
		 (long) MAX_ADMIN_PORT_PATH_COST);
      return STP_Large_Port_Path_Cost;
    }
  }

  RSTP_CRITICAL_PATH_START;
  tev = RSTP_PORT_MNGR_T;
  INCR100 (nev);

  this = stpapi_stpm_find (stpm_id);
  if (!this) {			/* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    stp_trace ("RSTP instance with %d hasn't been created", (int) stpm_id);
    return STP_Had_Not_Yet_Been_Created;
  }

  for (port_no = 1; port_no <= max_port; port_no++) {
    if (!K_LPortMaskGetPort (&uid_cfg->port_bmp, port_no - 1))
      continue;

    port = _stpapi_port_find (this, port_no);
    if (!port) {		/* port is absent in the stpm :( */
      continue;
    }

    if (PT_CFG_MCHECK & uid_cfg->field_mask) {
      /* if (this->ForceVersion >= NORMAL_RSTP) */
	port->mcheck = uid_cfg->admin_mcheck;
    }

    if (PT_CFG_COST & uid_cfg->field_mask) {
      port->adminPCost = uid_cfg->admin_port_path_cost;
    }

    if (PT_CFG_PRIO & uid_cfg->field_mask) {
      port->port_id = (uid_cfg->port_priority << 8) + port_no;
    }

    if (PT_CFG_P2P & uid_cfg->field_mask) {
      port->adminPointToPointMac = uid_cfg->admin_point2point;
      port->p2p_recompute = True;
    }

    if (PT_CFG_EDGE & uid_cfg->field_mask) {
      port->adminEdge = uid_cfg->admin_edge;
      port->operEdge = port->adminEdge;
#ifdef STP_DBG
      if (port->edge->debug) {
	stp_trace ("port %s is operEdge=%c in STP_IN_set_port_cfg",
		   port->port_name, port->operEdge ? 'Y' : 'n');
      }
#endif
    }

    if (PT_CFG_NON_STP & uid_cfg->field_mask) {
#ifdef STP_DBG
      if (port->roletrns->debug
	  && port->admin_non_stp != uid_cfg->admin_non_stp) {
	stp_trace ("port %s is adminNonStp=%c in STP_IN_set_port_cfg",
		   port->port_name, uid_cfg->admin_non_stp ? 'Y' : 'n');
      }
#endif
      port->admin_non_stp = uid_cfg->admin_non_stp;
    }

    port->reselect = True;
    port->selected = False;
  }

  STP_stpm_update (this);

  RSTP_CRITICAL_PATH_END;

  return 0;
}

#ifdef STP_DBG
int
STP_IN_dbg_set_port_trace (char *mach_name, int enadis,
			   int stpm_id, TstLPortMask * ports, int is_print_err)
{
  register STPM_T *this;
  register PORT_T *port;
  register int port_no;

  RSTP_CRITICAL_PATH_START;
  this = stpapi_stpm_find (stpm_id);
  if (!this) {			/* it had not yet been created :( */
    RSTP_CRITICAL_PATH_END;
    if (is_print_err) {
      stp_trace ("RSTP instance with id %d hasn't been created",
		 (int) stpm_id);
    }
    return STP_Had_Not_Yet_Been_Created;
  }

  for (port_no = 1; port_no <= max_port; port_no++) {
    if (!K_LPortMaskGetPort (ports, port_no - 1))
      continue;

    port = _stpapi_port_find (this, port_no);
    if (!port) {		/* port is absent in the stpm :( */
      continue;
    }
    STP_port_trace_state_machine (port, mach_name, enadis, stpm_id);
  }

  RSTP_CRITICAL_PATH_END;

  return 0;
}
#endif

const char *
STP_IN_get_stamp (void)
{
  static char buff[30];
  char bnev[4];

  strcpy (buff, UT_sprint_time_stamp (1));
  strcat (buff, ":");
  switch (tev) {
    case RSTP_PORT_EN_T:
    strcat (buff, "e:");
    break;
    case RSTP_PORT_DIS_T:
    strcat (buff, "d:");
    break;
    case RSTP_PORT_SPEED_T:
    strcat (buff, "S:");
    break;
    case RSTP_PORT_DPLEX_T:
    strcat (buff, "D:");
    break;
    case RSTP_PORT_RX_T:
    strcat (buff, "r:");
    break;
    case RSTP_PORT_TIME_T:
    strcat (buff, "T:");
    break;
    case RSTP_PORT_MNGR_T:
    strcat (buff, "M:");
    break;
    default:
    strcat (buff, " :");
    break;

  }
  sprintf (bnev, "%02d", nev);
  //strcat (buff, bnev);
  return buff;
}

const char *
STP_IN_get_error_explanation (int rstp_err_no)
{
#define CHOOSE(a) #a
  static char *rstp_error_names[] = RSTP_ERRORS;
#undef CHOOSE
  if (rstp_err_no < STP_OK) {
    return "Too small error code :(";
  }
  if (rstp_err_no >= STP_LAST_DUMMY) {
    return "Too big error code :(";
  }

  return rstp_error_names[rstp_err_no];
}

int
stp_in_dbg_sttx (int stpd, int numbf, int timef, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);

  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  K_LPortMaskAnd (&stpm->sttx_bmp, stpm->portmap, port_bmp);
  stpm->sttx_numbf = numbf;
  stpm->sttx_timef = timef;
  stpm->sttx_timet = 1;
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_force_sttx (int stpd, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);

  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  K_LPortMaskAnd (&stpm->sttx_bmp, stpm->portmap, port_bmp);
  STP_stpm_dbg_sttx (stpm, stpm->sttx_type, &stpm->sttx_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

static void
_stp_in_display_bit (unsigned char bitmask,
		     char *bit_name, char *bit_fmt, unsigned char flags)
{
  int the_bit = (flags & bitmask) ? 1 : 0;

  Print ("    ");
  Print (bit_fmt, the_bit);
  Print (" %-20s  %s\n", bit_name, the_bit ? "- yes" : "");
}

static void
_stp_in_dbg_stt_sniffer (int stpd, char *port_name, BPDU_T * bpdu, size_t len)
{
  PRIO_VECTOR_T vect;
  TIMEVALUES_T times;
  int role_ind;
  static char *txt_role[] = { "Unknown", "Alternate", "Root", "Designated" };

  stp_trace ("port %s got %s BPDU", port_name,
	     BPDU_CONFIG_TYPE == bpdu->hdr.bpdu_type ? "Config" :
	     (BPDU_TOPO_CHANGE_TYPE == bpdu->hdr.bpdu_type ? "Tcn" :
	      (BPDU_RSTP ? "RST" : "Unknown type")));
  if (BPDU_TOPO_CHANGE_TYPE == bpdu->hdr.bpdu_type)
    return;
  Print ("  Type=0x%lX Version=0x%lX, Flags=0x%lX\n",
	 (unsigned long) bpdu->hdr.bpdu_type,
	 (unsigned long) bpdu->hdr.version, (unsigned long) bpdu->body.flags);

  _stp_in_display_bit (TOPOLOGY_CHANGE_ACK_BIT, "TcAck", "%d.......",
		       bpdu->body.flags);
  _stp_in_display_bit (AGREEMENT_BIT, "Agreement", ".%d......",
		       bpdu->body.flags);
  _stp_in_display_bit (FORWARD_BIT, "Forward", "..%d.....", bpdu->body.flags);
  _stp_in_display_bit (LEARN_BIT, "Learning", "...%d....", bpdu->body.flags);
  role_ind = (int) ((bpdu->body.flags & PORT_ROLE_MASK) >> PORT_ROLE_OFFS);
  Print ("    ....XX.. Role = %s (%lx)\n", txt_role[role_ind],
	 (unsigned long) role_ind);
  _stp_in_display_bit (PROPOSAL_BIT, "Proposal", "......%d.",
		       bpdu->body.flags);
  _stp_in_display_bit (TOPOLOGY_CHANGE_BIT, "Tc", ".......%d",
		       bpdu->body.flags);

  STP_VECT_get_vector (&bpdu->body, &vect);
  Print ("  %s\n", STP_VECT_sprint (&vect));
  STP_get_times (&bpdu->body, &times);
  Print ("  MessageAge=%lu MaxAge=%lu ForwardDelay=%lu HelloTime=%lu\n\n",
	 (long) times.MessageAge, (long) times.MaxAge,
	 (long) times.ForwardDelay, (long) times.HelloTime);
}

/* stpm->sttx_type (bpdu_type): 0-rapid, 1-tcn, 2- config */
int
stp_in_dbg_stt_set_type (int stpd, int bpdu_type)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);

  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  stpm->sttx_type = bpdu_type;
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_brid (int stpd, int bridge_id_type,
			 unsigned int prio, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);

  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_brid (stpm, bridge_id_type, prio, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_rpc (int stpd, unsigned long val, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);

  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_rpc (stpm, val, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_port_id (int stpd, unsigned long val, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);
  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_port_id (stpm, val, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_message_age (int stpd, unsigned long val,
				TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);
  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_message_age (stpm, val, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_max_age (int stpd, unsigned long val, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);
  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_max_age (stpm, val, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_hellot (int stpd, unsigned long val, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);
  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_hellot (stpm, val, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_fd (int stpd, unsigned long val, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);
  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_fd (stpm, val, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_flags (int stpd, unsigned long val, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);
  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_flags (stpm, val, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_role (int stpd, unsigned long val, TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);
  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_role (stpm, val, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_set_bit (int stpd, unsigned char is_set, unsigned long val,
			TstLPortMask * port_bmp)
{
  register STPM_T *stpm;

  RSTP_CRITICAL_PATH_START;

  stpm = STP_stpm_get_the_list ();	//stpm = _stpapi_stpm_find_by_id (stpd);
  if (!stpm || STP_TEST != stpm->admin_state) {
    RSTP_CRITICAL_PATH_END;
    stp_trace ("Invalid mode=%d :(", (int) stpm->admin_state);
    return STP_Another_Error;
  }
  STP_stpm_dbg_stt_set_bit (stpm, is_set, val, port_bmp);
  RSTP_CRITICAL_PATH_END;

  return 0;
}

int
stp_in_dbg_stt_show (int stpd, TstLPortMask * port_bmp)
{
  register STPM_T *this;
  register PORT_T *port;
  register int cnt = 0;

  RSTP_CRITICAL_PATH_START;

  this = STP_stpm_get_the_list ();	//this = _stpapi_stpm_find_by_id (stpd);
  if (!this) {
    RSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
  }
  for (port = this->ports; port; port = port->next) {
    if (!K_LPortMaskGetPort (port_bmp, (port->port_index - 1))) {
      continue;
    }
    if (cnt++ > 2) {
      Print ("not more that 3 ports...\n");
      break;
    }
    switch (this->sttx_type) {
      case 0:
      Print ("RSTP   ");
      break;
      case 1:
      Print ("TCN    ");
      break;
      case 2:
      Print ("Config ");
      break;
      default:
      Print ("Unk %d", this->sttx_type);
      break;
    }
    STP_VECT_br_id_print ("rootBr", &port->portPrio.root_bridge, False);
    STP_VECT_br_id_print ("designBr", &port->portPrio.design_bridge, True);
    Print ("      RootPathCost=%ld designPort=%04lx bridgePort=%04lx\n",
	   (long) port->portPrio.root_path_cost,
	   (long) port->portPrio.design_port,
	   (long) port->portPrio.bridge_port);
    Print ("      MessageAge=%lu MaxAge=%lu ForwardDelay=%lu HelloTime=%lu\n",
	   (long) port->portTimes.MessageAge, (long) port->portTimes.MaxAge,
	   (long) port->portTimes.ForwardDelay,
	   (long) port->portTimes.HelloTime);

    Print ("      %d....... tcAck     = %c\n", port->tcAck ? 1 : 0,
	   port->tcAck ? 'Y' : 'n');
    Print ("      .%d...... Agreement = %c\n", port->synced ? 1 : 0,
	   port->synced ? 'Y' : 'n');
    Print ("      ..%d..... Forwarding= %c\n", port->forwarding ? 1 : 0,
	   port->forwarding ? 'Y' : 'n');
    Print ("      ...%d.... Learning  = %c\n", port->learning ? 1 : 0,
	   port->learning ? 'Y' : 'n');

    Print ("      ....XX.. Role      = ");

    switch (port->selectedRole) {
      default:
      Print ("?(%d) ", (int) port->selectedRole);
      case DisabledPort:
      Print ("Disabled\n");
      break;
      case AlternatePort:
      Print ("AlternateOrBackup(1)\n");
      break;
      case BackupPort:
      Print ("AlternateOrBackup(2)\n");
      break;
      case RootPort:
      Print ("Root\n");
      break;
      case DesignatedPort:
      Print ("Designated\n");
      break;
    }
    Print ("      ......%d. Proposal  = %c\n", port->proposing ? 1 : 0,
	   port->proposing ? 'Y' : 'n');
    Print ("      .......%d Tc        = %c\n", port->tcWhile ? 1 : 0,
	   port->tcWhile ? 'Y' : 'n');
  }

  RSTP_CRITICAL_PATH_END;

  return 0;
}
