/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************
 *
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 *
 **************************************************************************/

#include <linux/netfilter.h>
#include <asm/uaccess.h>

#include "krndef.h"
#include "krnportmask.h"
#include "krntype.h"

#include "base.h"
#include "stp_in.h"
#include "stpm.h"
#include "../stp_uni/stp_uni_netfilter.h"
#include "rstp_netfilter.h"

extern Bool rstp_debug;
extern Bool rstp_tick_enable;

int do_rstp_get_ctl(struct sock *, int, void *, unsigned int);
int do_rstp_set_ctl(struct sock *, int, void *, unsigned int);

struct nf_sockopt_ops rstp_sockopts =
{
list:		{NULL, NULL},
pf:		PF_INET,
set_optmin:	STP_BASE_CTL,
set_optmax:	STP_SET_MAX+1,
set:		do_rstp_set_ctl,
compat_set: NULL,
get_optmin:	STP_BASE_CTL,
get_optmax:	STP_GET_MAX+1,
get:		do_rstp_get_ctl,
compat_get: NULL,
use:		0,
cleanup_task:	NULL
};

static void rstp_get_port_status (Tuint8 ucPortId, UID_STP_CFG_T *pcfg, UID_STP_PORT_CFG_T *pcfg_pt, UID_STP_PORT_STATE_T *pstate_pt)
{
	TstLPortMask ports_bitmap;
	int rc, index;

	K_LPortMaskClrAll (&ports_bitmap);
	K_LPortMaskSetPort (&ports_bitmap, ucPortId); /* cli did -1 */

	for (index = 1; index <= pcfg->number_of_ports; index++) {
		if (!K_LPortMaskGetPort (&ports_bitmap, index - 1))
			continue;
		if (!K_LPortMaskGetPort (&(pcfg->ports), index - 1))
			continue;

		pstate_pt->port_no = index; /* index starts from 1 */

		rc = STP_IN_port_get_cfg (RSTP_STPM_ID, index, pcfg_pt);
		if (rc) {
			Print ("RSTP: can't get port config: %s\n",
					STP_IN_get_error_explanation (rc));
			continue;
		}

		rc = STP_IN_port_get_state (RSTP_STPM_ID, pstate_pt);
		if (STP_OK != rc) {
			Print ("RSTP: can't get port state: %s\n",
					STP_IN_get_error_explanation (rc));
			continue;
		}

		break; /* got it */
	}
}

int do_rstp_get_ctl (struct sock *sk, int cmd, void *user, unsigned int len)
{
	UID_STP_CFG_T cfg = {0};
	UID_STP_STATE_T state = {0};
	UID_STP_PORT_CFG_T cfg_pt = {0};
	UID_STP_PORT_STATE_T state_pt = {0};

	Tuint8 aucMac[MAC_ADDR_LEN];
	Tbool bBool;
	Tuint8 ucUByte;
	Tuint16 usUShort;
	Tuint32	ulULong;
	TstStpPortBool stStpPortBool;
	TstStpPortUByte stStpPortUByte;
	TstStpPortUShort stStpPortUShort;
	TstStpPortULong stStpPortULong;
	TstStpPortMac stStpPortMac;
	int rc;

	rc = STP_IN_stpm_get_state (RSTP_STPM_ID, &state);
	if (STP_OK != rc) {
		Print ("RSTP: can't get state:%s\n",
				STP_IN_get_error_explanation (rc));
		return 0;
	}

	rc = STP_IN_stpm_get_cfg (RSTP_STPM_ID, &cfg);
	if (STP_OK != rc) {
		Print ("RSTP: can't get config:%s\n",
				STP_IN_get_error_explanation (rc));
		return 0;
	}

	switch (cmd) {
	case STP_GET_EBL:
		bBool = state.stp_enabled == STP_ENABLED? True : False;
		copy_to_user (user, &bBool, sizeof(Tbool));
		break;

	case STP_GET_DEBUG:
		bBool = rstp_debug;
		copy_to_user (user, &bBool, sizeof(Tbool));
		break;

	case STP_GET_FORCE_VERSION:
		ucUByte = (unsigned char) cfg.force_version;
		copy_to_user (user, &ucUByte, sizeof(Tuint8));
		break;

	case STP_GET_FRWRD_DLY:
		ucUByte = (unsigned short) cfg.forward_delay;
		copy_to_user (user, &ucUByte, sizeof(Tuint8));
		break;

	case STP_GET_HELLO_TIME:
		ucUByte = (unsigned short) cfg.hello_time;
		copy_to_user (user, &ucUByte, sizeof(Tuint8));
		break;

	case STP_GET_MAX_AGE:
		ucUByte = (unsigned short) cfg.max_age;
		copy_to_user (user, &ucUByte, sizeof(Tuint8));
		break;

	case STP_GET_PRIORITY:
		usUShort = state.bridge_id.prio;
		copy_to_user (user, &usUShort, sizeof(Tuint16));
		break;

	case STP_GET_ROOT_HELLO_TIME:
		ucUByte = (unsigned short) state.hello_time;
		copy_to_user (user, &ucUByte, sizeof(Tuint8));
		break;

	case STP_GET_ROOT_FRWRD_DLY:
		ucUByte = (unsigned short) state.forward_delay;
		copy_to_user (user, &ucUByte, sizeof(Tuint8));
		break;

	case STP_GET_ROOT_MAX_AGE:
		ucUByte = (unsigned short) state.max_age;
		copy_to_user (user, &ucUByte, sizeof(Tuint8));
		break;

	case STP_GET_ROOT_PRIORITY:
		usUShort = state.designated_root.prio;
		copy_to_user (user, &usUShort, sizeof(Tuint16));
		break;

	case STP_GET_ROOT_MAC:
		memcpy (aucMac, state.designated_root.addr, 6);
		copy_to_user (user, aucMac, sizeof(Tuint8)*MAC_ADDR_LEN);
		break;

	case STP_GET_ROOT_PORT:
		ucUByte = (Tuint8)((state.root_port & 0xFF) - 1);
		copy_to_user (user, &ucUByte, sizeof(Tuint8));
		break;

	case STP_GET_ROOT_PATH_COST:
		ulULong = state.root_path_cost;
		copy_to_user (user, &ulULong, sizeof(Tuint32));
		break;

	case STP_GET_NUM_TCN:
		ulULong = state.Topo_Change_Count;
		copy_to_user (user, &ulULong, sizeof(Tuint32));
		break;

	case STP_GET_LAST_TCN:
		ulULong = state.timeSince_Topo_Change;
		copy_to_user (user, &ulULong, sizeof(Tuint32));
		break;

	case STP_GET_PORT_PATH_COST:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortULong.ulULong = state_pt.path_cost;
		copy_to_user (user, &stStpPortULong, sizeof(TstStpPortULong));
		break;

	case STP_GET_PORT_PRIORITY:
		copy_from_user (&stStpPortUByte, user, sizeof(TstStpPortUByte));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortUByte.ucUByte = cfg_pt.port_priority;
		copy_to_user (user, &stStpPortUByte, sizeof(TstStpPortUByte));
		break;

	case STP_GET_PORT_MCHECK:
		break;

	case STP_GET_PORT_EDGE:
		copy_from_user (&stStpPortBool, user, sizeof(TstStpPortBool));
		rstp_get_port_status (stStpPortBool.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortBool.bBool = state_pt.oper_edge;
		copy_to_user (user, &stStpPortBool, sizeof(TstStpPortBool));
		break;
 
	case STP_GET_PORT_NONSTP:
		copy_from_user (&stStpPortBool, user, sizeof(TstStpPortBool));
		rstp_get_port_status (stStpPortBool.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortBool.bBool = cfg_pt.admin_non_stp;
		copy_to_user (user, &stStpPortBool, sizeof(TstStpPortBool));
		break;
 
	case STP_GET_PORT_P2P:
		copy_from_user (&stStpPortBool, user, sizeof(TstStpPortBool));
		rstp_get_port_status (stStpPortBool.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortBool.bBool = state_pt.oper_point2point;
		copy_to_user (user, &stStpPortBool, sizeof(TstStpPortBool));
		break;
 
	case STP_GET_PORT_DES_PORT_PRIORITY:
		copy_from_user (&stStpPortUByte, user, sizeof(TstStpPortUByte));
		rstp_get_port_status (stStpPortUByte.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortUByte.ucUByte = state_pt.designated_port >> 8;
		copy_to_user (user, &stStpPortUByte, sizeof(TstStpPortUByte));
		break;

	case STP_GET_PORT_DES_PORT_ID:
		copy_from_user (&stStpPortUByte, user, sizeof(TstStpPortUByte));
		rstp_get_port_status (stStpPortUByte.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortUByte.ucUByte = (state_pt.designated_port & 0xFF) - 1;
		copy_to_user (user, &stStpPortUByte, sizeof(TstStpPortUByte));
		break;

	case STP_GET_PORT_DES_PATH_COST:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortULong.ulULong = state_pt.designated_cost;
		copy_to_user (user, &stStpPortULong, sizeof(TstStpPortULong));
		break;

	case STP_GET_PORT_DES_BRIDGE_PRIORITY:
		copy_from_user (&stStpPortUShort, user, sizeof(TstStpPortUShort));
		rstp_get_port_status (stStpPortUShort.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortUShort.usUShort = state_pt.designated_bridge.prio;
		copy_to_user (user, &stStpPortUShort, sizeof(TstStpPortUShort));
		break;

	case STP_GET_PORT_DES_BRIDGE_MAC:
		copy_from_user (&stStpPortMac, user, sizeof(TstStpPortMac));
		rstp_get_port_status (stStpPortMac.ucPortId, &cfg, &cfg_pt, &state_pt);
		memcpy (stStpPortMac.aucMac, state_pt.designated_bridge.addr, 6);
		copy_to_user (user, &stStpPortMac, sizeof(TstStpPortMac));
		break;
 
	case STP_GET_PORT_STATE:
		copy_from_user (&stStpPortUByte, user, sizeof(TstStpPortUByte));
		rstp_get_port_status (stStpPortUByte.ucPortId, &cfg, &cfg_pt, &state_pt);
		switch (state_pt.state)
		{
		case UID_PORT_DISCARDING:
			stStpPortUByte.ucUByte = STP_PORT_STATE_DISCARDING;
			break;
		case UID_PORT_LEARNING:
			stStpPortUByte.ucUByte = STP_PORT_STATE_LEARNING;
			break;
		case UID_PORT_FORWARDING:
			stStpPortUByte.ucUByte = STP_PORT_STATE_FORWARDING;
			break;
		case UID_PORT_NON_STP:
			stStpPortUByte.ucUByte = STP_PORT_STATE_NON_STP;
			break;
		case UID_PORT_DISABLED:
		default:
			stStpPortUByte.ucUByte = STP_PORT_STATE_DISABLED;
			break;
		}
		copy_to_user (user, &stStpPortUByte, sizeof(TstStpPortUByte));
		break;

	case STP_GET_PORT_ROLE:
		copy_from_user (&stStpPortUByte, user, sizeof(TstStpPortUByte));
		rstp_get_port_status (stStpPortUByte.ucPortId, &cfg, &cfg_pt, &state_pt);
		switch (state_pt.role)
		{
		case 'A':
			stStpPortUByte.ucUByte = STP_PORT_ROLE_ALTERNATE;
			break;
		case 'B':
			stStpPortUByte.ucUByte = STP_PORT_ROLE_BACKUP;
			break;
		case 'D':
			stStpPortUByte.ucUByte = STP_PORT_ROLE_DESIGNATED;
			break;
		case 'R':
			stStpPortUByte.ucUByte = STP_PORT_ROLE_ROOT;
			break;
		case '-':
			stStpPortUByte.ucUByte = STP_PORT_ROLE_NON_STP;
			break;
		default:
			stStpPortUByte.ucUByte = STP_PORT_ROLE_DISABLED;
			break;
		}
		copy_to_user (user, &stStpPortUByte, sizeof(TstStpPortUByte));
		break;

	case STP_GET_PORT_UPTIME:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortULong.ulULong = state_pt.uptime;
		copy_to_user (user, &stStpPortULong, sizeof(TstStpPortULong));
		break;

	case STP_GET_PORT_RX_CFG:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortULong.ulULong = state_pt.rx_cfg_bpdu_cnt;
		copy_to_user (user, &stStpPortULong, sizeof(TstStpPortULong));
		break;
 
	case STP_GET_PORT_RX_RST:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortULong.ulULong = state_pt.rx_rstp_bpdu_cnt;
		copy_to_user (user, &stStpPortULong, sizeof(TstStpPortULong));
		break;
 
	case STP_GET_PORT_RX_TCN:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortULong.ulULong = state_pt.rx_tcn_bpdu_cnt;
		copy_to_user (user, &stStpPortULong, sizeof(TstStpPortULong));
		break;
 
	case STP_GET_PORT_TX_CFG:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortULong.ulULong = state_pt.tx_cfg_bpdu_cnt;
		copy_to_user (user, &stStpPortULong, sizeof(TstStpPortULong));
		break;
 
	case STP_GET_PORT_TX_RST:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortULong.ulULong = state_pt.tx_rstp_bpdu_cnt;
		copy_to_user (user, &stStpPortULong, sizeof(TstStpPortULong));
		break;
 
	case STP_GET_PORT_TX_TCN:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		rstp_get_port_status (stStpPortULong.ucPortId, &cfg, &cfg_pt, &state_pt);
		stStpPortULong.ulULong = state_pt.tx_tcn_bpdu_cnt;
		copy_to_user (user, &stStpPortULong, sizeof(TstStpPortULong));
		break;

	default:
		break;
	}

	return 0;
}

void set_bridge_cfg_value (unsigned long value, unsigned long mask)
{
	UID_STP_CFG_T uid_cfg;
	char *val_name;
	int rc;

	uid_cfg.field_mask = mask;
	switch (mask) {
	case BR_CFG_STATE:
		uid_cfg.stp_enabled = value;
		val_name = "state";
		break;
	case BR_CFG_PRIO:
		uid_cfg.bridge_priority = value;
		val_name = "priority";
		break;
	case BR_CFG_AGE:                                                    
		uid_cfg.max_age = value;
		val_name = "max age";
		break;
	case BR_CFG_HELLO:
		uid_cfg.hello_time = value;
		val_name = "hello time";
		break;
	case BR_CFG_DELAY:
		uid_cfg.forward_delay = value;
		val_name = "forward delay";
		break;
	case BR_CFG_FORCE_VER:
		if (1 == value)
			uid_cfg.force_version = FORCE_STP_COMPAT;
		else if (2 == value)
			uid_cfg.force_version = NORMAL_RSTP;
		else
			uid_cfg.force_version = NORMAL_RSTP;
		val_name = "force version";
		break;
	case BR_CFG_AGE_MODE:
	case BR_CFG_AGE_TIME:
	case BR_CFG_HOLD_TIME:
	default:
		Print ("RSTP: Invalid value mask 0X%lx\n", mask);
		return;
	}

	rc = STP_IN_stpm_set_cfg (0, &uid_cfg);
	if (STP_OK != rc)
		Print ("RSTP: Can't change bridge %s:%s\n", val_name,
				STP_IN_get_error_explanation (rc));
}


static void
set_port_cfg_value (struct set_rstp_port_t pvalue, unsigned long mask)
{
	int port_index = pvalue.port + 1;
	unsigned long value = pvalue.value;
	UID_STP_PORT_CFG_T uid_cfg;
	int rc;
	char *val_name; 

	if (port_index > 0) {
		K_LPortMaskClrAll (&uid_cfg.port_bmp);
		K_LPortMaskSetPort (&uid_cfg.port_bmp, port_index - 1);
	} else {
/*                K_LPortMaskSetAll (&uid_cfg.port_bmp);*/
		K_LPortMaskClrAll (&uid_cfg.port_bmp);
	}

	uid_cfg.field_mask = mask;
	switch (mask) {
	case PT_CFG_MCHECK:
		uid_cfg.admin_mcheck = value;
		val_name = "mcheck";
		break;
	case PT_CFG_COST:
		uid_cfg.admin_port_path_cost = value;
		val_name = "path cost";
		break;
	case PT_CFG_PRIO:
		uid_cfg.port_priority = value;
		val_name = "priority";
		break;
	case PT_CFG_P2P:
		if (NF_P2P_FORCE_TRUE == value)
			uid_cfg.admin_point2point = P2P_FORCE_TRUE;
		else if (NF_P2P_FORCE_FALSE == value)
			uid_cfg.admin_point2point = P2P_FORCE_FALSE;
		else /* NF_P2P_AUTO */
			uid_cfg.admin_point2point = P2P_AUTO;
		val_name = "p2p flag";
		break;
	case PT_CFG_EDGE:
		uid_cfg.admin_edge = value;
		val_name = "adminEdge";
		break;
	case PT_CFG_NON_STP:
		uid_cfg.admin_non_stp = value;
		val_name = "adminNonStp";
		break;
	case PT_CFG_STATE:
	default:
		Print ("Invalid value mask 0X%lx\n", mask);
		return;
	}
	rc = STP_IN_set_port_cfg (0, &uid_cfg);
	if (STP_OK != rc)
		Print ("can't change rstp port[s] %s: %s\n",
				val_name, STP_IN_get_error_explanation (rc));
}


int do_rstp_set_ctl (struct sock *sk, int cmd, void *user, unsigned int len)
{
	UID_STP_CFG_T cfg;
	int rc;
	unsigned long value = 0;
	struct set_rstp_port_t pvalue = {0};
	Tbool bBool;
	TstStpPortBool stStpPortBool = {0};
	TstStpPortUByte stStpPortUByte = {0};
	TstStpPortULong stStpPortULong = {0};
	TstStpPortMachine stStpPortMachine = {0};
	TstLPortMask LMask;

	switch (cmd) {
	case STP_SET_EBL:
		copy_from_user (&bBool, user, sizeof(Tbool));
		if (bBool)
		{
			cfg.field_mask = BR_CFG_STATE;
			cfg.stp_enabled = STP_ENABLED;
			cfg.number_of_ports = MAX_LOGIC_PORT;
			snprintf (cfg.name, BRIDGE_NAME_LEN, BRIDGE_NAME);
			rc = STP_IN_stpm_set_cfg (RSTP_STPM_ID, &cfg);
			if (STP_OK != rc) 
			{
				Print ("RSTP: can't enable:%s\n",
					STP_IN_get_error_explanation (rc));
				break;
			}
			rstp_tick_enable = True;
		}
		else
		{
			cfg.field_mask = BR_CFG_STATE;
			cfg.stp_enabled = STP_DISABLED;
			cfg.number_of_ports = MAX_LOGIC_PORT;
			snprintf (cfg.name, BRIDGE_NAME_LEN, BRIDGE_NAME);
			rc = STP_IN_stpm_set_cfg (RSTP_STPM_ID, &cfg);
			if (STP_OK != rc) 
			{
				Print ("RSTP: can't disable:%s\n",
					STP_IN_get_error_explanation (rc));
				break;
			}
			rstp_tick_enable = False;
			STP_IN_set_forward_all_port ();
		}
		break;

	case STP_SET_DEBUG:
		copy_from_user (&bBool, user, sizeof(Tbool));
		if (bBool)
			rstp_debug = True;
		else
			rstp_debug = False;
		break;

	case STP_SET_FORCE_VERSION:
		copy_from_user (&value, user, sizeof(Tuint8));
		set_bridge_cfg_value (value, BR_CFG_FORCE_VER);
		break;

	case STP_SET_FRWRD_DLY:
		copy_from_user (&value, user, sizeof(Tuint8));
		set_bridge_cfg_value (value, BR_CFG_DELAY);
		break;

	case STP_SET_HELLO_TIME:
		copy_from_user (&value, user, sizeof(Tuint8));
		set_bridge_cfg_value (value, BR_CFG_HELLO);
		break;

	case STP_SET_MAX_AGE:
		copy_from_user (&value, user, sizeof(Tuint8));
		set_bridge_cfg_value (value, BR_CFG_AGE);
		break;

	case STP_SET_PRIORITY:
		copy_from_user (&value, user, sizeof(Tuint16));
		set_bridge_cfg_value (value, BR_CFG_PRIO);
		break;

	case STP_SET_PORT_PATH_COST:
		copy_from_user (&stStpPortULong, user, sizeof(TstStpPortULong));
		pvalue.port = stStpPortULong.ucPortId;
		pvalue.value = stStpPortULong.ulULong;
		set_port_cfg_value (pvalue, PT_CFG_COST);
		break;

	case STP_SET_PORT_PRIORITY:
		copy_from_user (&stStpPortUByte, user, sizeof(TstStpPortUByte));
		pvalue.port = stStpPortUByte.ucPortId;
		pvalue.value = stStpPortUByte.ucUByte;
		set_port_cfg_value (pvalue, PT_CFG_PRIO);
		break;

	case STP_SET_PORT_MCHECK:
		copy_from_user (&stStpPortBool, user, sizeof(TstStpPortBool));
		pvalue.port = stStpPortBool.ucPortId;
		pvalue.value = stStpPortBool.bBool;
		set_port_cfg_value (pvalue, PT_CFG_MCHECK);
		break;

	case STP_SET_PORT_EDGE:
		copy_from_user (&stStpPortBool, user, sizeof(TstStpPortBool));
		pvalue.port = stStpPortBool.ucPortId;
		pvalue.value = stStpPortBool.bBool;
		set_port_cfg_value (pvalue, PT_CFG_EDGE);
		break;

	case STP_SET_PORT_NONSTP:
		copy_from_user (&stStpPortBool, user, sizeof(TstStpPortBool));
		pvalue.port = stStpPortBool.ucPortId;
		pvalue.value = stStpPortBool.bBool;
		set_port_cfg_value (pvalue, PT_CFG_NON_STP);
		break;

	case STP_SET_PORT_P2P:
		copy_from_user (&stStpPortUByte, user, sizeof(TstStpPortUByte));
		pvalue.port = stStpPortUByte.ucPortId;
		pvalue.value = stStpPortUByte.ucUByte;
		set_port_cfg_value (pvalue, PT_CFG_P2P);
		break;

	case STP_SET_PORT_DEBUG:
		copy_from_user (&stStpPortMachine, user, sizeof(TstStpPortMachine));
		K_LPortMaskClrAll (&LMask);
		K_LPortMaskSetPort (&LMask, stStpPortMachine.ucPortId);
		if (stStpPortMachine.bEbl)
			STP_IN_dbg_set_port_trace (stStpPortMachine.aucMachName,
					True, RSTP_STPM_ID, &LMask, True);
		else
			STP_IN_dbg_set_port_trace (stStpPortMachine.aucMachName,
					False, RSTP_STPM_ID, &LMask, True);
		break;

	default:
		break;
	}

	return 0;
}

