/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************
 *
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 *
 **************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <asm/unaligned.h>
#include <linux/netfilter.h>
#include <linux/spinlock.h>

#include "krndef.h"
#include "krnmac.h"
#include "krnportmask.h"
#include "krnstp.h"
#include "krnsys.h"

#include "base.h"
#include "stp_in.h"
#include "stp_bpdu.h"
//#include "../stp_uni/stp_uni_netfilter.h"
#include "rstp_netfilter.h"

#define Config_bpdu_type    0
#define Tcn_bpdu_type       128
#define BPDU_PRIORITY       5


extern void (*gvrp_stp_port_forwarding)(int );
extern void (*gvrp_stp_port_disabled)(int );
extern void (*gmrp_stp_port_forwarding)(int );
extern void (*gmrp_stp_port_disabled)(int );
extern void (*igmp_stp_port_forwarding)(int );
extern void (*igmp_stp_port_disabled)(int );

unsigned char rstp_bridge_mac[6];
static spinlock_t rstp_ieee_sem;
TstLPortMask enabled_ports;
extern struct nf_sockopt_ops rstp_sockopts; 

static int periodical_thread_pid;
static int port_thread_pid;
Bool rstp_tick_enable;


static void rstp_sleep(unsigned howlong)
{
  if(in_interrupt())
   {
    return;
   }

	current->state = TASK_INTERRUPTIBLE;
	schedule_timeout(howlong);
}


/*
 * this thread will use PortGetLink to find out any port changes 
 * and report them to the ieee 802.1w code.
 */
static int rstp_port_thread(void * vp)
{
	int i;
	Tuint8 new_link, new_ebl, new_speed, new_duplex;
	Tuint8 link[MAX_LOGIC_PORT], ebl[MAX_LOGIC_PORT];
	Tuint8 speed[MAX_LOGIC_PORT], duplex[MAX_LOGIC_PORT];

	memset (link, 0, sizeof(link));
	memset (ebl, 0, sizeof(ebl));
	memset (speed, 0, sizeof(speed));
	memset (duplex, 0, sizeof(duplex));

	while (1) {
		if (!rstp_tick_enable) {
			rstp_sleep (200);
			continue;
		}

		for (i = 0; i < MAX_LOGIC_PORT; i++) {
			new_link = cclmx_PortGetLink (i);
			new_ebl = cclmx_PortGetEbl (i);
			cclmx_PortGetSpeedDuplex (i, &new_speed, &new_duplex);

			if (new_duplex != duplex[i] && new_duplex != CCLMX_PORT_DUPLEX_UNKNOWN) {
				STP_IN_changed_port_duplex (i+1);
				duplex[i] = new_duplex;
			}

			if (new_speed != speed[i]) {
				switch (new_speed) {
				case CCLMX_PORT_SPEED_0:
					K_LPortMaskClrPort (&enabled_ports, i);
					STP_IN_changed_port_speed (i+1, 0);
					break;
				case CCLMX_PORT_SPEED_10:
					STP_IN_changed_port_speed (i+1, 10);
					break;
				case CCLMX_PORT_SPEED_20:
					STP_IN_changed_port_speed (i+1, 20);
					break;
				case CCLMX_PORT_SPEED_30:
					STP_IN_changed_port_speed (i+1, 30);
					break;
				case CCLMX_PORT_SPEED_40:
					STP_IN_changed_port_speed (i+1, 40);
					break;
				case CCLMX_PORT_SPEED_100:
					STP_IN_changed_port_speed (i+1, 100);
					break;
				case CCLMX_PORT_SPEED_200:
					STP_IN_changed_port_speed (i+1, 200);
					break;
				case CCLMX_PORT_SPEED_300:
					STP_IN_changed_port_speed (i+1, 300);
					break;
				case CCLMX_PORT_SPEED_400:
					STP_IN_changed_port_speed (i+1, 400);
					break;
				case CCLMX_PORT_SPEED_1000:
					STP_IN_changed_port_speed (i+1, 1000);
					break;
				case CCLMX_PORT_SPEED_2000:
					STP_IN_changed_port_speed (i+1, 2000);
					break;
				}
				speed[i] = new_speed;
			}

			if (new_link == link[i] && new_ebl == ebl[i]) {
				rstp_sleep (10);
				continue;
			}
			else if (new_link == CCLMX_PORT_LINKUP && new_ebl == CCLMX_PORT_ON) {
				/* state: off -> on */
				spin_lock(&rstp_ieee_sem);
				K_LPortMaskSetPort (&enabled_ports, i);
				STP_IN_enable_port (i+1, True);
				spin_unlock(&rstp_ieee_sem);
			}
			else if (link[i] == CCLMX_PORT_LINKUP && ebl[i] == CCLMX_PORT_ON) {
				/* state: on -> off */
				spin_lock(&rstp_ieee_sem);
				K_LPortMaskClrPort (&enabled_ports, i);
				STP_IN_enable_port (i+1, False);
				spin_unlock(&rstp_ieee_sem);
			}
			else
				; /* state unchanged */

			link[i] = new_link;
			ebl[i] = new_ebl;
		}
	}
	return 0;
}


/*
 * periodically run
 */
static int rstp_periodical_thread(void * vp)
{
	while(1) 
	{
		if(!rstp_tick_enable) {
			rstp_sleep(200);
			continue;
		}

		spin_lock(&rstp_ieee_sem); /* protect the ieee802.1w code */
		STP_IN_one_second();
		spin_unlock(&rstp_ieee_sem);

		rstp_sleep(95);
	}
	return 0;
}


/* The CallBack function.
 * This function will get bpdu packets from
 * the VT6526 and report them to the ieee 802.1w code.
 */
static int ccl_rstp_rx_bpdu (struct sk_buff *skb)
{
	BPDU_T bpdu;
	unsigned char * ptr;
	int port_no;

	UID_STP_STATE_T state = {0};

	/* if stp is not enabled, don't waste time */
	if (STP_OK != STP_IN_stpm_get_state (RSTP_STPM_ID, &state)) {
		//Print("RSTP not ready\n");
		return CCL_MX_DROP;
	}
	if (!state.stp_enabled)
		return CCL_MX_DROP;

#if 0 //def STP_DBG
	Print("RSTP: program receives a packet!! %d %d %d %d %d %d\n",
			CCL_LP(skb)->wSize, CCL_LP(skb)->wVid,
			CCL_LP(skb)->byPriority, CCL_LP(skb)->bySrcPortId,
			CCL_LP(skb)->byFwdReason,CCL_LP(skb)->bIfTagged);
#endif

	if (CCL_LP(skb)->wSize < 17) {
		Print("RSTP: Error, packet too short!\n");
		return CCL_MX_DROP;
	}

	/* SrcPortId starts from 0 in VIA driver,
	 * but port_index starts from 1 in rstp */
	port_no = CCL_LP(skb)->bySrcPortId + 1;

	if(CCL_LP(skb)->bIfTagged) {
		//Print("RSTP: tagged BPDU rcvd!\n");
		return CCL_MX_DROP;
	}
	else {
		/* skip DA,SA */
		ptr = (unsigned char *)(skb->mac.raw) + MAC_HEADER_T_SZ;
		/* Becuse cpu will auto-align structs, we copy them one by one
		 * !!And, avoid to use sizeof(struct name) */
		memcpy (&bpdu.eth, ptr, ETH_HEADER_T_SZ);
		ptr += ETH_HEADER_T_SZ;
		memcpy (&bpdu.hdr, ptr, BPDU_HEADER_T_SZ);
		ptr += BPDU_HEADER_T_SZ;
		memcpy (&bpdu.body, ptr, BPDU_BODY_T_SZ);
#if 0
			Print("bpdu fg:%x root id:%02x%02x%02x%02x%02x%02x%02x%02x  ",
					bpdu.body.flags, bpdu.body.root_id[0],
					bpdu.body.root_id[1], bpdu.body.root_id[2],
					bpdu.body.root_id[3], bpdu.body.root_id[4],
					bpdu.body.root_id[5], bpdu.body.root_id[6],
					bpdu.body.root_id[7]);
			Print("ptr fg:%x root id:%02x%02x%02x%02x%02x%02x%02x%02x\n",
					*ptr, *(ptr+1), *(ptr+2), *(ptr+3), *(ptr+4),
					*(ptr+5), *(ptr+6), *(ptr+7), *(ptr+8));
			Print("bpdu RPC:%02x%02x%02x%02x  ptrRPC:%02x%02x%02x%02x\n",
					bpdu.body.root_path_cost[0], bpdu.body.root_path_cost[1],
					bpdu.body.root_path_cost[2], bpdu.body.root_path_cost[3],
					*(ptr+9), *(ptr+10), *(ptr+11), *(ptr+12));
			Print("bpdu bridge id:%02x%02x%02x%02x%02x%02x%02x%02x  ",
					bpdu.body.bridge_id[0], bpdu.body.bridge_id[1],
					bpdu.body.bridge_id[2], bpdu.body.bridge_id[3],
					bpdu.body.bridge_id[4], bpdu.body.bridge_id[5],
					bpdu.body.bridge_id[6], bpdu.body.bridge_id[7]);
			Print("ptr bridge id:%02x%02x%02x%02x%02x%02x%02x%02x\n",
					*(ptr+13), *(ptr+14), *(ptr+15), *(ptr+16),
					*(ptr+17), *(ptr+18), *(ptr+19), *(ptr+20));
			Print("bpdu port id:%02x%02x  ptr port id:%02x%02x\n",
					bpdu.body.port_id[0], bpdu.body.port_id[1],
					*(ptr+21), *(ptr+22));
			Print("bpdu msg age:%02x%02x  ptr msg age:%02x%02x\n",
					bpdu.body.message_age[0], bpdu.body.message_age[1],
					*(ptr+23), *(ptr+24));
			Print("bpdu max age:%02x%02x  ptr max age:%02x%02x\n",
					bpdu.body.max_age[0], bpdu.body.max_age[1],
					*(ptr+25), *(ptr+26));
			Print("bpdu hello:%02x%02x  ptr hello:%02x%02x\n",
					bpdu.body.hello_time[0], bpdu.body.hello_time[1],
					*(ptr+27), *(ptr+28));
			Print("bpdu fwd delay:%02x%02x  ptr fwd delay:%02x%02x\n",
					bpdu.body.forward_delay[0], bpdu.body.forward_delay[1],
					*(ptr+29), *(ptr+30));
#endif
		ptr += BPDU_BODY_T_SZ;
		memcpy (&bpdu.ver_1_length, ptr, BPDU_V1_LEN_SZ);
	}

	STP_IN_rx_bpdu (port_no, &bpdu , CCL_LP(skb)->wSize -
			sizeof(MAC_HEADER_T));

	return CCL_MX_DROP;
}


void ccl_rstp_tx_bpdu (int port_no, unsigned char *bpdu_packet, size_t bpdu_len)
{
	struct sk_buff *skb;
	BPDU_T * bpdu = (BPDU_T *) (bpdu_packet + MAC_HEADER_T_SZ);
	unsigned char *ptr;

	/* should not happen */
	if (port_no <= 0) return;

	port_no--; /* VIA driver starts from 0 */
	bpdu_len += MAC_HEADER_T_SZ + 2; /* 2 is 802.3 len in BPDU_T */

#if 0 //def STP_DBG
	Print("RSTP: xmit a pkt %d in len on port %d\n", bpdu_len, port_no+1);
#endif

	/* Allocate and zero memory for message */
	{
		skb = dev_alloc_skb (bpdu_len);
		if (NULL == skb)
		{
			Print("RSTP: memory allocation error while tx BPDU\n");
			return;
		}
		ptr = skb_put (skb, bpdu_len);
		memset (ptr, 0, bpdu_len);
	}

	/* copy data */
	memcpy (ptr, bpdu_packet, MAC_HEADER_T_SZ);
	ptr += MAC_HEADER_T_SZ;
	memcpy (ptr, &bpdu->eth, ETH_HEADER_T_SZ);
	ptr += ETH_HEADER_T_SZ;
	memcpy (ptr, &bpdu->hdr, BPDU_HEADER_T_SZ);
	if (BPDU_CONFIG_TYPE == bpdu->hdr.bpdu_type ||
			BPDU_RSTP == bpdu->hdr.bpdu_type)
	{
		ptr += BPDU_HEADER_T_SZ;
		memcpy (ptr, &bpdu->body, BPDU_BODY_T_SZ);
	}
	if (BPDU_RSTP == bpdu->hdr.bpdu_type)
	{
		ptr += BPDU_BODY_T_SZ;
		memcpy (ptr, &bpdu->ver_1_length, BPDU_V1_LEN_SZ);
	}

	K_LPortMaskClrAll (&CCL_LP(skb)->dwDstPortMsk);
	K_LPortMaskSetPort (&CCL_LP(skb)->dwDstPortMsk, port_no);
	K_LPortMaskClrAll (&CCL_LP(skb)->dwTagPortMsk);

	CCL_LP(skb)->bIfTagged = 0;
	CCL_LP(skb)->wVid = 1;

	cclmx_TxPkt (skb);
}


long CCL_RSTP_GetSpeed (int port_index)
{
	Tuint8 speed, duplex;

	cclmx_PortGetSpeedDuplex (port_index - 1, &speed, &duplex);
	switch (speed) {
	case CCLMX_PORT_SPEED_0:	return 0L;
	case CCLMX_PORT_SPEED_10:	return 10L;
	case CCLMX_PORT_SPEED_20:	return 20L;
	case CCLMX_PORT_SPEED_30:	return 30L;
	case CCLMX_PORT_SPEED_40:	return 40L;
	case CCLMX_PORT_SPEED_100:	return 100L;
	case CCLMX_PORT_SPEED_200:	return 200L;
	case CCLMX_PORT_SPEED_300:	return 300L;
	case CCLMX_PORT_SPEED_400:	return 400L;
	case CCLMX_PORT_SPEED_1000:	return 1000L;
	case CCLMX_PORT_SPEED_2000:	return 2000L;
	default:			return 100L;
	}
}


int CCL_RSTP_GetDuplex (int port_index)
{
	Tuint8 speed, duplex;

	/* driver's port number starts from 0, but rstp's index starts from 1 */
	cclmx_PortGetSpeedDuplex (port_index - 1, &speed, &duplex);
	if (duplex == CCLMX_PORT_DUPLEX_FULL)
		return 1;
	else
		return 0;
}


void CCL_RSTP_SetPortState(int port_no, IN RSTP_PORT_STATE state)
{
	if(port_no == 0)
		return;

	port_no--; 

	switch (state) 
	{
	case UID_PORT_DISABLED:
		//HIvSTP_SetPortState (port_no, DISABLED);
		cclmx_PortSetStpState (port_no, STP_PORT_DISABLED);
		if (gvrp_stp_port_disabled != NULL)
			gvrp_stp_port_disabled (port_no);
		if (gmrp_stp_port_disabled != NULL)
			gmrp_stp_port_disabled (port_no);
		if (igmp_stp_port_disabled != NULL)
			igmp_stp_port_disabled (port_no);
		break;

	case UID_PORT_DISCARDING:
		//HIvSTP_SetPortState (port_no, BLOCKING);
		cclmx_PortSetStpState (port_no, STP_PORT_BLOCKING);
		if (gvrp_stp_port_disabled != NULL)
			gvrp_stp_port_disabled (port_no);
		if (gmrp_stp_port_disabled != NULL)
			gmrp_stp_port_disabled (port_no);
		if (igmp_stp_port_disabled != NULL)
			igmp_stp_port_disabled (port_no);
		break;

	case UID_PORT_LEARNING:
		//HIvSTP_SetPortState (port_no, LEARNING);
		cclmx_PortSetStpState (port_no, STP_PORT_LEARNING);
		break;

	case UID_PORT_FORWARDING:
		//HIvSTP_SetPortState (port_no, FORWARDING);
		cclmx_PortSetStpState (port_no, STP_PORT_FORWARDING);
		if (gvrp_stp_port_forwarding != NULL)
			gvrp_stp_port_forwarding (port_no);
		if (gmrp_stp_port_forwarding != NULL)
			gmrp_stp_port_forwarding (port_no);
		if (igmp_stp_port_forwarding != NULL)
			igmp_stp_port_forwarding (port_no);
		break;

	case UID_PORT_NON_STP:
	default:
		break;
	}
}  


static int rstp_thread_create(void)
{
	unsigned long ulFourOctetsHigh = 0x80008030;
	unsigned long ulFourOctetsLow =  0x01030467;

	/* get switch MAC address and store into rstp_bridge_mac */
	ulFourOctetsHigh = (0x80000000) |
				(rstp_bridge_mac[0] << 8) |
				(rstp_bridge_mac[1]);
	ulFourOctetsLow = (rstp_bridge_mac[2] << 24) |
				(rstp_bridge_mac[3] << 16) |
				(rstp_bridge_mac[4] << 8) |
				(rstp_bridge_mac[5]);

	spin_lock_init(&rstp_ieee_sem);

	periodical_thread_pid = kernel_thread (
			(int (*)(void *))rstp_periodical_thread, NULL, 0);
	port_thread_pid = kernel_thread (
			(int (*)(void *))rstp_port_thread, NULL, 0);

	return 1;
}


static void rstp_cleanup(void)
{
	//sem_destroy(&rstp_ieee_sem);
	//up(&rstp_ieee_sem);

}


/***************Module parameter information*********************/
MODULE_AUTHOR ("Winfred Lu <winfred@itri.org.tw>");
MODULE_DESCRIPTION ("Rapid Spanning-Tree Protocol");
MODULE_LICENSE("GPL");

static char mac[6] = {0x01,0x80,0xC2,0x00,0x00,0x00};
static char mask[6] = {0xFF,0xFF,0xFF,0xFF,0xFF,0xFF};
/****************************************************************/


int rstp_init(void)
{
	UID_STP_CFG_T ipc_cfg;
	int number_of_ports = MAX_LOGIC_PORT;
	int iii;

	K_StpSetEbl(Enable); // Enable BPDU forward to CPU

	STP_IN_init (number_of_ports);
	K_LPortMaskClrAll (&enabled_ports);
	K_LPortMaskClrAll (&ipc_cfg.ports);
	for (iii = 0; iii < number_of_ports; iii++) 
	{
		K_LPortMaskSetPort (&ipc_cfg.ports, iii);
	}

	ipc_cfg.field_mask = BR_CFG_STATE;
	ipc_cfg.stp_enabled = STP_ENABLED;
	ipc_cfg.number_of_ports = number_of_ports;
	snprintf (ipc_cfg.name, BRIDGE_NAME_LEN, BRIDGE_NAME);
	iii = STP_IN_stpm_set_cfg (RSTP_STPM_ID, &ipc_cfg);
	if (STP_OK != iii) 
	{
		Print ("FATAL: can't enable:%s\n", STP_IN_get_error_explanation (iii));
		return (-1);
	}
	rstp_tick_enable=1;

	return 0;
}


int rstp_stop(void)
{
	int i;
	int rc;

	K_StpSetEbl(Disable); // Disable BPDU forward to CPU
	rstp_tick_enable=0;

	for(i=0; i<MAX_LOGIC_PORT; i++)
		CCL_RSTP_SetPortState(i+1, UID_PORT_FORWARDING);

	rc = STP_IN_stpm_delete (0);
	if (STP_OK != rc) 
	{
		Print ("FATAL: can't delete:%s\n", STP_IN_get_error_explanation (rc));
		return (-1);
	}
	return (0);
}


static int __init ccl_rstp_init_module(void)
{
	//Print("============= CCL RSTP program ============\n");	

	rstp_tick_enable = 0;
	K_SysGetMac(rstp_bridge_mac);

	rstp_init();
	rstp_thread_create();
	if(nf_register_sockopt(&rstp_sockopts))
	{
		Print(" cannot register sockopt \n");
		return False;
	}
	ccl_mx_register(mac, mask, ccl_rstp_rx_bpdu);

	/* 
	 * I need to check the reserved range of 
	 * address in the bpdu DMAC described in 
	 * 1998 IEEE standard sec. 7.12.6 p.54. 
	 * This is to pass the ANVL test case 1.5
	 */

	return 0;
}	


static void __exit ccl_rstp_cleanup_module(void)
{
	rstp_stop();
	rstp_cleanup();
	//Print("============= CCL RSTP program ============\n");	
}


module_init(ccl_rstp_init_module);
module_exit(ccl_rstp_cleanup_module);



