/************************************************************************ 
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Port Role Selection state machine : 17.22 */

#include "base.h"
#include "stpm.h"

#define STATES { \
  CHOOSE(INIT_BRIDGE),      \
  CHOOSE(ROLE_SELECTION),   \
}

#define GET_STATE_NAME STP_rolesel_get_state_name
#include "choose.h"

#ifdef STP_DBG
void
stp_dbg_break_point (PORT_T * port, STPM_T * stpm)
{
}
#endif

static Bool
_is_backup_port (PORT_T * port, STPM_T * this)
{
  if (!STP_VECT_compare_bridge_id
      (&port->portPrio.design_bridge, &this->BrId)) {
#if 0				/* def STP_DBG */
    stp_dbg_break_point (port, this);
#endif
    return True;
  } else {
    return False;
  }
}

static const char *
getRoleName (PORT_ROLE_T role)
{
  switch (role) {
    case DisabledPort:
    return "Disabled";
    break;
    case AlternatePort:
    return "Alternate";
    break;
    case BackupPort:
    return "Backup";
    break;
    case RootPort:
    return "Root";
    break;
    case DesignatedPort:
    return "Designated";
    break;
    case NonStpPort:
    return "NonStp";
    break;
    default:
    stp_trace ("Can't get role name %d", (int) role);
    return "???";
  }
}

static void
setRoleSelected (char *reason, STPM_T * stpm, PORT_T * port,
		 PORT_ROLE_T newRole)
{
  if (newRole == port->selectedRole && NonStpPort != newRole)
    return;

  port->selectedRole = newRole;

#ifdef STP_DBG
  if (port->roletrns->debug || port->p2p->debug || port->sttrans->debug) {
    stp_trace ("%s(%s-%s) %s=>%s",
	       reason, stpm->name, port->port_name,
	       getRoleName (port->role), getRoleName (newRole));
  }
#endif

  if (newRole == NonStpPort)
    port->role = newRole;
}

static void
updtRoleDisableBridge (STPM_T * this)
{				/* 17.10.20 */
  register PORT_T *port;

  for (port = this->ports; port; port = port->next) {
    port->selectedRole = DisabledPort;
  }
}

static void
clearReselectBridge (STPM_T * this)
{				/* 17.19.1 */
  register PORT_T *port;

  for (port = this->ports; port; port = port->next) {
    port->reselect = False;
  }
}

static void
updtRootPrio (STATE_MACH_T * this)
{
  PRIO_VECTOR_T rootPathPrio;	/* 17.4.2.2 */
  register PORT_T *port;
  register STPM_T *stpm;
  register unsigned int dm;

  int cmp;

  stpm = this->owner.stpm;

  for (port = stpm->ports; port; port = port->next) {
    if (port->admin_non_stp) {
      continue;
    }

    if (Disabled == port->infoIs) {
      continue;
    }
    if (Aged == port->infoIs) {
      continue;
    }
    if (Mine == port->infoIs) {
      continue;
    }

    if (!STP_VECT_compare_bridge_id
	(&port->portPrio.design_bridge, &stpm->BrId)) {
      continue;
    }

    STP_VECT_copy (&rootPathPrio, &port->portPrio);
    rootPathPrio.root_path_cost += port->operPCost;

    cmp = STP_VECT_compare_vector (&rootPathPrio, &stpm->rootPrio, True);
    if (cmp < 0) {
      STP_VECT_copy (&stpm->rootPrio, &rootPathPrio);
      STP_copy_times (&stpm->rootTimes, &port->portTimes);
      dm = (stpm->rootTimes.MaxAge + 8) / 16;
      if (!dm)
	dm = 1;
      stpm->rootTimes.MessageAge += dm;
    }
  }
}

void
updtRolesBridge (STATE_MACH_T * this, Bool debug)
{				/* 17.19.21 */
  register PORT_T *port;
  register STPM_T *stpm;
  PORT_ID old_root_port;	/* for tracing of root port changing */

  stpm = this->owner.stpm;
  old_root_port = stpm->rootPortId;

  STP_VECT_create (&stpm->rootPrio, &stpm->BrId, 0, &stpm->BrId, 0, 0);
  STP_copy_times (&stpm->rootTimes, &stpm->BrTimes);
  stpm->rootPortId = 0;

  updtRootPrio (this);

  for (port = stpm->ports; port; port = port->next) {
    if (port->admin_non_stp) {
      continue;
    }
    STP_VECT_create (&port->designPrio,
		     &stpm->rootPrio.root_bridge,
		     stpm->rootPrio.root_path_cost,
		     &stpm->BrId, port->port_id, port->port_id);
    STP_copy_times (&port->designTimes, &stpm->rootTimes);
  }

  stpm->rootPortId = stpm->rootPrio.bridge_port;

  if (old_root_port != stpm->rootPortId) {
    if (!stpm->rootPortId) {
      stpm->newRoot = True;
#ifdef STP_DBG
      stp_trace ("brige %s became root: %s", stpm->name,
		 STP_VECT_br_id_sprint (&stpm->rootPrio.root_bridge));
#endif
    } else {
      stpm->newRoot = False;
#ifdef STP_DBG
      stp_trace ("brige %s new root port: %s : %s",
		 stpm->name,
		 STP_stpm_get_port_name_by_id (stpm, stpm->rootPortId),
		 STP_VECT_br_id_sprint (&stpm->rootPrio.root_bridge));
#endif
    }
  }

  for (port = stpm->ports; port; port = port->next) {
    if (port->admin_non_stp) {
      setRoleSelected ("Non", stpm, port, NonStpPort);
      port->forward = port->learn = True;
      continue;
    }

    switch (port->infoIs) {
      case Disabled:
      setRoleSelected ("Dis", stpm, port, DisabledPort);
      break;
      case Aged:
      setRoleSelected ("Age", stpm, port, DesignatedPort);
      port->updtInfo = True;
      break;
      case Mine:
      setRoleSelected ("Mine", stpm, port, DesignatedPort);
      if (0 != STP_VECT_compare_vector (&port->portPrio,
					&port->designPrio,
					True) ||
	  0 != STP_compare_times (&port->portTimes, &port->designTimes)) {
	port->updtInfo = True;
      }
      break;
      case Received:
      if (stpm->rootPortId == port->port_id) {
	setRoleSelected ("Rcv", stpm, port, RootPort);
      } else if (STP_VECT_compare_vector (&port->designPrio,
					  &port->portPrio, True) < 0) {
	/* Note: this important piece has been inserted after
	 * discussion with Mick Sieman and reading 802.1y Z1 */
	setRoleSelected ("Rcv", stpm, port, DesignatedPort);
	port->updtInfo = True;
	break;
      } else {
	if (_is_backup_port (port, stpm)) {
	  setRoleSelected ("Rcv", stpm, port, BackupPort);
	} else {
	  setRoleSelected ("Rcv", stpm, port, AlternatePort);
	}
      }
      port->updtInfo = False;
      break;
      default:
      stp_trace ("undef infoIs=%d", (int) port->infoIs);
      break;
    }
  }
}


static Bool
setSelectedBridge (STPM_T * this)
{
  register PORT_T *port;

  for (port = this->ports; port; port = port->next) {
    if (port->reselect) {
#ifdef STP_DBG
      stp_trace ("setSelectedBridge: TRUE=reselect on port %s",
		 port->port_name);
#endif
      return False;
    }
  }

  for (port = this->ports; port; port = port->next) {
    port->selected = True;
  }

  return True;
}

void
STP_rolesel_enter_state (STATE_MACH_T * this)
{
  STPM_T *stpm;

  stpm = this->owner.stpm;

  switch (this->State) {
    case BEGIN:
    case INIT_BRIDGE:
    updtRoleDisableBridge (stpm);
    break;
    case ROLE_SELECTION:
    clearReselectBridge (stpm);
    updtRolesBridge (this, False);
    setSelectedBridge (stpm);
    break;
  }
}

Bool
STP_rolesel_check_conditions (STATE_MACH_T * s)
{
  STPM_T *stpm;
  register PORT_T *port;

  if (BEGIN == s->State) {
    STP_hop_2_state (s, INIT_BRIDGE);
  }

  switch (s->State) {
    case BEGIN:
    return STP_hop_2_state (s, INIT_BRIDGE);
    case INIT_BRIDGE:
    return STP_hop_2_state (s, ROLE_SELECTION);
    case ROLE_SELECTION:
    stpm = s->owner.stpm;
    for (port = stpm->ports; port; port = port->next) {
      if (port->reselect) {
	/* stp_trace ("reselect on port %s", port->port_name); */
	return STP_hop_2_state (s, ROLE_SELECTION);
      }
    }
    break;
  }

  return False;
}

Bool
STP_rolesel_update_root_priority (STPM_T * stpm)
{
  register PORT_T *port;
  Bool bret = False;

  STP_VECT_create (&stpm->rootPrio, &stpm->BrId, 0, &stpm->BrId, 0, 0);
#ifdef STP_DBG
  stp_trace ("%s rootPrio=%s", "new", STP_VECT_sprint (&stpm->rootPrio));
#endif

  for (port = stpm->ports; port; port = port->next) {
    if (port->admin_non_stp)
      continue;

    if (Received != port->infoIs && Mine != port->infoIs)
      continue;
    STP_VECT_create (&port->portPrio,
		     &stpm->rootPrio.root_bridge,
		     stpm->rootPrio.root_path_cost,
		     &stpm->BrId, port->port_id, port->port_id);
    /* stp_trace ("forse port %s : %s", port->port_name, STP_VECT_sprint (&port->portPrio)); */
    bret = True;
  }

  return bret;
}
