/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* STP priority vectors API : 17.4.2 */

#include <asm/unaligned.h>

#include "mstp.h"
#include "base.h"
#include "stp_bpdu.h"
#include "vector.h"

/*---------------------------------------------------------------------------------*/
int STP_VECT_compare_bridge_id (BRIDGE_ID * b1, BRIDGE_ID * b2)
 {
  if ((b1->prio&0xF000) < (b2->prio&0xF000))
    return -1;

  if ((b1->prio&0xF000) > (b2->prio&0xF000))
    return 1;

  return memcmp (b1->addr, b2->addr, 6);
 }

/*---------------------------------------------------------------------------------*/
void STP_VECT_copy (OUT PRIO_VECTOR_T * t, IN PRIO_VECTOR_T * f)
{
  memcpy (t, f, sizeof (PRIO_VECTOR_T));
}

/*---------------------------------------------------------------------------------*/
void
STP_VECT_create (OUT PRIO_VECTOR_T * t,
		 IN BRIDGE_ID * root_br,
		 IN unsigned long extern_root_path_cost,
		 IN BRIDGE_ID * region_root_br,
		 IN unsigned long intern_root_path_cost,
		 IN BRIDGE_ID * design_bridge,
		 IN PORT_ID design_port, IN PORT_ID rcv_port)
{
  if(root_br != 0)
   memcpy (&t->root_bridge, root_br, sizeof (BRIDGE_ID));
  else
//   bzero(&t->root_bridge,sizeof(BRIDGE_ID));
   memset(&t->root_bridge,0,sizeof(BRIDGE_ID));

  t->extern_path_cost = extern_root_path_cost;
  memcpy (&t->region_root_bridge, region_root_br, sizeof (BRIDGE_ID));
  t->intern_path_cost = intern_root_path_cost;
  memcpy (&t->design_bridge, design_bridge, sizeof (BRIDGE_ID));
  t->design_port = design_port;
  t->rcv_port = rcv_port;
}

/*---------------------------------------------------------------------------------*/
int STP_VECT_compare_vector (PRIO_VECTOR_T * v1, PRIO_VECTOR_T * v2, Bool shouldCheckBridgePort, Bool notMstp)
 {
  int bridcmp;

  bridcmp = STP_VECT_compare_bridge_id (&v1->root_bridge, &v2->root_bridge);
  if (bridcmp < 0)
    return bridcmp;
  if (!bridcmp) 
   {
    bridcmp = v1->extern_path_cost - v2->extern_path_cost;
    if (bridcmp < 0)
      return bridcmp;

    if (!bridcmp) 
     {
      if(notMstp == False)
       {
        bridcmp = STP_VECT_compare_bridge_id (&v1->region_root_bridge, &v2->region_root_bridge);

        if (bridcmp < 0)
          return bridcmp;

        if (!bridcmp) 
         {
          bridcmp = v1->intern_path_cost - v2->intern_path_cost;
          if (bridcmp < 0)
            return bridcmp;
         }
       }
      if (!bridcmp) 
       {
        bridcmp = STP_VECT_compare_bridge_id (&v1->design_bridge, &v2->design_bridge);
	
        if (bridcmp < 0)
	        return bridcmp;
        if (!bridcmp) 
         {
	         bridcmp = v1->design_port - v2->design_port;
	         if (bridcmp < 0)
	          return bridcmp;
	         if (shouldCheckBridgePort && !bridcmp)
	          return v1->rcv_port - v2->rcv_port;
         }
       }
     }
   }
  return bridcmp;
 }

/*---------------------------------------------------------------------------------*/
int STP_VECT_compare_msti_vector (PRIO_VECTOR_T * v1, PRIO_VECTOR_T * v2, Bool shouldCheckBridgePort)
 {
  int bridcmp;

  bridcmp = STP_VECT_compare_bridge_id (&v1->region_root_bridge, &v2->region_root_bridge);
  if (bridcmp < 0)
    return bridcmp;

  if (!bridcmp) 
   {
    bridcmp = v1->intern_path_cost - v2->intern_path_cost;
    if (bridcmp < 0)
     return bridcmp;
//    printk(" v1 inter path cost %d   ... v2 inter path cost %d\n",v1->intern_path_cost , v2->intern_path_cost);

    if (!bridcmp) 
     {
      bridcmp = STP_VECT_compare_bridge_id (&v1->design_bridge, &v2->design_bridge);
	
      if (bridcmp < 0)
	      return bridcmp;

      if (!bridcmp) 
       {
	       bridcmp = v1->design_port - v2->design_port;
	       if (bridcmp < 0)
	        return bridcmp;
	       if (shouldCheckBridgePort && !bridcmp)
	        return v1->rcv_port - v2->rcv_port;
       }
     }
   }
  return bridcmp;
 }

/*---------------------------------------------------------------------------------*/
static unsigned short stp_vect_get_short (IN unsigned char *f)
{
/*  return ntohs (*(unsigned short *) f);*/
  return ntohs(get_unaligned((unsigned short *)f));
//  return get_unaligned((unsigned short *)f);
}

/*---------------------------------------------------------------------------------*/
void stp_vect_set_short (IN unsigned short f, OUT unsigned char *t)
{
  put_unaligned (htons(f), (unsigned short *) t);
}

/*---------------------------------------------------------------------------------*/
void stp_vect_get_bridge_id (IN unsigned char *c_br, OUT BRIDGE_ID * bridge_id)
{
  bridge_id->prio = stp_vect_get_short (c_br);
  memcpy (bridge_id->addr, c_br + 2, 6);
}

/*---------------------------------------------------------------------------------*/
void stp_vect_set_bridge_id (IN BRIDGE_ID * bridge_id, OUT unsigned char *c_br)
{
  stp_vect_set_short (bridge_id->prio, c_br);
  memcpy (c_br + 2, bridge_id->addr, 6);
}

/*---------------------------------------------------------------------------------*/
void STP_VECT_get_vector (IN MSTP_BPDU_T * b, OUT PRIO_VECTOR_T * v)
 {
//RD
  stp_vect_get_bridge_id (b->cist_body.root_id, &v->root_bridge);
//ERCD
  v->extern_path_cost = ntohl (get_unaligned ((long *) b->cist_body.root_path_cost));

//RRD
  stp_vect_get_bridge_id (b->cist_body.bridge_id, &v->region_root_bridge);
//  stp_vect_get_bridge_id (b->Cist_bridge_id, &v->region_root_bridge);
//IRCD
  v->intern_path_cost = ntohl (get_unaligned ((long *) b->Cist_internal_root_path_cost));
//D
  stp_vect_get_bridge_id (b->Cist_bridge_id, &v->design_bridge);
//  stp_vect_get_bridge_id (b->cist_body.bridge_id, &v->design_bridge);
//PD
  v->design_port = stp_vect_get_short (b->cist_body.port_id);
//PB
//  v->rcvPort
 }

/*---------------------------------------------------------------------------------*/
void STP_VECT_get_msti_vector (IN MSTP_BPDU_T * b, IN int msti, OUT PRIO_VECTOR_T * v)
 {
//RRD
  stp_vect_get_bridge_id (b->msti_conf[msti].msti_internal_root_id, &v->region_root_bridge);
//IRCD
  v->intern_path_cost = ntohl (get_unaligned ((long *) b->msti_conf[msti].msti_internal_root_path_cost));
//D
  stp_vect_get_bridge_id (b->Cist_bridge_id, &v->design_bridge);  
 
  v->design_bridge.prio=b->msti_conf[msti].msti_bridge_priority<<8 | (b->msti_conf[msti].msti_internal_root_id[1]);
//  printk(" msti %d prio %x %x\n",msti, b->msti_conf[msti].msti_bridge_priority<<8,v->design_bridge.prio);
//  v->design_bridge.prio=(v->design_bridge.prio & 0xf000) & msti;

//PD
  v->design_port = stp_vect_get_short (b->cist_body.port_id);
//  v->design_port=(v->design_port & 0x00ff) | (b->msti_conf[msti].msti_port_priority<<8 | (b->msti_conf[msti].msti_internal_root_id[1])<<4);
  v->design_port=(v->design_port & 0x00ff) | (b->msti_conf[msti].msti_port_priority<<8);

 }

/*---------------------------------------------------------------------------------*/
void STP_VECT_set_vector (IN PRIO_VECTOR_T * v, OUT MSTP_BPDU_T * b)
 {
  unsigned long root_path_cost;
/*  unsigned short design_port;*/

  stp_vect_set_bridge_id (&(v->root_bridge), (b->cist_body.root_id));

  root_path_cost = htonl (v->extern_path_cost);
  memcpy ((b->cist_body.root_path_cost), &root_path_cost, 4);

//?????  flash
  stp_vect_set_bridge_id (&(v->region_root_bridge), (b->cist_body.bridge_id));
//  stp_vect_set_bridge_id (&(v->design_bridge), (b->cist_body.bridge_id));

  stp_vect_set_bridge_id (&(v->design_bridge), (b->Cist_bridge_id));

  root_path_cost = htonl (v->intern_path_cost);
  memcpy ((b->Cist_internal_root_path_cost), &root_path_cost, 4);

}

/*---------------------------------------------------------------------------------*/
#ifdef STP_DBG

const char *STP_VECT_br_id_sprint (IN BRIDGE_ID * br_id)
{
  static char reslt[4 + 1 + 6 + 2];

  sprintf (reslt,
	"%04lX-%02x%02x%02x%02x%02x%02x",
	(unsigned long) br_id->prio,
	(unsigned char) br_id->addr[0], (unsigned char) br_id->addr[1],
	(unsigned char) br_id->addr[2], (unsigned char) br_id->addr[3],
	(unsigned char) br_id->addr[4], (unsigned char) br_id->addr[5]);
  return reslt;
}


/*---------------------------------------------------------------------------------*/
const char *STP_VECT_sprint (IN PRIO_VECTOR_T * v)
{
  static char reslt[120];

  reslt[0]=0;
  sprintf (reslt,
	   "%04lX-%02x%02x%02x%02x%02x%02x %6ld "
	   "%04lX-%02x%02x%02x%02x%02x%02x %6ld "
	   "%04lX-%02x%02x%02x%02x%02x%02x %4lx %4lx",
	   (unsigned long) v->root_bridge.prio,
	   (unsigned char) v->root_bridge.addr[0],
	   (unsigned char) v->root_bridge.addr[1],
	   (unsigned char) v->root_bridge.addr[2],
	   (unsigned char) v->root_bridge.addr[3],
	   (unsigned char) v->root_bridge.addr[4],
	   (unsigned char) v->root_bridge.addr[5],
	   (long) v->extern_path_cost,
	   (unsigned long) v->region_root_bridge.prio,
	   (unsigned char) v->region_root_bridge.addr[0],
	   (unsigned char) v->region_root_bridge.addr[1],
	   (unsigned char) v->region_root_bridge.addr[2],
	   (unsigned char) v->region_root_bridge.addr[3],
	   (unsigned char) v->region_root_bridge.addr[4],
	   (unsigned char) v->region_root_bridge.addr[5],
	   (long) v->intern_path_cost,
	   (unsigned long) v->design_bridge.prio,
	   (unsigned char) v->design_bridge.addr[0],
	   (unsigned char) v->design_bridge.addr[1],
	   (unsigned char) v->design_bridge.addr[2],
	   (unsigned char) v->design_bridge.addr[3],
	   (unsigned char) v->design_bridge.addr[4],
	   (unsigned char) v->design_bridge.addr[5],
	   (unsigned long) v->design_port, (unsigned long) v->rcv_port);

  return reslt;
}

/*---------------------------------------------------------------------------------*/
void STP_VECT_br_id_print (IN char *title, IN BRIDGE_ID * br_id, IN Bool cr)
{
  printk ("%s=%04lX-%02x%02x%02x%02x%02x%02x",
	 title,
	 (unsigned long) br_id->prio,
	 (unsigned char) br_id->addr[0],
	 (unsigned char) br_id->addr[1],
	 (unsigned char) br_id->addr[2],
	 (unsigned char) br_id->addr[3],
	 (unsigned char) br_id->addr[4], (unsigned char) br_id->addr[5]);
  printk (cr ? "\n" : " ");
}
#endif
