
/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* This file contains API from an operation system to the RSTP library */

#include "mstp.h"
#include "base.h"
#include "xst.h"
#include "xst_port.h"
#include "stp_in.h"
#include "uid_stp.h"

extern unsigned char MSTP_ConfigName[32];
extern unsigned short MSTP_RevisionLevel;         /* 0 */
extern unsigned char mstp_debug;

int max_port = 0;

typedef enum
 {
  MSTP_PORT_EN_T,
  MSTP_PORT_DIS_T,
  MSTP_PORT_SPEED_T,
  MSTP_PORT_DPLEX_T,
  MSTP_PORT_RX_T,
  MSTP_PORT_TIME_T,
  MSTP_PORT_MNGR_T,
  MSTP_EVENT_LAST_DUMMY
 } MSTP_EVENT_T;


/* ----------------------------------------------------------------------------------------------*/
int STP_IN_refersh()
{
 XST_T *xst;
 XST_PORT_T *xst_port;
 int i,j;

 for(i=0;i<MSTP_MAX_INSTANCE;i++)
  {
   xst = STP_xst_find (i);
   if(xst->valid == False) continue;
   if(xst->admin_state == STP_DISABLED) continue;

   for(j=0;j<MAX_LOGIC_PORT;j++)
    {
     xst_port=&(xst->ports[j]);
    
     if(xst_port->valid == False) continue;
   
     if(xst_port->forwarding == True)
      STP_OUT_set_port_state (xst->xst_id, xst_port->port_index, UID_PORT_FORWARDING); 
     else if(xst_port->learning == True)
      STP_OUT_set_port_state (xst->xst_id, xst_port->port_index, UID_PORT_LEARNING);
     else
      STP_OUT_set_port_state (xst->xst_id, xst_port->port_index, UID_PORT_DISCARDING);

    }

  }
 return 0;
}
/* ----------------------------------------------------------------------------------------------*/
int STP_IN_xst_delete (int xst_id)
 {
  register XST_T *this;
  int iret = 0;

  MSTP_CRITICAL_PATH_START;
   
  this = STP_xst_find(xst_id);

  if(this->valid == False)
   return 0;

  if (this->admin_state == STP_ENABLED) 
   {
    if (STP_xst_enable (this, STP_DISABLED)!=0) 
     { /* can't disable :( */
      iret = STP_Another_Error;
     } 
    else
     {
      if(xst_id == 0) /* it's CIST */
       STP_OUT_set_hardware_mode (STP_DISABLED);
     }
   }

  if (iret == 0) 
   {
    STP_xst_delete (this);
   }
  MSTP_CRITICAL_PATH_END;

  return iret;
 }
/* ----------------------------------------------------------------------------------------------*/

void *stp_in_xst_create(int xst_id, int *err_code)
 {
  int port_index;
  register XST_T *this;

  this = STP_xst_find(xst_id);
  
/* it had just been created :( */
  if (this->valid == True) 
   {   
    *err_code = STP_Nothing_To_Do;
    return this;
   }

  this = STP_xst_create (xst_id);

/* can't create xst :( */
  if (this->valid == False) 
   {   
    *err_code = STP_Cannot_Create_Instance;
    return NULL;
   }

  this->xst_id=xst_id;

#ifdef STP_DBG
  if(mstp_debug)
   printk(" Created xst %d\n",this->xst_id);
#endif
  
  for (port_index = 0; port_index < MAX_LOGIC_PORT; port_index++) 
   {
    if (STP_port_create (this, port_index)==NULL) 
     {
      /* can't add port :( */
      STP_xst_delete (this);
      *err_code = STP_Cannot_Create_Instance_For_Port;
      return NULL;
     } 
#ifdef STP_DBG
//    printk(" create port %d\n",port_index);
//    printk("yyy XST=%08X rolesel=%08X machines=%08x Enter=%08X\n",this ,this->rolesel,this->machines, this->rolesel->concreteEnterState);
#endif
   }

  *err_code = STP_OK;
  return this;
 }
/* ----------------------------------------------------------------------------------------------*/

int _stp_in_xst_enable (int xst_id, UID_STP_MODE_T admin_state)
 {
  register XST_T *this;
  int rc;

  this = STP_xst_find (xst_id);
 
#if 0
  if (admin_state != STP_DISABLED) 
   {
    register XST_T *xst1;

    if (xst_id==0)  //disable all
     {
      for(i=0; i<MSTP_MAX_INSTANCE; i++) 
       {
        xst1=STP_xst_find(i);
        if (xst1->admin_state != STP_DISABLED) 
         {
          STP_OUT_set_hardware_mode (STP_DISABLED);
          STP_xst_enable (xst1, STP_DISABLED);
         }
       }
     }
   }
#endif

  if (this->admin_state == admin_state) 
   { /* nothing to do :) */
    return 0;
   }
#ifdef STP_DBG
//  printk("kkk %08x\n",this);
#endif 
  rc = STP_xst_enable (this, admin_state);
  if (!rc && this->xst_id==0) 
   {
    STP_OUT_set_hardware_mode (admin_state);
   }

  return rc;
 }

/* ----------------------------------------------------------------------------------------------*/

static XST_PORT_T *_stpapi_port_find (XST_T * this, int port_index)
 {
  return &(this->ports[port_index]);
 }
/* ----------------------------------------------------------------------------------------------*/

static void _conv_br_id_2_uid (IN BRIDGE_ID * f, OUT UID_BRIDGE_ID_T * t)
 {
  memcpy (t, f, sizeof (UID_BRIDGE_ID_T));
 }
/* ----------------------------------------------------------------------------------------------*/

static int _check_xst_config (IN UID_STP_CFG_T * uid_cfg)
 {
  if (uid_cfg->bridge_priority < MIN_BR_PRIO)//0 
   {
    return STP_Small_Bridge_Priority;
   }

  if (uid_cfg->bridge_priority > MAX_BR_PRIO)//61440 
   {
    return STP_Large_Bridge_Priority;
   }

  if (0 != (uid_cfg->bridge_priority % BR_PRIO_STEP)) //4096
   {   
#ifdef STP_DBG
//    printk("   prio = %d",uid_cfg->bridge_priority);
#endif
    return STP_Bridge_Priority_Granularity;
   }

  if (uid_cfg->hello_time < MIN_BR_HELLOT) //1
   {
    return STP_Small_Hello_Time;
   }

  if (uid_cfg->hello_time > MAX_BR_HELLOT) //10
   {
    return STP_Large_Hello_Time;
   }

  if (uid_cfg->max_age < MIN_BR_MAXAGE) //6
   {
    return STP_Small_Max_Age;
   }

  if (uid_cfg->max_age > MAX_BR_MAXAGE) //40
   {
    return STP_Large_Max_Age;
   }

  if (uid_cfg->forward_delay < MIN_BR_FWDELAY) //4
   {
    return STP_Small_Forward_Delay;
   }

  if (uid_cfg->forward_delay > MAX_BR_FWDELAY) //30
   {
    return STP_Large_Forward_Delay;
   }

  if (2 * (uid_cfg->forward_delay - 1) < uid_cfg->max_age) 
   {
    return STP_Forward_Delay_And_Max_Age_Are_Inconsistent;
   }

  if (uid_cfg->max_age < 2 * (uid_cfg->hello_time + 1)) 
   {
    return STP_Hello_Time_And_Max_Age_Are_Inconsistent;
   }

  return 0;
 }

/* ----------------------------------------------------------------------------------------------*/
static void _stp_in_enable_port_on_xst(XST_T * xst, int port_index, Bool enable)
 {
  register XST_PORT_T *port;

  port = _stpapi_port_find (xst, port_index);

  if (port->valid==False)
  {
#if 0
    if(port->port_index==8)
    printk(" not valid \n");
#endif
    return;
  }

  if (xst->xst_id == 0 && port->portEnabled == enable) 
   { /* nothing to do :) */
#if 0
    if(port->port_index==8)
    printk(" port Enabled == enable\n");
#endif
    return;
   }
  port->uptime = 0;
  if (True == enable) 
   {   /* clear port statistics */
    port->rx_cfg_bpdu_cnt = port->rx_rstp_bpdu_cnt =
     port->rx_mstp_bpdu_cnt = port->rx_tcn_bpdu_cnt = 0;
    port->tx_cfg_bpdu_cnt = port->tx_rstp_bpdu_cnt =
     port->tx_mstp_bpdu_cnt = port->tx_tcn_bpdu_cnt = 0;
   }

  port->adminEnable = enable;
  if (xst->admin_state != STP_ENABLED) 
   {
    port->portEnabled = port->forwarding = port->forward =  port->learning = port->learn = enable;
    port->role = enable ? NonStpPort : DisabledPort;
#if 0
    if(port->port_index==8)
    printk(" port Enabled == %d\n",port->portEnabled);
#endif
    return;
   }

#if 0
    if(port->port_index==8)
    printk(" port init %d %d\n",xst->xst_id,port->port_index);
#endif
  STP_xst_port_init (port, xst, False);
 }

/* ----------------------------------------------------------------------------------------------*/
void STP_IN_init ()
 {
  max_port = MAX_LOGIC_PORT;
  MSTP_INIT_CRITICAL_PATH_PROTECTIO;
 }

/* ----------------------------------------------------------------------------------------------*/
/* ----------------------------------------------------------------------------------------------*/

int STP_IN_get_mstp_info (SMstpInfo * info)
{
 register XST_T *this;

 MSTP_CRITICAL_PATH_START;

 this = STP_xst_find (CIST_INSTANCE_ID);

 if (this->valid == False) 
 {
  MSTP_CRITICAL_PATH_END;
  return STP_Had_Not_Yet_Been_Created;
 }

 info->bGenEnable = this->admin_state;
 info->bGenDebugEnable = mstp_debug; 

 strncpy (info->aucGenCfgName, MSTP_ConfigName, NAME_LEN);
 info->usGenRevLevel = MSTP_RevisionLevel;

 _conv_br_id_2_uid (&this->BridgeIdentifier, &info->stGenBridgeId);
 _conv_br_id_2_uid (&this->xstRootPriority.root_bridge, &info->stGenCistRoot);
 info->ulGenExternalRootCost = this->xstRootPriority.extern_path_cost;

 info->usGenBridgeMaxAge = this->xstBridgeTimes.MaxAge;
 info->usGenBridgeHelloTime = this->xstBridgeTimes.HelloTime;
 info->usGenBridgeForwardDelay = this->xstBridgeTimes.ForwardDelay;
 info->usGenMaxAge = this->xstRootTimes.MaxAge;
 info->usGenHelloTime = this->xstRootTimes.HelloTime;
 info->usGenForwardDelay = this->xstRootTimes.ForwardDelay;

//by flash, 2005/04/26
// info->usGenMaxHops = this->MaxHops;
 info->usGenMaxHops = this->xstBridgeTimes.remainingHops;

 //info->ucGenHoldTime;  /* winfred: no such thing? */
 info->ucGenHoldTime = 1;
 info->ucGenMigrateTime = this->MigrateTime;
 info->ucGenForceVersion = this->ForceVersion;

 info->usRootPort=this->xstRootPortId & 0xff;

 MSTP_CRITICAL_PATH_END;
 return STP_OK;
}

int STP_IN_get_port_info (SMstpPortInfo * info)
{
 register XST_T *this;
 register XST_PORT_T *port;

 if (info->ucPortIndex < 0 || info->ucPortIndex > MAX_LOGIC_PORT - 1)
  return STP_Invalid_Instance_Or_Port;

 MSTP_CRITICAL_PATH_START;

 this = STP_xst_find (CIST_INSTANCE_ID);
 if (!this) 
 {
  MSTP_CRITICAL_PATH_END;
  return STP_Had_Not_Yet_Been_Created;
 }

 port = _stpapi_port_find (this, info->ucPortIndex);
 if (port->valid==False) 
 {
  MSTP_CRITICAL_PATH_END;
  return STP_Port_Is_Absent_In_The_Stp;
 }

 info->bPortOperMacEnable = port->portEnabled;

 info->ulPortUptime = port->uptime;
 info->bPortAdminNonStp = port->admin_non_stp;

 info->ulPortAdminExternalPathCost = port->adminExtPCost; // flash
 info->ulPortOperExternalPathCost = port->operExtPCost; //flash
 info->bPortAdminEdge = port->adminEdge;
 info->bPortOperEdge = port->operEdge ? 1 : 0;
 info->ucPortAdminPointToPoint = port->adminPointToPointMac;
 info->bPortOperPointToPoint = port->operPointToPointMac;
 info->ulDesPortPathCost = port->xstDesignatedPriority.extern_path_cost;

 info->ulPortRxTcnBpduCnt = port->rx_tcn_bpdu_cnt;
 info->ulPortRxCfgBpduCnt = port->rx_cfg_bpdu_cnt;
 info->ulPortRxRstBpduCnt = port->rx_rstp_bpdu_cnt;
 info->ulPortRxMstBpduCnt = port->rx_mstp_bpdu_cnt;
 info->ulPortTxTcnBpduCnt = port->tx_tcn_bpdu_cnt;
 info->ulPortTxCfgBpduCnt = port->tx_cfg_bpdu_cnt;
 info->ulPortTxRstBpduCnt = port->tx_rstp_bpdu_cnt;
 info->ulPortTxMstBpduCnt = port->tx_mstp_bpdu_cnt;
 MSTP_CRITICAL_PATH_END;
 return STP_OK;
}

int STP_IN_get_inst_info (SMstpInstInfo * info)
{
 register XST_T *this;

 if (info->ucXstId < 0 || info->ucXstId > MSTP_MAX_INSTANCE - 1)
  return STP_Invalid_Instance_Or_Port;

 MSTP_CRITICAL_PATH_START;

 this = STP_xst_find (info->ucXstId);

 if (this->valid == False) 
 {
  MSTP_CRITICAL_PATH_END;
  return STP_Had_Not_Yet_Been_Created;
 }

// info->usXstPriority = this->xstBridgePriority.region_root_bridge.prio;
// _conv_br_id_2_uid (&this->xstBridgePriority.region_root_bridge, &info->stXstBridgeId);
 info->usXstPriority = this->BridgeIdentifier.prio;
 _conv_br_id_2_uid (&this->BridgeIdentifier, &info->stXstBridgeId);
 /* winfred:
  * 13.23.9: MSTI Regional Root Identifier and Designated Bridge Identifier
  * components are equal to the value of MSTI Bridge Identifier
  * note: not to use the value of root_bridge (it is useless in MSTI)
  */
//by flash, 2005/04/28
#if 0
 if(this->xst_id == 0)
  _conv_br_id_2_uid (&this->xstRootPriority.root_bridge, &info->stXstRegionaldRoot);
 else
  _conv_br_id_2_uid (&this->xstRootPriority.region_root_bridge, &info->stXstRegionaldRoot);
#endif
  _conv_br_id_2_uid (&this->xstRootPriority.region_root_bridge, &info->stXstRegionaldRoot);

 info->ulXstInternalRootCost = this->xstRootPriority.intern_path_cost;
 info->ucXstRootPort = (unsigned char)(this->xstRootPriority.rcv_port & 0x00FF);

 info->ulXstTimeSinceTopologyChange = this->timeSince_Topo_Change;
 info->ulXstTopologyChangeCnt = this->Topo_Change_Count;
 info->bXstTopologyChangeFlag = this->Topo_Change;

 MSTP_CRITICAL_PATH_END;
 return STP_OK;
}

int STP_IN_get_instport_info (SMstpInstPortInfo * info)
{
 register XST_T *this;
 register XST_PORT_T *port;
 PORT_ROLE_T role2show;

 if (info->ucXstId < 0 || info->ucXstId > MSTP_MAX_INSTANCE - 1 ||
     info->ucXstPortIndex < 0 || info->ucXstPortIndex > MAX_LOGIC_PORT -1)
  return STP_Invalid_Instance_Or_Port;
 MSTP_CRITICAL_PATH_START;

 this = STP_xst_find (info->ucXstId);

 if (this->valid == False) 
 {
  MSTP_CRITICAL_PATH_END;
  return STP_Had_Not_Yet_Been_Created;
 }

 port = _stpapi_port_find (this, info->ucXstPortIndex);
 if (port->valid==False) 
 {
  MSTP_CRITICAL_PATH_END;
  return STP_Port_Is_Absent_In_The_Stp;
 }

 if (port->admin_non_stp)
  info->ucXstPortState = UID_PORT_NON_STP;
 else if (DisabledPort == port->role)
  info->ucXstPortState = UID_PORT_DISABLED;
 else if (port->forwarding)
  info->ucXstPortState = UID_PORT_FORWARDING;
 else if (port->learning)
  info->ucXstPortState = UID_PORT_LEARNING;
 else
  info->ucXstPortState = UID_PORT_DISCARDING;

 if (NonStpPort != port->selectedRole)
  role2show = port->role;
 else
  role2show = port->selectedRole;
 switch (role2show)
 {
  case DisabledPort:	info->ucXstPortRole = ' '; break;
  case AlternatePort:	info->ucXstPortRole = 'A'; break;
  case BackupPort:	info->ucXstPortRole = 'B'; break;
  case RootPort:	info->ucXstPortRole = 'R'; break;
  case DesignatedPort:	info->ucXstPortRole = 'D'; break;
  case MasterPort:	info->ucXstPortRole = 'M'; break;
  case NonStpPort:	info->ucXstPortRole = '-'; break;
  default:		info->ucXstPortRole = '?'; break;
 }

 _conv_br_id_2_uid (&port->xstPortPriority.design_bridge, &info->stXstPortDesignatedBridge);
 info->usXstPortDesignatedPort = port->xstPortPriority.design_port;
 info->usXstPortPriority = port->portId >> 8;
 /* my port priority? */

 info->ulXstPortAdminInternalPathCost = port->adminIntPCost; // flash
 info->ulXstPortOperInternalPathCost = port->operIntPCost; //flash

 info->ulXstDesPathCost=port->xstPortPriority.intern_path_cost;

 MSTP_CRITICAL_PATH_END;
 return STP_OK;
}

/* ----------------------------------------------------------------------------------------------*/
int STP_IN_xst_get_cfg (IN int xst_id, OUT UID_STP_CFG_T * uid_cfg)
 {
  register XST_T *this;

  uid_cfg->field_mask = 0;

  MSTP_CRITICAL_PATH_START;
  this = STP_xst_find (xst_id);

  if (this->valid==False) 
   {   /* it had not yet been created :( */
    MSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
   }

  if (this->admin_state != STP_DISABLED) 
   {
    uid_cfg->field_mask |= BR_CFG_STATE;
   }
  uid_cfg->stp_enabled = this->admin_state;

  if (this->ForceVersion != BR_CFG_FORCE_VER) 
   {
    uid_cfg->field_mask |= BR_CFG_FORCE_VER;
   }
  uid_cfg->force_version = this->ForceVersion;

  if ((this->BridgeIdentifier.prio&0xff00) != DEF_BR_PRIO) 
   {
    uid_cfg->field_mask |= BR_CFG_PRIO;
   }
  uid_cfg->bridge_priority = this->BridgeIdentifier.prio & 0xff00;

  if (this->xstBridgeTimes.MaxAge != DEF_BR_MAXAGE) 
   {
    uid_cfg->field_mask |= BR_CFG_AGE;
   }
  uid_cfg->max_age = this->xstBridgeTimes.MaxAge;

  if (this->xstBridgeTimes.HelloTime != DEF_BR_HELLOT) 
   {
    uid_cfg->field_mask |= BR_CFG_HELLO;
   }
  uid_cfg->hello_time = this->xstBridgeTimes.HelloTime;

  if (this->xstBridgeTimes.ForwardDelay != DEF_BR_FWDELAY) 
   {
    uid_cfg->field_mask |= BR_CFG_DELAY;
   }
  uid_cfg->forward_delay = this->xstBridgeTimes.ForwardDelay;

  uid_cfg->hold_time = TxHoldCount;

  if(xst_id == 0)
   {
    uid_cfg->revision=MSTP_RevisionLevel;
    strncpy (uid_cfg->name, MSTP_ConfigName, NAME_LEN);
   }

  MSTP_CRITICAL_PATH_END;
  return 0;
 }

/* ----------------------------------------------------------------------------------------------*/


/* call it, when link Up/Down */
int STP_IN_enable_port (int port_index, Bool enable)
 {
  register XST_T *this;
  int i;

  MSTP_CRITICAL_PATH_START;

  if (!enable) 
   {
    STP_OUT_flush_single_lt (port_index, "disable port");
   }

  for (i=0;i<MSTP_MAX_INSTANCE;i++) 
   {
    this = STP_xst_find(i);

    if(this->valid==False)
     continue;
    if(this->admin_state==STP_DISABLED)
     continue;

    
    if(mstp_check_vlan_member(i, port_index)==True)
    {

//     printk(" is a member, xst %d port %d\n",i,port_index);
     _stp_in_enable_port_on_xst (this, port_index, enable);
#if 0
     if(port_index==8)
      printk("enable port %d %d %d \n",this->xst_id, port_index, enable);
#endif
    }
    else
    {
//     printk(" not a member, xst %d port %d \n",i,port_index);

     STP_OUT_flush_single_lt (port_index, "disable port");

     _stp_in_enable_port_on_xst (this, port_index, False);
#if 0
     if(port_index==8)
      printk("disable port %d %d %d \n",this->xst_id, port_index, enable);
#endif
    }
   }

  MSTP_CRITICAL_PATH_END;
  return 0;
 }
/* ----------------------------------------------------------------------------------------------*/

/* call it, when port speed has been changed, speed in Kb/s  */
int STP_IN_changed_port_speed (int port_index, long speed)
 {
  int i;
  register XST_T *this;
  register XST_PORT_T *port;

  MSTP_CRITICAL_PATH_START;

  for (i=0;i<MSTP_MAX_INSTANCE;i++) 
   {
    this = STP_xst_find(i);

    if(this->valid==False)
     continue;

    if ( this->admin_state != STP_ENABLED)
      continue;
    port = _stpapi_port_find (this, port_index);
    if (port->valid==False || port->operSpeed == speed)
      continue;
    port->operSpeed = speed;

    port->reselect = True;
    port->selected = False;
   }
  MSTP_CRITICAL_PATH_END;
  return 0;
}
/* ----------------------------------------------------------------------------------------------*/

/* call it, when port duplex mode has been changed  */
int STP_IN_changed_port_duplex (int port_index)
 {
  int i;
  register XST_T *this;
  register XST_PORT_T *port;

  MSTP_CRITICAL_PATH_START;

  for (i=0;i<MSTP_MAX_INSTANCE;i++) 
   {
    this = STP_xst_find(i);

    if(this->valid==False)
     continue;

    if ( this->admin_state != STP_ENABLED)
      continue;

    port = _stpapi_port_find (this, port_index);
    if (port->valid==False)
      continue;
    port->p2p_recompute = True;
    port->reselect = True;
    port->selected = False;
   }
  MSTP_CRITICAL_PATH_END;
  return 0;
 }

/* ----------------------------------------------------------------------------------------------*/
#if 0
int STP_IN_check_bpdu_header (MSTP_BPDU_T * bpdu, size_t len)
 {
  unsigned short len8023;

  len8023 = ntohs (*(unsigned short *) bpdu->eth.len8023);
  if (len8023 > 1500) 
   {  /* big len8023 format :( */
    return STP_Big_len8023_Format;
   }

  if (len8023 < MIN_BPDU) 
   { /* small len8023 format :( */
    return STP_Small_len8023_Format;
   }

  if (len8023 + 14 > len) 
   { /* len8023 format gt len :( */
    return STP_len8023_Format_Gt_Len;
   }

  if (bpdu->eth.dsap != BPDU_L_SAP || bpdu->eth.ssap != BPDU_L_SAP || bpdu->eth.llc != LLC_UI) 
   {
    /* this is not a proper 802.3 pkt! :( */
    return STP_Not_Proper_802_3_Packet;
   }

  if (bpdu->hdr.protocol[0] || bpdu->hdr.protocol[1]) 
   {
    return STP_Invalid_Protocol;
   }
#if 0
  if (bpdu->hdr.version != BPDU_VERSION_ID) 
   {
    return STP_Invalid_Version;
   }
#endif
  /* see also 9.3.4: think & TBD :( */
  return 0;
 }

#endif
/* ----------------------------------------------------------------------------------------------*/
int STP_IN_rx_bpdu (int port_index, MSTP_BPDU_T * bpdu, size_t len)
 {
  register XST_PORT_T *port;
  register XST_T *this;
  int iret;
  int i;

  MSTP_CRITICAL_PATH_START;

  this = STP_xst_find(CIST_INSTANCE_ID);
  if (this->valid == False) 
   {   /*  the xst had not yet been created :( */
    MSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
   }
  if (this->admin_state == STP_DISABLED) 
   {   /*  the xst had not yet been created :( */
    MSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
   }

  port = _stpapi_port_find (this, port_index);
  if (port->valid == False) 
   {   /* port is absent in the xst :( */
    MSTP_CRITICAL_PATH_END;
    return STP_Port_Is_Absent_In_The_Stp;
   }

// by flash, 20050421
  if (!port->portEnabled) 
   { 
#if STP_DBG
//      printk("port %d not enable\n",port_index);
#endif
    if(mstp_check_vlan_member(this->xst_id, port->port_index))
    {
     _stp_in_enable_port_on_xst (this, port->port_index, True);
#if 0
     if(port->port_index==8)
      printk("rx bpdu true\n");
#endif
    }
    else
    {
     _stp_in_enable_port_on_xst (this, port->port_index, False);
#if 0
     if(port->port_index==8)
      printk("rx bpdu False\n");
#endif
    }
   }

  port->operEdge = False;
  port->wasInitBpdu = True;

  iret = STP_port_rx_bpdu (port, bpdu, len);

  for(i=0;i<MSTP_MAX_INSTANCE;i++)
   {
    this = STP_xst_find(i);

    if(this->valid==False)
     continue;
    if(this->admin_state==STP_DISABLED)
     continue;

#if 0
    if(i && port->msgMSTConfId_match == False)
     break;
#endif

    STP_xst_update (this);
   }

  MSTP_CRITICAL_PATH_END;

  return iret;
 }
/* ----------------------------------------------------------------------------------------------*/

int STP_IN_one_second (void)
 {
  register XST_T *this;
  int i;

  MSTP_CRITICAL_PATH_START;

  for (i=0; i<MSTP_MAX_INSTANCE ;i++) 
   {
    this = STP_xst_find (i);

    if(this->valid == False)
     continue;
#if STP_DBG
//    printk(" xst%d admin_state= %d \n",i,this->admin_state);
#endif
    switch (this->admin_state) 
     {
      case STP_ENABLED:
         STP_xst_one_second (this);
         break;
      default:
         break;
     }
   }

  MSTP_CRITICAL_PATH_END;

  return 0;
 }
/* ----------------------------------------------------------------------------------------------*/

extern Bool STP_rolesel_update_root_priority (XST_T * xst);
/* ----------------------------------------------------------------------------------------------*/

int STP_IN_xst_set_cfg (IN int xst_id, IN UID_STP_CFG_T * uid_cfg)
 {
  int rc = 0;
  register XST_T *this;
  UID_STP_CFG_T old;
  Bool enabled_here=False;

  if (STP_IN_xst_get_cfg (xst_id, &old) != 0) 
   {
#if 0
    printk("  get default \n");
#endif
    STP_OUT_get_init_xst_cfg (xst_id, &old);
   }

  MSTP_CRITICAL_PATH_START;

  if (uid_cfg->field_mask & BR_CFG_PRIO) 
   {
#ifdef STP_DBG
//    printk("stp in set priority\n");
#endif
    old.bridge_priority = uid_cfg->bridge_priority;
   }

  if (BR_CFG_AGE & uid_cfg->field_mask) 
   {
#ifdef STP_DBG
//    printk("stp in set max age\n");
#endif
    old.max_age = uid_cfg->max_age;
   }

  if (BR_CFG_HELLO & uid_cfg->field_mask) 
   {
#ifdef STP_DBG
//    printk("stp in set hello \n");
#endif
    old.hello_time = uid_cfg->hello_time;
   }

  if (BR_CFG_DELAY & uid_cfg->field_mask) 
   {
    old.forward_delay = uid_cfg->forward_delay;
   }

  if (BR_CFG_FORCE_VER & uid_cfg->field_mask) 
   {
    old.force_version = uid_cfg->force_version;
   }

  if (BR_CFG_MAX_HOP & uid_cfg->field_mask) 
   {
    old.max_hop = uid_cfg->max_hop;
   }

  if (BR_CFG_REVISION & uid_cfg->field_mask) 
   {
    old.revision = uid_cfg->revision;
   }

  if (BR_CFG_NAME & uid_cfg->field_mask) 
   {
    strncpy(old.name, uid_cfg->name, 32);
   }

#if STP_DBG
//    printk("Check config  %d\n",xst_id);
#endif
#if 0
  rc = _check_xst_config (&old);
  if (rc != 0) 
   {
#if STP_DBG
    printk("Check config fail!! \n");
#endif
    MSTP_CRITICAL_PATH_END;
    return rc;
   }
#endif

  if ((uid_cfg->field_mask & BR_CFG_STATE ) && (uid_cfg->stp_enabled == STP_DISABLED)) 
   {
#if STP_DBG
//    printk("STP==disabled !!\n");
#endif
    rc = _stp_in_xst_enable (xst_id, STP_DISABLED);
    if (!rc) 
     {
      MSTP_CRITICAL_PATH_END;
      return rc;
     }

    uid_cfg->field_mask &= !BR_CFG_STATE;
    if (!uid_cfg->field_mask) 
     {
      MSTP_CRITICAL_PATH_END;
      return 0;
     }
   }

  /* get current state */
  this = STP_xst_find (xst_id);
  if (this->valid == False) 
   {   
    MSTP_CRITICAL_PATH_END;
    return -1;
    
   }

  this->BridgeIdentifier.prio = (old.bridge_priority & 0xff00) | (this->BridgeIdentifier.prio & 0xff);

  if (this->admin_state == STP_ENABLED) 
   {
#if STP_DBG2
//    printk("update root prio %d !!\n",this->xst_id);
#endif
    STP_rolesel_update_root_priority (this);
   }


  if(xst_id == 0)
   {
//    this->MaxHops=old.max_hop;

    this->ForceVersion = (PROTOCOL_VERSION_T) old.force_version;

    MSTP_RevisionLevel=old.revision;
    strncpy(MSTP_ConfigName,old.name,32);

#if 0
    printk(" copy revision %d\n",MSTP_RevisionLevel);
    printk(" copy name %s\n",MSTP_ConfigName);
#endif
   }

//by flash, 2005/04/26
//  else
   {
    this->xstBridgeTimes.MessageAge = 0;
    this->xstBridgeTimes.MaxAge = old.max_age;
    this->xstBridgeTimes.HelloTime = old.hello_time;
    this->xstBridgeTimes.ForwardDelay = old.forward_delay;

    this->xstBridgeTimes.remainingHops = old.max_hop;
   }

#if 0
  if (this->valid == False) 
   {   
    MSTP_CRITICAL_PATH_END;
    return -1;
    
   }
#endif

  if ((uid_cfg->field_mask & BR_CFG_STATE ) && uid_cfg->stp_enabled != STP_DISABLED && uid_cfg->stp_enabled != this->admin_state) 
   {
    rc =_stp_in_xst_enable (xst_id, uid_cfg->stp_enabled);
      
    if (rc != 0) 
     {
      MSTP_CRITICAL_PATH_END;
      return rc;
     }
    enabled_here = True;
   }

  if (!enabled_here && this->admin_state != STP_DISABLED ) 
   {
    STP_xst_update_after_bridge_management (this);
   }
  MSTP_CRITICAL_PATH_END;
  return 0;
 }

/* ----------------------------------------------------------------------------------------------*/
extern int mstp_tick_enable;
int STP_IN_set_port_cfg (IN int xst_id, IN UID_STP_PORT_CFG_T * uid_cfg)
 {
  register XST_T *this;
  register XST_PORT_T *port;
  register int port_no=uid_cfg->port;

  /* Validation */
  if (PT_CFG_PRIO & uid_cfg->field_mask) 
   {

    if (uid_cfg->port_priority > MAX_PORT_PRIO) 
     {
      return STP_Large_Port_Priority;
     }

    if (0 != (uid_cfg->port_priority % PORT_PRIO_STEP)) 
     {
      return STP_Port_Priority_Granularity;
     }
   }

  if ((PT_CFG_INTCOST | PT_CFG_EXTCOST) & uid_cfg->field_mask) 
   {
    if (uid_cfg->admin_port_path_cost > MAX_ADMIN_PORT_PATH_COST) 
     {
      return STP_Large_Port_Path_Cost;
     }
   }

  MSTP_CRITICAL_PATH_START;

  this = STP_xst_find (xst_id);
  if (this->valid!=True) 
   {   /* it had not yet been created :( */
    MSTP_CRITICAL_PATH_END;
    return STP_Had_Not_Yet_Been_Created;
   }

   {
    port = _stpapi_port_find (this, uid_cfg->port);
    if (port->valid!=True) 
     {  /* port is absent in the xst :( */
      MSTP_CRITICAL_PATH_END;
      return 0;
     }

    if (PT_CFG_MCHECK & uid_cfg->field_mask) 
     {
      port->mcheck = uid_cfg->admin_mcheck;
     }

    if (PT_CFG_EXTCOST & uid_cfg->field_mask) 
     {
#ifdef STP_DBG2
//      printk("set xst %d port %d pcost %d\n",xst_id,port_no, uid_cfg->admin_port_path_cost);
#endif
      port->adminExtPCost = uid_cfg->admin_port_path_cost;
     }

    if (PT_CFG_INTCOST & uid_cfg->field_mask) 
     {
#ifdef STP_DBG2
//      printk("set xst %d port %d pcost %d\n",xst_id,port_no, uid_cfg->admin_port_path_cost);
#endif
      port->adminIntPCost = uid_cfg->admin_port_path_cost;
     }

    if (PT_CFG_PRIO & uid_cfg->field_mask) 
     {
#ifdef STP_DBG
//      printk("set xst %d port %d prio %d\n",xst_id,port_no, uid_cfg->port_priority);
#endif
      port->portId = (uid_cfg->port_priority << 8) + port_no + 1;
     }

    if (PT_CFG_P2P & uid_cfg->field_mask) 
     {
      port->adminPointToPointMac = uid_cfg->admin_point2point;
      port->p2p_recompute = True;
     }

    if (PT_CFG_EDGE & uid_cfg->field_mask) 
     {
      port->adminEdge = uid_cfg->admin_edge;
      port->operEdge = port->adminEdge;
     }

    if (PT_CFG_NON_STP & uid_cfg->field_mask) 
     {
      port->admin_non_stp = uid_cfg->admin_non_stp;
     }

    port->reselect = True;
    port->selected = False;
   }

#ifdef STP_DBG
//      printk(" xst %d update\n",this->xst_id);
#endif

   if (!mstp_tick_enable) 
    {
     MSTP_CRITICAL_PATH_END;
     return 0;
    }

  STP_xst_update (this);

#ifdef STP_DBG
//      printk(" after xst %d update\n",this->xst_id);
#endif
  MSTP_CRITICAL_PATH_END;

  return 0;
 }

