/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************
 * RSTP library - Rapid Spanning Tree (802.1t, 802.1w) 
 * Copyright (C) 2001-2003 Optical Access 
 * Author: Alex Rozin 
 * 
 * This file is part of RSTP library. 
 * 
 * RSTP library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by the 
 * Free Software Foundation; version 2.1 
 * 
 * RSTP library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details. 
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with RSTP library; see the file COPYING.  If not, write to the Free 
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 **********************************************************************/

/* Port Role Transitions state machine : 17.24 */

#include "mstp.h"
#include "base.h"
#include "xst.h"

#define STATES { \
   CHOOSE(INIT_PORT),       \
   CHOOSE(BLOCK_PORT),      \
   CHOOSE(BLOCKED_PORT),    \
   CHOOSE(BACKUP_PORT),     \
   CHOOSE(REROOT),      \
   CHOOSE(SYNCED),   \
   CHOOSE(AGREES),     \
   CHOOSE(PROPOSING),  \
   CHOOSE(PROPOSED),  \
   CHOOSE(ROOT),       \
   CHOOSE(REROOTED),        \
   CHOOSE(LISTEN),      \
   CHOOSE(LEARN),      \
   CHOOSE(FORWARD),    \
   CHOOSE(ACTIVE_PORT),   \
}

#define GET_STATE_NAME STP_roletrns_get_state_name
#include "choose.h"

extern unsigned char mstp_debug,mstp_debug_min,mstp_debug_max;
/*
 * Sets sync TRUE for this tree (the CIST or a given MSTI) for all Ports of the Bridge.
 */

static void setSyncTree (STATE_MACH_T * this)
 {
  register XST_PORT_T *port;
  int i;

  for (i=0;i<MAX_LOGIC_PORT;i++) 
   {
    port = &(this->owner.port->owner->ports[i]);
    port->sync = True;		/* in ROOT_PROPOSED (setSyncTree) */
   }
 }

/*
 * This procedure sets reRoot TRUE for this tree (the CIST or a given MSTI) for all Ports of the Bridge.
 */
static void setReRootTree (STATE_MACH_T * this)
 {
  register XST_PORT_T *port;
  int i;

  for (i=0;i<MAX_LOGIC_PORT;i++) 
   {
    port = &(this->owner.port->owner->ports[i]);
    port->reRoot = True;	/* In setReRootTree */
   }
 }

static Bool compute_all_synced (XST_PORT_T * this)
 {
  register XST_PORT_T *port;
  int i;

  for (i=0;i<MAX_LOGIC_PORT;i++) 
   {
    port = &(this->owner->ports[i]);
    if (port->port_index == this->port_index)
      continue;
    if (!port->synced) 
     {
      return False;
     }
   }

  return True;
 }

static Bool compute_re_rooted (XST_PORT_T * this)
 {
  register XST_PORT_T *port;
  int i;

  for (i=0;i<MAX_LOGIC_PORT;i++) 
   {
    port = &(this->owner->ports[i]);
    if (port->port_index == this->port_index)
      continue;
    if (port->rrWhile) 
     {
      return False;
     }
   }
  return True;
 }

void STP_roletrns_enter_state (STATE_MACH_T * this)
 {
  register XST_PORT_T *port = this->owner.port;
  register XST_T *that;
  register XST_T *cist=STP_xst_find(CIST_INSTANCE_ID);
  register XST_PORT_T *cist_port = &(cist->ports[port->port_index]);

  that = port->owner;

#ifdef STP_DBG2
if(mstp_debug && port->port_index>=mstp_debug_min && port->port_index<=mstp_debug_max)
  printk(" STATE MACHINE: role trans state:%s inst %d port %d \n", GET_STATE_NAME(this->State),that->xst_id,port->port_index);
#endif
  switch (this->State) 
   {
    case BEGIN:
    case INIT_PORT:
//#if 0				/* due 802.1y Z.4 */
#if 1				/* flash */
          port->role = DisabledPort;
#else
          port->role = port->selectedRole = DisabledPort;
          port->reselect = True;
#endif
          port->synced = False;	/* in INIT */
          port->sync = True;		/* in INIT */
          port->reRoot = True;	/* in INIT_PORT */
          port->rrWhile = that->xstRootTimes.ForwardDelay;
          port->fdWhile = that->xstRootTimes.ForwardDelay;
          port->rbWhile = 0;
          break;
    case BLOCK_PORT:
          port->role = port->selectedRole;
          port->learn = port->forward = False;
          break;
    case BLOCKED_PORT:
          port->fdWhile = that->xstRootTimes.ForwardDelay;
          port->synced = True;	/* In BLOCKED_PORT */
          port->rrWhile = 0;
          port->sync = port->reRoot = False;	/* BLOCKED_PORT */
          break;
    case BACKUP_PORT:
          port->rbWhile = 2 * that->xstRootTimes.HelloTime;
          break;

    case REROOT:
          setReRootTree (this);
          break;
    case SYNCED:
          if(port->role != RootPort)
           port->rrWhile=0;
          port->synced=True;
          port->sync=False;
          break;
    case AGREES:
//          port->proposed = port->sync = False;	/* in ROOT_AGREED */
//          port->synced = True;	/* In ROOT_AGREED */
          port->proposed = False;
          port->agree = True;
//          port->newInfoXst=True;  
          cist_port->newInfoCist=True;  
          cist_port->newInfoMsti=True;  
          break;
    /* 17.23.3 */
    case PROPOSING:
          port->proposing = True;	/* in DESIGNATED_PROPOSE */
//          port->newInfoXst = True;
          cist_port->newInfoCist = True;
          cist_port->newInfoMsti = True;
          break;
    /* 17.23.2 */
    case PROPOSED:
          setSyncTree (this);
          port->proposed = False;
          break;
    case ROOT:
//          port->role = RootPort;
          port->rrWhile = that->xstRootTimes.ForwardDelay;
          break;
    case REROOTED:
          port->reRoot = False;	/* In REROOTED */
          break;
    case LISTEN:
          port->learn = port->forward = False;
          port->fdWhile = that->xstRootTimes.ForwardDelay;
          break;
    case LEARN:
          port->fdWhile = that->xstRootTimes.ForwardDelay;
          port->learn = True;
          break;
    case FORWARD:
          port->fdWhile = 0;
          port->forward = True;
          break;
    case ACTIVE_PORT:
          port->role=port->selectedRole;
          break;

   }
 }

Bool STP_roletrns_check_conditions (STATE_MACH_T * this)
{
  register XST_PORT_T *port = this->owner.port;
  register XST_T *that;
  Bool allSynced;
  Bool allReRooted;

  that = port->owner;

  if (this->State == BEGIN) 
   {
    return STP_hop_2_state (this, INIT_PORT);
   }

  if (port->role != port->selectedRole && port->selected && !port->updtInfo) 
   {
    switch (port->selectedRole) 
     {
      case DisabledPort:
      case AlternatePort:
      case BackupPort:
                      return STP_hop_2_state (this, BLOCK_PORT);
      case RootPort:
      case DesignatedPort:
      case MasterPort:
                      return STP_hop_2_state (this, ACTIVE_PORT);
      default:
                      return False;
     }
   }

  switch (this->State) 
   {
    /* 17.23.1 */
    case INIT_PORT:
                   return STP_hop_2_state (this, BLOCK_PORT);
    case BLOCK_PORT:
                   if (!port->selected || port->updtInfo)
                    break;
                   if (!port->learning && !port->forwarding) 
                    {
                     return STP_hop_2_state (this, BLOCKED_PORT);
                    }
                   break;
    case BLOCKED_PORT:
                   if (!port->selected || port->updtInfo)
                    break;
                   if (port->fdWhile != that->xstRootTimes.ForwardDelay || port->sync || port->reRoot || !port->synced) 
                    {
                     return STP_hop_2_state (this, BLOCKED_PORT);
                    }
                   if (port->rbWhile != 2 * that->xstRootTimes.HelloTime && port->role == BackupPort) 
                    {
                     return STP_hop_2_state (this, BACKUP_PORT);
                    }
                   break;
    case BACKUP_PORT:
                   return STP_hop_2_state (this, BLOCKED_PORT);

    case REROOT:
    case SYNCED:
    case AGREES:
    case PROPOSING:
    case PROPOSED:
    case FORWARD:
    case LEARN:
    case LISTEN:
    case REROOTED:
    case ROOT:
                   return STP_hop_2_state (this, ACTIVE_PORT);
    case ACTIVE_PORT:
                   if (!port->selected || port->updtInfo) break;

                   if(port->reRoot && (((port->role == RootPort) && port->forward) || (port->rrWhile == 0)))
                    {
                     return STP_hop_2_state (this, REROOTED);
                    }

                   if((port->rrWhile != that->xstRootTimes.ForwardDelay) && (port->role == RootPort))
                    {
                     return STP_hop_2_state (this, ROOT);
                    }

                   if ((!port->forward) && (!port->reRoot) && (port->role==RootPort)) 
                    {
                     return STP_hop_2_state (this, REROOT);
                    }

                   if((!port->learning && !port->forwarding && !port->synced && (port->role != RootPort)) || (port->agreed && !port->synced) || (port->operEdge && !port->synced) || (port->sync && port->synced))
                    {
                     return STP_hop_2_state (this, SYNCED);
                    }

                   allSynced = compute_all_synced (port);
                   if (allSynced && (port->proposed || !port->agree)) 
                    {
                     return STP_hop_2_state (this, AGREES);
                    }

                   if (!port->forward && !port->agreed && !port->proposing && !port->operEdge && (port->role == DesignatedPort)) 
                    {
                     return STP_hop_2_state (this, PROPOSING);
                    }
               
                   if (port->proposed && !port->synced) 
                    {
                     return STP_hop_2_state (this, PROPOSED);
                    }
               

                   allSynced = compute_all_synced (port);
                   allReRooted = compute_re_rooted (port);
                   if (port->learn && !port->forward && ((port->fdWhile==0) || ((port->role == RootPort) && (allReRooted && (port->rbWhile == 0)) && (that->ForceVersion >= FORCE_RSTP)) || ((port->role == DesignatedPort) && (port->agreed || port->operEdge) && ((port->rrWhile ==0) || !port->reRoot) && !port->sync) || ((port->role == MasterPort) && allSynced))) 
                    {
                     return STP_hop_2_state (this, FORWARD);
                    }

                   allSynced = compute_all_synced (port);
                   allReRooted = compute_re_rooted (port);
                   if(!port->learn && ((port->fdWhile == 0) || ((port->role == RootPort) && (allReRooted && (port->rbWhile == 0)) && (that->ForceVersion >= FORCE_RSTP)) || ((port->role == DesignatedPort) && (port->agreed || port->operEdge) && ((port->rrWhile ==0) || !port->reRoot) && !port->sync) || ((port->role == MasterPort) && allSynced)))
                    {
                     return STP_hop_2_state (this, LEARN);
                    }

                   if((port->learn || port->forward) && !port->operEdge && (port->role != RootPort) && ((port->sync && !port->synced) || (port->reRoot && (port->rrWhile != 0))))
                    {
                     return STP_hop_2_state (this, LISTEN);
                    }

                   break;
   }

  return False;
 }
