/***************************************************************************
 *
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *

***************************************************************************
*
 *  This work includes source codes from the http://www.st.rim.or.jp/~yumo/
 *  Copyright (C) Yumo (Katsuyuki Yumoto) 2000-2002
*/

/* $Id: veth.c,v 1.1.1.1 2009/02/17 09:44:45 anderson Exp $ */
/* VETH: A virtual ethernet device driver for Linux. 
 *
 * This device driver is an implementation of Aggregation of Multiple
 * Link Segments (IEEE 802.3ad).
 *
 * Author: Yumo (Katsuyuki Yumoto) 2000-2002
 *         yumoto@jpn.hp.com or yumo@st.rim.or.jp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */ 


#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/string.h>
#include <linux/ptrace.h>
#include <net/sock.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <linux/in.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/lp.h>
#include <linux/init.h>

#include <linux/etherdevice.h>
#include <linux/inetdevice.h>
#include <linux/skbuff.h>

#include <linux/ioport.h>
#include <asm/bitops.h>
#include <asm/irq.h>
#include <asm/byteorder.h>
#include <linux/spinlock.h>
#include <linux/mii.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>

#include <linux/timer.h>

#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/version.h>
#include <linux/netfilter.h>

#include <asm/unaligned.h>
#include "krndef.h"
#include "krnlacp.h"
#include "krnport.h"
#include "krntrk.h"
#include "krntype.h"
#include "../cclmx/ccl.h"
#include "../cclmx/ccl_mx.h"
#include "veth.h"

/* If it is sure all the LAN adapters report IFF_RUNNING correctly,
 * define following line. E.g. eepro100 device driver report link
 * status by using IFF_RUNNING.  */
#undef VETH_USE_IFF_RUNNING

#define LACP_PER_PORT

struct nf_sockopt_ops lacp_sockopts;
static  char mask6B[6]={0xff,0xff,0xff,0xff,0xff,0xff};

int nveth = MAX_VETH;
int lacpStart = LACP_NOALIVE;
int nport = MAX_PHYIF;
unsigned short sysPri = 0x08000;
struct timeval tv;
unsigned char bAddr[ETH_ALEN];
int ports[MAX_MEMBER]; /* keeps port which state is really "distributing" */
struct lacpdu lacpBuf;


/* Aggregator wildcard. This shall be changed to 0 after most of
 * Ethernet device drivers report their status correctly */
int agg_wc = 0;

#ifdef __LINUX_2_6_19__
#else
MODULE_PARM(nveth, "1-" __MODULE_STRING(MAX_VETH) "i");
MODULE_PARM(agg_wc, "1-" __MODULE_STRING(1) "i");
MODULE_PARM(nport, "1-" __MODULE_STRING(MAX_PHYIF) "i");

/* Round robing distribution. !!!Attention!!! IEEE 802.3ad does not
 * allow round robbing on frame distribution. Distributed frame order
 * may change.  Don't use this feature if you don't want to allow
 * frame order changing. This is disabled by default. */
int rr = 0;
MODULE_PARM(rr, "1-" __MODULE_STRING(1) "i");
#endif


/* Write the buffer (contents of the port statuses) to a PROCfs file */

static char *state2str_mux(enum muxm_state s)
{
	switch (s)
	{
	case DETACHED:
		return "DETACH";
	case WAITING:
		return "WAIT";
	case ATTACHED:
		return "ATTACH";
	case COLLECTING:
		return "CLLCT";
	case DISTRIBUTING:
		return "DSTRBT";
	}
	return NULL;
}

static char *state2str_rcv(enum rcvm_state s)
{
	switch (s)
	{
	case INITIALIZE:
		return "INIT";
	case PORT_DISABLED:
		return "PORTds";
	case EXPIRED:
		return "EXPR";
	case LACP_DISABLED:
		return "LACPds";
	case DEFAULTED:
		return "DFLT";
	case CURRENT:
		return "CRRNT";
	}
	return NULL;
}

static char *state2str_pr(enum perio_state s)
{
	switch (s)
	{
	case NO_PERIODIC:
		return "NoPRD";
	case FAST_PERIODIC:
		return "FstPRD";
	case SLOW_PERIODIC:
		return "SlwPRD";
	case PERIODIC_TX:
		return "PrdTX";
	}
	return NULL;
}

#if 0
static char *state2str_ac(enum acm_state s)
{
	switch (s)
	{
	case NO_ACTOR_CHURN:
		return "No_ACT";
	case ACTOR_CHURN_MONITOR:
		return "ActC_M";
	case ACTOR_CHURN:
		return "ActCHR";
	}
	return NULL;
}


static char *state2str_pc(enum pcm_state s)
{
	switch (s)
	{
	case NO_PARTNER_CHURN:
		return "NoPRTN";
	case PARTNER_CHURN_MONITOR:
		return "PtnC_M";
	case PARTNER_CHURN:
		return "PtnCHR";
	}
	return NULL;
}
#endif

/* 
 * Linux module subsystem does not provide module handling
 * facility by corresponding device name.
 * I don't want to make veth driver depend on module name
 * but it depends on device name (e.g. eth0).
 * However there are no way to lock/unlock the module by device name.
 * Therefore I need to convert device name into module name, and
 * then lock/unlock the module...
 */
/* find_module() is exported by patch */


static u16 get_link_stat(unsigned short portNo)
{
	Tbool	bLinkState = False;
	struct 	ifreq ifr;
	struct 	mii_ioctl_data *data = (struct mii_ioctl_data *)&ifr.ifr_data;
		
	/* Status register is mapped at address 1 in 
	 * MII management register set space.
	 * (IEEE Std 802.3, 1998)*/
	 
//	 printk("get_link_stat=====Begin\n");
	data->reg_num = MII_BMSR; 
	K_PortGetLink(portNo-1, &bLinkState);
//	printk("get_link_stat====portNo=%d,bLinkState=%d\n",portNo, bLinkState);
	if (bLinkState) {
		/* m->val_out contains status
		 * bit 2: 1 = link is up (BMSR_LSTATUS)
		 * bit 5: 1 = auto-negotiation process completed (BMSR_ANEGCOMPLETE)
		 */
		return (BMSR_LSTATUS | BMSR_ANEGCOMPLETE);
	}
	return 0;
}


/********************************************************
 *
 *   Link Aggregation Control Protocol
 *
 ********************************************************/


/* 43.4.9 */
static void veth_record_pdu(struct lacpdu *buf, struct veth_port *p)
{
	int param_match;
	int is_active_link;
	int aggNo;

	p->pv.partner_oper_port_num = ntohs(buf->actor_port);
	p->pv.partner_oper_port_pri = ntohs(buf->actor_port_pri);
	memcpy(p->pv.partner_oper_sys, buf->actor_sys, ETH_ALEN);

	if (VETH_DEBUG(VETH_DEBUG_RCVM, 1)){
		printk(KERN_INFO "veth_record_pdu(): port:%d, partn:%02x%02x%02x-%02x%02x%02x\n",
			   p->pv.actor_port_num,
			   p->pv.partner_oper_sys[0],
			   p->pv.partner_oper_sys[1],
			   p->pv.partner_oper_sys[2],
			   p->pv.partner_oper_sys[3],
			   p->pv.partner_oper_sys[4],
			   p->pv.partner_oper_sys[5]);
	}

	p->pv.partner_oper_sys_pri = ntohs(buf->actor_sys_pri);
	p->pv.partner_oper_key = ntohs(buf->actor_key);
	p->pv.partner_oper_port_state = buf->actor_state;
	

	p->pv.actor_oper_port_state &= ~PS_DEFAULTED;


	/* param_match shows whether partner recognizes actor's parameters
       correctly */

	param_match = 
		(p->pv.actor_port_num == ntohs(buf->partner_port)) &&
		(p->pv.actor_port_pri == ntohs(buf->partner_port_pri)) &&
		(memcmp(veth_sysvar.actor_sys, buf->partner_sys, ETH_ALEN) == 0) &&
		(veth_sysvar.actor_sys_pri == ntohs(buf->partner_sys_pri)) &&
		(p->pv.actor_oper_port_key == ntohs(buf->partner_key)) &&
		((p->pv.actor_oper_port_state & PS_AGGREGATION) ==
		 (buf->partner_state & PS_AGGREGATION));

	is_active_link = (buf->actor_state & PS_LACP_ACTIVITY) ||
		((p->pv.actor_oper_port_state & PS_LACP_ACTIVITY) && 
		(buf->partner_state & PS_LACP_ACTIVITY));


	if (is_active_link &&
		((param_match && (buf->actor_state & PS_SYNCHRONIZATION)) ||
		((buf->actor_state & (PS_AGGREGATION | PS_SYNCHRONIZATION)) ==
		 PS_SYNCHRONIZATION))){
		p->pv.partner_oper_port_state |= PS_SYNCHRONIZATION;
	} else {
		p->pv.partner_oper_port_state &= ~PS_SYNCHRONIZATION;
	}


	/* update aggregator's partner_sys if this port attaches an
     * aggregator */
	if (p->pv.actor_port_agg_id){

		aggNo = p->pv.actor_port_agg_id-1;

		memcpy(veth_aggs[aggNo].av.partner_sys, p->pv.partner_oper_sys, ETH_ALEN);
		veth_aggs[aggNo].av.partner_sys_pri = p->pv.partner_oper_sys_pri;
	}

	return;
}


/* copy operational partner's parameter values from administrative
   parameter values */
static void veth_record_default(struct veth_port *p)
{
	p->pv.partner_oper_port_num = p->pv.partner_admin_port_num;
	p->pv.partner_oper_port_pri = p->pv.partner_admin_port_pri;
	memcpy(p->pv.partner_oper_sys, p->pv.partner_admin_sys, ETH_ALEN);
	p->pv.partner_oper_sys_pri = p->pv.partner_admin_sys_pri;
	p->pv.partner_oper_key = p->pv.partner_admin_key;
	p->pv.partner_oper_port_state = p->pv.partner_admin_port_state;
	p->pv.partner_oper_port_state &= ~PS_SYNCHRONIZATION;
	p->pv.actor_oper_port_state |= PS_DEFAULTED;
	return;
}

static void veth_update_selected(struct lacpdu *buf, struct veth_port *p)
{
	int param_match;

	/* param_match shows whether actor recognizes partner's parameters
       correctly */
	param_match = 
		(p->pv.partner_oper_port_num == ntohs(buf->actor_port)) &&
		(p->pv.partner_oper_port_pri == ntohs(buf->actor_port_pri)) &&
		(memcmp(p->pv.partner_oper_sys, buf->actor_sys, ETH_ALEN) == 0) &&
		(p->pv.partner_oper_sys_pri == ntohs(buf->actor_sys_pri)) &&
		(p->pv.partner_oper_key == ntohs(buf->actor_key)) &&
		((p->pv.partner_oper_port_state & PS_AGGREGATION) ==
		(buf->actor_state & PS_AGGREGATION));

	if (!param_match){
		if (VETH_DEBUG(VETH_DEBUG_RCVM, 1)){
			printk (KERN_INFO "p=%d,p_num:%d p_pri:%d sys:%d s_pri:%d key:%d p_st:%d\n",
					p->pv.actor_port_num,
					(p->pv.partner_oper_port_num == ntohs(buf->actor_port))?1:0,
					(p->pv.partner_oper_port_pri == ntohs(buf->actor_port_pri))?1:0,
					(memcmp(p->pv.partner_oper_sys, buf->actor_sys, ETH_ALEN) == 0)?1:0,
					(p->pv.partner_oper_sys_pri == ntohs(buf->actor_sys_pri))?1:0,
					(p->pv.partner_oper_key == ntohs(buf->actor_key))?1:0,
					((p->pv.partner_oper_port_state & PS_AGGREGATION) ==
					 (buf->actor_state & PS_AGGREGATION))?1:0);
		}
		p->pv.selected = UNSELECTED;
	}
#ifndef LACP_PER_PORT
	else{

		p->pv.selected = SELECTED;
	}
#endif

	return;
}

static void veth_update_default_selected(struct veth_port *p)
{
	int param_match;

	/* param_match shows whether administrative parameter values match
       partner's parameters which actor recognizes */
	param_match = 
		(p->pv.partner_admin_port_num == p->pv.partner_oper_port_num) &&
		(p->pv.partner_admin_port_pri == p->pv.partner_oper_port_pri) &&
		(memcmp(p->pv.partner_admin_sys, p->pv.partner_oper_sys, ETH_ALEN) == 0) &&
		(p->pv.partner_admin_sys_pri == p->pv.partner_oper_sys_pri) &&
		(p->pv.partner_admin_key == p->pv.partner_oper_key) &&
		((p->pv.partner_admin_port_state & PS_AGGREGATION) ==
		(p->pv.partner_oper_port_state & PS_AGGREGATION));

	if (!param_match){
		p->pv.selected = UNSELECTED;
	} else {
		/* 43.4.12 
		 *
		 * Since all operational parameters are now set to locally
		 * administrated values, there can be no disagreement as to
		 * the Link Aggregation Group, so the
		 * partner_oper_port_state.synchronization variable is set to
		 * _TRUE.*/
		p->pv.partner_oper_port_state |= PS_SYNCHRONIZATION;
	}

	return;
}

static void veth_update_ntt(struct lacpdu *buf, struct veth_port *p)
{
	int param_match;

	/* param_match shows partner recognizes actor's parameters correctly */
	param_match = 
		(p->pv.actor_port_num == ntohs(buf->partner_port)) &&
		(p->pv.actor_port_pri == ntohs(buf->partner_port_pri)) &&
		(memcmp(veth_sysvar.actor_sys, buf->partner_sys, ETH_ALEN) == 0) &&
		(veth_sysvar.actor_sys_pri == ntohs(buf->partner_sys_pri)) &&
		(p->pv.actor_oper_port_key == ntohs(buf->partner_key)) &&
		((p->pv.actor_oper_port_state & (PS_LACP_ACTIVITY | PS_LACP_TIMEOUT | PS_AGGREGATION)) ==
		(buf->partner_state & (PS_LACP_ACTIVITY | PS_LACP_TIMEOUT | PS_AGGREGATION)));

	if (!param_match){
		p->pv.need_to_xmit = _TRUE;
	}
	return;
}

static void veth_attach_mux_to_agg(struct veth_port *p)
{
	p->pv.attached = _TRUE;
	return;
}

static void veth_detach_mux_from_agg(struct veth_port *p)
{
	p->pv.attached = _FALSE;
	return;
}
#ifdef Taihua	

	int agg, port, i, m, n;
	int *lp;
	struct veth_agg *ag;

	p->pv.attached = _FALSE;

	/* get aggregator selected for this port */
	agg = p->pv.actor_port_agg_id;

	if (!agg){
		/* already detached */
		return;
	}

	/* get port id */
	port = p->pv.actor_port_num;

	if (!port){
		/* it shall not come here... */
		return;
	}

	agg--;

	/* find the port from lag_port list */
	ag = veth_aggs+agg;
	lp = ag->av.lag_ports;
	for (i = 0; i < MAX_PHYIF; i++){
		if (lp[i] == port && p->pv.selected == UNSELECTED){
			/* found. detach it */
			p->pv.actor_port_agg_id = 0;
			for (m = i, n = i+1; n < MAX_PHYIF; m++, n++){
				lp[m] = lp[n];
			}
			lp[m] = 0;
			if (m == 0){ /* last one */
				memset(ag->av.partner_sys, 0, ETH_ALEN);
				ag->av.partner_sys_pri = 0;
			}
			break;
		}
	}

}
#endif //Taihua

static void veth_ena_collect(struct veth_port *p)
{
	int byPortId;
	
	p->pv.actor_oper_port_state |= PS_COLLECTING;
	
	byPortId = p->pv.actor_port_num - 1;
	// Enable Nway mask

#ifdef Taihua
	BIOvRegSpecBitOn(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2));
        BIOvRegSpecBitOn(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2 + 1));
        // Enable MAC in
        BIOvRegBitsOn ( (UINT16)(PHY_PORT_ABL_BASE + byPortId),
                            PTN_PHY_PORT_ABL_IN_ENABLE );
#endif //Taihua

	return;
}

static void veth_dis_collect(struct veth_port *p)
{
	int byPortId;
	
	p->pv.actor_oper_port_state &= ~PS_COLLECTING;
		
	byPortId = p->pv.actor_port_num - 1;
	// Disable Nway mask

#ifdef Taihua
        BIOvRegSpecBitOff(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2));
        BIOvRegSpecBitOff(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2 + 1));
        // Disable MAC in
        BIOvRegBitsOff( (UINT16)(PHY_PORT_ABL_BASE + byPortId),
                        PTN_PHY_PORT_ABL_IN_ENABLE);
#endif //Taihua

	return;
}


/* Search for proper device (port) for transmitting this skb.
 *
 * Distribution algorithm shall not cause mis-ordering of frames
 * that are part of any given conversation (43.2.4). So this 
 * implementation uses hash by destination MAC address */
static void veth_find_distrib_port(struct veth_agg *ag)
{
	int i, j;
	int k; /* num of ports which attaches this aggregator */
	struct veth_port *p;

	memset(ports, 0, MAX_MEMBER * sizeof(int));
	
	for (i = 0, k = 0; i < MAX_MEMBER; i++){
		if (!ag->av.lag_ports[i]){ /* lag_ports is built by selection logic */
			ports[k] = 0;
			break;
		} else {
			
			j = ag->av.lag_ports[i]-1;
			p = veth_ports+j;
			if (p->pv.attached && (p->pv.actor_oper_port_state & PS_DISTRIBUTING)){
				ports[k++] = j;
			}else{
			     	K_TrkSetPortAggrId((unsigned char)j, (unsigned char)ag->av.agg_id);
#ifdef LACP_PER_PORT
				cclmx_AddPortToTrk((unsigned char)ag->av.agg_id-1, (unsigned char)j);
#endif
			     	K_TrkSetPortAdmCri((unsigned char)j, 0x0);
			}
//			printk("veth_find_distrib_port() port_%d k=%d\n",ag->av.lag_ports[i],k);
		}
	}

	spin_lock(&ag->lock);
	if ((1 <= k) && (k <= MAX_MEMBER)){
		memcpy(ag->ports, ports, k*sizeof(int));
		ag->n_ports = k;
	}

	for (i = 0; (i < k) && (i < MAX_MEMBER); i++) {
		K_TrkSetPortAdmCri((unsigned char)ports[i], KRN_TRK_PORT_ADM_CRI[k-1][i]);
	}

	spin_unlock(&ag->lock);
	return;
}

static void veth_ena_distrib(struct veth_port *p)
{
	struct veth_agg *ag;
	int agg_id, byPortId;

	agg_id = p->pv.actor_port_agg_id;
	ag = veth_aggs + agg_id - 1;

	p->pv.actor_oper_port_state |= PS_DISTRIBUTING;

	if (agg_id){
		veth_find_distrib_port(ag);

		/* MTU check */
	}

	byPortId = p->pv.actor_port_num - 1;

#ifdef Taihua
	// Enable Nway mask
	BIOvRegSpecBitOn(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2));
        BIOvRegSpecBitOn(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2 + 1));
        // Enable MAC Out
        BIOvRegBitsOn ( (UINT16)(PHY_PORT_ABL_BASE + byPortId),
                            PTN_PHY_PORT_ABL_OUT_ENABLE );
#endif //Taihua
	return;
}

static void veth_dis_distrib(struct veth_port *p)
{
	struct veth_agg *ag;
	int agg_id, byPortId;

	agg_id = p->pv.actor_port_agg_id;
	ag = veth_aggs + agg_id - 1;

	p->pv.actor_oper_port_state &= ~PS_DISTRIBUTING;

	if (agg_id)
		veth_find_distrib_port(ag);
	
	byPortId = p->pv.actor_port_num - 1;
	
#ifdef Taihua
	// Disable Nway mask
        BIOvRegSpecBitOff(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2));
        BIOvRegSpecBitOff(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2 + 1));
        // Disable MAC Out
        BIOvRegBitsOff( (UINT16)(PHY_PORT_ABL_BASE + byPortId),
                        PTN_PHY_PORT_ABL_OUT_ENABLE);
#endif //Taihua
	return;
}


/* end of 43.4.9 */


/* This is an optional function */
static void veth_marker_receiver(struct sk_buff *skb, struct net_device *dev)
{
	/* veth_marker_receiver is not implemented yet */
	/* discard */
#ifdef __LINUX_2_6_19__
#else
	skb_unlink(skb);
#endif
	kfree_skb(skb);
	return;
}

static void veth_marker_responder(struct sk_buff *skb, struct net_device *dev)
{
	struct lacpdu *buf;

	if (!skb){
		printk(KERN_CRIT "veth: veth_rx_frame: skb is null!\n");
		return;
	}

	buf = (struct lacpdu *)skb->mac.raw;

	/* modify TLV_type */
	buf->first_tlv_type = 0x02; /* marker response info */
	skb_push(skb, skb->data - skb->mac.raw);
	skb->protocol = eth_type_trans(skb, skb->dev);
	
 	CCL_P(skb)->wSize = MARKERPDU_LEN;
  	CCL_P(skb)->wVid = 0;
  	CCL_P(skb)->byPriority = 0;
  	put_unaligned(0x1L << CCL_P(skb)->bySrcPortId, (unsigned long *) &(CCL_P(skb)->dwDstPortMsk));
  	put_unaligned(0, (unsigned long *) &(CCL_P(skb)->dwTagPortMsk));

	dev_queue_xmit(skb);

	return;
}




/*-------------------- Selection logic ---------------------*/

static void sort_ports_by_pri(void)
{
	int i, j, max;
	struct veth_port **p, **q, **t;
	static struct veth_port *t_opl[MAX_PHYIF];
	
//	printk( "sort_ports_by_pri: =====Begin !!\n");

	for (j = 0; j < MAX_PHYIF; j++){
		t_opl[j] = NULL;
	}

	p = t_opl;
	for (j = 0, max = 0; j < MAX_PHYIF; j++){
		if (veth_ports[j].pv.interceptedCCL){
//			printk( "sort_ports_by_pri: =====the %dst port exist!!\n",j+1);
			*(p++) = veth_ports+j;
			max++;
		}
	}

	for (j = 0, q = t_opl; j < max; j++, q++){
		for (i = j+1, p = q+1; i < max; i++, p++){
			if ((*p)->pv.actor_port_pri < (*q)->pv.actor_port_pri){
				t = p; p = q; q = t;
			}
		}
	}

	memcpy(opl, t_opl, MAX_PHYIF*sizeof(struct veth_port *));
	need_to_rebuild_opl = _FALSE;

	return;
}

static int get_num_of_port_attached(int an)
{
	int i;
	int ret = -1;

	if (!an){
		return ret;
	}

//	if (!veth_aggs[an-1].dev){
//		return ret;
//	}

	for (i = 0, ret = 0; i < MAX_PHYIF; i++){
		if (veth_ports[i].pv.actor_port_agg_id == an &&
			veth_ports[i].pv.attached){
			ret++;
		}
	}

	return ret;
}


static int not_assigned_mac(unsigned char *ad)
{
	int ret = _FALSE;
	int i;

	for (i = 0; i < ETH_ALEN/2; i++){
		if (ad[i] == 0xfc){
			ret = _TRUE;
		}
	}

	return ret;
}



/* This function search for ports which ready_n is _TRUE, then
   caluculate each aggregator's Ready variable */

/* NOTE.  At this time point, there are no ethernet device driver report
 * media speed, link status and duplex mode by standard way. Therefore
 * veth cannot recognize correct status of such devices. On the other
 * hand, if veth follows the standard, veth cannot handle such
 * "unknown" status devices. i.e. there are no places where veth can
 * work. Therefore I decided to implement a wildcard option (agg_wc)
 * for aggregator until most of ethernet devices report their status
 * correctly... I've introduced media, link_status and duplex
 * mode field in the net_device_stats structure with veth. I hope
 * ethernet device drivers report their status using it in the
 * future... */

static void veth_scan_ready_n(struct veth_port *p)
{
	int i, j, k;
	int key;
	struct veth_agg *ag;
	struct net_device *dev;

	if ( !(p->pv.interceptedCCL)) {
		return;
	}

	/* loopback detection (links may be on the shared media) */
	if (memcmp(veth_sysvar.actor_sys, p->pv.partner_oper_sys, ETH_ALEN) == 0 &&
		(p->pv.partner_oper_key & SUBKEY_MASK_UPPER) == (p->pv.actor_oper_port_key & SUBKEY_MASK_UPPER)){
		i = p->pv.actor_port_agg_id;
		if (i){
			p->pv.selected = UNSELECTED;
			p->pv.loop = _TRUE; /* 43.6.4 */
			p->pv.actor_oper_port_state &= ~PS_AGGREGATION; /* 43.6.4 */
		}
		return;
	}

	if ( p->pv.tbd_cnt){
		return;
	}

	if (p->muxm.state == DETACHED &&
		p->pv.selected == UNSELECTED &&
		p->pv.ready_n &&
		!p->pv.actor_port_agg_id){

		
		/* If agg_wc is disabled, veth does not allow unknown media
		 * speed device */
		if (!agg_wc && ((p->pv.actor_oper_port_key & SUBKEY_MASK_LOWER) == SUBKEY_UNKNOWN)){
			return;
		}

#ifdef LACP_PER_PORT
		/* if partner_oper_sys empty, return */
		if (memcmp(p->pv.partner_oper_sys, ZERO_ETH, ETH_ALEN) == 0){
			return;
		}
#endif

		/* apply admin params to oper params */
		p->pv.actor_oper_port_key &= 
			SUBKEY_MASK_LOWER; /* comes from veth_port_mon() */
		p->pv.actor_oper_port_key |= 
			(p->pv.actor_admin_port_key & SUBKEY_MASK_UPPER);
		key = p->pv.actor_oper_port_key;

		/* search for proper aggregator */
		for (i = 0, ag = veth_aggs; i < MAX_VETH && ag->av.agg_id; i++, ag++){

			if (!ag->av.lag_ports[0] &&
				ag->av.actor_oper_agg_key != ag->av.actor_admin_agg_key){
				/* There are no ports belong to this
				 * aggregator. So it is safe to change
				 * operational aggregator key.  */
				ag->av.actor_oper_agg_key = ag->av.actor_admin_agg_key;
			}

#ifndef LACP_PER_PORT
			if ((ag->av.actor_oper_agg_key & SUBKEY_MASK_UPPER) == (key & SUBKEY_MASK_UPPER)){
#endif

				/* goto tail (k never reach MAX_PHYIF) */
				for (k = 0; k < MAX_PHYIF; k++){
					if (!ag->av.lag_ports[k]){
						break;
					}
				}

				if (k > 0){
					struct veth_port *firstPort;
					unsigned char aggAble, sn;


					/* If an individual port has been assigned, or if
					 * this port is an individual port and a port has
					 * been assigned, then not select. */
					firstPort = veth_ports + ag->av.lag_ports[0] - 1;
					aggAble = (firstPort->pv.actor_oper_port_state & firstPort->pv.partner_oper_port_state)
						& PS_AGGREGATION;
					sn = (p->pv.actor_oper_port_state & p->pv.partner_oper_port_state)
						& PS_AGGREGATION;
					if (!aggAble || !sn){
#ifdef LACP_PER_PORT
						break;
#else
						continue;
#endif
					}

					/* If LAG_ID is different from first
					 * port's one, not aggregate each other */

//					if ((memcmp(p->pv.partner_oper_sys, ZERO_ETH ,ETH_ALEN) !=0) &&
					if ((firstPort->pv.partner_oper_key != p->pv.partner_oper_key) ||
						(firstPort->pv.actor_oper_port_key != p->pv.actor_oper_port_key) ||
						(memcmp(firstPort->pv.partner_oper_sys, p->pv.partner_oper_sys, ETH_ALEN) != 0)){

//					   printk(KERN_INFO "p:%d 1stP partnSys:%12x, partnSys:%12x\n",
//					   p->pv.actor_port_num,
//					   firstPort->pv.partner_oper_sys,
//					   p->pv.partner_oper_sys);

						continue;
					}
				}

				/* media speed of the first port is applied to the aggregator.
				 * It will be used until all the ports detach MUX from the aggregator. */
				if (!k){
					ag->av.actor_oper_agg_key &= SUBKEY_MASK_UPPER;
					ag->av.actor_oper_agg_key |= (key & SUBKEY_MASK_LOWER);
				}

				if (not_assigned_mac(ag->av.agg_mac)){


				 		dev = dev_get_by_name("eth0");
				 		memcpy(bAddr, dev->dev_addr, ETH_ALEN);
				 		bAddr[ETH_ALEN -1] += p->pv.actor_port_num-1;
//						memcpy(ag->dev->dev_addr, bAddr, ETH_ALEN);
						memcpy(ag->av.agg_mac, bAddr, ETH_ALEN);

//						printk("veth_scan_ready_n(): ===== ag-mac_addr=");
//                                                for (j=0; j<6; j++) printk("%x:",*(ag->av.agg_mac+j));
//                                                printk("\n");
				}

				ag->av.lag_ports[k] = p->pv.actor_port_num;
				p->pv.actor_port_agg_id = i+1;
//		printk("veth_scan_ready_n()##1##-agId=%d, ready=%d,portNo=%d, ready_n=%d\n",ag->av.agg_id,ag->av.ready, p->pv.actor_port_num, p->pv.ready_n);
					
				/* make it SELECTED after make sure no restriction for
				   attaching */
				p->pv.selected = STANDBY;
				break;
#ifndef LACP_PER_PORT
			}
#endif
		}
	}


	if (p->pv.selected == STANDBY){
		int n = get_num_of_port_attached(p->pv.actor_port_agg_id);

		if (n < 0){
			printk(KERN_CRIT "veth_scan_ready_n(): unexpected agg_id\n");
			return;
		}

		if (n < nport){
			if (VETH_DEBUG(VETH_DEBUG_SEL, 1)){
				printk(KERN_INFO "port:%d act_sys:%02x%02x%02x-%02x%02x%02x, partn_sys:%02x%02x%02x-%02x%02x%02x\n",
					   p->pv.actor_port_num,
					   veth_sysvar.actor_sys[0],
					   veth_sysvar.actor_sys[1],
					   veth_sysvar.actor_sys[2],
					   veth_sysvar.actor_sys[3],
					   veth_sysvar.actor_sys[4],
					   veth_sysvar.actor_sys[5],
					   p->pv.partner_oper_sys[0],
					   p->pv.partner_oper_sys[1],
					   p->pv.partner_oper_sys[2],
					   p->pv.partner_oper_sys[3],
					   p->pv.partner_oper_sys[4],
					   p->pv.partner_oper_sys[5]);
			}
			p->pv.selected = SELECTED;
			p->pv.need_to_xmit = _TRUE;
		} else {
			/* remain STANDBY because of restriction (43.6.1) */
		}
	}

	/* calculate ready value from ready_N value of each port which
       belongs to the aggregator. */

	i = p->pv.actor_port_agg_id - 1;
	if (i < 0){
		return;
	}

	ag = veth_aggs + i;

	for (j = 0; j < MAX_PHYIF ; j++){
		if (ag->av.lag_ports[j]){
			p = veth_ports + (ag->av.lag_ports[j]-1);
//			printk("veth_scan_ready_n()-##2## agId=%d, ready=%d,p=%d\n",ag->av.agg_id,ag->av.ready, p->pv.actor_port_num);
			if (p->pv.ready_n){
				ag->av.ready = _TRUE;
				break;
			}
		}
	}

	return;
}


/*-------------------- End of Selection logic ---------------------*/


/*-------------------- Timers ---------------------*/

static void vt_start(struct veth_timer *vt, int exp)
{
	vt->active = _TRUE;
	vt->exp = exp;
	vt->tim = 0;

	return;
}

static void vt_stop(struct veth_timer *vt)
{
	vt->tim = 0;
	vt->active = _FALSE;

	return;
}

static int is_vt_active(struct veth_timer *vt)
{
	return vt->active;
}




static void vt_count(struct veth_timer *vt)
{
	if (vt->active){
		if (vt->tim == vt->exp){
			vt->tim = 0;					
			vt->active = _FALSE;
		} else {
			vt->tim++;
		}
	}

	return;
}


static void veth_clear_timer(struct veth_timer *vt)
{
	memset(vt, 0, sizeof(struct veth_timer));
	return;
}

static void veth_unregister_timer(struct veth_timer *vt)
{
	struct veth_timer *p, *q;

	if (!vt){
		return;
	}

	if (ltimer == vt){
		p = ltimer->next;
		ltimer->next = 0;
		ltimer = p;
		return;
	}

	p = ltimer->next;
	q = ltimer;
	for (; p; p = p->next, q = q->next){
		if (vt == p){
			q->next = p->next;
			p->next = 0;
		}
	}
	return;
}

static void veth_register_timer(struct veth_timer *vt)
{
	if (!vt){
		return;
	}

	vt->tick_func = vt_count;
	vt->next = ltimer;
	ltimer = vt;
	return;
}

/*-------------------- end of Timers --------------*/


/*-------------- Periodic Transmit Machine ---------------*/
/* 43.4.13 */
static void veth_prdTx_tick(struct veth_port *p)
{
	struct perio_mach *m = &(p->prm);
	int go_no_perio;

	go_no_perio = p->pv.begin || !p->pv.lacp_ena || !p->pv.port_enabled ||
		(!(p->pv.actor_oper_port_state & PS_LACP_ACTIVITY) &&
		 !(p->pv.partner_oper_port_state & PS_LACP_ACTIVITY));

	/* state machine */
	if (m->state == NO_PERIODIC){
		if (m->do_it){
			vt_stop(&(p->periodic_timer));
			m->do_it = _FALSE;
		}
		if (!go_no_perio){
			m->state = FAST_PERIODIC;
			m->do_it = _TRUE;
		}
	} else if (m->state == FAST_PERIODIC){
		if (go_no_perio){
			m->state = NO_PERIODIC;
			m->do_it = _FALSE;
		} else {
			if (m->do_it){
				vt_start(&(p->periodic_timer), Fast_Periodic_Time);
				/* 43.4.16 */
				p->pv.xmit_tim_update = Fast_Periodic_Time;
				p->pv.xmit_ena = _TRUE;
				m->do_it = _FALSE;
			}
			if (!(p->pv.partner_oper_port_state & PS_LACP_TIMEOUT)){
				/* long timeout */
				m->state = SLOW_PERIODIC;
				m->do_it = _TRUE;
			}
			/* short timeout */
			if (!is_vt_active(&(p->periodic_timer))){
				m->state = PERIODIC_TX;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == SLOW_PERIODIC){
		if (go_no_perio){
			m->state = NO_PERIODIC;
			m->do_it = _FALSE;
		} else {
			if (m->do_it){
				vt_start(&(p->periodic_timer), Slow_Periodic_Time);
				/* 43.4.16 */
				p->pv.xmit_tim_update = Fast_Periodic_Time;
				p->pv.xmit_ena = _TRUE;
				m->do_it = _FALSE;
			}
			if ((p->pv.partner_oper_port_state & PS_LACP_TIMEOUT) ||
				!is_vt_active(&(p->periodic_timer))){
				m->state = PERIODIC_TX;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == PERIODIC_TX){
		if (go_no_perio){
			m->state = NO_PERIODIC;
			m->do_it = _FALSE;
		} else {
			if (m->do_it){
				p->pv.need_to_xmit = _TRUE;
				m->do_it = _FALSE;
			}
			if (p->pv.partner_oper_port_state & PS_LACP_TIMEOUT){
				/* short timeout */
				m->state = FAST_PERIODIC;
				m->do_it = _TRUE;
			} else {
				m->state = SLOW_PERIODIC;
				m->do_it = _TRUE;
			}
		}
	}
	return;
}
/*-------------- Periodic Xmit Machine ---------------*/


/*-------------- receive machine -------------------*/

/* 
 * 43.4.12 (receive machine) 
 * This describes state machine for each port
 */

/* action at INITIALIZE state */
static void veth_rcvm_init(struct veth_port *p)
{
	p->pv.selected = UNSELECTED;
	veth_record_default(p);
	p->pv.port_moved = _FALSE;
	p->pv.actor_oper_port_state &= ~PS_EXPIRED;

	return;
}

/* action at PORT_DISABLED state */
static void veth_rcvm_port_dis(struct veth_port *p)
{

	p->pv.partner_oper_port_state &= ~PS_SYNCHRONIZATION;

	/* The standard does not mention explicitly where PS_LACP_ACTIVITY
	 * shall be set...  */
//	p->pv.actor_oper_port_state |= PS_LACP_ACTIVITY;

	/* reset "loop condition" */
	p->pv.loop = _FALSE;
	p->pv.actor_oper_port_state |= PS_AGGREGATION;

	return;
}

static void veth_check_if_port_moved(struct veth_port *p)
{
	int i;

	for (i = 0; i < MAX_PHYIF; i++){
		if (veth_ports[i].pv.interceptedCCL &&
			p != veth_ports+i &&
			(veth_ports[i].pv.partner_oper_port_num == p->pv.partner_oper_port_num) &&
			(memcmp(veth_ports[i].pv.partner_oper_sys, p->pv.partner_oper_sys, ETH_ALEN) == 0)){
			p->pv.port_moved = _TRUE;
		}
	}
	return;
}

/* action at EXPIRED state */
static void veth_rcvm_exp(struct veth_port *p)
{
	p->pv.partner_oper_port_state &= ~PS_SYNCHRONIZATION;
	p->pv.partner_oper_port_state |= PS_LACP_TIMEOUT; /* short */
	vt_start(&(p->current_while_timer), Short_Timeout_Time);
	p->pv.actor_oper_port_state |= PS_EXPIRED;
	return;
}

/* action at LACP_DISABLED state */
static void veth_rcvm_lacp_dis(struct veth_port *p)
{
	p->pv.selected = UNSELECTED;
	veth_record_default(p);
	p->pv.partner_oper_port_state &= ~(PS_AGGREGATION | PS_EXPIRED);

	/* The standard does not mention explicitly when PS_LACP_ACTIVITY
	 * shall be set...  */
	p->pv.actor_oper_port_state &= ~PS_LACP_ACTIVITY;
	return;
}

/* action at DEFAULTED state */
static void veth_rcvm_def(struct veth_port *p)
{
	veth_update_default_selected(p);
	veth_record_default(p);
	p->pv.actor_oper_port_state &= ~PS_EXPIRED;
	return;
}

/* action at CURRENT state */
static void veth_rcvm_cur(struct veth_port *p)
{
	if (p->pv.actor_oper_port_state & PS_LACP_TIMEOUT){
		vt_start(&(p->current_while_timer), Short_Timeout_Time);
	} else {
		vt_start(&(p->current_while_timer), Long_Timeout_Time);
	}
	p->pv.actor_oper_port_state &= ~PS_EXPIRED;
	return;
}



static void veth_drop_lacpdu_buf(struct veth_port *p)
{
	for (;p->rcvm.ring.r != p->rcvm.ring.w;){
		/* LACPDU arrives (drop it)*/
		kfree(p->rcvm.ring.buf[p->rcvm.ring.r]);
		p->rcvm.ring.buf[p->rcvm.ring.r] = NULL;
		p->rcvm.ring.r++;
		if (p->rcvm.ring.r == MAX_RING){
			p->rcvm.ring.r = 0;
		}
	}
	return;
}

static void veth_rcvm_tick(struct veth_port *p)
{
	struct rcv_mach *m = &(p->rcvm);

	int go_init;
	int go_port_dis;

	go_init = p->pv.begin;
	go_port_dis = !p->pv.begin &&
		!p->pv.port_enabled &&
		!p->pv.port_moved;

	if (m->state == INITIALIZE){
//		printk("veth_rcvm_tick(): ===INITIALIZE\n");

#ifdef Taihua
		if (m->do_it && go_init){
			veth_rcvm_init(p);
			m->state = PORT_DISABLED;
		}
#endif  //Taihua
		if (m->do_it){
			veth_rcvm_init(p);
			m->do_it = _FALSE;
		}
		if (!go_init){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		}
		veth_drop_lacpdu_buf(p);
	} else if (m->state == PORT_DISABLED){
//		printk("veth_rcvm_tick(): ===PORT_DISABLED Begin\n");
		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else {
			if (m->do_it){
				veth_rcvm_port_dis(p);
				m->do_it = _FALSE;
			}

			if (!go_port_dis){
				if (p->pv.port_enabled &&
					p->pv.lacp_ena){
					m->do_it = _TRUE;
					m->state = EXPIRED;
				} else if (p->pv.port_moved){
					m->do_it = _TRUE;
					m->state = INITIALIZE;
				} else if (p->pv.port_enabled &&
						   !p->pv.lacp_ena){
					m->do_it = _TRUE;
					m->state = LACP_DISABLED;
				}
			}
			veth_check_if_port_moved(p);
			veth_drop_lacpdu_buf(p);
		}
	} else if (m->state == EXPIRED){
//		printk("veth_rcvm_tick(): ===EXPIRED Begin\n");

		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else if (go_port_dis){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		} else {
			if (m->do_it){
				veth_rcvm_exp(p);
				m->do_it = _FALSE;

				if (VETH_DEBUG(VETH_DEBUG_RCVM, 1)){
					printk(KERN_INFO "veth_rcvm_tick(): vt started\n");
				}

			}
			if (!is_vt_active(&p->current_while_timer)){
				/* stay this state until current_while_timer expires */

				if (VETH_DEBUG(VETH_DEBUG_RCVM, 1)){
					printk(KERN_INFO "veth_rcvm_tick(): vt expired\n");
				}

				m->state = DEFAULTED;
				m->do_it = _TRUE;
			}
			if (p->rcvm.ring.r != p->rcvm.ring.w){
				/* LACPDU arrives */
				m->state = CURRENT;
				m->do_it = _TRUE;
			}

		}
		/* stay this state until LACPDU received */
	} else if (m->state == LACP_DISABLED){
//		printk("veth_rcvm_tick(): ===LACP_DISABLED Begin\n");

		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else if (go_port_dis){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		} else {
			if (m->do_it){
				veth_rcvm_lacp_dis(p);
				m->do_it = _FALSE;
			}
			veth_drop_lacpdu_buf(p);
		}
	} else if (m->state == DEFAULTED){
//		printk("veth_rcvm_tick(): ===DEFAULT Begin\n");

		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else if (go_port_dis){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		} else {
			if (m->do_it){
				veth_rcvm_def(p);
				m->do_it = _FALSE;
			}
			
			if (p->rcvm.ring.r != p->rcvm.ring.w){
				/* LACPDU arrives */
				m->state = CURRENT;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == CURRENT){
//		printk("veth_rcvm_tick(): ===CURRENT Begin\n");

		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else if (go_port_dis){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		} else {

			if (m->do_it){
				struct lacpdu *buf;

				for (; p->rcvm.ring.r != p->rcvm.ring.w;){
					buf = (struct lacpdu *)p->rcvm.ring.buf[p->rcvm.ring.r];
					
					/* update_selected() should be done before
					 * record_pdu(). Because update_selected()
					 * compares actor's "partner information" with
					 * PDU's "actor information". If record_pdu() is
					 * done before update_selected(), above comparison
					 * cannot detect PDU's (partner's) information
					 * does not match one which actor holds. */
					veth_update_selected(buf, p);
					veth_update_ntt(buf, p);
					veth_record_pdu(buf, p);
					
					kfree(p->rcvm.ring.buf[p->rcvm.ring.r]);
					p->rcvm.ring.buf[p->rcvm.ring.r] = NULL;
					if (p->rcvm.ring.r == MAX_RING-1){
						p->rcvm.ring.r = 0;
					} else {
						p->rcvm.ring.r++;
					}
				}
				
				veth_rcvm_cur(p);
				m->do_it = _FALSE;
			}

			if (p->rcvm.ring.r != p->rcvm.ring.w){
				/* LACPDU arrives */
				m->do_it = _TRUE;
			}

			if (!is_vt_active(&p->current_while_timer)){
				m->state = EXPIRED;
				m->do_it = _TRUE;
			}
		}
	}

	return;
}

/*--------------- end of receive machine -----------*/

/*------ xmit machine ----------*/

static void xmit_lacpdu(struct veth_port *p)
{
	struct sk_buff *skb;
	struct ethhdr *eh;
	struct net_device *dev;
	unsigned char state;


	int pdu_len = LACPDU_LEN;
	int len = LACPDU_LEN + ETH_HLEN;

	memset(&lacpBuf, 0x0, pdu_len);

	/* build LACPDU */
	lacpBuf.subtype = LACP_Subtype;
	lacpBuf.version_number = LACP_Version;

	lacpBuf.first_tlv_type = 0x1; /* Actor Information */
	lacpBuf.actor_info_len = 0x14;
	lacpBuf.actor_sys_pri = htons(veth_sysvar.actor_sys_pri);
	memcpy(lacpBuf.actor_sys, veth_sysvar.actor_sys, ETH_ALEN);
	lacpBuf.actor_key = htons(p->pv.actor_oper_port_key);
	lacpBuf.actor_port_pri = htons(p->pv.actor_port_pri);
	lacpBuf.actor_port = htons(p->pv.actor_port_num);
	lacpBuf.actor_state = p->pv.actor_oper_port_state;

	lacpBuf.second_tlv_type = 0x2; /* Partner Information */
	lacpBuf.partner_info_len = 0x14;
	lacpBuf.partner_sys_pri = htons(p->pv.partner_oper_sys_pri);
	memcpy(lacpBuf.partner_sys, p->pv.partner_oper_sys, ETH_ALEN);
	lacpBuf.partner_key = htons(p->pv.partner_oper_key);
	lacpBuf.partner_port_pri = htons(p->pv.partner_oper_port_pri);
	lacpBuf.partner_port = htons(p->pv.partner_oper_port_num);
	lacpBuf.partner_state = p->pv.partner_oper_port_state;

	lacpBuf.third_tlv_type = 0x3; /* Collector Information */
	lacpBuf.collector_info_len = 0x10;
	lacpBuf.collector_max_del = htons(veth_sysvar.collector_max_del);

	lacpBuf.fourth_tlv_type = 0x0; /* Terminator */
	lacpBuf.terminator_len = 0x0;

	skb = dev_alloc_skb(len);
	if (!skb){
		return;
	}

	skb->mac.raw = skb_put(skb, len);
	
	dev = dev_get_by_name("eth0");
	memcpy(bAddr, dev->dev_addr, ETH_ALEN);
	bAddr[ETH_ALEN -1] += p->pv.actor_port_num;

	memcpy(skb->mac.raw, Slow_Protocols_Multicast, ETH_ALEN);
	memcpy(skb->mac.raw+ETH_ALEN, bAddr, ETH_ALEN);
	eh = (struct ethhdr *)skb->mac.raw;
	eh->h_proto = Slow_Protocols_Type;

	skb->nh.raw = skb->mac.raw + ETH_HLEN;
	memcpy(skb->nh.raw, &lacpBuf, pdu_len);

	skb->dev=dev;
 	CCL_P(skb)->wSize = len;
  	CCL_P(skb)->wVid = 0;
  	CCL_P(skb)->byPriority = 0;
  	put_unaligned(0x1L << (p->pv.actor_port_num-1), (unsigned long *) &(CCL_P(skb)->dwDstPortMsk));
  	put_unaligned(0, (unsigned long *) &(CCL_P(skb)->dwTagPortMsk));

	if (VETH_DEBUG(VETH_CONFOR, 1)){
  	
  		printk("xmit_lacpdu():pNo=%d RC=%s",p->pv.actor_port_num, (char *)state2str_rcv(p->rcvm.state));
  		state = p->pv.actor_oper_port_state;
		printk(" a_s=%c%c%c%c%c%c%c%c",
				   state&1?'A':'_',
				   state&2?'T':'_',
				   state&4?'G':'_',
				   state&8?'S':'_',
				   state&16?'C':'_',
				   state&32?'D':'_',
				   state&64?'F':'_',
				   state&128?'E':'_'
				   );
  		printk(" PT=%s", state2str_pr((enum perio_state)p->prm.state));
  		printk(" MX=%s", state2str_mux(p->muxm.state));
  		do_gettimeofday( &tv);
		printk( " %ld.%ld\n",tv.tv_sec,tv.tv_usec);

  		state = p->pv.partner_oper_port_state;
		printk(" p_s=%c%c%c%c%c%c%c%c Begin=%d, SELECT=%d\n",
				   state&1?'A':'_',
				   state&2?'T':'_',
				   state&4?'G':'_',
				   state&8?'S':'_',
				   state&16?'C':'_',
				   state&32?'D':'_',
				   state&64?'F':'_',
				   state&128?'E':'_',
				   p->pv.begin,
				   p->pv.selected
				   );
	}
//	do_gettimeofday( &tv);
//	printk( "xmit_lacpdu():%ld.%ld,port:%d, packet send!!\n",
//	       	tv.tv_sec,tv.tv_usec, p->pv.actor_port_num);
	dev_queue_xmit(skb);

	return;
}


static void veth_xmit_tick(struct veth_port *p)
{
	/* pv.xmit_ena is controlled by periodic machine */
	/* If lacp_ena is _FALSE, do not xmit LACPDU (43.4.16)*/
//	printk("veth_xmit_tick()1- p->pv.xmit_ena=%d, p->pv.lacp_ena=%d\n",p->pv.xmit_ena,p->pv.lacp_ena);

	if (!p->pv.xmit_ena || !p->pv.lacp_ena){
		p->pv.need_to_xmit = _FALSE;
		p->xmit_timer.tim = 0;
		return;
	}

	/* pv.xmit_update is controlled by periodic machine */
	/* 43.4.16 */
	if (p->pv.xmit_tim_update){
		p->xmit_timer.exp = p->pv.xmit_tim_update;
		p->pv.xmit_tim_update = 0;
		p->xmit_timer.tim = 0;
	}

	/* pv.xmit_count indicates LACPDU xmit count in a timer lifetime */
	if (p->xmit_timer.tim == p->xmit_timer.exp){
		p->pv.xmit_count = 0;
		p->xmit_timer.tim = 0;
	}


	if (p->pv.need_to_xmit && p->pv.xmit_count < 3){
		p->pv.xmit_count++;
		xmit_lacpdu(p);
	}
	p->pv.need_to_xmit = _FALSE;

	p->xmit_timer.tim++;

	return;
}
/*------ end of xmit machine ---*/


/*--------------- mux machine -----------*/

/* DETACHED state */
static void veth_muxm_detached(struct veth_port *p)
{
	veth_detach_mux_from_agg(p);
	p->pv.actor_oper_port_state &= ~PS_SYNCHRONIZATION;
	if (VETH_DEBUG(VETH_CONFOR, 1)){
		printk("veth_muxm_detached():portNo =%d\n",p->pv.actor_port_num);
	}
	veth_dis_distrib(p);
	veth_dis_collect(p);
//	K_TrkSetPortStatus((unsigned char)p->pv.actor_port_num -1, Disable);
	p->pv.need_to_xmit = _TRUE;

	/*
	 * Tell selection logic this port is waiting for selection
	 * if this port is under STANDBY state, do nothing because
	 * selection is already done.
	 */
//		printk("veth_muxm_detached():port_%d ready_n=%d\n",p->pv.actor_port_num,p->pv.ready_n);
	if (p->pv.selected == UNSELECTED){
		p->pv.ready_n = _TRUE;
		
	}
	return;
}

/* WAITING state */
static void veth_muxm_waiting(struct veth_port *p)
{
	vt_start(&(p->wait_while_timer), Aggregate_Wait_Time);
	return;
}

/* ATTACHED state */
static void veth_muxm_attached(struct veth_port *p)
{
	if (VETH_DEBUG(VETH_CONFOR, 1)){
		printk("veth_muxm_attached():portNo =%d\n",p->pv.actor_port_num);
	}
	veth_attach_mux_to_agg(p);
	p->pv.actor_oper_port_state |= PS_SYNCHRONIZATION;
	p->pv.actor_oper_port_state &= ~PS_COLLECTING;
	veth_dis_collect(p);
	p->pv.need_to_xmit = _TRUE;
//	p->pv.ready_n = _FALSE; /* because this port attached to the aggregator */
	return;
}

/* COLLECTING state */
static void veth_muxm_collect(struct veth_port *p)
{
//	printk("veth_muxm_collect():portNo =%d\n",p->pv.actor_port_num);
	veth_ena_collect(p);
	veth_dis_distrib(p);
	p->pv.need_to_xmit = _TRUE;
	return;
}

/* DISTRIBUTING state */
static void veth_muxm_distrib(struct veth_port *p)
{
//	printk("veth_muxm_distrib()=portNo=%d\n", p->pv.actor_port_num);
	veth_ena_distrib(p);
}


static void veth_muxm_tick(struct veth_port *p)
{
	struct mux_mach *m = &(p->muxm);
	struct veth_agg *ag;
	int go_detached;


	go_detached = p->pv.begin;

//        printk("veth_muxm_tick():port_%d ready_n=%d\n",p->pv.actor_port_num,p->pv.ready_n);
			  
	if (m->state == DETACHED){
		if (m->do_it){
			if (VETH_DEBUG(VETH_CONFOR, 1)){
				printk("veth_muxm_tick()===DETACHED ");
				do_gettimeofday( &tv);
				printk( " %ld.%ld\n",tv.tv_sec,tv.tv_usec);
			}

			veth_muxm_detached(p);
			m->do_it = _FALSE;
		}
		if (!go_detached){
			if (p->pv.selected == SELECTED ||
				p->pv.selected == STANDBY){
				m->state = WAITING;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == WAITING){
		if (go_detached){
			m->state = DETACHED;
			m->do_it = _TRUE;
		} else {
			if (m->do_it){
				if (VETH_DEBUG(VETH_CONFOR, 1)){
					printk("veth_muxm_tick()===WAITING ");
					do_gettimeofday( &tv);
					printk( " %ld.%ld\n",tv.tv_sec,tv.tv_usec);
				}
				veth_muxm_waiting(p);
				m->do_it = _FALSE;
			}
			
			if (!is_vt_active(&p->wait_while_timer)){
				p->pv.ready_n = _TRUE;
				ag = veth_aggs + p->pv.actor_port_agg_id-1;
//			        printk("WAITING:p=%d READY=%d\n",p->pv.actor_port_num,ag->av.ready);
				if (p->pv.selected == SELECTED &&
					ag->av.ready == _TRUE){
					m->state = ATTACHED;
					m->do_it = _TRUE;
				} else if (p->pv.selected == UNSELECTED){
					m->state = DETACHED;
					m->do_it = _TRUE;
				}
			}
		}
		/* stay this state while wait_while_timer does not expire */
	} else if (m->state == ATTACHED){
		
		if (go_detached){
			m->state = DETACHED;
			m->do_it = _TRUE;
		} else {
			if (m->do_it){
				if (VETH_DEBUG(VETH_CONFOR, 1)){
					printk("veth_muxm_tick()===ATTACHED ");
					do_gettimeofday( &tv);
					printk( " %ld.%ld\n",tv.tv_sec,tv.tv_usec);
				}
				veth_muxm_attached(p);
				ag = veth_aggs + p->pv.actor_port_agg_id-1;
				m->do_it = _FALSE;
			}
			if (p->pv.selected == SELECTED &&
				p->pv.partner_oper_port_state & PS_SYNCHRONIZATION){
				m->state = COLLECTING;
				m->do_it = _TRUE;
			} else if (p->pv.selected == UNSELECTED ||
					   p->pv.selected == STANDBY){
				m->state = DETACHED;
				
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == COLLECTING){
		if (go_detached){
			m->state = DETACHED;
			m->do_it = _TRUE;
		} else {
			if (m->do_it){
				veth_muxm_collect(p);
				m->do_it = _FALSE;
			}
			if (p->pv.selected == UNSELECTED ||
				p->pv.selected == STANDBY ||
				!(p->pv.partner_oper_port_state & PS_SYNCHRONIZATION)){
				m->state = ATTACHED;
				m->do_it = _TRUE;
			} else if (p->pv.selected == SELECTED &&
					   p->pv.partner_oper_port_state & PS_SYNCHRONIZATION &&
					   p->pv.partner_oper_port_state & PS_COLLECTING ){
				m->state = DISTRIBUTING;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == DISTRIBUTING){
		if (go_detached){
			m->state = DETACHED;
			m->do_it = _TRUE;
		} else {
			if (m->do_it){
				veth_muxm_distrib(p);
				m->do_it = _FALSE;
			}
			if (p->pv.selected == UNSELECTED ||
				p->pv.selected == STANDBY ||
				!(p->pv.partner_oper_port_state & PS_SYNCHRONIZATION) ||
				!(p->pv.partner_oper_port_state & PS_COLLECTING)){
				m->state = COLLECTING;
				m->do_it = _TRUE;
			}		
		}
	}
	return;
}


/*--------------- end of mux machine -----------*/

/*--------------- actor churn machine -----------*/

static void veth_acm_no_actor_churn(struct veth_port *p)
{
		p->pv.actor_churn = _FALSE;
		return;
}

static void veth_acm_actor_churn_mon(struct veth_port *p)
{
		p->pv.actor_churn = _FALSE;
		vt_start(&(p->actor_churn_timer), Churn_Detection_Time);
		return;
}

static void veth_acm_actor_churn(struct veth_port *p)
{
		p->pv.actor_churn = _TRUE;
		return;
}

static void veth_acm_tick(struct veth_port *p)
{
	struct ac_mach *m = &(p->acm);
	int go_churn_mon;

	go_churn_mon = p->pv.begin ||
		!p->pv.port_enabled;

	if (m->state == NO_ACTOR_CHURN){
		if (m->do_it){
			veth_acm_no_actor_churn(p);
			m->do_it = _FALSE;
		}
		if (go_churn_mon){
			m->state = ACTOR_CHURN_MONITOR;
			m->do_it = _TRUE;
		} else {
			if (!p->pv.actor_oper_port_state & PS_SYNCHRONIZATION){
				m->state = ACTOR_CHURN_MONITOR;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == ACTOR_CHURN_MONITOR){
		if (m->do_it){
			veth_acm_actor_churn_mon(p);
			m->do_it = _FALSE;
		}
		if (!go_churn_mon){
			if (p->pv.actor_oper_port_state & PS_SYNCHRONIZATION){
				m->state = NO_ACTOR_CHURN;
				m->do_it = _TRUE;
			}
			if (!is_vt_active(&p->actor_churn_timer)){
				m->state = ACTOR_CHURN;
				m->do_it = _TRUE;
			}
		}
		/* stay this state while wait_while_timer does not expire */
	} else if (m->state == ACTOR_CHURN){
		if (m->do_it){
			veth_acm_actor_churn(p);
			m->do_it = _FALSE;
		}
		if (go_churn_mon){
			m->state = ACTOR_CHURN_MONITOR;
			m->do_it = _TRUE;
		} else {
			if (p->pv.actor_oper_port_state & PS_SYNCHRONIZATION){
				m->state = NO_ACTOR_CHURN;
				m->do_it = _TRUE;
			}
		}
	}
	return;
}
/*--------------- end of actor churn machine -----------*/

/*--------------- partner churn machine -----------*/

static void veth_pcm_no_partner_churn(struct veth_port *p)
{
		p->pv.partner_churn = _FALSE;
		return;
}

static void veth_pcm_partner_churn_mon(struct veth_port *p)
{
		p->pv.partner_churn = _FALSE;
		vt_start(&(p->partner_churn_timer), Churn_Detection_Time);
		return;
}

static void veth_pcm_partner_churn(struct veth_port *p)
{
		p->pv.partner_churn = _TRUE;
		return;
}

static void veth_pcm_tick(struct veth_port *p)
{
	struct pc_mach *m = &(p->pcm);
	int go_churn_mon;

	go_churn_mon = p->pv.begin ||
		!p->pv.port_enabled;

	if (m->state == NO_PARTNER_CHURN){
		if (m->do_it){
			veth_pcm_no_partner_churn(p);
			m->do_it = _FALSE;
		}
		if (go_churn_mon){
			m->state = PARTNER_CHURN_MONITOR;
			m->do_it = _TRUE;
		} else {
			if (!p->pv.partner_oper_port_state & PS_SYNCHRONIZATION){
				m->state = PARTNER_CHURN_MONITOR;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == PARTNER_CHURN_MONITOR){
		if (m->do_it){
			veth_pcm_partner_churn_mon(p);
			m->do_it = _FALSE;
		}
		if (!go_churn_mon){
			if (p->pv.partner_oper_port_state & PS_SYNCHRONIZATION){
				m->state = NO_PARTNER_CHURN;
				m->do_it = _TRUE;
			}
			if (!is_vt_active(&p->partner_churn_timer)){
				m->state = PARTNER_CHURN;
				m->do_it = _TRUE;
			}
		}
		/* stay this state while wait_while_timer does not expire */
	} else if (m->state == PARTNER_CHURN){
		if (m->do_it){
			veth_pcm_partner_churn(p);
			m->do_it = _FALSE;
		}
		if (go_churn_mon){
			m->state = PARTNER_CHURN_MONITOR;
			m->do_it = _TRUE;
		} else {
			if (p->pv.partner_oper_port_state & PS_SYNCHRONIZATION){
				m->state = NO_PARTNER_CHURN;
				m->do_it = _TRUE;
			}
		}
	}
	return;
}

/*--------------- end of partner churn machine -----------*/

/*--------------- port deletion machine -----------*/
/* This machine is not defined by the standard */

static void veth_port_del(struct veth_port *p)
{
	int portNo;
	/* unregister timers */
	veth_unregister_timer(&(p->current_while_timer));
	veth_unregister_timer(&(p->actor_churn_timer));
	veth_unregister_timer(&(p->periodic_timer));
	veth_unregister_timer(&(p->partner_churn_timer));
	veth_unregister_timer(&(p->wait_while_timer));
	veth_unregister_timer(&(p->xmit_timer));

	for (portNo = 0;  portNo < MAX_PHYIF; portNo ++){
		if(veth_aggs[p->pv.actor_port_agg_id-1].av.lag_ports[portNo] 
		   == p->pv.actor_port_num){
			veth_aggs[p->pv.actor_port_agg_id-1].av.lag_ports[portNo] =0;
		}
	}
//	printk("veth_port_del()== port=%d it's info has been clean\n",p->pv.actor_port_num); 

	/* make the physical i/f's empty */
	memset( p, 0, sizeof(struct veth_port));

	return;
}


static void veth_port_del_tick(struct veth_port *p)
{
	if (!p->pv.tbd_cnt){
		return;
	}

	p->pv.tbd_cnt--;

	if (p->pv.tbd_cnt < 3){
		p->pv.begin = _TRUE; /* reset this port */

		vt_stop(&(p->current_while_timer));
		vt_stop(&(p->actor_churn_timer));
		vt_stop(&(p->periodic_timer));
		vt_stop(&(p->partner_churn_timer));
		vt_stop(&(p->wait_while_timer));
		vt_stop(&(p->xmit_timer));
		
		if (p->pv.tbd_cnt){
			return;
		} else {
			veth_port_del(p);
			need_to_rebuild_opl = _TRUE;
		}
	}
}
/*--------------- end of port deletion machine -----------*/


/*--------- port status monitor -----------*/
/* Purpose of this block is periodic calculation of operational key
 * values and monitoring link status */

static void veth_port_mon(struct veth_port *p)
{
//	unsigned long ls;
	unsigned short	key;
	Tbool		bLinkState = False;
	Tuint8		ucSpeed;
	Tuint8		ucDuplex;

	K_PortGetLink(p->pv.actor_port_num-1, &bLinkState);
	if(bLinkState)
		p->pv.port_enabled = _TRUE;
	else
		p->pv.port_enabled = _FALSE;


#ifdef Taihua
	if (bLinkState) {
		K_PortGetEbl(p->pv.actor_port_num-1, (BOOL *)&(p->pv.port_enabled));
	}
#endif
	
//	printk("veth_port_mon====port_%d, enabled=%d\n",p->pv.actor_port_num, p->pv.port_enabled);

	K_PortGetSpeedDuplex(p->pv.actor_port_num-1, &ucSpeed, &ucDuplex);

	if (!p->pv.loop && (lacpStart == LACP_ALIVE)){
		/* If the link is half duplex mode, we cannot enable LACP on
		 * the link... 43.1.2 (p) */
		if (ucDuplex == PORT_DUPLEX_HALF) {
                    	/* 43.6.3 */
			p->pv.lacp_ena = _FALSE;
                }else{
                	p->pv.lacp_ena = _TRUE;
        	}
        }else{
        	p->pv.lacp_ena = _FALSE;
        }

	/* In this implementation, Lower 8-bit of key means media speed. If
     * lower 8-bit of an aggregator's key is zero, it means the
     * aggregator accepts any media speed. See selection logic
     * comments...  */

	if ((ucSpeed == PORT_SPEED_1000) && (ucDuplex == PORT_DUPLEX_FULL)) {
               	key = SUBKEY_1000M;
	}
	else if ((ucSpeed == PORT_SPEED_100) && (ucDuplex == PORT_DUPLEX_FULL)) {
                key = SUBKEY_100M;
	}
	else if ((ucSpeed == PORT_SPEED_10) && (ucDuplex == PORT_DUPLEX_FULL)) {
       		key = SUBKEY_10M;
	}
	else {
		key = SUBKEY_UNKNOWN;
	}

//	printk("veth_port_mon===portSpeed=%d, portDuplex=%d,key=%d\n", ucSpeed, ucDuplex,key);

	/* agg_wc comes from module parameter */
//	printk( "veth_port_mon: === agg_wc=%d\n", agg_wc);

	if (agg_wc == 1){
		key = SUBKEY_UNKNOWN;
	}
		
	if (key == SUBKEY_UNKNOWN && agg_wc){
		p->pv.actor_oper_port_key &= SUBKEY_MASK_UPPER;
		return;
	}

	/* if port's media speed is known and key has changed, then
     * unselects aggregator */
	if ((p->pv.actor_oper_port_key & SUBKEY_MASK_LOWER) != key){
		p->pv.actor_oper_port_key &= SUBKEY_MASK_UPPER;
		p->pv.actor_oper_port_key |= key;
		p->pv.selected = UNSELECTED;
	}
//	printk( "veth_port_mon: === portId=%d, key=%x\n",p->pv.actor_port_num, p->pv.actor_oper_port_key);

	return;
}

static void veth_port_mon_tick(struct veth_port *p)
{
	if (p->pv.pm_intval > VETH_PORT_MON_INTERVAL){
		p->pv.pm_intval = VETH_PORT_MON_INTERVAL;
	}

	if (!p->pv.pm_intval){
		p->pv.pm_intval = VETH_PORT_MON_INTERVAL;
		veth_port_mon(p);
	} else {
		p->pv.pm_intval--;
	}
	return;
}

/*--------- end of port status monitor ----*/

static void veth_tick(unsigned long data)
{
	struct veth_timer *vt;
	struct veth_port *p;
	int i;

	/* periodic machine and other timers */
	for (i = 0, vt = ltimer; vt;i++, vt = vt->next){
		if (vt->tick_func){
			vt->tick_func(vt);
		}
	}

	if (need_to_rebuild_opl){
//		printk( "veth_tick: =====Begin sort_ports_by_pri\n");
		sort_ports_by_pri();
	}

	for (i = 0; opl[i] && i < MAX_PHYIF; i++){

		p = opl[i];
//		printk( "veth_tick: =====Before veth_port_mon_tick\n");
		/* port status monitor */
		veth_port_mon_tick(p);

		/* receive machine */
		veth_rcvm_tick(p);

		/* mux machine */
		veth_muxm_tick(p);

		/* actor churn machine */
		veth_acm_tick(p);

		/* partner churn machine */
		veth_pcm_tick(p);

		/* periodic xmit machine */
		veth_prdTx_tick(p);

		/* xmit machine */
		veth_xmit_tick(p);

		/* selection logic */
		veth_scan_ready_n(p);
//		printk( "veth_tick: =====After veth_scan_ready_n(p)\n\n");
	}

	for (p = veth_ports, i = 0; i < MAX_PHYIF; p++, i++){
		if (!veth_ports[i].pv.interceptedCCL){
			continue;
		}
		/* port deletion machine */
		veth_port_del_tick(p);
	}

	tl.expires = jiffies + HZ/10; /* 100 msec */
	add_timer(&tl);
	
	return;
}



/*----------- aggregator parser (43.2.8.1) --------------------*/

		
static void veth_aggregator_parser(struct sk_buff *skb, int b_slowp, struct net_device *dev)
{
	unsigned char *buf;
	int b_subtype;
	int agg, portNo;
	struct veth_port *p;


	buf = skb->mac.raw + ETH_HLEN;
	b_subtype = (buf[0] == Marker_Subtype);

	if (b_slowp && b_subtype && buf[2] == Marker_Information){
		veth_marker_responder(skb, dev);
	} else if (b_slowp && b_subtype && buf[2] == Marker_Response_Information){
		veth_marker_receiver(skb, dev);
	} else if (b_slowp && (buf[0] < 1 || 10 < buf[0])){
		goto freeandout;
	} else {
		portNo = CCL_P(skb)->bySrcPortId  +1;
		if (!portNo)
			goto freeandout;
		p = veth_ports + portNo -1;

		/* calculate agg num */
		agg = p->pv.actor_port_agg_id - 1;
		if (agg < 0)
			/* proper aggregator not assigned */
			goto freeandout;

		/* If the port state is not COLLECTING then discards this skb */
		if (!p->pv.actor_oper_port_state & PS_COLLECTING)
			goto freeandout;

//		veth_frame_collector(skb, veth_aggs[agg].dev);
	}
	return;

freeandout:
#ifdef __LINUX_2_6_19__
#else
	skb_unlink(skb);
#endif
	kfree_skb(skb);
	return;
}

static void veth_lacp(struct sk_buff *skb, struct net_device *dev)
{

	struct veth_port *p;
	unsigned char *buf;
	int portNo = 0;

	buf = skb->mac.raw + ETH_HLEN;
	
	portNo = CCL_P(skb)->bySrcPortId + 1;
	if (!portNo)
		goto freeandout;

	p = veth_ports + portNo -1;

	if (p->pv.tbd_cnt)
		goto freeandout;
//	do_gettimeofday( &tv);
//	printk( "veth_lacp():%ld.%ld,port:%d,ring.w=%d,ring.r=%d\n",
//		tv.tv_sec,tv.tv_usec,portNo, p->rcvm.ring.w,p->rcvm.ring.r);

	p->rcvm.ring.buf[p->rcvm.ring.w] = kmalloc(LACPDU_LEN+ETH_HLEN+5, GFP_ATOMIC);
	memcpy(p->rcvm.ring.buf[p->rcvm.ring.w++], buf, LACPDU_LEN+ETH_HLEN);
	if (p->rcvm.ring.w == MAX_RING){
		p->rcvm.ring.w = 0;
		if (p->rcvm.ring.r == p->rcvm.ring.w){
			/* Overwriting oldest entry.. This should not occur! */
			p->rcvm.ring.r++;
		}
	}

freeandout:
#ifdef __LINUX_2_6_19__
#else
	skb_unlink(skb);
#endif
	kfree_skb(skb);
	return;
}


/* control parser (43.2.9.1.4) */

static int veth_control_parser(struct sk_buff *skb)
{
	unsigned char *buf;
	struct sk_buff *nskb;
	int b_mac_addr;
	int b_mac_type;
	int portNo;

	if (!skb)
		return 1;


	nskb = skb;
	skb = skb_copy(nskb, GFP_ATOMIC);
#ifdef __LINUX_2_6_19__
#else
	skb_unlink(nskb);
#endif
	kfree_skb(nskb);

	if (!skb)
		return 0;

	portNo = CCL_P(skb)->bySrcPortId + 1;
//	printk("veth_control_parser()== portNo=%d\n", portNo);
	buf = skb->mac.raw + ETH_HLEN;


	if (VETH_DEBUG(VETH_DEBUG_CTLP, 1)){
#ifdef __LINUX_2_6_19__
		printk(KERN_INFO "h_dest:%02x%02x%02x-%02x%02x%02x, h_proto:%04x, lacp-sub:%02x\n",
			   eth_hdr(skb)->h_dest[0],
			   eth_hdr(skb)->h_dest[1],
			   eth_hdr(skb)->h_dest[2],
			   eth_hdr(skb)->h_dest[3],
			   eth_hdr(skb)->h_dest[4],
			   eth_hdr(skb)->h_dest[5],
			   eth_hdr(skb)->h_proto,
			   buf[0]);
#else
		printk(KERN_INFO "h_dest:%02x%02x%02x-%02x%02x%02x, h_proto:%04x, lacp-sub:%02x\n",
			   skb->mac.ethernet->h_dest[0],
			   skb->mac.ethernet->h_dest[1],
			   skb->mac.ethernet->h_dest[2],
			   skb->mac.ethernet->h_dest[3],
			   skb->mac.ethernet->h_dest[4],
			   skb->mac.ethernet->h_dest[5],
			   skb->mac.ethernet->h_proto,
			   buf[0]);
#endif
	}

#ifdef __LINUX_2_6_19__
	b_mac_addr = (memcmp(eth_hdr(skb)->h_dest, Slow_Protocols_Multicast, ETH_ALEN) == 0)?_TRUE:_FALSE;
	b_mac_type = (eth_hdr(skb)->h_proto == Slow_Protocols_Type);
#else
	b_mac_addr = (memcmp(skb->mac.ethernet->h_dest, Slow_Protocols_Multicast, ETH_ALEN) == 0)?_TRUE:_FALSE;
	b_mac_type = (skb->mac.ethernet->h_proto == Slow_Protocols_Type);
#endif

	if (b_mac_addr && b_mac_type)
		if (buf[0] == LACP_Subtype)
			veth_lacp(skb, skb->dev);
		else
			veth_aggregator_parser(skb, _TRUE, skb->dev);
	else
		veth_aggregator_parser(skb, _FALSE, skb->dev);

	return CCL_MX_ACCEPT;

}

/***************************************************************
 *              End of LACP
 ***************************************************************/

#if 0 // #if LINUX_VERSION_CODE != KERNEL_VERSION(2,4,20)
static struct packet_type veth_packet_type __initdata = {
	__constant_htons(ETH_P_VETH),
	NULL,           /* Listen to all devices */
	veth_control_parser,
	NULL,
	NULL
};

static inline void ic_veth_init(void)
{
	dev_add_pack(&veth_packet_type);
}

static inline void ic_veth_cleanup(void)
{
	dev_remove_pack(&veth_packet_type);
}
#endif


static void veth_set_admin_port_vars(struct veth_port *p, unsigned short key)
{
	p->pv.actor_admin_port_key = p->pv.actor_oper_port_key = key;
	p->pv.actor_admin_port_state = PS_LACP_ACTIVITY | PS_DEFAULTED | PS_AGGREGATION;
	p->pv.actor_oper_port_state |= PS_LACP_ACTIVITY | PS_AGGREGATION;
	p->pv.partner_admin_key = key;
	memset(p->pv.partner_admin_sys, 0, ETH_ALEN);
	p->pv.partner_admin_sys_pri = sysPri;
	p->pv.partner_admin_port_num = p->pv.actor_port_num;
	p->pv.partner_admin_port_pri = 0x8000;
	p->pv.partner_admin_port_state = PS_DEFAULTED | PS_SYNCHRONIZATION | PS_COLLECTING | PS_AGGREGATION | PS_LACP_TIMEOUT;
	return;
}

static void veth_port_vars_init(struct veth_port *p)
{
	/* state machines */
	p->rcvm.state = INITIALIZE;
	p->acm.state = NO_ACTOR_CHURN;
	p->pcm.state = NO_PARTNER_CHURN;
	p->muxm.state = DETACHED;
	p->prm.state = NO_PERIODIC;

	if (lacpStart == LACP_ALIVE){
		p->rcvm.do_it = _TRUE;
		p->acm.do_it = _TRUE;
		p->pcm.do_it = _TRUE;
		p->muxm.do_it = _TRUE;
		p->prm.do_it = _TRUE;

	}else{
		p->rcvm.do_it = _FALSE;
		p->acm.do_it = _FALSE;
		p->pcm.do_it = _FALSE;
		p->muxm.do_it = _FALSE;
		p->prm.do_it = _FALSE;
	}
	
	p->pv.tbd_cnt = 0;
	p->pv.selected = UNSELECTED;
	p->pv.ready_n = _FALSE;

	p->avrlen = 750;
}


/*
 * ioctl() support functions
 */
static int veth_get_group_id_by_name(unsigned short portNo)
{
	int i, j;

	for (i = 0; i < MAX_VETH; i++)
		if (veth_aggs[i].av.agg_id)
			for (j = 0; j < MAX_PHYIF; j++)
				if (veth_aggs[i].av.lag_ports[j] == portNo){
//					printk("veth_get_group_id_by_name==Got AggId=%d\n",i+1);
					return i+1;
				}
//	printk("veth_get_group_id_by_name==Got No AggId!!\n");
	
	return 0; /* fail */
}


static int veth_catch_device(struct veth_port *p, char *modname)
{
//	memset(p, 0, sizeof(struct veth_port));

//        printk("veth_catch_device():received port_%d\n",p->pv.actor_port_num);


	if (p->pv.interceptedCCL) 
		return 0;		/* already used by a virtual device driver */

	p->pv.interceptedCCL = _TRUE;
 
//        printk("veth_catch_device(): check ok End!!\n");


	/* make the interface to promiscuous mode */
//	dev_set_promiscuity(p->dev, 1);
#if 0
	/* Demand loaded and "unused" device driver 
	 * module causes OOPS even though 
	 * set_multicast_list is not NULL!!! */
	if (p->dev->set_multicast_list != NULL){
		p->dev->set_multicast_list(p->dev);
	}
#endif

	return 1;
}

static int veth_release_device(int vifNo, struct veth_port *p)
{
	int portNo;
	p->pv.selected = UNSELECTED;
	p->pv.tbd_cnt = 3; /* delete this phyif 2 ticks later... */

#ifndef LACP_PER_PORT
	if (vifNo)
#endif
	for (portNo = 0;  portNo < MAX_PHYIF; portNo ++){
		if(veth_aggs[vifNo-1].av.lag_ports[portNo] == p->pv.actor_port_num){
			veth_aggs[vifNo-1].av.lag_ports[portNo] =0;
//			printk("veth_release_device==aggNo=%d,it's port=%d has been remove!!\n",
//                                 vifNo,p->pv.actor_port_num);
		}
	}

	K_TrkSetPortStatus((unsigned char)(p->pv.actor_port_num-1), Disable);
	K_TrkSetPortAggrId((unsigned char)(p->pv.actor_port_num -1), 0);
#ifdef LACP_PER_PORT
	cclmx_DelPortFromTrk((unsigned char)(p->pv.actor_port_num-1));
#endif
	K_TrkSetPortAdmCri((unsigned char)(p->pv.actor_port_num -1), 0x0);


#ifdef Taihua
	// Enable Nway mask
	BIOvRegSpecBitOn(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2));
        BIOvRegSpecBitOn(PHY_PORT_NWAY_EN_MASK, (UINT8)(byPortId*2 + 1));
        // Enable MAC I/O
        BIOvRegBitsOn ( (UINT16)(PHY_PORT_ABL_BASE + byPortId),
                            PTN_PHY_PORT_ABL_OUT_ENABLE |PTN_PHY_PORT_ABL_IN_ENABLE);
#endif //Taihua

	return 1;
}

/* add physical i/f to pool */
static int veth_add_phyif(unsigned short portNo, char *modname, unsigned short key)
{
	struct veth_port *p;
	struct veth_agg *ag;
	int i;
	unsigned short keyBuf=0;
//	struct net_device *dev;

	/* We cannot add a veth i/f to veth
	 * We cannot add a phy i/f already registered
	 * search for vacant veth_ports entry */
 	p = veth_ports+portNo-1;

	if (veth_get_group_id_by_name(portNo)  ||
		(p->pv.interceptedCCL))
		return 0;

	if (!veth_catch_device(p, modname))
		return 0;

#ifdef Taihua	
	if (portNo == 1){
		dev = dev_get_by_name("eth0");
		memcpy(veth_sysvar.actor_sys, dev->dev_addr, ETH_ALEN);
		veth_sysvar.actor_sys_pri = sysPri;
	}
#endif

	p->pv.begin = _TRUE;


	/* register timers */

//        printk("veth_add_phyif(): ===== begin register timer module=%s\n",modname);
	veth_clear_timer(&(p->current_while_timer));
	veth_clear_timer(&(p->actor_churn_timer));
	veth_clear_timer(&(p->periodic_timer));
	veth_clear_timer(&(p->partner_churn_timer));
	veth_clear_timer(&(p->wait_while_timer));
	veth_clear_timer(&(p->xmit_timer));
	
	veth_register_timer(&(p->current_while_timer));
	veth_register_timer(&(p->actor_churn_timer));
	veth_register_timer(&(p->periodic_timer));
	veth_register_timer(&(p->partner_churn_timer));
	veth_register_timer(&(p->wait_while_timer));
	veth_register_timer(&(p->xmit_timer));

	veth_port_vars_init(p);
	
	p->pv.actor_port_num = portNo;
	p->pv.actor_port_pri = sysPri;
	need_to_rebuild_opl = _TRUE;
	
	p->pv.lacp_ena = _TRUE;
	
	veth_set_admin_port_vars(p, key);

	K_TrkSetPortStatus(portNo - 1, Enable);
        /* start state machines */
        p->pv.begin = _FALSE;

	keyBuf=key>>8;
	if( lacpStart == LACP_NOALIVE){
		p->pv.lacp_ena = _FALSE;
	        ag = veth_aggs + keyBuf - 1;
		for (i = 0; i < MAX_PHYIF; i++){
			if (!ag->av.lag_ports[i]){
				ag->av.lag_ports[i] = portNo; 
				break;
			}
		}
	}


//        printk("veth_add_phyif()== ADD Port=%d OK, ready_n=%d\n",portNo,p->pv.ready_n);

	return 1; /* true */
}


/* delete physical i/f from pool */
static int veth_del_phyif(unsigned short portNo)
{
	struct veth_port *p;
	int vifNo =0;

	p = veth_ports+portNo-1;

	if ( !(p->pv.interceptedCCL))
		return 0;

	/* We cannot delete a phy i/f not registered */

	vifNo =veth_get_group_id_by_name(portNo);  
//	printk("veth_del_phyif== aggId=%d\n",vifNo);
#ifndef LACP_PER_PORT
	if (!vifNo)return 0;
#endif

	veth_release_device(vifNo, p);
	return 1; /* true */
}

static int veth_get_agg_info(struct u_agg_info *u, unsigned short vifNo)
{
	int i;
	struct veth_agg *ag;
	struct u_agg_info ku;

	for (i = 0, ag = veth_aggs; i < MAX_VETH && ag->av.agg_id; i++, ag++){

		if (ag->av.agg_id == vifNo){
			memcpy(ku.agg_mac, ag->av.agg_mac, ETH_ALEN);
			ku.agg_id = ag->av.agg_id;
			ku.ind_agg = ag->av.ind_agg;
			ku.actor_admin_agg_key = ag->av.actor_admin_agg_key;
			ku.actor_oper_agg_key = ag->av.actor_oper_agg_key;
			memcpy(ku.partner_sys, ag->av.partner_sys, ETH_ALEN);
			ku.partner_sys_pri = ag->av.partner_sys_pri;
			ku.partner_oper_agg_key = ag->av.partner_oper_agg_key;
			ku.ready = ag->av.ready;
			ku.distmode = ag->distmode;

//			printk("veth_get_agg_info ===== vif_name=%s\n",vif_name);
			if (copy_to_user(u, &ku, sizeof(struct u_agg_info)))
				return -EFAULT;

			return sizeof(struct u_agg_info);
		}
	}

	return -ENODEV;
}

static int veth_set_agg_info(struct u_agg_info *u, unsigned short vifNo)
{
	int i;
	struct veth_agg *ag;
	struct u_agg_info ku;

	if (copy_from_user(&ku, u, sizeof(struct u_agg_info)))
		return -EFAULT;

	for (i = 0, ag = veth_aggs; i < MAX_VETH && ag->av.agg_id; i++, ag++){
		if (ag->av.agg_id == vifNo){
			ag->av.actor_admin_agg_key = ku.actor_admin_agg_key;
			ag->distmode = ku.distmode;

			return sizeof(struct u_agg_info);
		}
	}

	return -ENODEV;
}




static int veth_get_port_info(struct u_port_info *u, unsigned short pifId)
{
	int i;
	struct veth_port *p;
	struct u_port_info ku;

	for (i = 0, p = veth_ports; i < MAX_PHYIF; i++, p++){
		if (p->pv.actor_port_num == pifId){
//			printk("veth_get_port_info =catch portNo=%d \n",p->pv.actor_port_num);
			ku.actor_port_num = p->pv.actor_port_num;
			ku.actor_port_pri = p->pv.actor_port_pri;
			ku.actor_port_agg_id = p->pv.actor_port_agg_id;
			ku.actor_admin_port_key = p->pv.actor_admin_port_key;
			ku.actor_oper_port_key = p->pv.actor_oper_port_key;
			ku.actor_admin_port_state = p->pv.actor_admin_port_state;
			ku.actor_oper_port_state = p->pv.actor_oper_port_state;

			memcpy(ku.partner_admin_sys, p->pv.partner_admin_sys, ETH_ALEN);
			memcpy(ku.partner_oper_sys, p->pv.partner_oper_sys, ETH_ALEN);
			ku.partner_admin_sys_pri = p->pv.partner_admin_sys_pri;
			ku.partner_oper_sys_pri = p->pv.partner_oper_sys_pri;
			ku.partner_admin_key = p->pv.partner_admin_key;
			ku.partner_oper_key = p->pv.partner_oper_key;
			ku.partner_admin_port_num = p->pv.partner_admin_port_num;
			ku.partner_oper_port_num = p->pv.partner_oper_port_num;
			ku.partner_admin_port_pri = p->pv.partner_admin_port_pri;
			ku.partner_oper_port_pri = p->pv.partner_oper_port_pri;
			ku.partner_admin_port_state = p->pv.partner_admin_port_state;
			ku.partner_oper_port_state = p->pv.partner_oper_port_state;

			ku.selected = p->pv.selected;

			if (copy_to_user(u, &ku, sizeof(struct u_port_info))){
				printk("veth_get_port_info=Port%d copy_to_user err\n",p->pv.actor_port_num);
				return -EFAULT;
			}
			return sizeof(struct u_port_info); /* success */
		}
	}

	return -ENODEV; /* fail */
}


static int veth_set_port_info(struct u_port_info *u, unsigned short pifId)
{
	int i;
	struct veth_port *p;
	struct u_port_info ku;

	if (copy_from_user(&ku, u, sizeof(struct u_port_info)))
		return -EFAULT;
//	printk("set_port_info()== port=%d ",pifId);

	for (i = 0, p = veth_ports; i < MAX_PHYIF; i++, p++){
		if (p->pv.actor_port_num == pifId){
			p->pv.actor_admin_port_key = ku.actor_admin_port_key;
//			p->pv.selected = UNSELECTED;
			p->pv.actor_admin_port_state = ku.actor_admin_port_state;
			p->pv.actor_oper_port_state = ku.actor_admin_port_state;
			p->pv.actor_oper_port_state = PS_LACP_ACTIVITY;
			
			memcpy(p->pv.partner_admin_sys, ku.partner_admin_sys, ETH_ALEN);
			p->pv.partner_admin_sys_pri = ku.partner_admin_sys_pri;
			p->pv.partner_admin_key = ku.partner_admin_key;
			p->pv.partner_admin_port_num = ku.partner_admin_port_num;
			p->pv.partner_admin_port_pri = ku.partner_admin_port_pri;
			p->pv.partner_admin_port_state = ku.partner_admin_port_state;

			return sizeof(struct u_port_info); /* success */
		}
	}

	return 0; /* fail */
}


static int veth_get_version(struct u_drv_info *u)
{
	struct u_drv_info ku;

	ku.major = VETH_VER_MAJOR;
	ku.minor = VETH_VER_MINOR;
	ku.patch = VETH_VER_PATCH;
	strcpy(ku.name, "VETH");
	strcpy(ku.author, "CCL/N300 2003/10");

	if (copy_to_user(u, &ku, sizeof(struct u_drv_info)))
		return -EFAULT;

	return sizeof(struct u_drv_info);
}

static int lacp_stop(void)
{
	int i,j;
	unsigned char portNo;
	struct veth_port *p;


		/* make the physical i/f's non-promiscuous mode */

	for (i = 0; i < MAX_VETH; i++) {
		for( j=0; veth_aggs[i].av.lag_ports[j] && j < MAX_PHYIF;j++){
			portNo = veth_aggs[i].av.lag_ports[j]-1;
			p = veth_ports+portNo;
//			printk("lacp_stop()== agg=%d,Port=%d has been reset!!\n",veth_aggs[i].av.agg_id,p->pv.actor_port_num);
			p->rcvm.state = INITIALIZE;
			p->acm.state = NO_ACTOR_CHURN;
			p->pcm.state = NO_PARTNER_CHURN;
			p->muxm.state = DETACHED;
			p->prm.state = NO_PERIODIC;
			p->rcvm.do_it = _FALSE;
			p->acm.do_it = _FALSE;
			p->pcm.do_it = _FALSE;
			p->muxm.do_it = _FALSE;
			p->prm.do_it = _FALSE;
			p->pv.xmit_ena = _FALSE;
			p->pv.selected = UNSELECTED;
			p->pv.actor_oper_port_state &= 0x00;
			K_TrkSetPortStatus(portNo, Disable);
//			K_TrkSetPortAggrId(portNo, 0);
			K_TrkSetPortAdmCri(portNo, 0x0);
		}

	}
	ccl_mx_unregister(Slow_Protocols_Multicast,mask6B);
	K_LacpSetEbl(Disable);
   
	return 0;
}


static int lacp_dispatch(struct sock *sk, int cmd, void *user, int *len)
{
	int n;
	int ret = 0;
	struct ifreq *rq = (struct ifreq *)user;
	struct vethconf *vc = (struct vethconf *) rq->ifr_data;
	struct vethconf kvc;

	if (copy_from_user(&kvc, vc, sizeof(struct vethconf)) ||
			(kvc.magic != VETH_MAGIC))
		return -EOPNOTSUPP;

//	printk("lacp_dispatch()== Command=%d\n",cmd);


	switch (cmd)
	{
	case VETHCTL_VERSION:
//		printk("lacp_dispatch =====check VERSION \n");
		n = veth_get_version((struct u_drv_info *)vc->buf);
		(n < 0)?  (ret = n) : (vc->size = n);
		break;

	case VETHCTL_GET_SYS_PRI:
		if (copy_to_user((unsigned short *)vc->buf, &sysPri, sizeof(unsigned short))) {
			ret = -EFAULT;
		}
		else {
			ret = 0;
		}
		break;
		
	case VETHCTL_SET_SYS_PRI:
		memcpy(&sysPri, (unsigned short *)vc->buf, sizeof(unsigned short));
		ret = 0;
		break;

	case VETHCTL_ADD_PHYIF:
		{
			struct u_port_info *u = (struct u_port_info *)vc->buf;
//			printk("lacp_dispatch ==ADD portNo=%d, key=%d\n",vc->pifId,(u->actor_admin_port_key >>8));

			if (!veth_add_phyif(vc->pifId, vc->pif_module, u->actor_admin_port_key)) {
				ret = -ENODEV;
			}
		}
		break;

	case VETHCTL_DEL_PHYIF:
//		printk("lacp_dispatch ==DEL portNo=%d\n", vc->pifId);

		if (!veth_del_phyif(vc->pifId)){
			ret = -ENODEV;
		}
		break;

	case VETHCTL_GET_AGG_INFO:
//		printk("lacp_dispatch =====GET_AGG_INFO vif=%s\n",vc->vif_name);
		n = veth_get_agg_info((struct u_agg_info *)vc->buf, vc->vifNo);
		(n < 0)? (ret = n) : (vc->size = n);
		break;

	case VETHCTL_SET_AGG_INFO:
//		printk("lacp_dispatch =====SET_AGG_INFO vif=%d\n",vc->vifNo);
		n = veth_set_agg_info((struct u_agg_info *)vc->buf, vc->vifNo);
		(n < 0)? (ret = n) : (vc->size = n);
		break;

	case VETHCTL_GET_PORT_INFO:
//		printk("lacp_dispatch =====GET_PORT_INFO port=%d\n",vc->pifId);
		n = veth_get_port_info((struct u_port_info *)vc->buf, vc->pifId);
		(n < 0)? (ret = n) : (vc->size = n);
		break;

	case VETHCTL_SET_PORT_INFO:
//		printk("lacp_dispatch ===SET_PORT_INFO agg=%s, buf=%s\n",vc->vif_name,vc->buf);
		n = veth_set_port_info((struct u_port_info *)vc->buf, vc->pifId);
		(n < 0)? (ret = n) : (vc->size = n);
		break;

	case VETHCTL_START:

		if (lacpStart == LACP_ALIVE) {
			//printk("lacp_dispatch: The Lacp has Started\n");
			//ret = -EFAULT;
			ret = 0;
		}
		else {
			
			K_LacpSetEbl(Enable);
			K_TrkSetMacHash(TRK_MAC_HASH_DIRECT_MAP);
			K_TrkSetTcktGenAlg(TRK_TCKT_GEN_ALG_DMAC_SMAC_PORTID);
			/* packet handler registration for receiving */
			ccl_mx_register(Slow_Protocols_Multicast,mask6B, veth_control_parser);

			lacpStart = LACP_ALIVE;

		}
		break;

	case VETHCTL_STOP:
		lacpStart = LACP_NOALIVE;
		ret = lacp_stop();
		break;
	case VETHCTL_ALIVE:
		if (copy_to_user((int *)vc->buf, &lacpStart, sizeof(int))) {
			ret = -EFAULT;
		}
		else {
			ret = 0;
		}

		break;

	default:
		ret = -EOPNOTSUPP;
		break;
	}
//	printk("lacp_dispatch end===== ret=%d\n",ret);
	return ret;
}




static int inline veth_searchfor(int list[], int a)
{
	int i;
	for (i = 0; i < MAX_VETH && list[i] != -1; i++) {
		if (list[i] == a) return 1;
	}
	return 0;
}





static int sprintf_a_line(char *buf, struct veth_port *p)
{
	int len;
	int j;
	unsigned char uc;
	char c;

	j = p->pv.actor_port_agg_id-1;

	if (p->pv.selected == UNSELECTED){
		c = 'U';
	} else if (p->pv.selected == STANDBY){
		c = 'D';
	} else if (p->pv.selected == SELECTED){
		c = 'S';
	} else {
		c = '?';
	}
	len = sprintf(buf,"%2d",p->pv.actor_port_num);
	len += sprintf(buf+len, " %2d", (j >= 0)?veth_aggs[j].av.agg_id:0);
	len += sprintf(buf+len, " %02x%02x%02x%02x%02x%02x",
				p->pv.partner_oper_sys[0],
				p->pv.partner_oper_sys[1],
				p->pv.partner_oper_sys[2],
				p->pv.partner_oper_sys[3],
				p->pv.partner_oper_sys[4],
				p->pv.partner_oper_sys[5]);
	len += sprintf(buf+len, "  %04x", p->pv.partner_oper_key);
	len += sprintf(buf+len, "  %04x", p->pv.actor_oper_port_key);
	len += sprintf(buf+len, "  %c", c);
	len += sprintf(buf+len, "%7s", state2str_mux(p->muxm.state));
	len += sprintf(buf+len, "%7s", state2str_rcv(p->rcvm.state));
	len += sprintf(buf+len, "%7s", state2str_pr(p->prm.state));

	uc = p->pv.actor_oper_port_state;
	len += sprintf(buf+len, " %c%c%c%c%c%c%c%c",
				   uc&1?'A':'_',
				   uc&2?'T':'_',
				   uc&4?'G':'_',
				   uc&8?'S':'_',
				   uc&16?'C':'_',
				   uc&32?'D':'_',
				   uc&64?'F':'_',
				   uc&128?'E':'_'
				   );
	uc = p->pv.partner_oper_port_state;
	len += sprintf(buf+len, " %c%c%c%c%c%c%c%c",
				   uc&1?'A':'_',
				   uc&2?'T':'_',
				   uc&4?'G':'_',
				   uc&8?'S':'_',
				   uc&16?'C':'_',
				   uc&32?'D':'_',
				   uc&64?'F':'_',
				   uc&128?'E':'_'
				   );
	len += sprintf(buf+len, " %d %d\n", veth_aggs[j].av.ready,p->pv.ready_n);

	return len;
}

static int veth_get_info(char *pbuf, char **start, off_t offset, int length)
{
#define LIMIT (PAGE_SIZE-80);

	int size;
	int len = 0;
	int i;

	if(!offset){
		/* first time write the header */
		size = sprintf(pbuf,"LACP=%7s\n",(lacpStart == LACP_ALIVE)? "Enable":"Disable");
		len = size;
		size = sprintf(pbuf+len,"%s","Pt Ag PartnerSysId PnKey AtKey Sel  Mux  Receiv  PrdTx  AtState  PnState R r\n");
		len += size;
	}
	

	for (i = 0; i < MAX_PHYIF; i++){
		if (veth_ports[i].pv.interceptedCCL){
			len += sprintf_a_line(pbuf+len, veth_ports+i);
		}
	}

	if (len > PAGE_SIZE-80){
		len = PAGE_SIZE-80;
	}

	return len;
}



struct nf_sockopt_ops lacp_sockopts = {
                { NULL, NULL }, PF_INET,
                0, 0, NULL, NULL,
                LACP_BASE_CTL, VETHCTL_MAX+1, lacp_dispatch, NULL
};


static void __exit veth_cleanup_module(void)
{
	int ret;
	
	ret = lacp_stop();
        if(ret){
                printk(KERN_INFO "veth:===lacp_stop():error \n");
//                return ret;
        }
        
	del_timer(&tl);
	nf_unregister_sockopt(&lacp_sockopts);
#ifdef CONFIG_PROC_FS
	proc_net_remove("veth");
#endif
#if 0 // #if LINUX_VERSION_CODE != KERNEL_VERSION(2,4,20)
	/* remove packet handler */
	ic_veth_cleanup();
#endif

}

static int veth_init_agg(int agn)
{
	struct veth_agg *ag;
	int i;

	i = agn;
	ag = veth_aggs + i;
	ag->av.actor_admin_agg_key = (i+1) << 8;

	memset(ag->av.agg_mac, 0, ETH_ALEN);
	ag->av.agg_mac[ETH_ALEN-1] = i+1;
//	printk("veth_init_agg =====agg dev=%s,mac=",ag->dev->name);
//        for (j=0; j<8; j++) printk("%x:",*(ag->av.agg_mac+j));
//        printk("\n");

	ag->av.agg_id = i+1;

	ag->count = 0;
	ag->distmode = DIST_TCPUDPHASH;
//	memset(ag->dht, 0, 32*sizeof(struct dist_hash_table *));
	ag->lock = SPIN_LOCK_UNLOCKED;

	return 0;
}


/* 
 * Entry point of VETH driver.
 * Register/initialize the driver.
 */

static int lacp_init(void)
{
	int i;
	int ret;

//	printk("CCL LACP ver%d.%d \n",VETH_VER_MAJOR, VETH_VER_MINOR);

	/* nveth comes from module parameter and it is default to 1 */
	for (i = 0; i < nveth; i++){

		ret = veth_init_agg(i);
		if (ret){
			printk(KERN_INFO "veth: no devices registered\n");
			return ret;
		}

	}
	
        for (i = 0; i < MAX_PHYIF; i++){
		memset( veth_ports +i, 0, sizeof(struct veth_port));
        }


	tl.expires = jiffies + HZ/10; /*     */
	tl.function = veth_tick;

	add_timer(&tl);

	return 0;
	
}

static int __init veth_init_module(void)
{
	int ret;
	struct net_device *dev;

	//----------------- replace register_netdev to nf_register_sockopt---
        ret = nf_register_sockopt(&lacp_sockopts);
        if(ret){
                printk(KERN_INFO "veth:===nf_register_sockopt():cannot register sockopt \n");
                return ret;
        }

       	dev = dev_get_by_name("eth0");
	memcpy(veth_sysvar.actor_sys, dev->dev_addr, ETH_ALEN);
	veth_sysvar.actor_sys_pri = sysPri;

 	ret = lacp_init();
        if(ret){
                printk(KERN_INFO "veth:===lacp_init():error \n");
                return ret;
        }
  
	
#ifdef CONFIG_PROC_FS
	proc_net_create("veth", 0, veth_get_info);
#endif

	/* packet handler registration for receiving */
//	ccl_mx_register(Slow_Protocols_Multicast,mask6B, veth_control_parser);

	return 0;
}




module_init(veth_init_module);
module_exit(veth_cleanup_module);


