/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  THIS IS AN UNPUBLISHED WORK WHICH CONTAINS CONFIDENTIAL INFORMATION
 *  FROM CCL, ITRI.  NO PART OF THIS WORK MAY BE USED IN ANY WAY WITHOUT
 *  THE PRIOR WRITTEN PERMISSION.  ANY UNAUTHORIZED USE COULD SUBJECT THE
 *  PERPETRATOR TO CRIMINAL AND CIVIL LIABILITY.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 *  Author: Winfred Lu (N300, CCL/ITRI)
 *
 ***************************************************************************/

#include <linux/kernel.h>
#include <asm/unaligned.h>

#include "krnerr.h"
#include "krnipmcst.h"
#include "krndef.h"
#include "krnportmask.h"
#include "krnstp.h"		/* need to know spanning-tree blocking ports */

#include "../cclmx/ccl.h"
#include "../cclmx/ccl_mx.h"
#include "db.h"
#include "util.h"
#include "igmp_netfilter.h"


TstLPortMask igmpi_upstream_mask;
TstLPortMask igmpi_disabled_mask;
extern TstIgmpSnpStatus		igmpsnp_global_stats;
extern igmp_router_t		igmp_router;



void debug(const char *fmt, ...)
{
	if (igmpsnp_global_stats.debug_enable) {
		static char mydebug_buf[1024];
		int len;
		va_list args;

		va_start(args, fmt);
		len = vsnprintf(mydebug_buf, 1024, fmt, args);
		va_end(args);

		printk("IGMP_Snooping: %s", mydebug_buf);
	}
	return;
}


void update_portmasks (void)
{
	Tuint8	ucPortId;

	K_LPortMaskClrAll (&igmpi_upstream_mask);
	if (igmp_router.addr != 0 && igmp_router.port < MAX_LOGIC_PORT) {
		K_LPortMaskSetPort (&igmpi_upstream_mask, igmp_router.port);
	}

	for (ucPortId = 0; ucPortId < MAX_LOGIC_PORT; ucPortId++) {
#if 0
		if (CCLMX_PORT_LINKUP == cclmx_PortGetLink (ucPortId) &&
		    CCLMX_PORT_ON == cclmx_PortGetEbl (ucPortId)) {
#endif
		if (STP_PORT_FORWARDING == cclmx_PortGetStpState (ucPortId)) {
			K_LPortMaskClrPort (&igmpi_disabled_mask, ucPortId);
		}
		else {
			K_LPortMaskSetPort (&igmpi_disabled_mask, ucPortId);
		}
	}

	/* it is possible that router port is also disabled */
	K_LPortMaskClrPorts (&igmpi_upstream_mask, &igmpi_disabled_mask);
}


void igmpsnp_xmit_dir (struct sk_buff * skb, int dir)
{
	update_portmasks ();

	switch (dir) {
	case IGMPI_UPSTREAM:		
		K_LPortMaskCopy (&CCL_LP(skb)->dwDstPortMsk, &igmpi_upstream_mask);
		K_LPortMaskClrPorts (&CCL_LP(skb)->dwDstPortMsk, &igmpi_disabled_mask);
		break;
	case IGMPI_DOWNSTREAM:
		K_LPortMaskSetAll (&CCL_LP(skb)->dwDstPortMsk);
		K_LPortMaskClrPorts (&CCL_LP(skb)->dwDstPortMsk, &igmpi_upstream_mask);
		K_LPortMaskClrPorts (&CCL_LP(skb)->dwDstPortMsk, &igmpi_disabled_mask);
		break;
	default:
		return;
	}
	if (K_LPortMaskIsZero (&CCL_LP(skb)->dwDstPortMsk))
		return;

	CCL_LP(skb)->wVid = 1;
	CCL_LP(skb)->byPriority = 0;

	cclmx_TxPkt (skb);
#if 0
	debug("Tx a pkt %s (mask_up:%08x,%08x, disabled:%08x,%08x)\n",
		IGMPI_UPSTREAM == dir? "up" : "down",
		igmpi_upstream_mask.ulMask[1], igmpi_upstream_mask.ulMask[0],
		igmpi_disabled_mask.ulMask[1], igmpi_disabled_mask.ulMask[0]);
#endif
}


void igmpsnp_xmit_portmask(struct sk_buff * skb, TstLPortMask mask)
{
	K_LPortMaskCopy (&CCL_LP(skb)->dwDstPortMsk, &mask);
	CCL_LP(skb)->wVid = 1;
	CCL_LP(skb)->byPriority = 0;

	cclmx_TxPkt (skb);
#if 0
	debug("Tx a pkt to %08x,%08x\n", mask.ulMask[1], mask.ulMask[0]);
#endif
}


void igmpsnp_xmit_exc_port (struct sk_buff * skb, unsigned char port_no)
{
	TstLPortMask mask;

	update_portmasks ();
	if (port_no >= MAX_LOGIC_PORT)
		return;
	K_LPortMaskSetAll (&mask);
	K_LPortMaskClrPort (&mask, port_no);
	K_LPortMaskClrPorts (&mask, &igmpi_disabled_mask);
	igmpsnp_xmit_portmask (skb, mask);
}


unsigned short in_cksum(unsigned short *addr, int len)
{
	register int sum = 0;
	register int nleft = len;
	unsigned short answer = 0;

	while (nleft > 1) {
		sum += get_unaligned (addr);
		++addr;
		nleft -= 2;
	}
	if (nleft == 1) {
		sum += *((unsigned char *)addr);
	}
	sum = (sum >> 16) + (sum & 0xffff);
	answer = (unsigned short) ~sum;
	return answer;
}


/* test the existence of given multicast group ip */
int table_test_group_existence (Tuint32 ulGroup)
{
	Tbool		bStatic;
	TstPPortMask	stPhyMask;

	if (K_IpMcstGetEntry (ulGroup, &bStatic, &stPhyMask) == KRN_RET_OK) {
		if (!bStatic) {
			return 1;
		}
	}

	return 0;
}


/* change mask of all entries in the table for new router port */
void table_modify_new_root (int old_root_port, int new_root_port)
{
	Tuint16		usSize;
	Tuint16		usSlot;
	Tuint32		ulIp;
	Tbool		bStatic;
	TstPPortMask	stPhyMask;

	TstPPortMask	old, new;

	if (old_root_port == new_root_port) {
		return;
	}

	cclmx_LId2PMask (&old, old_root_port);
	cclmx_LId2PMask (&new, new_root_port);

	K_IpMcstGetTblSize (&usSize);
	for (usSlot = 0; usSlot < usSize; ++usSlot) {
		if (K_IpMcstGetTblEntry (usSlot, &ulIp, &bStatic, &stPhyMask) == KRN_RET_OK) {
			if (bStatic)
				continue;
			K_PPortMaskClrPorts (&stPhyMask, &old);
			K_PPortMaskSetPorts (&stPhyMask, &new);
			K_IpMcstInsEntry (ulIp, bStatic, &stPhyMask);
		}
	}
}


/*
 * modify multicast table for entries with disabled_port was set
 *   -> clear disabled_port bit for all these entries
 */
int table_modify_new_disabled (int disabled_port)
{
	Tuint16		usSize;
	Tuint16		usSlot;
	Tuint32		ulIp;
	Tbool		bStatic;
	TstPPortMask	stPhyMask;

	K_IpMcstGetTblSize (&usSize);
	for (usSlot = 0; usSlot < usSize; ++usSlot) {
		if (K_IpMcstGetTblEntry (usSlot, &ulIp, &bStatic, &stPhyMask) == KRN_RET_OK) {
			if (bStatic)
				continue;
			K_PPortMaskClrPort (&stPhyMask, disabled_port);
			if (K_PPortMaskIsZero (&stPhyMask)) {
				K_IpMcstDelEntry(ulIp);
			}
			else {
				K_IpMcstInsEntry(ulIp, bStatic, &stPhyMask);
			}
		}
	}
	return 0;
}


int table_del_group_with_port (Tuint32 ulGroup, Tuint8 ucPort)
{
	Tbool bStatic;
	TstPPortMask stPhyMask;
	TstLPortMask stLogicMask;

	if (K_IpMcstGetEntry (ulGroup, &bStatic, &stPhyMask) != KRN_RET_OK)
		return 0;

	if (bStatic) /* don't modify static entries */
		return 0;

	cclmx_PMask2LMask (&stLogicMask, &stPhyMask);
	/* remove the specified port */
	K_LPortMaskClrPort (&stLogicMask, ucPort);
	/* add the router port */
	K_LPortMaskSetPorts (&stLogicMask, &igmpi_upstream_mask);
	cclmx_LMask2PMask (&stPhyMask, &stLogicMask);

	if ( (K_LPortMaskIsZero (&stLogicMask))
		|| (!(K_LPortMaskCmp (&stLogicMask, &igmpi_upstream_mask))) )
	{
		K_IpMcstDelEntry(ulGroup);
	}
	else {
		K_IpMcstInsEntry(ulGroup, bStatic, &stPhyMask);
	}
	return 0;
}


int table_add_group_with_port (Tuint32 ulGroup, Tuint8 ucPort)
{
	Tbool bStatic;
	TstPPortMask stPhyMask;
	TstLPortMask stLogicMask;

	/* don't add it if the port is disabled */
	if (K_LPortMaskGetPort (&igmpi_disabled_mask, ucPort))
		return 0;

	if (K_IpMcstGetEntry (ulGroup, &bStatic, &stPhyMask) == KRN_RET_OK) {
		if (bStatic) /* don't modify static entries */
			return 0;
	}
	else {
		bStatic = False;
		K_PPortMaskClrAll (&stPhyMask);
	}

	cclmx_PMask2LMask (&stLogicMask, &stPhyMask);
	/* add the specified port */
	K_LPortMaskSetPort (&stLogicMask, ucPort);
	/* add the router port */
	K_LPortMaskSetPorts (&stLogicMask, &igmpi_upstream_mask);
	cclmx_LMask2PMask (&stPhyMask, &stLogicMask);

	K_IpMcstInsEntry(ulGroup, bStatic, &stPhyMask);
	return 0;
}


void _igmp_stp_port_forwarding (int port_no)
{
	K_LPortMaskClrPort (&igmpi_disabled_mask, port_no);
	return;
}


void _igmp_stp_port_disabled (int port_no)
{
	igmp_db_t del;
	K_LPortMaskSetPort (&igmpi_disabled_mask, port_no);
#ifndef __CCL_TUNING_SLOW_IP_MCST_TBL__
	table_modify_new_disabled (port_no);
#endif
	while (igmp_db_find_port (&del, port_no)) {
		igmp_db_del_entry (del);
	}
	return;
}


int is_port_disabled (unsigned char port_no)
{
	if (K_LPortMaskGetPort (&igmpi_disabled_mask, port_no))
		return 1;
	else
		return 0;
}


void update_router_port (unsigned char new_port)
{
	unsigned char old_port;

	old_port = igmp_router.port;
	igmp_router.port = new_port;

	update_portmasks ();
	table_modify_new_root (old_port, new_port);
#if 0
	debug("mask_up %08x,%08x disabled %08x,%08x\n",
		igmpi_upstream_mask.ulMask[1], igmpi_upstream_mask.ulMask[0],
		igmpi_disabled_mask.ulMask[1], igmpi_disabled_mask.ulMask[0]);
#endif
}


