/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  THIS IS AN UNPUBLISHED WORK WHICH CONTAINS CONFIDENTIAL INFORMATION
 *  FROM CCL, ITRI.  NO PART OF THIS WORK MAY BE USED IN ANY WAY WITHOUT
 *  THE PRIOR WRITTEN PERMISSION.  ANY UNAUTHORIZED USE COULD SUBJECT THE
 *  PERPETRATOR TO CRIMINAL AND CIVIL LIABILITY.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 *  Author: Winfred Lu (N300, CCL/ITRI)
 *
 ***************************************************************************/

#include <linux/kernel.h>
#include <linux/skbuff.h>	/* for struct sk_buff */
#include <linux/in.h>
#include <asm/checksum.h>	/* for ip_fast_csum */
#ifdef __LINUX_2_6_19__
#include <linux/hardirq.h>
#else
#include <asm/hardirq.h>	/* for in_interrupt */
#endif

#ifdef __DRAFT_IETF_MAGMA_IGMP_PROXY__
#include "krnerr.h"
#include "krnipmcst.h"
#include "krnsys.h"
#endif

/* ccl multiplexer */
#include "../cclmx/ccl.h"	/* for additional info in cb in sk_buff */
#include "../cclmx/ccl_mx.h"	/* for ccl mutiplexer registration */

#include "igmpsnoop.h"
#include "db.h"
#include "util.h"
#include "igmp_netfilter.h"


#ifdef __DRAFT_IETF_MAGMA_IGMP_PROXY__
extern Tuint8		sys_mac[6]; /* system MAC address */
#endif
extern TstIgmpSnpStatus	igmpsnp_global_stats;
extern igmp_router_t	igmp_router;



int igmp_check_header(struct iphdr *iph)
{
	struct igmphdr * igmph;

	if (iph->ihl < 5 || iph->version != 4){
		debug("check_header: ip version or length error!\n");
		return -1;
	}
	if (!IN_MULTICAST(ntohl(iph->daddr))) {
		debug("check_header: dst ip not in multicast range\n");
		return -1;
	}
/*        if (in_cksum((unsigned short *)iph, 20) != 0){}*/
	if (ip_fast_csum((u8 *)iph, iph->ihl) != 0){
		debug("check_header: ip header checksum error!\n");
		return -1;
	}

	igmph = (struct igmphdr *)((char *)iph + iph->ihl * 4);

	if (igmph->type != IGMP_HOST_MEMBERSHIP_QUERY &&
	    igmph->type != IGMP_HOST_MEMBERSHIP_REPORT &&
	    igmph->type != IGMP_HOST_NEW_MEMBERSHIP_REPORT &&
	    igmph->type != IGMP_HOST_LEAVE_MESSAGE) {
		debug("warning: a pkt with uncompatible IGMP version rcved!\n");
		return 0;
	}

	if (in_cksum((__u16 *)igmph, sizeof(struct igmphdr)) != 0) {
		debug("check_header: igmp header checksum error!\n");
		return -1;
	}

	if (iph->protocol != IPPROTO_IGMP) {
		debug("warning: a pkt protocol type not IGMP received!\n");
		return 0;
	}

	return 1;
}


#if 0
void build_igmp_query (struct sk_buff ** skb)
{
	unsigned char dst_mac[6] = {0x01, 0x00, 0x5e, 0x00, 0x00, 0x01};
	unsigned char src_mac[6] = {0x01, 0x00, 0x5e, 0x00, 0x00, 0x02};
	unsigned short type	= htons(0x0800);

	unsigned char lv	= 0x45;
	unsigned char tos	= 0x00;
	unsigned short tot_len	= htons (sizeof(struct iphdr)
			+ sizeof(struct igmphdr));
	unsigned short id	= htons(0x1234);
	unsigned short frag_off	= htons(0x0000);
	unsigned char ttl	= 64;
	unsigned char protocol	= IPPROTO_IGMP;
	unsigned short csum	= 0;
	unsigned long saddr	= htonl(0xE0000002L);
	unsigned long daddr	= htonl(0xE0000001L);

	unsigned char itype	= 0x11;
	unsigned char code	= 0x23;
	unsigned long gda	= 0xE0010203L;

	unsigned short * iph;
	unsigned short * igmph;

	/* ethernet II header */
	memcpy (skb_put(*skb, 6), &dst_mac, 6);
	memcpy (skb_put(*skb, 6), &src_mac, 6);
	memcpy (skb_put(*skb, 2), &type, 2);

	/* ip header */
	iph = (unsigned short *)((*skb)->tail);
	memcpy (skb_put(*skb, 1), &lv, 1);
	memcpy (skb_put(*skb, 1), &tos, 1);
	memcpy (skb_put(*skb, 2), &tot_len, 2);
	memcpy (skb_put(*skb, 2), &id, 2);
	memcpy (skb_put(*skb, 2), &frag_off, 2);
	memcpy (skb_put(*skb, 1), &ttl, 1);
	memcpy (skb_put(*skb, 1), &protocol, 1);
	memcpy (skb_put(*skb, 2), &csum, 2);
	memcpy (skb_put(*skb, 4), &saddr, 4);
	memcpy (skb_put(*skb, 4), &daddr, 4);
	/* calculate cksum */
	(*skb)->tail -= 10;
	(*skb)->len -= 10;
	csum = in_cksum(iph, sizeof(struct iphdr));
	csum = htons(csum);
	memcpy (skb_put(*skb, 2), &csum, 2);
	skb_put (*skb, 4+4);		/* push saddr, daddr */

	/* igmp header */
	iph = (unsigned short *)((*skb)->tail);
	memcpy (skb_put(*skb, 1), &itype, 1);
	memcpy (skb_put(*skb, 1), &code, 1);
	csum = 0;
	memcpy (skb_put(*skb, 2), &csum, 2);
	memcpy (skb_put(*skb, 4), &gda, 4);
	/* calculate cksum */
	(*skb)->tail -= 6;
	(*skb)->len -= 6;
	csum = in_cksum(iph, sizeof(struct igmphdr));
	csum = htons(csum);
	memcpy (skb_put(*skb, 4), &csum, 4);
	skb_put (*skb, 4);		/* push gda */
}
#endif


#ifdef __DRAFT_IETF_MAGMA_IGMP_PROXY__
#define IGMP_REPORT_PDU_LEN	42 /* MAC + IP + IGMP headers length */
struct sk_buff * create_report_pdu(unsigned short version, __u32 group)
{
	struct sk_buff *skb;
	unsigned char *	skb_buffer;
	unsigned short	csum; /* checksum */

	skb = dev_alloc_skb(IGMP_REPORT_PDU_LEN);
	if (!skb) {
		debug("create_report_pdu: dev_alloc_skb() failed.\n");
		return NULL;
	}

	skb_buffer = (unsigned char *)skb_put(skb, IGMP_REPORT_PDU_LEN);
	memset(skb_buffer, 0, IGMP_REPORT_PDU_LEN);
	CCL_LP(skb)->wSize = IGMP_REPORT_PDU_LEN;
	CCL_LP(skb)->bIfTagged = 0;
	skb->dev = dev_get_by_name("eth0");
	group = htonl(group);

	/* MAC: Destination Address: Multicast All Routers */
	/*   Since skb_buffer[] is already cleared, all 0x00 need not be set */
	skb_buffer[0] = 0x01;
	/*   skb_buffer[1] = 0x00; */
	skb_buffer[2] = 0x5E;
	/*   skb_buffer[3] = 0x00; */
	/*   skb_buffer[4] = 0x00; */
	skb_buffer[5] = 0x02;
	/* MAC: Source Address */
	skb_buffer[6] = sys_mac[0];
	skb_buffer[7] = sys_mac[1];
	skb_buffer[8] = sys_mac[2];
	skb_buffer[9] = sys_mac[3];
	skb_buffer[10] = sys_mac[4];
	skb_buffer[11] = sys_mac[5];
	/* MAC: Type */
	skb_buffer[12] = 0x08;
	/*   skb_buffer[13] = 0x00; */
	/* IP: Version & Header Length */
	skb_buffer[14] = 0x45;
	/* IP: Type of Service */
	/*   skb_buffer[15] = 0x00; */
	/* IP: Total Length: 28 */
	/*   skb_buffer[16] = 0x00; */
	skb_buffer[17] = 0x1C;
	/* IP: Identification */
	/*   skb_buffer[18] = 0x00; */
	/*   skb_buffer[19] = 0x00; */
	/* IP: Flags */
	/*   skb_buffer[20] = 0x00; */
	/*   skb_buffer[21] = 0x00; */
	/* IP: Time to Live: 64 */
	skb_buffer[22] = 0x40;
	/* IP: Protocol: IGMP */
	skb_buffer[23] = 0x02;
	/* IP: Checksum (calculated below) */
	/* IP: Source Address: 0.0.0.0 */
	/*   skb_buffer[26] = 0x00; */
	/*   skb_buffer[27] = 0x00; */
	/*   skb_buffer[38] = 0x00; */
	/*   skb_buffer[39] = 0x00; */
	/* IP: Destination Address */
	memcpy(skb_buffer+30, &group, 4);
	/* IP: Calculate Checksum */
	csum = in_cksum(skb_buffer+14, 20); /* no need to do htons(csum) */
	memcpy (skb_buffer+24, &csum, 2);
	/* IGMP: Version & Type */
	if (version == 1) {	/* v1 Report */
		skb_buffer[34] = 0x12;
	}
	else {			/* v2 Report */
		skb_buffer[34] = 0x16;
	}
	/* IGMP: Max Response Time: 0 */
	skb_buffer[35] = 0x00;
	/* IGMP: Checksum (calculated below) */
	/* IGMP: Group */
	memcpy(skb_buffer+38, &group, 4);
	/* IGMP: Calculate Checksum */
	csum = in_cksum(skb_buffer+34, 8); /* no need to do htons(csum) */
	memcpy (skb_buffer+36, &csum, 2);
	debug("create_report_pdu: packet content:\n%02X %02X %02X %02X %02X %02X %02X %02X-%02X %02X %02X %02X %02X %02X %02X %02X\n%02X %02X %02X %02X %02X %02X %02X %02X-%02X %02X %02X %02X %02X %02X %02X %02X\n%02X %02X %02X %02X %02X %02X %02X %02X-%02X %02X\n", skb_buffer[0], skb_buffer[1], skb_buffer[2], skb_buffer[3], skb_buffer[4], skb_buffer[5], skb_buffer[6], skb_buffer[7], skb_buffer[8], skb_buffer[9], skb_buffer[10], skb_buffer[11], skb_buffer[12], skb_buffer[13], skb_buffer[14], skb_buffer[15], skb_buffer[16], skb_buffer[17], skb_buffer[18], skb_buffer[19], skb_buffer[20], skb_buffer[21], skb_buffer[22], skb_buffer[23], skb_buffer[24], skb_buffer[25], skb_buffer[26], skb_buffer[27], skb_buffer[28], skb_buffer[29], skb_buffer[30], skb_buffer[31], skb_buffer[32], skb_buffer[33], skb_buffer[34], skb_buffer[35], skb_buffer[36], skb_buffer[37], skb_buffer[38], skb_buffer[39], skb_buffer[40], skb_buffer[41]);
	return skb;
}
#endif


void forward_packet_dir (struct sk_buff * skb, int dir)
{
	struct sk_buff * new_skb;

	if ((new_skb = skb_copy (skb, GFP_ATOMIC)) != NULL){
		new_skb->data -= 14;
		igmpsnp_xmit_dir (new_skb, dir);
	}
}


void forward_packet_exc_port (struct sk_buff * skb, unsigned char port)
{
	struct sk_buff * new_skb;

	if ((new_skb = skb_copy (skb, GFP_ATOMIC)) != NULL){
		new_skb->data -= 14;
		igmpsnp_xmit_exc_port (new_skb, port);
	}
}


void igmp_query_handle(
		struct sk_buff * pdu,
		struct iphdr * iph,
		struct igmphdr * igmph)
{
#ifdef __DRAFT_IETF_MAGMA_IGMP_PROXY__
	struct sk_buff * report_pdu;
#endif
	unsigned char src_port;
	unsigned char t;
	__u32 router, group;

	src_port = CCL_LP(pdu)->bySrcPortId;
	router = ntohl (iph->saddr);
	group = ntohl (igmph->group);

	if (igmph->group == 0) { /* general query */
		t = igmph->code? igmph->code : QUERY_RESPONSE_INTV;
		++igmpsnp_global_stats.g_query_rcvd;
		debug("A v%d general QUERY received from port %d. (interval:%d.%d)\n",
				igmph->code ? 2 : 1,
				src_port + 1, t/10, t%10);
	} else { /* v2 group specific query */
		t = igmph->code? igmph->code : QUERY_RESPONSE_INTV/10;
		++igmpsnp_global_stats.gs_query_rcvd;
		debug("A v2 gs-QUERY received from port %d. (group:%d.%d.%d.%d interval:%d.%d)\n",
			src_port + 1,
			(group & 0xff000000L) >> 24,
			(group & 0x00ff0000L) >> 16,
			(group & 0x0000ff00L) >> 8,
			(group & 0x000000ffL),
			t/10, t%10);
	}

	if (is_port_disabled (src_port)) {
		debug("drop packet due to port disabled\n");
		return;
	}

	if (router == 0) /* non-querier ? */
		return;

	igmp_router.addr = router;
	if (src_port < MAX_LOGIC_PORT) {
		update_router_port (src_port);
	}

#ifdef __DRAFT_IETF_MAGMA_IGMP_PROXY__
	if (igmpsnp_global_stats.proxy_enable) {
		if (group == 0) { /* general query */
			Tuint16		usSize;
			Tuint32		ulGroup;
			Tbool		bStatic;
			TstPPortMask	stPortMask;
			int		i;

			K_IpMcstGetTblSize(&usSize);
			for (i = 0; i < usSize; ++i) {
				if (K_IpMcstGetTblEntry(i, &ulGroup, &bStatic, &stPortMask) == KRN_RET_OK) {
					if (igmph->code) { /* v2 g-Query  */
						report_pdu = create_report_pdu (2, ulGroup);
						if (report_pdu) {
							igmpsnp_xmit_dir (report_pdu, IGMPI_UPSTREAM);
							++igmpsnp_global_stats.report_xmit;
						}
					}
					else { /* v1 g-Query */
						report_pdu = create_report_pdu (1, ulGroup);
						if (report_pdu) {
							igmpsnp_xmit_dir (report_pdu, IGMPI_UPSTREAM);
							++igmpsnp_global_stats.report_xmit;
						}
					}
				}
			}
		}
		else { /* v2 group specific query */
			if (table_test_group_existence (group)) {
				report_pdu = create_report_pdu (2, group);
				if (report_pdu) {
					igmpsnp_xmit_dir (report_pdu, IGMPI_UPSTREAM);
					++igmpsnp_global_stats.report_xmit;
				}
			}
		}
	}
#endif
#if defined(__DRAFT_IETF_MAGMA_SNOOP__) || defined(__DRAFT_IETF_MAGMA_IGMP_PROXY__)
	forward_packet_dir (pdu, IGMPI_DOWNSTREAM);
	if (group == 0) { /* general query */
		++igmpsnp_global_stats.g_query_xmit;
	}
	else { /* v2 group specific query */
		++igmpsnp_global_stats.gs_query_xmit;
	}
#endif
}


void igmp_report_handle(
		struct sk_buff * pdu,
		struct iphdr *iph,
		struct igmphdr * igmph)
{
	unsigned char src_port;
	__u32 group;
	igmp_db_t db, * new_db;

	src_port = CCL_LP(pdu)->bySrcPortId;
	group = ntohl (igmph->group);

	if (MAX_LOGIC_PORT <= src_port)
		return;

	debug("A v%d REPORT received from port %d. (group:%d.%d.%d.%d)\n",
		igmph->type == IGMP_HOST_MEMBERSHIP_REPORT? 1 : 2,
		src_port + 1,
		(group & 0xff000000L) >> 24,
		(group & 0x00ff0000L) >> 16,
		(group & 0x0000ff00L) >> 8,
		(group & 0x000000ffL));

	if (is_port_disabled (src_port)) {
		debug("drop packet due to port disabled\n");
		return;
	}

#ifdef __DRAFT_IETF_MAGMA_IGMP_PROXY__
	if (!( (igmpsnp_global_stats.proxy_enable) && (table_test_group_existence (group)) ) )
#endif
	{
#if defined(__DRAFT_IETF_MAGMA_SNOOP__) || defined(__DRAFT_IETF_MAGMA_IGMP_PROXY__)
		if (igmp_router.port != src_port) {
			forward_packet_dir (pdu, IGMPI_UPSTREAM);
			++igmpsnp_global_stats.report_xmit;
		}
#endif
	}

	/* notify switch + */
	table_add_group_with_port (group, src_port);
	if (!table_test_group_existence (group)) /* insert failed */
		return;

	if (igmp_db_find_entry (&db, group, src_port)) {
		/* modify timer for db entry that existed */
		igmp_db_modify_life (db, GROUP_MEMBERSHIP_INTV);
		return;
	}

	/* new coming report */
	new_db = igmp_db_new_entry (group, src_port, GROUP_MEMBERSHIP_INTV);
	if (NULL != new_db) {
		igmp_db_insert_entry (new_db);
#ifdef IGMPSNP_DEBUG
		igmp_db_debug_print ("insert");
#endif
	}
}


#if 0
void build_igmp_gs_query(struct sk_buff ** skb, struct in_addr addr)
{
	unsigned char dst_mac[6] = {0x01, 0x00, 0x5e, 0x00, 0x00, 0x01}; /* all hosts */
	unsigned char src_mac[6];
	unsigned short type	= htons(0x0800);

	unsigned char lv	= 0x45;
	unsigned char tos	= 0x00;
	unsigned short tot_len	= htons (sizeof(struct iphdr)
			+ sizeof(struct igmphdr));
	unsigned short id	= htons(0x1234);
	unsigned short frag_off	= htons(0x0000);
	unsigned char ttl	= 1;
	unsigned char protocol	= IPPROTO_IGMP;
	unsigned short csum	= 0;
	unsigned long saddr	= htonl(0x00000000L);
	unsigned long daddr	= htonl(0xE0000002L);

	unsigned char itype	= IGMP_HOST_MEMBERSHIP_QUERY;
	unsigned char code	= GS_QUERY_RESP_INTV;
	unsigned long gda	= addr.s_addr;

	unsigned short * iph;

	K_SysGetMac(src_mac); /* get Mac address of the switch */

	/* ethernet II header */
	memcpy (skb_put(*skb, 6), &dst_mac, 6);
	memcpy (skb_put(*skb, 6), &src_mac, 6);
	memcpy (skb_put(*skb, 2), &type, 2);

	/* ip header */
	iph = (unsigned short *)((*skb)->tail);
	memcpy (skb_put(*skb, 1), &lv, 1);
	memcpy (skb_put(*skb, 1), &tos, 1);
	memcpy (skb_put(*skb, 2), &tot_len, 2);
	memcpy (skb_put(*skb, 2), &id, 2);
	memcpy (skb_put(*skb, 2), &frag_off, 2);
	memcpy (skb_put(*skb, 1), &ttl, 1);
	memcpy (skb_put(*skb, 1), &protocol, 1);
	memcpy (skb_put(*skb, 2), &csum, 2);
	memcpy (skb_put(*skb, 4), &saddr, 4);
	memcpy (skb_put(*skb, 4), &daddr, 4);
	/* calculate ip header cksum */
	(*skb)->tail -= 10;
	(*skb)->len -= 10;
	csum = in_cksum(iph, sizeof(struct iphdr));
	csum = htons(csum);
	memcpy (skb_put(*skb, 2), &csum, 2);
	skb_put (*skb, 4+4);		/* push saddr, daddr */

	/* igmp header */
	iph = (unsigned short *)((*skb)->tail);
	memcpy (skb_put(*skb, 1), &itype, 1);
	memcpy (skb_put(*skb, 1), &code, 1);
	csum = 0;
	memcpy (skb_put(*skb, 2), &csum, 2);
	memcpy (skb_put(*skb, 4), &gda, 4);
	/* calculate cksum */
	(*skb)->tail -= 6;
	(*skb)->len -= 6;
	csum = in_cksum(iph, sizeof(struct igmphdr));
	csum = htons(csum);
	memcpy (skb_put(*skb, 4), &csum, 4);
	skb_put (*skb, 4);		/* push gda */
}
#endif


void igmp_leave_handle(
		struct sk_buff *pdu,
		struct iphdr * iph,
		struct igmphdr * igmph)
{
	unsigned char src_port;
	__u32 group;
	igmp_db_t db;

	src_port = CCL_LP(pdu)->bySrcPortId;
	group = ntohl (igmph->group);

	debug("A v2 LEAVE received from port %d. (group:%d.%d.%d.%d)\n",
		src_port + 1,
		(group & 0xff000000L) >> 24,
		(group & 0x00ff0000L) >> 16,
		(group & 0x0000ff00L) >> 8,
		(group & 0x000000ffL));

	if (is_port_disabled (src_port)) {
		debug("drop packet due to port disabled\n");
		return;
	}

	if (igmpsnp_global_stats.fastleave_enable) {
		if (0 < src_port && src_port < MAX_LOGIC_PORT) {
			/* notify switch - */
			table_del_group_with_port (group, src_port);
			db.gda = group;
			db.port = src_port;
			igmp_db_del_entry (db);
		}
	}
	/* else: should send gs-query? */

#ifdef __DRAFT_IETF_MAGMA_IGMP_PROXY__
	if ( (igmpsnp_global_stats.proxy_enable) && (igmpsnp_global_stats.fastleave_enable) ) {
		if (table_test_group_existence (group)) {
			return;
		}
	}
#endif
#if defined(__DRAFT_IETF_MAGMA_SNOOP__) || defined(__DRAFT_IETF_MAGMA_IGMP_PROXY__)
	forward_packet_dir (pdu, IGMPI_UPSTREAM);
	++igmpsnp_global_stats.leave_xmit;
#endif
}


int igmp_received (struct sk_buff * rcv_pdu)
{
	struct iphdr * iph;
	struct igmphdr * igmph;

	iph = rcv_pdu->nh.iph;

	++igmpsnp_global_stats.total_pkt_rcvd;

	switch (igmp_check_header(iph)) {
	case 0:
		++igmpsnp_global_stats.valid_pkt_rcvd;
		++igmpsnp_global_stats.other_rcvd;
		forward_packet_exc_port (rcv_pdu, CCL_LP(rcv_pdu)->bySrcPortId);
		//skb_unlink(rcv_pdu);
		//kfree_skb(rcv_pdu);
		return CCL_MX_DROP;
		break;
	case -1:
		++igmpsnp_global_stats.invalid_pkt_rcvd;
		//skb_unlink(rcv_pdu);
		//kfree_skb(rcv_pdu);
		return CCL_MX_DROP;
		break;
	}
	++igmpsnp_global_stats.valid_pkt_rcvd;

	if (!igmpsnp_global_stats.igmpsnp_enable) {
		/* broadcast this packet */
		return 0;
	}

#if 0
	debug("iphdr -> ihl %d, ver %d, tos %d, tot_len 0x%x,\n"
		"        id 0x%d, frag_off 0x%d, ttl %d, protocol %d,\n"
		"        saddr 0x%x, daddr 0x%x\n\n", 
		iph->ihl, iph->version, iph->tos, iph->tot_len,
		iph->id, iph->frag_off, iph->ttl, iph->protocol,
		iph->saddr, iph->daddr);
#endif

	igmph = (struct igmphdr *)((char *)iph + iph->ihl*4);

	switch (igmph->type) {
	case IGMP_HOST_MEMBERSHIP_QUERY:
		igmp_query_handle(rcv_pdu, iph, igmph);
		break;
	case IGMP_HOST_MEMBERSHIP_REPORT: /* v1 */
	case IGMP_HOST_NEW_MEMBERSHIP_REPORT: /* v2 */
		++igmpsnp_global_stats.report_rcvd;
		igmp_report_handle(rcv_pdu, iph, igmph);
		break;
	case IGMP_HOST_LEAVE_MESSAGE:
		++igmpsnp_global_stats.leave_rcvd;
		igmp_leave_handle(rcv_pdu, iph, igmph);
		break;
	default:
		debug("An unrecognized igmp message received.\n");
		++igmpsnp_global_stats.other_rcvd;
		//forward_packet_exc_port (rcv_pdu, CCL_LP(rcv_pdu)->bySrcPortId);
		break;
	}
#ifdef __LINUX_2_6_19__
#else
	skb_unlink(rcv_pdu);
#endif
	kfree_skb(rcv_pdu);
	return CCL_MX_ACCEPT;
}


void igmp_sleep (unsigned howlong)
{
	if (in_interrupt ()) {
		return;
	}
	current->state = TASK_INTERRUPTIBLE;
	schedule_timeout (howlong);
}


int igmp_report_cleaner (void * vp)
{
	igmp_db_t del = {0};

	while(1)
	{
		if (!igmpsnp_global_stats.igmpsnp_enable) {
			igmp_sleep (600);
			continue;
		}

		igmp_db_decrease_life ();

		while (igmp_db_find_dead (&del)) {
			/* notify switch - */
			table_del_group_with_port (del.gda, del.port);
			igmp_db_del_entry (del);
#ifdef IGMPSNP_DEBUG
			igmp_db_debug_print ("delete");
#endif
		}
		igmp_sleep (110);
	}
	return 0;
}


