/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  THIS IS AN UNPUBLISHED WORK WHICH CONTAINS CONFIDENTIAL INFORMATION
 *  FROM CCL, ITRI.  NO PART OF THIS WORK MAY BE USED IN ANY WAY WITHOUT
 *  THE PRIOR WRITTEN PERMISSION.  ANY UNAUTHORIZED USE COULD SUBJECT THE
 *  PERPETRATOR TO CRIMINAL AND CIVIL LIABILITY.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************/

// ********************************************
// Name   : gvr.c
// Author : Liu, Ren Hao (N300, CCL/ITRI)
// Date   : 2003/06/26
// Note   : GVR = "GARP VLAN REGISTRATION APPLICATION"
// ********************************************

#include "common.h"
#include "garp_common.h"
#include "gvrp_common.h"


extern garp_statistics	gvrp_statistics[MAX_LOGIC_PORT];
extern Tuint32		gvrp_debug_level;
extern Gvr		*my_gvr;

/*
* Creates a new instance of GVR, allocating and initialising a control
* block, returning True and a pointer to this instance if creation suceeds.
* Also creates instances of GVD (the GARP VLAN database) and of GIP
* (which controls information propagation).
*
* Ports are created by the system and added to GVR separately (see
* gvr_added_port() and gvr_removed_port() below).
*
* The operating system supplied process_id is for use in subsequent calls
* to operating system services. The system itself ensures the temporal
* scope of process_id, guarding against timers yet to expire for destroyed
* processes etc. Although process_id will be implicitly supplied by many
* if not most systems, it is made explicit in this implementation for
* clarity.
* 
*/

Tbool gvr_create_gvr(Tuint32 process_id, void **gvr)
{  
 Garp *my_gvr;

 if (!sysmalloc(sizeof(Gvr),(void **)&my_gvr)) {
     goto gvr_creation_failure;
 }

 my_gvr->process_id = process_id;
 my_gvr->gid = NULL;

 my_gvr->max_gid_index = GVRP_DB_SIZE - 1;
 my_gvr->last_gid_used = 0;

 my_gvr->join_indication_fn  = gvr_join_indication;
 my_gvr->leave_indication_fn = gvr_leave_indication;
 my_gvr->join_propagated_fn  = gvr_join_leave_propagated;
 my_gvr->leave_propagated_fn = gvr_join_leave_propagated;
 my_gvr->transmit_fn         = gvr_tx;
 my_gvr->receive_fn          = gvr_rcv; 
 my_gvr->added_port_fn       = gvr_added_port;
 my_gvr->removed_port_fn     = gvr_removed_port;

 ((Gvr *)my_gvr)->number_of_gvd_entries = GVRP_DB_SIZE;
 ((Gvr *)my_gvr)->last_gvd_used_plus1   = 0;

 *gvr = my_gvr;
 return(True);

gvr_creation_failure: return (False);

}


/*
* Destroys an instance of GVR, destroying and deallocating the associated
* instances of GVD and GIP, and any instances of GID remaining.
*/
void gvr_destroy_gvr(void *gvr)
{  
  Gid *my_port;
  Garp *application= &my_gvr->garp;

  while((my_port = application->gid) != NULL)
  {
    gid_stop_all_timer(my_port);
    gid_destroy_port( application, my_port->port_no);
    memset(&gvrp_statistics[my_port->port_no],0,sizeof(garp_statistics)); 
  }

  gvd_destroy_gvd(my_gvr->gvd);
  gip_destroy_gip(application->gip);
}

/*
* The system has created a new port for this application and added it to
* the ring of GID ports. This function ensures that Static VLAN Entries
* from the Permanent Database are represented in the GVD database (which
* provides VLAN ID to GID index mapping) and have GID machines in the newly
* added port (with the correct management control state). This can result
* in the creation of new GID machines or modification of the state of
* existing machines.
*
* Newly created ports are 'connected' for the purpose of GARP information
* propagation using the separate function gip_connect_port(). This should
* be called after this function, gvr_added_port. It may cause GVRP/GIP
* to propagate information from the static management controls through
* other ports.
*
* It is assumed that new ports will be 'connected' correctly before the
* application continues as determined by the active topology of the network,
* i.e. if stp_forwarding(port_no) gvr_connect_port(port_no);.
*
* As the system continues to run it should invoke gip_disconnect_port()
* and gip_connect_port() as required to maintain the required connectivity.


* Query the Permanent Database for Static VLAN Entries with "Registration
* Forbidden" or "Registration Fixed" for this port. Repeat the following
* steps until there are no more entries to be found.
*
* Check that the VLAN ID is represented in VLD. If not create it, and
* create GID machines for all the other ports with control state "Normal
* Registration" and create the GID machine for this port. Change the
* control state for this port's GID machine to forbidden or fixed as
* required.
*
*/
//add gid into gip ring when port state in forwarding state
void gvr_added_port(void *gvr, Tuint32 port_no)
{
 Garp *application= &my_gvr->garp;

 if(cclmx_PortGetStpState(port_no) == STP_PORT_FORWARDING){
     gip_connect_port(application, port_no);
 }
}


/*
* The system has removed and destroyed the GID port. This function should
* provide any application specific cleanup required.

* Provide any GVR specific cleanup or management alert functions for the
* removed port.
*/
void gvr_removed_port(void *gvr, Tuint32 port_no)
{ 
}


/******************************************************************************
* GVR : GARP VLAN REGISTRATION APPLICATION : JOIN, LEAVE INDICATIONS
*******************************************************************************/

//add vlan entry into vlan table
void gvr_join_indication(void *gvr, void *port, Tuint32 joining_gid_index)
{  
  Vlan_id key;
  Gid *my_port=(Gid *)port;

  if(gvd_get_key( my_gvr->gvd, joining_gid_index, &key)) {	     
     udelay(50);
     vfdb_forward(my_port->port_no, key);
  }  
}

//remove vlan entry from vlan table
void gvr_leave_indication(void *gvr, void *port, Tuint32 leaving_gid_index)
{
  Vlan_id  key;
  Gid *my_port=(Gid *)port;

  if(gvd_get_key(my_gvr->gvd, leaving_gid_index, &key))
   {
      udelay(50);
      vfdb_filter(my_port->port_no, key);
   }
}

/*
* Nothing to be done since, unlike GMR with its Forward All Unregistered
* port mode, a join indication on one port does not cause filtering to be
* instantiated on another.
*/
void gvr_join_leave_propagated(void *gvr, void *port, Tuint32 gid_index)
{  
}

/******************************************************************************
* GVR : GARP VLAN REGISTRATION APPLICATION : RECEIVE MESSAGE PROCESSING
*******************************************************************************/

/*
* Place holder for management alert functions indicating registrations
* for more VLANs have been received than can be accepted.
*/
static void gvr_db_full(Gvr *gvr, Gid *my_port)
{ 
 if(gvrp_debug_level & GVRP_DEBUG_DB){
    printk("GVRP: GVRP Database Full...\n");
 }
}


/*
* Process one received message.
*
* A LeaveAll message never causes an indication (join or leave directly),
* even for the poTuint32 to poTuint32 link protocol enhancements (where an
* ordinary Leave does). No further work is needed here.
*
* A LeaveAllRange message is currently treated exactly as a LeaveAll
* (i.e. the range is ignored). 
*
* All the remaining messages refer to a single attribute (i.e. a single
* registered VLAN). Try to find a matching entry in the gvd database.
* If one is found dispatch the message to a routine which will
* handle both the local GID effects and the GIP propagation to other ports.
*
* If no entry is found Leave and Empty messages can be discarded, but
* JoinIn and JoinEmpty messages demand further treatment. First, an attempt
* is made to create a new entry using free space (in the database, which
* corresponds to a free GID machine set). If this fails an attempt may be
* made to recover space from a machine set which is in an unused or less
* significant state. Finally the database is consider full and the received
* message is discarded.
*
* Once (if) an entry is found, Leave, Empty, JoinIn, and JoinEmpty are
* all submitted to GID (gid_rcv_msg()).
*
* JoinIn and JoinEmpty may cause Join indications, which are then propagated
* by GIP.
*
* On a shared medium, Leave and Empty will not give rise to indications
* immediately. However this routine does test for and propagate
* Leave indications so that it can be used unchanged with a poTuint32 to point
* protocol enhancement.
*/
static void gvr_rcv_msg(Gvr *gvr, Gid *my_port, Gvf_msg *msg)
{ 
 Tuint32 gid_index  = GVRP_DB_SIZE; 
 Garp    *application = &my_gvr->garp;

 if (msg->event == Gid_rcv_leaveall) {
    gid_rcv_leaveall(my_port);
 }
 else {
  if(!gvd_find_entry(my_gvr->gvd, msg->key1, &gid_index)) {
   if((msg->event == Gid_rcv_joinin)|| (msg->event == Gid_rcv_joinempty)) {
     if(!gvd_create_entry(my_gvr->gvd, msg->key1, &gid_index)) {
        if(gid_find_unused( application, 0 , &gid_index)) {
           gvd_delete_entry(my_gvr->gvd, gid_index);
           gvd_create_entry(my_gvr->gvd, msg->key1,&gid_index);
	}
        else {
           gvr_db_full(my_gvr, my_port);
	}
     }//if
   }//if
  }//if

  if (gid_index != GVRP_DB_SIZE) {
     gid_rcv_msg(my_port, gid_index, msg->event);
  }
 }//else

}


/* Process an entire received pdu for this instance of GVR */
Tuint32 gvr_rcv(void *gvr, void *port, void *rx_pdu)
{ 

  Gvf_msg  msg;
  Gid *my_port=(Gid *)port;
  Tuint32 i;

  Pdu *rcv_pdu=(Pdu *)rx_pdu;
  Tuint8 *pdu= rcv_pdu->data+3; //3byte = skip dsap,ssap,ctl field
#ifdef __LINUX_2_6_19__
  Tuint16 pdu_len =ntohs(eth_hdr(rcv_pdu)->h_proto);
#else
  Tuint16 pdu_len =ntohs(rcv_pdu->mac.ethernet->h_proto);
#endif
  
  // *** Parsing GVRP PDU
  if(gvrp_debug_level & GVRP_DEBUG_PDU){
   for(i=0;i< pdu_len ;i++){
    printk("%x-",*(rcv_pdu->data+i));
   }
   
    printk("\n");
  }


   // get protocol id 
  if( get_2_bytes(&pdu) != 0x01 ){
     if(gvrp_debug_level & GVRP_DEBUG_ERR){
        printk(" GVRP: protocol id != 0x01 \n");
     }
     
     return(False);
  }   
  

  // Maybe more than one message in the same PDU
  while((check_1_byte(&pdu)!= 0x00) && (pdu < rcv_pdu->data + pdu_len))
  {
   // get attribute type
   if(get_1_byte(&pdu)!=0x01) { // VID
     if(gvrp_debug_level & GVRP_DEBUG_ERR){
        printk(" GVRP: VID Type != 1 \n");
     }
     
     if(skip_attribute_list(&pdu, pdu_len)==True){
        continue;
     }
     else {
        return(True); //no other message in PDU 
     }
   }   
     
   // parsing attribute list
   // if attribute length or event or value has error...skip this attribute
   while(check_1_byte(&pdu)!=0x00) {//check attribute length
    
    //There is another attribute in the message X
    //boundary check
    if( (rcv_pdu->data+pdu_len)-pdu >= 4 ) {//4bytes = len+event+value (2 x end_of_mark)
       gvrp_parsing_attribute(&pdu,&msg); //get attribute length/event/value
    }
    else {
       return(False);
    }

    // invaild value for attribute event
    if(msg.event==Gid_null) {
      if(gvrp_debug_level & GVRP_DEBUG_ERR){
        printk(" GVRP: msg.event ==Gid_null \n");
      }
      
      continue;   
    }
    
    //attribute_length of LeaveAll msg is 2(attribute_length+attribute_event)
    if( (msg.event==Gid_rcv_leaveall) && (msg.length!=2) ) {
      if(gvrp_debug_level & GVRP_DEBUG_ERR){
        printk(" GVRP: msg.event==Gid_rcv_leaveall && msg.length=%lu !=2 \n",msg.length);
      }

      continue;
    }
   
    if( (msg.event!=Gid_rcv_leaveall) && (msg.length!=4) ) {
      if(gvrp_debug_level & GVRP_DEBUG_ERR){
        printk(" GVRP: msg.event!=Gid_rcv_leaveall  && msg.length=%lu !=4 \n",msg.length);
      }
      
      continue;
    }
  
        
    //VLAN ID= 0 / 4095 are reserved  (802.1Q-1998 p12)
    //The maximum number of VLANs that can be supported is 4094 
    //rather than 4096, as the VID values 0 and FFF are reserved,
    if( msg.event!=Gid_rcv_leaveall && (msg.key1 < 1 || msg.key1 > 4094)) {
      if(gvrp_debug_level & GVRP_DEBUG_ERR){
        printk(" GVRP: 4094 < VLAN_ID=%d < 1 \n",msg.key1);
      }

      continue;
    }
    
  
     //update gvrp_statistics counter 
     if(msg.event==Gid_rcv_leaveall) {
        gvrp_statistics[my_port->port_no].LeaveAll_rx++;
     }
     else if(msg.event==Gid_rcv_joinempty) { 
        gvrp_statistics[my_port->port_no].JoinEmpty_rx++;
     }
     else if(msg.event==Gid_rcv_joinin) { 
        gvrp_statistics[my_port->port_no].JoinIn_rx++;
     }
     else if(msg.event==Gid_rcv_empty) {
        gvrp_statistics[my_port->port_no].Empty_rx++;
     }
     else if(msg.event==Gid_rcv_leavein) {
        gvrp_statistics[my_port->port_no].LeaveIn_rx++;
     }
     else if(msg.event==Gid_rcv_leaveempty) {
        gvrp_statistics[my_port->port_no].LeaveEmpty_rx++;
     }
     
     gvr_rcv_msg(my_gvr, my_port, &msg); 
    
   }//while
  
   (void)get_1_byte(&pdu); //There is an end_of_mark field between two message 
      
  }//while

  return(True);
}


/******************************************************************************
* GVR : GARP VLAN REGISTRATION APPLICATION : TRANSMIT PROCESSING
*******************************************************************************/
/* Fill in msg fields for transmission. */

Tbool gvr_tx_msg(Gvr *my_gvr, Tuint32 gid_index, Gvf_msg *msg, Tuint32 port_no)
{  
  Garp *application=&my_gvr->garp;
  
  msg->attribute = Vlan_attribute; //attribute type=0x01=VID
  
  if(msg->event != Gid_tx_leaveall) { 
    
    if(gid_index==1) { 
       msg->key1=1 ;//default vlan
       return True;
    }
    else {  
       gvd_get_key(my_gvr->gvd, gid_index, &msg->key1);//fill in attribute value

       if( (msg->key1==0) || ( application->gip[gid_index]==0)){
         return False;//can't find or inactive
       }
       else {
         return True;
       }
    }
  }  
 
  return True;
}


/*
* Get and prepare a pdu for the transmission, if one is not available
* simply return, if there is more to transmit GID will reschedule a call
* to this function.
*
* Get messages to transmit from GID and pack them into the pdu using GVF
* (GARP VLAN pdu Formatter).
*/
void gvr_tx(void *gvr, void *port)
{  
 Pdu         	 *pdu;
 Gvf_msg     	 msg;
 Gid_event   	 tx_event;
 Tuint32     	 gid_index;
 Gid         	 *my_port=port;
 Tuint8		 end_mark=0x00;
 Tuint32         pkt_valid=0;
 garp_statistics tmp_counter;
 
 memset(&tmp_counter,0,sizeof(garp_statistics));
 
 /*if there is more to transmit GID will reschedule a call to this function*/
 if ((tx_event = gid_next_tx(my_port, &gid_index)) != Gid_null) {
   
   if (syspdu_alloc(&pdu)) {
   
     gvrp_wrmsg_init(&pdu);
     
     do {
        
	msg.event = tx_event; //attribute event
	
	if(!gvr_tx_msg(my_gvr, gid_index, &msg, my_port->port_no)) {// get attribute_type & vid to msg
	  continue; //vid value is invalid
	}
          
       	if (!gvrp_wrmsg(&pdu, &msg)) { // fill out message N
	  syspdu_free(pdu);//added by rhliu
	  return;
	}	

	if(msg.event==0)
  	   tmp_counter.LeaveAll_tx++;
	else if(msg.event==1) 
	   tmp_counter.JoinEmpty_tx++;
	else if(msg.event==2) 
	   tmp_counter.JoinIn_tx++;
	else if(msg.event==3) 
	   tmp_counter.LeaveEmpty_tx++;
	else if(msg.event==4) 
   	   tmp_counter.LeaveIn_tx++;
	else if(msg.event==5) 
	   tmp_counter.Empty_tx++;

	//Now, PDU have payload
	pkt_valid=True;	
	
     } while((tx_event = gid_next_tx(my_port, &gid_index))!= Gid_null);
   
     put_unaligned(end_mark,(Tuint8 *)skb_put(pdu,1));    
     put_unaligned(end_mark,(Tuint8 *)skb_put(pdu,1));  
    
     if(pkt_valid==False) {// pkt is invalid (only have garp header)
       syspdu_free(pdu);
     }
     else if( (cclmx_PortGetStpState(my_port->port_no)!=STP_PORT_FORWARDING)
            ||(cclmx_PortGetLink(my_port->port_no)==CCLMX_PORT_LINKDOWN)) {
       syspdu_free(pdu);
     }
     else {
       gvrp_statistics[my_port->port_no].LeaveAll_tx   +=tmp_counter.LeaveAll_tx;
       gvrp_statistics[my_port->port_no].JoinEmpty_tx  +=tmp_counter.JoinEmpty_tx;
       gvrp_statistics[my_port->port_no].JoinIn_tx     +=tmp_counter.JoinIn_tx;
       gvrp_statistics[my_port->port_no].LeaveEmpty_tx +=tmp_counter.LeaveEmpty_tx;
       gvrp_statistics[my_port->port_no].LeaveIn_tx    +=tmp_counter.LeaveIn_tx;
       gvrp_statistics[my_port->port_no].Empty_tx      +=tmp_counter.Empty_tx;
     
       syspdu_tx(pdu,my_port->port_no);
     }
    }//if
 }//if

}

